/*
ntpdate - set the time of day by polling one or more NTP servers
*/

/*
History

1.0     24-Jun-1994     The first translation of ntpdate for Unix
1.1      6-Jan-1995     The number of seconds in binary time now
                        overflows long fixed point in ntp.  Fixed
                        by subtracting the UTC offset first.
/*

/*
   Includes
*/

#ifdef __decc
#include        <in.h>
#include        <inet.h>
#endif

#include <descrip.h>
#include <iodef.h>
#include <libdef.h>
#include <libdtdef.h>
#include <ssdef.h>
#include <starlet.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ucx$inetdef.h>
#include "config.h"

/*
   Defines
*/

#define FALSE 0                     /* Boolean stuff */
#define TRUE 1
#define TIMEOUT_FSECS 0.2           /* Timer timeout unit in frac secs */
#define DEFTIMEOUT 5                /* Default num of timeout units */
#define HIGH_PRI 10                 /* High priority value */
#define MAX_ERRSTR 257              /* Max len of an error msg string */
#define MAX_TIMESTR 32              /* Max len of a time string */
#define INET_DEVNAME "_BG0:"        /* Internet device name */
#define NTP_SHIFT 8
#define NTP_PORT 123                /* Network port for ntp server */
#define MIN_MAC_LEN (sizeof(u_long) + 8)            /* DES */
#define MAX_MAC_LEN (sizeof(u_long) + 16)           /* MD5 */
#define NTP_VERSION 3               /* Default ntp version */
#define MODE_CLIENT 3               /* Client mode */
#define MODE_SERVER 4               /* Server mode */
#define NTP_MAXSTRATUM 15           /* Largest stratum */
#define UTCBASE "01-JAN-1900 00:00:00.00"    /* UTC starts on this date */
#define SANE_TIME "01-JAN-1995 00:00:00.00"  /* A known, sane time */
#define NSPS 1000000000             /* Nanoseconds per second */
#define HNSPS 10000000              /* Hundreds of nanoseconds per second */
#define LEAP_NOTINSYNC 3
#define STRATUM_UNSPEC (NTP_MAXSTRATUM + 1) /* Stratum unspecified */
#define STRATUM_PKT_UNSPEC 0        /* Packet stratum unspecified */
#define PKT_TO_STRATUM(s)       (((s) == STRATUM_PKT_UNSPEC) ?\
                                 (STRATUM_UNSPEC) : (s))
#define STRATUM_TO_PKT(s)       (((s) == (STRATUM_UNSPEC)) ?\
                                (STRATUM_PKT_UNSPEC) : (s))
#define NTP_MINPOLL 4
#define NTPDATE_PRECISION -6     /* Our claimed precision */
#define NTPDATE_DISTANCE 0x10000 /* 1 fixed-point second */
#define NTPDATE_DISP 0x10000     /* Dispersion */
#define NTPDATE_REFID 0          /* Reference ID */
#define DEFSAMPLES 4             /* Default number of samples to take */
#define HIGHBIT_MASK 0x80000000  /* Get the high bit */
#define MASK_HIGHBIT 0x7fffffff  /* Get everything but the high bit */
#define HR_SECS      60 * 60     /* Seconds in an hour */
#define DAY_SECS     24 * HR_SECS   /* Seconds in a day */
#define DST_ADJ      HR_SECS     /* Secs Daylight Savings Time adjustment */
#define LTZ_ADJ      UTC_OFFSET * HR_SECS    /* Secs from UTC to local TZ */
#define NTP_MAXSKW 100000        /* 0.01 in hnsps */
#define NTP_MINDIST 200000       /* 0.02 in hnsps */
#define PEER_MAXDISP 64          /* In seconds */
#define NTP_MAXCLOCK 10
#define NTP_INFIN 15
#define NTP_MAXWGT (8 * HNSPS)
#define NTP_MAXAGE 86400         /* maximum age, one day in seconds */
#define NTP_MAXLIST 5            /* Pick only the top best servers */

/*
   Type Definitions
*/

typedef unsigned char u_char;       /* Convenience types */
typedef unsigned short u_short;
typedef unsigned long u_long;

typedef struct {
   u_long qd_lo;                    /* Low-order word */
   long qd_hi;                      /* High-order word, negative is delta */
   } bintime;                       /* Quad word (64 bits) */

typedef struct {
   u_short cond;                    /* Condition code */
   u_short count;                   /* Transfer count */
   u_long dev_spec;                 /* Device-specific information */
   } itm_lst;                       /* An item list */

typedef struct {
   u_short len;                     /* Length of information */
   u_short item_code;               /* Item code */
   char *address;                   /* Address of information */
   } itm_lst2;                      /* Item list 2 */

typedef struct {
   u_short buff_len;                /* Length of the buffer */
   u_short item_code;               /* Item code */
   char *buff;                      /* Pointer to the buffer */
   u_long *ret_len;                 /* Length of returned information */
   } itm_lst3;                      /* Item list 3 */

typedef struct {
   u_short proto;                   /* Protocol (tcp, udp, ip) */
   u_char ptype;                    /* Prototype (dgram, stream, raw) */
   u_char dname;                    /* Domain name (default of Internet) */
   } sock_parms;                    /* Socket parameters */

typedef struct {
   u_short family;                  /* Protocol family */
   u_short port;                    /* Protocol port */
   u_long addr;                     /* Address */
   u_long reserved1;
   u_long reserved2;
   } sock_addr;                     /* Socket address */

typedef struct {
   u_char type;                     /* Type of subfunction */
   u_char call_code;
   u_short reserved;
   } sub_funct;

typedef struct {
   u_long l_ui;                     /* Long unsigned integer */
   u_long l_uf;                     /* Long unsigned fixed point*/
   } l_fp;                          /* Long fixed point */

typedef struct {
   char *hostname;         /* The server's name */
   sock_addr s_addr;       /* The socket address */
   u_char leap;            /* Leap second indicator */
   u_char stratum;         /* Server's startum */
   char precision;         /* Server's precision */
   u_long rootdelay;       /* Delay to the primary clock */
   u_long rootdispersion;  /* dispersion */
   u_long refid;           /* reference clock ID */
   bintime reftime;        /* Server's time of last update */
   u_long event_time;      /* Current clock period */
   u_short xmtcnt;         /* Number of packets sent so far */
   u_short filter_nextpkt; /* Number/order of next packet */
   u_long filter_delay[NTP_SHIFT];    /* Delay of packets */
   bintime filter_offset[NTP_SHIFT];  /* Offset from packets */
   bintime org;            /* Time the server originated the packet */
   bintime srec;           /* Time the server received the packet */
   bintime rec;            /* Time we received the packet back */
   bintime xmt;            /* Time we sent the packet */
   u_long delay;           /* Filter estimated delay */
   bintime dispersion;     /* Filter estimated dispersion */
   bintime offset;         /* Filter estimated clock offset */
   } server;               /* Info from/about a server */

typedef struct {
   unsigned mode:3;        /* Mode */
   unsigned version:3;     /* Version */
   unsigned leap:2;        /* Leap indicator */
   u_char stratum;         /* peer's stratum */
   u_char ppoll;           /* the peer polling interval */
   char precision;         /* peer clock precision */
   u_long rootdelay;         /* distance to primary clock */
   u_long rootdispersion;    /* clock dispersion */
   u_long refid;           /* reference clock ID */
   l_fp reftime;           /* time peer clock was last updated */
   l_fp org;               /* originate time stamp */
   l_fp rec;               /* receive time stamp */
   l_fp xmt;               /* transmit time stamp */
   u_long keyid;           /* (Here and below optional) key identification */
   u_char mac[MAX_MAC_LEN - sizeof(u_long)]; /* message-authentication code */
   } ntp_pkt;

typedef struct rcvbuf {
   struct rcvbuf *next;    /* Point to next one of these */
   ntp_pkt buf;            /* Holds one ntp packet */
   itm_lst iosb;           /* IO status block */
   sock_addr s_addr;       /* The socket address */
   itm_lst3 remote_host;   /* The remote host */
   u_long itm3_retlen;     /* Length of returned information */
   bintime bt_rcv;         /* Time packet was received */
   } rcv_buf;              /* A receive buffer */

typedef struct {
   u_short yr;                /* Year since year 0 */
   u_short mon;               /* Month of year */
   u_short day;               /* Day of month */
   u_short hr;                /* Hour of day */
   u_short min;               /* Minute of hour */
   u_short sec;               /* Second of minute */
   u_short hun;               /* Hundredths of second */
   } numtim;                  /* Structure for sys$numtim's "numeric time" */

typedef struct dsc$descriptor_s dscr_s; /* A string descriptor */

#define MIN_PKTLEN (sizeof(ntp_pkt) - MAX_MAC_LEN)

/*
   Function prototypes
*/

void Init(void);
void Sanity_Check(void);
void Get_Args(int *argcnt, char *args[]);
void Usage(void);
void Open_Net(void);
void Get_Servers(int maxarg, int argnum, char *args[],
                 int *nsrvs, server **srvs_lst[]);
u_long Get_HostAddr(char *name_str);
rcv_buf *Get_RcvBufs(int nsrvs);
void Poll_Srvs(void);
void Set_NetRead(void);
void NetRead_AST(void);
void Set_Timer(bintime bt_delta);
void Timer_AST(void);
u_long Set_Priority(u_long priority);
void Print_Error(u_long err_code);
void Poll_Inactive(void);
void Process_RcvPkts(void);
void Calc_Delays(u_long *dly, bintime *c_off, server *sptr);
server *Find_Server(sock_addr s_addr);
void Update_Filter(server *sptr, u_long dly, bintime c_off);
void Poll_Server(server *sptr);
void Send_Pkt(sock_addr *s_addr, ntp_pkt *pkt_ptr, u_long pkt_len);
void Close_Net(void);
void Clock_Filter(server *sptr);
server *Clock_Select(void);
void Clock_Adjust(void);
bintime FSecs2DBT(float frac_secs);
void *Get_ZMem(int num, int bytes);
bintime Str2Bt(char *time_str);
char *Bt2ASC(bintime bt_time, char *time_str, int str_len);
int Bt_gt(bintime bt_time1, bintime bt_time2);
int Bt_lt(bintime bt_time1, bintime bt_time2);
int Bt_ge(bintime bt_time1, bintime bt_tim2);
void Bt2SecsRem(bintime bt_time, long *seconds, long *remainder);
int Bt_eq(bintime bt_time1, bintime bt_time2);
l_fp Bt2Lfp(bintime bt_time);
bintime EMultUl(u_long multiplier, u_long multiplicand);
bintime EMultL(long multiplier, long multiplicand);
bintime MultHigh(u_long num1, u_long num2);
bintime Lfp2Bt(l_fp lfp_time);
bintime Bt_Sub(bintime bt_time1, bintime bt_time2);
bintime Bt_Add(bintime bt_time1, bintime bt_time2);
bintime Bt_Abs(bintime bt_time);
bintime Bt_ShiftR(bintime bt_time, int bits);
bintime Get_Time(void);
void Set_Time(bintime bt_time);
l_fp ntohl_fp(l_fp lfp_time);
l_fp htonl_fp(l_fp lfp_time);
int Lfp_eq(l_fp lfp_time1, l_fp lfp_time2);
int Lfp_ge(l_fp lfp_time1, l_fp lfp_time2);
int Dst(bintime bt_time);
void Dow_Adj(int cur_dow, int new_dow, int cmp, bintime *bt_dst);
char *Bt2Str(bintime bt_time, char *str);
void Print_Server(server *sptr);

/*
   Global Variables
*/

char *Program_Name;     /* How we were invoked */
u_short inet_chan;      /* Internet device channel number */
u_long timer_exp;       /* Timer expire time */
u_long timevt_flag;     /* Timer event flag */
u_long readevt_flag;    /* Read event flag */
u_long writevt_flag;    /* Write event flag */
rcv_buf *rbufq_head;    /* Received buffer queue head */
rcv_buf *rbufq_tail;    /* Received buffer queue tail */
server **srv_lst;       /* List of servers */
int sys_numservers;     /* Number of servers */
int sys_version = NTP_VERSION;   /* Our ntp version */
bintime bt_utcoffset;   /* Binary time version of offset from utc */
bintime bt_dstadj;      /* Binary time version of DST adjustment */
bintime bt_zero;        /* Zero in binary time */
bintime bt_maxage;
char *utcbase_str = UTCBASE;  /* utc base time */
char *sanetime_str = SANE_TIME;  /* A sane time */
u_long current_time = 1;   /* Our pseudo-clock with generic units */
u_long sys_timeout = DEFTIMEOUT; /* Timeout after this many periods */
int sys_samples = DEFSAMPLES;    /* Number of packets to sample */
int poll_num;                    /* Number of servers we are polling */
int simple_query = FALSE;        /* True if just checking, not setting */
int debug = FALSE;               /* True if debugging */

$DESCRIPTOR(inet_dev, INET_DEVNAME);

main(int argc, char *argv[])
{
   int cur_arg;  /* Points to the current argument in the list */

   Init();
   Get_Args(&cur_arg, argv);
   Open_Net();
   Get_Servers(argc, cur_arg, argv, &sys_numservers, &srv_lst);
   rbufq_head = rbufq_tail = Get_RcvBufs(sys_numservers);
   Poll_Srvs();
   Close_Net();
   Clock_Adjust();
}

void Init(void)
/*
Description:  Calculate the amount of the daylight saving time
adjustment.  Calculate the adjustment from UTC.  Calculate the maximum
age we will allow on a packet.  Reserve event flags for the various
asynchronous functions.
*/
{
   bt_dstadj = EMultUl(DST_ADJ, HNSPS);
   bt_utcoffset = Bt_Add(Str2Bt(utcbase_str), EMultL(LTZ_ADJ, HNSPS));
   bt_maxage = EMultUl(NTP_MAXAGE, HNSPS);
   lib$get_ef(&timevt_flag);
   lib$get_ef(&readevt_flag);
   lib$get_ef(&writevt_flag);
}

void Sanity_Check(void)
/*
Description:  If the current time is somewhere in the past, set the time
to something sane.
*/
{
   bintime bt_sanetime;  /* Binary version of the time */

   bt_sanetime = Str2Bt(sanetime_str);
   if (Bt_gt(bt_sanetime, Get_Time())) Set_Time(bt_sanetime);
}

void Get_Args(int *argcnt, char *args[])
/*
Description:  Get the command line switches and arguments.  If
something's not right, set the error flag.  If there is an error or no
arguments are left for server names, tell the user how to invoke the
program.
*/
{
   int error_flag;      /* Counts each error encountered */
   int temp_samples;    /* Temporary number of samples */

   Program_Name = args[0];
   error_flag = 0;
   *argcnt = 1;   /* Skip program name */
   while (args[*argcnt] && (args[*argcnt][0] == '-'))
   {
      if (strlen(args[*argcnt]) != 2)
         printf("Unknown flag '%s'.\n", args[*argcnt]);
      else
         switch (tolower(args[*argcnt][1]))
         {
            case 'd':         /* Set debugging */
               debug++;
               break;
            case 'o':         /* Select ntp version */
               (*argcnt)++;
               sscanf(args[*argcnt], "%d", &sys_version);
               break;
            case 'p':         /* Number of packets to sample */
               (*argcnt)++;
               sscanf(args[*argcnt], "%d", &temp_samples);
               if (temp_samples <= 0 || temp_samples > NTP_SHIFT)
               {
                  printf("Number of packets to sample (%d) ", temp_samples);
                  printf("is not between 0 and %d ", NTP_SHIFT);
                  printf("(default of %d).\n", DEFSAMPLES);
                  error_flag++;
               }
               else sys_samples = temp_samples;
               break;
            case 'q':         /* Just a simple query */
               simple_query = TRUE;
               break;
            default:
               printf("Unknown flag '%s'.\n", args[*argcnt]);
               break;
         }
      (*argcnt)++;
   }
   if (!args[*argcnt] || error_flag) Usage();
}

void Usage(void)
/*
Description:  Tell the user how to invoke the program.  Say what default
and min/max values are, if possible.
*/
{
   printf("\nUsage:  ntpdate [-d] [-o version] [-p packets] [-q] [-v] ");
   printf("host [host ...]\n\n");
   printf("        -d            turn on debugging\n");
   printf("        -o version    NTP version ");
   printf("(default is %d)\n", NTP_VERSION);
   printf("        -p packets    number of packets to sample ");
   printf("(default is %d)\n", DEFSAMPLES);
   printf("        -q            simple query, don't set system time\n");
   printf("\n");
   exit(1);
}

void Open_Net(void)
/*
Description:  Assign the internet device and then open a UDP channel on
the NTP port.
*/
{
   u_long status;          /* Return status of the system calls */
   u_long evt_flag;        /* Event flag for QIO */
   itm_lst iosb;           /* I/O status block */
   sock_parms s_parms;     /* The socket parameters */
   itm_lst2 host;          /* Item list for the host address */
   sock_addr host_addr;    /* Our socket address */

   status = sys$assign(&inet_dev, &inet_chan, 0, 0);
   if (status != SS$_NORMAL)
   {
      printf("Failed to assign channel to Internet device.\n");
      lib$signal(status);
   }

   s_parms.proto = INET$C_UDP;
   s_parms.ptype = INET_PROTYP$C_DGRAM;
   s_parms.dname = 0;
   host_addr.family = INET$C_AF_INET;
   host_addr.port = htons(NTP_PORT);
   host_addr.addr = 0;
   host_addr.reserved1 = 0;
   host_addr.reserved2 = 0;
   host.len = sizeof(sock_addr);
   host.item_code = 0;
   host.address = &host_addr;

   lib$get_ef(&evt_flag);
   status = sys$qiow(evt_flag,
      inet_chan,
      IO$_SETMODE,
      &iosb, 0, 0,
      &s_parms, 0,
      &host, 0, 0, 0);
   lib$free_ef(&evt_flag);

   if (status != SS$_NORMAL)
   {
      printf("Failed to create UDP socket.\n");
      lib$signal(status);
   }
   if ((iosb.cond & 1) == 0)
   {
      printf("Socket bind error.\n");
      lib$signal(iosb.cond);
   }
}

void Get_Servers(int maxarg, int argnum, char *args[],
                 int *nsrvs, server **srvs_lst[])
/*
Description:  Get server names out of args[] until no more are left. 
Get the IP number of each server.  If an IP number is not found, warn
the user, ignore the name and get the next one.  If an IP number is
found, create a server structure and fill it with the necessary
information.
*/
{
   u_long ipnum;        /* The server's IP number */
   server **srvs_ptr;   /* Pointer to the list of server pointers */
   server *srv_ptr;     /* Points to one server */
   int num_srv;         /* Number of acceptable servers found */
   int max_srv;         /* Max possible servers from arguments */

   num_srv = 0;         /* None found yet */
   max_srv = maxarg - argnum;    /* Could find this many servers */
   srvs_ptr = Get_ZMem(max_srv, sizeof(server *)); /* List of pointers */
   while (args[argnum])
   {
      ipnum = Get_HostAddr(args[argnum]);
      if (!ipnum)
         printf("Server not found - %s\n", args[argnum]);
      else
      {
         srv_ptr = Get_ZMem(1, sizeof(server));
         srv_ptr->hostname = Get_ZMem(strlen(args[argnum]) + 1, sizeof(char));
         strcpy(srv_ptr->hostname, args[argnum]);
         srv_ptr->s_addr.family = UCX$C_AF_INET;
         srv_ptr->s_addr.port = htons(NTP_PORT);
         srv_ptr->s_addr.addr = ipnum;
         srvs_ptr[num_srv++] = srv_ptr;
         srv_ptr->event_time = num_srv;
      }
      argnum++;
   }
   if (!num_srv)
   {
      printf("No servers found.\n");
      exit(1);
   }
   *srvs_lst = srvs_ptr;
   *nsrvs = num_srv;
}

u_long Get_HostAddr(char *name_str)
/*
Description:  Do a hostname lookup and return the IP number if it is
found.
*/
{
   u_long status;          /* Return status of the system call */
   u_long evt_flag;        /* Event flag for the system call */
   u_long ip_num;          /* IP number of the host */
   sub_funct sfun;         /* A subfunction of the call */
   dscr_s subfunct_code;   /* Subfunction code */
   dscr_s hname;           /* The host name string descriptor */
   dscr_s host_addr;       /* The host address descriptor */
   itm_lst iosb;           /* I/O status block */
   u_short addr_len;       /* The length of the returned address */

   sfun.type = INETACP_FUNC$C_GETHOSTBYNAME; /* Specific subfunction */
   sfun.call_code = INETACPC$C_TRANS;
   sfun.reserved = 0;
   subfunct_code.dsc$w_length = sizeof sfun;
   subfunct_code.dsc$b_dtype = DSC$K_DTYPE_T;
   subfunct_code.dsc$b_class = DSC$K_CLASS_S;
   subfunct_code.dsc$a_pointer = &sfun;

   hname.dsc$w_length = strlen(name_str);
   hname.dsc$b_dtype = DSC$K_DTYPE_T;
   hname.dsc$b_class = DSC$K_CLASS_S;
   hname.dsc$a_pointer = name_str;

   host_addr.dsc$w_length = sizeof ip_num;
   host_addr.dsc$b_dtype = DSC$K_DTYPE_T;
   host_addr.dsc$b_class = DSC$K_CLASS_S;
   host_addr.dsc$a_pointer = &ip_num;

   lib$get_ef(&evt_flag);
   status = sys$qiow(evt_flag,
      inet_chan,
      IO$_ACPCONTROL,
      &iosb, 0, 0,
      &subfunct_code,
      &hname,
      &addr_len,
      &host_addr, 0, 0);
   lib$free_ef(&evt_flag);

   if ((status != SS$_NORMAL) || ((iosb.cond & 1) == 0)) ip_num = 0;
   return(ip_num);
}

rcv_buf *Get_RcvBufs(int nsrvs)
/*
Description:  This function creates a circular receive queue using a
linked list for storing the packets from the servers when they arrive.
*/
{
   int cnt;             /* Count through the buffers as they are linked in */
   int nbufs;           /* Number of buffers to create */
   rcv_buf *head_ptr;   /* Head of the linked list */
   rcv_buf *tail_ptr;   /* Tail of the linked list */
   rcv_buf *tmp_ptr;    /* Temporarily points to buffer before linked in */

   nbufs = nsrvs + ((nsrvs + 1) / 2);           /* 50% more */
   head_ptr = Get_ZMem(1, sizeof(rcv_buf));     /* Plus 1 additional */
   tail_ptr = head_ptr;
   tail_ptr->remote_host.buff_len = sizeof tail_ptr->s_addr;
   tail_ptr->remote_host.item_code = 0;
   tail_ptr->remote_host.buff = &tail_ptr->s_addr;
   tail_ptr->remote_host.ret_len = &tail_ptr->itm3_retlen;
   for (cnt = 0; cnt < nbufs; cnt++)
   {
      tmp_ptr = Get_ZMem(1, sizeof(rcv_buf));
      tail_ptr->next = tmp_ptr;
      tail_ptr = tmp_ptr;
      tail_ptr->remote_host.buff_len = sizeof tail_ptr->s_addr;
      tail_ptr->remote_host.item_code = 0;
      tail_ptr->remote_host.buff = &tail_ptr->s_addr;
      tail_ptr->remote_host.ret_len = &tail_ptr->itm3_retlen;
   }
   tail_ptr->next = head_ptr;    /* Make it circular */
   return(head_ptr);
}

void Poll_Srvs(void)
/*
Description:  First set the time to something sane.  Then raise our
priority, set the asynchronous routine for reading packets from the
network, and every time the timer has expired, poll any inactive servers
again, reset the timer and then hibernate until either a packet arrives
or the timer expires.  If any packets have arrived, process them and go
through this all again as long as there are still servers we need to
poll.
*/
{
   u_long norm_pri;     /* What the normal priority was */
   bintime bt_timeout;  /* Timer timeout */

   Sanity_Check();      /* Make the time sane */
   bt_timeout = FSecs2DBT(TIMEOUT_FSECS);
   timer_exp = TRUE;    /* Assume it has already expired */
   norm_pri = Set_Priority(HIGH_PRI);
   Set_NetRead();
   poll_num = sys_numservers;
   while (poll_num)
   {
      if (timer_exp)
      {
         Poll_Inactive();
         timer_exp = FALSE;
         Set_Timer(bt_timeout);
      }
      sys$hiber();      /* Wait for a packet or timer expiration */
      if (rbufq_head != rbufq_tail) Process_RcvPkts();  /* Queue empty? */
   }
   sys$cantim(0, 0);    /* Cancel the timer, we're done */
   Set_Priority(norm_pri);
}

void Set_NetRead(void)
/*
Description:  Specify the function to handle asynchronous reads when a
packet arrives.
*/
{
   u_long status;    /* Return status of the system call */

   status = sys$qio(readevt_flag,
      inet_chan,
      IO$_READVBLK,
      &rbufq_tail->iosb,
      NetRead_AST, 0,
      &rbufq_tail->buf,          /* The buffer to receive the packet */
      sizeof rbufq_tail->buf,
      &rbufq_tail->remote_host, 0, 0, 0);  /* Sending host */
   if (status != SS$_NORMAL)
   {
      printf("Failure to read from network.\n");
      Print_Error(status);
   }
}

void NetRead_AST(void)
/*
Description:  Check if there was some kind of error on reception or if
the read was cancelled somehow (e.g. internet device was closed). 
Otherwise, stick the time in the packet, find the next free buffer and
set ourselves up to read the next one.
*/
{
   if ((rbufq_tail->iosb.cond & 1) == 0)
   {
      if (rbufq_tail->iosb.cond != SS$_CANCEL)
      {
         printf("NetRead I/O error.\n");
         lib$signal(rbufq_tail->iosb.cond);
      }
   }
   else
   {
      rbufq_tail->bt_rcv = Get_Time();
      rbufq_tail = rbufq_tail->next;   /* Next free buffer */
      sys$wake(0, 0);   /* Wake ourselves up if sleeping */
      Set_NetRead();    /* Set ourselves up for the the next packet */
   }
}

void Poll_Inactive(void)
/*
Description:  Go through the list of servers.  If any server has an
event time (0 means we finished polling it) and it's less than the
current time period, it is a candidate for polling again.  If we
transmitted a packet to it already, it didn't respond before the timer
expiration (or we wouldn't be here), so put a zero in the filter.  Then poll.
*/
{
   int cnt;    /* Count through the list of servers */

   for (cnt = 0; cnt < sys_numservers; cnt++)
      if ((srv_lst[cnt]->event_time) &&
          (srv_lst[cnt]->event_time <= current_time))
      {
         if (srv_lst[cnt]->xmtcnt)
            Update_Filter(srv_lst[cnt], 0, bt_zero);
         Poll_Server(srv_lst[cnt]);
      }
}

void Set_Timer(bintime bt_delta)
/*
Description:  Set when the timer should expire (delta time) and the
function to execute when it does.  Resolution of the timer on a VAX is
10 milliseconds.
*/
{
   u_long status;    /* Return status of the system call */

   status = sys$setimr(timevt_flag,
      &bt_delta,
      Timer_AST, 0, 0);
   if (status != SS$_NORMAL)
   {
      printf("Failure to set the timer.\n");
      lib$signal(status);
   }
}

void Timer_AST(void)
/*
Description:  Set the flag that the timer has expired, increment the
time period, and wake ourselves up.
*/
{
   timer_exp = TRUE;
   current_time++;
   sys$wake(0,0);
}

u_long Set_Priority(u_long priority)
/*
Description:  Set our priority to the requested priority and return the
previous priority value.
*/
{
   u_long status;    /* Return status of the system call */
   u_long prev_pri;  /* Value of the previous priority */

   status = sys$setpri(0, 0, priority, &prev_pri);
   if (status != SS$_NORMAL)
   {
      printf("Failure to set priority to %lu.\n", priority);
      Print_Error(status);
   }
   return(prev_pri);
}

void Print_Error(u_long err_code)
/*
Description:  Print the system error message using the specified error
code.
*/
{
   char err_str[MAX_ERRSTR];        /* Holds error message */
   $DESCRIPTOR(err_msg, err_str);   /* Error string descriptor */
   u_short msglen;                  /* Length of the error message */

   sys$getmsg(err_code, &msglen, &err_msg, 0, 0);
   err_str[msglen] = '\0';
   printf("%s\n", err_str);
}

void Process_RcvPkts(void)
/*
Description:  Process all of the packets currently in the receive queue. 
If the packet passes all of the tests (there are a lot of them), then we
move the information into the server structure, calculate the delays,
move to the next buffer, update the filter for this server and poll this
server again.  Do this to the next packet in the buffer.
   As times are moved into the server structure, they are adjusted by
the UTC offset.  This is so that the times are at least within one hour
of what our time should be.  This just makes it easier down the road for
calculations.  It would be a lot easier if VMS stored a
timezone/daylight saving time independent time.
   UTC is safe, but we can't add the DST adjustment here.  Though, it
would be really nice if we could.  First, it means a lot of calculations
for each packet to find the current time when we are really only
interested in the offset at this point.  There would mucho problems with
filtering if one packet arrived and was adjusted but that the next one
wasn't.  Even more unlikely but possible is a packet which was received
in DST and was sent in standard time.
*/
{
   ntp_pkt *pkt_ptr;    /* Points to a packet from the server */
   server *srv_ptr;     /* Points to server of the packet */
   u_long delay;        /* Delay of the packet */
   bintime clk_offset;  /* The clock offset */
   l_fp lfp_zero;       /* A zero */

   lfp_zero.l_ui = lfp_zero.l_uf = 0;
   delay = 0;
   clk_offset.qd_hi = clk_offset.qd_lo = 0;
   srv_ptr = NULL;
   while (rbufq_head != rbufq_tail)    /* As long as there are packets */
   {
      pkt_ptr= &rbufq_head->buf;       /* Get a packet */
               /* Is the packet the right size? */
      if ((rbufq_head->iosb.count >= MIN_PKTLEN) &&
               /* Is it the version we requested? */
          (pkt_ptr->version == sys_version) &&
               /* Is the server in server mode? */
          (pkt_ptr->mode == MODE_SERVER) &&
               /* Is the stratum OK? */
          (pkt_ptr->stratum <= NTP_MAXSTRATUM) &&
               /* Is this a server we polled? */
          ((srv_ptr = Find_Server(rbufq_head->s_addr)) != NULL) &&
               /* Is our xmt and packet originate time the same? */
          (Bt_eq(Bt_Add(Lfp2Bt(ntohl_fp(pkt_ptr->org)), bt_utcoffset),
                 srv_ptr->xmt)) &&
               /* Is the receive time OK? */
          (!Lfp_eq(pkt_ptr->rec, lfp_zero)) &&
               /* >= 0 delay between server receive and transmit? */
          (Lfp_ge(ntohl_fp(pkt_ptr->xmt), ntohl_fp(pkt_ptr->rec))))
      {                                   /* Record the information */
         srv_ptr->leap = pkt_ptr->leap;
         srv_ptr->stratum = pkt_ptr->stratum;
         srv_ptr->precision = pkt_ptr->precision;
         srv_ptr->rootdelay = ntohl(pkt_ptr->rootdelay);
         srv_ptr->rootdispersion = ntohl(pkt_ptr->rootdispersion);
         srv_ptr->refid = pkt_ptr->refid;
         srv_ptr->reftime =
            Bt_Add(Lfp2Bt(ntohl_fp(pkt_ptr->reftime)), bt_utcoffset);
         srv_ptr->org = Bt_Add(Lfp2Bt(ntohl_fp(pkt_ptr->xmt)), bt_utcoffset);
         srv_ptr->srec = Bt_Add(Lfp2Bt(ntohl_fp(pkt_ptr->rec)), bt_utcoffset);
         srv_ptr->rec = rbufq_head->bt_rcv;
         Calc_Delays(&delay, &clk_offset, srv_ptr);
      }
      rbufq_head = rbufq_head->next;
      if (srv_ptr != NULL)
      {
         Update_Filter(srv_ptr, delay, clk_offset);
         Poll_Server(srv_ptr);
      } 
   }
}

void Calc_Delays(u_long *dly, bintime *c_off, server *sptr)
/*
Description:  Calculate the delay and clock offset.
*/
{
   bintime bt_t10;      /* Time 1 minus time 0 */
   bintime bt_t23;      /* Time 2 minus time 3 */
   bintime bt_clkoff;   /* Clock offset */
   bintime bt_delay;    /* The delay */
   bintime bt_tmp;      /* Temporarily holds a time value */

   bt_t10 = Bt_Sub(sptr->org, sptr->rec);    /* Delay from server to us */
   bt_t23 = Bt_Sub(sptr->srec, sptr->xmt);   /* Delay from us to server */
   bt_clkoff = Bt_Add(bt_t23, bt_t10);       /* Total travel time */
   bt_clkoff = Bt_ShiftR(bt_clkoff, 1);      /* Divide by 2 */
   bt_delay = Bt_Sub(bt_t23, bt_t10);        /* Total processing delay */
   bt_tmp.qd_hi = 0;
   bt_tmp.qd_lo = (HNSPS >> -NTPDATE_PRECISION) +
                  (HNSPS >> -sptr->precision) +
                  NTP_MAXSKW;
   bt_delay = Bt_Add(bt_delay, bt_tmp);
   if (Bt_lt(bt_delay, bt_zero))
   {
      bt_clkoff.qd_hi = bt_clkoff.qd_lo = 0;
      bt_delay.qd_hi = bt_delay.qd_lo = 0;
   }
   else
   {
      bt_tmp.qd_lo = NTP_MINDIST;
      if (Bt_gt(bt_tmp, bt_delay))
         bt_delay = bt_tmp;
   }
   *dly = bt_delay.qd_lo;
   *c_off = bt_clkoff;
}

server *Find_Server(sock_addr s_addr)
/*
Description:  Given a socket address, run through the list of servers and
see if this is one that we know about.  Check the source port so that no
one spoofs us.  If found, return its pointer.
*/
{
   int cnt;    /* Count through the list of servers */

   cnt = 0;
   while ((cnt < sys_numservers) &&
          (s_addr.addr != srv_lst[cnt]->s_addr.addr))
      cnt++;
   if ((cnt >= sys_numservers) || (ntohs(s_addr.port) != NTP_PORT))
      return(NULL);
   else
      return(srv_lst[cnt]);
}

void Update_Filter(server *sptr, u_long dly, bintime c_off)
/*
Description:  Update the filter for the server with the delay time and
the clock offset.
*/
{
   int num_fpkts;    /* Number of filter packets */

   num_fpkts = sptr->filter_nextpkt;
   if (num_fpkts < NTP_SHIFT)    /* Have room for one more? */
   {
      sptr->filter_delay[num_fpkts] = dly;
      sptr->filter_offset[num_fpkts] = c_off;
      sptr->filter_nextpkt++;    /* Count it */
   }
}

void Poll_Server(server *sptr)
/*
Description:  If the filter array has enough samples, then if the server
has an even time change it to zero to indicate we are finished polling
it.  Otherwise, fill in the ntp packet and send it to the server.
NB:  Ever since 1995, the number of seconds in binary time is greater
than l_fp in ntp.  Be sure to subtract bt_utcoffset so that it will fit. 
Remember to add it later on when comparing.
*/
{
   ntp_pkt pkt;   /* A network packet */

   if (sptr->filter_nextpkt >= sys_samples)
   {
      if (sptr->event_time)
      {
         poll_num--;
         sptr->event_time = 0;
      }
   }
   else
   {
      pkt.leap = LEAP_NOTINSYNC;
      pkt.version = sys_version;
      pkt.mode = MODE_CLIENT;
      pkt.stratum = STRATUM_TO_PKT(STRATUM_UNSPEC);
      pkt.ppoll = NTP_MINPOLL;
      pkt.precision = NTPDATE_PRECISION;
      pkt.rootdelay = htonl(NTPDATE_DISTANCE);
      pkt.rootdispersion = htonl(NTPDATE_DISP);
      pkt.refid = htonl(NTPDATE_REFID);
      pkt.reftime.l_ui = pkt.reftime.l_uf = 0;
      pkt.org.l_ui = pkt.org.l_uf = 0;
      pkt.rec.l_ui = pkt.rec.l_uf = 0;
      sptr->xmt = Get_Time();    /* Remember when we sent it */
      pkt.xmt = htonl_fp(Bt2Lfp(Bt_Sub(sptr->xmt, bt_utcoffset)));
      Send_Pkt(&(sptr->s_addr), &pkt, MIN_PKTLEN);
      sptr->event_time = current_time + sys_timeout;  /* When request expires */
      sptr->xmtcnt++;            /* Count the packet transmitted */
   }
}

void Send_Pkt(sock_addr *s_addr, ntp_pkt *pkt_ptr, u_long pkt_len)
/*
Description:  Given an address and a packet, send it on its way.
*/
{
   u_long status;          /* Return status from the system call */
   u_long evt_flag;        /* Event flag for the system call */
   itm_lst iosb;           /* I/O status block */
   itm_lst2 remote_host;   /* Remote host information */

   remote_host.len = sizeof *s_addr;
   remote_host.item_code = 0;
   remote_host.address = s_addr;
   status = sys$qiow(writevt_flag,
      inet_chan,
      IO$_WRITEVBLK,
      &iosb, 0, 0,
      pkt_ptr,
      pkt_len,
      &remote_host, 0, 0, 0);
   if (status != SS$_NORMAL)
   {
      printf("Unable to send packet.\n");
      Print_Error(status);
   }
   if ((iosb.cond & 1) == 0)
   {
      printf("Error writing packet to network.\n");
      Print_Error(iosb.cond);
   }
}

void Close_Net(void)
/*
Description:  Close the network by deassigning the network device.
*/
{
   sys$dassgn(inet_chan);
}

void Clock_Filter(server *sptr)
/*
Description:  Sort the delays to find the best one that isn't zero.  If
the best one is zero, save default information for the server.  If it
isn't zero, save the delay and offset for the server.  Then calculate
and save the dispersion (the sum of the difference between the best
offset and all the rest of the offsets) for the server.
*/
{
   int order[NTP_SHIFT];   /* Holds each filter delay's order */
   int tmp;                /* Temporary storage */
   int cnt1;               /* Count through samples */
   int cnt2;               /* Count through samples */
   bintime bt_peermaxdisp; /* Peer max dispersion */
   bintime bt_disp;        /* The dispersion */

   bt_peermaxdisp = EMultUl(PEER_MAXDISP, HNSPS);
   for (cnt1 = 0; cnt1 < sys_samples; cnt1++)
      order[cnt1] = cnt1;
                              /* Sort delays into order */
   for (cnt1 = 0; cnt1 < (sys_samples - 1); cnt1++)
      for (cnt2 = cnt1 + 1; cnt2 < sys_samples; cnt2++)
         if (sptr->filter_delay[order[cnt2]] != 0 &&
             (sptr->filter_delay[order[cnt1]] == 0 ||
              sptr->filter_delay[order[cnt1]] >
               sptr->filter_delay[order[cnt2]]))
         {
            tmp = order[cnt1];
            order[cnt1] = order[cnt2];
            order[cnt2] = tmp;
         }
   if (sptr->filter_delay[order[0]] == 0)  /* Best delay is 0? */
   {
      sptr->delay = 0;
      sptr->offset.qd_hi = sptr->offset.qd_lo = 0;
      sptr->dispersion = bt_peermaxdisp;
   }
   else
   {
      sptr->delay = sptr->filter_delay[order[0]];   /* Best delay */
      sptr->offset = sptr->filter_offset[order[0]]; /* It's offset */
      sptr->dispersion = bt_zero;
      for (cnt1 = 1; cnt1 < sys_samples; cnt1++)    /* Missing samples? */
      {
         if (sptr->filter_delay[order[cnt1]] == 0)
            bt_disp = bt_peermaxdisp;
         else
         {
            bt_disp = Bt_Abs(Bt_Sub(sptr->filter_offset[order[cnt1]],
                                     sptr->filter_offset[order[0]]));
            if (Bt_gt(bt_disp, bt_peermaxdisp))
               bt_disp = bt_peermaxdisp;
         }
         sptr->dispersion = Bt_Add(sptr->dispersion,
                                    Bt_ShiftR(bt_disp, cnt1));
      }
   }
}

server *Clock_Select(void)
/*
Description:  The clock selection function.
*/
{
   int done;
   int num_good;
   int cnt1;
   int cnt2;
   int cnt3;
   server *sptr;
   server *sys_server;
   server *server_list[NTP_MAXCLOCK];
   bintime server_badness[NTP_MAXCLOCK];
   bintime bt_disp;
   bintime bt_lthresh;
   bintime bt_tmp;

   num_good = 0;
   for (cnt1 = 0; cnt1 < sys_numservers; cnt1++)
   {
      sptr = srv_lst[cnt1];
      if ((sptr->delay != 0) &&
          (sptr->stratum <= NTP_INFIN) &&
          (sptr->delay <= NTP_MAXWGT) &&
          (sptr->leap != LEAP_NOTINSYNC) &&
          (Bt_ge(sptr->org, sptr->reftime)) &&
          Bt_lt(Bt_Sub(sptr->org, sptr->reftime), bt_maxage))
      {
         bt_disp = Bt_Add(sptr->dispersion, sptr->dispersion);
         cnt2 = 0;
         while (cnt2 < num_good &&
                sptr->stratum > server_list[cnt2]->stratum)
            cnt2++;
         while (cnt2 < num_good &&
                sptr->stratum >= server_list[cnt2]->stratum &&
                Bt_ge(bt_disp, server_badness[cnt2]))
            cnt2++;
         if (cnt2 < NTP_MAXLIST)
         {
            for (cnt3 = num_good; cnt3 > cnt2; cnt3--)
               if (cnt3 < NTP_MAXLIST)
               {
                  server_list[cnt3] = server_list[cnt3 - 1];
                  server_badness[cnt3] = server_badness[cnt3 - 1];
               }
            server_list[cnt2] = sptr;
            server_badness[cnt2] = bt_disp;
            if (num_good < NTP_MAXLIST) num_good++;
         }
      }
   }
   cnt2 = 0;
   cnt3 = 0;
   while (cnt2 < num_good - 1 && cnt3 <= 2)
   {
      if (server_list[cnt2 + 1]->stratum > server_list[cnt2]->stratum)
         cnt3++;
      cnt2++;
   }
   if (cnt3 >= 2) num_good = --cnt2;
   if (num_good == 0)
      sys_server = NULL;
   else
      if (num_good == 1)
         sys_server = server_list[0];
      else
      {
         for (cnt2 = 0; cnt2 < num_good - 1; cnt2++)
            for (cnt3 = cnt2 + 1; cnt3 < num_good; cnt3++)
               if ((server_list[cnt2]->stratum >= server_list[cnt3]->stratum) &&
                   (server_list[cnt2]->delay >= server_list[cnt3]->delay))
               {
                  sptr = server_list[cnt2];
                  server_list[cnt2] = server_list[cnt3];
                  server_list[cnt3] = sptr;
               }
         bt_lthresh.qd_hi = 0;
         bt_lthresh.qd_lo = (HNSPS >> -NTPDATE_PRECISION) + NTP_MAXSKW;
         done = FALSE;
         while (num_good > 1 && !done)
         {
            for (cnt1 = 0; cnt1 < num_good; cnt1++)
            {
               server_badness[cnt1].qd_hi = server_badness[cnt1].qd_lo = 0;
               for (cnt3 = 0; cnt3 < num_good; cnt3++)
                  if (cnt3 != cnt1)
                  {
                     bt_disp = Bt_Abs(Bt_Sub(server_list[cnt3]->offset,
                                              server_list[cnt1]->offset));
                     for (cnt2 = 0; cnt2 < cnt3; cnt2++)
                        bt_disp = Bt_Add(Bt_ShiftR(bt_disp, 1),
                                          Bt_ShiftR(bt_disp, 2));
                     server_badness[cnt1] = Bt_Add(bt_disp,
                                                    server_badness[cnt1]);
                  }
            }
            cnt2 = 0;
            cnt1 = server_list[0]->precision;
            for (cnt3 = 1; cnt3 < num_good; cnt3++)
            {
               if (Bt_ge(server_badness[cnt3], server_badness[cnt2]))
                  cnt2 = cnt3;
               if (cnt1 > server_list[cnt3]->precision);
                  cnt1 = server_list[cnt3]->precision;
            }
            bt_tmp.qd_hi = 0;
            bt_tmp.qd_lo = HNSPS >> -cnt1;
            if (Bt_ge(server_badness[cnt2], Bt_Add(bt_lthresh, bt_tmp)))
            {
               for (cnt3 = cnt2 + 1; cnt3 < num_good; cnt3++)
                  server_list[cnt3 - 1] = server_list[cnt3];
               num_good--;
            }
            else done = TRUE;
         }
         sys_server = server_list[0];   /* The best server */
      }
   return(sys_server);
}

void Clock_Adjust(void)
/*
Description:  Print information about the servers and adjust the clock
as requested.  The adjustment is done in one fell swoop.
*/
{
   int cnt;                      /* Count through the servers */
   server *sptr;                 /* Points to a server */
   bintime bt_offset;            /* Adjust system clock by this amount */
   char time_str[MAX_TIMESTR];   /* Holds time as string */

   for (cnt = 0; cnt < sys_numservers; cnt++)
   {
      Clock_Filter(srv_lst[cnt]);   /* Find best delays and offsets */
      if (debug || simple_query) Print_Server(srv_lst[cnt]);
   }
   sptr = Clock_Select();        /* Select best server */
   if (sptr == NULL)
      printf("No server suitable for synchronization found.\n");
   else
   {
      bt_offset = sptr->offset;
      if (Dst(Bt_Add(Get_Time(), bt_offset)))   /* Daylight saving time? */
         bt_offset = Bt_Add(bt_offset, bt_dstadj);
      if (!simple_query)
         Set_Time(Bt_Add(Get_Time(), bt_offset));
      printf("Selected server is %s (%s) offset %s\n", sptr->hostname,
         inet_ntoa(sptr->s_addr.addr), Bt2Str(bt_offset, time_str));
   }
}

bintime Get_Time(void)
/*
Description:  Return the current system time.
*/
{
   bintime bt_time;
   sys$gettim(&bt_time);
   return(bt_time);
}

void Set_Time(bintime bt_time)
/*
Description:  Set the system time.
*/
{
   u_long status;    /* Return status from the system call */

   status = sys$setime(&bt_time);
   if (status != SS$_NORMAL)
   {
      printf("Unable to set the system time.\n");
      Print_Error(status);
   }
}

bintime FSecs2DBT(float frac_secs)
/*
Description:  Convert fractional seconds to a delta binary time.
*/
{
   u_long cvt_opt = LIB$K_DELTA_SECONDS_F;  /* Conversion opertion */
   bintime bt_result;   /* The binary time result */

   lib$cvtf_to_internal_time(&cvt_opt, &frac_secs, &bt_result);
   return(bt_result);
}

void *Get_ZMem(int num, int bytes)
/*
Description:  Get zeroed memory.  This function returns a pointer to the
number or requested bytes or prints a warning message and exits.
*/
{
   void *memptr;  /* Pointer to the memory */

   memptr = calloc(num, bytes);
   if (memptr == NULL)
   {
      printf("Cannot allocate requested memory.\n");
      exit(1);
   }
   return(memptr);
}

bintime Str2Bt(char *time_str)
/*
Description:  This function takes the time string and converts it to the
binary time equivalent.
*/
{
   dscr_s time_dscr;    /* Time string descriptor */
   bintime bt_result;   /* Binary time result */

   time_dscr.dsc$w_length = strlen(time_str);
   time_dscr.dsc$b_dtype = DSC$K_DTYPE_T;
   time_dscr.dsc$b_class = DSC$K_CLASS_S;
   time_dscr.dsc$a_pointer = time_str;

   sys$bintim(&time_dscr, &bt_result);
   return(bt_result);
}

char *Bt2ASC(bintime bt_time, char *time_str, int str_len)
/*
Description:  Accept a binary time, an empty string and the length of
the string, and return the string time and date equivalent of the binary
time.
*/
{
   dscr_s time_dscr;  /* Time string descriptor */
   u_short time_len;  /* Length of the time string */

   time_dscr.dsc$w_length = str_len;
   time_dscr.dsc$b_dtype = DSC$K_DTYPE_T;
   time_dscr.dsc$b_class = DSC$K_CLASS_S;
   time_dscr.dsc$a_pointer = time_str;

   sys$asctim(&time_len, &time_dscr, &bt_time, 0);
   time_str[time_len] = '\0'; /* Don't forget that null character */
   return(time_str);
}

int Bt_gt(bintime bt_time1, bintime bt_time2)
/*
Description:  Return (bt_time1 > bt_time2).
*/
{
   return((bt_time1.qd_hi > bt_time2.qd_hi) ||
          ((bt_time1.qd_hi == bt_time2.qd_hi) &&
           (bt_time1.qd_lo > bt_time2.qd_lo)));
}

int Bt_lt(bintime bt_time1, bintime bt_time2)
/*
Description:  Return (bt_time1 < bt_time2).
*/
{
   return((bt_time1.qd_hi < bt_time2.qd_hi) ||
          ((bt_time1.qd_hi == bt_time2.qd_hi) &&
           (bt_time1.qd_lo < bt_time2.qd_lo)));
}

int Bt_ge(bintime bt_time1, bintime bt_time2)
/*
Description:  Return (bt_time1 >=  bt_time2).
*/
{
   return((bt_time1.qd_hi > bt_time2.qd_hi) ||
          ((bt_time1.qd_hi == bt_time2.qd_hi) &&
           (bt_time1.qd_lo >= bt_time2.qd_lo)));
}

int Bt_eq(bintime bt_time1, bintime bt_time2)
/*
Description:  Return (bt_time1 == bt_time2).
*/
{
   return((bt_time1.qd_hi == bt_time2.qd_hi) &&
          (bt_time1.qd_lo == bt_time2.qd_lo));
}

void Bt2SecsRem(bintime bt_time, long *seconds, long *remainder)
/*
Description:  Convert a binary time to seconds and a remainder (in
hundred nanoseconds per second).  This function does not handle a
negative binary time.
*/
{
   long nsps;        /* Nanoseconds per second */

   nsps = NSPS;
   lib$ediv(&nsps, &bt_time, seconds, remainder);
   *seconds = (*seconds * (NSPS / HNSPS)) + *remainder / HNSPS;
   *remainder = *remainder % HNSPS;
}

l_fp Bt2Lfp(bintime bt_time)
/*
Description:  Convert a binary time to a long fixed point (ntp type). 
Care must be taken.  Since 1995, the number of seconds in binary time
will no longer fit into l_fp.  One solution is to subtract bt_utcoffset
so that the resulting value is in line with the ntp protocol expects.
*/
{
   l_fp lfp_time;    /* A long fixed point time */
   long nsps;        /* Nanoseconds per second */
   long quotient;    /* Quotient from the division */
   long remainder;   /* The remainder from the division */
   bintime bt_frac;  /* Fraction of time in binary */

   Bt2SecsRem(bt_time, &lfp_time.l_ui, &remainder);
   bt_frac.qd_hi = remainder;    /* << 32 */
   bt_frac.qd_lo = 0;
   Bt2SecsRem(bt_frac, &lfp_time.l_uf, &remainder);
   lfp_time.l_uf += remainder / (HNSPS / 2);    /* Round up */
   return(lfp_time);
}

bintime MultHigh(u_long num1, u_long num2)
/*
Description:  Mutiply by the high bit only and return the value as a
binary time.
*/
{
   bintime bt_result;      /* The result of the multiply */

   bt_result.qd_hi = bt_result.qd_lo = 0;
   if (num1 & HIGHBIT_MASK)
   {                             /* num2 << 31 */
      bt_result.qd_hi = num2;
      bt_result.qd_lo = (bt_result.qd_hi & 1) << 31;
      bt_result.qd_hi >>= 1;
   }
   return(bt_result);
}

bintime EMultUl(u_long multiplier, u_long multiplicand)
/*
Description:  Do an extended multiply with unsigned long numbers and
return the result as a binary time.
*/
{
   bintime bt_result;   /* The result of the multiply */
   bintime bt_high1;    /* Value of multiplying by high order bit */
   bintime bt_high2;    /* Value of multiplying by high order bit */
   long addend;         /* The addend of an extended multiply */

   addend = 0;
   bt_high1 = MultHigh(multiplier, multiplicand);
   multiplier &= MASK_HIGHBIT;
   bt_high2 = MultHigh(multiplicand, multiplier);
   multiplicand &= MASK_HIGHBIT;
   lib$emul(&multiplier, &multiplicand, &addend, &bt_result);
   lib$addx(&bt_high1, &bt_result, &bt_result, 0);
   lib$addx(&bt_high2, &bt_result, &bt_result, 0);
   return(bt_result);
}

bintime EMultL(long multiplier, long multiplicand)
/*
Description:  Do an extended multiply with long numbers and return the
result as a binary time.
*/
{
   long addend;         /* Addend for the extended multiply */
   bintime bt_result;   /* Resut of the extended multiply */

   addend = 0;
   lib$emul(&multiplier, &multiplicand, &addend, &bt_result);
   return(bt_result);
}

bintime Lfp2Bt(l_fp lfp_time)
/*
Description:  Convert a long fixed point (ntp type) value to a binary
time and return the binary time.
*/
{
   bintime bt_result;   /* The final result */
   bintime bt_frac;     /* Fractional part of time */
   bintime bt_int;      /* Integer portion of the time */
   bintime bt_round;    /* Number to round on */
   u_long frac_secs;    /* Fractional seconds */
   u_long multiplier;   /* Multiplier for time */

   bt_round.qd_hi = 0;
   bt_round.qd_lo = HIGHBIT_MASK;
   multiplier = HNSPS;
   bt_frac = Bt_Add(EMultUl(multiplier, lfp_time.l_uf), bt_round);
   bt_frac.qd_lo = bt_frac.qd_hi;   /* >> 32 */
   bt_frac.qd_hi = 0;
   bt_int = EMultUl(multiplier, lfp_time.l_ui);
   lib$addx(&bt_frac, &bt_int, &bt_result, 0);
   return(bt_result);
}

bintime Bt_Sub(bintime bt_time1, bintime bt_time2)
/*
Description:  Return (bt_time1 - bt_time2).
*/
{
   bintime bt_result;   /* Result of the subtract */

   lib$subx(&bt_time1, &bt_time2, &bt_result, 0);
   return(bt_result);
}

bintime Bt_Add(bintime bt_time1, bintime bt_time2)
/*
Description:  Return (bt_time1 + bt_time2).
*/
{
   bintime bt_result;   /* Result of the addition */

   lib$addx(&bt_time1, &bt_time2, &bt_result, 0);
   return(bt_result);
}

bintime Bt_Abs(bintime bt_time)
/*
Description:  Return abs(bt_time).
*/
{
   if (Bt_lt(bt_time, bt_zero))
      bt_time = Bt_Sub(bt_zero, bt_time);
   return(bt_time);
}

bintime Bt_ShiftR(bintime bt_time, int bits)
/*
Description:  Return (bt_time1 >> bits).
*/
{
   if (bits < 32)
   {
      bt_time.qd_lo >>= bits;
      bt_time.qd_lo |= (bt_time.qd_hi & ((2 << bits) - 1)) << (32 - bits);
      bt_time.qd_hi >>= bits;
   }
   else
   {
      bt_time.qd_lo = bt_time.qd_hi >> (bits - 32);
      bt_time.qd_hi >>= 32;
   }
   return(bt_time);
}

l_fp ntohl_fp(l_fp lfp_time)
/*
Description:  Accept a long fixed point time, convert it from network
to host order long fixed point and return it.
*/
{
   lfp_time.l_ui = ntohl(lfp_time.l_ui);
   lfp_time.l_uf = ntohl(lfp_time.l_uf);
   return(lfp_time);
}

l_fp htonl_fp(l_fp lfp_time)
/*
Description:  Convert a long fixed point time from host to network
byte order and return it.
*/
{
   lfp_time.l_ui = htonl(lfp_time.l_ui);
   lfp_time.l_uf = htonl(lfp_time.l_uf);
   return(lfp_time);
}

int Lfp_eq(l_fp lfp_time1, l_fp lfp_time2)
/*
Description:  Return (lfp_time1 == lfp_time2).
*/
{
   return((lfp_time1.l_ui == lfp_time2.l_ui) &&
          (lfp_time1.l_uf == lfp_time2.l_uf));
}

int Lfp_ge(l_fp lfp_time1, l_fp lfp_time2)
/*
Description:  Return (lfp_time1 >= lfp_time2).
*/
{
   return((lfp_time1.l_ui > lfp_time2.l_ui) ||
          ((lfp_time1.l_ui == lfp_time2.l_ui) &&
           (lfp_time1.l_uf >= lfp_time2.l_uf)));
}

int Dst(bintime bt_time)
{
/*
Description:  This function returns true or false depending on whether
the given time is during the Daylight Savings Time period.
  Convert the given binary time into numeric time.  Use the
same year and find the binary time of the beginning of DST.  If a day of
the week was specified, adjust the beginning of DST to that particular
day.  Do the same thing for the end of DST.
   Convert the begin DST, end DST, and time to seconds of the year. 
Then return the result of whether the time falls into the period of DST
or not.
   The day-of-week adjustment requires the dow number to be 0 to 6,
Sunday through Saturday.  VMS dow is 1 to 7, Monday through Sunday, and
the defines for DST are 1 to 7, Sunday through Saturday.  These numbers
are modified before being given to the dow adjustment function.
*/

   numtim timbuf;    /* Buffer for storing "numeric time" */
   bintime bt_bdst;  /* Begin of Daylight Savings Time in binary time */
   bintime bt_edst;  /* End of Daylight Savings Time in binary time */
   u_long day_num;   /* Day of week number */
   u_long cvt_opt = LIB$K_SECOND_OF_YEAR;
   u_long bdst_secs; /* Seconds in year of begin of DST */
   u_long edst_secs; /* Seconds in year of end of DST */
   u_long time_secs; /* Seconds in year of time */

   if (DSTON_DOW < 0 || DSTOFF_DOW < 0) return(0); /* Check turned off */

   sys$numtim(&timbuf, &bt_time);      /* Convert bin time to numeric time */
   timbuf.mon = DSTON_MON;             /* Insert begin of DST info */
   timbuf.day = DSTON_DOM;
   timbuf.hr = DSTON_TIM / 100;
   timbuf.min = DSTON_TIM % 100;
   timbuf.sec = 0;
   timbuf.hun = 0;
   lib$cvt_vectim(&timbuf, &bt_bdst);  /* Get begin of DST in binary time */
   if (DSTON_DOW)                      /* Begin on specific day of week? */
   {
      lib$day_of_week(&bt_bdst, &day_num);
      Dow_Adj(day_num % 7, DSTON_DOW - 1, DSTON_CMP, &bt_bdst);
   }

   timbuf.mon = DSTOFF_MON;            /* Insert end of DST info */
   timbuf.day = DSTOFF_DOM;
   timbuf.hr = DSTOFF_TIM / 100;
   timbuf.min = DSTOFF_TIM % 100;
   timbuf.sec = 0;
   timbuf.hun = 0;
   lib$cvt_vectim(&timbuf, &bt_edst);  /* Get end of DST in binary time */
   if (DSTOFF_DOW)                     /* End on specific day of week? */
   {
      lib$day_of_week(&bt_edst, &day_num);
      Dow_Adj(day_num % 7, DSTOFF_DOW - 1, DSTOFF_CMP, &bt_edst);
   }
                                       /* Get seconds in year for each */
   lib$cvt_from_internal_time(&cvt_opt, &bdst_secs, &bt_bdst);
   lib$cvt_from_internal_time(&cvt_opt, &edst_secs, &bt_edst);
   lib$cvt_from_internal_time(&cvt_opt, &time_secs, &bt_time);

   return(bdst_secs <= time_secs && time_secs < edst_secs);
}

void Dow_Adj(int cur_dow, int new_dow, int cmp, bintime *bt_dst)
{
/*
Description:  This function calculates the number of days to adjust a
time in order for the current day of week to be the new day of week
based on a specified comparison.
   This allows a person to find the first new day of week before the
current day, before or on the current day, after the current day, and
after or on the current day.
   If any days were calculated for adjustment, the days are converted to
binary time and then added to or subtracted from the given time.
*/

   int days;         /* Number of days to adjust time */
   bintime bt_dayadj;/* Days adjustment in binary time */

   days = 0;
   switch (cmp)
   {
      case 0:
         break;
      case 1: /* < before */
      case 2: /* <= before or on */
         days -= (cur_dow - new_dow + 7) % 7;
         if (cmp == 1 && !days) days = -7;
         break;
      case 3: /* > after */
      case 4: /* >= after or on */
         days = (new_dow - cur_dow + 7) % 7;
         if (cmp == 3 && !days) days = 7;
         break;
      default:
         break;
   }

   if (days)         /* Any days to adjust? */
   {
      bt_dayadj = EMultUl(abs(days) * DAY_SECS, HNSPS);
      if (days > 0)
         *bt_dst = Bt_Add(*bt_dst, bt_dayadj);
      else
         *bt_dst = Bt_Sub(*bt_dst, bt_dayadj);
   }
}

char *Bt2Str(bintime bt_time, char *str)
/*
Description:  Convert a binary time to its string equivalent in seconds
and fractions thereof.  We assume the string is long enough.
*/
{
   int is_neg;       /* True if the number is negative */
   long seconds;     /* Seconds */
   long remainder;   /* Remainder of seconds */

   is_neg = FALSE;
   if (bt_time.qd_hi < 0)
   {
      is_neg = TRUE;
      bt_time = Bt_Abs(bt_time);
   }
   Bt2SecsRem(bt_time, &seconds, &remainder);
   sprintf(str, "%s%d.%07d", is_neg ? "-" : "+", seconds, remainder);
   return(str);
}

void Print_Server(server *sptr)
/*
Description:  Print the information about the server.
*/
{
   int cnt;             /* Count through the filter entries */
   bintime bt_offset;   /* Offset of true time from our time */
   bintime bt_tmp;      /* Temporarily holds a time */
   char time_str[MAX_TIMESTR];   /* Holds time in string form */
   char tmp_str[5];     /* Holds bytes of reference id */

   bt_offset = bt_zero;
   if (Dst(Bt_Add(Get_Time(), sptr->offset)))  /* Daylight savings time? */
      bt_offset = bt_dstadj;
   bt_tmp.qd_hi = 0;
   bt_tmp.qd_lo = sptr->delay;
   if (!debug)    /* Short form */
   {
      printf("Server %s (%s), stratum %d\n",
             sptr->hostname,
             inet_ntoa(sptr->s_addr.addr), sptr->stratum);
      printf("offset %s, ", Bt_eq(sptr->offset, bt_zero) ?
         Bt2Str(bt_zero, time_str) :
         Bt2Str(Bt_Add(sptr->offset, bt_offset), time_str));
      printf("delay %s\n", Bt2Str(bt_tmp, time_str));
   }
   else           /* Long form */
   {
      printf("server %s (%s), port %d\n", sptr->hostname,
               inet_ntoa(sptr->s_addr.addr), ntohs(sptr->s_addr.port));
      printf("stratum %d, precision %d, leap %c%c\n",
               sptr->stratum, sptr->precision,
               sptr->leap & 0x2 ? '1' : '0',
               sptr->leap & 0x1 ? '1' : '0');
      if (sptr->stratum == 1)    /* Primary servers have names of clocks */
      {                          /* for the time source */
         tmp_str[4] = 0;
         memcpy(tmp_str, sptr->refid, 4);
         printf("refid [%s] ", tmp_str);
      }
      else
         printf("refid [%lu] ", sptr->refid);   /* Others have IP numbers */
      printf("delay %s ", Bt2Str(bt_tmp, time_str));
      printf("dispersion %s ", Bt2Str(sptr->dispersion, time_str));
      printf("offset %s\n\n", Bt_eq(sptr->offset, bt_zero) ?
         Bt2Str(bt_zero, time_str) :
         Bt2Str(Bt_Add(sptr->offset, bt_offset), time_str));
      printf("transmitted %d, in filter %d\n",
         sptr->xmtcnt, sptr->filter_nextpkt);
      printf("reference time:      %s\n",
         Bt2ASC(Bt_Add(sptr->reftime, bt_offset), time_str, MAX_TIMESTR));
      printf("originate timestamp: %s\n",
         Bt2ASC(Bt_Add(sptr->org, bt_offset), time_str, MAX_TIMESTR));
      printf("transmit timestamp:  %s\n",
         Bt2ASC(sptr->xmt, time_str, MAX_TIMESTR));
      printf("filter delay:       ");
      bt_tmp.qd_hi = 0;
      for (cnt = 0; cnt < NTP_SHIFT; cnt++)     /* Print filter delays */
      {
         bt_tmp.qd_lo = sptr->filter_delay[cnt];
         printf(" %-8.8s", Bt2Str(bt_tmp, time_str));
         if (cnt == (NTP_SHIFT >> 1) - 1) printf("\n                    ");
      }
      printf("\n");

      printf("filter offset:      ");
      for (cnt = 0; cnt < NTP_SHIFT; cnt++)     /* Print filter offsets */
      {
         printf(" %-8.8s", Bt_eq(sptr->filter_offset[cnt], bt_zero) ?
            Bt2Str(bt_zero, time_str) :
            Bt2Str(Bt_Add(sptr->filter_offset[cnt], bt_offset), time_str));
         if (cnt == (NTP_SHIFT >> 1) - 1) printf("\n                    ");
      }
      printf("\n\n");
   }
}
