/* Copyright 1989 Phil Andrews, Pittsburgh Supercomputing Center */
/* all rights reserved */
/* taken over by Phil Andrews, 1989 */
/********************************************************************/
/*                                                                  */
/* copyright 1987, 1988                                             */
/* by Joel Welling, Pittsburgh Supercomputing Center                */
/*                                                                  */
/********************************************************************/

/*

This module implements a device driver for GKS (of the Vax/VMS variety).

*/

#include descrip.h	/* VMS string-passing descriptors */
#include stdio.h	/* to write diagnostic and error messages */
#include "Sys$library:gksdefs.h"	/* GKS entry points */
#include "defs.h"	/* structures to interface to GPLOT */

/* Auxiliary descriptor definitions */

struct dsc$descriptor_a2 {
	unsigned short	dsc$w_length;
	unsigned char 	dsc$b_dtype;
	unsigned char	dsc$b_class;
	char		*dsc$a_pointer;
	char		dsc$b_scale;
	unsigned char	dsc$b_digits;
	struct	{
		unsigned			:4;
		unsigned dsc$v_fl_redim		:1;
		unsigned dsc$v_fl_column	:1;
		unsigned dsc$v_fl_coeff		:1;
		unsigned dsc$v_fl_bounds	:1;
	} dsc$b_aflags;
	unsigned char	dsc$b_dimct;
	unsigned long	dsc$l_arsize;
	char		*dsc$a_a0;
	long		dsc$l_m[2];
	struct {
		long	dsc$l_l;
		long	dsc$l_u;
	} dsc$bounds[2];
};

#define DESC_ARRAY(name,length,ptr) struct dsc$descriptor_a            \
	name={4, DSC$K_DTYPE_L, DSC$K_CLASS_A, ptr, 0, 0, {0,0,0,0,0}, \
		1, length*4}

#define DESC_ARRAY_2(name, dim1, dim2, ptr) struct dsc$descriptor_a2   \
	name = {4,DSC$K_DTYPE_L, DSC$K_CLASS_A, ptr, 0, 0, {0,0,0,1,1},\
		2, dim1*dim2*4, ptr, {dim1,dim2},{0,dim1-1,0,dim2-1}}

/* Interesting part of code begins here */

/*
The following pointers provide storage for the driver's own copies of
several important structure pointers.  The struct definitions come
from defs.h .
*/
static struct one_opt *Op;		/* Command line option array */
static struct mf_d_struct *State_c1;	/* Current CGM metafile desc. data */
static struct pic_d_struct *State_c2;	/* Current CGM picture descrip. data */
static struct control_struct *State_c3;	/* Current CGM control element data */
static struct attrib_struct *State_c5;	/* Current CGM attribute data */
static struct info_struct *Dev_info;	/* Device characteristics communicated
					via this struct */

/* The following macro is a shorthand for the 'debug' command line option */
#define debugmode Op[debug].val.i

/*
Function return code conventions:  Unless otherwise stated,
the following function return values have the following
meanings:

	0	Terrible error, die immediately
	1	Successful function execution
	2	The function has made a survivable error
	4	The function has detected but not introduced a
		survivable error
	8	No error, but for some reason the device is currently
		incapable of fulfilling the request

*/

/*
state_level starts off -1 (set by gks_setup), is incremented each time
gks_begin gets called, and is decremented each time gks_end gets called.
It may exceed zero if GPLOT is being used to composite multiple images
on the screen;  in this case the driver will be called with an extra
set of gks_begin, gks_bpage and gks_epage, gks_end calls bracketting the
full set of calls associated with the composited images (including
their gks_begin's and gks_end's).  Thus gks_bpage and gks_epage have
to check state_level to avoid doing repeat initializations.
*/
static int state_level;

/*
Coordinate model:  GKS requires real values for coordinates.  GPLOT
supplies integers based on the device information set by gks_setup.
To make the conversion, gks_setup also creates a pair of conversion 
factors, xscale and yscale.  These are set to (screen size/pixels per 
inch) in each direction by gks_setup;  they are used by the driver to
get the needed real coordinates.
*/   
static float xscale,yscale;

/* Data concerning the physical display device, set by gks_setup */
static int ws_id=1,raster_x,raster_y;
static float dev_x,dev_y;

/* Coordinate buffers, enlarged as needed by gks_getcmem */
static float *xbuf, *ybuf;
static int xbuf_sz=0, ybuf_sz=0;

/* Image buffer, enlarged as needed by gks_getimem */
static int *ibuf;
static int ibuf_sz=0;

/* Cell array row buffers are statically allocated.  Their length is: */
#define ROWSIZE 1024

/* Arrays to hold font index list and font precision list, set by gks_setup */
#define NFONTS 20
#define NPREC 3
static int font_index[NFONTS],font_prec[NPREC],num_fonts=0;

/* 
Values for character heights available.
*/
static int num_c_heights, c_h_min, c_h_max;

/* Array for line types list, set by gks_setup */
#define NLINES 6     
static int line_types[NLINES];

/* Array for marker types list, set by gks_setup */
#define NMARKERS 10
static int marker_types[NMARKERS];

/* Total number of color indices available, set by gks_setup */
static int clr_t_sz=0;

/*
	Direct color simulation variables, set by gks_fake_dc and
	gks_dc_colors.  The flag dc_ready_flag is only reset
	by gks_ctab, which puts explicit colors in the physical
	device color table but does *not* change clr_tbl, the
	direct color simulation color table.  The driver believes
	it is in direct color or indexed color mode based on whether
	the color selection mode in State_c2 is or is not non-zero 
	at the time of the call to gks_bpage.
*/
#define BEST_C_SZ 32768   /* = 32**3;  not a free parameter! */
#define f_to_b(f) ( (unsigned char) ((int)((f)*255.0) % 256 ))
#define b_to_f(b) ( (1./255.)*((float) (b)) )
#define mask_8_thru_4 248
#define pack_clr(ir,ig,ib) \
  	( ( ( (ir) & mask_8_thru_4 )<<7 ) \
	| ( ( (ig) & mask_8_thru_4 )<<2 ) \
	| ( ( (ib) & mask_8_thru_4 )>>3 ) )
static int dc_init_flag=0, dc_ready_flag=0, dc_totclrs;
static int *best_clr;
static float *clr_tbl;
                                   
/* 
	This routine finishes setting up the GKS workstation.
 */
gks_begin(comment, file_name, prog_name)
char *comment, *file_name, *prog_name;
{
	$DESCRIPTOR(title_dsc,"Dummy");

	if (debugmode)
		{
		fprintf(stderr," gks_begin: \n");
		fprintf(stderr,"    comment: <%s>\n",comment);
		fprintf(stderr,"    file name: <%s>\n",file_name);
		fprintf(stderr,"    program name: <%s>\n",prog_name);
		}
                                                  
	state_level++;
	if (state_level > 0) return(1);  /* already set up */

	/* Set up the workstation.  At the moment, workstation type and
	 * connection ID are the defaults.  If you are using the VAXstation
	 * workstation type, you can make the file name be the window title
	 * by replacing &GKS$K_CONID_DEFAULT with &title_dsc in the
	 * gks$open_ws line below.
	 */
	title_dsc.dsc$w_length= strlen(file_name);
	title_dsc.dsc$a_pointer= file_name;
	gks$open_ws(&ws_id, &GKS$K_CONID_DEFAULT, &GKS$K_WSTYPE_DEFAULT);
	gks$activate_ws(&ws_id);

	return(1);
}           
                                                        
/* 
	This routine shuts down the workstation and GKS.
*/ 
gks_end(pages_done)
int pages_done;
{
	if (debugmode) 
		fprintf(stderr," gks_end: pages done = %d\n",pages_done);

	state_level--;
	if (state_level > 0) return(1);  /* not yet done */

	gks$deactivate_ws(&ws_id);
	gks$close_ws(&ws_id);
	gks$close_gks();
	if (xbuf_sz>0) { free(xbuf); xbuf_sz= 0; };
	if (ybuf_sz>0) { free(ybuf); ybuf_sz= 0; };

	return(1);
}

/*                                   
	This routine starts a new page.  Returns 1 if properly executed.
	X and y offset are in pixels;  rotation is in floating point degrees.
	rb, gb, and bb are the background color.
*/                                                              
gks_bpage(pic_title,xoff, yoff, rotation, rb, gb, bb, pgnum, xsize, ysize)
char *pic_title;
float rotation, rb, gb, bb;
int xoff, yoff, pgnum;
int xsize, ysize;
{                   
	int rc,one= 1,zero= 0;

	if (debugmode) 
 		{
		fprintf(stderr,
			" gks_bpage:   page number %d\n",pgnum);
		fprintf(stderr,"    x, y offset, rotation = %d %d %f\n",
			xoff,yoff,rotation);
		};

	if (state_level > 0) return(1);  /* The page is already set up */

	/* Clear the workstation window  */
	gks$clear_ws(&ws_id,&one);

	/*
	Could check max color index, but will use clr_t_sz (set by
	gks_setup) instead throughout the driver.
	*/

	/* Set background color */
	gks$set_color_rep(&ws_id,&zero,&rb,&gb,&bb);

	/*
	Set proper color selection mode for this frame, skipping
	index 0 (just set) if it's direct color 
	*/
	if (State_c2->c_s_mode) 	/* Direct color */
		{ 
		if ( !dc_ready_flag ) rc= gks_dc_colors(); 
		else rc= 1;
		}
                                                     
	else		/* Indexed color - load color table */
	   	rc= gks_ctab(1, State_c1->max_c_index, State_c5->ctab);
	if (!rc) return(rc);
                                                        
	/* Reset a number of attributes to their proper values */
	rc= gks_t_colour(State_c5->text_colour.red,State_c5->text_colour.green,
			State_c5->text_colour.blue,State_c5->text_colour.ind);
	if (!rc) return(rc);
	rc= gks_t_align(State_c5->text_align.hor,State_c5->text_align.ver,
		State_c5->text_align.cont_hor,State_c5->text_align.cont_ver);
	if (!rc) return(rc);
    	rc= gks_c_height(State_c5->c_height);
	if (!rc) return(rc);
	rc= gks_t_font(State_c5->t_f_index);
	if (!rc) return(rc);
	rc= gks_mk_colour(State_c5->mk_colour.red,State_c5->mk_colour.green,
			State_c5->mk_colour.blue,State_c5->mk_colour.ind);
	if (!rc) return(rc);
	rc= gks_mk_size(State_c5->mk_size.i,State_c5->mk_size.r);
	if (!rc) return(rc);
	rc= gks_mk_type(State_c5->mk_type);
	if (!rc) return(rc);
	rc= gks_fl_colour(State_c5->fill_colour.red,State_c5->fill_colour.green,
		      	State_c5->fill_colour.blue,State_c5->fill_colour.ind);
	if (!rc) return(rc);
	rc= gks_fl_style(State_c5->int_style);
	if (!rc) return(rc);
	rc= gks_l_type(State_c5->line_type);
	if (!rc) return(rc);
	rc= gks_l_width(State_c5->line_width.i,State_c5->line_width.r);
	if (!rc) return(rc);
	rc= gks_l_colour(State_c5->line_colour.red,State_c5->line_colour.green,
			State_c5->line_colour.blue,State_c5->line_colour.ind);

	return(rc);
}                                                                     

/* 
	This routine ends the page, clearing the screen if necessary.
*/
gks_epage(copies)
int copies;
{
	if (debugmode) fprintf(stderr," gks_epage: copies = %d\n",copies);

	/* 
	state_level might be relevant here, but isn't needed for this
	device.
	*/

	return(1);                                              
}
                                                       
/*                                                          
  	Set the text colour.
*/
gks_t_colour(r, g, b, index)
float r, g, b;
int index;
{
	int rc= 1;

	if (debugmode) 
		fprintf(stderr," gks_t_colour: r, g, b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else 
			{
			fprintf(stderr, 
		     	   " Direct text color with indexed color set!\n");
			rc= 4;
			index= 1;
			}
		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
			fprintf(stderr, 
			   " Indexed text color with direct color set!\n");
			rc= 4;
			};
		};

	if ( index < clr_t_sz ) gks$set_text_color_index(&index);
	else rc= 8;

	return(rc);
}
                
/* 
	Write text.
*/
gks_text(x, y, final, buffer)
int x, y;
enum boolean final;
char *buffer;
{
	float xloc,yloc;
	$DESCRIPTOR(txt_desc,"dummy");

	if (debugmode) 
		{
		fprintf(stderr," gks_text: x= %d, y= %d, final= %d\n",
			x,y,(int)final);
		fprintf(stderr,"      text=<%s>\n",buffer);
		}
	txt_desc.dsc$w_length= strlen(buffer);
	txt_desc.dsc$a_pointer= buffer;
	xloc= x*xscale;
	yloc= y*yscale;
	gks$text(&xloc,&yloc,&txt_desc);

	return(1);
}

/*
	Set the text alignment.
*/
gks_t_align(hor, ver, cont_hor, cont_ver)
enum hor_align hor;
enum ver_align ver;
float cont_hor, cont_ver;
{
	int ih,iv;

	if (debugmode) 
		fprintf(stderr," gks_t_align: hor= %d, ver= %d\n",hor,ver);
	if ((int)hor>3)
		{
		fprintf(stderr,
		   " Unknown or unimplemented hor = %d\n", hor);
		return(8);
		}
	if ((int)ver>5)
		{
		fprintf(stderr,
		   " Unknown or unimplemented ver = %d\n", ver);
		return(8);
		}

	gks$set_text_align(&hor,&ver);

	return(1);
}

/*
Plot a set of lines.
*/
gks_pline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	int i, *x1_ptr_cpy=x1_ptr, *y1_ptr_cpy= y1_ptr;
	float *xcpy, *ycpy;

	if (debugmode) 
		fprintf(stderr," gks_pline: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs <= 1 ) return(1);

	if ( gks_getcmem(no_pairs,no_pairs) != 1 )
		{ 
		fprintf(stderr," Error allocating memory for pline buffer.");
		return(2);
		};

	xcpy= xbuf; ycpy= ybuf;
	for (i=0; i<no_pairs; i++)
		{
		*xcpy++= xscale* *x1_ptr_cpy++;
		*ycpy++= yscale* *y1_ptr_cpy++;
		};

	gks$polyline(&no_pairs,xbuf,ybuf);

	return(1);
}

/* plot a set of lines between alternate points */
gks_dpline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	int i,two= 2;
	float x[2],y[2];

	if (debugmode) 
		fprintf(stderr," gks_dpline: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs <= 1 ) return(1);

	for (i=0; i < no_pairs ; i+=2)
		{
		x[0]= x1_ptr[i]*xscale; x[1]= x1_ptr[i+1]*xscale;
		y[0]= y1_ptr[i]*yscale; y[1]= y1_ptr[i+1]*yscale;
		gks$polyline(&two,x,y);
		};

	return(1);
}

/* 
	Set the marker colour.
*/
gks_mk_colour(r, g, b, index)
float r, g, b;
int index;
{
	int rc= 1;

	if (debugmode) 
		fprintf(stderr," gks_mk_colour: r, g, b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
	    	{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else 
			{
			fprintf(stderr, 
			   " Direct marker color with indexed color set!\n");
			rc= 4;
		       	index= 1;
			}
      		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
			fprintf(stderr, 
			   " Indexed marker color with direct color set!\n");
			rc= 4;
			};
		};

	if ( index < clr_t_sz ) gks$set_pmark_color_index(&index);
	else rc= 8;

	return(rc);
}

/*
	Set marker size.
*/
gks_mk_size(mk_a_size,mk_s_size)
int mk_a_size;
float mk_s_size;
{
	if (debugmode) fprintf(stderr," gks_mk_size: size scaled to %f\n",
				mk_s_size);

	gks$set_pmark_size(&mk_s_size);

	return(1);	
}

/*
	Set marker type.
*/
gks_mk_type(marker)
int marker;
{                                              
	int type;

	if (debugmode) fprintf(stderr,
			" gks_mk_type: marker type set to %c\n",marker);

	type = marker;
	gks$set_pmark_type(&type);

	return(1);
}

/* 
	Put up a series of markers.
*/
gks_pmarker(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	int i, *x1_ptr_cpy=x1_ptr, *y1_ptr_cpy=y1_ptr;
	float *xcpy, *ycpy;

	if (debugmode) 
		fprintf(stderr," gks_pmarker: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs < 1 ) return(1);

	if ( gks_getcmem(no_pairs,no_pairs) != 1 )
		{ 
		fprintf(stderr," Error allocating memory for pmarker buffer.");
		return(2);
		};

	xcpy= xbuf;
	ycpy= ybuf;
	for (i=0; i<no_pairs; i++)
		{
		*xcpy++= xscale* *x1_ptr_cpy++;
		*ycpy++= yscale* *y1_ptr_cpy++;
	    	};
	gks$polymarker(&no_pairs,xbuf,ybuf);

	return(1);
}

/* 
      	Set the text font. 
*/
gks_t_font(index)
int index;
{
	int one= 1;

	if (debugmode) fprintf(stderr," gks_t_font:  font index= %d \n",
		index);

	if ( (index<=num_fonts) && (index>0) )
		{
		gks$set_text_fontprec(&font_index[index-1],&one);
		}
	else
		fprintf(stderr," Font index %d out of range; ignored.\n",index);

	return(1);
}

/*        
  	Set character height, and set Dev_info->c_height and c_width to
	the values which result.
*/                
gks_c_height(t_a_height)
int t_a_height;
{
	float height;

	if (debugmode) fprintf(stderr,
	   	" gks_c_height: character height set to %d pixels\n",
		t_a_height);

	height= t_a_height*yscale;

	gks$set_text_height(&height);

	/*  
	Can't actually check character height and width now in effect,
	but a good guess is the next height down from the one requested
	(if possible).
	*/
	if ( t_a_height < c_h_min ) Dev_info->c_height= c_h_min;
	else if ( t_a_height > c_h_max ) Dev_info->c_height= c_h_max;
	else if ( num_c_heights == 0 ) Dev_info->c_height= t_a_height;
	else Dev_info->c_height= 
		(num_c_heights*(t_a_height - c_h_min))
		/(c_h_max - c_h_min)
		+ c_h_min;
	/* Assume square characters */
	Dev_info->c_width= Dev_info->c_height;

	return(1);
}

/*
	Set filled area interior style.
*/
gks_fl_style(style)
enum is_enum style;
{
	int istyle;

	if (debugmode) fprintf(stderr,
			" gks_fl_style: interior style set to style %d\n",
			(int)style);

	switch( (int)style )
		{
		case (int)hollow:	istyle= 0; break;
		case (int)solid_i:	istyle= 1; break;
		case (int)pattern:	istyle= 2; break;
		case (int)hatch:	istyle= 3; break;
		case (int)empty:	istyle= 0; break;
		default:		
			fprintf(stderr,
	"Unknown or unsupported interior style; using hollow.\n");
			(void) gks_fl_style(hollow);
			return(2);
		};

	gks$set_fill_int_style(&istyle);

	return(1);
}

/* 
	Set polygon fill colour.
*/
gks_fl_colour(r, g, b, index)
float r, g, b;                     
int index;
{
	int rc= 1;

	if (debugmode) 
		fprintf(stderr," gks_fl_colour: r,g,b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else 
			{
			fprintf(stderr, 
			   " Direct fill color with indexed color set!\n");
			rc= 4;
			index= 1;
			}
		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
			fprintf(stderr, 
			   " Indexed fill color with direct color set!\n");
			rc= 4;
			};
		};

	if ( index < clr_t_sz ) gks$set_fill_color_index(&index);
	else rc= 8;

	return(rc);
}

/* 
	Draw a polygon.
*/
gks_pgon(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	int i,rc,*x1_ptr_cpy=x1_ptr,*y1_ptr_cpy=y1_ptr;
	float *xcpy, *ycpy;

	if (debugmode) 
		fprintf(stderr," gks_pgon: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs <= 1 ) return(1);

	if ( gks_getcmem( no_pairs+1, no_pairs+1 ) != 1 )
		{ 
		fprintf(stderr," Error allocating memory for pgon buffer.");
		return(2);
		};

	xcpy= xbuf;
	ycpy= ybuf;
	for (i=0; i<no_pairs; i++)
		{
		*xcpy++= xscale* *x1_ptr_cpy++;
		*ycpy++= yscale* *y1_ptr_cpy++;
		};
	/* 
	Make another copy of the initial points, to close the polyline
	in case edge visibility is set.
	*/
	*xcpy= *xbuf;
	*ycpy= *ybuf;

	gks$fill_area(&no_pairs,xbuf,ybuf);

	/* Draw edges if appropriate */
	if (State_c5->edge_vis)
		{
		rc= gks_l_type(State_c5->edge_type);
		if (!rc) return(rc);
		rc= gks_l_width(State_c5->edge_width.i,State_c5->edge_width.r);
		if (!rc) return(rc);
	     	rc= gks_l_colour(State_c5->edge_colour.red,
			State_c5->edge_colour.green,
			State_c5->edge_colour.blue,
			State_c5->edge_colour.ind);
		if (!rc) return(rc);

		no_pairs= no_pairs+1;
		gks$polyline(&no_pairs, xbuf, ybuf);

		rc= gks_l_type(State_c5->line_type);
		if (!rc) return(rc);
		rc= gks_l_width(State_c5->line_width.i,State_c5->line_width.r);
		if (!rc) return(rc);
	     	rc= gks_l_colour(State_c5->line_colour.red,
			State_c5->line_colour.green,
			State_c5->line_colour.blue,
			State_c5->line_colour.ind);
		if (!rc) return(rc);
		};

	return(1);

}

/*                   
	Set the line type. 
*/              
gks_l_type(l_type)
enum line_enum l_type;
{
	int itype;

	if (debugmode) fprintf(stderr," gks_l_type: type= %d.\n",l_type);

	switch (l_type) 
		{
		case solid_l:	itype= 1; break;
		case dash:	itype= 2; break;
		case dot_l:   	itype= 3; break;
		case dash_dot:	itype= 4; break;
                                
		default:
			fprintf(stderr, 
	   "Unknown or unsupported line type [%d]; using solid lines.\n", 
				(int)l_type);
			(void) gks_l_type(1);
			return(2);
	}

	gks$set_pline_linetype(&itype);

	return(1);
}

/* 
Set the line width. 
*/
gks_l_width(l_a_width,l_s_width)
int l_a_width;
float l_s_width;
{
	if (debugmode) fprintf(stderr," gks_l_width: width scaled to %f\n",
		l_s_width);

	gks$set_pline_linewidth(&l_s_width);

	return(1);
}

/*
	Set the line colour.
*/
gks_l_colour(r, g, b, index)
float r, g, b;
int index;
{
	int rc= 1;                       
                                                                
	if (debugmode) 
		fprintf(stderr," gks_l_colour: r, g, b, index= %f %f %f %d\n", 
		r,g,b,index);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else                                 
			{
			fprintf(stderr, 
			   " Direct line color with indexed color set!\n");
			rc= 4;
			index= 1;
			}
     		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
	       		fprintf(stderr, 
			   " Indexed line color with direct color set!\n");
			rc= 4;
	    		};
		};

	if ( index < clr_t_sz ) gks$set_pline_color_index(&index);
	else rc= 8;

	return(rc);
}

/* 
	Draw a cell array.
*/
gks_carray(p, q, r, nx, ny, prec, image_ptr, mode, no_bytes)
int p[2], q[2], r[2], nx, ny, prec, mode;
unsigned char *image_ptr;
long int no_bytes;
{
	int imagesize,rc,rdummy[2],zero= 0;
	float px,py,qx,qy,rx,ry;
    	DESC_ARRAY_2(image_desc,0,0,NULL);
	extern cla_p_fb();

	if (debugmode) 
		{
		fprintf(stderr,
			" gks_carray: nx= %d, ny= %d, prec= %d, mode= %d\n",
			nx,ny,prec,mode);
		fprintf(stderr,"      p= [%d,%d], q= [%d,%d], r= [%d,%d].\n",
			p[0],p[1],q[0],q[1],r[0],r[1]);
		};

	/* 
	For skewed or rotated cell arrays, use the polygon fallback
	routine.
	*/
	if ( (p[1] != r[1]) || (q[0] != r[0]) )
		{
	       	cla_p_fb( p, q, r, nx, ny, prec, image_ptr, mode, no_bytes );
		return(1);
		};

	/* Check row buffer space availability */
	if ( nx > ROWSIZE )
		{
		fprintf(stderr,
		  " Cell array too wide for internal buffer; call ignored.\n");
		return(8);
		};

	/* Make sure buffer memory is available */
	imagesize= nx*ny;
	if ( gks_getimem(imagesize) != 1 )
		{ 
		fprintf(stderr," Error allocating memory for image buffer.");
		return(2);
		};
	
	/* Set up coordinates */
	px= xscale*p[0]; py= yscale*p[1];
	qx= xscale*q[0]; qy= yscale*q[1];

	/* Copy the image into the integer image buffer */
	if (State_c2->c_s_mode)		/* Direct color */    
		{
		if (dc_ready_flag)
			gks_cdca(nx,ny,prec,mode,image_ptr);
		else 
			{
			fprintf(stderr, 
			  " Direct color cell array with indexed color set!\n");
			return(4);
			}
	    	}

	else		/* Indexed color */
		{
	     	if (!dc_ready_flag)
			gks_cica(nx,ny,prec,mode,image_ptr);
		else 
			{
			fprintf(stderr, 
			  " Indexed color cell array with direct color set!\n");
			return(4);
			}
	    	}

	/* Fix up the array descriptor */
	image_desc.dsc$a_pointer= ibuf;  /*Needed because ibuf changes*/
	image_desc.dsc$l_arsize= 4*nx*ny;
	image_desc.dsc$l_m[0]= nx;
	image_desc.dsc$l_m[1]= ny;
	image_desc.dsc$bounds[0].dsc$l_u= nx-1;
	image_desc.dsc$bounds[1].dsc$l_u= ny-1;

	/* Draw the cell array */
	gks$cell_array(&px,&py,&qx,&qy,&zero,&zero,&nx,&ny,&image_desc);

	return(1);
}

/*                                            
	This routine implements the copying of an indexed color cell
	array into the (previously allocated) image memory buffer.
	Return 1 if successful.  The acronym is 'gks_copy_indexed_
	cell_array'.
*/
gks_cica(nx,ny,prec,mode,image_ptr)
int nx,ny,prec,mode;
unsigned char *image_ptr;
{
	int i,j,*cellptr;
	extern unsigned char *cla_i_row();
	static int rowbuf[ROWSIZE];  /* buffer for a row of image data */

	if (debugmode) 
		fprintf(stderr,
			" gks_cica: nx= %d, ny= %d, prec= %d, mode= %d\n",
			nx,ny,prec,mode);

	for (j=0; j<ny; j++)
		{ 
  		image_ptr= cla_i_row(image_ptr, nx, rowbuf, prec, mode);
		cellptr= ibuf + j;
		for (i=0; i<nx; i++) 
			{
			*cellptr= (rowbuf[i]>=0 && rowbuf[i]<clr_t_sz ) ? 
					rowbuf[i] : 0;
			cellptr= cellptr + ny;
			};
		};

	return(1);
}

/*                
	This routine implements the copying of a direct color cell
	array into the (previously allocated) image memory buffer.
	Return 1 if successful.  The acronym is 'gks_copy_direct_
	cell_array'.
*/
gks_cdca(nx,ny,prec,mode,image_ptr)
int nx,ny,prec,mode;
unsigned char *image_ptr;
{
	int i, j, pixel, *cellptr;
	extern unsigned char *cla_dc_row();
	static float redbuf[ROWSIZE],greenbuf[ROWSIZE],bluebuf[ROWSIZE];
		/* buffers for a row of image data */

	if (debugmode) 
		fprintf(stderr,
			" gks_cdca: nx= %d, ny= %d, prec= %d, mode= %d\n",
			nx,ny,prec,mode);

	for (j=0; j<ny; j++)
		{ 
		image_ptr= cla_dc_row(image_ptr,nx,redbuf,greenbuf,
			bluebuf,prec,mode);
		cellptr= ibuf + j;
		for (i=0; i<nx; i++) 
			{
     			pixel= best_clr[ pack_clr(
				f_to_b(redbuf[i]),
				f_to_b(greenbuf[i]),
				f_to_b(bluebuf[i])) ];
			*cellptr= ( pixel>=0 && pixel<clr_t_sz ) ? pixel : 0;
			cellptr= cellptr + ny;
			};
		};

	return(1);
}
                     
/* 
	Get a colour table update here.  Return 1 if successful.
*/
gks_ctab(beg_index, no_entries, pctab)
int beg_index, 	/* beginning index */
no_entries; 	/* number of entries to add starting at beg_index */
float *pctab;	/* direct colour array, *(pctab + i*3) is the red
		   entry for the i'th index, followed by g and b */
{
       	int loopmax,index;
	float *cptr;

	if (debugmode)
		fprintf(stderr,
		   " gks_ctab: %d entries with first index= %d\n",
			no_entries,beg_index);

	cptr= pctab+3*beg_index;
	loopmax= beg_index+no_entries;
	if ( loopmax > clr_t_sz ) loopmax= clr_t_sz;
	for (index=beg_index; index<loopmax; index++)
		{
		gks$set_color_rep(&ws_id,&index,cptr,cptr+1,cptr+2);
		cptr+=3;
		};

	/* A side effect of this routine is that it leaves GKS unready
		to simulate direct color;  set flag saying so. */
	dc_ready_flag= 0;

	return(1);
}

/*
	This function checks that sufficient memory is available for
	coordinate storage, allocating more as needed.  Returns 1 if
	successful.
*/	
gks_getcmem(nx,ny)
int nx,ny;
{
	char *malloc();

	if (nx>xbuf_sz)
		{
		if (debugmode) fprintf(stderr,
		   " gks_getcmem: increasing x coord mem to %d floats.\n",nx);
		if (xbuf_sz>0) { free(xbuf); xbuf_sz= 0; };
		if ( (xbuf= (float *)malloc(nx*sizeof(float))) == 0 )
			{ 
			fprintf(stderr,"Unable to allocate x memory.\n");
		 	return(2);
			};
		xbuf_sz= nx;
		};
	if (ny>ybuf_sz)
		{
		if (debugmode) fprintf(stderr,
		   " gks_getcmem: increasing y coord mem to %d floats.\n",ny);
		if (ybuf_sz>0) { free(ybuf); ybuf_sz= 0; };
		if ( (ybuf= (float *)malloc(ny*sizeof(float))) == 0 )
			{ 
			fprintf(stderr,"Unable to allocate y memory.\n");
			return(2);
			};
		ybuf_sz= ny;
		};
	return(1);
}

/*
	This function checks that sufficient memory is available for
	image storage, allocating more as needed.  Returns 1 if
	successful.
*/	
gks_getimem(isize)
int isize;
{
	char *malloc();

	if (isize>ibuf_sz)
		{
 		if (debugmode) fprintf(stderr,
		   " gks_getimem: increasing image mem to %d ints.\n",isize);
 		if (ibuf_sz>0) { free(ibuf); ibuf_sz= 0; };
		if ( (ibuf= (int *)malloc(isize*sizeof(int))) == 0 )
			{ 
			fprintf(stderr," Unable to allocate image memory.\n");
		 	return(2);
			};
		ibuf_sz= isize;
		};
	return(1);
}
            
/*                              
	This routine sets up a color table with which to 'fake' direct color
  	on an indexed color device.  Return 1 if setup was successful.
*/
gks_fake_dc()  
{
	float r,g,b,rtbl,gtbl,btbl,rnxt,gnxt,bnxt;
	int ir,ig,ib,nr,ng,nb,index,itbl,irtbl,igtbl,ibtbl,
		inxt,irnxt,ignxt,ibnxt,roff,goff,ctoffset;
	char *malloc();

	if (debugmode) fprintf(stderr,
		" gks_fake_dc: setting up direct color simulation.\n");
 
	if (dc_init_flag) return(1);

	if (clr_t_sz==0)
		{
		fprintf(stderr," GKS driver not initialized.\n");
		return(2);
		};
	if (clr_t_sz<8)
		{
		fprintf(stderr,
			" Color table too small to fake direct color\n");
		return(2);
		};

	nb= gks_cube_rt(clr_t_sz);
	ng= gks_sqr_rt(clr_t_sz/nb);
	nr= clr_t_sz/(nb*ng);
	dc_totclrs= nr*ng*nb;
	ctoffset= clr_t_sz - dc_totclrs;

	/* Allocate memory for color table and best-color correspondence
		table */
	if ( (clr_tbl= (float *)malloc(3*dc_totclrs*sizeof(float))) 
		== 0 )
		{ 
		fprintf(stderr,
			" Unable to allocate color table memory.\n");
		return(2);
		};
	if ( (best_clr= (int *)malloc(BEST_C_SZ*sizeof(int))) == 0 )
		{ 
		fprintf(stderr,
			" Unable to allocate best-color list memory.\n");
		return(2);
		};

	/* Build the color table */
	itbl= 0;
	for (ir=0; ir<nr; ir++)
		for (ig=0; ig<ng; ig++)
			for (ib=0; ib<nb; ib++)
			   {
		 	   clr_tbl[3*itbl]= ((float) ir)/((float)(nr-1));
		 	   clr_tbl[3*itbl+1]= ((float) ig)/((float)(ng-1));
		 	   clr_tbl[3*itbl+2]= ((float) ib)/((float)(nb-1));
			   itbl++;
			   };             
                         
	/* Build the nearest-color-index table */
	irtbl= 0;  roff= 0;
	rtbl= clr_tbl[0];
	rnxt= clr_tbl[3*ng*nb];
	for ( ir=7; ir<256; ir+=8 )
		{
		r= b_to_f(ir);
		if ( (r-rtbl>rnxt-r) && ( irtbl < nr-1 ) )
		      	{
			irtbl++;  roff += ng*nb;
			rtbl= rnxt;
			rnxt= clr_tbl[ 3*(irtbl+1)*ng*nb ];
			};
		igtbl= 0;  goff= 0;
		gtbl= clr_tbl[3*roff+1];
		gnxt= clr_tbl[3*(roff+nb)+1];
		for ( ig=7; ig<256; ig+=8 )
			{
			g= b_to_f(ig);
			if ( (g-gtbl>gnxt-g) && (igtbl < ng-1 ) )
				{
				igtbl++;  goff += nb;
				gtbl= gnxt;
				gnxt= clr_tbl[ 3*(roff+(igtbl+1)*nb) +1];
				};
			ibtbl= 0;
			btbl= clr_tbl[3*(roff+goff)+2];
			bnxt= clr_tbl[3*(roff+goff+1)+2];
			for ( ib=7; ib<256; ib+=8 )
				{                      
				b= b_to_f(ib);
				if ( (b-btbl>bnxt-b) && ( ibtbl < nb-1 ) )
					{
				     	ibtbl++;
					btbl= bnxt;
					bnxt= clr_tbl[3*(roff+goff+ibtbl+1)+2];
					};
				index= pack_clr(ir,ig,ib);
				best_clr[index]= ctoffset+roff+goff+ibtbl;
			 	};
			};
		};

	dc_init_flag= 1;

	return(1);
}

/*                        
	This routine swaps in the direct color simulation color map
	if it is not currently in effect.  If necessary, the map and
	related data structures are generated by calling gks_fake_dc.
	Returns 1 if successful.
*/
gks_dc_colors()
{
	int loopmax,index,rc=1;
	float *cptr;

	if (debugmode) fprintf(stderr,
		" gks_dc_colors: swapping in direct color sim table.\n");
 
	if (dc_ready_flag) return(1);

	if ( (dc_init_flag) || ( (rc=gks_fake_dc()) ==1 ) )
		{
 		cptr= clr_tbl;
		for (index=clr_t_sz-dc_totclrs; index<clr_t_sz; index++)
			{
			gks$set_color_rep(&ws_id,&index,cptr,cptr+1,cptr+2);
			cptr+=3;
			};
		dc_ready_flag= 1;
		};

	return(rc);
}

/* 
	This routine returns the greatest integer less than or equal to
	the cube root of the small integer i.
*/
gks_cube_rt(i)
int i;
{
	int j;
	for (j=0; j<i; j++) if (j*j*j>i) return(j-1);
	return(i);
}

/* 
	This routine returns the greatest integer less than or equal to 
	the square root of the small integer i.
*/
gks_sqr_rt(i)
int i;
{
	int j;
	for (j=0; j<i; j++) if (j*j>i) return(j-1);
	return(i);
}

/*                 
	This routine sets up the interface between the GKS driver
	and GPLOT.  It appears at the end of the module so that it
	will have access to the symbolic names of functions defined
	so far.
*/                                   
void gks_setup(pOp, pDev_info, pc1, pc2, pc3, pc5, pdelim, mfdesc, pdesc, 
	mfctrl, pgprim, pattr, escfun, extfun, ctrl, pargc, argv)
struct one_opt 		*pOp;	/* the command line options, in only */
struct info_struct *pDev_info;	/* device info to fill out, out only */
struct mf_d_struct 	*pc1;	/* the class 1 elements, in only */
struct pic_d_struct 	*pc2;	/* the class 2 elements, in only */
struct control_struct	*pc3;	/* the class 3 elements, in only */
struct attrib_struct	*pc5;	/* the class 5 elements, in only */
int (*pdelim[])(); 		/* delimiter functions, out only */
int (*mfdesc[])();     		/* metafile descriptor functions */
int (*pdesc[])();      		/* page descriptor functions */
int (*mfctrl[])();     		/* metafile control functions */
int (*pgprim[])();     		/* graphical primitives, out only */
int (*pattr[])();      		/* the attribute functions, out only */
int (*escfun[])();     		/* the escape functions */
int (*extfun[])();     		/* the external functions *
int (*ctrl[])();       		/* controller functions */
int *pargc;			/* pointer to argc of main routine */
char *argv[];			/* argv of main routine */
{
	int ierror,imeters,num_char_exp,num_indexes,prec_ret_sz,
		font_ret_sz,num_colors,color_or_mono,num_l_types,
		num_l_widths,num_l_indices,l_type_ret_sz,num_m_types,
		num_m_sizes,num_m_indices,m_type_ret_sz;

	float nom_l_width,l_w_min,l_w_max,nom_m_size,m_s_min,m_s_max,
	      	xmax,ymax,h_min,h_max,ch_x_min,ch_x_max;

	$DESCRIPTOR(error_file_dsc,"sys$error:");
	DESC_ARRAY(font_i_desc,NFONTS,font_index);
	DESC_ARRAY(font_p_desc,NPREC,font_prec);
	DESC_ARRAY(line_t_desc,NLINES,line_types);
	DESC_ARRAY(mark_t_desc,NMARKERS,marker_types);

	/* Initialize the cell array utilities */
	cla_init( pc1, pc2, pc5, pgprim, pattr );

	/* store the command line argument pointer */
	Op = pOp;

	if (debugmode) fprintf(stderr," gks_setup \n");
                                                  
	/* store the CGM data structure and device info pointers*/
	State_c1 = pc1;
	State_c2 = pc2;
	State_c3 = pc3;
	State_c5 = pc5;
	Dev_info = pDev_info;

	/* now fill out the function pointer arrays for CGM */
	/* the delimiter functions */
	pdelim[(int) B_Mf] 	= gks_begin;
	pdelim[(int) E_Mf]	= gks_end;
	pdelim[(int) B_Pic_Body]= gks_bpage;
	pdelim[(int) E_Pic]	= gks_epage;

	/* the graphical primitives */
	pgprim[(int) PolyLine]	= gks_pline;
	pgprim[(int) Dis_Poly]	= gks_dpline;
	pgprim[(int) PolyMarker]= gks_pmarker;
	pgprim[(int) Text]	= gks_text;
	pgprim[(int) Polygon]	= gks_pgon;
	pgprim[(int) Cell_Array]	= gks_carray;

	/* the attributes */
	pattr[(int) LType]	= gks_l_type;
	pattr[(int) LWidth]	= gks_l_width;
	pattr[(int) LColour]	= gks_l_colour;
	pattr[(int) MColour]	= gks_mk_colour;
	pattr[(int) MType]	= gks_mk_type; 
	pattr[(int) MSize]	= gks_mk_size;
	pattr[(int) TColour]	= gks_t_colour;
	pattr[(int) CHeight]	= gks_c_height;
	pattr[(int) TAlign]	= gks_t_align;
	pattr[(int) FillColour]	= gks_fl_colour;
	pattr[(int) IntStyle]	= gks_fl_style;
	pattr[(int) ColTab]	= gks_ctab;

	/* 
	fill out the device info structure, as far as now known.
	Some data will be set later in the routine.
	*/
	Dev_info->x_offset	= 0.0;
	Dev_info->y_offset	= 0.0;
	Dev_info->capability	= h_center + v_center + string_text; 
	strcpy(Dev_info->out_name, ".GKS"); /* This won't be used */
	Dev_info->rec_size	= 80;

	/* 
	The following long series of magic initializes GKS, and 
	inquires about certain device capabilities.
	*/

	/* Initialize GKS. */
	gks$open_gks(&error_file_dsc);

	/* 
	Inquire about display size and pixel counts, and use them
	to set device size and resolution.  Note we assume the
	window will be square and that the distances will be in
	meters (=3.29 feet= 3.29*12 inches) 
	*/
	gks$inq_max_ds_size(&GKS$K_WSTYPE_DEFAULT,&ierror,
		&imeters,&dev_x,&dev_y,&raster_x,&raster_y);
	if (ierror!=0) exit(ierror);
	Dev_info->x_size 	= 3.28*12.0*dev_y;
	Dev_info->y_size 	= 3.28*12.0*dev_y;
	Dev_info->pxl_in 	= raster_x/(3.28*12.0*dev_x);
	Dev_info->ypxl_in 	= raster_y/(3.28*12.0*dev_y);

	/* Set up default line width, and edge width (assumed same) */
	gks$inq_pline_fac(&GKS$K_WSTYPE_DEFAULT,&ierror,
		&num_l_types,&line_t_desc,&num_l_widths,&nom_l_width,
		&l_w_min,&l_w_max,&num_l_indices,&l_type_ret_sz);
	if (ierror != 0) exit(ierror);
	if (l_type_ret_sz > NLINES)
		{
		fprintf(stderr, " Too many line types for available list!\n");
		exit(2);
		}
	Dev_info->d_l_width	= 3.28*12.0*nom_l_width*Dev_info->pxl_in;
	Dev_info->d_e_width	= Dev_info->d_l_width;

	/* Set up default marker size */
	gks$inq_pline_fac(&GKS$K_WSTYPE_DEFAULT,&ierror,
		&num_m_types,&mark_t_desc,&num_m_sizes,&nom_m_size,
		&m_s_min,&m_s_max,&num_m_indices,&m_type_ret_sz);
	if (ierror != 0) exit(ierror);
	if (m_type_ret_sz > NMARKERS)
		{
		fprintf(stderr, " Too many marker types for available list!\n");
		exit(2);
		}
	Dev_info->d_m_size	= 3.28*12.0*nom_m_size*Dev_info->pxl_in;

	/* Set up global list of available fonts */
	gks$inq_text_fac(&GKS$K_WSTYPE_DEFAULT,&ierror,
		&num_fonts,&font_i_desc,&font_p_desc,
		&num_c_heights,&h_min,&h_max,
		&num_char_exp,&ch_x_min,&ch_x_max,
		&num_indexes,&prec_ret_sz,&font_ret_sz);
	if (ierror!=0) exit(ierror);
	if (font_ret_sz>NFONTS)
		{
		fprintf(stderr, " Too many fonts for available list!\n");
		exit(2);
		}
	if (prec_ret_sz>NPREC)
		{
		fprintf(stderr, 
			" Too many font precisions for available list!\n");
		exit(2);
		}
	/* Rescale min and max heights to pixels and save copies */
	c_h_min= (int)(3.28 * 12.0 * h_min * Dev_info->ypxl_in);
	c_h_max= (int)(3.28 * 12.0 * h_max * Dev_info->ypxl_in);
	/* Character height default is 1% of the default norm. window */
	Dev_info->c_height= (int)(0.01 * Dev_info->y_size * Dev_info->ypxl_in);
	/* Assume square characters to get character width */
	Dev_info->c_width	= Dev_info->c_height;

	/* Set up global color table information */
	gks$inq_color_fac(&GKS$K_WSTYPE_DEFAULT,&ierror,
		&num_colors,&color_or_mono,&clr_t_sz);
	if (ierror!=0) exit(ierror);
	if (clr_t_sz==0)
		{
		fprintf(stderr, 
		   " This GKS driver only works for indexed color devices!\n");
		exit(2);
		};

	/* Set up coordinate information (driver static variables). */
	xscale= 1.0/(Dev_info->x_size * Dev_info->pxl_in);
	yscale= 1.0/(Dev_info->y_size * Dev_info->ypxl_in);
	
	state_level= -1;  /* Just starting driver */
}

