 /*
  * On socket-only systems, fromhost() is nothing but an alias for the
  * socket-specific sock_host() function.
  * 
  * On systems with sockets and TLI, fromhost() determines the type of API
  * (sockets, TLI), then invokes the appropriate API-specific routines.
  * 
  * The API-specific routines determine the nature of the service (datagram,
  * stream), and the name and address of the host at the other end. In case
  * of an IP service, these routines also determine the local address and
  * port, and the remote username if username lookups are done irrespective
  * of client. All results are in static memory.
  * 
  * The return value is (-1) if the remote host pretends to have someone elses
  * name, or if the remote host name is available but could not be verified;
  * in either case the hostname will be ignored. The return status is zero in
  * all other cases (the hostname is unavailable, or the host name double
  * check succeeds).
  * 
  * Diagnostics are reported through syslog(3).
  * 
  * Author: Wietse Venema, Eindhoven University of Technology, The Netherlands.
  */

#ifndef lint
static char sccsid[] = "@(#) fromhost.c 1.14 93/12/13 22:23:05";
#endif

/* System libraries. */

#include <sys/types.h>
#include <stdio.h>
#include <syslog.h>

#if defined(TLI) || defined(TLI_SOCKETS)
#include <sys/tiuser.h>
#include <stropts.h>
#endif

/* Local stuff. */

#include "log_tcp.h"

#if !defined(TLI) && !defined(TLI_SOCKETS)

/* fromhost - compatibility wrapper for socket-only systems */

int     fromhost(client)
struct client_info *client;
{
    int     client_fd = 0;		/* XXX compatibility */

    return (sock_host(client, client_fd));
}

#endif /* !defined(TLI) && !defined(TLI_SOCKETS) */

#ifdef TLI

/* fromhost - find out what network API we should use */

int     fromhost(client)
struct client_info *client;
{
    int     client_fd = 0;		/* XXX compatibility */

    /*
     * On systems with streams support the IP network protocol family may
     * have more than one programming interface (sockets and TLI).
     * 
     * Thus, we must first find out what programming interface to use: sockets
     * or TLI. On some systems, sockets are not part of the streams system,
     * so if stdin is not a stream we assume sockets.
     */

    if (ioctl(client_fd, I_FIND, "timod") > 0) {
	return (tli_host(client, client_fd));
    } else {
	return (sock_host(client, client_fd));
    }
}

#endif /* TLI */

#ifdef TLI_SOCKETS

/* fromhost - use socket primitives even with TLI */

int     fromhost(client)
struct client_info *client;
{
    int     client_fd = 0;		/* XXX compatibility */
    int     ret;

    /*
     * Some System V.3 based systems have TLI but the supporting libraries
     * are not as complete as with System V.4. Dynix/PTX is an example. The
     * workaround is to temporarily switch to the socket interface, identify
     * the client name/address with socket calls, then to switch back to TLI.
     */

#define SWAP_MODULE(f, old, new) (ioctl(f, I_POP, old), ioctl(f, I_PUSH, new))

    if (ioctl(client_fd, I_FIND, "timod") > 0) {
	if (SWAP_MODULE(client_fd, "timod", "sockmod") != 0)
	    syslog(LOG_ERR, "swap timod for sockmod: %m");
	ret = sock_host(client, client_fd);
	if (SWAP_MODULE(client_fd, "sockmod", "timod") != 0)
	    syslog(LOG_ERR, "swap sockmod for timod: %m");
	return (ret);
    } else {
	return (sock_host(client, client_fd));
    }
}

#endif /* TLI_SOCKETS */
