/*  File: gmapcpt.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 **  Manipulates the genetic map.                             **
 * Exported functions:
     gMapCompute()
 * HISTORY:
 * Last edited: Mar 26 13:13 1992 (mieg)
 * Created: Tue Nov 19 15:54:48 1991 (mieg)
 *-------------------------------------------------------------------
 */


#include "acedb.h"
#include  "lex.h"
#include "bs.h"
#include "a.h"
#include "sysclass.wrm"
#include "classes.wrm"
#include "systags.wrm"
#include "tags.wrm"
#include "plot.h"
#include "keyset.h"
#include "graph.h"
#include "display.h"

typedef struct LOOKSTUFF
  { int   magic;        /* == MAGIC*/
    Graph graph ;
    Array bands ;
    int projectBox ;
    char project[50] ;
    int curr, line ;
  } *LOOK ;


#define MAGIC  34153
static BOOL doRecompute = TRUE ;
 
void gMapCompute(void) ;

#define GCPTGET(name)     LOOK look ;\
                                       \
                          if (!graphAssFind (gMapCompute,&look)) \
		            messcrash ("%s can't find graph",name) ; \
			  if (!look) \
                            messcrash ("%s  received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)  ;\
                          displayPreserve()  


typedef struct gAstruct
   { KEY g;  int n ; } GASt, *GA ;
#define gastFormat "ki"

typedef struct cAstruct
  { KEY c, g1, g2 ; float p, dp ;} CASt, *CA ;
#define castFormat "3k2f"

static int gLine ;
static   Graph gCptGraph = 0 ;

/********************************/

static int distanceOrder(void*a, void*b)
{
 return ( ((CA) a) -> p > ((CA) b)->p  )  ?
	      1
	    : ( (CA) a)->p == ((CA)b)->p ? 0 : -1   ;
}


/********************************/

static int firstGeneOrder(void*a, void*b)
{
 return lexstrcmp (name( ((CA) a) -> g1),
                   name( ((CA) b) -> g1) ) ;
}

/********************************/

static int  geneFind(Array gA,KEY g1)
{
  int i = arrayMax(gA) ;
  
  while(i--)
    if(array(gA,i,GASt).g == g1)
      return i ;
  messcrash("cannot find %s", name(g1)) ;
  return 0 ; /* for compiler happiness  */
}

/**********************************/


	/* Compute the most probable genetic distance p
	 * given a 2 point cross ab/++
	 * a and b recessive
	 * observed values s = wild type, a=A (b is a let),
	 * dp is the error on p
	 */

static BOOL dist2pType2
     (KEY key, int s, int a, float *p, float *dp)
{
  int n = s + a ;
  double  a1 = ((float)a)/n , u ;
   
  if( a== -1 )
    return FALSE ;
  /* I solve p = 1 - sqrt(1 - 3 a/n) */
  u =  1 - 3 * a1 ;
  if (u < 0)
    { fprintf(stderr,"Negative discriminant in dist2pType2 %s",name(key));
      return FALSE ;
    }
  
   *p = (float) ( 1 - sqrt(u) ) ;
  if (*p < 0)
    { fprintf(stderr,"Negative *p distance in dist2pType2 %s",name(key));
      return FALSE ;
    }

  *p *= 100 ; /* to be in centiMorgan */
  *dp = 0. ;  /* for the moment */
/*
  graphText(messprintf("p = %d",(int)(100 *(*p))), 60,gLine++) ;
*/
  return TRUE ;
}

/**********************************/

	/* Compute the most probable genetic distance p
	 * given a 2 point cross ab/++
	 * a and b recessive
	 * observed values ab = AB , a = A, s and b not counted
	 * dp is the error on p
	 */

static BOOL dist2pType21
     (KEY key, int ab, int a, float *p, float *dp)
{
  int n = ab + a ;
  double  q2 = ((float) ab) / n ;
   
  if (q2 < 0)
    { fprintf(stderr,"Negative discriminant in dist2pType21 %s",name(key));
      return FALSE ;
    }
  
   *p = (float) ( 1 - sqrt(q2) ) ;
  if (*p < 0)
    { fprintf(stderr,"Negative *p distance in dist2pType2 %s",name(key));
      return FALSE ;
    }

  *p *= 100 ; /* to be in centiMorgan */
  *dp = 0. ;  /* for the moment */
/*
  graphText(messprintf("p = %d",(int)(100 *(*p))), 60,gLine++) ;
*/
  return TRUE ;
}

/**********************************/

	/* Compute the most probable genetic distance p
	 * given a 2 point cross ab/++
	 * a and b recessive
	 * observed values ab = AB , total
	 * dp is the error on p
	 */

static BOOL dist2pType22
     (KEY key, int ab, int total, float *p, float *dp)
{
   double  q2 =  4 * ((float) ab) / total ;
   
  if (q2 < 0)
    { fprintf(stderr,"Negative discriminant in dist2pType21 %s",name(key));
      return FALSE ;
    }
  
   *p = (float) ( 1 - sqrt(q2) ) ;
  if (*p < 0)
    { fprintf(stderr,"Negative *p distance in dist2pType2 %s",name(key));
      return FALSE ;
    }

  *p *= 100 ; /* to be in centiMorgan */
  *dp = 0. ;  /* for the moment */
/*
  graphText(messprintf("p = %d",(int)(100 *(*p))), 60,gLine++) ;
*/
  return TRUE ;
}

/**********************************/

	/* Compute the most probable genetic distance p
	 * given a 2 point cross ab/++
	 * a and b recessive
	 * observed values s = wild type, a=A, b=B, d =AB
	 * dp is the error on p
	 */

static BOOL dist2pType4
     (KEY key, int s, int a, int b, int d, float *p, float *dp)
{
  int n = s + a + b + d ;
  double s1 = (1.0 * s)/n ,
  d1 = (1.0 * d ) /n ,
  u, v, delta, x ;
  
/*
  graphText( name(key),2,gLine) ;
  graphText(messprintf("s = %d",s), 10,gLine) ;
  graphText(messprintf("a = %d",a), 20,gLine) ;
  graphText(messprintf("b = %d",b), 30,gLine) ;
  graphText(messprintf("d = %d",d), 40,gLine) ;
  graphText(messprintf("tot = %d",n), 50,gLine++) ;
*/  
  
  if( a== -1 || b== -1 || s == -1 || d == -1)
    return FALSE ;
  /* I solve x^2 + 2 u x + v = 0 */
  u =  1 - (3. * s1 + d1) / 2. ;
  v =  - 2. * d1 ;
  /* Note that v<=0 hence the discriminant is >0 and one root is positive
     or null */
  delta = u*u - v ;
  if (delta < 0)
    { fprintf(stderr,"Negative discriminant in dist2pType4 %s",name(key));
      return FALSE ;
    }
  
  x = -u + sqrt(delta) ; /* take positive root */
  if (x < 0)
    { fprintf(stderr,"Negative x in dist2pType4 %s",name(key));
      return FALSE ;
    }
  *p = (float) ( 1 - sqrt(x) ) ;
  if (*p < 0)
    { fprintf(stderr,"Negative *p distance in dist2pType4 %s",name(key));
      return FALSE ;
    }

  *p *= 100 ; /* to be in centiMorgan */
  *dp = 0. ;  /* for the moment */
/*
  graphText(messprintf("p = %d",(int)(100 *(*p))), 60,gLine++) ;
*/
  return TRUE ;
}

/*********************************************************/

static BOOL  nameCheck2(char *cp, char *cq)
{ int i = 3 ;
   while(i--)
    if(freeupper(*cp++) != freeupper(*cq++))
      return FALSE ;
  return TRUE ;
}

/*****************/

static BOOL  nameCheck3(KEY gene, char *cq)
{ OBJ Gene ; 
  KEY gene2 ;
  char *cp ;
  
  if (nameCheck2( name(gene), cq))
    return TRUE ;
  if( Gene = bsCreate(gene))
    {
      if (bsGetData(Gene, _Nick_name,_Text, &cp)
	  && nameCheck2( name(gene), cp))
	{ bsDestroy(Gene) ;
	  return TRUE ;
	}

      if (bsGetKey(Gene, _Other_name, &gene2)
	  && nameCheck2( name(gene2), cq))
	{ bsDestroy(Gene) ;
	  return TRUE ;
	}

      bsDestroy(Gene) ;
    }

  return FALSE ;
}

/*****************/

   /* Check double name ex: DpyUnc matches dpy-5 unc-52 */
static BOOL  nameCheckDouble(KEY g1, KEY g2, char *t4)
{    
  return
    ( nameCheck3(g1, t4) &&
     nameCheck3(g2, t4 + 3)  )
      ||
    ( nameCheck3(g2, t4) &&
     nameCheck3(g1, t4 + 3)  )
      ;	
}

/****************/

static void nameSwitch(char *cp, char *cq, int *n1, int *n2)
{ char buf[12] ; int i ;

  strncpy(buf, cp, 12) ;
  strncpy(cp, cq, 12) ;
  strncpy(cq, buf, 12) ;

  i = *n1 ;
  *n1 = *n2 ;
  *n2 = i ;
}

/****************/

  /* Decides on the type of the cross */
static int nameCheck(KEY g1, KEY g2, char *t1, char *t2, char*t3, char *t4, 
		      int *n1, int *n2, int *n3, int *n4, int nRes)
{ 
  if(nRes == 2)
    {            /* Check WT  */
      if(nameCheck2("WT", t1))
	{
	  if(nameCheck3(g1, t2) ||
	     nameCheck3(g2, t2)) 
	    return 2 ;
	  else
	    return 0 ;
	}
      else
	{ 
	  if(nameCheckDouble(g1, g2, t1))
	    {
	      if ( nameCheck3(g1, t2) ||
		  nameCheck3(g2, t2))
		return 21 ;
	      else if (nameCheck2("total",t2))
		return 22 ;
	      else
		return 0 ;
	    }
	  if(nameCheckDouble(g1, g2, t2))
	    {
	      if ( nameCheck3(g1, t1) ||
		  nameCheck3(g2, t1))
		return 25 ;
      /* i.e. 107:  15Dpy, 210 DpyUnc */
	    }
	  if ((nameCheck3(g1, t1) && nameCheck2("total", t2) && nameCheck3(g2,"let"))
	      || (nameCheck3(g2, t1) && nameCheck2("total", t2) && nameCheck3(g1,"let")))
	    return 23 ;
	  if ((nameCheck3(g1, t1) && nameCheck2("adults", t2) && nameCheck3(g2,"let"))
	       || (nameCheck3(g2, t1) && nameCheck2("adults", t2) && nameCheck3(g1,"let")))
	    return 23 ;
	  if ((nameCheck3(g1, t1) && nameCheck2("total", t2))
	     || (nameCheck3(g2, t1) && nameCheck2("total", t2)))
	    return 24 ;
	}
    }
       
  if(nRes == 3)
    {
      if(strlen(t3) <= 4 && strlen(t2) == 6)
	nameSwitch(t2, t3, n2, n3) ;

            /* Check WT  */
      if(nameCheck2("WT", t1))
	{ /* Check first name ex: Dpy matches dpy-5 */
	  if(!nameCheck3(g1, t2)) 
	    { if(nameCheck3(g1, t3)) 
		nameSwitch(t2, t3, n2, n3) ;
	    else
	      return 0 ;
	    }
	  
	  /* Check second name ex Unc matches unc-52 */
	  if(!nameCheck3(g2, t3)) 
	    return 0 ;
	  return 3 ;
	}
      else
	{ /* Check first name ex: Dpy matches dpy-5 */
	  if(!nameCheck3(g1, t1)) 
	    { if(nameCheck3(g1, t2)) 
		nameSwitch(t1, t2, n1, n2) ;
	    else
	      return 0 ;
	    }
	  
	  /* Check second name ex Unc matches unc-52 */
	  if(!nameCheck3(g2, t2)) 
	    return 0 ;
	  if(!nameCheckDouble(g1, g2, t3))
	    return 0 ;
	  return 5 ;
	}
    }

  if(nRes == 4)
    {            /* Check WT  */
      if(!nameCheck2("WT", t1))
	return 0 ;

      if(strlen(t4) <= 4 && strlen(t2) == 6)
	nameSwitch(t2, t4, n2, n4) ;

      /* Check first name ex: Dpy matches dpy-5 */
      if(!nameCheck3(g1, t2)) 
	{ if(nameCheck3(g1, t3)) 
	    nameSwitch(t2, t3, n2, n3) ;
	else
	  return 0 ;
	}
      
      
      /* Check second name ex Unc matches unc-52 */
      if(!nameCheck3(g2, t3)) 
	return 0 ;
      
      if(!nameCheckDouble(g1, g2, t4))
	return 0 ;
      return 4 ;
    }
  return 0 ;
}

/*********************************************************/
static KEYSET mismatch = 0 ;
static int nMismatch = 0 ;

static BOOL get2pResult(KEY cross)
{
  char * cp , *text, t1[12], t2[12], t3[12], t4[12] , cutter; 
  KEY g1, g2 ; float p = 0 , dp = 0  , p1 ;
  int  n1 = 0, n2 = 0, n3 = 0, n4 = 0 , total, nRes = 0 ;
  OBJ Cross = bsUpdate(cross) ;

  if(!Cross)
    return FALSE ;


  if(bsFindTag(Cross,_ACEDB))
    { bsRemove(Cross) ;
      bsSave(Cross) ;
      Cross = bsUpdate(cross) ;
    }

  if(!bsGetKey(Cross,_Gene1,&g1) ||
     !bsGetKey(Cross,_Gene2,&g2) ||
     ! bsGetData(Cross,_Genotype,_Text,&text) )
    goto abort ;

  /* interpret the genotype */
  freeforcecard(text) ;
  cp = freewordcut(" /+",&cutter) ;
  if(cutter != ' ' ||
     strcmp(cp, name(g1)))
    { 
      graphText
	(messprintf("Bad 1st gene %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }
  
  freenext() ;
  cp = freewordcut(" /+",&cutter) ;
  if(cutter != '/' ||
     strcmp(cp, name(g2)))
    { 
      graphText
	(messprintf("Bad 2nd gene %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }
  
  freenext() ;
  cp = freewordcut(" +", &cutter) ;
  if(cutter != '+')
    { 
       graphText
	 (messprintf("Bad 3rd gene %s in genotype of %s : %s",
		     cp ? cp : "NULL",
		     name(cross), text),     1, gLine++ ) ;
       while(cp = freeword()) ;
       goto abort ;
     }

  freenext() ;
  cp = freewordcut(" +", &cutter) ;
  if(cutter != '+')
    { 
       graphText
	 (messprintf("Bad 4th gene %s in genotype of %s : %s",
		     cp ? cp : "NULL",
		     name(cross), text),     1, gLine++ ) ;
       while(cp = freeword()) ;
       goto abort ;
     }

  while(cp = freeword()) ;

  if(! bsGetData(Cross,_Results,_Text,&text) )
    goto abort ;

  /* interpret the Results */

  /* remove the [comment part] */

  cp = text ;
  while(*cp && *cp != '[') cp++ ;
  *cp = 0 ;

  freeforcecard(text) ;

  nRes = 0 ;
  if(freeint(&n1) &&
     (cp = freeword()))
    strcpy(t1,cp) ;
  else 
    goto verif ;

  nRes++ ;
  if(freeint(&n2) &&
     (cp = freeword()))
    strcpy(t2,cp) ;
  else 
    goto verif ;

  if(!strcmp(t1, "recombinants"))
    { goto verif ;
    }
    
  nRes++ ;
  if(freeint(&n3) &&
     (cp = freeword()))
    strcpy(t3,cp) ;
  else 
    goto verif ;

  nRes++ ;
  if(freeint(&n4) &&
     (cp = freeword()))
    strcpy(t4,cp) ;
  else 
    goto verif ;

  nRes++ ;
  while(cp = freeword()) ;

  /* Verify the results subfields */
 verif:
 
  switch( nameCheck(g1, g2, t1, t2, t3, t4, &n1, &n2, &n3, &n4, nRes))
    {
    case 2:
      total = n1 + n2 + n3 + n4 ;
      dist2pType2( cross, n1, n2, &p, &dp) ;
      bsAddData(Cross, _Counts, _Int, &n1) ;
      bsAddData(Cross, _bsRight, _Text,&t1) ;
      bsAddData(Cross, _bsRight, _Int, &n2) ;
      bsAddData(Cross, _bsRight, _Text,&t2) ;
      bsAddData(Cross, _Total, _Int, &total) ;
      bsAddData(Cross, _ACEDB_dist, _Float, &p) ;
      bsAddData(Cross, _bsRight, _Float,&dp) ;
      graphText
	(messprintf("Cross: %s  Results :%s", name(cross), text), 
	 1, gLine++ ) ;
      graphText
	(messprintf("        %d %s, %d %s,  total %d,  d= %6.2f, dp = %4.2f ",
		    n1, t1, n2, t2, total, p, dp),
	 1, gLine++ ) ;
      break ;

    case 21:
      total = n1 + n2 + n3 + n4 ;
      dist2pType21( cross, n1, n2, &p, &dp) ;
      bsAddData(Cross, _Counts, _Int, &n1) ;
      bsAddData(Cross, _bsRight, _Text,&t1) ;
      bsAddData(Cross, _bsRight, _Int, &n2) ;
      bsAddData(Cross, _bsRight, _Text,&t2) ;
      bsAddData(Cross, _Total, _Int, &total) ;
      bsAddData(Cross, _ACEDB_dist, _Float, &p) ;
      bsAddData(Cross, _bsRight, _Float,&dp) ;
      graphText
	(messprintf("Cross: %s  Results :%s", name(cross), text), 
	 1, gLine++ ) ;
      graphText
	(messprintf("        %d %s, %d %s,  total %d,  d= %6.2f, dp = %4.2f ",
		    n1, t1, n2, t2, total, p, dp),
	 1, gLine++ ) ;
      break ;

    case 22:
      total = n2 ;
      dist2pType22( cross, n1, n2, &p, &dp) ;
      bsAddData(Cross, _Counts, _Int, &n1) ;
      bsAddData(Cross, _bsRight, _Text,&t1) ;
      bsAddData(Cross, _bsRight, _Int, &n2) ;
      bsAddData(Cross, _bsRight, _Text,&t2) ;
      bsAddData(Cross, _Total, _Int, &total) ;
      bsAddData(Cross, _ACEDB_dist, _Float, &p) ;
      bsAddData(Cross, _bsRight, _Float,&dp) ;
      graphText
	(messprintf("Cross: %s  Results :%s", name(cross), text), 
	 1, gLine++ ) ;
      graphText
	(messprintf("        %d %s, %d %s,  total %d,  d= %6.2f, dp = %4.2f ",
		    n1, t1, n2, t2, total, p, dp),
	 1, gLine++ ) ;
      break ;

    case 23:
      total = n2 ;
      dist2pType2( cross, n2 - n1, n1, &p, &dp) ;
      bsAddData(Cross, _Counts, _Int, &n1) ;
      bsAddData(Cross, _bsRight, _Text,&t1) ;
      bsAddData(Cross, _bsRight, _Int, &n2) ;
      bsAddData(Cross, _bsRight, _Text,&t2) ;
      bsAddData(Cross, _Total, _Int, &total) ;
      bsAddData(Cross, _ACEDB_dist, _Float, &p) ;
      bsAddData(Cross, _bsRight, _Float,&dp) ;
      graphText
	(messprintf("Cross: %s  Results :%s", name(cross), text), 
	 1, gLine++ ) ;
      graphText
	(messprintf("        %d %s, %d %s,  total %d,  d= %6.2f, dp = %4.2f ",
		    n1, t1, n2, t2, total, p, dp),
	 1, gLine++ ) ;
      break ;

    case 3:  /* ++, A, B,  we estimate AB  = (++ - A - B ) / 3 */
      total = n1 + n2 + n3 + n4 ;
      dist2pType4( cross, n1, n2, n3, (n1 - n2 - n3) / 3, &p, &dp) ;
      bsAddData(Cross, _Counts, _Int, &n1) ;
      bsAddData(Cross, _bsRight, _Text,&t1) ;
      bsAddData(Cross, _bsRight, _Int, &n2) ;
      bsAddData(Cross, _bsRight, _Text,&t2) ;
      bsAddData(Cross, _bsRight, _Int, &n3) ;
      bsAddData(Cross, _bsRight, _Text,&t3) ;
      bsAddData(Cross, _bsRight, _Int, &n4) ;
      bsAddData(Cross, _bsRight, _Text,&t4) ;
      bsAddData(Cross, _Total, _Int, &total) ;
      bsAddData(Cross, _ACEDB_dist, _Float, &p) ;
      bsAddData(Cross, _bsRight, _Float,&dp) ;
      graphText
	(messprintf("Cross: %s  Results :%s", name(cross), text), 
	 1, gLine++ ) ;
      graphText
	(messprintf("        %d %s, %d %s, %d %s, %d %s total %d,  d= %6.2f, dp = %4.2f ",
		    n1, t1, n2, t2,  n3, t3,  n4, t4, total, p, dp),
	 1, gLine++ ) ;
      break ;
    case 4:
      total = n1 + n2 + n3 + n4 ;
      dist2pType4( cross, n1, n2, n3, n4, &p, &dp) ;
      bsAddData(Cross, _Counts, _Int, &n1) ;
      bsAddData(Cross, _bsRight, _Text,&t1) ;
      bsAddData(Cross, _bsRight, _Int, &n2) ;
      bsAddData(Cross, _bsRight, _Text,&t2) ;
      bsAddData(Cross, _bsRight, _Int, &n3) ;
      bsAddData(Cross, _bsRight, _Text,&t3) ;
      bsAddData(Cross, _bsRight, _Int, &n4) ;
      bsAddData(Cross, _bsRight, _Text,&t4) ;
      bsAddData(Cross, _Total, _Int, &total) ;
      bsAddData(Cross, _ACEDB_dist, _Float, &p) ;
      bsAddData(Cross, _bsRight, _Float,&dp) ;
      graphText
	(messprintf("Cross: %s  Results :%s", name(cross), text), 
	 1, gLine++ ) ;
      graphText
	(messprintf("        %d %s, %d %s, %d %s, %d %s total %d,  d= %6.2f, dp = %4.2f ",
		    n1, t1, n2, t2,  n3, t3,  n4, t4, total, p, dp),
	 1, gLine++ ) ;
      break ;
 case 5:  /* A, B, AB we estimate ++ = A + B + 3 AB */
      total = n1 + n2 + n3 + n4 ;
      dist2pType4( cross, n1 + n2 + 3 * n3, n1, n2, n3, &p, &dp) ;
      bsAddData(Cross, _Counts, _Int, &n1) ;
      bsAddData(Cross, _bsRight, _Text,&t1) ;
      bsAddData(Cross, _bsRight, _Int, &n2) ;
      bsAddData(Cross, _bsRight, _Text,&t2) ;
      bsAddData(Cross, _bsRight, _Int, &n3) ;
      bsAddData(Cross, _bsRight, _Text,&t3) ;
      bsAddData(Cross, _bsRight, _Int, &n4) ;
      bsAddData(Cross, _bsRight, _Text,&t4) ;
      bsAddData(Cross, _Total, _Int, &total) ;
      bsAddData(Cross, _ACEDB_dist, _Float, &p) ;
      bsAddData(Cross, _bsRight, _Float,&dp) ;
      graphText
	(messprintf("Cross: %s  Results :%s", name(cross), text), 
	 1, gLine++ ) ;
      graphText
	(messprintf("        %d %s, %d %s, %d %s, %d %s total %d,  d= %6.2f, dp = %4.2f ",
		    n1, t1, n2, t2,  n3, t3,  n4, t4, total, p, dp),
	 1, gLine++ ) ;
      break ;
    default:
      goto abort ;
    }

  bsGetData(Cross, _Distance, _Float, &p1) ;
  bsSave(Cross) ;

  p1 -= p ;
  if(p1 < 0) p1 = -p1 ;
  if(p1 > .2 * p)
    keySet(mismatch,nMismatch++) = cross ;
  return TRUE ;

 abort:
  while(cp = freeword()) ;
  bsDestroy(Cross) ;
  return FALSE ;
}


/*********************************************************/

static void getAll2pResults(void)
{
  KEY key = 0 ;
  int nn = 0 ;
  KEYSET bad = keySetCreate() ;
  int nBad = 0 ;
  
  graphActivate(gCptGraph) ;

  nMismatch = 0 ;
  mismatch = keySetCreate() ;

  while(lexNext(_V2_point_data, &key))
    if( KEYKEY(key) && 
	( (lexGetStatus(key) & CALCULSTATUS) 
       /* i.e. key has been modified */
       || doRecompute)    /* forced calcul */
       )
      if( get2pResult(key))
	nn++ ;
      else
	keySet(bad, nBad++) = key ;

  graphText(
	    messprintf("Found %d usable 2 point results out of %d",
		       nn, lexMax(_V2_point_data)),
	    3, 7) ;


  graphTextBounds(60,gLine+3) ;

  graphRedraw() ;
  if(keySetMax(bad))
    { lexaddkey("Bizare 2p", &key, _VKeySet) ;
      arrayStore(key, bad, "k") ;
    }
  arrayDestroy(bad) ;
  if(keySetMax(mismatch))
    { lexaddkey("Mismatch 2p", &key, _VKeySet) ;
      arrayStore(key, mismatch, "k") ;
    }
  arrayDestroy(mismatch) ;
  display(key,0,0) ;
}

/*********************************************************/

static int analyse3p(char *text, KEY *k1, KEY* k2, KEY *k3)
{
  char *cp, c ;
  
  freeforcecard(text) ;
  
  freenext() ;
  if (!freestep('('))
    {  
      if (!(cp = freewordcut("(",&c)) || c != '(' ||  /* get a ( */
	  !lexword2key(cp, k1, _VGene) )
	goto abort ;
	
      freenext() ;
      if (!(cp = freewordcut(")",&c)) || c != ')' )    /* get 9/9)  */
	goto abort ;
       
      freenext() ;
      if (freestep('('))                             /* get (     */
	{                                            /* get b c)  */
	  if (!(cp = freewordcut(" ",&c)) || !lexword2key(cp, k2, _VGene) )        
	    goto abort ;
	  
	  if (!(cp = freewordcut(")",&c)) || c != ')' ||
	      !lexword2key(cp, k3, _VGene) )
	    goto abort ;
	  while (freeword()) ;
	  return 3 ;
	}

      if (!(cp = freewordcut("(",&c)) || c != '(' ||  /* get b ( */
	  !lexword2key(cp, k2, _VGene) )
	    goto abort ;

      freenext() ;
      if (!(cp = freewordcut(")",&c)) || c != ')' )    /* get 9/9)  */
	goto abort ;

      freenext() ;
      if (!(cp = freewordcut(" .([+",&c)) ||              /* get c */
	  !lexword2key(cp, k3, _VGene) )
	    goto abort ;

      while (freeword()) ;
      return 1 ;
    }
  else  /* case (ab) (9/0) c */
    {  
      if (! (cp = freewordcut(" ", &c)) || !lexword2key(cp, k1, _VGene) )               /* get ( a b ) */
	goto abort ;
	
      freenext() ;
      if (!(cp = freewordcut(")",&c)) || c != ')' ||
	  !lexword2key(cp, k2, _VGene) )
	    goto abort ;
      
      freenext() ;
      if ((freestep('(') && !(cp = freewordcut(")",&c))) || c != ')' )    /* get (9/9)  */
	goto abort ;
      
      freenext() ;
      if (!(cp = freewordcut(" .(",&c)) ||               /*  get c */
	  !lexword2key(cp, k3, _VGene) )
	    goto abort ;

      while (freeword()) ;
      return 2 ;
    }

 abort:
  while (freeword()) ;
  return 0 ;
}

/*********************************************************/

static void switch3p(OBJ obj,KEY  k1,KEY  k2,KEY  k3,
		     KEY  g1,KEY  g2,KEY  g3,KEY  a1,KEY  a2,KEY  a3) 
{
  if (g1 != k1)
    { if (bsFindTag(obj, _Gene1))
	bsRemove(obj) ;
      if (bsFindTag(obj, _Allele1))
	bsRemove(obj) ;
      bsAddKey(obj, _Gene1, k1) ;
      if (a1)
	bsAddKey(obj, _Allele1, a1) ;
    }
  if (g2 != k2)
    { if (bsFindTag(obj, _Gene2))
	bsRemove(obj) ;
      if (bsFindTag(obj, _Allele2))
	bsRemove(obj) ;
      bsAddKey(obj, _Gene2, k2) ;
      if (a2)
	bsAddKey(obj, _Allele2, a2) ;
    }
  if (g3 != k3)
    { if (bsFindTag(obj, _Gene3))
	bsRemove(obj) ;
      if (bsFindTag(obj, _Allele3))
	bsRemove(obj) ;
      bsAddKey(obj, _Gene3, k3) ;
      if (a3)
	bsAddKey(obj, _Allele3, a3) ;
    }
}

/*********************************************************/

static BOOL get3pResult(KEY cross)
{ char *cp ;
  OBJ obj = bsUpdate(cross) ;
  KEY g1, g2, g3, k1, k2, k3, a1, a2, a3 ;

  if (!obj)
    goto abort ;

  if (bsFindTag(obj, _Result_type))
    bsRemove(obj) ;
  if (!bsGetKey(obj, _Gene1, &g1) ||
      !bsGetKey(obj, _Gene1, &g1) ||
      !bsGetKey(obj, _Gene1, &g1) )
    goto abort ;

  a1 = a2 = a3 = 0 ;
  bsGetKey(obj, _Allele1, &a1) ;
  bsGetKey(obj, _Allele2, &a2) ;
  bsGetKey(obj, _Allele3, &a3) ;

  if (bsGetData(obj, _Results, _Text, &cp))
    switch (analyse3p(cp, &k1, &k2, &k3))
      {
      case 1:
	bsAddTag(obj, _ABC) ;
	break ;
      case 2:
	bsAddTag(obj, _AB_C) ;
	break ;
      case 3:
	bsAddTag(obj, _A_BC) ;
	break ;
      default:
	goto abort ;
      }
  switch3p(obj, k1, k2, k3, g1, g2, g3, a1, a2, a3) ;
  bsSave(obj) ;
  return TRUE ;

 abort:
  bsSave(obj) ;
  return FALSE ;
}

/*********************************************************/

static void getAll3pResults(void)
{
  KEY key = 0;
  int nn = 0 ;
  KEYSET bad = keySetCreate() ;
  int nBad = 0 ;
  
  graphActivate(gCptGraph) ;

  while(lexNext(_V3_point_data, &key))
    if( KEYKEY(key) && 
	( (lexGetStatus(key) & CALCULSTATUS) 
       /* i.e. key has been modified */
       || doRecompute)    /* forced calcul */
       )
      if( get3pResult(key))
	nn++ ;
      else
	keySet(bad, nBad++) = key ;

 
  graphText(
	    messprintf("Found %d usable 3 point results out of %d",
		       nn, lexMax(_V3_point_data)),
	    3, 1) ;
  graphTextBounds(60,gLine+3) ;
  graphRedraw() ;
  if(keySetMax(bad))
    { lexaddkey("Bizare 3p", &key, _VKeySet) ;
      arrayStore(key, bad, "k") ;
    }
  arrayDestroy(bad) ;
}

/*********************************************************/

static BOOL getDfDpResult(KEY cross)
{
  char * cp , *text, cutter; 
  KEY rear, gene ;
  int  type = 0 ;
  OBJ Cross = bsUpdate(cross) ;

  if(!Cross)
    return FALSE ;

  if(bsFindTag(Cross,_ACEDB))
    { bsRemove(Cross) ;
      bsSave(Cross) ;
      Cross = bsUpdate(cross) ;
    }

  if(!bsGetKey(Cross,_Rearrangement, &rear) ||
     !bsGetKey(Cross,_Gene,&gene) ||
     ! bsGetData(Cross,_Results,_Text,&text) )
    goto abort ;

  /* interpret the genotype */
  freeforcecard(text) ;
  cp = freewordcut(" /+",&cutter) ;
  if(cutter != ' ' ||
     strcmp(cp, name(rear)))
    { 
      graphText
	(messprintf("Bad rear %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }
  
  freenext() ;
  cp = freewordcut(" ",&cutter) ;
  if(cutter != ' ')
    { 
      graphText
	(messprintf("Bad text %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }
  
  if(!strcmp(cp, "deletes"))
    { type = 1 ;
      goto getGeneName ;
    }

  if(!strcmp(cp, "includes"))
    { type = 2 ;
      goto getGeneName ;
    }

  if(strcmp(cp, "does"))
    {
      graphText
	(messprintf("Bad text %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }

  freenext() ;
  cp = freewordcut(" ",&cutter) ;
  if(cutter != ' ' ||
     strcmp(cp, "not") )
    {
      graphText
	(messprintf("Bad text %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }

 
  freenext() ;
  cp = freewordcut(" ",&cutter) ;
  if(cutter != ' ')
    {
      graphText
	(messprintf("Bad text %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }

  if(!strcmp(cp, "delete"))
    { type = 3 ;
      goto getGeneName ;
    }

  if(!strcmp(cp, "include"))
    { type = 4 ;
      goto getGeneName ;
    }

 getGeneName:
  freenext() ;
  cp = freewordcut(".",&cutter) ;
  if(cutter != '.' ||
      strcmp(cp, name(gene)))
    { 
      graphText
	(messprintf("Bad gene %s in genotype of %s : %s",
		    cp ? cp : "NULL",
		    name(cross), text),     1, gLine++ ) ;
      while(cp = freeword()) ;
      goto abort ;
    }
  while(cp = freeword()) ;

   /****************** Save the interpreted results **********/

 switch(type)
    {
    case 1:   /* deletes */
      bsAddTag(Cross, _A_deletes_B) ;
      break ;
    case 2:   /* includes */
      bsAddTag(Cross, _A_includes_B) ;
      break ;
    case 3:   /* does not delete */
      bsAddTag(Cross, _A_does_not_delete_B) ;
      break ;
    case 4:   /* does not include */
      bsAddTag(Cross, _A_does_not_include_B) ;
      break ;
    default:
      goto abort ;
    }

  bsSave(Cross) ;
  return TRUE ;

 abort:
  while(cp = freeword()) ;
  bsDestroy(Cross) ;

  return FALSE ;
}

/*********************************************************/

static void getAllDfDpResults(void)
{
  KEY key = 0;
  int nn = 0 ;
   KEYSET bad = keySetCreate() ;
  int nBad = 0 ;
 
  graphActivate(gCptGraph) ;

  while(lexNext(_VDf_Dup_data, &key))
    if( KEYKEY(key) && 
	( (lexGetStatus(key) & CALCULSTATUS) 
       /* i.e. key has been modified */
       || doRecompute)    /* forced calcul */
       )
      if( getDfDpResult(key))
	nn++ ;
      else
	keySet(bad, nBad++) = key ;

  graphText(
	    messprintf("Found %d usable Df/Dp results out of %d",
		       nn, lexMax(_VDf_Dup_data)),
	    3, 1) ;
  graphTextBounds(60,gLine+3) ;
  graphRedraw() ;
  if(keySetMax(bad))
    { lexaddkey("Bizare DfDp", &key, _VKeySet) ;
      arrayStore(key, bad, "k") ;
    }
  arrayDestroy(bad) ;
}

/**********************************************/
/**********************************************/

static BOOL dist2Pbs(KEY key,CA ca)
{
  OBJ obj = bsCreate(key) ;

  float p, dp ;
  BOOL done = FALSE ;
  KEY g1, g2 ;  
  if(!obj) return FALSE ;
  
  if( bsGetKey(obj,_Gene1, &g1)
     && bsGetKey(obj,_Gene2, &g2)
     && bsGetData(obj, _ACEDB_dist,_Float,&p))
    {
      bsGetData(obj,_bsRight,_Float,&dp) ;
      lexUnsetStatus(key,CALCULSTATUS) ;
      done = TRUE ;
      ca->g1 = g1 ;
      ca->g2 = g2 ;
      ca->p = p;
      ca->dp = dp;
    }
  
  bsDestroy(obj) ;
  return done ;
}

/**************************************/

   /* Construct  a reoodeded Cross Set */
static void reorder_cA(void)
{
  Array cc, c1, c2 ;
  KEY cross ;
  int i, j ;
  
  lexaddkey("Cross_Array", &cross,_VCalcul) ;
  cc  = arrayGet(cross,CASt, castFormat) ;
  if(!cc)
    {
      messout(" First Construct the Array of 2 point data") ;
      return ;
    }
  c1 = cc ; c2 = arrayCopy(cc) ;
  
  { register int i = arrayMax(c2) ;
    while (i--)
      { register CA cp = arrp(c2,i,CASt) ;
	register KEY k = cp->g2 ;
	cp->g2 = cp->g1; cp->g1 = k;
      }
  }

  i = j = arrayMax(c2) ;
  arrayExtend(c1,i+j) ;
  while(i--)
   arr(c1,i + j, CASt) = arr(c2,i,CASt) ;
  arraySort(c1,firstGeneOrder) ;
  
  lexaddkey("Cross_Doubled_Array", &cross,_VCalcul) ;

  arrayStore(cross, c1, castFormat) ;

  arrayDestroy(c1) ;
  arrayDestroy(c2) ;
}

/*********************************************************/
/*********************************************************/

void construct_cA(void)
{
  Array cc ; CASt c ;
  KEY cross , key = 0;
  int n, i ;
  float x ;
  Array plot = arrayCreate(50,int) ;
  
  graphActivate(gCptGraph) ;

  lexaddkey("Cross_Array", &cross,_VCalcul) ;
  cc  = arrayGet(cross,CASt, castFormat) ;
  if(!cc)
    cc = arrayCreate(50,CASt) ;
  n = arrayMax(cc) ;
  while(lexNext(_V2_point_data, &key))
    if((KEYKEY(key) && lexGetStatus(key) & CALCULSTATUS) 
       /* i.e. key has been modified */
       || doRecompute)    /* forced calcul */
      if( dist2Pbs(key,&c))
	array(cc, n++,CASt) = c ;
  
  arraySort(cc, distanceOrder) ;
  i = arrayMax(cc) ;
  while(i--)
    {
      x= arr(cc,i,CASt).p ;
      x = 100*x ; 
      if(x>50) x = 50 ;
      if(x<0) x = 0 ;
      array(plot,(int)x,int) ++ ;
    }
  graphText(
	    messprintf("Found %d usable 2 point dta out of %d",
		       arrayMax(cc), lexMax(_V2_point_data)),
	    3, 1) ;
  graphTextBounds(60,gLine+3) ;
  graphRedraw() ;
  plotHisto("Distances",plot) ;
  arrayStore(cross, cc, castFormat) ;
  reorder_cA() ;
}

/*********************************************************/

static void construct_gA(void)
{
  Array cA, gA ;
  KEY cross , gak ;
  KEYSET gg ;
  int  i, j = 0 ;

  graphActivate(gCptGraph) ;

  lexaddkey("Gene_Array", &gak,_VCalcul) ;
  lexaddkey("Cross_Array", &cross,_VCalcul) ;

  gA = arrayGet(gak, GASt, gastFormat) ;
  if(!gA)
    gA = arrayCreate(50,GASt) ;
  cA  = arrayGet(cross,CASt, castFormat) ;
  if(!cA)
    {
      arrayDestroy(gA) ;
      messout("First create the cA array") ;
      return ;
    }


  gg = keySetCreate() ;
  for(i=0; i<arrayMax(cA);i++)
      { keySet(gg,j++) = arr(cA,i,CASt).g1 ;
	keySet(gg,j++) = arr(cA,i,CASt).g2 ;
      }

  keySetSort(gg) ;
  keySetCompress(gg) ;

  arrayMax(gA) = 0 ;
  for(i=0; i<keySetMax(gg);i++)
    array(gA,i,GASt).g = keySet(gg,i) ;
  arrayStore(gak, gA, gastFormat) ;
  arrayDestroy(gA) ;
  arrayDestroy(cA) ;
  
  gCptGraph = graphCreate (TEXT_SCROLL,"Genetics", 0.0,0.0,0.55,0.35) ;
  graphCreate(TEXT_FIT,"Genes Found",.8,.7,.5,.3) ;
  keySetShow(gg,0) ;
}

/*********************************************************/

static void computeLinkageGroups(void)
{

  Array cA , gA, plot ;
  KEY cross , gak ;
  KEYSET gg = keySetCreate() ;
  int i , j, threshold , nLG ;
  GA ga, gb ; CA ca ;  float x , xmax ;
  
  if(!graphPrompt
  (" Threshold for the definition of linkage groups in cM",
   "40","i"))
    return ;

  freeint(&threshold) ;
  xmax = threshold/100. ;

  graphActivate(gCptGraph) ;

  lexaddkey("Gene_Array", &gak,_VCalcul) ;
  lexaddkey("Cross_Doubled_Array", &cross,_VCalcul) ;
  
  gA = arrayGet(gak,GASt, gastFormat) ;
  cA = arrayGet(cross,CASt, castFormat) ;
  if(!cA || !gA)
    {
      messout("First create the cA and gA arrays") ;
      return ;
    }

  for(i=0; i<arrayMax(gA);i++)
    arr(gA,i,GASt).n = 0;

  nLG = 0 ;

  for(i=0; i<arrayMax(cA);i++)
    { ca = arrp(cA,i,CASt) ;
      j = geneFind(gA,ca->g1) ;
      ga = arrp(gA,j,GASt) ;
      if(!ga->n)
	{
	  ga->n = ++nLG ;
	  graphText(messprintf("New LG %d : %s", nLG,
	      name(ca->g1)), 2,gLine++) ;
	}
      x = ca->p ;
      if(ca->p < xmax)
	{
	  j = geneFind(gA,ca->g2) ;
	  gb = arrp(gA,j,GASt) ;
	  if(!gb->n)
	    {
	      gb->n = ga->n ;
	      graphText(messprintf(": %s",
	      name(ca->g2)), 2,gLine++) ;
	    }
	  if(ga->n != gb->n)
	    graphText(messprintf("linkage group contradiction %s=%d %s=%d",
      name(ca->g1), ga->n,name(ca->g2), gb->n),3,gLine++);
	}
    }

       /* plot will be destroyed by plotHisto */

  plot = arrayCreate(12,int) ;
  i = arrayMax(gA) ;
  while(i--)
    array(plot,array(gA,i,GASt).n,int) ++ ;


  graphTextBounds(60,gLine+3) ;
  graphRedraw() ;
  plotHisto("Linkage Groups",plot) ;

  arrayDestroy(cA) ;
  arrayStore(gak,gA, gastFormat) ;
  arrayDestroy(gA) ;
  keySetDestroy(gg) ;
}

/*********************************************************/
/*********************************************************/

static void gCptDestroy (void)
{
  GCPTGET("gCptDestroy");

  look->magic = 0 ;

  messfree(look) ;
  gCptGraph = 0 ;
}

/**********************************/

MENUOPT gCptMenu[] =
{   graphDestroy,"Quit",
    help, "Help",  
    getAll2pResults,"get 2p results",
    getAll3pResults,"get 3p results",
    getAllDfDpResults,"get DfDp results",
    0,0,
    construct_cA, "2 point data",  
    construct_gA, "list Genes",  
    computeLinkageGroups,"Linkage Groups",
    0,0
    }
 ;

/*************************************************************/

static void gCptDisplay(void)
{ 
  GCPTGET("gCptDisplay") ;
 
  graphActivate(look->graph) ;
  graphClear() ;
  graphButtons(gCptMenu,3.,1., 50) ;

  gLine = 8 ;
  graphTextBounds(60,200) ; /* gLine+3) ;*/
  graphRedraw() ;
}

/*********************************************************************/
/********************  public routines   *****************************/

void gMapCompute(void)
{ LOOK look ;
  if(graphActivate(gCptGraph))
    { 
      graphPop() ;
      return ;
    }

  gCptGraph = graphCreate (TEXT_SCROLL,"Genetics", 0.0,0.0,0.55,0.35) ;
  if (!gCptGraph)
      return ;
 /* provision for multi windows */

  look=(LOOK)messalloc(sizeof(struct LOOKSTUFF));
  look->magic = MAGIC;
  look->graph = gCptGraph ;

  graphAssociate(gMapCompute,look) ;
  graphMenu(gCptMenu) ;
  graphHelp("Genetics") ;
  graphRegister (DESTROY,gCptDestroy) ;
  graphRegister(RESIZE,(GraphFunc)gCptDisplay);

  gCptDisplay() ;
}

/*************************************************************/
/*************************************************************/




