/*  File: alignmaps.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 **  Realigns the various maps in dna unit                    **
 * Exported functions:
 * HISTORY:
 * Last edited: Apr  3 15:46 1992 (mieg)
 * * Dec 11 12:18 1991 (mieg): myText for non interactive messages
 * Created: Fri Nov 29 14:31:10 1991 (mieg)
 *-------------------------------------------------------------------
 */


#include "acedb.h"
#include "keyset.h"
#include "graph.h"
#include "sysclass.wrm"
#include "systags.wrm"
#include "tags.wrm"
#include "classes.wrm"
#include "session.h"
#include "a.h"
#include "bs.h"
#include "dna.h"
#include "lex.h"
#include "plot.h"
#include "query.h"
#include "regression.h"

static Graph alignGraph = 0 ;
static int line = 0 ;
static Array xx = 0 ;

#include "display.h"	/* must come after ALIGN definition */

static void alignDisplay(void) ;
void alignMaps (void) ;


#define ALIGNGET(name)   if (!graphActivate(alignGraph)) \
		            messcrash ("(%s) lost its graph.",name) ; 

/************************************************************************/
/************************************************************************/

   /* Graphic or non graphic printouts */

static void textGraphic(char *cp, int x, int y)
{
  line += y ;
  graphText(cp, x, line) ;
}

static void textNonGraphic(char *cp, int x, int y)
{
  if(!y)
    fprintf(stderr,"  ") ;
  while(y--)
    putc('\n',stderr) ;
  fprintf(stderr,cp) ;
}

static void (*myText)(char *, int,int) ;

/*************************************************************************/
     typedef struct { KEY contig; int length, pos ;
		      float a, b, mid, from, to ;} YC ;
     typedef struct { KEY clone ;
		      KEY situCh ;
		      float inSitu ;
		      KEY gene ;
		      KEY gChromo ;
		      float gPos ;
		      KEY dna ;
		      int pMap1 , pMap2, length ;
		      int bMap1, bMap2 ;
		    } XX ;

static int XXOrder(void *a, void *b)
{ return  
    ((XX *)a)->pMap1  - ((XX *)b)->pMap1 ;
}

static int genesOrder(void *a, void *b)
{ float x = ((XX *)a)->gPos  - ((XX *)b)->gPos ;
  if(x>0) return  1 ;
  else if(x<0) return -1 ;
  else return 0 ;
}

static int contigOrder(void *a, void *b)
{ float x = ((YC *)a)->mid - ((YC *)b)->mid ;
  if (x<0) return -1;
  else if(x>0) return 1;
  return 0 ;
}

/******************************/
    /* Gathers the data about a contig */
static BOOL gatherData(KEY contig) 
{
  OBJ  Clone, Gene ;
  int i, j = 0 ;
  KEY dummy , clone ;
  XX *x ;
  float y ;
  Array clones ; Stack DNA ;
  
  arrayDestroy(xx) ;
  xx =  arrayCreate(50,XX) ;
  myText(name(contig), 1, 1) ;
  clones = queryKey(contig,">Clone") ;
  i = arrayMax(clones) ;
  while(i--)
    { if(Clone = bsUpdate(clone = keySet(clones,i) ))
	 { x = arrayp(xx,j++,XX) ;
	   x->clone = keySet(clones,i) ;
	   x->situCh = 0 ;
	   bsGetKey(Clone,_Chromosome,&(x->situCh)) ;
	   x->inSitu = 9999 ;
	   if(bsGetData(Clone,_In_Situ,_Float,&y))
	     x->inSitu = y ;
	   x->gene = 0 ;
	   x->gChromo = 0 ;
	   x->gPos = 9999 ;
	   if(bsGetKey(Clone,_Gene,&(x->gene)))
	     { if (Gene = bsCreate(x->gene))
		 {  if(bsGetKey(Gene,_gMap,&(x->gChromo) ) &&
		      bsGetData(Gene,_bsRight,_Float,&y))
  /* Disconsider mapping data without justifications, 
     some are actually inferred from the physical map so
     it would be circular to use these to position the contigs
     */
		      if(bsGetKey(Gene,_2Point,&dummy)  || 
			 bsGetKey(Gene,_3Point,&dummy)  ||
			 bsGetKey(Gene,_Df_Dup,&dummy) )
			x->gPos = y ;
		    bsDestroy(Gene) ;
		  }
	     }
	   if(bsGetKey(Clone,_bsDown,&dummy))
	     myText(messprintf("%s also contain gene %s ",
			       name(clone),
			       name(dummy)),
		       3, 1) ;
	   bsGetKey(Clone,_DNA,&(x->dna)) ;
	  if(x->dna && (DNA = stackGet(x->dna)))
	    { x->length = stackMax(DNA) ;
	      bsAddData(Clone,_Length,_Int,&(x->length)) ;
	      stackDestroy(DNA) ;
	    }
	   if(bsGetKey(Clone,_pMap,&dummy) &&
	      bsGetData(Clone,_bsRight,_Int,&(x->pMap1)) &&
	      bsGetData(Clone,_bsRight,_Int,&(x->pMap2) )) ;
	   else
	     x->pMap1 = x->pMap2 = 9999 ;
	   bsSave(Clone) ;
	 }
    }
  arraySort(xx,XXOrder) ;
  i = arrayMax(xx) ;
  while(i--)
    { x = arrayp(xx,i,XX) ;
      if(! x->gChromo && ! x-> situCh)
	continue ;
      myText(name(x->clone), 1, 1) ;
      if(x->situCh)
       {  myText(name(x->situCh), 10, 0) ;
	  if(x->inSitu != 9999)
	    myText(messprintf("%f",x->inSitu), 13, 0 ) ;
	}
      if(x->gene)
	{ myText(name(x->gene), 22, 0 ) ;
	  if(x->gChromo)
	    { myText(name(x->gChromo), 32, 0 ) ;
	      if(x->gPos != 9999)
		myText(messprintf("%f",x->gPos), 36, 0 ) ;
	    }
	}
      myText(messprintf("%d",x->pMap1), 50, 0 ) ;
      myText(messprintf("%d",x->pMap2), 60, 0 ) ;
    }
  return TRUE ;
}

/******************************/
    /* Bumps the contigs along the gMap */
static void bumpContigChromo(Array ctgS, KEY chromo)
{
  OBJ Contig ;
  KEY contig , dummy ;
  int i, j , j1 , length , pos ;
  YC * yc , *yc1, *yc2 ;
  Array ycA ;
  float from, to, a, b , x ;
  
  /* Get all data */
  i = keySetMax(ctgS) ;
  ycA = arrayCreate(10,YC) ;
  j = 0 ;
  while(i--)
    if( Contig = bsCreate(contig = keySet(ctgS,i)) )
      { if(bsGetData(Contig,_b2g,_Float,&a) &&
	   bsGetData(Contig,_bsRight,_Float,&b) &&
	   bsGetData(Contig,_Length,_Int,&length) &&
	   bsGetKey(Contig,_gMap,&dummy) &&
	   bsGetData(Contig,_bsRight,_Float,&from) &&
	   bsGetData(Contig,_bsRight,_Float,&to) )
	  { yc = arrayp(ycA,j++,YC) ;
	    yc->contig = contig ;
	    if(a<=0) a = .000001 ; /* better than nothing */
	    yc->mid = a* length/2. + b ;
	    yc->length = length ;
	    yc->a = a ;
	    yc->b = b ;
	    if(to < from)
	      { x = from ; from = to; to = x ;}
	    if (from == to)
	      { from -= .1; to += .1 ; }
	    yc->from = from ;
	    yc->to = to ;
	  }
	bsDestroy(Contig) ;
	myText
	  (messprintf("%s    from %f  to %f",
		      name(contig),from, to),
	   3, 1) ;
      }
  arraySort(ycA,contigOrder) ;
  
  /* Study the intervals */
  for(i=1 ; i<arrayMax(ycA); i++)
    { yc1 = arrp(ycA,i-1,YC) ;
      yc2 = arrp(ycA,i, YC) ;
      if(yc1->from > yc2->from)  /* then 1 is short since mid1 < mid2 */
	{ yc2->from = yc1->to + .2 ;
	}
      else if (yc1->to > yc2->to) /* 1 is longer */
	{ yc1->to = yc2->from - .2 ;
	}
      else if(yc1->to > yc2->from) /* cas chevauchant */
	{ x = (yc1->to - yc2->from + .2)/ 2. ;
	  yc1->to -= x ;
	  yc2->from += x ;
	}
    }
/* Reorder the contigs and reset their midpoints */
  for(i=arrayMax(ycA), yc1 = arrp(ycA,0,YC); i--; yc1++)
    yc1->mid = (yc1->from + yc1->to) / 2. ;
  arraySort(ycA,contigOrder) ;
    
/* Verify that the contig ends are not mispositioned with respect to the mapped clones */
  { Array genes = queryKey(chromo,">Gene Clone && gMap") ;
    KEY gene, clone, contig ;
    OBJ Gene, Clone ;
    float x ;
    Array gg = arrayCreate(arrayMax(genes),XX) ; /* all cloned genes on this chromo */

    i = arrayMax(genes) ; j = 0 ;
    while(i--)
      { gene = keySet(genes, i) ;
	if(Gene = bsCreate(gene))
	  { if(bsGetKey(Gene,_gMap,&dummy) &&
	       bsGetData(Gene,_bsRight,_Float,&x) &&
	       bsGetKey(Gene,_Clone,&clone) &&
	       (Clone = bsCreate(clone)) )
	      { if(bsGetKey(Clone,_pMap,&contig))
		  { array(gg,j,XX).gPos = x ;
		    array(gg,j,XX).gene = gene ;
		    array(gg,j,XX).gChromo = contig ;
		    j++ ;
		  }
		bsDestroy(Clone) ;
	      }
	    bsDestroy(Gene) ;
	  }
      }			   
    arraySort(gg,genesOrder) ;

    if(arrayMax(gg))
      for(i=0 ; i<arrayMax(ycA) ; i++)
	{
	  /* start from contig center
	     scan genes belonging to this contig
	     this gives a minimum limit
	     */
	  x = arr(ycA,i,YC).mid ;
	  contig = arr(ycA,i,YC).contig ;
	  for(j=0 ; j<arrayMax(gg) ; j++)
	    if(arr(gg,j,XX).gPos > x)
	      break ;
	  if(j == arrayMax(gg))
	    j-- ;
	  if(arr(gg,j,XX).gChromo != contig) /* The one just down */
	    { if(j) j-- ;                    /* the one just up */
	      if(arr(gg,j,XX).gChromo != contig)
		continue ;  /* Hopeless case */
	    }
	     /* Look down Stream */
	  for(j1 = j; j1<arrayMax(gg) && arr(gg,j1,XX).gChromo == contig ; j1++) ;
	  x = arr(gg,j1-1,XX).gPos ;
	  if(arr(ycA,i,YC).to < x)
	    { arr(ycA,i,YC).to = x + .1 ;
	      if(i+1 < arrayMax(ycA) && arr(ycA,i+1,YC).from < x)
		arr(ycA,i+1,YC).from = x + .3 ;
	    }
	     /* Look up Stream */
	  for(j1 = j; j1 >= 0 && arr(gg,j1,XX).gChromo == contig ; j1--) ;
	  x = arr(gg,j1+1,XX).gPos ;
	  if(arr(ycA,i,YC).from > x)
	    { arr(ycA,i,YC).from = x - .1 ;
	      if(i-1 >= 0 && arr(ycA,i-1,YC).to > x)
		arr(ycA,i-1,YC).to = x - .3 ;
	    }
	}
    arrayDestroy(gg) ;
    arrayDestroy(genes) ;
  }
      

  /* recompute the b2g values */
  for(i=0 ; i<arrayMax(ycA); i++)
    { yc = arrp(ycA,i,YC) ;
      yc->a = (yc->to - yc->from)/ yc->length ;
      yc->b = yc-> from ;
    }
  
  /* Measure the intervals and
     position the contigs in b units */
  length = arr(ycA,0,YC).length  ; /* running total */
  for(i=1 ; i<arrayMax(ycA); i++)
    { yc1 = arrp(ycA,i-1,YC) ;
      yc2 = arrp(ycA,i, YC) ;
      a = (yc1->a + yc2->a) /2 ;
      if(!a) a = 1./1000000 ;
      length += (yc2->from - yc1->to) / a ;
      yc2->pos = length ;     /* zeroth is implicitly zero */
      length += yc2->length ;
    }

  myText("Moved to", 40, 1) ;
  /* Save all data */
  for(i=0, yc = arrp(ycA,0,YC); i<arrayMax(ycA); yc++ , i++ )
    if( Contig = bsUpdate( yc->contig ))
      { a = yc->a ;
	b = yc->b ;
	from = yc->from ;
	to = yc->to ;
	pos = yc->pos ;
	bsAddData(Contig,_b2g,_Float,&a) ;
	bsAddData(Contig,_bsRight,_Float,&b) ;
	bsAddData(Contig,_Position,_Int,&pos) ;
	bsAddKey(Contig,_gMap,chromo) ;
	bsAddData(Contig,_bsRight,_Float,&from) ;
	bsAddData(Contig,_bsRight,_Float,&to) ;
	bsSave(Contig) ;
	myText
	  (messprintf("%s    from %f  to %f",
		      name(contig),from, to),
	   3, 1) ;

      }
  arrayDestroy(ycA) ;
}

extern  void  gMapMakeAll(void) ; /* in gmapdisp */
/* Bumps the contigs along the gMap */
static void bumpContigs2(void)
{
  KEY kkch = 8, chromo ;
  KEYSET ctgS ;
  
  while(--kkch)  /* Not chromosome 0 */
    {
      line += 3 ;
      chromo = KEYMAKE(_VChromosome,kkch) ;
      myText(name(chromo), 1, 2) ;
      ctgS = queryKey(chromo,">Contig  Length && b2g ;") ;
      if (keySetMax(ctgS) > 1)
	bumpContigChromo(ctgS, chromo) ;
      keySetDestroy(ctgS) ;
    }
}

static void bumpContigs(void)
{
  alignDisplay() ;
  bumpContigs2() ;
  graphTextBounds (80, line += 2 ) ;
  graphRedraw() ;
}

/*************************************************************************/
/*************************************************************************/

static BOOL contig2gMap(KEY contig)
{
  OBJ  Contig ;
  KEY chromo ;
  int i, np , length ;
  float y ;
  double aa, bb, rr, ww ;
  Array xy = arrayCreate(50,POINT) ;
  XX *x ;

  np = 0 ;
  i = arrayMax(xx) ;
  while(i--)
    { x = arrp(xx,i,XX) ;
      if(x->pMap1 != 9999  && x->pMap2 != 9999
	 && x->gPos != 9999 )
	{ array(xy,np,POINT).x = (double)(x->bMap1 + x-> bMap2)/2. ;
	  array(xy,np,POINT).y = (double) x->gPos ;
	  np++ ;
	}
    }
  
  if(np)
	{ aa = .000001 ;
	  linearRegression(xy,&aa, &bb,&rr,&ww) ;
	  Contig = bsUpdate(contig) ;
	  length = 0 ;
	  bsGetData(Contig,_Length,_Int,&length) ;
	  y = (float) aa ;
	  bsAddData(Contig,_b2g,_Float,&y) ;
	  y = (float) bb ;
	  bsAddData(Contig,_bsRight,_Float,&y) ;
          bsGetKey(Contig, _gMap, &chromo) ;
	  y = (float) bb ;
	  bsAddData(Contig,_bsRight,_Float,&y) ;
	  y = (float) ( aa * length + bb) ;
	  bsAddData(Contig,_bsRight,_Float,&y) ;
	  bsSave(Contig) ;
	  myText
	    ( messprintf
	     ("%s %d genes, %s, y = %lg x + %lf",
	      name(chromo), np,
	      name(contig), aa,bb),
	     2, 1 ) ;
	}
  arrayDestroy(xy) ;
  return TRUE ;
}

/*********************************/

static BOOL contigLengths(KEY contig)
{
  OBJ  Contig ;
  int i , length, from, to ;
  XX *x ;

       /* look for extrema of pMap */
  to = from = arrp(xx,0,XX)->pMap1 ;
  i = arrayMax(xx) ;
  while(i--)
    { x = arrp(xx,i,XX) ;
      if(to<x->pMap2 && x->pMap2 != 9999)
	to = x->pMap2 ;
      x->bMap1 = 4096*(x->pMap1 - from) ;
      x->bMap2 = 4096*(x->pMap2 - from) ;
    }
  /*     
     I should also treat YACs and dna length exactly when the sequence
     is known


  to = -(1<<29) ;
  for(j=0;j<arrayMax(xcA); j++)
    { if(to < array(xcA,j,XC).from) 
	ilot++ ;
      array(xcA,j,XC).ilot = ilot ;
      if (to < array(xcA,j,XC).to)
	to = array(xcA,j,XC).to 
      x->length =  4096 * (x->pMap2 - x->pMap1) ;
    }
     */  
  /* 4096 is the distance between hind3 sites, also used in
     geometrical picking from gMap and vice versa in pmapDisp
     It is used also in pMapToGMap
  */
  
  Contig = bsUpdate(contig) ;
  length =  4096 * (to - from );
  bsAddData(Contig,_Length,_Int,&length) ;
  bsAddData(Contig,_pMap,_Int,&from) ;
  bsAddData(Contig,_bsRight,_Int,&to) ;
  bsSave(Contig) ;
  
  return TRUE ;
}

/*************************************************************************/
  /* attributes the contigs to the chromosomes */
static BOOL contig2chromosome(KEY contig)
{
  OBJ Contig ;
  int  i ;
  XX *x ;
  KEY ch, c2 ;
  
  i = arrayMax(xx) ;
  ch = 0 ;
  while(i--)
    { x = arrp(xx,i,XX) ;
      if(c2 = x->situCh)
	if(ch)
	  { if (ch != c2)
	      { myText
		  (messprintf
		   ("Two chromosomes %s %s for contig %s ",
		    name(ch) ,
		    name(c2) ,
		    name(contig) ), 3, 1 ) ;
		ch = 0 ;
		break ;
	      }
	  }
	else
	  ch = c2 ;

      if(c2 = x->gChromo)
	if(ch)
	  { if (ch != c2) /* double assignation */
	      { myText
		  (messprintf
		   ("Two chromosomes %s %s for contig %s ",
		    name(ch) ,
		    name(c2) ,
		    name(contig) ), 3, 1 ) ;
		ch = 0 ;
		break ;
	      }
	  }
	else
	  ch = c2 ;
    }
		  
  if (Contig = bsUpdate(contig))
    { if(ch)
	{ bsAddKey(Contig,_gMap, ch) ;
	  myText
	    (messprintf
	     ("****** Success, contig %s on chromosome %s ***",
	      name(contig),
	      name(ch)),
	     5, 1 ) ;
	}
      else
	{ myText
	   (messprintf
	    ("No chromosome for contig %s",
	     name(contig)), 5, 1 ) ;
	  if(bsFindTag(Contig,_gMap))
	    bsRemove(Contig) ;
	}

       bsSave(Contig) ;
    }
  else
    myText
      (messprintf
       ("No chromosome for contig %s",
	name(contig)), 5, 1 ) ;
  return ch != 0 ;
}

/*****************************************/

static void alignAllContigs2(void)
{
  KEY contig = 0 ;
  
  lexNext(_VContig,&contig) ;  /* skip the class definition */
  while(lexNext(_VContig,&contig))
    { if(iskey(contig) == 2 &&
	 gatherData(contig) &&
	 contig2chromosome(contig) &&
	 contigLengths(contig) &&
	 contig2gMap(contig)) ;
    }

  graphTextBounds (80, line += 1 ) ;
  graphRedraw() ;
}

/*****************************************/

static void alignAllContigs(void)
{
  alignDisplay() ;
  alignAllContigs2() ;
  graphTextBounds (80, line += 1 ) ;
  graphRedraw() ;
}

/*****************************************/
/************* action routines ***********/
/*****************************************/

static void alignDestroy (void)
{    
  arrayDestroy(xx) ;
  alignGraph = 0 ;
  myText = textNonGraphic ;
}

/**********************************************************/

extern void pMapMakeAll(void), gMapMakeAll(void) ;

static MENUOPT alignMenu[] =
  {
   graphDestroy, "Quit",
   help, "Help",
   graphPrint,"Print",
   pMapMakeAll,"Make pMaps",
   gMapMakeAll,"Make gMaps",
   alignAllContigs,"Align contigs",
   bumpContigs, "Bump contigs",
   0, 0 
   } ;

/*************************************************************/

static void alignDisplay(void)
{
  ALIGNGET("alignDisplay") ;
 
  graphPop() ;
  graphClear() ;
  graphButtons(alignMenu,3.,1.,40.) ;
  
  line = 6 ;
  graphTextBounds (80, line += 1 ) ;
  graphRedraw() ;
}

/*************************************************************/
/*********************************************************************/
/********************  public routines   *****************************/

void alignMaps(void)
{
  if(!isWriteAccess())
    { messout("Sorry, you do not have write access.") ;
      return ;
    }

  if(graphActivate(alignGraph))
    { myText = textGraphic ;
      graphPop() ;
      return ;
    }

  alignGraph = displayCreate(DtAlign) ;
  if (!alignGraph)
      return ;

  graphTextBounds (80,50) ;   /* needed to for text box sizing */
  graphRegister (DESTROY,alignDestroy) ;
  
  graphMenu(alignMenu) ;  
 
  xx = arrayCreate(50,XX) ;
  myText = textGraphic ;
  alignDisplay() ;
}

void alignMapsNonInteractive(void)
{
  myText = textNonGraphic ;
  mainActivity("Aligning maps") ;
  
  xx = arrayCreate(50,XX) ;
  myText("Automatic alignment of genetic and physical maps",
	    7,5) ;
  alignAllContigs2() ;
  mainActivity("Bump contigs") ;
  bumpContigs2() ;
  myText = textGraphic ;
}

/*************************************************************/
/*************************************************************/







