/*  File: pmapdisp.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: physical map display
 * Exported functions: pmapDisplay
 * HISTORY:
 * Last edited: Apr  2 17:54 1992 (mieg)
 * * Mar 10 18:22 1992 (mieg): added an extra probe line - changed SHOWY, YMIN
 * * Feb  3 17:00 1992 (mieg): added segFormat, needed now by arrayGet/Store
 * * Jan 31 1992 (neil): add proforma for exported routine pMapGetCMapKeySet();
                         fix bug I introduced to drawing minicontig;
 * * Jan 16 1992 (neil): use showFacXY not separate scale factor for each object;
                         reintroduce menu entry for scaling to allow explicit scale factor;
                         make minicontig X-scale a constant (X0.5);
                         bug in graphical zoom - always increased scale, so:
                         make midbutton Y+ -> increase scale, midbutton Y- -> decrease scale
                         (gmap has X+ -> increase, X- -> decrease);
                         register MIDDLE_DRAG and MIDDLE_UP actions only once, on creation;
                         add routine pMapSetHighlightKeySet() to allow a caller via display 
                         to send a KEYSET to be highlit in pMapDisplay;
 * * Jan 13 01:30 1992 (mieg): Introduced look->mag, suppressed globals showFacXY
                               Horizontal midButton zooming copied from gMap
 * * Jan  6 18:48 1992 (rd): change probe localization to use gridDisp calls
 * * Jan  6 14:50 1992 (rd): add horizontal scale change to menu
 * * Dec 19 1991 (neil): add RESIZE, write pMapResize(); add centre marker;
                         correct virtually all the calls to LOOKGET;
                         add pMapHighlight(), analogous to gMapHighlight(); use keySetFind;
                         make highlight on selection RED, not BLUE, so it means the same everywhere;
                         change "Show All Objects" to "Unhighlight, Revert" (menu);
                         change free() to messfree();
                         add attachBuriedClones(), though it isn't called yet;
                         change compareSeg in line with ANSI qsort and modify for buried clones;
                         remove showWorkRemarks int to flag on LOOK flag reg;
                         add names for menu entry selections;
 * * Dec 16 14:32 1991 (rd): changed hybridisation -> map rules
 * * Dec  1 18:10 1991 (mieg): added intersection with keyset and FLAG_HIDE
 * * Oct 22 18:54 1991 (mieg): adjusted size of gMapBox 
 * * Oct 21 13:46 1991 (mieg): factor 20 in follow sequence
 * * Oct 18 20:38 1991 (mieg): made seqbox >=2,
 added a sequence box if the exists seq(gene(clone))
 * * Oct 14 22:02 1991 (rd): draw mini contig with scroll box
 * * Oct 14 18:45 1991 (rd): use TextFit window instead of MapScroll
 * * Oct  7 22:20 1991 (rd): changed cursor to redraw properly
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "array.h"
#include "graph.h"
#include "key.h"
#include "lex.h"
#include "bs.h"
#include "plot.h"
#include "a.h"
#include "sysclass.wrm"
#include "systags.wrm"
#include "classes.wrm"
#include "tags.wrm"
#include "disptype.wrm"
#include "display.h"
#include "session.h"
#include "keyset.h"
#include "pmap.h"
/*<<--not sure if I need the following - probably not wormtype.h, anyway:*/
#if 0
#include "query.h"
#include "wormtype.h"
#endif
#include "grid.h"

typedef struct LOOKSTUFF
  { int   magic ;        /* == MAGIC */
    KEY   key ;
    int   activebox, min, max ;
    int   winHeight, winWidth, scrollBox ;
    float centre ;
    float cursor ;     /* position of the vertical Xor bar */
    unsigned int flag; /* for switches dealing with the display entity, eg when "Show Buried Clones" gets selected*/
    Array segs ;       /* content of all possible boxes : SEG */
    Array box2seg ;    /* displayed boxes : SEG* */
  } *LOOK ;

#define MAGIC  8273468

/*..... ..... ..... .....macros for handling flags:
*/

#define ClearFlagsReg(__obj) (__obj)->flag=(unsigned int) 0
#define SetFlag(__obj, __m) (__obj)->flag|=(__m)
#define ClearFlag(__obj, __m) (__obj)->flag&= ~(__m)
#define ToggleFlag(__obj, __m) \
  { \
      unsigned int __f=(__obj)->flag; \
      (__obj)->flag=(__f&~(__m))|((__f&(__m))^(__m)); \
  }
#define IsSetFlag(__obj, __m) (((__obj)->flag & (__m))!=(unsigned int) 0)
#define IsClearFlag(__obj, __m) (((__obj)->flag & (__m))==(unsigned int) 0)

/*..... ..... ..... .....masks for LOOK flags:
*/

#define FLAG_SHOW_BURIED_CLONES 0x00000001
#define FLAG_BURIED_CLONES_ATTACHED 0x00000002
#define FLAG_SHOW_WORK_REMARKS 0x00000004
#define FLAG_REVERT_DISPLAY 0x00000008

/*..... ..... ..... .....names for menu options:
*/

#define MENU_QUIT               1
#define MENU_HELP               (MENU_QUIT+1)
#define MENU_PRINT              (MENU_QUIT+2)
#define MENU_PRESERVE           (MENU_QUIT+3)
#define MENU_REDRAW             (MENU_QUIT+4)
#define MENU_RECALCULATE        (MENU_QUIT+5)
#define MENU_SHOW_REMARKS       (MENU_QUIT+6)
#define MENU_SHOW_SELECTED      (MENU_QUIT+7)
#define MENU_HIGHLIGHT_SELECTED (MENU_QUIT+8)
#define MENU_REVERT_DISPLAY     (MENU_QUIT+9)
#define MENU_SHOW_BURIED_CLONES (MENU_QUIT+10)
#define MENU_GENETIC_MAP        (MENU_QUIT+11)
#define MENU_SEQUENCE           (MENU_QUIT+12)
#define MENU_SCALE_CHANGE	(MENU_QUIT+13)

/*..... ..... ..... .....the menu:
*/

#define MENU_NO_OF_ENTRIES 13
  /* **NB update this when the menu is changed*/

static FREEOPT options[] =
{
    MENU_NO_OF_ENTRIES, "Map ?",

    MENU_QUIT, "Quit",
    MENU_HELP, "Help",
    MENU_PRINT, "Print",
    MENU_PRESERVE, "Preserve",
    MENU_REDRAW, "Redraw",

    MENU_RECALCULATE, "Recalculate",
    MENU_SHOW_REMARKS, "Show All Remarks",
    MENU_SHOW_SELECTED,"Show Selected Objects",
    MENU_HIGHLIGHT_SELECTED, "Highlight Selected Objects",
    MENU_REVERT_DISPLAY,"Unhighlight, Revert",

#if 0
    MENU_SHOW_BURIED_CLONES, "Show Buried Clones",
#endif
    MENU_GENETIC_MAP, "To Genetic Map",
    MENU_SEQUENCE, "Sequence",
    MENU_SCALE_CHANGE, "Change scale",
};

/*..... ..... ..... .....
*/

typedef struct
  { KEY key ;
    KEY parent ;		/* clone associated with this item */
    int x, dx ;
    unsigned int flag ;
  } SEG ;
#define segFormat "2k3i"

/*..... ..... ..... .....masks for SEG flags:
*/

#define FLAG_FINGERPRINT     0x00000001
#define FLAG_CANONICAL	     0x00000002
#define FLAG_MORE_TEXT       0x00000004
#define FLAG_HIDE            0x00000008
#define FLAG_COSMID_GRID     0x00000010
#define FLAG_YAC_GRID        0x00000020
#define FLAG_WORK_REMARK     0x00000040
#define FLAG_PROBE	     0x00000080
#define FLAG_HIGHLIGHT       0x00000100
#define FLAG_IS_BURIED_CLONE 0x00000200

/*..... ..... ..... .....static routines:
*/

static char* pMapName(KEY key) ;
static char pMapNameBuffer[128] ;
static void mapDestroy (void) ;
static void mapMenu (KEY k) ;
static void pMapPick  (int box, double x , double y) ;
static void pMapMiddleDown (double x , double y) ;
static void pMapMiddleDrag (double x, double y); 
static void pMapMiddleUp (double x, double y);
static void mapKbd (int k) ;
static void pMapShowAll(void) ;
static void pMapHide(void) ;
static void pMapDraw (LOOK look, KEY key) ;
static Array pmapConvert (KEY contig, KEY pmap) ;
static void pMapShowSegs (LOOK look, KEY key) ;
static int  compareSeg (void *a, void *b);
static void pMapSelectBox (LOOK look, int box) ;
static void pMapFollow  (LOOK look, double x, double y) ;
static void limitsFind (LOOK look) ;
static void pMapScroll (float *px, float *py, BOOL isDone) ;
static void pMapHighlight(KEYSET k, KEY key);
static void pMapResize (void); /*<<--*/
#if 0
static void attachBuriedClones(LOOK look); /*<<--*/
#endif
extern void pMapToGMap (KEY contig ,KEY key, int position) ; /*<<--code for this is in gmapdisp.c*/
/*extern Array pMapFingerPrint(KEY clone) ; */

/*..... ..... ..... .....
*/

static int gMapBox = 0 ;
static float showFacX = 0.4 ;
static float showFacY = 1.3 ;
static float offset ;
static float miniOffset ;

#define SHOWX(z) (showFacX*(z) - offset)
#define UNSHOWX(z) (((z) + offset)/showFacX)
#define SHOWY(z) (1+showFacY*(z))
#define YMINI 46
#define MINIFAC 0.05
#define MINIX(z) (MINIFAC*(z) - miniOffset)

static double oldy , oldDx, oldx;
static BOOL dragFast ;
static LOOK lookDrag = 0 ;
#define DRAGFASTLIMIT 35

#define LOOKGET(name)     LOOK look ; \
  			  if (!graphAssFind (pMapDisplay,&look)) \
		            messcrash ("graph not found in %s",name) ; \
			  if (!look) \
                            messcrash ("%s received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

/*..... ..... ..... .....
*/

static KEYSET pMapPendingKeySet=NULL;

/*..... ..... ..... .....
*/

extern BOOL pMapDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  LOOK look=(LOOK)messalloc(sizeof(struct LOOKSTUFF));
  KEY contig = 0, clone = 0, pmap ;
  int pos ;
  OBJ  Clone ;
  BOOL isProbe = FALSE ;

  if (class(key) == _VClone)	/* use the contig */
    { clone = key ;

      if (!(Clone = bsCreate(clone)))
	goto abort ;

      if (!bsGetKey (Clone,_pMap,&contig) &&
	  !bsGetKey (Clone,_Exact_Match_to,&clone) &&
	  !bsGetKey (Clone,_Approximate_Match_to,&clone) &&
	  !bsGetKey (Clone,_Funny_Match_to,&clone))
	if (bsGetKey (Clone,_Hybridizes_to, &key))
	  { bsDestroy (Clone) ;
	    display (key, clone, 0) ; /* key is a Grid_Clone */
	    goto abort ;
	  }
	else
	  goto abortNoMap ;
      if (clone == from)
	goto abortClone ;
      if (!contig)
	{
	  bsDestroy (Clone) ;
	  if (!(Clone = bsCreate(clone)))
	    goto abort ;
	  if(!bsGetKey (Clone,_pMap,&contig))
	    goto abortNoMap ;
	}
      if (!bsGetData(Clone,_bsRight,_Int,&pos))
	goto abortNoMap ;

      bsDestroy (Clone) ;
      if (isProbe)
	clone = key ;
      key = contig ;
    }

  if (class(key) == _VpMap)
    pmap = key ;
  else if (class(key) == _VContig)
    lexaddkey (name(key), &pmap, _VpMap) ;
  else
    goto abort ;

  if (!(look->segs = arrayGet (pmap, SEG, segFormat)) &&
      (class(key) != _VContig ||
       !(look->segs = pmapConvert (key,pmap))))
    goto abort ;

  look->magic = MAGIC ;
  look->key = pmap ;
  look->activebox = 0 ;
  look->box2seg = arrayCreate (32,SEG*) ;
  look->centre = 0 ;
  if (isOldGraph)
    { mapDestroy () ;
      graphAssRemove (pMapDisplay) ;
      graphRetitle (pMapName (key)) ;
    }
  else 
    { if (!displayCreate (PMAP))
	goto abort ;
      
      graphRetitle (pMapName(key)) ;
      graphRegister(DESTROY, mapDestroy) ;
      graphRegister(RESIZE, (GraphFunc) pMapResize);
      graphRegister(PICK, (GraphFunc)pMapPick) ;
      graphRegister(MIDDLE_DOWN, (GraphFunc)pMapMiddleDown) ;
      graphRegister(MIDDLE_DRAG,(GraphFunc) pMapMiddleDrag) ;
      graphRegister(MIDDLE_UP,(GraphFunc) pMapMiddleUp) ;
      graphRegister(KEYBOARD, (GraphFunc)mapKbd) ;
      graphFreeMenu(mapMenu, options) ;
    }

  graphAssociate (pMapDisplay,look) ;

  if (!clone && ((class(from) == _VClone)
       || (class(from) == _VCalcul)))  /* Geometrical positioning */
    clone = from ;
  limitsFind (look) ;
  if (pMapPendingKeySet!=NULL) /*<<--*/
  {
    pMapHighlight(pMapPendingKeySet, clone); /*<<--calls pMapDraw, so pass 'clone down to it*/
    pMapPendingKeySet=NULL;
  }
  else
  {
    pMapDraw(look, clone);
  }
  return TRUE ;

abortNoMap :
abortClone :
  bsDestroy (Clone) ;
  display (key,from,TREE) ;	/* just show as a TREE */

abort :
  messfree (look) ;
  return FALSE ;
}

void pMapMakeAll (void)	/* to be called from main menu */
{
  KEY contig = 0, pmap ;	/* 0 primes lexNext() */
  Array segs ;

  if(!isWriteAccess())
    { messout("Sorry, you do not have Write Access");
      return ;
    }

  if (lexNext (_VContig,&contig)) /* skip the model */
    while (lexNext (_VContig,&contig))
      { lexaddkey (name(contig), &pmap, _VpMap) ;
	if (segs = pmapConvert (contig,pmap))
	  arrayDestroy (segs) ;
      }
}

/************************************************************/

static void mapDestroy (void)
{
  LOOKGET("mapDestroy") ;

  arrayDestroy (look->segs) ;
  arrayDestroy (look->box2seg) ;

  look->magic = 0 ;
  messfree (look) ;
}

/*************************************************************/

static char* pMapName(KEY key) 
{ KEY contig , chromo ;
  float start , stop ;
  OBJ Contig ;

  strcpy(pMapNameBuffer,name(key)) ;
  if (class(key) == _VContig)
    contig = key ;
  else if (class(key) == _VpMap)
    lexaddkey (name(key), &contig, _VContig) ;
  else
    return pMapNameBuffer ;

  if (Contig = bsCreate(contig))
    { if(bsGetKey(Contig,_gMap,&chromo))
	{ strcpy(pMapNameBuffer,name(chromo)) ;
	  if(bsGetData(Contig,_bsRight,_Float,&start) &&
	     bsGetData(Contig,_bsRight,_Float,&stop) )
	    strcat(pMapNameBuffer,
		   messprintf(":  %4.2f to %4.2f", start, stop)) ;
	  else
	    strcat(pMapNameBuffer, "  position unknown ") ;
	}
      else
	strcpy(pMapNameBuffer, "This contig is not localised") ;
      bsDestroy(Contig) ;
    }
  
  return pMapNameBuffer ;
}

/************************************************************/

static void mapMenu (KEY k)
{
  SEG *seg ;
  KEY contig,key ;
  LOOKGET("mapMenu") ;

  switch ((int) k)
    {
    case MENU_HELP: helpOn("Physical-map"); break;
    case MENU_SHOW_REMARKS:
      ToggleFlag(look, FLAG_SHOW_WORK_REMARKS);
      pMapDraw(look, 0);
      break ;

    case MENU_GENETIC_MAP:
      if (!look->activebox)
	{ messout ("You must have selected a clone or gene first") ;
	  break ;
	}
      if (!lexword2key(name(look->key),&contig,_VContig))
	{ messout ("Sorry, can't find contig object") ;
	  break ;
	}
      seg = arr(look->box2seg,look->activebox,SEG*) ;
      pMapToGMap (contig, seg->key, seg->x) ;
      break ;

    case MENU_PRINT: graphPS("pmap"); break;
    case MENU_PRESERVE: displayPreserve(); break;
    case MENU_REDRAW: limitsFind(look); pMapDraw(look,0); break;
    case MENU_RECALCULATE:
      if (lexword2key (name(look->key),&contig,_VContig))
	{ if (look->activebox)
	    key = arr(look->box2seg,look->activebox,SEG*)->key ;
	  else
	    key = 0 ;
	  arrayDestroy (look->segs) ;
	  look->segs = pmapConvert (contig,look->key) ;
	  pMapDraw (look, key) ;
	}
      break ;

    case MENU_SEQUENCE:
      if (!look->activebox)
	{ messout("First select a clone") ;
	  break ;
	}
      if (!lexword2key(name(look->key),&contig,_VContig))
	{ messout ("Sorry, can't find contig object") ;
	  break ;
	}
      seg = arr(look->box2seg,look->activebox,SEG*) ;
      if(class(seg->key) != _VClone)
	{ messout("First select a clone") ;
	  break ;
	}
      { OBJ Clone ;
	KEY seq ;
	
	if(Clone = bsCreate(seg->key))
	 { if(bsGetKey(Clone,_Sequence,&seq))
	     display (seq,0,0) ;
	   else
	     messout("Sorry, sequence of %s unknown",
		     name(seg->key)) ;
	   bsDestroy(Clone) ;
	 }
      }
      break ;
/*
    case 40 :
      if (!look->activebox)
	{ messout("First select a clone") ;
	  break ;
	}
      if (!lexword2key(name(look->key),&contig,_VContig))
	{ messout ("Sorry, can't find contig object") ;
	  break ;
	}
      seg = arr(look->box2seg,look->activebox,SEG*) ;
      if(class(seg->key) != _VClone)
	{ messout("First select a clone") ;
	  break ;
	}
      { Array histo ;
	Array  fp = pMapFingerPrint(seg->key) ;
	int i ;
	
	if(fp)
	  { i = arrayMax(fp) ;
	    if(i)
	      { histo = arrayCreate(2000,int) ;
		while(i--)
		  array(histo,(int)(array(fp,i,short)),int)++ ;
		plotHisto
		  (messprintf("%s Finger Print",
			      name(seg->key)), 
		   histo) ;
	      }
	    arrayDestroy(fp) ;
	  }
      }
      break ;
*/
    case MENU_SHOW_SELECTED: pMapHide(); break;
    case MENU_HIGHLIGHT_SELECTED: pMapHighlight(NULL, 0); break;
    case MENU_REVERT_DISPLAY: pMapShowAll(); break;
#if 0
    case MENU_SHOW_BURIED_CLONES:
      ToggleFlag(look, FLAG_SHOW_BURIED_CLONES); /*cd be a switch menu entry for this and for "Show Remarks"*/
      attachBuriedClones(look);
      pMapDraw(look, 0);
      break;
#endif
    case MENU_SCALE_CHANGE:
      if (graphPrompt("Increase to stretch, decrease to compress:", messprintf("%.2f",showFacX), "fz"))
      {
        freefloat(&showFacX);
        pMapDraw(look, 0);
      }
      break;
    case MENU_QUIT: graphDestroy(); break;
    }
}


/*..... ..... ..... .....
*/

#if 0
static void attachBuriedClones(LOOK look)
{
    if (IsClearFlag(look, FLAG_BURIED_CLONES_ATTACHED)) /*then check for buried clones..*/
    {
        int nGivenSegs, nSegs, i;
        SEG *seg;
        KEY k, b;
        OBJ parentp /*the associated clone*/, buried_clonep;

#ifdef __TRACE
    int nBuriedClones=0;
#endif

#ifdef __TRACE
    tracef(" --attachBuriedClones:\n");
#endif

        /*..... ..... ..... .....the first seg:
        */

        if (1<(nGivenSegs=arrayMax(look->segs)))
	{

#if 0
    tracef(" --attachBuriedClones: given %d segs in this look: initial whole segs array:\n", nGivenSegs);
    {
        int i;
        SEG *s;
        for (i=1; i<nGivenSegs; i++)
        {
            s=arrayp(look->segs, i, SEG);
            tracef(" --given: '%s'\n", name(s->key));
        }  
    }
#endif

            seg=arrayp(look->segs, 1, SEG);
        }
        else
        {
            return;
        }

        /*..... ..... ..... .....for all segs, find and attach buried clones:
        */

        i=1, nSegs=nGivenSegs;
        do
        {
          if (!((0<iskey(k=seg->key)) && (parentp=bsCreate(k)) && bsGetKey(parentp, _Canonical_for, &b) && (0<iskey(b))))
	  {
/*
              messout();
*/
          }
          else /*at least one valid buried clone: get all buried clones for this seg*/
          {

#if 0
    tracef(" --attachBuriedClones: attaching valid buried clones to [%s (%d)]..\n", name(k), i);
#endif

              do
              {
                  SEG *bseg=arrayp(look->segs, nSegs++, SEG);
                  buried_clonep=bsCreate(b);
                  bseg->key=b; /*the buried clone*/
                  bseg->parent=k; /*nominally*/
                  bseg->x=seg->x; bseg->dx=seg->dx; /*for now make them the same as for the parent clone*/
                  SetFlag(bseg, FLAG_IS_BURIED_CLONE);
                  SetFlag(bseg, FLAG_FINGERPRINT);

#ifdef __TRACE
    nBuriedClones++;
    tracef(" --attachBuriedClones: attached '%s (%d)[%s (%d)]'..\n", name(b), nSegs-1, name(k), i);
#endif

	      }
              while (/*more valid buried clones*/ bsGetKey(parentp, _bsDown, &b) && (0<iskey(b)));
          }
        }
        while (i++<nGivenSegs && ((seg=arrayp(look->segs, i, SEG))!=NULL));

        /*..... ..... ..... .....then sort them all for display
        */

        arraySort(look->segs, compareSeg);

        /*..... ..... ..... .....
        */

        SetFlag(look, FLAG_BURIED_CLONES_ATTACHED);

#ifdef __TRACE
    tracef(" --attachBuriedClones: total segs %d, attached %d (nBuriedClones %d)\
 buried clones giving whole segs array:\n", nSegs, nSegs-nGivenSegs, nBuriedClones);
    {
        int i;
        SEG *s;
        for (i=1; i<nSegs; i++)
        {
            s=arrayp(look->segs, i, SEG);
            if (IsSetFlag(s, FLAG_IS_BURIED_CLONE))
	    {
                tracef(" -- '%s[%s]', coords: (%d+%d)\n", name(s->key), name(s->parent), s->x, s->dx);
	    }
            else
            {
                tracef(" -- '%s', coords: (%d+%d)\n", name(s->key), s->x, s->dx);
            }
            
        }  
    }
#endif

    }
    else
    {
#ifdef __TRACE
        tracef(" --attachBuriedClones: buried clones already attached\n");
#endif
    }

    return;
}
#endif

/**********************************/

static void pMapPick  (int box, double x , double y) 
{
  LOOKGET("pMapPick") ;

  if (!box)
    return ;
  if (box > arrayMax(look->box2seg))
    { messerror("LookPick received a wrong box number");
      return ;
    }

  if (box == look->scrollBox)
    graphBoxDrag (look->scrollBox, pMapScroll) ;
  else if (box == look->activebox)  /* a second hit - follow it */
    pMapFollow (look,x,y) ;
  else                              /* change now */
    pMapSelectBox (look,box) ;
}

/***********************************/

static void mapKbd (int k)
{
  int box, table ;
  SEG *seg ;
  LOOKGET("mapKbd") ;

  if (!look->activebox)
    return ;

  box = look->activebox ;
  if (!(seg = arr(look->box2seg,box,SEG*)))
    return ;
  table = class(seg->key) ;
  switch (k)
    {
    case LEFT_KEY :
      while ( --box > 0 &&
	     (seg = arr(look->box2seg,box,SEG*)) &&
	     class(seg->key) != table) ;
      break ;
    case RIGHT_KEY :
      while (++box < arrayMax(look->box2seg) &&
	     (seg = arr(look->box2seg,box,SEG*)) &&
	     class(seg->key) != table) ;
      break ;
    }

  if (box < arrayMax(look->box2seg) && arr(look->box2seg, box, SEG*))
    { pMapSelectBox (look, box) ;
      pMapFollow (look,0,0) ;
    }
}

/**********************************/

static void limitsFind (LOOK look)
{
  int i, min = 1000000000, max = -1000000000 ;
  Array segs = look->segs ;
  SEG   *seg ;

  for (i = 1 ; i < arrayMax(segs) ; ++i)
    { seg = arrp(segs,i,SEG) ;
      if (seg->x - seg->dx < min)
	min = seg->x - seg->dx ;
      if (seg->x + seg->dx > max)
	max = seg->x + seg->dx ;
    }
  look->min = min ;
  look->max = max ;
}

/******************************/

static void pMapDraw (LOOK look, KEY key)     /* key becomes activebox, unless reverting*/
{
  float y=0.0;

  graphClear () ;
  graphLinewidth (0.10) ;
  
  if (!arrayMax(look->segs))
    return ;

  graphFitBounds (&look->winWidth, &look->winHeight) ;
  ++look->winWidth ;

  graphLine(look->winWidth/2, y, look->winWidth/2, y+0.9);

  pMapShowSegs (look, key) ;

  graphLine(look->winWidth/2, look->winHeight-2.0, look->winWidth/2, look->winHeight-2.0+0.9); /*<<--temporarily*/

  graphRedraw () ;
}

/*********************************************/

static int compareSeg (void *a, void *b)     /* for arraySort() call */

#define S_BEFORE_T -1
#define T_BEFORE_S 1
/*trigraphs miscompiled by gcc (code always evals test to TRUE):
#define OrderBy(__m, __n) ((int) ((__m)<=(__n)? S_BEFORE_T: T_BEFORE_S))
*/
#define OrderBy(__m, __n) if ((__m)<=(__n)) return S_BEFORE_T; else return T_BEFORE_S
#define SameKey(__k, __l) ((__k)==(__l)) /*<<<<<this shd probably be used generally*/
#define IsParentOf(__s, __t) SameKey((__s)->parent, (__t)->parent) /*not sure if this is right*/

/* **NB
  1/ seg[0] gets sorted, so any change to the sort primitive must preserve its pre-eminence
  2/ display is constructed from bottom up, so "after" there means "before" here <neil>
*/

{
    SEG *s=(SEG *) a, *t=(SEG *) b;

    if (IsSetFlag(s, FLAG_IS_BURIED_CLONE) && IsParentOf(s, t)) return S_BEFORE_T;
    else if (IsSetFlag(t, FLAG_IS_BURIED_CLONE) && IsParentOf(t, s)) return T_BEFORE_S;
    else OrderBy(s->x, t->x);
/*was
    OrderBy(s->x, t->x);
*/
}

#undef S_BEFORE_T
#undef T_BEFORE_S
#undef OrderBy
#undef SameKey
#undef IsParentOf

/*..... ..... ..... .....
*/

static Associator probeHash = 0 ;

static void addClone (Array segs, int *pj, KEY clone, OBJ Clone, int pos1)
{
  int	i, j = *pj ;
  KEY   gene, locus, sequence, probe ;
  OBJ   Gene , Locus ;
  int   pos2, pos ;
  static Array flatA = 0 ;
  SEG 	*seg ;
  Array probeArray ;
  GRIDMAP *map ;

  if (!flatA)
    flatA = arrayCreate (8, BSunit) ;

  seg = arrayp(segs,j++,SEG) ;
  seg->key = clone ;
  seg->parent = clone ;
  if (!bsGetData(Clone,_bsRight,_Int,&pos2))
    pos2 = pos1 ;
  seg->x = pos = 0.5 * (pos1 + pos2) ;
  seg->dx = 0.5 * (pos2 - pos1) ;
  seg->flag = 0 ;
  if (bsFindTag (Clone,_Bands))	/* FingerPrint no good because of Flag */
    seg->flag |= FLAG_FINGERPRINT ;
  if (bsFindTag (Clone,_Canonical_for))
    seg->flag |= FLAG_CANONICAL ;
  if (bsFindTag (Clone,_Cosmid_grid)
      || bsFindTag (Clone,_Canon_for_cosmid))
    seg->flag |= FLAG_COSMID_GRID ;
  if (bsFindTag (Clone,_YAC_polytene))
    seg->flag |= FLAG_YAC_GRID ;
  if (bsGetKey (Clone,_Sequence, &sequence))
    { seg = arrayp(segs,j++,SEG) ;
      seg->key = sequence ;
      seg->parent = clone ;
      seg->x = pos ;
      seg->dx = 0.5 * (pos2 - pos1) ;
      if (seg->dx <= 2)
	seg->dx = 2 ;
    }
  if (bsGetKey (Clone,_Gene,&gene)) do
    { seg = arrayp(segs,j++,SEG) ;
      seg->key = gene ;
      seg->parent = clone ;
      seg->x = pos ;
      if(Gene = bsCreate(gene))
	 { if (bsGetKey (Gene,_Sequence, &sequence))
	     { seg = arrayp(segs,j++,SEG) ;
	       seg->key = sequence ;
	       seg->parent = clone ;
	       seg->x = pos ;
	       seg->dx = 4 ; /* Since this is not the clone sequence */
	     }
	   bsDestroy(Gene) ;
	 }
    } while (bsGetKey (Clone,_bsDown,&gene)) ;

  if (bsGetKey (Clone,_Locus, &locus)) do
    { seg = arrayp(segs,j++,SEG) ;
      seg->key = locus ;
      seg->parent = clone ;
      seg->x = pos ;
      if(Locus = bsCreate(locus))
	 { if (bsGetKey (Locus,_Sequence, &sequence))
	     { seg = arrayp(segs,j++,SEG) ;
	       seg->key = sequence ;
	       seg->parent = clone ;
	       seg->x = pos ;
	       seg->dx = 4 ; /* Since this is not the clone sequence */
	     }
	   bsDestroy(Locus) ;
	 }
    } while (bsGetKey (Clone,_bsDown,&locus)) ;

  if (bsFindTag(Clone, _Remark)
      && bsFlatten(Clone,2,flatA))
    for (i = 0 ; i<arrayMax(flatA) ; i+=2 )
      { seg = arrayp(segs,j++,SEG) ;
	seg->key = arr(flatA, i+1, BSunit).k ;
	seg->parent = clone ;
	seg->x = pos ;
	seg->flag = 0 ;
	if (arr(flatA, i, BSunit).k  != _General_remark)
	  seg->flag |= FLAG_WORK_REMARK ;
      } 

    if (bsGetKey (Clone,_Positive_probe,&probe)) do
      { if (!assFind (probeHash, (void*)probe, &probeArray))
	  { probeArray = arrayCreate (8, GRIDMAP) ;
	    assInsert (probeHash, (void*)probe, probeArray) ;
	  }
	map = arrayp(probeArray, arrayMax(probeArray), GRIDMAP) ;
	map->x = pos ;
	map->ctg = 1 ;		/* 0 value is special */
      } while (bsGetKey (Clone,_bsDown,&probe)) ;
  
  *pj = j ;
}

static Array pmapConvert (KEY contig, KEY pmap)
{
  OBJ	Clone, Contig ;
  KEY 	clone, jprobe ;
  static Array mapArray ;
  Array segs, probeArray ;
  int 	i, j, pos1 ;

  if (!(Contig = bsCreate (contig)))
    { messout ("Can't find raw data for contig %s",
	       name(contig)) ;
      return 0 ;
    }
  if (!lexlock (pmap))
    { messout ("Structure for %s map is locked by another user",
	       name (pmap)) ;
      return 0 ;
    }

  if (!(segs = arrayGet (pmap, SEG, segFormat)))
    segs = arrayCreate (32, SEG) ;
  arrayMax(segs) = 0 ;

  probeHash = assReCreate (probeHash) ;
  mapArray = arrayReCreate (mapArray, 8, GRIDMAP) ;
  
  j = 1 ;	/* segs are boxes, so start at 1 */
  if (bsGetKey (Contig, _Clone, &clone)) do
    if (Clone = bsCreate(clone))
      { if (bsGetKey (Clone, _pMap, 0) && 
	    bsGetData(Clone, _bsRight, _Int, &pos1))
	  addClone (segs, &j, clone, Clone, pos1) ;
	bsDestroy(Clone);
      } 
    while (bsGetKey (Contig, _bsDown, &clone)) ;

				/* now cluster probes */
  clone = 0 ; probeArray = 0 ;
  while (assNext (probeHash, &clone, &probeArray))
    { if (Clone = bsCreate(clone))
	{ gridCluster (probeArray, mapArray) ;
	  for (i = 0 ; i < arrayMax(mapArray) ; ++i)
	    { jprobe = j ;
	      addClone (segs, &j, clone, Clone, 
			arrp(mapArray,i,GRIDMAP)->x) ;
	      arrp(segs, jprobe, SEG)->flag |= FLAG_PROBE ;
	    }
	  bsDestroy(Clone) ;
	}
      arrayDestroy (probeArray) ;
    }
  
  array (segs,0,SEG).x = -1000000 ;
  arraySort (segs, compareSeg);

  bsDestroy (Contig) ;
  arrayStore (pmap, segs, segFormat) ;
  lexunlock (pmap) ;
  
  return segs ;
}

/*************************************************************/
/******** pmap show intersect with active keyset *************/
 
static void pMapShowAll(void) /*show pmap without any highlighting*/
{
  int i, n ;
  SEG *seg ;
 
  LOOKGET("pMapShowAll") ;

  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  n = FLAG_HIDE | FLAG_HIGHLIGHT; /*<<--*/
  while(i--)
  {
      seg->flag&= ~n; /*<<--reset these bits*/
      seg++;
  }
  ClearFlag(look, FLAG_SHOW_WORK_REMARKS);
    /*make sure extra remarks aren't shown*/
  SetFlag(look, FLAG_REVERT_DISPLAY);
    /*<<--this is just to communicate down the call stack without passing arg: it only happens here*/
  pMapDraw (look, 0);
  ClearFlag(look, FLAG_REVERT_DISPLAY);
}

/**********/

static void pMapHide(void)
{
  int i, j, n ;
  SEG *seg ;
  void *dummy ;
  KEYSET keySet = 0 ;
 
  LOOKGET("pmapHide") ;

  if(!keySetActive(&keySet, &dummy))
    { messout("First select a keySet window, thank you.") ;
      return ;
    }
  seg = arrp(look->segs,0,SEG) ;
  i = arrayMax(look->segs) ;
  n = FLAG_HIDE ;
  while(i--)
    { seg->flag &= ~n ;  /* bitwise NOT */
      if (!arrayFind(keySet,&seg->key,&j, keySetOrder) &&
	  !arrayFind(keySet,&seg->parent,&j, keySetOrder))
	seg->flag |= FLAG_HIDE ;
      seg++ ;
    }
  pMapDraw (look, 0);
}


/*********************************************/
/******* local display routines **************/

/************* first a mini package for bumping text *************/

typedef struct bumpStruct
 { int n ;
   double y0 ;
   double *end ;
 } *BUMP ;

static BUMP bumpCreate (int n, double y0)
{
  int i ;
  BUMP bump = (BUMP) messalloc (sizeof (struct bumpStruct)) ;

  bump->n = n ;
  bump->y0 = y0 ;
  bump->end = (double*) messalloc (n*sizeof (double)) ;
  for (i = 0 ; i < n ; ++i)
    bump->end[i] = -100000000.0 ;
  return bump ;
}

static void bumpDestroy (BUMP bump)
{
  messfree (bump->end) ;
  messfree (bump) ;
}

static void bumpWrite (LOOK look, BUMP bump, char *text, double x)
{
  int i ;
  int len = strlen(text) ;
  int min ;
  
  for (i = 0 ; i < bump->n ; ++i)
    if (bump->end[i] < x)
      { graphText (text, SHOWX(x)-3.5, SHOWY(bump->y0 + i)) ;
	bump->end[i] = x + (len+1)/showFacX ;
	return ;
      }

  min = 0 ;
  for (i = 1 ; i < bump->n ; ++i)
    if (bump->end[i] < bump->end[min])
      min = i ;

  graphText (text, SHOWX(bump->end[min])-3.5, SHOWY(bump->y0 + min)) ;
  bump->end[min] += (len+1)/showFacX ;
}


static void bumpNoWrite (LOOK look, BUMP bump, char *text, double x)
{
  int i ;
  int len = strlen(text) ;
  int min ;
  
  for (i = 0 ; i < bump->n ; ++i)
    if (bump->end[i] < x)
      { bump->end[i] = x + (len+1)/showFacX ;
	return ;
      }

  min = 0 ;
  for (i = 1 ; i < bump->n ; ++i)
    if (bump->end[i] < bump->end[min])
      min = i ;

  bump->end[min] += (len+1)/showFacX;
}

/********* use middle button for cursor **********/
  /* Borrowed from gMap but with x,y exchanged */

#define ACCEL (1./25.)

static void pMapMiddleDown (double x, double y) 
{ 
  float x1,x2,y1,y2 ;
  LOOKGET("pMapMiddleDown") ;

  graphBoxDim (look->scrollBox, &x1, &y1, &x2, &y2) ;
  oldDx = 0.5 * (x2-x1) ;
  oldx=x; /*make sure line gets undrawn*/
  lookDrag = look ;
  dragFast = y > DRAGFASTLIMIT ? TRUE : FALSE ;

  if(dragFast)
    { graphXorLine (oldx - oldDx , DRAGFASTLIMIT, oldx - oldDx , YMINI + 3) ;
      graphXorLine (oldx + oldDx , DRAGFASTLIMIT, oldx + oldDx , YMINI + 3) ;
    }
  else
    graphXorLine (oldx, 0, oldx, DRAGFASTLIMIT) ;
 
  oldy = y ;
#if 0
printf("DOWN: oldx %f, oldy %f, oldDx %f, x %f, y %f\n",  oldx, oldy, oldDx, x, y);
#endif
  return;
}

static void pMapMiddleDrag (double x, double y) 
{
  if(dragFast)
    { graphXorLine (oldx - oldDx , DRAGFASTLIMIT, oldx - oldDx , YMINI + 3) ;
      graphXorLine (oldx + oldDx , DRAGFASTLIMIT, oldx + oldDx , YMINI + 3) ;
    }
  else
    graphXorLine (oldx, 0, oldx, DRAGFASTLIMIT) ;

  oldx = x ;

  if(dragFast)
    {
      oldDx*=exp((oldy-y)*ACCEL); /**NL get the magnification the right way round*/
      oldy=y;
      graphXorLine (oldx - oldDx , DRAGFASTLIMIT, oldx - oldDx , YMINI + 3) ;
      graphXorLine (oldx + oldDx , DRAGFASTLIMIT, oldx + oldDx , YMINI + 3) ;
    }
  else
    graphXorLine (oldx, 0, oldx, DRAGFASTLIMIT) ;
#if 0
printf("DRAG: oldx %f, oldy %f, oldDx %f, x %f, y %f\n",  oldx, oldy, oldDx, x, y);
#endif
  return;
}

static void pMapMiddleUp (double x, double y) 
{ 
  float x1, x2, y1, y2 ;
  LOOK look = lookDrag ;

  if(dragFast)
    { graphBoxDim (look->scrollBox, &x1, &y1, &x2, &y2) ;
      look->centre+=(x - look->winWidth*.50  )  / MINIFAC ;
      showFacX *= 0.5*(x2-x1)/oldDx;
    }
  else
    look->centre += (x - look->winWidth*.50)/showFacX ;
#if 0
printf("UP: oldx %f, oldy %f, oldDx %f, x %f, y %f\n",  oldx, oldy, oldDx, x, y);
#endif
  pMapDraw (look, 0);
  return;
}

#undef ACCEL


/*********************************/

typedef enum {NORMAL, SELECTED, SISTER, HIGHLIT} DrawType ; /*<<--SELECTED here means the currently-picked box*/

static void mapBoxDraw (int box, DrawType drawType, SEG *seg)
{
  if (!seg)			/* e.g. gMapBox */
    { if (box == gMapBox)
	graphBoxDraw (box, BLACK, 
		      drawType==SELECTED ? GREEN : LIGHTGREEN) ;
      return ;
    }

  switch (drawType)
    {
    case HIGHLIT: /*<<--*/
        graphBoxDraw(box, BLACK, MAGENTA);
        break;
    case SELECTED:
#if 0
      graphBoxDraw (box, WHITE, BLUE) ; break ;
#endif
      graphBoxDraw (box, WHITE, RED) ; break ; /*<<--make highlighting the same everywhere*/
    case SISTER:
#if 0
      graphBoxDraw (box, BLACK, LIGHTBLUE) ; break ;
#endif
      graphBoxDraw (box, BLACK, LIGHTRED) ; break ; /*<<--make highlighting the same everywhere*/
    case NORMAL:
      switch (class(seg->key))
	{
	case _VSequence:
	  graphBoxDraw (box, BLACK, YELLOW) ; break ;
	case _VClone:
	  graphBoxDraw (box, BLACK, WHITE) ; break ;
	default:
	  graphBoxDraw (box, BLACK, WHITE) ;
	}
      break ;
    }
}

/******************************/


void drawMiniContig (LOOK look)
{
  SEG   *seg ;
  float x, lastx = 100000.0, height, x0, x1 ;
  int   i ;
#define DY 0.75

  offset = look->centre*showFacX - look->winWidth/2 ;
  miniOffset = look->centre*MINIFAC - look->winWidth/2 ;

  graphFillRectangle (MINIX(look->min), YMINI, 
		      MINIX(look->max), YMINI+0.2) ;

  graphTextHeight (0.75) ;
  for (i = arrayMax(look->segs) ; --i ;)
    { seg = arrp(look->segs, i, SEG) ;
      if (class(seg->key) == _VGene ||
	  class(seg->key) == _VLocus)
	{ x = MINIX(seg->x) ;
	  if (x < 0)
	    break ;
	  if (x > look->winWidth)
	    continue ;
	  if (x + strlen(name(seg->key)) < lastx)
	    height = 1 ;
	  else
	    height += DY ;
	  graphLine (x, YMINI - height + DY, x, YMINI) ;
	  graphText (name(seg->key), x,  YMINI - height) ;
	  lastx = x ;
	}
    }
  graphTextHeight (0) ;		/* default */

/* map 0 in top region to pmap units x0, winWidth to x1 */

  x0 = UNSHOWX(0) ;
  x1 = UNSHOWX(look->winWidth) ;

  look->scrollBox = graphBoxStart () ;
  graphRectangle (MINIX(x0), YMINI-0.2, MINIX(x1), YMINI+0.4) ;
  graphBoxEnd () ;
  graphBoxDraw (look->scrollBox, BLACK, GREEN) ;
}

/**********************************/

static void pMapScroll (float *px, float *py, BOOL isDone)
{
  LOOKGET("pMapScroll") ;

  *py = YMINI - 0.2 ;
  if (isDone)
    { look->centre += (*px - look->winWidth*0.45) / MINIFAC ;
      pMapDraw (look, 0);
    }
}

/**********************************/

static void pMapResize (void) /*<<--*/
{
  LOOKGET("pMapResize");
  pMapDraw(look, 0);
  return;
}

/******************************/

static void pMapShowSegs (LOOK look, KEY key)
{
  int i,box,keybox = 0 ;
  float xx1,xx2,y,oldx2 ;
  SEG *seg ;
  char *text ;
  BUMP remBump,geneBump,probeBump ;
  int iYac = 0, iReg = 0 ;

  probeBump = bumpCreate (2,- 0.25) ;
  geneBump = bumpCreate (3,20) ;
  remBump = bumpCreate (7,23) ;

          /* Direct interpolating move from the gmap */
          /* The position is passed approximately in pmap units */

  if (class(key) == _VCalcul)
    { xx1 = (int)KEYKEY(key) - (int)0x800000 ;
      if (xx1 < look->min)
	xx1 = look->min ;
      else if (xx1 > look->max)
	xx1 = look->max ;
      look->centre = xx1 ;
    }
  else if (key)
    for (i = arrayMax(look->segs) ; --i ;)
      if (arrp(look->segs,i,SEG)->key == key)
	{ look->centre = arrp(look->segs,i,SEG)->x ;
	  break ;
	}

  if (!key && look->activebox && 
      (seg = arr(look->box2seg,look->activebox,SEG*)))
    key = seg->key ;
  look->activebox = 0 ;

  arrayMax(look->box2seg) = 0 ;

/* Communication box to the gMap */
  array(look->box2seg,gMapBox = graphBoxStart(),SEG*) = 0 ;
  graphRectangle (0, SHOWY(19.2), look->winWidth, SHOWY(19.8)) ;
  graphTextHeight (0.6) ;
  for (i = 10 ; i < look->winWidth ; i += 40)
    graphText ("Pick here to go to the genetic map", 
	       i, SHOWY(19.25)) ;
  graphTextHeight (0) ;
  graphBoxEnd() ;
  graphBoxDraw(gMapBox,BLACK,LIGHTGREEN) ;

  offset = look->centre*showFacX - look->winWidth/2 ;

  graphText ("Press middle button in upper zone to recentre slowly, "
	     "in lower to recentre faster and to zoom." ,
	     2, look->winHeight-1) ;

  oldx2 = -10000001.0 ;
  for (i = 1 ; i < arrayMax(look->segs) ; ++i)
    { box = 0 ;
      seg = arrp(look->segs,i,SEG) ;
      if(seg->flag & FLAG_HIDE)
	continue ;
      xx1 = seg->x - seg->dx ;
      xx2 = seg->x + seg->dx ;
      if (SHOWX(xx2) < 0)
	{  /* Must ensure that the vertical placing of 
	      clones, remarks, genes etc. stays constant.
	   */
	  switch (class(seg->key))
	    {
	    case _VClone:
	      if (seg->flag & FLAG_FINGERPRINT)
		{ if (xx1 > oldx2)
		    iReg = 0 ;
		  if (xx2 > oldx2)
		    oldx2 = xx2 ;
		  ++iReg ;
		}
	      else if (!(seg->flag & FLAG_PROBE))
		++iYac ;
	      break ;
	    case _VGene:
	    case _VLocus:
	      bumpNoWrite (look, geneBump,name(seg->key),seg->x) ;
	      break ;
	    case _VText:
 	      if (IsClearFlag(look, FLAG_SHOW_WORK_REMARKS) && IsSetFlag(seg, FLAG_WORK_REMARK))
		break ;
	      if (seg->flag & FLAG_MORE_TEXT)
		text = messprintf("%s(*)",name(seg->key)) ;
	      else
		text = name(seg->key) ;
	      bumpNoWrite (look, remBump,text,seg->x) ;
	      break ;
	    }
	  continue ;		/* skip drawing */
	}
      if (SHOWX(xx1) > look->winWidth)
	continue ;		/* no need to be so careful */
      switch (class (seg->key))
	{
	case _VClone:
	  if (seg->flag & FLAG_FINGERPRINT)
	    { if (xx1 > oldx2)
		{ iReg = 0 ;	/* start new islands from bottom */
		  if (oldx2 > -10000000.0)
		    { graphLine (SHOWX(xx1-7),SHOWY(7+10),
				 SHOWX(xx1-7),SHOWY(7+11)) ;
		      graphLine (SHOWX(xx1-8),SHOWY(7+10),
				 SHOWX(xx1-8),SHOWY(7+11)) ;
		    }
		}
	      box = graphBoxStart() ;
	      if (xx2 > oldx2)
		oldx2 = xx2 ;
	      y = 7 + 10.5 - (iReg % 10) - 0.5*((iReg/10)%2) ;
	      if (seg->flag & FLAG_COSMID_GRID)
		graphFillRectangle (SHOWX(xx1),SHOWY(y-0.1),
				    SHOWX(xx2),SHOWY(y+0.1)) ;
	      else
		graphLine (SHOWX(xx1),SHOWY(y),SHOWX(xx2),SHOWY(y));
	      if (seg->flag & FLAG_CANONICAL)
		text = messprintf ("%s *",name(seg->key)) ;
	      else
		text = name(seg->key) ;
	      graphText (text,SHOWX(seg->x)-3.5,SHOWY(y-0.8)) ;
	      ++iReg ;
	    }
	  else if (seg->flag & FLAG_PROBE)
	    { box = graphBoxStart() ;
	      bumpWrite (look, probeBump,name(seg->key),seg->x) ;
	    }
	  else
	    { y = 1 + 5 - (iYac%5) + 0.5*((iYac/5)%2) ;
	      if (seg->flag & FLAG_YAC_GRID)
		graphFillRectangle (SHOWX(xx1),SHOWY(y-0.1),
				    SHOWX(xx2),SHOWY(y+0.1)) ;
	      else
		graphLine (SHOWX(xx1),SHOWY(y),SHOWX(xx2),SHOWY(y)) ;
	      box = graphBoxStart() ;
	      text = messprintf ("(%s)",name(seg->key)) ;
	      graphText (text,SHOWX(seg->x)-3.5,SHOWY(y-0.4)) ;
	      ++iYac ;
	    }
	  graphBoxEnd() ;
	  break ;
	case _VSequence:
	  box = graphBoxStart() ;
	  graphRectangle (SHOWX(xx1),SHOWY(18.2),
			  SHOWX(xx2),SHOWY(18.8)) ;
	  graphBoxEnd() ;
	  break ;
	case _VGene:	
	case _VLocus:
	  box = graphBoxStart() ;
	  bumpWrite (look, geneBump,name(seg->key),seg->x) ;
	  graphBoxEnd() ;
	  break ;
	case _VText:
 	  if (IsClearFlag(look, FLAG_SHOW_WORK_REMARKS) && IsSetFlag(seg, FLAG_WORK_REMARK))
	    break ;
	  box = graphBoxStart() ;
	  if (seg->flag & FLAG_MORE_TEXT)
	    text = messprintf("%s(*)",name(seg->key)) ;
	  else
	    text = name(seg->key) ;
	  bumpWrite (look, remBump,text,seg->x) ;
	  graphBoxEnd() ;
	  break ;
	default:
	  messout ("Don't know how to put (%s) on a physical map",
		   name(seg->key)) ;
	}
      if (box)			/* something drawn */
	{ mapBoxDraw (box, (seg->flag & FLAG_HIGHLIGHT)? HIGHLIT: NORMAL, seg) ; /*<<--*/
	  array(look->box2seg,box,SEG*) = seg ;
	  if (seg->key == key)
	    keybox = box ;
	}
    }

  if (keybox && IsClearFlag(look, FLAG_REVERT_DISPLAY)) pMapSelectBox(look, keybox);

  drawMiniContig (look) ;
  
  bumpDestroy (remBump) ;
  bumpDestroy (geneBump) ;
  bumpDestroy (probeBump) ;
}

/**********************************/

static Array sisterBox, sisterSeg ;

static void addPositiveYacs (OBJ obj, Array box2seg, int min, int max)
{
  static Array flat = 0 ;
  int nsib = arrayMax (sisterBox) ;
  KEY yac ;
  int i, j ;
  SEG *subseg ;

  flat = arrayReCreate (flat, 32, BSunit) ;

  if (bsFindTag (obj, _Hybridizes_to) && bsFlatten (obj, 2, flat))
    for (j = 0 ; j < arrayMax(flat) ; j += 2)
      { yac = arr(flat, j+1, BSunit).k ;
	for (i = min ; i <= max ; ++i)
	  if ((subseg = arr(box2seg,i,SEG*))->key == yac)
	    { array(sisterBox, nsib, int) = i ;
	      array(sisterSeg, nsib++, SEG*) = subseg ;
	    }
      }
}

static void addPositiveProbes (OBJ obj, Array box2seg, int min, int max)
{
  int nsib = arrayMax (sisterBox) ;
  KEY probe ;
  int i ;
  SEG *subseg ;

  if (bsGetKey (obj, _Positive_probe, &probe)) do
    for (i = min ; i <= max ; ++i)
      if ((subseg = arr(box2seg,i,SEG*))->key == probe)
	{ array(sisterBox, nsib, int) = i ;
	  array(sisterSeg, nsib++, SEG*) = subseg ;
	}
    while (bsGetKey (obj, _bsDown, &probe)) ;
}

static void makeSisterList (Array box2seg, int activebox, int segmax)
{
  OBJ obj ;
  SEG	*seg, *subseg ;
  int	i, min, max, nsib = 0 ;

  if (sisterBox)
    arrayMax(sisterBox) = arrayMax(sisterSeg) = 0 ;
  else
    { sisterBox = arrayCreate (8, int) ;
      sisterSeg = arrayCreate (8, SEG*) ;
    }
 
  seg = arr(box2seg,activebox,SEG*) ;

  min = max = activebox ;
  for (i = activebox ; --i > 1 &&	/* RMD 14/8/91  0->1 */
       arr(box2seg,i,SEG*)->x > seg->x - 250 ;)
    min = i ;
  for (i = activebox ; ++i < segmax &&
       arr(box2seg,i,SEG*)->x < seg->x + 250 ;)
    max = i ;

  for (i = min ; i <= max ; ++i)
    if (i != activebox &&
	(subseg = arr(box2seg,i,SEG*)) &&
	subseg->x == seg->x &&
	subseg->parent == seg->parent)
      { array(sisterBox, nsib, int) = i ;
	array(sisterSeg, nsib++, SEG*) = subseg ;
      }

  if (class (seg->key) == _VClone && (obj = bsCreate (seg->key)))
    { addPositiveProbes (obj, box2seg, min, max) ;
      if (seg->flag & FLAG_PROBE)
	addPositiveYacs (obj, box2seg, min, max) ;
      bsDestroy (obj) ;
    }
}

/*********************************************************/

static void pMapSelectBox (LOOK look, int box)	/* switch activebox */
{
  int i ;
  SEG *seg ;

  if (look->activebox)
    { seg = arr(look->box2seg,look->activebox,SEG*) ;
      mapBoxDraw (look->activebox, 
		  (seg && (seg->flag & FLAG_HIGHLIGHT)) ? HIGHLIT: NORMAL, seg) ;
      if (seg)  /* i.e. not for special boxes */
	{ makeSisterList (look->box2seg, look->activebox, 
			  arrayMax(look->box2seg)) ;
	  for (i = arrayMax(sisterBox) ; i-- ;)
	  {
	      mapBoxDraw(
                arr(sisterBox, i, int),
                ((arr(sisterSeg, i, SEG*))->flag & FLAG_HIGHLIGHT)? HIGHLIT: NORMAL,
                arr(sisterSeg, i, SEG*)
              ) ; /*<<--*/
	  }
	}
    }

  seg = arr(look->box2seg,box,SEG*) ;
  mapBoxDraw (box, SELECTED, seg) ;
  if(seg)  /* i.e. not for special boxes */
    { makeSisterList (look->box2seg, box, 
		      arrayMax(look->box2seg)) ;
      for (i = arrayMax(sisterBox) ; i-- ;)
	mapBoxDraw (arr(sisterBox,i,int), SISTER, arr(sisterSeg,i,SEG*)) ;
    }

  look->activebox = box ;
}

/*********************************************************/

extern int sequenceLength (KEY seq) ;

static void pMapFollow  (LOOK look, double x, double y)
{ KEY  from ;
  SEG  *seg ;

  if (look->activebox == gMapBox)
    { KEY contig ;

      if (!lexword2key(name(look->key), &contig, _VContig))
	return ;
      pMapToGMap (contig, 0, 
	  (int)(look->centre + (x - look->winWidth/2)/showFacX)) ;
    }
  else
    { seg = arr(look->box2seg,look->activebox,SEG*) ;
      
      switch (class(seg->key))
	{
	case _VGene: 
	case _VLocus:
	  display (seg->key, look->key, GMAP) ;
	  break ;
	case _VSequence:
	  if (seg->dx)
	    from = KEYMAKE(_VCalcul,
		   sequenceLength(seg->key)*x/(20.*showFacX*seg->dx)) ;
	  else
	    from = 0 ;
	  display (seg->key, from, 0) ;
	  break ;
	case _VClone:
	  display (seg->key, look->key, TREE) ;
	  break ;
	case _VText:
	  display (seg->parent, look->key, TREE) ;
	  break ;
	default:
	  display (seg->key, look->key, 0) ;
	}
    }
}

/*.......... ..... .....
*/

static void pMapHighlight(KEYSET k, KEY key)
{
  int i, j;
  SEG *seg;
  void *dummy;
  KEYSET keySet=NULL; 
  LOOKGET("pMapHighlight");

  if (k!=NULL && keySetExists(k) /*I probably ought to check this more thoroughly*/) keySet=k;
  if (keySet==NULL && !keySetActive(&keySet, &dummy))
  {
     messout("First select a keySet window, thank you.");
     return;
  }
  seg=arrp(look->segs, 0, SEG);
  i=arrayMax(look->segs);
  while (i--)
  {
    seg->flag&= ~FLAG_HIGHLIGHT; /*clear this bit on all segs, except for selected objects:*/
    if (keySetFind(keySet, seg->key, &j)) seg->flag|=FLAG_HIGHLIGHT;
    seg++;
  }
  pMapDraw(look, (k==NULL? 0: key));
  return;
}

extern void pMapSetHighlightKeySet(KEYSET k)
{
  pMapPendingKeySet=k;
  return;
}

extern BOOL pMapGetCMapKeySet(KEY contig, int *xMin, int *xMax, KEYSET clones)
{
#if 0
  if (not a good contig)
  {
    return FALSE;
  }
  else if (not provided with a valid clones keyset by caller)
  {
    return FALSE;
  }
  else
    /*caller has provided me with a usable clones KEYSET, so provide it if there*/
  {
    zero the keyset;
    if (there is one to get)
    {
      get it in keyset
      return TRUE;
    }
    else
    {
      return FALSE;
    }
  }
#endif
  messout("**pMapGetCMapKeySet not implemented");
  return FALSE;
}

/*********************************************************/
/*********************************************************end of file*/
