/*  File: griddisp.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:  to display clone grids, such as the Yac grid
 * Exported functions: gridDisplay() only
 * HISTORY:
 * Last edited: Apr  2 17:52 1992 (mieg)
 * * Mar  2 13:45 1992 (rd): Pool handling, surrounds for comparison, getting
	probe and clone names by clicking
 * * Feb 21 20:02 1992 (rd): CloneDisp looks for matches in any grid
	FLAG_CHANGED
 * * Jan  9 19:01 1992 (rd): FLAG_DUPLICATE and gridKeyFind()
 * * Jan  9 15:06 1992 (rd): FLAG_BLANK
 * * Dec 31 13:22 1991 (rd): added support for _No_stagger
 * Created: Thu Nov  7 15:30:07 1991 (rd)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "keyset.h"
#include "graph.h"
#include "key.h"
#include "lex.h"
#include "bs.h"
#include "systags.wrm"
#include "classes.wrm"
#include "sysclass.wrm"
#include "tags.wrm"
#include "disptype.wrm"
#include "display.h"
#include "query.h"
#include "wormtype.h"
#include "session.h"
#include "grid.h"
#include "pmap.h"

typedef struct LOOKSTUFF
  { int   magic;        /* == MAGIC */
    KEY   key ;
    KEY   probe ;
    KEY   probeStore ;
    Array segs ;      	/* array of SEG's from the obj */
    int   dxLine, dyLine, dxSpace, dySpace ;
    char  *title ;
    int   flag ;
    char  selectName[12] ;
    char  probeName[12] ;
    Array map ;
    int   activeBox ;
    int   crossBox ;
  } *LOOK ;

#define MAGIC 26931

BOOL gridDisplay (KEY key, KEY from, BOOL isOldGraph) ;
static void gridDestroy (void) ;
static void gridConvert (LOOK look) ;     
static void gridDraw (LOOK look) ;
static void gridPick (int box) ;
static void gridKeyboard (int k) ;
static void gridSelect (LOOK look, int box) ;
static void gridSelectEntry (char *string) ;
static void gridToggleSymbolic (void) ;
static void gridProbeDisplay (LOOK look, KEY probe) ;
static void gridDump (void) ;
static void gridTreeProbe (void) ;
static void gridSave (void) ;
static void gridSetRange (void) ;
static BOOL gridKeyFind (LOOK look, KEY clone, int *pbox) ;
static void gridKeySetStats (void) ;
static void gridComparisonStore (void) ;
static void gridSelectButton (void) ;
static void gridProbeButton (void) ;
static void gridChangeClone (void) ;
static void gridMapMode (void) ;

enum BoxNames { BACKGROUND=0, 
		SELECT_BOX, SELECT_BUTTON_BOX, 
		  SELECT_NAME_BOX, SELECT_NAME_BOX2,
		CLEAR_BOX, 
		PROBE_BOX, PROBE_BUTTON_BOX,
		  PROBE_NAME_BOX, PROBE_NAME_BOX2,
		  CLONE_BOX, POOL_BOX,
		MODE_BOX, EDIT_MODE_BOX, MAP_MODE_BOX, 
		MIN_LIVE_BOX } ;  /* MIN_LIVE_BOX must be last */

static MENUOPT gridMenu[] =
            { graphDestroy, "Quit",
	      help, "Help",
	      graphPrint, "Print",
	      displayPreserve, "Preserve",
	      gridComparisonStore, "Centre <-> Surround",
	      gridToggleSymbolic, "Names",
	      gridTreeProbe, "Display probe as tree",
	      gridSave, "Save data with probe",
	      gridSetRange, "Set cluster range",
	      gridKeySetStats, "Stats on Keyset",
	      gridChangeClone, "Change Gridded Clone",
	      0, 0, 
	      gridDump, "Dump",
	      0, 0
            } ;

typedef struct
  { KEY key ;
    int x, y ;
    int flag ;
    int mapIndex ;
  } SEG ;

	/* per LOOK flags */
#define FLAG_NAMES		0x0001
#define FLAG_EDIT_MODE		0x0002
#define FLAG_EDITED		0x0004
#define FLAG_STAGGER		0x0008
#define FLAG_A1			0x0010
#define FLAG_IS_POOL		0x0020

	/* per SEG flags */
#define FLAG_STRONG		0x0001
#define FLAG_WEAK		0x0002
#define FLAG_HYB		0x0003
#define FLAG_BLANK		0x0004
#define FLAG_DUPLICATE		0x0008
#define FLAG_STORE_STRONG	0x0010
#define FLAG_STORE_WEAK		0x0020
#define FLAG_STORE_HYB		0x0030
#define FLAG_CHANGED		0x0040
#define FLAG_STORE_CHANGED	0x0400

#define LOOKGET(name)     LOOK look ; \
                          if (!graphAssFind (gridDisplay,&look)) \
		            messcrash ("graph not found in %s",name) ; \
			  if (!look) \
                            messcrash ("%s received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

/*********************************/

BOOL gridDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  int box ;
  LOOK look ;

  if (key && class(key) != _VClone_Grid)
    { from = key ;
      key = 0 ;
    }

  if (class(from) != _VClone && class(from) != _VPool)
    from = 0 ;

/* in the future get "key" from "from" either because from is
   in key, or because from has been hybridised to key */

  if (!key)
    return FALSE ;

  if (isOldGraph)
    { if (graphAssFind (gridDisplay,&look) &&
	  look->magic == MAGIC &&
	  look->key == key)
	{ if (gridKeyFind (look, from, &box))
	    gridSelect (look, box) ;
	  else if (from)
	    gridProbeDisplay (look, from) ;
	  return TRUE ;
	}
      graphRetitle (name(key)) ;
      gridDestroy () ;
      graphAssRemove (gridDisplay) ;
    }
  else 
    {
      if (!displayCreate (GRID))
	return FALSE ;
      
      graphRegister (DESTROY, gridDestroy) ;
      graphRegister (PICK,(GraphFunc) gridPick) ;
      graphRegister (KEYBOARD,(GraphFunc) gridKeyboard) ;
      graphRegister (MESSAGE_DESTROY, displayUnBlock) ;
      graphMenu (gridMenu) ;
    }

  look = (LOOK) messalloc (sizeof (struct LOOKSTUFF)) ;      
  look->magic = MAGIC;
  look->key = key ;
  look->map = arrayCreate (16, GRIDMAP) ;
  gridConvert (look) ;		/* makes ->segs, dx values etc. */
  
  graphAssociate (gridDisplay,look) ;

  if (from)
    gridProbeDisplay (look, from) ;
  else
    gridDraw (look) ;

  return TRUE ;
}

/************************************************************/
/***************** Registered rourtines *********************/

static void gridDestroy (void)
{
  LOOKGET("gridDestroy") ;

  arrayDestroy (look->segs) ;
  arrayDestroy (look->map) ;

  look->magic = 0 ;
  messfree (look) ;
}

/***********************************/

static void gridEdit (LOOK look, int box)
{
  SEG *seg = arrp(look->segs, box, SEG) ;
  int box2 ;
  static BOOL doDups = TRUE ;

  if (seg->flag & FLAG_BLANK)
    return ;

  if (seg->flag & FLAG_STRONG)
    { seg->flag &= ~FLAG_STRONG ;
      seg->flag |= FLAG_WEAK ;
      graphBoxDraw (box, -1, LIGHTBLUE) ;
    }
  else if (seg->flag & FLAG_WEAK)
    { seg->flag &= ~FLAG_HYB ;
      graphBoxDraw (box, -1, WHITE) ;
    }
  else
    { seg->flag |= FLAG_STRONG ;
      graphBoxDraw (box, -1, BLUE) ;
    }
  seg->flag |= FLAG_CHANGED ;

  if (doDups && (seg->flag & FLAG_DUPLICATE))
    { doDups = FALSE ;		/* prevents infinite loops */
      while (gridKeyFind (look, seg->key, &box2))
	if (box2 != box)
	  gridEdit (look, box2) ;
      doDups = TRUE ;
    }

  look->flag |= FLAG_EDITED ;
}

/**********************************/

static void colourNeighbours (LOOK look, int strong, int weak)
{
  int i, index ;
  SEG *seg ;

  if (look->flag & FLAG_EDIT_MODE)
    return ;

  index = arrp(look->segs, look->activeBox, SEG)->mapIndex ;
  if (!index)
    return ;

  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      if (seg->mapIndex == index)
	if (seg->flag & FLAG_STRONG)
	  graphBoxDraw (i, -1, strong) ;
	else if (seg->flag & FLAG_WEAK)
	  graphBoxDraw (i, -1, weak) ;
    }
}

static void gridSelect (LOOK look, int box)
{
  float x1,x2,y1,y2 ;
  SEG *seg ;
  int box2 ;

  if (look->activeBox == box)
    return ;

  if (look->activeBox)
    { graphBoxDraw (look->activeBox, BLACK, -1) ;
      seg = arrp(look->segs, look->activeBox, SEG) ;
      if (seg->flag & FLAG_DUPLICATE)
	while (gridKeyFind (look, seg->key, &box2))
	  if (box2 != look->activeBox)
	    graphBoxDraw (box2, BLACK, -1) ;
      colourNeighbours (look, BLUE, LIGHTBLUE) ;
    }

  look->activeBox = box ;

  graphBoxDraw (look->activeBox, RED, -1) ;
  seg = arrp(look->segs, look->activeBox, SEG) ;
  if (seg->flag & FLAG_DUPLICATE)
    while (gridKeyFind (look, seg->key, &box2))
      if (box2 != look->activeBox)
	graphBoxDraw (box2, RED, -1) ;
  colourNeighbours (look, RED, LIGHTRED) ;
  graphBoxDim (box, &x1, &y1, &x2, &y2) ;
  graphBoxShift (look->crossBox, x1, y1) ;

  strncpy (look->selectName, name(seg->key), 9) ;
  graphBoxDraw (SELECT_NAME_BOX, -1, -1) ;
}

/***********************************/

static void gridFollow (LOOK look)
{
  int index = arrp(look->segs, look->activeBox, SEG)->mapIndex ;
  GRIDMAP *map ;
  static KEYSET kset = 0 ;
  int i, j ;

  if (index)
    { map = arrp(look->map, index-1, GRIDMAP) ;
      kset = keySetReCreate (kset) ;
      j = 0 ;
      for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
	if (arrp(look->segs, i, SEG)->mapIndex == index)
	  keySet(kset, j++) = arrp(look->segs, i, SEG)->key ;
      keySetSort (kset) ;
      pMapSetHighlightKeySet (kset) ;
      display (map->ctg, 
		     KEYMAKE(_VCalcul, (0x800000 + (int)map->x)),
		     0) ;
      pMapSetHighlightKeySet (0) ;
    }
  else
    display (arrp(look->segs,look->activeBox,SEG)->key,
		   look->key, 0) ;
}

/***********************************/

extern int ksetClassComplete (char *text, int len, int class) ;

static void cloneCompletion (char *cp, int len)
{
  ksetClassComplete (cp, len, _VClone) ;
}

static void probeCompletion (char *cp, int len)
{
  LOOKGET("probeCompletion") ;
  ksetClassComplete (cp, len, 
		     (look->flag & FLAG_IS_POOL) ? _VPool : _VClone) ;
}

/***********************************/

static void gridPick (int box) 
{
  LOOKGET("gridPick") ;

  switch (box)
    {
    case PROBE_NAME_BOX:
      graphCompletionEntry (0, look->probeName, 0, 0, 0, 0) ; 
      break ;
    case SELECT_NAME_BOX:
      graphCompletionEntry (0, look->selectName, 0, 0, 0, 0) ; 
      break ;
    default:
      if (box >= MIN_LIVE_BOX &&
	  box < arrayMax (look->segs)) /* a seg */
	{ graphEntryDisable () ;
	  if (look->flag & FLAG_EDIT_MODE)
	    gridEdit (look, box) ;
	  else if (box == look->activeBox)
	    gridFollow (look) ;
	  gridSelect (look, box) ;
	}
    }
}

/************************************/

static void gridToggleSymbolic (void)
{
  LOOKGET("gridToggleSymbolic") ;

  if (look->flag & FLAG_NAMES)
    look->flag &= ~FLAG_NAMES ;
  else
    look->flag |= FLAG_NAMES ;
  gridDraw (look) ;
}

/***********************/

static void gridCloneProbe (void)
{
  LOOKGET("gridCloneProbe") ;

  look->flag &= ~FLAG_IS_POOL ;
  graphBoxDraw (CLONE_BOX, WHITE, DARKRED) ;
  graphBoxDraw (POOL_BOX, BLACK, WHITE) ;
}

static void gridPoolProbe (void)
{
  LOOKGET("gridPoolProbe") ;

  look->flag |= FLAG_IS_POOL ;
  graphBoxDraw (POOL_BOX, WHITE, DARKRED) ;
  graphBoxDraw (CLONE_BOX, BLACK, WHITE) ;
}

/***********************/

static void gridEditMode (void)
{
  LOOKGET("gridEditMode") ;

  colourNeighbours (look, BLUE, LIGHTBLUE) ;
  look->flag |= FLAG_EDIT_MODE ;
  graphBoxDraw (EDIT_MODE_BOX, WHITE, DARKRED) ;
  graphBoxDraw (MAP_MODE_BOX, BLACK, WHITE) ;
}

/*************************************************/

static int clusterRange = 200 ;
static Array posStructArray = 0 ;

static int mapOrder (void *a, void *b)
{
  GRIDMAP *map1 = arrp(posStructArray, *(int*)a, GRIDMAP) ;
  GRIDMAP *map2 = arrp(posStructArray, *(int*)b, GRIDMAP) ;

  if (map1->ctg < map2->ctg)
    return -1 ;
  if (map1->ctg > map2->ctg)
    return 1 ;

  if (map1->x > map2->x)
    return -1 ;
  if (map1->x < map2->x)
    return 1 ;

  return 0 ;
}

void gridCluster (Array posArray, Array mapArray)
{				/* also called from pmapdisp.c */
  GRIDMAP *map, *pos ;
  int i, n, nmap = -1 ;
  static Array ptr ;

  arrayMax(mapArray) = 0 ;

  if (!arrayMax(posArray))
    return ;
  posStructArray = posArray ;

  if (!ptr || arrayMax(ptr) < arrayMax(posArray))
    ptr = arrayReCreate (ptr, arrayMax(posArray), int) ;
  arrayMax(ptr) = arrayMax(posArray) ;

  for (i = arrayMax(posArray) ; i-- ;)
    arr(ptr,i,int) = i ;

  arraySort (ptr, mapOrder) ;	/* preserve posArray order */

  for (i = 0 ; i < arrayMax(posArray) ; i++)
    { pos = arrp(posArray, arr(ptr, i, int), GRIDMAP) ;
      if (nmap >= 0 &&
	  pos->ctg == map->ctg && 
	  pos->x < map->x + clusterRange && 
	  pos->x > map->x - clusterRange)
	{ map->x = (n * map->x + pos->x)/(n+1) ;
	  ++n ;
	}
      else
	{ map = arrayp(mapArray, ++nmap, GRIDMAP) ;
	  *map = *pos ;
	  n = 1 ;
	}
      pos->clump = nmap ;
    }
}

static int nYacHits ;		/* for gridKeySetStats */

BOOL gridClusterKey (KEY key, Array map)
{
  int j, x1, x2 ;
  static Array positives, units ;
  KEY ctg ;
  OBJ obj, Clone ;
  GRIDMAP *pos ;

  if (!arrayExists (map))
    return FALSE ;
  arrayMax(map) = 0 ;
  
  if (obj = bsCreate (key))
    { units = arrayReCreate (units, 16, BSunit) ;
      if (bsFindTag (obj, _Hybridizes_to) && bsFlatten (obj, 2, units))
	{ positives = arrayReCreate (positives, 16, GRIDMAP) ; 
	  for (j = 1 ; j < arrayMax(units) ; j += 2)
	    if (Clone = bsCreate(arr(units,j,BSunit).k))
	      { if (bsGetKey (Clone, _pMap, &ctg) &&
		    bsGetData (Clone, _bsRight, _Int, &x1) &&
		    bsGetData (Clone, _bsRight, _Int, &x2))
		  { pos = arrayp (positives, arrayMax(positives), GRIDMAP) ;
		    pos->ctg = ctg ;
		    pos->x = 0.5 * (x1+x2) ;
		    ++nYacHits ;
		  }
		bsDestroy (Clone) ;
	      }
	  gridCluster (positives, map) ;
	}
      bsDestroy (obj) ;
    }

  return (arrayMax (map) > 0) ;
}

static void gridSetRange (void)
{
  LOOKGET("gridSetRange") ;

  if (graphPrompt ("Give range in pmap bands for clustering into single loci",
		   messprintf ("%d",clusterRange), "iz"))
    { freeint (&clusterRange) ;
      if (!(look->flag & FLAG_EDIT_MODE))
	gridMapMode () ;
    }
}

static void gridKeySetStats (void)
{
  int i ;
  static Array map ;
  void *dummy ;
  KEYSET kset = 0 ;
  int nClones = 0, nSingle = 0, nLoci = 0 ;

  map = arrayReCreate (map, 16, GRIDMAP) ;
 
  if (!keySetActive(&kset, &dummy))
    { messout("First select a keySet window, thank you.") ;
      return  ;
    }

  nYacHits = 0 ;

  for (i = 0 ; i < keySetMax(kset) ; ++i)
    if (gridClusterKey (keySet(kset,i), map))
      { ++nClones ;
	nLoci += arrayMax(map) ;
	if (arrayMax(map) == 1)
	  ++nSingle ;
      }
  
  printf ("%d clones, %d Yac hits, %d loci, %d singles\n",
	  nClones, nYacHits, nLoci, nSingle) ;
}

/*******************************/

static void gridMapMode (void)
{
  int i, x1, x2, npositives = 0 ;
  static Array positives, reverse ;
  SEG *seg ;
  GRIDMAP *pos ;
  OBJ obj ;
  LOOKGET("gridMap") ;

  look->flag &= ~FLAG_EDIT_MODE ;
  graphBoxDraw (MAP_MODE_BOX, WHITE, DARKRED) ;
  graphBoxDraw (EDIT_MODE_BOX, BLACK, WHITE) ;

  positives = arrayReCreate (positives, 16, GRIDMAP) ; 
  reverse = arrayReCreate (reverse, 16, int*) ;

  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      seg->mapIndex = 0 ;
      if ((seg->flag & FLAG_HYB) && (obj = bsCreate(seg->key)))
	{ pos = arrayp (positives, npositives, GRIDMAP) ;
	  array (reverse, npositives, int*) = &(seg->mapIndex) ;
	  if (bsGetKey (obj, _pMap, &pos->ctg) &&
	      bsGetData (obj, _bsRight, _Int, &x1) &&
	      bsGetData (obj, _bsRight, _Int, &x2))
	    { pos->x = 0.5 * (x1+x2) ;
	      ++npositives ;
	    }
	  else
	    --arrayMax(positives) ;
	  bsDestroy (obj) ;
	}
    }

  gridCluster (positives, look->map) ;
  for (i = 0 ; i < arrayMax(positives) ; ++i)
    *(arr(reverse, i, int*)) = 
      arrp(positives, i, GRIDMAP)->clump + 1 ;
}

/*********************************/

static void gridHybBuild (LOOK look, KEY probe)
{
  int i, box, *pFlag ;
  KEY key ;
  OBJ obj ;
  static Array column = 0 ;

  column = arrayReCreate (column, 24, BSunit) ;

  if (obj = bsCreate (probe))
    { if (bsGetKey (obj, _Clone, &key))
	do gridHybBuild (look, key) ; while (bsGetKey (obj, _bsDown, &key)) ;
      if (bsGetKey (obj, _Subpool, &key))
	do gridHybBuild (look, key) ; while (bsGetKey (obj, _bsDown, &key)) ;
      if (bsFindTag (obj, _Hybridizes_to) && bsFlatten (obj, 3, column))
	for (i = 0 ; i < arrayMax(column) ; i += 3)
	  while (gridKeyFind(look,arr(column, i+1, BSunit).k,&box))
	    { pFlag = &(arrp(look->segs, box, SEG)->flag) ;
	      if (arr(column, i+2, BSunit).s)
		{ if (!(*pFlag & FLAG_STRONG))
		    *pFlag |= FLAG_WEAK ;
		}
	      else
		{ *pFlag |= FLAG_STRONG ;
		  *pFlag &= ~FLAG_WEAK ;
		}
	    }
      bsDestroy (obj) ;
    }
}

static void gridProbeDisplay (LOOK look, KEY probe)
{
  int i ;
  BOOL empty = TRUE ;

  if (class(probe) == _VClone)
    look->flag &= ~FLAG_IS_POOL ; /* not gridCloneProbe because may come before drawing */
  else if (class(probe) == _VPool)
    look->flag |= FLAG_IS_POOL ;
  else
    { messout ("Sorry, I can only show hybridisation data for the "
	       "Clone and Pool classes.  %s is in the %s class",
	       name(probe), className(probe)) ;
      return ;
    }

  look->probe = probe ;
  strncpy (look->probeName, name(probe), 10) ;

				/* clear hybridisation flags */
  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    arrp(look->segs, i, SEG)->flag &= ~FLAG_HYB ;

  gridHybBuild (look, probe) ;
  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    if (arrp(look->segs, i, SEG)->flag & FLAG_HYB)
      empty = FALSE ;

  if (empty)
    messout ("Sorry - no hybridisation of %s to this grid", 
	     name (probe)) ;
  gridDraw (look) ;
}

static void gridProbeEntry (char* string)
{
  KEY probe = 0 ;
  LOOKGET("gridProbeEntry") ;

  gridMapMode () ;

  if (!lexword2key (string, &probe, 
		    (look->flag & FLAG_IS_POOL) ? _VPool : _VClone))
    messout ("Sorry, %s is not a %s name", string,
		    (look->flag & FLAG_IS_POOL) ? "pool" : "clone") ;
  else 
    gridProbeDisplay (look, probe) ;
}

static void gridProbeAction (KEY key)
{
  LOOKGET ("gridProbeAction") ;

  gridMapMode () ;
  gridProbeDisplay (look, key) ;
  displayRepeatBlock () ;
}

static void gridProbeButton (void)
{
  displayBlock (gridProbeAction, 
		"The hybridisation pattern for that object will be shown.\n"
		"Remove this message to cancel.") ;
}

/*************************************************/

static void gridClear (void)
{
  int i ;
  LOOKGET("gridClear") ;

  if ((look->flag & FLAG_EDITED) && 
      isWriteAccess() &&
      look->probe && 
      graphQuery ("Do you want to save the current pattern?"))
    gridSave () ;

  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    arrp(look->segs, i, SEG)->flag &= 
      ~(FLAG_HYB | FLAG_CHANGED) ;

  look->probe = 0 ;
  *look->probeName = 0 ;
  gridEditMode() ;
  look->flag &= ~FLAG_EDITED ;
  gridDraw (look) ;
}

/************************************/

static void gridKeyboard (int k)
{
  int i, ibest ;
  float x, y, best ;
  SEG *seg ;
  LOOKGET("gridKeyboard") ;

  if (!look->activeBox)
    return ;

  if (k == RETURN_KEY)
    { if (look->flag & FLAG_EDIT_MODE)
	gridEdit (look, look->activeBox) ;
      else
	gridFollow (look) ;
      return ;
    }

  seg = arrp(look->segs, look->activeBox, SEG) ;
  x = seg->x ;  y = seg->y ;
  ibest = 0 ;
  if (k == LEFT_KEY || k == UP_KEY)
    best = -1000.0 ;
  else
    best = 1000.0 ;

  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      switch (k)
      {
      case LEFT_KEY:
	if (seg->y < y+0.5 && seg->y > y-0.5 && seg->x < x && seg->x > best)
	  { best = seg->x ; ibest = i ; }
	break ;
      case RIGHT_KEY:
	if (seg->y < y+0.5 && seg->y > y-0.5 && seg->x > x && seg->x < best)
	  { best = seg->x ; ibest = i ; }
	break ;
      case UP_KEY:
	if (seg->x < x+1 && seg->x > x-1 && seg->y < y && seg->y > best)
	  { best = seg->y ; ibest = i ; }
	break ;
      case DOWN_KEY:
	if (seg->x < x+1 && seg->x > x-1 && seg->y > y && seg->y < best)
	  { best = seg->y ; ibest = i ; }
	break ;
      }
    }
  if (ibest)
    gridSelect (look, ibest) ;
}

static void gridTreeProbe (void)
{
  LOOKGET("gridTreeProbe") ;

  if (look->probe)
    display (look->probe, look->key, TREE) ;
}

/*************************************************/
/************* conversion and drawing ************/

void gridConvert (LOOK look)
{ 
  int i, row, n, n2 ;
  OBJ obj ;
  KEY key ;
  SEG *seg ;
  static BSMARK mark = 0 ;

  if (!(obj = bsCreate (look->key)))
    messcrash ("Can't make object in gridConvert") ;

  if (bsGetKey (obj, _Title, &key))
    look->title = name (key) ;
  else
    look->title = name (look->key) ;

  if (bsFindTag (obj, _No_stagger))
    look->flag &= ~FLAG_STAGGER ;
  else
    look->flag |= FLAG_STAGGER ;

  if (bsFindTag (obj, _A1_labelling))
    look->flag |= FLAG_A1 ;
  else
    look->flag &= ~FLAG_A1 ;

  look->dxLine = look->dyLine = look->dxSpace = look->dySpace = 0 ;
  bsGetData (obj, _Lines_at, _Int, &look->dxLine) ;
  bsGetData (obj, _bsRight,  _Int, &look->dyLine) ;
  bsGetData (obj, _Space_at, _Int, &look->dxSpace) ;
  bsGetData (obj, _bsRight,  _Int, &look->dySpace) ;

  look->segs = arrayReCreate (look->segs, 512, SEG) ;

  n = MIN_LIVE_BOX ;
  if (bsGetData (obj, _Row, _Int, &row)) do
    { mark = bsMark (obj, mark) ;
      row-- ;
      i = 0 ; 
      while (bsGetKey (obj, _bsRight, &key))
	{ seg = arrayp (look->segs, n, SEG) ;
	  seg->flag = 0 ;
	  if (!strcmp (name(key),"-"))
	    seg->flag |= FLAG_BLANK ;
	  seg->key = 0 ;
	  if (gridKeyFind (look, key, &n2))
	    { seg->flag |= FLAG_DUPLICATE ;
	      arrp(look->segs, n2, SEG)->flag |= FLAG_DUPLICATE ;
	    }
	  seg->key = key ;
	  seg->x = i ;
	  seg->y = row ;
	  ++n ; ++i ;
	}
      bsGoto (obj, mark) ;
    } while (bsGetData (obj, _bsDown, _Int, &row)) ;
  else
    messout ("No clones in Clone_Grid %s", name(look->key)) ;

  bsDestroy (obj) ;
}

/***************************************/

void gridDraw (LOOK look)
{
  int   xmax = 0, ymax = 0 ;
  int   i ;
  float x, y, x1, y1, x2, y2 ;
  SEG   *seg ;
  float yOff = 0.5 ;
  float xOff = 2 ;
  BOOL  isSymbolic = !(look->flag & FLAG_NAMES) ;
  int   itemWidth =  isSymbolic ? 2 : 7 ;

#define i2x(z)	((z)*itemWidth + \
		 (look->dxLine ? 0.5*((z)/look->dxLine) : 0) + \
		 (look->dxSpace ? 2*((z)/look->dxSpace) : 0))
#define j2y(z)	((z) + \
		 (look->dyLine ? 0.5*((z)/look->dyLine) : 0) + \
		 (look->dySpace ? (z)/look->dySpace : 0))
  
  graphClear () ;

  graphText (look->title, 1, yOff) ; /* title */

  x = 3 + strlen(look->title) ;
  if (graphBoxStart() != SELECT_BOX)
    messcrash ("box screwup at %d in gridDraw", SELECT_BOX) ;
  if (graphButton ("Gridded Clone:", gridSelectButton, x, yOff) 
      != SELECT_BUTTON_BOX)
    messcrash ("box screwup at %d in gridDraw", SELECT_BUTTON_BOX) ;
  *look->selectName = 0 ;
  if (graphCompletionEntry (cloneCompletion, look->selectName, 
			    10, x+16, yOff+0.2, gridSelectEntry)
      != SELECT_NAME_BOX)
    messcrash ("box screwup at %d in gridDraw", SELECT_NAME_BOX) ;
  graphBoxDim (SELECT_BOX, &x1, &y1, &x2, &y2) ;
/*  graphRectangle (x1-0.4, y1-0.2, x2+0.4, y2+0.2) ; */
  graphBoxEnd () ;

  if (graphButton ("Clear", gridClear, x2 + 3, yOff) != CLEAR_BOX)
    messcrash ("box screwup at %d in gridDraw", CLEAR_BOX) ;

  yOff += 1.6 ;
  x = 1 ;
  
  if (graphBoxStart() != PROBE_BOX)
    messcrash ("box screwup at %d in gridDraw", PROBE_BOX) ;
  if (graphButton ("Probe:", gridProbeButton, x, yOff) 
      != PROBE_BUTTON_BOX)
    messcrash ("box screwup at %d in gridDraw", PROBE_BUTTON_BOX) ;
  if (graphCompletionEntry (probeCompletion, look->probeName, 
			    10, x+8, yOff+0.2, gridProbeEntry)
      != PROBE_NAME_BOX)
    messcrash ("box screwup at %d in gridDraw", PROBE_NAME_BOX) ;

  if (graphButton ("Clone", gridCloneProbe, x+19, yOff)
      != CLONE_BOX)
    messcrash ("box screwup at %d in gridDraw", CLONE_BOX) ;
  if (graphButton ("Pool", gridPoolProbe, x+26, yOff)
      != POOL_BOX)
    messcrash ("box screwup at %d in gridDraw", POOL_BOX) ;
  graphBoxDim (PROBE_BOX, &x1, &y1, &x2, &y2) ;
/*  graphRectangle (x1-0.4, y1-0.2, x2+0.4, y2+0.2) ; */
  graphBoxEnd () ;

  x = x2 + 2 ;

  if (graphBoxStart() != MODE_BOX)
    messcrash ("box screwup at %d in gridDraw", MODE_BOX) ;
  graphText ("Mode:", x, yOff+0.2) ;
  if (graphButton ("Edit Mode", gridEditMode, x+6, yOff)
      != EDIT_MODE_BOX)
    messcrash ("box screwup at %d in gridDraw", EDIT_MODE_BOX) ;
  if (graphButton ("Map Mode", gridMapMode, x+17, yOff)
      != MAP_MODE_BOX)
    messcrash ("box screwup at %d in gridDraw", MAP_MODE_BOX) ;
  graphBoxDim (MODE_BOX, &x1, &y1, &x2, &y2) ;
/*  graphRectangle (x1-0.4, y1-0.2, x2+0.4, y2+0.2) ; */
  graphBoxEnd () ;

  if (look->flag & FLAG_EDIT_MODE)
    gridEditMode () ;
  else
    gridMapMode () ;
  if (look->flag & FLAG_IS_POOL)
    gridPoolProbe () ;
  else
    gridCloneProbe () ;

  yOff += 1.6 ;
  graphLine (0, yOff, 200, yOff) ;
  yOff += 0.5 ; 

  if (look->flag & FLAG_A1)	/* labelling as for Guy's grids */
    { char label[2] = {'A',0} ;
      xOff += 3 ;
      yOff += 2 ;
      for (i = 1 ; i <= 12 ; i++) /* top */
	graphText (messprintf ("%d", i), xOff + i2x(i*look->dxLine - 2.5), yOff - 1.5) ;
      for (i = 0 ; i < 8 ; i++) /* LH side */
	{ graphText (label, 2, yOff + j2y(1 + i*look->dyLine)) ;
	  ++label[0] ;
	}
    }
  else				/* default Cambridge labelling */
    { xOff += 4 ;
      graphRectangle (1, 4+yOff, 4, 20+yOff) ;
      graphText ("L", 2, 8+yOff) ;
      graphText ("A", 2, 10+yOff) ;
      graphText ("B", 2, 12+yOff) ;
      graphText ("E", 2, 14+yOff) ;
      graphText ("L", 2, 16+yOff) ;
    }

				/* draw comparison boxes */
  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      if (seg->flag & FLAG_BLANK || !(seg->flag & FLAG_STORE_HYB))
	continue ;
      x = i2x(seg->x) ;
      if ((look->flag & FLAG_STAGGER) && isSymbolic && (seg->y%2))
	x += 0.5 ;
      y = j2y(seg->y) ;

      if (!isSymbolic)
	x += strlen(name(seg->key)) ;

      if (seg->flag & FLAG_STORE_STRONG)
	graphColor (GREEN) ;
      else if (seg->flag & FLAG_STORE_WEAK)
	graphColor (YELLOW) ;

      graphFillRectangle (xOff+x-0.3, yOff+y-0.1, 
			  xOff+x+1.3, yOff+y+1.0) ;
      graphColor (DARKGREEN) ;
      graphRectangle (xOff+x-0.3, yOff+y-0.1, 
		      xOff+x+1.3, yOff+y+1.0) ;
    }
  graphColor (BLACK) ;
				/* draw main hybridisation boxes */
  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      x = i2x(seg->x) ;
      if ((look->flag & FLAG_STAGGER) && isSymbolic && (seg->y%2))
	x += 0.5 ;
      y = j2y(seg->y) ;

      if (i != graphBoxStart())
	messcrash ("minLiveBox wrong in gridDraw()") ;
      if (seg->flag & FLAG_BLANK)
	graphText ("-", xOff+x, yOff+y) ;
      else if (isSymbolic)
	graphRectangle (xOff+x, yOff+y+0.1, 
			xOff+x+0.9, yOff+y+0.8) ;
      else
	graphText (name(seg->key), xOff+x, yOff + y) ;
      graphBoxEnd () ;

      if (seg->flag & FLAG_STRONG)
	graphBoxDraw (i, BLACK, BLUE) ;
      else if (seg->flag & FLAG_WEAK)
	graphBoxDraw (i, BLACK, LIGHTBLUE) ;
      else
	graphBoxDraw (i, BLACK, WHITE) ;

      if (x > xmax)
	xmax = x ;
      if (y > ymax)
	ymax = y ;
    }

  if (!isSymbolic)		/* draw grids */
    {
				/* the grid of lines */
      if (look->dxLine)
	for (i = look->dxLine ; i < xmax ; i += look->dxLine)
	  { x = i2x(i) - 1 ;
	    graphLine (xOff+x, yOff, xOff+x, ymax+yOff+1) ;
	  }
      if (look->dyLine)
	for (i = look->dyLine ; i < ymax ; i += look->dyLine)
	  { y = j2y(i) - 0.25 ;
	    graphLine (xOff, y+yOff, xOff+xmax+itemWidth, y+yOff) ;
	  }
      
      graphColor (WHITE) ;	/* the grid of spaces */
      if (look->dxSpace)
	for (i = look->dxSpace ; i < xmax ; i += look->dxSpace)
	  { x = i2x(i) - 2.5 ;
	    graphFillRectangle (xOff+x, yOff, xOff+x+2, ymax+yOff+1) ;
	  }
      if (look->dySpace)
	for (i = look->dySpace ; i < ymax ; i += look->dySpace)
	  { y = j2y(i) - 1.5 ;
	    graphFillRectangle (xOff, y+yOff, xOff+xmax+itemWidth, y+yOff+1.5) ;
	  }
    }

  look->crossBox = graphBoxStart () ;
  graphLine (-1.0, -0.9, -0.1, -0.2) ;
  graphLine (-1.0, -0.2, -0.1, -0.9) ;
  graphBoxEnd () ;
  graphBoxDraw (look->crossBox, BLACK, TRANSPARENT) ;
  graphBoxSetPick (look->crossBox, FALSE) ;

  look->activeBox = 0 ;

/*  graphTextBounds (xOff+xmax+itemWidth+1, yOff+ymax+1) ; */

  graphRedraw () ;
}

/**********************************/

static void gridDump (void)	/* for debugging */
{
  int i ;
  SEG *seg ;
  GRIDMAP *map ;
  LOOKGET("gridDump") ;

  printf ("Dump of Clone_Grid %s\n", name(look->key)) ;
  printf ("  Flag is %d\n", look->flag) ;
  printf ("  Segs:\n") ;
  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      if (seg->flag)
	printf ("    %8s  %3d %3d  %4d  %2d\n",	name(seg->key),
		seg->x, seg->y, seg->flag, seg->mapIndex) ;
    }
  printf ("  Map:\n") ;
  for (i = 0 ; i < arrayMax(look->map) ; ++i)
    { map = arrp(look->map, i, GRIDMAP) ;
      printf ("    %6s  %f\n", name(map->ctg), map->x) ;
    }
}

/**********************************/

static void gridSave (void)
{
  OBJ obj ;
  int i ;
  KEY key, weak, text ;
  SEG *seg ;
  static BSMARK mark = 0 ;
  BOOL isPool ;
  LOOKGET("gridSave") ;

  if (!isWriteAccess())
    { messout ("Sorry - you do not have write access") ;
      return ;
    }

  isPool = (look->flag & FLAG_IS_POOL) ;

  if (!look->probeName || 
      !lexword2key (look->probeName, &key, isPool ? _VPool : _VClone))
    { messout ("Please type a valid %s name into the probe name box "
	       "and reselect this option.  You can also use the "
	       "Clone/Pool buttons to switch probe type.\n"
	       "Do NOT hit 'Return' in the probe name box if you want "
	       "to save the current pattern.", 
	       isPool ? "pool" : "clone") ;
      return ;
    }

  if (look->probe && key != look->probe && 
      !graphQuery (messprintf ("Probe name or type has been edited - "
			       "save pattern in %s %s?", 
			       className(key), name(key))))
    return ;

  look->probe = key ;
  if (!(obj = bsUpdate (key)))
    { messout ("Sorry, object %s is locked", name(key)) ;
      return ;
    }

  lexaddkey ("weak", &weak, _VText) ;

  bsAddKey (obj, _Hybridizes_to, look->key) ;
  mark = bsMark (obj, mark) ;
  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      if ((seg->flag & FLAG_HYB) && (seg->flag & FLAG_CHANGED))
	{ bsGoto (obj, mark) ;
	  bsAddKey (obj, _bsRight, seg->key) ;
	  if ((seg->flag & FLAG_WEAK) && 
	      !bsGetKey (obj, _bsRight, &text))
	    bsAddKey (obj, _bsRight, weak) ;
	}
      seg->flag &= ~FLAG_CHANGED ;
    }
				/* now check deletes */
 delete_check:
  bsGoto (obj, mark) ;
  if (bsGetKey (obj, _bsRight, &key))
    do
      { if (!gridKeyFind (look, key, &i) ||
	    !(arrp(look->segs, i, SEG)->flag & FLAG_HYB))
	  { bsRemove (obj) ;	/* returns to root of tree */
	    goto delete_check ;
	  }
      } while (bsGetKey (obj, _bsDown, &key)) ;

  bsSave (obj) ;

  look->flag &= ~FLAG_EDITED ;
}

/**************************/

static BOOL gridKeyFind (LOOK look, KEY clone, int *pbox)
{
  static KEY oldClone ;
  static int ibase, oldBox ;
  int i ;

  if (clone != oldClone || *pbox != oldBox)
    ibase = MIN_LIVE_BOX ;

  for (i = ibase ; i < arrayMax(look->segs) ; ++i)
    if (arrp(look->segs, i, SEG)->key == clone)
      { oldClone = clone ;
	oldBox = *pbox = i ;
	ibase = i+1 ;
	return TRUE ;
      }

  return FALSE ;
}

/************************/

void gridComparisonStore (void)
{
  int i ;
  int tmp ;
  KEY keyTmp ;
  SEG *seg ;
  LOOKGET("gridComparisonStore") ;

  for (i = MIN_LIVE_BOX ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs, i, SEG) ;
      tmp = seg->flag & (FLAG_STORE_HYB | FLAG_STORE_CHANGED) ;
      seg->flag &= ~(FLAG_STORE_HYB | FLAG_STORE_CHANGED) ;
      seg->flag |= (seg->flag & (FLAG_HYB | FLAG_CHANGED)) << 4 ;
      seg->flag &= ~(FLAG_HYB | FLAG_CHANGED) ;
      seg->flag |= tmp >> 4 ;
    }

  keyTmp = look->probeStore ;
  look->probeStore = look->probe ;
  look->probe = keyTmp ;
  if (keyTmp)
    { strncpy (look->probeName, name(keyTmp), 10) ;
      if (class (keyTmp) == _VClone)
	gridCloneProbe () ;
      else
	gridPoolProbe () ;
    }
  else
    *look->probeName = 0 ;
  graphBoxDraw (PROBE_NAME_BOX, -1, -1) ;

  gridDraw (look) ;
}

/***********************************/

void gridSelectEntry (char* string)
{
  KEY key = 0 ;
  int box ;
  LOOKGET("gridSelectEntry") ;

  if (!lexword2key (string, &key, _VClone))
    messout ("Sorry, %s is not a clone name", string) ;
  else if (!gridKeyFind(look, key, &box))
    messout ("Sorry, that clone is not on this grid") ;
  else
    gridSelect (look, box) ;
}

static void gridSelectAction (KEY key)
{
  int box ;
  LOOKGET("gridSelectAction") ;

  if (gridKeyFind(look, key, &box))
    gridSelect (look, box) ;
  else
    messout ("Sorry, %s is not on this grid", name(key)) ;
  displayRepeatBlock () ;
}

static void gridSelectButton (void)
{
  displayBlock (gridSelectAction, 
		"When you pick a clone that exists on this grid "
		"its location will be indicated by a cross.\n"
		"Remove this message to cancel.") ;
}

/************************************/

static void gridChangeAction (KEY key)
{
  LOOKGET("gridChangeAction") ;

  if (!isWriteAccess())
    { messout ("Sorry, you lost write access.  Regain it and try again.") ;
      return ;
    }
  if (!look->activeBox)
    { messout ("Sorry, there is no current gridded clone (indicated by "
	       "name on the top line and a cross on the diagram) to be "
	       "replaced.") ;
      return ;
    }
  if (class (key) != _VClone)
    { messout ("Sorry, only clones can be placed on a Clone_Grid, and "
	       "%s is not a clone.", name (key)) ;
      return ;
    }

  gridClear () ;		/* clear everything */
  gridComparisonStore () ;
  gridClear () ;

  messout ("This doesn't do anything yet.  It is more complicated than "
	   "I thought.  Can I have a special kernel function to help?") ;
}

static void gridChangeClone (void)
{
  if (!isWriteAccess())
    { messout ("You must have write access to change a clone in the grid.") ;
      return ;
    }

  displayBlock (gridChangeAction,
		"The clone that you pick will replace the current active "
		"clone (indicated by name and a cross on the diagram).  "
		"Before doing this the current hybridisation pattern(s) will "
		"be cleared.\n"
		"Remove this message to cancel.") ;

}

/************** end of file ****************/
