/*  File: cmapdisp.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: physical unit display of a chromosome
 	based on gmap
 * Exported functions: cmapDisplay()
 * HISTORY:
 * Last edited: Apr  2 18:04 1992 (mieg)
 * Created: Thu Jan  9 22:54:23 1992 (rd)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "keyset.h"
#include "graph.h"
#include "key.h"
#include "lex.h"
#include "bs.h"
#include "a.h"
#include "systags.wrm"
#include "classes.wrm"
#include "sysclass.wrm"
#include "tags.wrm"
#include "session.h"
#include "disptype.wrm"
#include "display.h"
#include "query.h"
#include "wormtype.h"

/* the basic units are Mb - left end is 0
   assume that the length is 16 Mb
*/

typedef struct LOOKSTUFF
  { int   magic ;	/* == MAGIC */
    KEY   key ;		/* a chromosome */
    int   activeBox ;
    unsigned int flag ;
    float centre, mag ;	
    float min, max ;
    Array segs,      	/* array of SEG's from the obj */
          boxIndex ;    /* if >256 a SEG, else something special */
  } *LOOK ;

#define MAGIC 361718

typedef struct
  { KEY key ;
    float x1, x2 ;
    int flag ;
  } SEG ;
				/* seg->flag flags */
#define FLAG_HIGHLIGHT		0x0001

#define LOOKGET(name)     LOOK look ; \
                          if (!graphAssFind (cMapDisplay,&look)) \
		            messcrash ("graph not found in %s",name) ; \
			  if (!look) \
                            messcrash ("%s received a null pointer",name) ; \
                          if (look->magic != MAGIC) \
                            messcrash ("%s received a wrong pointer",name)

static void cMapDestroy (void) ;
static void cMapRecalculate(void) ;
static void cMapPick (int box, double x , double y) ;
static void cMapDragCursor (float *x, float *y, BOOL isDone) ;
static void cMapMiddleDown (double x, double y) ;
static void cMapDrag (double x, double y) ;
static void cMapUp (double x, double y) ;
static void cMapDraw (LOOK look, KEY key) ;
static BOOL cMapConvert (LOOK look) ;
static void cMapResize (void) ;
static void cMapSelect (LOOK look, int box) ;
static void cMapFollow (LOOK look, double x, double y) ;
static void cMapHighlight (void) ;
static void cMapAdd (void) ;
static void cMapSubtract (void) ;
static void cMapWhole (void) ;
static int  cMapOrder(void *a, void *b) ; /* for arraySort() call */

static MENUOPT cMapMenu[] =
              { graphDestroy, "Quit",
		help,"Help",
		graphPrint,"Print",
		displayPreserve,"Preserve",
		cMapRecalculate,"Clear",
		cMapHighlight,"Highlight Selected Objects",
		cMapAdd,"Add selected objects",
		cMapSubtract,"Subtract selected objects",
		0, 0
	      } ;

static int	cursorBox ;
static int	minLiveBox ;

extern BOOL gMapGetPos(KEY from, KEY *chromo, float *x1, float *x2) ;

/**********************/

BOOL cMapDisplay (KEY key, KEY from, BOOL isOldGraph)
{
  LOOK look=(LOOK)messalloc(sizeof(struct LOOKSTUFF)) ;
  float x1, x2 ;

  if (key && class(key) != _VChromosome)
    { from = key ;
      key = 0 ;
    }
  if (from && !gMapGetPos(from, &key, &x1, &x2))
    if (key)
      from = 0 ;
    else
      display (from, 0, TREE) ;
  
  if (!key)
    goto abort ;

  look->key = key ;
  if (!cMapConvert (look))
    goto abort ;

  look->boxIndex = arrayCreate (64,SEG*) ;
  look->activeBox = 0 ;
  look->magic = MAGIC;
  look->flag = 0 ;
  look->min = 0.0 ;
  look->max = 16.0 ;		/* arbitrary for now */
  
  if (isOldGraph)
    { graphRetitle (name (key)) ;
      cMapDestroy () ;
      graphAssRemove (cMapDisplay) ;
    }
  else 
    { if (!displayCreate (CMAP))
	goto abort ;
    
      graphRegister (RESIZE,(GraphFunc)cMapResize) ;
      graphRegister (DESTROY, cMapDestroy) ;
      graphRegister (PICK, (GraphFunc)cMapPick) ;
      graphRegister (MIDDLE_DOWN, (GraphFunc)cMapMiddleDown) ;
      graphRegister (MIDDLE_DRAG, (GraphFunc) cMapDrag) ;
      graphRegister (MIDDLE_UP, (GraphFunc) cMapUp) ;
      graphMenu (cMapMenu) ;
    }

  graphAssociate (cMapDisplay,look) ;

  cMapWhole () ;	/* sets ->centre, ->mag and calls Draw() */
  return TRUE ;

abort :
  messfree (look) ;
  return FALSE ;
}

/*********************/

static void cMapDestroy (void)
{
  LOOKGET("cMapDestroy") ;

  arrayDestroy(look->segs) ;
  arrayDestroy(look->boxIndex) ;

  look->magic = 0 ;
  messfree (look) ;
}

/**********************************/

static void cMapResize (void)
{
  LOOKGET("cMapResize") ;

  cMapDraw (look, 0) ;
}

/***********************************/

static void cMapPick (int box, double x, double y) 
{
  LOOKGET("cMapPick") ;

  if (!box)
    return ;

  if (box == cursorBox)
    graphBoxDrag (cursorBox, cMapDragCursor) ;
  else if (box = look->activeBox)
    cMapFollow (look, x, y) ;
  else
    cMapSelect (look, box) ;
}

/*********************/

static void cMapRecalculate (void)
{
  KEY curr ;
  LOOKGET("cMapRecalculate") ;
  
  if (look->activeBox)
    curr = arr(look->boxIndex,look->activeBox,SEG*)->key ;
  else 
    curr = 0 ;
  cMapConvert (look) ;
  cMapDraw (look, curr) ;
}

static void cMapSelect (LOOK look, int box) {}

static void cMapFollow (LOOK look, double x, double y) {}

/**************************************************/
/**************** drawing info ********************/

static int	nx, ny ;	/* window dimensions */
static float	yCentre ;	/* ny/2 */
static float	yLength ;	/* length of picture */
static float	topMargin = 2 ;	/* space at top for buttons etc */
static float	bottomMargin = 1 ; /* space at bottom */
static float	xCursor = 5 ;	/* X coord of mini-chromosome */
static float	xScale = 10 ;	/* X coord of scale bar */

#define MAP2GRAPH(look,x) \
  (yCentre  +  look->mag * ((x) - look->centre))
#define GRAPH2MAP(look,x) \
  (((x) - yCentre) / look->mag + look->centre)

/********* start off with some utility routines ********/

static void getNxNy(void)
{
  graphFitBounds (&nx, &ny) ;
  yLength = (ny - topMargin - bottomMargin) ;
  yCentre = topMargin + 0.5*yLength ;
}

/***************************************/

static BOOL cMapGetCurrentPos(LOOK look, KEY key, float *y)
{ 
  int i ;
  SEG *seg ;

  for (i = 0 ; i < arrayMax(look->segs) ; ++i)
    { seg = arrp(look->segs,i,SEG) ;
      if (seg->key == key)
	{ *y = 0.5 * (seg->x1 + seg->x2) ;
	  return TRUE ;
	}
    }
  return FALSE ;
}

static void cMapWhole (void)
{ 
  LOOKGET("cMapWhole") ; 
  getNxNy() ;
  look->centre = 0.5 * (look->min + look->max) ;
  look->mag = yLength / (look->max - look->min) ;
  cMapDraw (look, 0) ;
}

static void cMapZoomIn (void)
{ 
  LOOKGET("cMapZoomIn") ; 
  look->mag *= 2 ; 
  cMapDraw (look, 0) ;
}

static void cMapZoomOut (void)
{ 
  LOOKGET("cMapZoomOut") ; 
  look->mag /= 2 ; 
  cMapDraw (look, 0) ;
}

/**************************************************************/
/***************** dragging code - middle button **************/

static double	oldy, oldDy, oldx;
static BOOL	dragFast ;
#define DRAGFASTLIMIT xScale

static void cMapDrag (double x, double y) 
{
  if (dragFast)
    { graphXorLine (0, oldy - oldDy, DRAGFASTLIMIT, oldy - oldDy) ;
      graphXorLine (0, oldy + oldDy, DRAGFASTLIMIT, oldy + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, oldy, nx, oldy) ;

  oldy = y ;

  if (dragFast)
    { oldDy *= exp ((x - oldx) / 25.) ;
      oldx = x ;
      graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, nx, y) ;
}

static void cMapUp (double x, double y) 
{ 
  float x1,x2,y1,y2 ;
  LOOKGET("cMapUp") ;

  if (dragFast)
    { graphBoxDim (cursorBox, &x1, &y1, &x2, &y2) ;
      look->mag *= (y2 - y1) / (2. * oldDy) ;
      look->centre = look->min  +  (look->max - look->min) * \
	(y - topMargin) / yLength ;
    }
  else
    look->centre +=  (y - 0.5 - yCentre) / look->mag ;
  cMapDraw (look, 0) ;
}

static void cMapMiddleDown (double x, double y) 
{  
  float x1,x2,y1,y2 ;
  LOOKGET("cMapMiddleDown") ;

  getNxNy () ; 

  graphBoxDim (cursorBox, &x1, &y1, &x2, &y2) ;
  oldDy = (y2 - y1) / 2. ;

  dragFast = (x < DRAGFASTLIMIT) ? TRUE : FALSE ;

  if(dragFast)
    { graphXorLine (0, y - oldDy, DRAGFASTLIMIT, y - oldDy) ;
      graphXorLine (0, y + oldDy, DRAGFASTLIMIT, y + oldDy) ;
    }
  else
    graphXorLine (DRAGFASTLIMIT, y, nx, y) ;
   
  oldx = x ;
  oldy = y ;
  graphRegister (MIDDLE_DRAG, (GraphFunc) cMapDrag) ;
  graphRegister (MIDDLE_UP, (GraphFunc) cMapUp) ;
}

static void cMapDragCursor (float *x, float *y, BOOL isDone)
{
  if (isDone)
    { float x1,y1,x2,y2 ;
      LOOKGET("cMapDragCursor") ;

      getNxNy() ;
      graphBoxDim (cursorBox, &x1, &y1, &x2, &y2) ;
      look->centre = look->min + (look->max - look->min) *
	((*y + 0.5*(y2-y1)) - topMargin) / yLength ;
      cMapDraw (look, 0) ;
    }
  else
    *x = xCursor - 0.5 ;
}

/**************************************************/
/**************************************************/

static void drawScale (LOOK look)
{
  float cutoff = 5 / look->mag ;
  float unit = 0.01 ;
  float subunit = 0.001 ;
  float x,y ;

  while (unit < cutoff)
    { unit *= 2 ;
      subunit *= 5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2.5 ;
      if (unit >= cutoff)
	break ;
      unit *= 2 ;
      subunit *= 2 ;
    }

  x = GRAPH2MAP(look, topMargin) ;
  x = unit * (((x>=0)?1:0) + (int)(x/unit)) ;
  while ((y = MAP2GRAPH(look, x)) < ny  - 1)
    { 
      graphLine (xScale-1.5,y,xScale-0.5,y) ;
      if(unit >= 1)
	graphText (messprintf ("%-4.0f",x),xScale,y-.5) ;
      else if(unit >= .1)
	graphText (messprintf ("%-4.1f",x),xScale,y-.5) ;
      else if(unit >= .01)
	graphText (messprintf ("%-4.2f",x),xScale,y-.5) ;
      else if(unit >= .001)
	graphText (messprintf ("%-4.3f",x),xScale,y-.5) ;
      x += unit ;
    }

  x = look->centre - (yCentre-topMargin)/look->mag ;
  x = subunit * (((x>=0)?1:0) + (int)(x/subunit)) ;
   while ((y = MAP2GRAPH(look, x)) < ny  - 1)
    { 
      graphLine (xScale-1.0,y,xScale-0.5,y) ;
      x += subunit ;
    }

  graphLine (xScale-0.5, topMargin,
	     xScale-0.5, ny - bottomMargin) ;
}

static void drawChromosome (LOOK look)
{
  int i ;

#define MAP2CHROM(x) \
  (topMargin + yLength * ((x)-look->min) / (look->max-look->min))

  graphFillRectangle (xCursor - 0.25, topMargin, 
		      xCursor + 0.25, ny - bottomMargin) ;

  cursorBox = graphBoxStart() ;
  array(look->boxIndex,cursorBox,SEG*) = 0 ;
  graphRectangle (xCursor - 0.5, 
		  MAP2CHROM(GRAPH2MAP(look, topMargin)), 
		  xCursor + 0.5, 
		  MAP2CHROM(GRAPH2MAP(look, ny - bottomMargin))) ;
  graphBoxEnd () ;
  graphBoxDraw (cursorBox,DARKGREEN,GREEN) ;

  graphColor (DARKGRAY) ;
  graphLine (xCursor, MAP2CHROM(GRAPH2MAP(look,topMargin)), 
	     xScale-0.5, topMargin ) ;
  graphLine (xCursor, MAP2CHROM(GRAPH2MAP(look,ny - bottomMargin)),
	     xScale-0.5, ny - bottomMargin) ;
  graphColor (BLACK) ;

  graphTextHeight (0.75) ;
  for (i = 0 ; i <= 10 ; ++i)
    graphText (messprintf ("%3d%%",10*i),
	       1, topMargin + yLength * i / 10.0 - 0.25) ;
  graphTextHeight (0) ;
}

/***********************************************/


static MENUOPT buttonOpts[] = {
  cMapWhole, "Whole", 
  cMapZoomIn, "Zoom In",
  cMapZoomOut, "Zoom Out",
  0, 0} ;

static void cMapDraw (LOOK look, KEY curr)
{
  graphClear () ;
  graphColor (BLACK) ;
  getNxNy () ;
  if (yLength < 6)
    { messout ("Sorry, this window is too small for a chromo map") ;
      return ;
    }
  look->activeBox = 0 ;

  drawScale (look) ;
  drawChromosome (look) ;

  graphButtons (buttonOpts, 5, 0.5, nx) ; /* at end of mapDraw() */
  graphRedraw () ;
}

/*********************************************************/
/********************* conversion code *******************/

static int cMapOrder(void *a, void *b) /* for arraySort() call */
{
  SEG *seg1 = (SEG*)a, *seg2 = (SEG*)b ;
  float diff = seg1->x1 - seg2->x1 ;

  if (diff > 0)
    return 1 ;
  else if (diff < 0)
    return -1 ;
  else
    return 0 ;
}

static BOOL cMapConvert (LOOK look)
{
  return TRUE ;
}

static void cMapAdd (void) {}

static void cMapSubtract (void) {}

static void cMapHighlight (void) {}
