/*  File: bssubs.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: interface to the 'B' classes - tree objects
 		public declarations in bs.h
		private stuff in bs_.h (restricted to other w6 files)
 * Exported functions:
      The following preceeded by obj - used by objcache.c
        Get, Store, StoreNoCopy, Destroy, Copy
      The following all preceeded by bs
 	Save
	GetTag, GetKey, GetClass, GetData, Flatten
 	AddTag, AddKey, AddData, AddComment
	Type, TypeCheck
	Mark, Goto
	Remove, Prune
	FuseModel	(treedisp.c)
	MakePaths	(model.c)
 * HISTORY:
 * Last edited: Apr  3 18:31 1992 (mieg)
 * * Feb 21 14:05 1992 (mieg): getObj, addObj etc - not complete yet (remove etc)
 * * Jan 13 16:47 1992 (mieg): more asn stuff, rd's bsfuse
 * * Jan  8 19:02 1992 (mieg): messcrah when model is missing (I guess in wspec)
 * * Oct 22 13:30 1991 (mieg): added bsVisible.. to better queryGrep searches
 * * Oct 16 18:04 1991 (rd): fixed bug in xref deleting
 * Created: Wed Oct 16 17:33:54 1991 (rd)
 *-------------------------------------------------------------------
 */

#include <ctype.h>

#include "acedb.h"
#include "bs_.h"  /* Prototypes of private functions of bs package */
#include "bs.h"   /* Prototypes of public functions of bs package */
#include "cache.h"  /* Prototypes of public functions of cache package */
#include "lex.h"
#include "pick.h"
#include "systags.wrm"
#include "sysclass.wrm"
#include "chrono.h"

int BSTEST = 0 ;
#undef DELETE_DEBUG

static void     OBJfree (OBJ obj) ;
static void     makePaths (Associator ass, BS bs, int isUnique) ;
static BOOL     findTag (OBJ obj, KEY tag, BOOL makeIt) ;
static void	xrefPrune (OBJ obj, BS bs, BS bsm) ;

static BOOL     isXrefing = TRUE ;      /* disabled during doXref */

static Associator tabAssPath[256] ;     /* rely on clearing to 0 */
static BS         tabModel[256] ;

#define OBJ_MAGIC 294756

/* NB I assume that the first call to get or add functions in a
   sequence uses a tag, not _bsRight, _bsDown.  This means that I
   only need to check that an object is valid in bsFind/addTag.
   Also, I always leave x->curr pointing to a node that fits the
   model, and x->modCurr pointing to the respective model node.
*/


/**************************************************************/
/**************************************************************/
/* Manage OBJ allocation
   hope they speed things up/prevent fragmentation
   in any case, they will help monitoring memory usage
   NB they must be paired.
*/
/**************************************************************/
/**************************************************************/
 
static Stack freeOBJstack = 0 ;
static int nOBJused = 0, nOBJalloc = 0 ;        /* useful for debug */

OBJ OBJalloc (void)       /* self managed calloc */
{
  static int blocSize = 2048 ;
  OBJ p ;
  int i ;
 
  if (!freeOBJstack)
    freeOBJstack = stackCreate (4*blocSize) ;
  if (stackEmpty (freeOBJstack))
    { p = (OBJ) messalloc (blocSize * sizeof(struct sobj)) ;
      for (i = blocSize ; i-- ; ++p)
        push (freeOBJstack,p,OBJ) ;
      nOBJalloc += blocSize ;
      blocSize *= 2 ;
    }
  p = pop (freeOBJstack,OBJ) ;
  memset (p, 0, sizeof (struct sobj)) ;
  ++nOBJused ;
  p->magic = OBJ_MAGIC ;
  return p ;
}

void OBJfree (OBJ obj)
{ if (!obj)
    return ;
  if (obj->magic == OBJ_MAGIC)
    { obj->magic = 0 ;
      stackDestroy(obj->xref) ;
      push (freeOBJstack,obj,OBJ) ;
      --nOBJused ;
    }
  else
    messcrash("Bad obj in OBJfree") ;
}

void OBJstatus (int *used, int *alloc)
{ *used = nOBJused ; *alloc = nOBJalloc ;
}

/**************************************************************/
/**************************************************************/

/**************************************************************/
/**************************************************************/
/*             Object oriented operations                     */
/********************   Creations *****************************/
/**************************************************************/

                 /* cannot fail */
OBJ bsCreate (KEY key)
{
 OBJ obj ;
 BS bs ;
 CACHE x ;

 chrono("bsCreate") ;

 if (iskey(key) != 2 ||
      pickType(key) != 'B' ||
     !(x = cacheCreate(key, &bs)))
   return 0 ;

 obj = OBJalloc() ;

 obj->key = key ;
 obj->magic = OBJ_MAGIC ;
 obj->x = x ;
 obj->parent = 0 ;
 obj->numberOfChildren = 0 ;
 obj->curr = obj->root = bs ;

 if (!tabModel[class(key)] && KEYKEY(key))
   bsMakePaths (class(key)) ;
 obj->modCurr = tabModel[class(key)] ;
 obj->xref = 0 ;

 chronoReturn();
 return obj ;
}

/**************************************************************/
#ifndef READONLY
                 /* cannot fail */
OBJ bsUpdate (KEY key)
{
 OBJ obj ;
 BS bs ;
 CACHE x ;

 if (pickType(key) != 'B')
   messcrash ("bsUpdate called on non B key %s:%s",
	      className(key), name(key)) ;

 x = cacheUpdate(key, &bs) ;
 
 if (!x)
   return 0 ;

 chrono("bsUpdate") ;

 obj = OBJalloc() ;

 obj->key = key ;
 obj->magic = OBJ_MAGIC ;
 obj->x = x ; 
 obj->parent = 0 ;
 obj->numberOfChildren = 0 ;
 obj->curr = obj->root = bs ;

 if (!tabModel[class(key)] && KEYKEY(key))
   bsMakePaths (class(key)) ;
 obj->modCurr = tabModel[class(key)] ;
 obj->xref = 0 ;

 chronoReturn();
 return obj ;
}

#endif
/**************************************************************/
              /* Gets a #Class sub tree */
BOOL bsGetObj(OBJ obj, KEY target, OBJ *objp)
{ 
  OBJ new ;
  KEY type, t ;
  *objp = 0 ;

  if (target != _bsHere && !bsFindTag(obj, target))
    return FALSE ;
  t = bsType (obj,_bsRight) ;
  type = class(t) ;
  if (!type || t != KEYMAKE(type, 1))
    return FALSE ;
  
  /* think of xref and of destroying it */

  *objp = new = OBJalloc() ;
  new->magic = OBJ_MAGIC ;
  new->key = KEYMAKE(type, 0) ;
  new->x = obj->x ;
  new->parent = obj ;
  obj->numberOfChildren++ ;
  new->curr = new->root = obj->curr ;
  if (!tabModel[type])
    bsMakePaths (type) ;
  new->modCurr = tabModel[type] ;
  new->xref = 0 ;

  return TRUE ;
}

/**************************************************************/
              /* Gets a #Class sub tree */
#ifndef READONLY

BOOL bsAddObj(OBJ obj, KEY target, OBJ *objp)
{   
  *objp = 0 ;

  if (!isCacheLocked(obj->x))
    messcrash ("bsAddObj fails because %s is not locked\n",
               name (obj->key)) ;
  if (target != _bsHere)
    bsAddTag(obj, target) ; /* may fail if target already exists */
     
  return 
    bsGetObj(obj, target, objp) ;
        /* Check type as authorized by model */
        /* think of xref and of destroying it */
}
#endif

/**************************************************************/
/**************  Destructions   *******************************/
/**************************************************************/

void  bsDoDestroy(OBJ obj)
{
  if (!obj)
    return ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsDoDestroy received  a non magic B obj (%s)",name(obj->key)) ;
  if (obj->numberOfChildren)
    messcrash ("Trying to bsDestroy an  object %s, before destroying the children"
	       "that you created with the command bsGetObj()",name(obj->key)) ;

  if(obj->parent)  /* Simply release this subHandle */
    { if(!obj->parent->numberOfChildren--)
	messcrash("Parent missing link to his child in bsSave") ;
     }
  else
    cacheDestroy (obj->x) ; /*bsTreePrune (obj->root) ; */
    
  OBJfree(obj) ;
}

/**************************************************************/

#ifndef READONLY
static void     doXref (OBJ obj) ;

void  bsDoKill (OBJ obj)
{
  if (!obj)
    return ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsKill received  a non magic B obj (%s)",name(obj->key)) ;
  if (obj->numberOfChildren)
    messcrash ("Trying to bsKill an  object %s, before destroying the children"
	       "that you created with the command bsGetObj()",name(obj->key)) ;

  if (obj->parent)
    messcrash("You can't kill a child, you criminal") ;
  
  if (obj->key != obj->root->key)
    messcrash("key mismatch in objStore %s != %s",
	      name(obj->key), name(obj->root->key));
  
  obj->curr = obj->root ;
  obj->modCurr = tabModel[class(obj->key)] ;
  while (bsGetKeyTags (obj, _bsRight, 0))
    bsRemove (obj) ;
  if (obj->xref)
    doXref (obj) ;
  cacheSave (obj->x) ;  /* Note that we keep obj->root on disk */
  lexSetStatus (obj->key, EMPTYSTATUS) ;
  OBJfree (obj) ;
}
  
/**************************************************************/

void bsDoSave(OBJ obj)
{
  if (!obj)
    return ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsDoSave received  a non magic B obj (%s)",name(obj->key)) ;
  if (obj->numberOfChildren)
    messcrash ("Trying to bsSave object %s, before destroying the children"
	       "that you created withthe command bsGetObj()",name(obj->key)) ;
 
  if (obj->parent)  /* Simply release this subHandle */
    { if (!obj->parent->numberOfChildren--)
	messcrash ("Parent missing link to his child in bsSave") ;
      if (obj->xref)
	{ if (obj->parent->xref)
	    { stackCursor(obj->xref, 0) ;
	      while (!stackAtEnd (obj->xref))
		push(obj->parent->xref, stackNext(obj->xref,KEY), KEY) ;
	    }
	  else
	    obj->parent->xref = obj->xref ;
	}
     }
  else
    {  if (!obj->root)
	 messcrash ("bsDoSave says : Jean owes a beer to Richard") ;
       if (!obj->root->right)
	 { bsKill(obj) ;
	   return ;
	 }

       if (obj->key != obj->root->key)
	messcrash("key mismatch in objStore %s != %s",
		  name(obj->key), name(obj->root->key));
      
      if (obj->xref)
	doXref (obj) ;
	
      cacheSave (obj->x) ;
      lexUnsetStatus (obj->key, EMPTYSTATUS) ;
    }
  OBJfree(obj) ;
}

#endif

/**************************************************************/
/*                  Local surgery                             */
/**************************************************************/
/**************** first the get package ***********************/

/* if tag present, returns TRUE and moves curr, else nothing */

BOOL bsFindTag (OBJ obj, KEY target)
{
  BS    bs, bsm ;

  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsFindTag called with null or bad object");

  switch (target)
    {
    case _bsRight : 
      bs = obj->curr->right ; 
      bsm = bsModelRight(obj->modCurr) ; 
      if (!bs || !bsm)		/* off the end of the model */
	return FALSE ;
      obj->curr = bs ;
      obj->modCurr = bsm ;
      return TRUE ;
    case _bsDown :  
      messcrash ("bsFindTag called on bsDown, the result cannot be guaranteed") ;
      return FALSE ;
    case _bsHere:   
      return TRUE ;
    default:
      return findTag (obj, target, FALSE) ;      /* FALSE -> do not grow path  */
    }
}

/*******************************************************************/

        /* basic routine to read from an object
           target = tag or _bsRight or _bsDown or _bsHere (relative to curr)
	   ignores comments
           if there is something there,
             fills found, updates curr and returns TRUE
           else
             returns FALSE
        */

static BOOL bsGetKey2 (OBJ obj, KEY target, KEY *found, BOOL alsoGetTags)
{
  BS    bs, bsm ;

  bs = obj->curr ;
  bsm = obj->modCurr ;

  switch (target)
    {
    case _bsRight : 
      bs = bs->right ; 
      bsm = bsModelRight(bsm) ; 
      break ;
    case _bsDown :  
      bs = bs->down ; 
      break ;
    case _bsHere:   
      break ;
    default:
      if (!findTag (obj, target, FALSE))
        return FALSE ;
      bs = obj->curr->right ;   /* go right from tag */
      bsm = bsModelRight (obj->modCurr) ;
    }

  if (!bsm)		/* off the end of the model */
    return FALSE ;

  while (bs)
    { if (!bsIsComment(bs) && (class(bs->key) || alsoGetTags))
        { if (found)
            *found = bs->key ;
	  bsModelMatch (bs, &bsm) ;
          obj->curr = bs ;
	  obj->modCurr = bsm ;
          return TRUE ;
        }
      bs = bs->down ;     /* iterate downwards, ignoring user comments */
    }

   return FALSE ;
}

BOOL bsGetKey (OBJ obj, KEY target, KEY *found)
{
  return
    bsGetKey2(obj,target,found,FALSE) ;
}

   /* Also get tags, used by query package */
BOOL bsGetKeyTags (OBJ obj, KEY target, KEY *found)
{
  return
    bsGetKey2(obj,target,found,TRUE) ;
}

/*******************************************************************/

        /* matches to bs->key type and returns TRUE if found,
             else goes to bottom of column and returns FALSE
           can call with x=0 to test if match found and update curr
        */

BOOL bsGetData (OBJ obj, KEY target, KEY type, void *x)
{
  KEY k ;

  if (!bsGetKey2 (obj,target,&k, TRUE))
    return FALSE ;

  while (k != type)
    if (!bsGetKey2 (obj,_bsDown,&k, TRUE))
      return FALSE ;

  if (!x)
    return TRUE ;

  switch(type)
    {
    case _Int :
      *(int*)x = obj->curr->n.i ;
      break ;
    case _Float :
      *(float*)x = obj->curr->n.f ;
      break ;
    default:
      if(type<=_LastC)
        *(char**)x = bsText(obj->curr) ;
      else
        messcrash("Wrong type in bsGetData") ;
      break ;
    }

  return TRUE ;
}

/* Note: to get a vector of, say, int's into Array a, do something like:
   arrayMax(a) = 0 ; i = 0 ;
   if (bsFindTag (obj,tag))
     while (bsGetData (obj,_bsRight,_Int,&ix))
       array(a,i++,int) = ix ;
*/

/***************************************/

        /* class checked bsGetKey() - use to ensure you get what you expect */
/*
BOOL bsGetClass (OBJ obj, KEY target, int table, KEY *found)
{
  KEY k ;

  if (!bsGetKey (obj,target,&k))
    return FALSE ;

  while (class(k) != table)
    if (!bsGetKey (obj,_bsDown,&k))
      return FALSE ;

  if (found)
    *found = k ;
  return TRUE ;
}
*/
/***********************************************************/

    /* bsFlatten() flattens a subtree into an array of bsUnits
       provided by the user. The array contains a multiple of n 
       BSunits, each containing an int, float, char* 
       or KEY according to the model, which the user is presumed 
       to know.  Missing data are represented by 0.
    */

BOOL bsFlatten (OBJ obj, int n, Array a)
{
  int i, j, m = 0 ;
  KEY direction = _bsRight ;
  BS curr, bs ;
  BSunit *u ;
  static BSunit zero ;		/* rely on this being 0 */
  static Array currStack = 0, unit = 0 ;
  BOOL gotit = TRUE ;		/* don't want an empty list */
  BOOL added ;

  if (!arrayExists(a))
    messcrash("bsFlatten received a non existing array %d", a) ;
  arrayMax(a) = 0 ;

  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsFlatten got a bad obj") ;
  curr = obj->curr ;

  currStack = arrayReCreate (currStack, n, BS) ; arrayMax(currStack) = n ;
  unit = arrayReCreate (unit, n, BSunit) ; arrayMax(unit) = n ;

  for (i = 0 ; i >= 0 ;)
    { added = FALSE ;
      bs = (direction == _bsRight) ? curr->right : curr->down ;
      while (bs)
	{ if (!bsIsComment(bs)) /* add this node to the unit list */
	    { curr = bs ;
	      arr(currStack,i,BS) = curr ;
	      if (class (curr->key) || curr->key > _LastN)
		arr(unit,i,BSunit).k = curr->key ;
	      else 
		switch (curr->key)
		  {
		  case _Int: 
		    arr(unit,i,BSunit).i = curr->n.i ; 
		    break ;
		  case _Float:
		    arr(unit,i,BSunit).f = curr->n.f ; 
		    break ;
		  default:
		    if (curr->key <= _LastC)
		      arr(unit,i,BSunit).s = bsText(curr) ;
		    else
		      messcrash ("Bad type in bsFlatten") ;
		    break ;
		  }
	      ++i ;
	      gotit = FALSE ;
	      direction = _bsRight ;
	      added = TRUE ;
	      break ;
	    }
	  bs = bs->down ;	/* iterate downwards, ignoring user comments */
	}
      if (!added || i >= n)
	{ if (!gotit)
	    { array(a,n*(++m)-1,BSunit) = zero ; /* set max */
	      u = arrp(a,n*(m-1),BSunit) ;
	      for (j = 0 ; j < i ;)
		*u++ = arr(unit,j++,BSunit) ;
	      while (j++ < n)
		(*u++).k = 0 ;
	      --i ;
	      gotit = TRUE ;
	    }
	  if (direction == _bsDown)
	    { --i ;
	      if (i >= 0)
		curr = arr(currStack,i,BS) ;
	    }
	  direction = _bsDown ;
	}
    }

  return (arrayMax (a) > 0) ;
}

/***********************************************************/

static KEYSET viKeySet = 0 ;

static void bsViGet (BS bs)
{ 
  if (pickVisible(class(bs->key)) && !pickXref(class(bs->key)))
    keySet(viKeySet, keySetMax(viKeySet)) = bs->key ;

 if (bs->right)
    bsViGet (bs->right) ;

  if (bs->down)
    bsViGet (bs->down) ;
}
  
/*********************************/

     /* bsVisibleKeySet() 
       returns a keySet containing all the Visible Keys
       that can be found in Object
    */

KEYSET bsVisibleKeySet (KEY key)
{
  OBJ  obj ;

  if (!key || 
      pickType(key) != 'B' ||
      !(obj  = bsCreate(key) ))
    return 0 ;

  viKeySet = keySetCreate() ;
  bsViGet (obj->root) ;

  bsDestroy (obj) ;

  keySetSort (viKeySet) ;
  keySetCompress (viKeySet) ;
  return viKeySet ;
}

/***************************************************************/
/***************************************************************/
/************************ Add package **************************/


#ifndef READONLY

/* main function of bsAddTag is to leave curr in the right place.
   if necessary adds tag according to model to get there.
   returns TRUE if it adds stuff.
*/

BOOL bsAddTag (OBJ obj, KEY tag)
{
  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsAddTag received  a non magic B obj (%s)",name(obj->key)) ;

  if (!isCacheLocked(obj->x))
    { messout ("bsAddTag fails because %s is not locked\n",
               name (obj->key)) ;
      return FALSE ;
    }

  return findTag (obj,tag,TRUE) ;       /* i.e. new tags made */
}

/***************************************************************/


/* basic routine to add to an object.
   searches down the column for key new.
   if there is a key new there it moves curr to it and returns FALSE
   else it makes a BS with key new and puts curr on it and returns TRUE
   handles user comments properly (now, we hope)
   if target is _bsHere, both this and bsAddData corrupt whatever you
     were on.  Caveat user.
*/

BOOL bsAddKey (OBJ obj, KEY target, KEY new)
{
  BS    bs, bsNew, bsUp, oldCurr ;

  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsAddKey received  a non magic B obj (%s)",name(obj->key)) ;

  switch (target)
    {
    case _bsHere:
      messcrash("bsAddKey(bsHere) is forbidden") ;
    case _bsDown:
      bsUp = obj->curr ;  /* node on which we will add the hook */
      oldCurr = obj->modCurr ;
      bs = bsUp->down ;
      break ;
    default:
      if (!bsAddTag (obj,target))
	return FALSE ;
      target = _bsRight ;
    case _bsRight:	/* note fall through from default */
      bsUp = obj->curr ;	/* must come after bsAddTag() */
      oldCurr = obj->modCurr ;
      bs = bsUp->right ;
      if (!obj->modCurr->right)
	{ messerror ("Trying to add %s off end of model for %s",
		     name(new),name(obj->key)) ;
	  return FALSE ;
	}
      obj->modCurr = obj->modCurr->right ;
    }

/* from here on target must be _bsDown or _bsRight */

	/* next check that new isn't there already */
  while (bs)
    { if (bs->key == new)
        { bsModelMatch (bs, &obj->modCurr) ;
	  obj->curr = bs ;
	  return FALSE ;
	}
      bsUp = bs ;
      bs = bs->down ;
    }
	/* next delete if on a unique branch */
  if (target == _bsRight && bsUp != obj->curr && 
      obj->modCurr->n.key & UNIQUE_BIT)
    { if (bsUp->up->right == bsUp)	/* unhook */
	bsUp->up->right = 0 ;
      else
	bsUp->up->down = 0 ;
      xrefPrune (obj, bsUp, obj->modCurr) ;
      for (bsUp = obj->curr, bs = bsUp->right ; bs ;)
	{ bsUp = bs ; bs = bs->down ; }
    }

  if (!bsTypeCheck (obj,new))	/* sets modCurr to match new */
    { obj->modCurr = oldCurr ;
      return FALSE ;
    }

  bsNew = BSalloc () ;
  bsNew->key = new ;
  if (target == _bsRight && bsUp == obj->curr)  /* hook right */
    { bsNew->down = bsUp->right ;
      bsUp->right = bsNew ;
    }
  else                                          /* hook down */
    { bsNew->down = bsUp->down ;
      bsUp->down = bsNew ;
    }
  bsNew->up = bsUp ;

  obj->curr = bsNew ;
  cacheMark (obj->x) ;
  return TRUE ;
}

/***************************************************************/

BOOL bsAddData (OBJ obj, KEY target, KEY type, void *xp)
         /* NB must call with the ADDRESS of the numbers */
{
  BS bs ;

  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsAddData received  a non magic B obj (%s)",name(obj->key)) ;

  while (!bsAddKey (obj,target,type))   /* got a key match */
    {
      bs = obj->curr ;
      if (bs->key != type)
        return FALSE ; /* happens if bsAddKey failed because typeCheck did */
      switch (type)                     /* first check for data match */
        {
        case _Int:
          if (bs->n.i == *(int*)xp)
            return FALSE ;
          break ;
        case _Float:
          if (bs->n.f == *(float*)xp)
            return FALSE ;
          break ;
        default:
          if (type <= _LastC)
            { if (!strcmp (bsText(bs),(char*)xp))
                return FALSE ;
            }
          else
            messcrash ("Unknown type in bsAddData") ;
        }

      if (obj->modCurr->n.key & UNIQUE_BIT)
	{ if (bs->right)
	    { xrefPrune (obj, bs->right, obj->modCurr->right) ;
	      bs->right = 0 ;
	    }
	  if (bs->down)
	    { xrefPrune (obj, bs->down, obj->modCurr) ;
	      bs->down = 0 ;
	    }
	  break ;
	}

      target = _bsDown ;
    }

  bs = obj->curr ;

  switch(type)
    {
    case _Int :
      bs->n.i = *(int*)xp ;
      break ;
    case _Float :
      bs->n.f = *(float*)xp ;
      break ;
    default:
      if(type<=_LastC)
        { if (!bs->bt)
	    bs->bt = BTalloc() ;
	  else if (bs->bt->cp)
	    messfree (bs->bt->cp) ;
          bs->bt->cp = messalloc (strlen ((char*)xp) + 1) ;
          strcpy (bs->bt->cp, (char*)xp) ;
        }
      else
        messcrash("Wrong type in bsAddData") ;
      break ;
    }
  cacheMark (obj->x) ;
  return TRUE ;
}

/********************************************/

BOOL bsAddComment (OBJ obj, char* text, char type)
{ 
  BS bs ;
  BS new ;

  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsAddComment received  a non magic B obj (%s)",name(obj->key)) ;

  if (!text || !strlen(text))
    return FALSE ;

	/* add to end of list of comments */
  bs = obj->curr ;
  if (bs->right && bsIsComment(bs->right))
    { if (!strcmp(text,name(bs->right->key)))
	return FALSE ;
      for (bs = bs->right ; 
	   bs->down && bsIsComment(bs->down) ;
	   bs = bs->down) 
	if (!strcmp(text,name(bs->down->key)))
	  return FALSE ;
    }

  new = BSalloc() ;
  new->up = bs ;

  if (type == 'C')
    lexaddkey (text, &new->key, _VComment) ;
  else if (type == 'U')
    lexaddkey (text, &new->key, _VUserComment) ;
  else
    messcrash ("Attempt to aadd unknown comment type %c", type) ;

  if (isXrefing)
    { if (!obj->xref) obj->xref = stackCreate (64) ;
      push (obj->xref, new->key, KEY) ;
      push (obj->xref, _Quoted_in, KEY) ;
    }
  new->bt = BTalloc() ;		/* convenient for treedisp */

  if (bs == obj->curr)
    { new->down = bs->right ;
      bs->right = new ;
    }
  else
    { new->down = bs->down ;
      bs->down = new ;
    }
  if (new->down)
    new->down->up = new ;

  cacheMark (obj->x) ;
  return TRUE ;
}

#endif

/******* mark package to allow return to the same point ********/

typedef struct MarkStruct  { BS curr, modCurr ; } *MARK ;

BSMARK bsMark (OBJ obj, BSMARK mark)
{
  if (!mark)
    mark = (BSMARK) messalloc (sizeof (struct MarkStruct)) ;
  ((MARK)mark)->curr = obj->curr ;
  ((MARK)mark)->modCurr = obj->modCurr ;
  return mark ;
}

void bsGoto (OBJ obj, BSMARK mark)
{
  obj->curr = ((MARK)mark)->curr ;
  obj->modCurr = ((MARK)mark)->modCurr ;
}

/********** for deletion from BS trees ***************/

static void xrefPrune (OBJ obj, BS bs, BS bsm)	
     /* recursive, NB can be called on comments with bsm == 0 */
{
#ifdef DELETE_DEBUG
  static int level = 1 ;
  printf ("%*s%s",2*level,"",name(bs->key)) ;
#endif

	/* check for cross referencing */
  if (isXrefing)
    { if (bsIsComment(bs))
	{ if (!obj->xref) obj->xref = stackCreate (64) ;
	  push (obj->xref, bs->key, KEY) ;
	  push (obj->xref, (_Quoted_in | DELETE_BIT), KEY) ;
#ifdef DELETE_DEBUG
	  printf ("\txref") ;
#endif
	}
      else if (bsModelMatch (bs, &bsm) && KEYKEY(bsm->n.key))
	{ if (!obj->xref) obj->xref = stackCreate (64) ;
	  push (obj->xref, bs->key, KEY) ;
	  push (obj->xref, (KEYKEY(bsm->n.key) | DELETE_BIT), KEY) ;
#ifdef DELETE_DEBUG
	  printf ("\txref") ;
#endif
	}
    }

#ifdef DELETE_DEBUG
  printf ("\n") ;
#endif
      
  if (bs->down)
    xrefPrune (obj,bs->down,bsm) ;

#ifdef DELETE_DEBUG
  ++level ;
#endif

  if (bs->right)
    if (bsm)
      xrefPrune (obj,bs->right,bsModelRight(bsm)) ;
    else
      xrefPrune (obj,bs->right,0) ;

#ifdef DELETE_DEBUG
  --level ;
#endif

  BSfree (bs) ;
}

/*********************************/

BOOL bsRemove (OBJ obj)	  /* removes curr and subtree to the right 
			     returns curr to the root of the tree */
{
  BS    bs, bsm ;

  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsRemove received  a non magic B obj (%s)",name(obj->key)) ;

  bs = obj->curr ;
  bsm = obj->modCurr ;
   /* for this routine only, modCurr can be 0, if bs is a comment */

  if (obj->numberOfChildren)
    messcrash ("bsRemove can't kill %s because it has children",
	      name(obj->key)) ; 
  if (!bs || bs == obj->root)	/* can't remove the root node */
    return FALSE ;

  if (bs->up->right == bs)
    bs->up->right = bs->down ;
  else
    bs->up->down = bs->down ;
  if (bs->down)
    bs->down->up = bs->up ;
  bs->down = 0 ;
#ifdef DELETE_DEBUG
  printf ("bsRemove from %s\n", name(obj->key)) ;
#endif
  xrefPrune (obj, bs, bsm) ;

	/* must return to root because of REPEAT, FREE problems */
  obj->curr = obj->root ;
  obj->modCurr = tabModel[class(obj->key)] ;
  cacheMark (obj->x) ;
  return TRUE ;
}

/**********************************/

BOOL bsPrune (OBJ obj) 
     /* removes curr and everything left of it that is unique.  
	NB must go right back to root and come out to keep model
	straight because of possible REPEAT etc.
	Returns curr, modCurr to root at end */
{
  BS bs, bsm ;
  static Stack path = 0 ;

  if (!obj)
    return FALSE ;
  if (obj->magic != OBJ_MAGIC)
    messcrash ("bsPrune received  a non magic B obj (%s)",name(obj->key)) ;

  path = stackReCreate (path, 32) ;

  bs = obj->curr ;
  while (!bs->down && bs->up && bs->up->right == bs 
	 && bs->up != obj->root)		/* unique non-root */
    bs = bs->up ;
  while (bs != obj->root)
    { push (path,bs,BS) ;
      while (bs->up->down == bs)
	bs = bs->up ;
      bs = bs->up ;
    }
  bsm = tabModel[class(obj->key)] ;
  while (!stackEmpty(path))
    { bs = pop (path,BS) ;
      bsm = bsModelRight(bsm) ;
      if (bsIsComment (bs))
	{ bsm = 0 ;
	  break ;		/* must be the final elt */
	}
      if (!bsModelMatch (bs, &bsm))
	messcrash ("Model screwup in bsPrune: %s unmatched in %s",
		   name(bs->key), name(obj->key)) ;
    }

  obj->curr = bs ;
  obj->modCurr = bsm ;
  return bsRemove (obj) ;
}

/******************************************************/

BOOL bsFuseObjects (KEY old, KEY new)
{ 
#ifdef READONLY
  return FALSE ;
#else
  Stack s = 0 ;
  OBJ Old, New ;
  BS copy ;
  extern BOOL parseBuffer(char *buffer) ;

  /* for use in lexalias() - call this BEFORE aliasing the keys.
     idea is to ace dump new to a temp file with object name old,
     make a copy of the old object,
     rewind the file,
     parse it in on top of old with parseFile().
     If success, delete the copy, move the (now fused) object
       from old to new, and leave old empty.
     else delete the half-changed old and replace with the
       saved copy
  */
 
   if (s && parseBuffer(stackText(s,0)))
    { 
      return TRUE ;
    }
  else
    {
      return FALSE ;
    }
#endif
}

/***************************************************************************/
/******* Local Routines - all concern the formation and use of paths *******/

        /* make set of paths for model only when needed by bsUpdate
           set of paths held as an associator: tag -> path
           these are stored in a table over models: tabAssPath
           findHook() sets ->curr to the end of the path
        */

static Array workPath ;         /* 2 statics needed for makepaths */
static int depth ;
static int pathTable ;  /* JTM, to help debugging the errors in models.wrm*/

/*****************************************/

void bsMakePaths (int table)
{
  KEY key = KEYMAKE(table,0) ;
  OBJ model ;

  if (pickType(key) != 'B')
    messcrash ("Trying to update a non bs voc with bs package");

  if (tabAssPath[table])
    assClear (tabAssPath[table]) ;
  else
    tabAssPath[table] = assCreate () ;

  model =  bsCreate(key) ;
  tabModel[table] = bsTreeCopy(model->root) ;
  bsDestroy (model) ;

  if (!workPath)
    workPath = arrayCreate (4,KEY) ;
  depth = 0 ;
  pathTable = table ;
  if (!tabModel[table]->right)
    messcrash("Model for class %s missing",
	      name(tabModel[table]->key)) ;
  makePaths (tabAssPath[table], tabModel[table]->right, FALSE) ;
}

/******************************************/

static void makePaths (Associator ass, BS bs, int isUnique)
{				/* recursively makes paths */

  if (bs->down)
    if (bs->key == _FREE)
      messcrash ("FREE is followed by branch in model") ;
    else
      makePaths (ass,bs->down,isUnique) ;  /* don't include bs in path */

  if (bs->key == _FREE)		/* unchecked subtree */
    { bs->key = _ANY ;
/*      bs->down = bs ; */
      bs->right = bs ;
    }

  array(workPath,depth,KEY) = bs->key ;
  ++depth ;
  array(workPath,depth,KEY) = 0 ;

  if (bsIsTag(bs) && bs->key != _ANY &&
      !assInsert (ass,(void *)bs->key,arrayCopy(workPath)))
    messcrash ("Duplicate tag %s in model %s",
	       name(bs->key), pickClass2Word(pathTable)) ;

  bs->n.key = 0 ;         /* compress unique and xref keys into ->n field */

  if (pickXref(class(bs->key)))
    bs->n.key = _Quoted_in ;

  if (bs->right && bs->right->key == _XREF)
    { BS temp = bs->right ;
      bs->n.key = bs->right->right->key ; /* pick up back ref tag */
      bs->right = bs->right->right->right ;
      if (bs->right)
	bs->right->up = bs ;
      BSfree (temp->right) ;
      BSfree (temp) ;
    }
  if (bs->right && bs->right->key == _UNIQUE)
    { BS temp = bs->right ;
      bs->right = bs->right->right ;
      if (bs->right)
	bs->right->up = bs ;
      BSfree (temp) ;
      isUnique = TRUE ;
    }
  if (isUnique)
    bs->n.key |= UNIQUE_BIT ;

  if (bs->right && bs->right->key == _REPEAT)
    bs->right = bs ;

  if (bs->right && bs->right != bs)	/* can also happen via FREE above */
    makePaths (ass, bs->right, isUnique) ;

  --depth ;                     /* important to restore to original value */
}

/*****************************************/

        /*  set curr to the tag position in obj.  Also set modCurr.
            construct path to tag if necessary only if makeIt is TRUE.
            return TRUE if found (made), else FALSE
        */

static BOOL findTag (OBJ obj, KEY tag, BOOL makeIt)
{
  Array path ;
  int table = class(obj->key) ;
  Associator ass = tabAssPath[table] ;
  BS    bs = obj->root, bsm, new, bs1, top ;
  int   i = 0 ;
  BOOL  ismodif = FALSE, isDown ;
  KEY   key ;

  if (tag == 0)
    return FALSE ;

  if (!assFind (ass,(void*)tag,&path))
    return FALSE ;

  bsm = tabModel[table] ;

	/* RMD 3/1/91 rewrote more robustly along lines of bsFuseModel().  
	   Now handles  (a) interior comments
	   		(b) out of order columns (due to old models)
	*/

  while (key = arr(path,i++,KEY))
    { 
      isDown = FALSE ;
      for (top = bs->right ; 
	   top && bsIsComment(top) ;
	   top = top->down)
	{ bs = top ;
	  isDown = TRUE ;
	}
      for (bs1 = top ; bs1 ; bs1 = bs1->down)
	if (bs1->key == key)
	  { bs = bs1 ; 
	    for (bsm = bsModelRight(bsm) ; bsm->key != key ; bsm = bsm->down) ;
	    goto gotIt ;
	  }
		/* if get here key is not there */
      if (!makeIt)
	return FALSE ;

      new = BSalloc() ;
      new->key = key ;
      ismodif = TRUE ;

      for (bsm = bsModelRight(bsm), bs1 = top ; bsm->key != key ; bsm = bsm->down)
	while (bs1 && bs1->key == bsm->key)
	  { new->up = bs1 ; 
	    bs1 = bs1->down ;
	  }

      if (!new->up)		/* insert at the top */
	{ new->down = top ;
	  if (top)
	    top->up = new ;
	  top = new ;
	}
      else
	{ new->down = new->up->down ;
	  new->up->down = new ;
	  if (new->down)
	    new->down->up = new ;
	}

      if (isDown)
	bs->down = top ;
      else
	bs->right = top ;
      if (top)
	top->up = bs ;

      bs = new ;
gotIt : ;
    }

  obj->modCurr = bsm ;
  obj->curr = bs ;
  if (ismodif)
    cacheMark (obj->x) ;
  return TRUE ;
}

/*********************************************/

        /* for testing ahead in the model to see what types allowed */
        /* currently you must go once right then down the column
           to make sure - not really done properly */

KEY bsType (OBJ obj, KEY target)
{
  BS bsm = obj->modCurr ;

  if (!bsm)
    return FALSE ;

  switch (target)
    {
    case _bsHere:  break ;
    case _bsDown:  bsm = bsm->down ; break ;
    case _bsRight: bsm = bsModelRight(bsm) ; break ;
    default:   messcrash ("bsType called with %s - must be right or down",
                          name(target)) ;
    }

  if (bsm)
    return bsm->key ;
  else
    return FALSE ;
}

/*****************************************************/

BOOL bsTypeCheck (OBJ obj, KEY type)
{	/* checks match of key to modCurr and sets up Xreffing */
  BS bsm = obj->modCurr ;
  int t = class(type) ;
  KEY original = type ;

  if (t) 
    type = KEYMAKE(t,0) ;
  while (bsm->up->down == bsm)    /* go to top of allowed column */
    bsm = bsm->up ;

  while (TRUE)                  /* go back down looking for a match */
    { if (bsm->key == _ANY || bsm->key == type)
        { if (isXrefing && t && KEYKEY(bsm->n.key))
            { if (!obj->xref) obj->xref = stackCreate (64) ;
              push (obj->xref, original, KEY) ;
              push (obj->xref, KEYKEY(bsm->n.key), KEY) ;
            }
	  obj->modCurr = bsm ;
          return TRUE ;
        }
      if (bsm->down && bsm->down != bsm)
	bsm = bsm->down ;
      else
	break ;
    }

  messout ("Type \"%s\" does not check in B object \"%s\"",
	   name(type), name(obj->key)) ;
  return FALSE ;
}

/********** typematch bsm in column to bs ************/

BOOL bsModelMatch (BS bs, BS *bsmp)
{
  KEY bskey = bs->key ;
  BS  bsm = *bsmp ;
  int type = class(bskey) ? KEYMAKE(class(bskey),0) : 
				 bskey ;

  if (!bsm)
    return FALSE ;

  if (bsm->key == type)
    return TRUE ;

  while (bsm->up->down == bsm)    /* go to top of allowed column */
    bsm = bsm->up ;

  while (TRUE)                  /* go back down looking for a match */
    { if (bsm->key == _ANY || bsm->key == type)
        { *bsmp = bsm ;
	  return TRUE ;
        }
      if (bsm->down && bsm->down != bsm)
	bsm = bsm->down ;
      else
	break ;
    }
  return FALSE ;
}

/*****************************************************/

#ifndef READONLY

#undef XREF_DEBUG

static void doXref (OBJ originalObj)
{
  Stack s  = originalObj->xref ;
  OBJ obj ;
  KEY key, tag , original = originalObj->key ;

  if (originalObj->parent)  /* never cross ref a subType */
    return ; 
  isXrefing = FALSE ;

  stackCursor(s,0) ;
  while (!stackAtEnd (s))
    { key = stackNext(s,KEY) ;
      tag = stackNext(s,KEY) ;
      if (obj = bsUpdate (key))
        { bsAddKey (obj, KEYKEY(tag), original) ;
	  if (tag & DELETE_BIT) /* delete: obj->curr is in place */
#ifdef XREF_DEBUG
	    { printf ("Xref: deleting %s from %s\n",
		      name(original), name(key)) ;
#endif
	      bsPrune (obj) ;
#ifdef XREF_DEBUG
	    }
	  else
	    { printf ("Xref: adding %s to %s\n",
		      name(original), name(key)) ;
	    }
#endif	      
          bsSave (obj) ;
        }
      else
	messout ("Can't crossref %s onto %s because %s is locked",
		 name(original),name(key),name(key)) ;
/* could put it onto a global stack to try at the session end? */
    }

  isXrefing = TRUE ;
}

#endif

/****************************************************/
/***** routines for updating in treedisp.c **********/

static void bsSubFuse (BS bs, BS bsm) /* recursively fuses model into object */
{  /* assumes bs matches bsm - this routine does the whole column right */
  BS new,bsr,top,old ;

  if (!bs->size && (bsm->n.key & UNIQUE_BIT) && !bsIsTag(bs))
    bs->size = 2 ;
  if (!bs->bt)
    bs->bt = BTalloc() ;
  bs->bt->bsm = bsm ;

  if (!bsModelRight(bsm) || (bs->size == 1 && bsModelRight(bsm) == bsm))
				/* 2nd poss. if REPEAT or FREE */
    return ;

	/* top will be the top of the column - hook it on later */

  for (top = bs->right ; 
       top && bsIsComment (top) ;
       top = top->down)
    if (!top->bt)		/* need bt->bsm == 0 on comments */
      top->bt = BTalloc() ;

  if (top)
    top->up = 0 ;

     /* JTM fuse to types */

  if (bsIsType(bsModelRight(bsm))) /* then nothing else on that column in model */
    { if (top)
	{ bsm = bsModelRight(bsm) ;  /* keep expanding the model */
	}
      else
	{
	}
    }

    /* take the model keys in the column, one by one */
  for (bsm = bsModelRight(bsm) ; bsm ; bsm = bsm->down)
    {
      		/* first fuse to anything already there that matches */
      for (bsr = top, old = 0 ; bsr ; old = bsr, bsr = bsr->down)
	if ((bsr->key == bsm->key) || 
	    (class(bsr->key) && (class(bsr->key) == class(bsm->key)))) {
	    bsSubFuse (bsr,bsm) ;
	    if (bsIsTag(bsm) || bsm->n.key & UNIQUE_BIT)
	      goto skipNewNode ;
	  }

		/* now make new node */

      new = BSalloc () ;
      new->size = 1 ;		/* flag that it is from model */
      new->key = bsm->key ;
      if (bsIsType(bsm))
	new->size |= 64 ;

  /* insert below last occurence, or previous tag, so search backwards */
  
      if (old)
	for (bsr = old ; bsr ; bsr = bsr->up)
	   if ((bsr->key == bsm->key) || (bsr->key == bsm->up->key) ||
	       (class(bsr->key) && (class(bsr->key) == class(bsm->key)))) {
	     break ;
	   }
      if (bsr)			/* put new below it */
	{ new->up = bsr ;
	  new->down = bsr->down ;
	  bsr->down = new ;
	  if (new->down)
	    new->down->up = new ;
	}
      else			/* put new at the top */
	{ new->down = top ;
	  if (top)
	    top->up = new ;
	  top = new ;
	}

      if (!bsIsType(bsm))
	bsSubFuse (new, bsm) ;		/* fuse to it */

skipNewNode: ;
    }

	/* now hook our new column back onto bs (+ comments) */

  if (bs->right && bsIsComment(bs->right))
    { for (bs = bs->right ; 
	   bs->down && bsIsComment(bs->down) ; 
	   bs = bs->down) ;
      bs->down = top ;
    }
  else
    bs->right = top ;
  if (top)
    top->up = bs ;
}

void bsSubFuseType (OBJ obj, BS bs)
{
  int table = class(bs->key) ;
  BOOL isUp ;
  BS new, bsup ;

  if (!tabModel[table])
    bsMakePaths (table) ;

  /* unhook*/
  bsup = bs->up ;
  if (bsup->down == bs) 
    isUp = TRUE ;
  else if (bsup->right == bs) 
    isUp = FALSE ;
  else
    messcrash("hook error in bsSubFuseType") ;
  bs->up = 0 ;
  bsTreePrune(bs) ;
  
  new = BSalloc () ;
     
  bsSubFuse (new, tabModel[table]) ; 

  /*  rehook */
  if (isUp)
    bsup->down = new->right ;
  else
    bsup->right = new->right ;
  new->right->up = bsup ;
  
  BSfree(new) ;
}

void bsFuseModel (OBJ obj)
{
  int table = class(obj->key) ;

  if (!tabModel[table])
    bsMakePaths (table) ;

  bsSubFuse (obj->root, tabModel[table]) ;
}

/*********************************/

BS bsModelRight (BS bsm)
{
  KEY key = bsm->key ;
  int c = class(key) ;

  if (c && KEYKEY(key) == 1)
    { if (!tabModel[c])
	bsMakePaths (c) ;
      return tabModel[c]->right ;
    }
  else
    return bsm->right ;
}

BS bsModelRoot (OBJ obj)
{
  if (!obj) 
    return 0 ;

  return tabModel[class(obj->key)] ;
}

/********** end of file **********/
