/*  File: objcache.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 * Exported functions:
 * HISTORY:
 * Last edited: Apr 22 16:16 1992 (mieg)
 * * Nov  6 18:26 1991 (mieg): added READING_MODELS protection
 * Created: Wed Nov  6 18:25:34 1991 (mieg)
 *-------------------------------------------------------------------
 */

      /***************************************************************/
      /**  File objcache.c :                                         **/
      /**  Handles the secondary cache of the ACeDB program.        **/
      /***************************************************************/
      /***************************************************************/
      /*                                                             */
      /*  ? routines are public :                                    */
      /*          create/destroy, update/save, acedump               */
      /*    Get/Release, Lock/Unlock, ForceFree                      */

      /*    The secondary cache is a linked list of up to            */
      /*  MAXKNOWNOBJ  objects.                                      */
      /*                                                             */
      /* The cache has two piles controlled by Get and Lock/Unlock   */
      /*   1 : the "knownCache", a LIFO stack, on top of which each  */
      /* cacheEntry is put when "get" (implying a load the first time*/
      /* or a pop, or a new). These cacheEntries are lost, LIFO way, */
      /* when the cache is full, automatically.                      */
      /*   2 : the "Locked CacheEntries". For counting purpose       */
      /* This pile is controlled explicitely by Lock/Unlock          */

      /* cacheEntrySave saves one marked cacheEntry to disk.         */
      /*                                                             */
      /* cacheStatus gives the status.                               */
      /*                                                             */
      /* The Primary Cache handling is done in the separate file     */
      /* blocksub.c invoked only by the static routines load/unload. */
      /*                                                             */
      /*         R.Durbin & J.Thierry-Mieg.                          */
      /*                    last modified  7/3/1991 by JTM.          */
      /*                                                             */
      /***************************************************************/

#include "acedb.h"
#include "cache_.h"
#include "cache.h"  /* Prototypes of public functions of cache package */
#include "bstree.h"
#include "lex_bl_.h"  /* for q->cache */
#include "lex.h"
#include "array.h"
#include "pick.h"
#include "chrono.h"

#define MAXKNOWNCACHE 1000            /*number of cacheEntries in the cache*/

static CACHE	knownCacheTop = NULL ,
		knownCacheEnd  = NULL ,
		lockedCacheTop = NULL ;
static int nKnownCache = 0;
static int nAllocatedCache = 0 ;        /* useful for debug */

static CACHE cacheNew(KEY key) ;
static CACHE cacheCopy(CACHE cache);

static void cacheDestruct(CACHE v);

static void cachePop(CACHE v);
static void cacheKOadd(CACHE v);
static void cacheKOremove(CACHE v);
static void cacheLOadd(CACHE v);
static void cacheLOremove(CACHE v);


static CACHE cacheAlloc (void) ;     /* self managed calloc */
static void cacheFree (CACHE p);

BOOL   READING_MODELS = FALSE ;
#define TESTCACHE_to_test

/*****************************************************************/
/*****************************************************************/
                 /* General utility routines */
/*****************************************************************/

void cacheMark(CACHE cache)
{
  if (cache && cache->magic == CACHEMAGIC && cache->lock)
    cache->lock=2 ;
  else
    messcrash("CacheMark received a bad pointer") ;
}

/**********************/

BOOL isCacheLocked(CACHE cache)
{ return cache->lock ? TRUE : FALSE ;
}

/***********************/

BOOL isCacheModified(CACHE cache)
{ return cache->lock == 2 ;
}

/*****************************************************************/
                 /*  Public;   Calls cacheStore   */
#ifndef READONLY
static int cacheStore(CACHE v, BOOL doCopy) ;

int cacheSaveAll(void)
{
  CACHE v = knownCacheEnd ;
  chrono("cacheSaveAll") ;
  
  while(v)
   { if(v->hasBeenModified)
       cacheStore(v,TRUE) ;
     v = v->up ;
   }

   chronoReturn() ; return(0);        
}
#endif

/************************************************************************/
                          /* To be called only by session manager */
/*
void cacheClear(void)
{
  CACHE v, w ;;
 
  v = knownCacheTop;
  while(v) 
    { if (v->x)
	switch(v->type)
	  {
	  case 'B' :
	    bsTreePrune(v->x) ;
	    break ;
	    
	  default:
	    messcrash("cacheDestruct received unknown type") ;
	  }
      w = v ;
      v = v->next ;
      cacheFree(w);
    }

  v = lockedCacheTop;
  while(v) 
    { if (v->x)
	switch(v->type)
	  {
	  case 'B' :
	    bsTreePrune(v->x) ;
	    break ;
	    
	  default:
	    messcrash("cacheDestruct received unknown type") ;
	  }
      w = v ;
      v = v->next ;
      cacheFree(w);
    }

  knownCacheTop = lockedCacheTop = knownCacheEnd = 0 ;
}
*/
/************************************************************************/
                          /* Gives the cache status*/
                          /*    Public;  Calls nothing*/
int cacheStatus (int *used, int *locked, int*known, int *modified)
{
  register CACHE v;
  register int i;

  chrono("cacheStatus") ;

  *used = nAllocatedCache ;
 
  i = 0; v = knownCacheTop;
    while(v) {v=v->next; i++;}
    *known=i;
  i = 0; v = lockedCacheTop;
    while(v) {v=v->next; i++;}
    *locked=i;
  i=0;v = knownCacheEnd ;
    while(v) {if(v->hasBeenModified) i++; v=v->up;}
    *modified=i;

  chronoReturn() ; return TRUE;
}

/************************************************************************/
    /* For debugging, prints out the content of the cache
       Public;  Calls nothing; needs some work
    */

/*
int cacheShow(void)
{
  chrono("cacheShow") ;

  register CACHE v ;
  int line,bused,bfree,bpinned,bmodif;

  cacheEntryavail(&bused,&bpinned,&bfree,&bmodif);
  graphText(messprintf(
               "Cache usage : %d used, %d modified, %d pinned, %d freecacheEntries",
               bused,bmodif,bpinned,bfree),
            1,9);

  graphTextBounds (100,20+bused+bpinned+bmodif) ;

  line = 11 ; graphText("Modified cacheEntries  ",1,line) ; line++ ;
  for (v = knownCacheEnd ; v ; v = v->up)
    if (v->hasBeenModified)
      cacheEntrieshmod(v,&line) ;

  line += 2 ; graphText("Pinned cacheEntries  ",1,line) ; line++ ;
  for(v = pinnedcacheEntries ; v ; v = v->next)
    cacheEntrieshwks(v,&line);

  line += 2 ; graphText("Used cacheEntries  ",1,line) ; line++ ;
  for(v = knownCacheTop ; v ; v = v->next)
    cacheEntrieshwks(v,&line);

  graphRedraw() ;
  chronoReturn() ; return 0 ;
}
*/

/************************************************************************/
                      /*  Public ;  Called by cacheCreate and cacheUpdate     */
                      /*            Calls buCopy                  */
static CACHE cacheCopy(CACHE cache)
{
 CACHE copy = cacheAlloc();

  chrono("cacheCopy") ;

 *copy = *cache; /* everything but those relevant to this package */
 switch(cache->type)
   {
   case 'A' :
     copy->x = (CACHE_HANDLE) arrayCopy((Array)cache->x) ;
     break ;
   case 'B' :
     copy->x = bsTreeCopy(cache->x);
     break ;
   default :
       messcrash("cacheCopy received unknown type") ;
   }
 chronoReturn() ; return copy;
}

/**************************************************************/
  /* to get a cache address in main memory, with read only status*/
  /* getting it  if necessary  from the primary cache disk */

  /* the system is allowed to unload when refCount == 0 */
  /*   Called by bsCreate and cacheUpdate                 */
  /*   Calls either cachePop (if k is already in memory)*/
  /*             or cacheNew                            */
  /* or cacheCopy if the cacheEntry is locked but not yet modified */


CACHE cacheCreate(KEY k, CACHE_HANDLE *bsp)
{
  LEXP q=KEY2LEX(k);
  CACHE cache ;
  
  chrono("cacheCreate") ;

#ifdef TESTCACHE
  messout("Hello from cacheCreate(%s)",name(k));
#endif
  if(!q)
    {
      messerror("Error : cacheCreate called with unknown key");
      *bsp = 0 ;
      chronoReturn() ; return NULL ;
    }
  
  if (cache = q->cache )
    switch(cache->lock)
      {    
      case 0 :                  /* already in memory*/    
	(cache->refCount)++;
	cachePop(cache);
	*bsp = cache->x ;
	chronoReturn() ; return cache ;
	
      case 1 :    /* accessed read write else where, make a copy */
	cache = cacheCopy(cache);
	break;
	
      case 2 :    /* already modified, grab it again from disk */
	cache = 0 ;
	break;
      }
  
      /* cache->hasBeenModified is set to 0 by cacheNew */
      /* otherwise is stays as is */
  if (!cache)
    cache = cacheNew(k) ;
  if (cache)
    {
      cache->refCount = 1;
      cacheKOadd(cache);
      cache->lock = 0;
      q->cache = cache ;   /* for future access */
      *bsp = cache->x ;
    }
  else
    *bsp = 0 ;
  chronoReturn() ; 
 
  return cache ;
}

/**************************************************************/
                    /* new : to get a new cache address in cache, */
                    /*  Static; called by cacheCreate; Calls cacheAlloc*/
static CACHE cacheNew(KEY key)
{
 CACHE cache = cacheAlloc();

  chrono("cacheNew") ;

 cache->key = key; 
 cache->magic = CACHEMAGIC ;
 cache->type = pickType(key) ;
 cache->hasBeenModified = 0;
 
 
switch (cache->type)
   {
   case 'A' :
     messcrash("cacheNew received Array type, read a.h") ;

   case 'B' :
     cache->x = bsTreeGet(key) ;	/* cannot fail */
     break ;

   case 'S' :
     cacheFree(cache) ;
     cache = 0 ;
     break ;

   default :
     messcrash("cacheNew received unknown type") ;
   }
 chronoReturn() ; return cache ;
}

/************************************************************************/

                    /* Release : complementary to Get */
void cacheDestroy (CACHE v)
{
  chrono("cacheDestroy") ;

  if(!(v->refCount) --)
    messerror("Unbalanced Release of cacheEntry %s",
	      name(v->key));

  switch (v->lock)
    {
    case 0 :  /* This cacheEntry was opened read only */
      break ;
    case 1 :  /* This cacheEntry has not been modified,
		 thus we can still use it next time */
      lexunlock(v->key);  
      cacheLOremove(v);
      cacheKOadd(v);
      break ;
    case 2 :  /* We do want to lose the modifications */
      lexunlock(v->key);  
      cacheLOremove(v);
      cacheDestruct(v) ;
    }

  chronoReturn() ; 
}

/************************************************************************/
   /* Locks a cache, it must then be unlocked or ForceDestroy explicitely*/
                      /*before the memory can be recovered*/
                      /*    Public;   Calls cacheCreate */

CACHE cacheUpdate(KEY k, CACHE_HANDLE *bsp)
{
  LEXP q=KEY2LEX(k);
  CACHE  cache ;

  chrono("cacheUpdate") ;

  *bsp = 0 ;
  if(!q)
    {
      messerror("Error : cacheCreate called with unknown key");
      chronoReturn() ; return NULL ;
    }

  if (q->lock & LOCKSTATUS)
      { messout 
    ("Sorry, cacheEntry %s is already locked", name (k)) ;
        chronoReturn() ; 
	return NULL ;
      }


  if(!KEYKEY(k) && !READING_MODELS)
    { messout
	("Sorry, only Read Models in main menu can update models") ;
        chronoReturn() ; 
	return NULL ;
      }

  q->lock |= LOCKSTATUS ;
  if (cache = q->cache )                /* already in memory*/    
      {    
	if(!cache->refCount)           /* unused */
	    cacheKOremove(cache);
	else
	   cache = cacheCopy(cache);
      }
  else
    {
      cache = cacheNew(k) ;
      q->cache = cache;
    }
  if (cache)
    {
      cache->refCount = 1;
      cacheLOadd(cache);
      cache->lock = 1;        /* locking */
      *bsp = cache->x ;
    }

  chronoReturn() ; return cache ;
}

/************************************************************************/

                    /* unlock : to register the recent modifications */
            
void cacheSave(CACHE v)
{
  LEXP q = KEY2LEX(v->key) ;
  chrono("cacheSave") ;

#ifdef TESTCACHE
  messout("Hello from cacheSave(%s)",name(v->key));
#endif
  if(!v->lock)
      messcrash("Attempt to save an unlocked cacheEntry : %s",
		name(v->key)) ;
  
  lexunlock(v->key);  
  cacheLOremove(v);
  cacheKOadd(v);

  if(q->cache && (q->cache != v))
	   ((CACHE)(q->cache))->hasBeenModified = 0;
	   /* Lame duck, not to be saved */
  
  q->cache = v;  /* for future access */

  if(v->lock == 2)
      (v->hasBeenModified)++;

  v->lock = 0;
  if(!(v->refCount) --)
    messerror("Unbalanced Release of cacheEntry %s",
	      name(v->key));

  chronoReturn() ; 
}

/**************************************************************/
           /* to copy a cacheEntry into the block cache */
             /*        Static, called by cachefree */
             /*                calls ..Store */
/* Note i abandon the forced write system available inside block sub */

#ifndef READONLY

static int cacheStore(CACHE v, BOOL doCopy)
{
  chrono("cacheStore") ;

#ifdef TESTCACHE
  messout("Hello from cacheStore");
#endif
  
  if(v->hasBeenModified) 
    switch(v->type)
      {
/*
      case 'A' :
	arrayStore(v->key, v->x) ;
	break ;
*/	
#ifndef READONLY
      case 'B' :
	if (doCopy)
	  bsTreeStore(bsTreeCopy(v->x)) ;
	else
	  bsTreeStore(v->x);   /* Destructive operation */

	break ;
#endif	
	default :
	  messcrash("cacheStore received unknown type") ;
      }

  v->hasBeenModified = FALSE ;
  
  chronoReturn() ; return(0);
}
#endif

/**************************************************************/

static void cacheDestruct(CACHE v)
{
  LEXP q = KEY2LEX(v->key);

  chrono("cacheDestruct") ;


  if (q->cache == v)
    q->cache = 0;
  if (v)
    {
     if (v->x)
       switch(v->type)
	 {
/*
	 case 'A' :
	   arrayDestroy(v->x) ;
	   break ;
*/	   
	 case 'B' :
	   bsTreePrune(v->x) ;
	   break ;
	   
	 default:
	   messcrash("cacheDestruct received unknown type") ;
	 }

      cacheFree(v);
    }
  chronoReturn() ; 
}


/**************************************************************/
/**************************************************************/
    /* Linked Lists manipulations */

/************************************************************************/
                      /*pops a cache to the top of the knownCache line*/
                       /*  Static; called by cacheCreate; Calls nothing*/
static void cachePop(CACHE v)
{

  chrono("cachePop") ;

 cacheKOremove(v) ;
 cacheKOadd(v) ;
  chronoReturn() ; 
}

 /*********************************************************************/
                /* to add an cacheEntry to the Known CacheEntry List */
                /*  Static; called by cacheCreate and cacheSave        */

static void cacheKOadd(CACHE v)
{
 CACHE w ;
#ifdef TESTCACHE
  messout("Hello from cacheKOadd(%s)",name(v->key));
#endif
  chrono("cacheKOadd") ;

 nKnownCache++;
 w = knownCacheTop;
 knownCacheTop=v;

 if (!knownCacheEnd)
   knownCacheEnd = v;

 if (w) 
   w->up = v ;

 v->up =  NULL;

 v->next = w ;

 w = knownCacheEnd ;
 while( (nKnownCache > MAXKNOWNCACHE) && w)
   {
     v = w;
     w = w->up;
     if (! v->refCount)
       {
	 cacheKOremove(v);

	 if(v->hasBeenModified) 
	   switch(v->type)
	     {
/*
	     case 'A' :
	       arrayStore(v->key,v->x) ;
	       break ;
*/	       
#ifndef READONLY
	     case 'B' :
	       cacheStore(v, FALSE) ;
	       v->x = 0;
	       break ;
#endif	       
	       default :
		 messcrash("cacheKOadd received unknown type") ;
	     }
	 cacheDestruct(v);
       }
   }
  chronoReturn() ; 
}

 /*********************************************************************/
                /* to remove an cacheEntry to the Known CacheEntry List */
                /*  Static; called by cacheUpdate and cacheNew        */
                /*          Calls nothing.      */
static void cacheKOremove(CACHE v)
{
  CACHE u, w;
  chrono("cacheKOremove") ;

#ifdef TESTCACHE
  messout("Hello from cacheKOremove(%s)",name(v->key));
#endif
  nKnownCache--;
  u=v->up;
  w=v->next;

 if (knownCacheTop == v) 
   knownCacheTop = w ;

 if(knownCacheEnd == v)
   knownCacheEnd = u ;

 if(u)
   u->next = w;
 if (w) 
   w->up = u;

 v->up = v->next = NULL;
  chronoReturn() ; 
}

 /*********************************************************************/
                /* to add an cacheEntry to the Locked CacheEntry List */
                /*  Static; called by cacheSave and cacheForceFree */
                /*          Calls nothing.      */
static void cacheLOadd(CACHE v)
{
 CACHE w ;
  chrono("cacheLOadd") ;
#ifdef TESTCACHE
  messout("Hello from cacheLOadd(%s)",name(v->key));
#endif

  w= lockedCacheTop ;
 lockedCacheTop = v ;

 if (w) 
   w->up = v ;

 v->up =  NULL;
 v->next = w ;
  chronoReturn() ; 
}

 /*********************************************************************/
                /* to add an cacheEntry to the Locked CacheEntry List */
                /*  Static; called by cacheSave and cacheDestroy */
                /*          Calls nothing.      */
static void cacheLOremove(CACHE v)
{
 CACHE u, w ;
  chrono("cacheLOremove") ;

#ifdef TESTCACHE
  messout("Hello from cacheLOremove(%s)",name(v->key));
#endif
  u=v->up;
  w=v->next;

 if (lockedCacheTop == v) 
   lockedCacheTop = w ;

 if(u)
   u->next = w;
 if (w) 
   w->up = u;

 v->up = v->next = NULL;
  chronoReturn() ; 
}

/**************************************************************/
/**************************************************************/
            
static Stack freeStack = 0 ;


/***********************/

static CACHE cacheAlloc (void)       /* self managed calloc */
{
  static int blocSize = 512 ;
  CACHE p ;
  int i ;

  chrono("cacheAlloc") ;

  if (!freeStack)
    freeStack = stackCreate (4*blocSize) ;
  if (stackEmpty (freeStack))
    { p = (CACHE) messalloc (blocSize * SIZE_OF_CACHE) ;
      for (i = blocSize ; i-- ; ++p)
        push (freeStack,p,CACHE) ;
/* printf ("Adding %d to CACHE free list\n",blocSize) ; */
      blocSize *= 2 ;
    }
  p = pop (freeStack,CACHE) ;
  memset (p, 0, SIZE_OF_CACHE) ;
  p->magic = CACHEMAGIC ;
  ++nAllocatedCache ;
  chronoReturn() ; return p ;
}

/***********************/

static void cacheFree (CACHE p)
{
  chrono("cacheFree") ;

  push (freeStack,p,CACHE) ;
  --nAllocatedCache ;
  chronoReturn() ; 
}

/**************************************************************/
/**************************************************************/






