/*  File: blocksub.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 * Exported functions:
 * HISTORY:
 * Last edited: Mar 17 18:25 1992 (mieg)
 * * Jan 22 10:24 1992 (mieg): 
    Session object should not be shared.
    Session and bat obj should be modified en place.
 * Created: Wed Jan 22 10:24:47 1992 (mieg)
 *-------------------------------------------------------------------
 */


      /***************************************************************/
      /**  File blocksubs.c :                                       **/
      /**  Handles the cache of the ACeDB program.                  **/
      /***************************************************************/
      /***************************************************************/
      /*                                                             */
      /*  Twelve routines are public :  avail,  rewrite, reallocate, */
      /*   show, mark&save, init,  get/write, pinn/unpinn.           */
      /*              blockfriend, blockNext.                        */
      /*      The cache is an area which contains BLOCKMAX blocks.   */
      /*                                                             */
      /* The cache has two piles controlled by get and pinn/unpinn : */
      /*   1 : the "usedblocks", a LILO stack, on top of which each  */
      /* block is put when  "get" (implying a load the first time    */
      /* or a pop, or a new). These blocks are lost, LILO way, when  */
      /* the cache is full, automatically.                           */
      /*   2 : the "pinnedblocks". Those blocks stay in the cache.   */
      /* This pile is controlled explicitely by blockpinn and unpinn */
      /* Pinns implies a get but unpinn just moves back the block to */
      /* the "usedblocks" list.                                      */
      /*                                                             */
      /* If an object is too large to fit in a single block it       */
      /* is written in a set of blocks linked to the rigth an        */
      /* handled as a whole by the stacks .                          */
      /*                                                             */
      /* blockmark specifies that a block has been modified.         */
      /* blockwrite forces a copy of block to disk, if marked.       */
      /* It has  no action on the cache.                             */
      /* It can be called explicitely. It is invoked by save.        */
      /* blockSave saves the marked block to disk. It is called      */
      /* from the main menu.                                         */
      /*                                                             */
      /* Blockshow is for debugging. Blockavail gives the status.    */
      /*                                                             */
      /* The actual disk handling is done in the separate file       */
      /* disksubs.c invoked only by the static routines load/unload. */
      /*                                                             */
      /*         R.Durbin & J.Thierry-Mieg.                          */
      /*                    last modified    7/3/1991 by JTM.        */
      /*                                                             */
      /***************************************************************/


/* n.b. The TURBOC compiler warns on each use of KEY2LEX(k) that there is
a possible loss of significant digits, which I dont understand since
an LEXP lxi=KEYTOLEX(k) is by itself a huge pointer which can hold
a long, KEY2LEX is defined in ACEDB.h */

#include "acedb.h"
#include "lex_bl_.h"
#include "disk.h"   /*used by block load/unload*/
#include "lex.h"    /*used by blockshow*/
#include "array.h"  /* used by blockSave  */
#include "block.h"
#include "key.h"      /*used by blockshow*/
#include "graph.h"
#include "chrono.h"
#include "session.h"
#include "sysclass.wrm" 

#define BLOCKMAX  1000            /*number of blocks in the cache*/

static void   blockget    (LEXP q, KEY k,BP *p);
static void   blockpop    (ALLOCP v);
static void   blockload   (LEXP q, KEY key);
static void   blocknew    (ALLOCP *v);
static ALLOCP blockalloc  (void);
static ALLOCP blockgrab   (void);
static void   blockfree   (ALLOCP v);
static void   blockunload (ALLOCP v);
static void   blockshwks  (ALLOCP v,int *line);
static void   blockshmod  (ALLOCP v,int *line);
static BOOL    blockisload (ALLOCP v, KEY k);
static void   blockisunld (BP p);
static int    blocknewdsk (BP p);
static BOOL   blockNextKey (BP p, KEY *kp) ;
static ALLOCP freeblocks,
              pinnedblocks,
              usedbltop,
              usedblend;
static void * alloctop,
            * cachetop;

/*
#define TESTBLOCK
#define CHECKBLOCK
*/
extern void invokeDebugger(void) ;

/********************************************************/

BOOL blockCheck (void)
{
  ALLOCP u,v ;
  static Associator table = 0 ;
  int m, n = 0 ;
  int list, badlist ;
  static char* listNames[] = {"","FREE","PINNED","USED"} ;
  BOOL result = TRUE ;

  if (table)
    assClear (table) ;
  else
    table = assCreate () ;
  
  list = 1 ; m = n ;
  for (v = freeblocks ; v ; v = v->next)
    { ++n ;
      if (!assInsert (table, v, (void*)list))
	goto loop_found ;
      if (v->next && v->next->up != v)
	{ messout (
 "blockCheck: bad link at %d in %s list 0x%x -> 0x%x back to 0x%x",
		   n-m, listNames[list], v, v->next, v->next->up) ;
	  result = FALSE ;
	}
    }
  list = 2 ; m = n ;
  for (u = pinnedblocks ; u ; u = u->next)
    for (v = u ; v ; v = v->right)
      { ++n ;
	if (!assInsert (table, v, (void*)list))
	  goto loop_found ;
	if (v->next && v->next->up != v)
	{ messout (
 "blockCheck: bad link at %d in %s list 0x%x -> 0x%x back to 0x%x",
		   n-m, listNames[list], v, v->next, v->next->up) ;
	  result = FALSE ;
	}
      }
  list = 3 ; m = n ;
  for (u = usedbltop ; u ; u = u->next)
    for (v = u ; v ; v = v->right)
      { ++n ;
	if (!assInsert (table, v, (void*)list))
	  goto loop_found ;
	if (v->next && v->next->up != v)
	{ messout (
 "blockCheck: bad link at %d in %s list 0x%x -> 0x%x back to 0x%x",
		   n-m, listNames[list], v, v->next, v->next->up) ;
	  result = FALSE ;
	}
      }

  if (n != BLOCKMAX)
    { messout ("blockCheck found %d blocks instead of %d",
	       n, BLOCKMAX) ;
      result = FALSE ;
    }

  return result ;

 loop_found:
  assFind (table, v, &badlist) ;
  messout (
"blockCheck: loop found from 0x%x in %s list to %s list - %d'th item",
	   v, listNames[list], listNames[badlist], n) ;
  return FALSE ;
}

/********************************************************************/
                      /* constructs the Cache */
                      /*Allocates m<=n blocks and their control area*/
                      /* returns the number m really allocated*/
                      /*Called at least once on entering the program*/
                         /*     Public;  Calls nothing */
void blockInit(void)
{ int n=BLOCKMAX;
  void *v,*w;
  ALLOCP p;
  BP q;
  register unsigned int i;
  register int m;
  static BOOL firstpass = TRUE ;
  
  if (firstpass)
    { firstpass = FALSE ;
      freeblocks=
	pinnedblocks=usedbltop=usedblend=(ALLOCP )NULL;
    }
  else
    return ;
  i=(unsigned int)n*(unsigned int)sizeof(struct alloc);
  v = messalloc(i) ;

  i=(unsigned int)n*(unsigned int)sizeof(BLOCK);
  w = messalloc(i) ;

  memset((char *)w,0,(int)i);
  m=n;p=(ALLOCP)v; q=(BP)w;
  alloctop=v;cachetop=w;
  while(m--) 
    { p->next=freeblocks; 
      p->up=p->right = 0;
      p->p=q;
      p->ismodified=0;
      p->ispinned=0;
      if (freeblocks) freeblocks->up=p;
      freeblocks=p;
      p++;
      q++;
    }
}
/**************************************************************/

static int blockOrder(void *a, void*b)
{
 return 
 (int)(*(DISK *)
           (((ALLOCP)a)->p) - *(DISK *)(((ALLOCP)b)->p) ) ;
}

/**************************************************************/

                 /*Called when the machine is idle*/
                 /*returns the number of blocks written*/
                 /*  Public;   Calls blockunload   */
int blockSave(void)
{
  Array a ;
  register int i = 0 , j = 0 ;
  ALLOCP v = usedblend ;

  if (!isWriteAccess())
    { messout("Sorry, you do not have write access") ;
      return 0 ;
    }

  chrono("blockSave");

  if (!v)
    return 0;

  a = arrayCreate(BLOCKMAX,ALLOCP) ;
  while (v)
    {
      if(v->ismodified)
	array(a,j++,ALLOCP) = v ;
      v = v->up ;
    }

 arraySort(a,blockOrder);
 for(i=0;i<j;i++)
      blockunload(arr(a,i,ALLOCP)) ;

  arrayDestroy(a) ;
  chronoReturn() ;   return(0);       
}

/******************************************************************/
                          /* Gives the cache status*/
                          /*    Public;  Calls nothing*/
int blockavail(int *used,int *pinned,int *free,int *modif)
{
  register ALLOCP v, w;
  register int i;

  i=0;v=usedbltop;
    while(w=v) 
      { while (w=w->right) i++;
	v=v->next; i++;}
    *used=i;
  i=0;v=pinnedblocks;
    while(w=v) 
      { while (w=w->right) i++;
	v=v->next; i++;}
    *pinned=i;
  i=0;v=freeblocks;
    while(v) {v=v->next; i++;}
    *free=i;
  i=0;v = usedblend ;
    while(w=v) 
      { if(v->ismodified)
	  { while (w=w->right) i++;
	    i++ ;
	  }
	v=v->up;
      }
    *modif=i;

  return(0);

}

/****************************************************************/
    /* For debugging, prints out the content of the cache
       Public;  Calls nothing
    */

void blockshow(void)
{
  register ALLOCP v ;
  int line,bused,bfree,bpinned,bmodif;

  graphClear() ;
  blockavail(&bused,&bpinned,&bfree,&bmodif);
  graphText(messprintf(
  "Cache usage : %d used, %d modified, %d pinned, %d freeblocks",
               bused,bmodif,bpinned,bfree),
            1,9);

  graphTextBounds (100,20+bused+bpinned+bmodif) ;

  line = 11 ; graphText("Modified blocks  ",1,line) ; line++ ;
  for (v = usedblend ; v ; v = v->up)
    if (v->ismodified)
      blockshmod(v,&line) ;

  line += 2 ; graphText("Pinned blocks  ",1,line) ; line++ ;
  for(v = pinnedblocks ; v ; v = v->next)
    blockshwks(v,&line);

  line += 2 ; graphText("Used blocks  ",1,line) ; line++ ;
  for(v = usedbltop ; v ; v = v->next)
    blockshwks(v,&line);

  graphRedraw() ;
}

/*****************************************************************/
      /*Returns a pinned block to the top of the usedblock line*/
                   /*      Public;   Calls nothing.  */
void blockunpinn(KEY kk)
{
 LEXP q=KEY2LEX(kk);
 ALLOCP u,v,w;

#ifdef TESTBLOCK
 messout("Hello from blockunpinn");
#endif

 chrono("blockunpinn");
 v=q->addr;
#ifdef TESTBLOCK
 messout("Hello from blockunpinn(%s), old status=%d",
           name(kk),v->ispinned);
#endif
 if(!v->ispinned)
   messcrash(
	     "Unbalanced pinning found in blockunpinn \n%s",
	     name(kk));
             
 if(--(v->ispinned))
   {     chronoReturn() ; return;}
             /* the block is pinned in some other way*/
 if(v->ismodified & 2)
   blockunload(v); /*a blockwrite is pending*/
   
                                    u=v->up;
                                    w=v->next;
                                    if(!u) pinnedblocks=w;
                                       else u->next=w;
                                    if (w) w->up=u;
                                    v->next=usedbltop;
                                    v->up=(ALLOCP)NULL;
                                    if (usedbltop) usedbltop->up=v;
                                    usedbltop=v;
                                    if(!usedblend) usedblend=v;
#ifdef CHECKBLOCK
  if (usedblend && usedblend->ispinned) invokeDebugger() ;
#endif

 chronoReturn() ; return ;
}
/*****************************************************************/
                      /* Clears the disk address of an object */
                      /* and reallocate a new empty block for it */
                      /* This sub does not touch the former block */
                      /*  Public ;  Called by BSstore     */
                      /*            Calls blockpinn */
void blockreallocate(KEY kk,BP *p)
{
 LEXP q=KEY2LEX(kk);
 
 chrono("blockreallocate");

 q->addr=0;
 q->diskaddr=0;
 blockpinn(kk,p) ;

 chronoReturn() ;
}

/********************************************************************/
               /*pinns a block, it must then be unpinned explicitely*/
                      /*before the memory can be recovered*/
                      /* return the number of blocks used */
                      /*    Public;   Calls blockget */
static ALLOCP activeBlock = 0 ;
int blockpinn(KEY kk,BP *p)
{
 LEXP q=KEY2LEX(kk);
 ALLOCP u,v=q->addr,w;
 int n = 0;
 chrono("blockpinn");

 if (!v)
   {
     blockget(q,kk,p) ;
     v=q->addr;
   }

  *p=v->p;
  if (v->ispinned++)
    messcrash("Double pinning of %s",name(kk));

          /* I now modify the cache stacks */
                {
                 u=v->up;
                 w=v->next;
                 if(!u) usedbltop=w;
                   else u->next=w;
                 if(w) w->up=u;
                   else usedblend=u;
#ifdef CHECKBLOCK
  if (usedblend && usedblend->ispinned) invokeDebugger() ;
#endif
                 v->next=pinnedblocks;
                 v->up=(ALLOCP)NULL;
                 if (pinnedblocks) pinnedblocks->up=v;
                 pinnedblocks=v;
                 }
 activeBlock = v ;
 n = 1;        /* counts the cache blocks used by this object */
 while (v = v->right) n++ ;

 chronoReturn() ;
 return n ;
}
/**************************************************************/
  /* to get a block address in main memory, */
  /* loading it if necessary  from disk   : return 0 if success */
  /* this is a necessary step because of dynamical memory allocation */
  /* the system is allowed to unload when it whishes */
  /*   Called by blockpinn;  */
  /*   Calls either blockpop (if k is already in memory)*/
  /*             or blockload (if k is on disk) */
  /*             or blocknew (if k is unknown) */

  /*WARNING : you must  set activeblock if blockget becomes public */

static void blockget(LEXP q, KEY k, BP *p)
{
  ALLOCP a;
  
  chrono("blockget");
  
  if(a=q->addr)    /* already in memory*/    
    {
      *p=a->p;
      blockpop(a);
    }
  
  else
    if(q->diskaddr)
      {
	blockload(q,k) ;
	*p=(q->addr)->p;
      }
    else
      {
	blocknew(&a) ;
	q->addr=a;
	*p=a->p;
	a->p->h.key = k ;
      }

/*  activeBlock = a;   set it if blockget becomes public */
  chronoReturn() ;
}

/*********************************************************************/

void blockAlias(KEY key, KEY newKey)
{ BP p ;
  ALLOCP a;
  LEXP q = KEY2LEX(key) ;
  
  blockget(q, key, &p) ;
  a = q->addr ;
  if (a->p->h.key == key)
    a->p->h.key = newKey ;
  if (a->p->h.type == 'B')
    Balias(a->p, key, newKey) ;
  a->ismodified |= 1 ;
}

/*********************************************************************/

BOOL blockNext(BP *bpp)
{ 
 if (activeBlock->p != *bpp)
   messcrash("blockNext called out of context");
 if (activeBlock = activeBlock->right)
   {
     *bpp = activeBlock->p ;
     chrono ("blockNextFound") ;
     chronoReturn() ;     return TRUE ;
   }
 return FALSE ;
}

/*********************************************************************/

BP blockGetContinuation(int nc)
{ register ALLOCP v =  activeBlock ;
  while(v->up && (v->up->right == v))
    v = v->up ;

  while(nc--)
    { 
      if(v->right)
	v = v->right ;
      else
	messcrash("Cant find continuation %d of block %s",
		  nc, name(v->p->h.key));
    }
  activeBlock = v ;
  return v->p ;
}

/*********************************************************************/

void blockSetNext(BP *bpp)
{
  if (activeBlock->p != *bpp)
    messcrash("blockSetNext called out of context");
  if (!(activeBlock->right))
    activeBlock->right = blockgrab() ;
  activeBlock->right->up = activeBlock ;
  activeBlock = activeBlock->right ;
  *bpp = activeBlock->p ;
}

/*********************************************************************/

void blockSetEnd(BP bp)
{ 
  ALLOCP a;
  
  if (!activeBlock || activeBlock->p != bp)
    messcrash("blockSetEnd called out of context");
  a = activeBlock->right ; /* so a not on used or pinned list */
  activeBlock->right = 0 ;
  while(a)
    { diskfree(a->p) ;
      a->up = 0;
      a->ispinned = FALSE ;
      a->ismodified = FALSE ;
      a->next = freeblocks ;
      if (freeblocks)
	freeblocks->up = a;
      freeblocks = a;
      a = a->right ;
      freeblocks->right = 0 ; /* must be final */
    }
}

/*********************************************************************/
   /* Release the whole disk space */
void blockSetEmpty(BP bp)
{ 
  ALLOCP a,u,w ;
  
  if (!activeBlock || activeBlock->p != bp)
    messcrash("blockSetEmpty called out of context");
  a = activeBlock ;
  if (!a->ispinned)
    messcrash ("block must be pinned to call blockSetEmpty") ;

  u = a->up ;		/* unhook from pinned list */
  w = a->next;
  if (!u) pinnedblocks = w ;
  else u->next = w ;
  if (w) w->up = u ;
  
  while(a)
    { diskfree(a->p) ;
      a->up = 0 ;
      a->ispinned = FALSE ;
      a->ismodified = FALSE ;
      a->next = freeblocks ;
      if (freeblocks)
	freeblocks->up = a ;
      freeblocks = a ;
      a = a->right ;
      freeblocks->right = 0 ;	/* must be final */
    }
}

/*********************************************************************/
                    /*pops a block to the top of the usedblock line*/
                    /*  Static; called by blockget; Calls nothing*/
static void blockpop(ALLOCP v)
{
 ALLOCP u,w;

#ifdef TESTBLOCK
  messout("Hello from blockpop");
#endif
 chrono("blockpop");
 if (!v->ispinned && (u=v->up)) 
   {     
                 w=v->next;
                 u->next=w;
                 if (w) w->up=u;
                        else usedblend=u;
#ifdef CHECKBLOCK
  if (usedblend && usedblend->ispinned) invokeDebugger() ;
#endif
                 v->next=usedbltop;
                 v->up=0;
                 if (usedbltop) usedbltop->up=v;
                 usedbltop=v;
      }
 chronoReturn() ;
}

/*******************************************************************/
              /* to load a block from disk   : return 0 if success */
              /*  Static; called by blockget.                      */
              /*          Calls blockalloc and diskblockread.      */

static void blockload(LEXP q, KEY key)
{
 DISK d=q->diskaddr;
 ALLOCP v;

 chrono("blockload");
#ifdef TESTBLOCK
  messout("Hello from blockload");
#endif
                     /* Read the first block from q->diskaddress */
 v=blockalloc() ;
 
 diskblockread(v->p,d) ;
                          /* it may be shared */
 if (!blockisload(v,key))
   messcrash("ERROR : blockload loaded a wrong block : %s in place of %s",
	     name(v->p->h.key),name(key)
	     );
                      /* Read the continuations directly */
  while (d = v->p->h.nextdisk)
    {
      v->right = blockgrab() ;
      v->right->up = v ;
      v = v->right ;
                       /* they are private */
      diskblockread(v->p,d) ;
      if (key != v->p->h.key)
       messcrash("ERROR : blockload loaded a wrong block : %s",
		 name(key)) ;
    }

 chronoReturn() ; 
}

/**************************************************************/
              /* new : to get a new block address in main memory, */
              /*  return 0 if success */
              /*  Static; called by blockget; Calls blockalloc*/
static void blocknew(ALLOCP *v)
{
#ifdef TESTBLOCK
  messout("Hello from blocknew");
#endif
  chrono("blocknew");
  *v=blockalloc() ;
  (*v)->ismodified=1;
  chronoReturn() ;   
}

/*****************************************************************/
                     /*returns a free allocp address or NULL*/
                     /*  Static; called byblockload or blocknew;*/
                     /*          calls blockfree */
static ALLOCP blockgrab(void)
{
  ALLOCP v;
  static BOOL recursive = FALSE ;
  chrono("blockgrab");
  
  if(recursive)
    messcrash("recursive call of blockgrab");
  recursive = TRUE ;
 blfree:
  if(!freeblocks && usedblend)
    blockfree(usedblend);
  
  if(!freeblocks) 
    if(isWriteAccess())
      messcrash("%s\n%s\n%s",
		"Error : blockgrab failure",
		"All blocks are pinned",
		"Balance Pinn/Unpinn or Increase BLOCKMAX");
    else
      {
	sessionWriteAccess() ;
	if(isWriteAccess() &&  
	   graphQuery("The cache is full, do you want write access ?"))
	  goto blfree ;
	else
	  messcrash(
		    "Block grab : the cache is full and write access is denied") ;
      }
  
  v=freeblocks;
  freeblocks=v->next;
#ifdef CHECKBLOCK
  if (v->ispinned || v->ismodified) invokeDebugger() ;
  if (freeblocks && freeblocks->ispinned) invokeDebugger() ;
#endif
  if (freeblocks) 
    freeblocks->up=(ALLOCP)NULL;
  v->right = v->next= v->up = (ALLOCP)NULL;
  v->ispinned = v->ismodified = recursive = FALSE ;
  memset(v->p,0,sizeof(BLOCK)) ;
  chronoReturn() ; return(v);
}

/*****************************************************************/
                         /*returns a free allocp address or NULL*/
                         /*  Static; called byblockload or blocknew;*/

static ALLOCP blockalloc(void)
{ ALLOCP v = blockgrab() ;

 if (usedbltop) usedbltop->up=v;
                else usedblend=v;
#ifdef CHECKBLOCK
  if (usedblend && usedblend->ispinned) invokeDebugger() ;
#endif
 v->next=usedbltop;
 usedbltop=v;
 return(v);
}

/*******************************************************************/
                        /*frees the relevant block*/
                        /*  Static; called by blockgrab */
                        /*          calls blockunload */
static void blockfree(ALLOCP v)
{                       /*allways unpinn before free*/
  ALLOCP u,w;

#ifdef TESTBLOCK
  messout("Hello from blockfree");
#endif

  if(!isWriteAccess())
    while(v && v->ismodified)
      v = v->up ;   /* buffer the modified blocks */
  
  if(!v)
    return ;

  chrono("blockfree");
  blockunload(v) ;
  blockisunld(v->p);
  
  u=v->up;
  w=v->next;
  if(!u) usedbltop=w;
  else u->next=w;
  if(w) w->up=u;
  else usedblend=u;
  v->next=freeblocks;
  v->up=(ALLOCP)NULL;
  if (freeblocks) freeblocks->up=v;
  freeblocks=v;
#ifdef CHECKBLOCK
  if (usedblend && usedblend->ispinned) invokeDebugger() ;
  if (freeblocks->ispinned) invokeDebugger() ;
#endif

  while(w=v->right)
    { 
      freeblocks = w ;
#ifdef CHECKBLOCK
      if (freeblocks->ispinned) invokeDebugger() ;
#endif
      v->up = w;
      v->right = 0 ;
      w->up = 0;
      w->next = v;
      v = w;
    }
 chronoReturn() ;
}
/**************************************************************/
  /* to mark a key   as modified : returns 0 */
                      /*  Public ; calls nothing */
void blockmark(KEY k)
{
  LEXP q=KEY2LEX(k);
  ALLOCP v=q->addr ;

  chrono("blockmark");
  if(!v) 
    messcrash("ERROR : blockmarking an unloaded key :%s",
                            name(k));
  v->ismodified |= 1;
  if(!(v->ispinned))
    messcrash("WARNING : blockmarking an unpinned block %s",name(k));

 chronoReturn() ;
}

/**************************************************************/
               /* to rewrite a block sharing the addresses*/
                      /*  Public ; calls nothing */
                      /* called by BSstore */
void blockrewrite(KEY key)
{
 ALLOCP v = KEY2LEX(key)->addr;

 chrono("blockrewrite");

 blockisload(v,key);               /*to share memory address*/
 v->ismodified |= 1 ; /* == blockmark ,  to ensure rewriting*/
 blockunpinn(key);

 chronoReturn() ;
}

/**************************************************************/
  /* to force-write a key to disk if it is modified : return 0 if success */
                      /*  Public ;  Calls blockunload*/
void blockwrite(KEY k)
{
  LEXP q=KEY2LEX(k);
  ALLOCP v=q->addr;

  chrono("blockwrite");
  if (v)      /*else key is not in memory*/
    if(v->ispinned)
      {
	messout("%s%s",
		"Sorry : blockwrite invoking a pinned block,",
		"\n this is not an error, just a delay");
	v->ismodified |=2; /*will unload as soon as unpinned*/
      }
    else
      blockunload(v) ;

 chronoReturn() ;
}


/**************************************************************/
             /* to unload a block to disk   : return 0 if success */
             /*        Static, called by blockfree write and save */
             /*                calls diskalloc and diskblockwrite */
static void blockunload(ALLOCP v)
{
  ALLOCP w ;
  BP p=v->p, q;
  KEY key = 0 ; /* to init blockNextKey */

#ifdef TESTBLOCK
  messout("Hello from blockunload %lu", v);
#endif
  
  if(!v->ismodified) 
     return ;
  
  if (!isWriteAccess())
    return ;

  v->ismodified=0;

  chrono("blockunload");

  if (!blockNextKey(v->p,&key))
	{    /* i.e. the block is really empty */
	  diskfree(v->p) ;
	  chronoReturn() ;
          return ;
	}
  v->p->h.key = key ;
  p = v->p ;
 
  diskalloc(p) ;
  blocknewdsk(p) ;
      
  while(v)
    {
      p = v->p ;
      if (w=v->right)
	{
	  q = w->p ;
	  diskalloc(q) ;
	  p->h.nextdisk = q->h.disk ;
	  q->h.key = key ;
	}
      else
	  p->h.nextdisk = 0 ;
  
      diskblockwrite(p) ;

      v = v->right ;
    }
  chronoReturn() ;
}

/**************************************************************/
/**************************************************************/
             /*blockshow_keys, static, called by blockshow*/
static void blockshwks(ALLOCP v,int *line)
{ int i=1;
 KEY key=0;
 char *cp ;

 if(!(v->ismodified))
 while(blockNextKey(v->p,&key))
   {if(i>=80)                
      { (*line)++;i=1; }
   cp = messprintf (" : %s ",name(key)) ;
   graphText(cp,i,*line);
   i += strlen(cp) ;
  }
 (*line)++;
}
/**************************************************************/
             /*blockshow_keys, static, called by blockshow*/
static void blockshmod(ALLOCP v,int *line)
{  int i=1;
 KEY key=0;
 char *cp ;

 if(v->ismodified)
 while(blockNextKey(v->p,&key))
  {if(i>80)                   
     { (*line)++;i=1;}
   cp = messprintf (" : %s ",name(key)) ;
   graphText(cp,i,*line);
   i += strlen(cp) ;
  }
 (*line)++;
 }
/**************************************************************/
              /* block is loaded static, called by blockload*/
static BOOL blockisload (ALLOCP v, KEY k)
{
  KEY key=0;
  BOOL found = FALSE ;

  while(blockNextKey(v->p,&key))
	{
	  KEY2LEX(key)->addr = v ;
	  if(key==k)
	    found = TRUE ;  /*k has been found as hoped */
	}

  return found ;
}
/**************************************************************/
              /* block is unloaded static, called by blockfree*/
static void blockisunld (BP p)
{
 KEY key=0;
 
 while(blockNextKey(p,&key))
        KEY2LEX(key)->addr=0;
}

/**************************************************************/
              /* block new disk address static, called by blockload*/
              /* returns the number of objects in the block */
static int  blocknewdsk (BP p)
{
 KEY key=0;
 register int i=0;
 register LEXP q;
 DISK d = p->h.disk;
/*
 if(p->h.key == _continuationKey)
   return 1 ;
*/
 while(blockNextKey(p,&key))
               {
		 i++;
                 q=KEY2LEX(key);
                 if(q->diskaddr!=d)
                      {q->diskaddr=d;
                       lexmark(class(key));
                       }
                 }
 return i;
}
/**************************************************************/
        /* Returns a loaded unpinned key of same class, or key.
	 * Looks only for keys of same class, already touched
	 * during the same session and still in the cache.
         * Called from bstree.c : bsTreeFindBlock 
	 *  only when key has never been saved before. 
	 */

extern long random(void) ;   /* Unix */
KEY blockfriend(KEY key)
{
  register  int t = class(key);
  register ALLOCP v = usedbltop ;
  KEY k = 0;
  int i = ((int)random()) % BLOCKMAX , j =0 ;

  if(!v)
    return key ;
    
   /* Session object should not be shared.
    * Suppose you did, 
    * may be in session 8 you share session 5 block, hence 
    * session 5 would be relocated in a block allocated by session 8.
    * If you then destroyed session 8 before session 5, the 
    * session 5 obj would be lost.
    *
    * Caveat: this also holds for Bat obj if at some point shared
    * arrays were introduced.
    */
  if(class(key) == _VSession  || class(key) == _VBat)
    return key ;

  chrono("lexrandomkey") ;

  while (TRUE)
    {
     if(!v)       /* start from top again */
       { if(!i)
	   break ;
	 i = i%j ;
	 v = usedbltop ;
	 continue ;
       }
     if(i) 
       { j++ ; i-- ;}
     else
       if(
	  (   v->ismodified 
	  || (v->p->h.session == thisSession.session)
	  || (v->p->h.session == 0)
          )
	  && !v->right 
	  && blockNextKey(v->p,&k) && (class(k) == t)
	  )
	 { key = k ; break ; }
     k = 0;
     v = v->next ;
   }
  
  chronoReturn() ; return(key) ;
}
/**************************************************************/
extern int Bnextkey(BP p, KEY *key) ; /* in bsubs */

              /* block is loaded static, called by blockload*/
static BOOL blockNextKey (BP p, KEY *kp)
{
  static BP bp ;
 
  if(*kp)
    { if(bp != p)
	messcrash("WARNING : bad call to blockNextKey /n %s",
		 name(*kp));
   }
  else
    bp = p ;

  switch( bp -> h.type )
    {
    case 'B'  :
      return Bnextkey(p,kp) ;
    default:
      if (*kp)
	return FALSE ;   /* single object by block */
      return (*kp= bp-> h.key) ? TRUE : FALSE ;
    }
  return FALSE ;
}

/**************************************************************/
/**************************************************************/
