/*  File: session.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
     Session control, passwords, write acces, semaphores etc.
 * Exported functions:
 * HISTORY:
 * Last edited: Apr  2 18:34 1992 (mieg)
 * * Feb  2 11:46 1992 (mieg): interactive session control
 * * Jan 24 11:20 1992 (mieg): session fusion.
 * Created: Fri Jan 24 11:19:27 1992 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "disk_.h"   /* defines BLOCKHEADER */
#include "graph.h"  
#include "array.h"
#include "menu.h"
#include "mytime.h"
extern char* getlogin(void) ;
extern void wormClose (void) ;    /* in wormmain.c */
extern void lexOverLoad(KEY key,DISK disk) ;
extern DISK lexDisk(KEY key) ;


#ifdef THINK_C
	#include "unix.h"
#endif

/* Prototype not found on the SUN include, taken for SUN OS 4.1 doc */
#ifdef IPC	/* RMD - clean solution to IPC problems */
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
  extern int semget(key_t key, int nsems, int semflag) ;
  extern int semctl(int semid, int semnum, int cmd, union semun arg) ;
  extern int semop(int semid, struct sembuf* sops, unsigned nsops) ;
static int SEM = -1 ;
static int semid = -1 ;

#else

static int SEM = 0 ;
static int semid = 1 ;
#endif

#define BLKMX (BLOC_SIZE - sizeof(BLOCKHEADER) - sizeof(DISK)\
	       -sizeof(int) -sizeof(int) -sizeof(int))

typedef struct block /* sizeof(block) must be < BLOC_SIZE */
  {
    BLOCKHEADER  h ;
    DISK gAddress ;
    int mainRelease ,  subDataRelease, subCodeRelease ;
    char c[BLKMX] ;
  }
    BLOCK, *BLOCKP;   /*the transfer unit between disk and cache*/

#ifdef DEF_BP
  Fatal Error  Double definition of BP
#endif

typedef BLOCKP BP ;
#define DEF_BP

#include "session.h"
#include "chrono.h"
#include "a.h"
#include "bs.h"
#include "disk.h"
#include "display.h"
#include "query.h"
#include "lex.h"
#include "systags.wrm"
#include "sysclass.wrm"


static  int  writeAccess = 0 ;  /* Sets to gActive->user */
static void sessionFlipMenu(BOOL access) ;
static void sessionReleaseWriteAccess(void) ;
static BLOCK b ;
void sessionUser(void) ;
extern void readModels(void) ;

static Array sessionTree = 0 ;
static Stack stText = 0 ;
static Graph sessionSelectionGraph = 0 ;
static BOOL showDeads =  FALSE ; 
static int sessionFindSon(KEY father, KEY *sonp) ;
static void sessionTreeConstruct(void) ;
void sessionControl(void) ;

SESSION  thisSession ;

typedef struct STstructure *STP ;
typedef struct STstructure 
 { KEY key, father, ancester ;
   STP sta ;
   BOOL isDestroyed, permanent ;
   int user, date, title, color ; 
   int generation , x, y, len , box ;
 } ST ;

/****************************************************************/

#define DEBUG

/*************************************************************/
  /* If another process has written since I started
     either he has grabed the semaphore, and I cant write,
     or he may have released it but then he may have updated
     the disk.
     In this case, the session number on block 1 will be
     modified and I will know that my BAT is out of date
     */

static BOOL checkSessionNumber(void)
{
  BLOCK bb ;
  diskblockread(&bb,1) ;
  if( thisSession.gAddress ==  bb.gAddress ) 
    return TRUE ;
  return FALSE ;
}

/*************************************************************/

static BOOL do_I_own_the_process(void)
{
#if !defined(THINK_C) && !defined(DOS)
 extern int getuid(void), geteuid(void) ;   /* Unix */
 return
     (getuid() == geteuid())  ;
#else
 return TRUE ;
#endif
}

/*************************************************************/
/************** Disk Lock System *****************************/
/*************************************************************/

static BOOL setWriteLock()
{ FILE* lockFile = 0 ;
  
  lockFile = filopen("database/lock","wrm","w") ;
  if(!lockFile)
    { messout("setWriteLock cannot create the lock file") ;
      return FALSE ;
    }
  fprintf(lockFile,"%d %s locking the database",
	  thisSession.session, thisSession.name );
  fflush(lockFile) ;
  fclose(lockFile) ;
  return TRUE ;
}

/**************************/

static void releaseWriteLock()
{ FILE* lockFile = 0 ;
  
  lockFile = filopen("database/lock","wrm","w") ;
  if(!lockFile)
    messcrash("releaseWriteLock cannot create the lock file") ;

  fprintf(lockFile,"0 nobody locking the database") ;
  fflush(lockFile) ;
  fclose(lockFile) ;
}

/**************************/

static char * isWriteLocked()
{ static char name[80] ;
  int n ;
  FILE* lockFile = 0 ;

  lockFile = filopen("database/lock","wrm","r") ;
  if(!lockFile)
     return 0 ;

  fscanf(lockFile,"%d %s",&n, name) ;
  fclose(lockFile) ;
    /* if n == thisSession.session && name == getlogin()
       I understand that i crashed last time during the same session
       It is therefore licit to ignore the lock
       I could also do a check on pid
       then ps -aux | grep the locking pid
       and ignore the lock if this pid is dead
       but this is real bad because ps does not explore the network.
       So the pid system would fail if 2 copies of acedb run
       on different machines.
       */
  if(!n || 
     ( thisSession.session == n && !strcmp(name,getlogin())))
    return 0 ;
  else 
    return name ;
}



#ifdef IPC

/*************************************************************/
/************** Semaphore System *****************************/
/*************************************************************/

static BOOL getSemKey()
{
  FILE *fil ;
  int n ;
  char *cp ;

  if(SEM != -1)
    return TRUE ;

  fil = filopen("wspec/semkey","wrm","r") ;
  
  if(!fil)
    {
      messout("Sorry, I cannot read the File giving the SEM ") ;
      return FALSE ;
    }
  while(freeread(fil))
    if(    (cp = freeword())
       &&  !strcmp(cp,"SEM")
       &&  freestep('=')
       &&  freeint(&n))
      { SEM = n ; break ;}

  fclose(fil) ;
  return SEM != -1 ;
}

/*************************************************************/
    
static BOOL getSemaphore(void)
{
  struct sembuf sops ;
  int nsops = 1 ;
  struct semid_ds buf ;
  union semun arg ;

  /* Check if i created the semaphore myself, if not, grab it */
      /* I can now try to grab it, if no other process does it
	 in the meantime
	 */
  arg.buf = &buf ;

   /* get the semaphore KEY */
  if(!getSemKey())
    return FALSE ;

   /* No semaphore requested */
  if (!SEM)
    return TRUE ;

   /* get the semaphore semid */
  semid = semget( SEM, 0, 0) ;
  if (semid == -1)
    {    /* It does not exist, try to create it */
      semid = semget( SEM, 1,  IPC_CREAT  | 0664) ;
      if (semid == -1)
	{ messerror("%s%s%s%s%s%s\n%s",
		    "Sorry, I cannot IPC_CREAT the semaphore, ",
		    "May be the running code does not own the ACEDB directory, ",
		    "this should be cured by chmod 4755 acedb" ,
		    "or the semaphore is used by another program or another version ",
		    "of acedb. Each acedb database needs its own semaphore.",
		    "In that case, update the file wspec/semkey.wrm" ,
		    "Use SEM = 0 if semaphores are not part of your Unix kernel.") ;
	  return FALSE ;
	}

     /* This is my own creation, I init it to one no  sem_UNDO */   
     /* there is a slight possibility that someone else
	created the semaphore between my 2 calls to semget
	I do not know how to avoid this problem.
	*/

   /* Other problem the following semctl provokes a bus error 
      arg.val = 1 ;
      if(semctl(semid, 0, SETVAL, arg) == -1)
	{ messerror( "Sorry, getSemaphore semctl(SETVAL) failed") ;
	  return FALSE ;
	}
     So I rather do the following
	*/



      sops.sem_num = 0 ; /* single semaphore system */
      sops.sem_op = +1 ; /* Try to init at 1 */
  /* with IPC_NOWAIT : the system will not wait for liberation */
  /* no SEM_UNDO */
  sops.sem_flg = IPC_NOWAIT ;
  if (semop(semid,&sops,nsops) == -1)
	{ messerror( "Sorry, getSemaphore semop (init at 1 failed") ;
	  return FALSE ;
	}

     /* we now fall thru onto the normal situation as if the
	semaphore existed before hand
	*/
    }


  sops.sem_num = 0 ; /* single semaphore system */
  sops.sem_op = -1 ; /* Try to grab write access */
  /* with IPC_NOWAIT : the system will not wait for liberation */
  /* SEM_UNDO is good in case the system crashes */
  sops.sem_flg = IPC_NOWAIT | SEM_UNDO ;
  if (semop(semid,&sops,nsops) == -1)
    return FALSE ;
  else
    return TRUE ;
}
/**************************/


/*      this code does not work i do not understand the content of buf 
	JTM jan 22 91, somehow i should be able to know that
	semget(IPC_CREAT did or did not provoked a creation
	with the present alternative call 2 process asking at the
	same time for write access could create a knot

  the idea was to check after grabbing wheter i did create it
  if(semctl(semid, 0,IPC_STAT,arg) == -1)
    { messerror( "Sorry, getSemaphore semctl(IPC_STAT) failed") ;
      return FALSE ;
    }
  if(mypid == buf.sem_perm.cuid)
*/

/*************************************************************/
  /* Kill any earlier version of the semaphore and grab the new one*/
static void makeSemaphore(void)
{
  int  semnum, cmd;
  union semun arg ;
/*
    {
      int val;
      struct semid_ds *buf;
      ushort *array;
    } arg;

*/
   /* get the semaphore KEY */
  if(!getSemKey())  /* if the file does not exist create it */
    {  FILE *fil ;

       fil = filopen("wspec/semkey","wrm","w") ;
       if(!fil)
	 messcrash("Sorry, I cannot create a  File giving the SEM ") ;
       SEM = 7 ; /* random choice */

       fprintf(fil,
 "### This file contains the semaphore Key of the acedb system ###\n") ;
       fprintf(fil,"SEM =  7 \n") ;
       fclose(fil) ;
     }
  if(SEM == 0)
    { messout("%s, %s",
	      "The semaphore system is explicitly disabled",
	      "to reinstall it update the file wspec/semkey.wrm") ;
      return ;
    }

  semid = semget( SEM, 0, 0) ;
  if(semid >= 0 )  /* semaphore exists, kill it */
    {
      cmd = IPC_RMID ;
      if(  semctl(semid, semnum, cmd, arg ) == -1 )
	{ messerror
	    ("makeSemaphore could not IPC_RMID the semaphore") ;
	  messcrash
	    ("makeSemaphore could not IPC_RMID the semaphore") ;
	}
    }

  if (!getSemaphore())
    {
      messerror ("makeSemaphore failed, sorry") ;
      messerror("try to edit wspec/semkey to another value") ;
      messcrash ("makeSemaphore failed, sorry") ;
    }
}

/*************************/

static void releaseSemaphore(void)
{
  struct sembuf sops ;
  int nsops = 1 ;
  int presentSemid,  presentStatus ;

   /* No semaphore requested */
  if (!SEM)
    return ;

  if (semid == -1)
    messcrash("releaseSem was called with semid = -1 ") ;

  semaphoreStatus(&presentSemid, &presentStatus) ;
  if(semid != presentSemid)
    messcrash
      ("Somehow, the semaphore id changed between get and release") ;

  if(presentStatus)
    messcrash
      ("Somehow, releaseSemaphore found semState = %d, should be 0",
        presentStatus ) ;

  sops.sem_num = 0 ; /* single semaphore system */
  sops.sem_op = +1 ; /* Release write access */
   /* with IPC_NOWAIT : the system will not wait for liberation */
   /* SEM_UNDO must be balanced */
  sops.sem_flg = IPC_NOWAIT | SEM_UNDO ;
  if (semop(semid,&sops,nsops) == -1)
    {
      messerror("ReleaseSemaphore") ;
      messcrash("releaseSemaphore failed, see the dump file") ;
    }

   /*  else successful completion */
}
#endif

/*************************/

void semaphoreStatus(int * sid, int* state)
{
#ifdef IPC
  int localSemid ;  
  union semun arg ; /* not used, so no init required */

  *state = -1 ;
   /* get the semaphore KEY */
  if(!getSemKey())
    { *sid = -1 ; return ; }

   /* No semaphore requested */
  if (!SEM)
    { *sid = 0 ;
      return ;
    }

  localSemid = semget( SEM, 0, 0) ; /* nsems = 0 is a good default
				  when you do not IPC_CREAT */
  if(localSemid >= 0 )  /* semaphore exists */
    { *state =  semctl(localSemid, 0, GETVAL, arg) ;
      if(*state == -1)
	messerror("semaphoreStatus, semctl(GETVAL) failure") ;
    }
  else
    messerror("semaphoreStatus, semget(SEM = %d) failure",SEM) ;
  *sid = localSemid ;
#else
*sid = 0; *state = 0 ;
#endif
}

/*************************************************************/
/****************************************************************/
static KEY   sessionChosen = 0 ;

static void sessionStart(void)
{ OBJ Session ;
  int a, b, c , h ;
  extern void mainCleanUp(void) ;

  a = b = c = h = 0 ;
  Session = bsCreate(sessionChosen) ;
  if(!Session )
    { messout("Sorry, I cannot find %s",name(sessionChosen)) ;
      return ;
    }
  if(!bsGetData(Session,_VocLex,_Int,&a) ||
     !bsGetData(Session,_bsRight,_Int,&b)  ||
     !bsGetData(Session,_bsRight,_Int,&c)  ||
         /* Hasher only introduced in release 1.4 */
     /* I hope that the compiler is not too smart ! */
     ( !bsGetData(Session,_bsRight,_Int,&h) && FALSE)  ||
     !bsGetData(Session,_CodeRelease,
	    _Int,&thisSession.mainCodeRelease) ||
     !bsGetData(Session,_bsRight,
		_Int,&thisSession.subCodeRelease) ||
     !bsGetData(Session,_DataRelease,
	    _Int,&thisSession.mainDataRelease) ||
     !bsGetData(Session,_bsRight,
	    _Int,&thisSession.subDataRelease) ) 
    { messout("No address in this session") ;
      bsDestroy(Session) ;
      return ;
    }
  bsDestroy(Session) ;

  thisSession.from = sessionChosen ;
  thisSession.upLink = sessionChosen ;

  lexOverLoad(__lexi3,(DISK)a) ;
  lexOverLoad(__voc3,(DISK)c) ;

  lexOverLoad(__lexh3,(DISK)h) ;

  lexRead() ; 

  mainCleanUp() ;  /*  all but main  graph */
  sessionChosen = 0 ;
}

/*************************************************************/

void sessionAutoSelect(void)
{ 
  sessionChosen = lexLastKey(_VSession) ;

  if(!sessionChosen)
    messcrash("The session class is empty, Reconstruct, sorry") ;
  sessionStart() ;
}

/*************************************************************/

static BOOL  sessionOedipe(void) /* Kill the  father of the present session */
{ KEY father = thisSession.upLink, grandFather, fPlus, fMinus ;
  OBJ Father ; 
  Array fPlusArray, fMinusArray ;
  extern void diskFuseOedipeBats(Array fPlusArray, Array fMinusArray)  ;

   if(!father || !(Father = bsUpdate(father)))
    return FALSE ;
  if(!bsGetKey(Father,_BatPlus,&fPlus) ||
     !bsGetKey(Father,_BatMinus,&fMinus) ) 
    { messout("No address in this father") ;
      bsDestroy(Father) ;
      return FALSE ;
    }

  if(!bsFindTag(Father,_Permanent_session))
    { messout("The previous session beeing delared permanent, I keep it") ;
      bsDestroy(Father) ;
      return FALSE ;
    }

  if (bsGetKey(Father, _Up_linked_to,&grandFather))
    { thisSession.upLink = grandFather ;
      bsAddData(Father, _Destroyed_by_session, _Int, &thisSession.session) ;
      bsSave(Father) ;
    }
  else
   { thisSession.upLink = 0 ;
     bsDestroy(Father) ;
   }  
  
  fPlusArray = arrayGet(fPlus,unsigned char,"c") ;
  fMinusArray = arrayGet(fMinus,unsigned char,"c") ;

  diskFuseOedipeBats(fPlusArray, fMinusArray) ; 

  arrayDestroy(fPlusArray) ;
  arrayDestroy(fMinusArray) ;
              /* Kill the father bats */
  arrayKill(fPlus) ;
  arrayKill(fMinus) ;

  saveAll() ;
  return TRUE ;
}

/******************/

static BOOL sessionFuse(KEY father, KEY son) 
{ KEY fPlus, fMinus, sPlus, sMinus , grandFather ;
  OBJ Father = 0 , Son = 0 ; int n = 0 ;
  Array fPlusArray = 0, fMinusArray = 0, 
        sPlusArray = 0, sMinusArray = 0;
  extern int diskBatSet(Array bat) ;
  extern void diskFuseBats(Array fPlusArray, Array fMinusArray,
			   Array sPlusArray, Array sMinusArray) ;

  Father = bsUpdate(father) ;
  if(!bsGetKey(Father,_BatPlus,&fPlus) ||
     !bsGetKey(Father,_BatMinus,&fMinus) ||
      bsFindTag(Father, _Destroyed_by_session)) 
    { /* messout("Session fuse: No address in the father") ; */
      goto abort ;
    }

  fPlusArray = arrayGet(fPlus,unsigned char,"c") ;
  fMinusArray = arrayGet(fMinus,unsigned char,"c") ;

  if (son)
    { if (!(Son = bsUpdate(son)))
	messcrash("sessionFuse cannot bsupdate son = %s",
		  name(son) ) ;
      
      if( !bsGetKey(Son,_BatPlus,&sPlus) ||
	 !bsGetKey(Son,_BatMinus,&sMinus) )
	{ messout("Session fuse: No address in the son") ;
	  goto abort ;
	}
     
      sPlusArray = arrayGet(sPlus,unsigned char,"c") ;
      sMinusArray = arrayGet(sMinus,unsigned char,"c") ;
      
      if (!sPlusArray || ! sMinusArray)
	goto abort ;
      arrayKill(fPlus) ;  /* Ok iff present session is indeed a descendant */
      arrayKill(fMinus) ;

      diskFuseBats(fPlusArray, fMinusArray, sPlusArray, sMinusArray) ; 
      if (bsGetKey(Father, _Up_linked_to,&grandFather))
	bsAddKey(Son , _Up_linked_to, grandFather) ;
    }
  else
    diskFuseBats(fPlusArray, fMinusArray, 0, 0) ;

 
             /* Kill the father bats */
  bsAddData(Father, _Destroyed_by_session, _Int, &thisSession.session) ;
  bsSave(Father) ;
  

/* In principle one should:
         arrayKill(fPlus) ; 
         arrayKill(fMinus) ;
 * however, their disk block has already been allocated in father
 * and should only be freed by a direct descendant which is not necessarily the case.
 * Since in cache we did not arraySave the cache will emtpy itself automatically.
 * note however that it is crucial to saveAll between each sessionDestroy
 * because if i killed a's father before a then i will have done an arraySave(a's bat)
 */

  arrayDestroy(fPlusArray) ;
  arrayDestroy(fMinusArray) ;

            /* Store the sons bat */
  if (son)
    { bsAddKey(Son, _BatPlus, sPlus) ;   /* to reposition */
      n = diskBatSet(sPlusArray) ;
      bsAddData(Son, _bsRight, _Int, &n) ;
      bsAddKey(Son, _BatMinus, sMinus) ;
      n = diskBatSet(sMinusArray) ;
      bsAddData(Son, _bsRight, _Int, &n) ;
      bsSave(Son) ;
      
      arrayStore(sPlus, sPlusArray,"c") ;
      arrayStore(sMinus, sMinusArray,"c") ;
      arrayDestroy(sPlusArray) ;
      arrayDestroy(sMinusArray) ;
    }
   saveAll() ;
   return TRUE ;

 abort:
  bsSave(Son) ;
  bsSave(Father) ;
  return FALSE ;
}

/****************/

static BOOL  sessionKillAncestors(int n) /* Kill my father level n */
{ KEY father = thisSession.upLink, son ;
  ST *st ; int i ;

  for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
    if (st->key == father)
      break ;

  if (father  && st->key != father) 
    { messerror("sessionkillAncester did not find the father he wanted to murder.") ;
      return FALSE ;
    }
 
  while (st && --n)            /* Move up the ladder */
    st = st->sta ;
  if (!st || n)                /* Not many ancesters alive */
    return FALSE ;
  
  while (st && 
	 sessionFindSon(st->key, &son) != 1 )
    st = st->sta ;            /* Try higher up */
    
  if (st && st->key && son)
     return
      sessionFuse(st->key, son) ;

  return FALSE ;
}

/*************************************************************/

void sessionInit(void)
{
 FILE * f;
 char *cp ;

 extern void lexReadGlobalTables(void) ;
 
 if (f = filopen("database/ACEDB","wrm","r"))
    {
      fclose(f) ;

      diskblockread(&b,1) ;
      if(b.mainRelease != MAINCODERELEASE)
	messcrash
	  ("%s%d.%d%s%d%s%s",
	   "You are running release ",MAINCODERELEASE,
	   SUBCODERELEASE,
	   " of ACEDB.\n The data base was created under release ",
	   b.mainRelease, ".xx To reconstruct the database",
   "remove the file ACEDB.wrm (save the old blocks.wrm first)."
	   ) ;
      thisSession.session = ++b.h.session ;
      thisSession.gAddress = b.gAddress ;
      lexOverLoad(__lexi1,b.gAddress) ;
      lexReadGlobalTables() ;
      return ;
    }
 
       /* else reinitialise the system */

  /* Only the acedb administrator is entitled to proceed
   * I check this by trying to open passwd.wrm, a
   * In the install documentation, it is stated that
   * this file should be set chmod 644
   */

         /* First you must own the running process */

  if (!do_I_own_the_process() 
      || (!(f = filopen("wspec/passwd","wrm","a"))) ) 
    messcrash
 ("%s%s",
  "Sorry, the database seems empty.  Only the acedb administrator owning",
  "the $(ACEDB) directory can reinitialise it");

  fclose(f) ;

  if(!graphQuery(
  "The database seems empty.  Should I reinitialise the  system"))
    messcrash("Have a look at the documentation !, bye.");

#ifdef IPC
  makeSemaphore() ; /* Since I reinitialise the disk ! */
#endif
  
  writeAccess = TRUE  ; /* implied */
  thisSession.session = 1 ;
  strcpy(thisSession.name,"Empty_db") ;
  thisSession.gAddress = 0 ; /* so far */
  thisSession.mainCodeRelease = MAINCODERELEASE ; 
  thisSession.subCodeRelease = SUBCODERELEASE ; 
  thisSession.mainDataRelease = MAINCODERELEASE ; 
  thisSession.subDataRelease = 0 ; 
  thisSession.from = 0 ; 
  thisSession.upLink = 0 ; 
  if (graphPrompt("Choose a title for this session","","t")
      && (cp = freeword()))
    strcpy (thisSession.title, cp) ;

  b.h.disk = 1 ;
  b.h.nextdisk = 0 ;
  b.h.type = 'S' ;
  b.h.key  = 0 ;

  if (!(f = filopen("database/blocks","wrm","rb")))
    { messout ("I create the main database file") ;
      if(!( f = filopen("database/blocks","wrm","w")))
	messcrash ("Disk extend cannot open the main database file") ;
      fprintf(f, "Hello") ;
    }
  fclose(f) ;
  
  messdump ("**********************************************************\n") ;
  messdump ("Making a new database file and starting at session 1 again\n") ;

  readModels() ;
    /* application specific initialisations, in quovadis.wrm */
  diskPrepare() ;
  sessionClose(TRUE) ;

  f = filopen("database/ACEDB","wrm","w") ;
  fprintf (f,"Destroy this file to reinitialize the database\n") ;
  fclose (f) ;

  wormClose() ;
}

/*************************************************************/
static KEY sessionKey = 0 ;
void sessionRegister(void)
{
  OBJ Session , Model = bsCreate(KEYMAKE(_VSession,0)) ;
  KEY key , kPlus, kMinus ;
  DISK d ;
  int free, plus, minus, used ;

  if(!isWriteAccess())
    return ;

  if(!Model || 
     !bsFindTag(Model,_Up_linked_to)
     )  /* Always check on latest tag added to ?Session */
    { messerror("%s\n%s\n%s\n%s\n%s\n",
		"FATAL, an eror occured in sessionRegister",
		"Probably, your model for the session class is not correct",
		"and corresponds to an earlier release.",
		"Look for the latest version of the wspec/models.wrm file,",
		"read the model file and try again.") ;
      fprintf(stderr,"%s\n%s\n%s\n%s\n%s\n",
		"FATAL, an eror occured in sessionRegister",
		"Probably, your model for the session class is not correct",
		"and corresponds to an earlier release.",
		"Look for the latest version of the wspec/models.wrm file,",
		"read the model file and try again.") ;
      exit(1) ;
    }

  bsDestroy(Model) ;				 
				 
  lexaddkey(messprintf("s-%d-%s",
		       thisSession.session,thisSession.name),

	    &key, _VSession) ;
  sessionKey = key ;
  lexaddkey(messprintf("p-%d",thisSession.session),&kPlus,_VBat) ;
  lexaddkey(messprintf("m-%d",thisSession.session),&kMinus,_VBat) ;
  Session = bsUpdate(key) ;
  if(!Session)
    messcrash
      ("Session register fails to create the session object") ;

  if (thisSession.session == 1)
    bsAddTag(Session,_Permanent_session) ;
  bsAddData(Session,_User,_Text,&thisSession.name) ;
  { char buffer[72] ;
    if(timeStamp() && dateStamp() &&
       strlen(dateStamp()) + strlen(timeStamp()) < 70)
      { sprintf(buffer,"%s,  %s", timeStamp(), dateStamp()) ;
	bsAddData(Session,_Date,_Text,&buffer) ;
      }
  }
  if (*thisSession.title)
    bsAddData(Session,_Session_Title,_Text,thisSession.title) ;
  bsAddData(Session,_Session,_Int,&thisSession.session) ;
/*    if(graphPrompt("Choose a title for this session","","t")
       && (cp = freeword()))
      bsAddData(Session,_Session_Title,_Text,cp) ;
*/
  if(thisSession.from)
    bsAddKey(Session,_Created_from,thisSession.from) ;
  if(thisSession.upLink)
      bsAddKey(Session,_Up_linked_to,thisSession.upLink) ;


  thisSession.mainCodeRelease = MAINCODERELEASE ; 
  thisSession.subCodeRelease = SUBCODERELEASE ; 
  bsAddData(Session,_CodeRelease,
	    _Int,&thisSession.mainCodeRelease) ;
  bsAddData(Session,_bsRight,
	    _Int,&thisSession.subCodeRelease) ;
  bsAddData(Session,_DataRelease,
	    _Int,&thisSession.mainDataRelease) ;
  bsAddData(Session,_bsRight,
	    _Int,&thisSession.subDataRelease) ;

  diskavail(&free,&used,&plus, &minus) ;
  d = lexDisk(__Global_Bat) ;
  bsAddData(Session,_GlobalBat,_Int,&d) ; 
  bsAddData(Session,_bsRight,_Int,&used) ;

  d = lexDisk(__lexi1) ;
  bsAddData(Session,_GlobalLex,_Int,&d) ;

  d = lexDisk(__lexi2) ;
  bsAddData(Session,_SessionLex,_Int,&d) ;
  d = lexDisk(__lexa2) ;
  bsAddData(Session,_bsRight,_Int,&d) ;
  d = lexDisk(__voc2) ;
  bsAddData(Session,_bsRight,_Int,&d) ;
  d = lexDisk(__lexh2) ;
  bsAddData(Session,_bsRight,_Int,&d) ;

  d = lexDisk(__lexi3) ;
   bsAddData(Session,_VocLex,_Int,&d) ;
  d = lexDisk(__lexa3) ;
  bsAddData(Session,_bsRight,_Int,&d) ;
  d = lexDisk(__voc3) ;
  bsAddData(Session,_bsRight,_Int,&d) ;
  d = lexDisk(__lexh3) ;
  bsAddData(Session,_bsRight,_Int,&d) ;

  bsAddKey(Session,_BatPlus,kPlus) ;
  bsAddData(Session,_bsRight,_Int,&plus) ;

  bsAddKey(Session,_BatMinus,kMinus) ;
  bsAddData(Session,_bsRight,_Int,&minus) ;
  bsSave(Session) ;
}

/*************************************************************/
#define NFATHERS 3 /* Number of sessions kept alive, must be >= 1 */
void sessionClose(BOOL doSave)
{ int presentSemid , presentStatus ;
  KEY son ;

  if (!isWriteAccess())
    return ;
  if (!doSave)
    { sessionReleaseWriteAccess() ;
      return ;
    }

  showDeads = FALSE ;  /* Locate father  in the session tree */
  if (thisSession.session != 1)
    sessionTreeConstruct() ;
   
  if(saveAll()) 
    {     /* I keep only NFATHERS session alive */
       if (thisSession.session != 1)
	 {
	   if (NFATHERS == 1)
	     { if (sessionFindSon(thisSession.upLink, &son) == 1)
		 sessionOedipe () ;
	     }
	   else
	     while (sessionKillAncestors(NFATHERS)) ;
	 }

      messdump(messprintf("END OF SESSION : %s %s\n",
			  dateStamp(), timeStamp())) ;
      
      semaphoreStatus(&presentSemid, &presentStatus) ;
      if(SEM && semid != presentSemid)
	messcrash
	  ("%s%s","SessionWriteSuperBlock,",
	   " the semaphore id changed since i got write access") ;
                     /* increment session number */
      b.gAddress = thisSession.gAddress = lexDisk(__lexi1) ;
      b.mainRelease = MAINCODERELEASE ;
      b.subCodeRelease = SUBCODERELEASE ;
      thisSession.from =  sessionKey ;
      thisSession.upLink =  sessionKey ;
      b.h.session =  thisSession.session++ ;
      diskblockwrite(&b) ;
      diskWriteSuperBlock(&b) ;  /* writes and zeros the BATS */
    }

  sessionReleaseWriteAccess()  ;
}

/*************************************************************/

BOOL isWriteAccess(void)
{
  return writeAccess != 0 ;
  /* should be writeAccess != graphActive->user */
}

/*************************************************************/

static void sessionReleaseWriteAccess(void)
{
  if (!isWriteAccess())
    messcrash("Unbalanced Get/Release Write Access") ;
#ifdef IPC
  releaseSemaphore() ;
#endif
  releaseWriteLock() ;
  writeAccess = 0 ;
  sessionFlipMenu(FALSE) ;
}

/*************************************************************/

BOOL  sessionCheckUserName(void)
{
  int line = 0 ;
  FILE * fil ; 
  register char *cp , *name ;

  Stack uStack = stackCreate(32) ;

    /* Internal security,
     * Only users listed in wspec/passwd.wrm
     * may be granted write access to the database.
     */

    /* I cannot use cuserid
     * because I want the real userid
     * and, depending on systems, cuserid may return
     * either the real or the effective userid
     * which I would often reset, see Install documentation.
     */
  name = getlogin() ;
  if (!name)
    {
      messout("sessionWriteAccess : getlogin failed, sorry.");
      return FALSE ;
    }



  fil = filopen("wspec/passwd","wrm","r") ;
  if (!fil)
    { messout("Sorry, sessionWriteAccess cannot find wspec/passwd.wrm");
      return FALSE ;
    }
  
  while(line ++ ,freeread(fil))
    {
      freenext() ;
      if(freestep('#'))         /* skip # lines */
	continue ;
                               /* read user name */
      if (cp = freeword())
	  pushText(uStack,cp) ;
    }

  fclose(fil) ;
  while (!stackAtEnd(uStack))
    if(!strcmp(name,stackNextText(uStack)))
      { stackDestroy(uStack) ;
	return TRUE ;
      }
  stackDestroy(uStack) ;
  return FALSE ;
}

/*************************************************************/

static BOOL  sessionSecurity(void)
{
  FILE * fil ; 

    /* Recommanded security,
     * The acedb administrator must list the write authorized
     * users in wspec/passwrd.
     *
     * The program must belong to the acedb administrator
     * and must set the effective user id :
     *   chmod 4755 ${ACEDB}/bin/acedb       (or 4750)
     *
     */

    /* First, we verify that the user is not 
     * running his own version of the program,
     * i.e. that he is indeed allowed to write by Unix.
     * This is automatic if his effective id is indeed that of the
     * owner of the database as implied by chmod 4755
     */

  
  fil= filopen("database/ACEDB","wrm","a") ;
  if(fil)
    fclose(fil) ;
  else
    { messout("%s\n%s\n%s",
	      "Sorry, i would not be able to rewrite the ACEDB file ",
	      "please run the command",
	      "chmod 4755 bin/xace" );
      return FALSE ;
    }
  
  fil= filopen("logbook","wrm","a") ;
  if(fil)
    fclose(fil) ;
  else
    { messout("%s\n%s\n%s",
	      "Sorry, i would not be able to rewrite the logbook",
	      "please run the command",
	      "chmod 4755 bin/xace" );
      return FALSE ;
    }
  
  if(!(fil = filopen("database/blocks","wrm","a")))
    { messout("%s%s",
	      "Unix would not let you write to the database,",
	      "Run the command chmod 4755 bin/xace") ;
      return FALSE ;
    }
  
  fclose(fil) ;
  
  if(!sessionCheckUserName())
   { messout("%s\n%s\n%s",
	    "You are not a registered user,",
	    "Your userid is not listed in wspec/passwd.wrm",
	    "ask around or read the documentation." );
     return FALSE ;
   }
  return TRUE ;
} 

/*************************************************************/

void sessionWriteAccess(void)
{
  register char *cp ;

  if (isWriteAccess() || !sessionSecurity())
    return ;

        /* A signal I hope to other processes */
  if(cp = isWriteLocked())
    { messout("%s%s\n%s%s%s",
	      "Sorry",cp," already has write access.",
	      "or you crashed with write access",
	      "In this case remove the file database/lock.wrm") ;
      return ;
    }

  cp = getlogin() ;  /* Cannot fail, verified by sessionCheckUserName */
  strncpy(thisSession.name,cp,78) ;
  setWriteLock() ;

#ifdef IPC
      /* Semaphore system */
  if(!getSemaphore())
    { messout("Sorry, the semaphore is Locked elsewhere") ;
      releaseWriteLock() ;
      return ;
    }
#endif

         /* Further verification */
  if(!checkSessionNumber())
    {
      messout("%s\n%s\n%s",
	      "Sorry,",
 "while you where reading the database, another process updated it.",
 "You can go on reading, but to write, you must quit and rerun." ) ;
      releaseWriteLock() ;
      return ;
    }


  /* Success */
  writeAccess = TRUE ;
  messdump ("Write access : User %s\n", cp) ;
  sessionFlipMenu(TRUE) ;
}

/**************************************************************/

void sessionSaveAccess(void)
{
  if (isWriteAccess())
    sessionClose(TRUE) ;
  else
    sessionWriteAccess() ; /* try to grab it */
}

/*************************************************************/

extern MENUOPT* mainMenu ;
extern BOOL ACEDB_SU ;

static void sessionFlipMenu(BOOL access)
{
  if(access)
    { menuSuppress ("Write Access", mainMenu) ;
      menuRestore ("Save", mainMenu) ;
    }
  else 
    { menuSuppress ("Save", mainMenu) ;
      menuRestore ("Write Access", mainMenu) ;
    }
}

/*************************************************************/
/*************************************************************/
/**************** Interactive session control ****************/
/*************************************************************/

static void localDestroy(void)
{ 
  sessionSelectionGraph =  0 ;
  arrayDestroy(sessionTree) ;
  stackDestroy(stText) ;
}

/*************************************************************/

static void otherSessionStart(void)
{ extern void lexClear(void) ;
  extern void mainCleanUp(void) ;

  if(!sessionChosen)
   { messout("First pick a session in the Session List") ;
     return ;
   }

  if (isWriteAccess())
    { messout("First save your previous work") ;
      return ;
    }
  
  sessionWriteAccess() ;
  if (!isWriteAccess())
    { messout("Sorrry, you do not have write access") ;
      return ;
    }

  mainCleanUp() ;  /* all but active graph */
  sessionClose(TRUE) ;   /* To empty all caches */
  
  arrayDestroy(sessionTree) ;
  lexClear() ;

  sessionStart() ;
}

/****************************************************************/

static void sessionTreeConstruct(void)
{ KEY key = 0 , kk ; int n = 0 ; char *cp ;
  OBJ obj ;
  ST * st , *st1 ;
  int gen, i, j , max ;
  BOOL found ;

  sessionTree = arrayReCreate(sessionTree, 50,ST) ;

  if (!stText)
    stText = stackCreate(50) ;
  else
    stackClear(stText) ;
  
  pushText(stText,"dummy") ;

  if (lexNext(_VSession, &key))   /* jump model */
    while (lexNext(_VSession, &key))
      if (obj = bsCreate(key))
	{ st = arrayp(sessionTree, n++, ST) ;
	  st->key = key ;
	  if (bsGetKey(obj,_Created_from, &kk))
	    st->father = st->ancester = kk ;
	  if (bsGetKey(obj,_Up_linked_to, &kk))
	    st->ancester = kk ;
	  else if (st->key != KEYMAKE(_VSession, 1))
	    st->isDestroyed = TRUE ; /* i.e. to old a release */
	  if (bsFindTag(obj,_Destroyed_by_session))
	    st->isDestroyed = TRUE ;
	  if (bsFindTag(obj,_Permanent_session))
	    st->permanent = TRUE ;
	  if (bsGetData(obj,_User, _Text, &cp))
	      { st->user = stackMark(stText) ;
		pushText(stText,cp) ;
	      }
	  if (bsGetData(obj,_Date, _Text, &cp))
	      { st->date = stackMark(stText) ;
		pushText(stText,cp) ;
	      }
	  if (bsGetData(obj,_Session_Title, _Text, &cp))
	      { st->title = stackMark(stText) ;
		pushText(stText,cp) ;
	      }

	  bsDestroy(obj) ;
	}

  /* Add thisSession */
  st = arrayp(sessionTree, n++, ST) ;
  st->key = _This_session ;
  st->father = st->ancester = thisSession.from ;

  /* links the st */
  for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
    for (j = 0, st1 = arrp(sessionTree,0,ST) ; j < arrayMax(sessionTree) ; st1++, j++)
      if(st->key == st1->ancester)
	st1->sta = st ;

  /* Count generations */
  if (showDeads)
    { 
      for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
	if (!st->ancester)
	  st->generation = 1 ;
    }
  else
    {
      for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
	if ((!st->isDestroyed) && ( !st->ancester || (st->sta && st->sta->isDestroyed)) )
	  st->generation = 1 ;
    }
  gen = 1 ;
  max = arrayMax(sessionTree) ;
  found = TRUE ;
  while (found)
    { found = FALSE ;
      if (showDeads)
	{
	  for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
	    if (st->sta && st->sta->generation == gen)
	      { st->generation = gen + 1 ;
		found = TRUE ;
	      }
	}
      else
	{
	  for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
	    if (!st->isDestroyed && st->sta && st->sta->generation == gen)
	      { st->generation = gen + 1 ;
		found = TRUE ;
	      }
	}
      gen++ ;
    }
}
    
/************************************************************/  

static void sessionPick(KEY box)
{ static int previousBox = 0 ;
  int i ;
  ST *st ;

  for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
    if (st->box == previousBox)
      graphBoxDraw(st->box, BLACK, st->color) ;
      	
  for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
    if (st->box == box)
      { if (!st->isDestroyed && st->key != _This_session)
	  sessionChosen = st->key ;
	else
	  sessionChosen = 0 ;
	graphBoxDraw(st->box, BLACK, RED) ;
	if (st->key != 1)
	  display(st->key, 0, 0) ;
      }
  previousBox = box ;
}

/************************************************************/  

static int sessionFindSon(KEY father, KEY *sonp)
{ int i , n = 0 ; ST *st, *sta ;
  if (!sessionTree)
    sessionTreeConstruct() ;

  *sonp = 0 ;             /* Suicide */
  if (father == _This_session)
    return -2 ;
  if (!father)           /* Trivial */
    return -3 ;
  if (father == KEYMAKE(_VSession, 1))
    return -6 ;

  for (i = 0, sta = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; sta++, i++)
    if (sta->key == father)
      break ;

  if (sta->key != father)       /* not found error */
    return -5 ;
  
  if (sta->isDestroyed)          /* dead */
      return -3 ;
 
  for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
    if(!st->isDestroyed && st->sta == sta)
      { *sonp = st->key ;
	n++ ;
      }
  
  if (n == 1)            /* is st1 in my direct ancestors */
    { for (i = 0, st = arrp(sessionTree,0,ST) ; i < arrayMax(sessionTree) ; st++, i++)
	if (st->key == thisSession.from)
	  {
	    while (st)
	      { if (st == sta)
		  return 1 ;
		st = st->sta ;  
	      }
	    return -4 ;
	  }
      return -4 ;        /* not ancestor */
    }

  return n ;             /* number of sons */
}

/************************************************************/  

static void sessionDestroy(void)
{ KEY son ;
  
  if (!isWriteAccess())
    { messout("Sorrry, you do not have write access") ;
      return ;
    }

  if (!sessionChosen)
    { messout ("First select a living session.") ;
      return ;
    }

  if (!iskeyold(sessionChosen))
    { messout ("This session no longer exists, sorry") ;
      return ;
    }

  /* I save once here to let the system a chance to crash if need be */
  saveAll() ; 
 
  if (graphQuery("Destroy this session"))
    {  if (sessionChosen == thisSession.from)
	 { messout("Thou shall not kill thy own father, Oedipus") ;
	   return ;
	 }
      switch(sessionFindSon(sessionChosen, &son))
	{
        case -2:
	  messout("Thou shall not commit suicide") ;
	  return ;
	case -3:
	  messout ("This session no longer exists, sorry") ;
	  return ;
	case -4:
	  messout ("You can only kill your direct ancestors or an extremity") ;
	  return ;
	case -5:
	  messerror ("Sorry, sessionDestroy cannot locate this session") ;
	  return ;
	case -6:
	  messout ("You cannot kill the empty database") ;
	  return ;
	case 0: case 1:
	  sessionFuse(sessionChosen, son) ;
  /* This save will rewrite Session to disk without altering the BAT */
	  saveAll() ;
	  arrayDestroy(sessionTree) ;
	  sessionControl() ;
	  return ;
	  break ;
	default:
	  messout("This session has more than 1 son, i cannot kill it.") ;
	  return ;
	}
    }
}

/****************************************************************/

static void sessionRename(void)
{ OBJ Session ;
  char *cp ;

  if (!isWriteAccess())
    { messout("Sorrry, you do not have write access") ;
      return ;
    }
    
  if (!sessionChosen)
    { messout ("First select a session.") ;
      return ;
    }

  if (!iskeyold(sessionChosen))
    { messout ("This session no longer exists, sorry") ;
      return ;
    }

  /* I save once here to let the system a chance to crash if need be */
  saveAll() ; 
  Session = bsUpdate(sessionChosen) ;
  if (!bsFindTag(Session, _Up_linked_to))
    { messout("I cannot rename a session created before code 1-6") ;
      bsDestroy(Session) ;
      return ;
    }
  bsGetData(Session, _Session_Title,_Text,&cp) ;
  if (graphPrompt("Rename this session and make it permanent",cp,"t") &&
      ( cp = freeword()) )
    { if(strlen(cp) > 80)
	*(cp + 80) = 0 ;
      bsAddTag(Session, _Permanent_session) ;
      bsAddData(Session, _Session_Title,_Text,cp) ;
    }
  bsSave(Session) ;
  /* This save will rewrite Session to disk without altering the BAT */
  saveAll() ;
}

/************************************************************************/
  /* Emergency exit, in model.c and other places 
   * menacing the integrity of the database
   */
void sessionHardCrash(char* s)
{
  messerror(s) ;
  fprintf(stderr,"\n%s\n I Quit \n",s) ;
  sessionReleaseWriteAccess() ;
  wormClose() ;
}

/****************************************************************/
static void sessionDraw()  ;
static void sessionShowDeads (void)
{
  showDeads = showDeads ? FALSE : TRUE ;
  sessionTreeConstruct() ;
  sessionDraw() ;
}

/****************************************************************/

static MENUOPT
   sessionMenu[] = {
     graphDestroy,"Quit",
     help,"Help",
     otherSessionStart,"Start Session",
     sessionRename,"Rename Session",
     sessionDestroy,"Destroy Session",
     sessionShowDeads,"Invoke/Repulse the spirits",
     0,0
     } ;

/****************************************************************/

static void sessionDrawBranch(ST* st, Array x)
{ int i, g = st->generation , n = arrayMax(sessionTree) ;
  ST *st1 ;
  
  if(!array(x, g, int))
    array(x, g, int ) = 4 ;

         /* Draw self */
  st->box = graphBoxStart() ;
  st->x = array(x, g, int) ;
  st->y = 7 + 3*g ;
  graphText(name(st->key), st->x, st->y ) ;
  st->len =strlen(name(st->key)) ;
  array(x, g, int) += st->len + 5 ;

  graphBoxEnd() ;
  if(st->isDestroyed)  /* cannot pick ! */
    st->color = WHITE ;
  else
    st->color = YELLOW ;
  if (st->key == _This_session || st->key == KEYMAKE(_VSession, 1))
    st->color = LIGHTBLUE ;

  graphBoxDraw(st->box, BLACK, st->color) ;
  for (i = 0, st1 = arrp(sessionTree,0,ST) ; i < n ;
       st1++, i++)
    if (st->ancester == st1->key)
      graphLine(st1->x + st1->len/2, st1->y + 1.0, st->x + st->len/2 , st->y - .0) ;
  
               /* Recursion */
  for (i = 0, st1 = arrp(sessionTree,n - 1,ST) ; i < n ;
       st1--, i++)
     if (st1->sta == st)
       sessionDrawBranch(st1, x) ;
}

/****************************************************************/

static void sessionDraw() 
{ int i, max = 0 , maxGen = 0 , n ;
  ST *st ;
  Array maxX = arrayCreate(12, int) ;
  
  if(!sessionTree)
    sessionTreeConstruct() ;
  
  n = arrayMax(sessionTree) ;
  graphClear() ;
  graphPop() ;
  for (i = 0, st = arrp(sessionTree,n - 1,ST) ; i < n ; st--, i++)
    { if (st->generation == 1)
	sessionDrawBranch(st, maxX) ;
      if (st->generation > maxGen)
	maxGen = st->generation ;
    }

  for (i=0 ; i< arrayMax(maxX); i++)
    if (max < arr(maxX,i, int))
      max = arr(maxX,i,int) ;
  if (!max)
    max = 40 ;
 
  graphTextBounds (max + 3 , 7 + 3 * maxGen) ;
  graphButtons(sessionMenu, 2., 2., 40) ;
  graphRedraw() ;
  arrayDestroy(maxX) ;
}

/*************************************************************/

void sessionControl(void)
{ 
  sessionChosen = 0 ;

  if ( graphActivate(sessionSelectionGraph))
    graphPop() ;
  else
    sessionSelectionGraph =  displayCreate(DtSession) ;
  graphMenu(sessionMenu) ;

  graphRegister(DESTROY,localDestroy) ;
  graphRegister(RESIZE, sessionDraw) ;
  graphRegister(PICK, (GraphFunc) sessionPick) ;
  sessionDraw() ;
}

/*************************************************************/
/*************************************************************/
