/*  File: queryexe.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 *         Used anywhere for non interactive executions.       
 * Exported functions:
         query, queryKey, queryGrep                           
 * HISTORY:
 * Last edited: Apr  3 14:18 1992 (mieg)
 * Created: Tue Nov  5 16:53:55 1991 (mieg)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "keyset.h"
#include "bs.h"
#include "lex.h"
#include "pick.h"
#include "sysclass.wrm"
#include "systags.wrm"

 /* Query operators */
typedef enum { NULLQOP, SUBQUERY, SUBEXP, NUMBER, TEXT, CLASS, TAG, FOLLOW,
	       OR, XOR, AND, NOT, LT, LE, GT, GE, NEQ, EQ, COMPOSITE_TAG } QOP ;

typedef struct conditionNode *CDT;
struct conditionNode
{ QOP qop ;
  CDT up, d1, d2 ; /* down1, down2 tree structure */
  union {  KEY tag ; int class ; float x ;} n ;
  int mark ;   /* mark in the associated stack */
} ;

typedef struct condition 
{ Stack stack ;
  CDT cdt ;
} *COND ;

/**************************************************************/

/* Manage CDT allocation
   hope they speed things up/prevent fragmentation
   in any case, they will help monitoring memory usage
   NB they must be paired.
*/

static Stack freeCDTstack = 0 ;
static int nCDTused = 0, nCDTalloc = 0 ;        /* useful for debug */

static CDT CDTalloc (void)       /* self managed calloc */
{
  static int blocSize = 512 ;
  CDT p ;
  int i ;
 
  nCDTalloc++ ;

  if (!freeCDTstack)
    freeCDTstack = stackCreate (4*blocSize) ;
  if (stackEmpty (freeCDTstack))
    { p = (CDT) messalloc (blocSize * sizeof (struct conditionNode)) ;
      for (i = blocSize ; i-- ; ++p)
        push (freeCDTstack,p,CDT) ;
/*       printf ("Adding %d to CDT free list\n",blocSize) ; */
      blocSize *= 2 ;
    }
  p = pop (freeCDTstack,CDT) ;
  memset (p, 0, sizeof (struct conditionNode)) ;
  ++nCDTused ;
  return p ;
}

static void CDTfree (CDT cdt)
{
  push (freeCDTstack,cdt,CDT) ;
  --nCDTused ;
}

void CDTstatus (int *used, int *alloc)
{ *used = nCDTused ; *alloc = nCDTalloc ;
}

/**************************************************************/

/* Manage COND allocation
   hope they speed things up/prevent fragmentation
   in any case, they will help monitoring memory usage
   NB they must be paired.
*/

static Stack freeCONDstack = 0 ;
static int nCONDused = 0, nCONDalloc = 0 ;        /* useful for debug */

static COND CONDalloc (void)       /* self managed calloc */
{
  static int blocSize = 512 ;
  COND p ;
  int i ;
 
  nCONDalloc++ ;

  if (!freeCONDstack)
    freeCONDstack = stackCreate (4*blocSize) ;
  if (stackEmpty (freeCONDstack))
    { p = (COND) messalloc (blocSize * sizeof (struct condition)) ;
      for (i = blocSize ; i-- ; ++p)
        push (freeCONDstack,p,COND) ;
/*       printf ("Adding %d to COND free list\n",blocSize) ; */
      blocSize *= 2 ;
    }
  p = pop (freeCONDstack,COND) ;
  memset (p, 0, sizeof (struct condition)) ;
  ++nCONDused ;
  return p ;
}

static void CONDfree (COND cond)
{
  push (freeCONDstack,cond,COND) ;
  --nCONDused ;
}

void CONDstatus (int *used, int *alloc)
{ *used = nCONDused ; *alloc = nCONDalloc ;
}

/**************************************************************/
       /* Recursively destroys a tree of conditions */

static void cdtDestroy(CDT c)
{
  if(c)
    {
      if(c->d1)
	cdtDestroy(c->d1) ;
      if(c->d2)
	cdtDestroy(c->d2) ;
      c->d1 = c->d2 = 0 ;  /* stops eventual loop */
      CDTfree(c) ;
    }
}

/**************************************************************/
       /* Call with 0 if you want to rescan the same text */
       /* itemizes a text considering quotes and nested parentheses */
static QOP cdtItem(char *text, char **item)
{
  static char *previousText = 0 , *cp , backup = 0 ;
  int np = 0 , ncb ;
  static  QOP qop = 0 ;

  *item = 0 ;
  if(!text || (previousText != text))
    { cp = text ; previousText = text ; qop = 0; }
  else 
    {
      *cp = backup ;
      switch (qop)
	{   /*skip the former end delimiter */
	case SUBEXP :
	case TEXT :
	 if(*cp) cp++ ;
	default:
	    break;
	}
    }

  qop = 0 ;
  if(!text)
    return 0 ;

  while (TRUE) switch(*cp)
    {
    case 0 :
      goto done ;

    case '(':
      if (qop)
	goto done ;
      *item = ++cp ; /* inside the parenthese */
      np = 1 ;
      while (np)
	switch (*cp++)
	  {
          case 0:
	    messout("Unbalanced parenthesis :\n %s",text) ;
	    return FALSE ;
	  case '(':
	    np++ ;
	    break ;
	  case ')':
	    np--; 
	    break ;
	  }
      cp-- ; /* The paranthesis will be masked */
      qop = SUBEXP ;
      goto done ;

    case '{':
      if (qop)
	goto done ;
      *item = ++cp ; /* inside the parenthese */
      ncb = 1 ;
      while (ncb)
	switch (*cp++)
	  {
          case 0:
	    messout ("Unbalanced curly bracket :\n %s",text) ;
	    return FALSE ;
	  case '{':
	    ncb++ ;
	    break ;
	  case '}':
	    ncb--; 
	    break ;
	  }
      cp-- ; /* The paranthesis will be masked */
      qop = SUBEXP ;  /* change to SUBQUERY when code is complete */
      goto done ;

    case '\"':
      if (qop)
	goto done ;
      *item = ++cp ;
      while (*cp && (*cp != '\"')) cp++ ;
      if(!*cp)
	{ messout("Unbalanced double quote :\n %s",text) ;
	  return FALSE ;
	}
           /* The final double quote will be masked */
      qop = TEXT ;
      goto done ;

    case ')':
      messout("Unexpected closing  parenthesis :\n %s",text) ;
      return FALSE ;

    case ' ' : case '\t' : case '\n' : case ','  :  
      if (qop)
	goto done ;
      cp++ ; /* otherwise ignore */
      break ;
    
    case '!':
      if (qop)
	goto done ;
      cp++ ;
      if (*cp == '=')
	{ cp++ ;
	  qop = NEQ ;
	}
      else
	qop = NOT ;
      goto done ;

    case '^':
      if (qop)
	goto done ;
      cp++ ;
      qop = XOR ;
      goto done ;

    case '<':
      if (qop)
	goto done ;
      cp++ ;
      if (*cp == '=')
	{ cp++ ;
	  qop = LE ;
	}
      else
	qop = LT ;
      goto done ;

    case '>':  /* Multiple meaning, >Tag with nothing on the left
		  means FOLLOW tag */
      if (qop)
	goto done ;
      cp++ ;
      if (*cp == '=')
	{ cp++ ;
	  qop = GE ;
	}
      else
	qop = GT ;
      goto done ;

    case '=':   /* I treat = and == the same way */
      if (qop)
	goto done ;
      cp++ ;
      if (*cp == '=')
	{ cp++ ;
	}
      qop = EQ;
      goto done ;

    case '&':   /* I treat & and && the same way */
      if (qop)
	goto done ;
      cp++ ;
      if (*cp == '&')
	{ cp++ ;
	}
      qop = AND ;
      goto done ;

    case '|':   /* I treat | and || the same way */
      if (qop)
	goto done ;
      cp++ ;
      if (*cp == '|')
	{ cp++ ;
	}
      qop = OR ;
      goto done ;

    case '#':
      if (qop)
	goto done ;
      cp++ ;
      qop = COMPOSITE_TAG ;
      goto done ;

    default:
      if (!*item)
	{ qop = TAG ; /* to be reexamined when the word is complete */
	  *item = cp ;
	}
      cp++ ;
      break ;
    }
 done:
  backup = *cp ;
  *cp = 0 ;
  return qop ;
}

/**************************************************************/
/**************************************************************/
       /* Recursive construction of a tree of conditions */
/**************************************************************/

       /* Scans the cdt Tree
	  looking for the occurence of a query operator 
	  in order of increasing priority
	  */
static BOOL cdtReorder(CDT *top, Stack s)
{
  CDT c, cc, new, ca, cb, cr, cl ;
  QOP q ;
  c = *top ;
     /* Anything on d2 side has lower priority */

     /* A leading GT is a follow */
  if (c->qop == GT)
    { /* printf("found leading GT") ;*/
      c->qop = FOLLOW ;
    }

  while(c)
    {
      if (c->d2)
	if (!cdtReorder(&(c->d2), s)) 
	  return FALSE ;
      c = c->d1 ;
    }

     /* Now we reorder the d1 side recursively */
     /* We find highest operator and make of it a subexpression */
     /* In case of equality I work left to right as usual */
     /* i.e. the rightest binds more strongly */

  while (TRUE)
    {
      cc = c = *top ;
      q = c->qop ;
      while(c->d1)
	{ 
	  c = c->d1 ;
	  if (c->qop >=q) /* rightest of given precedence */
	    { q = c->qop ;
	      cc = c ;
	    }
	}
            /* Tags or FOLLOW  already reorganized */
      if(q <= TAG || ((q == FOLLOW) && cc->d2)) 
	break ;
      if(q == NOT)  /* Unary operator */
	{ if(!cc->d1)
	    { messout("op NOT with nothing to the right") ;
	      return FALSE ;
	    }
          cr = cc->d1 ;
	  ca = cc->up ; cb = cr->d1 ;
          
	  new = CDTalloc() ;
	  new->qop = cc->qop ;
          new->d2 = cr ;
	  cr->up = new ;
          new->up = cc ;
	  cc->d2 = new ; cc->d1 = 0 ;
	  cr->d1 = 0 ;

          if(ca) ca->d1 = cc ;
	  cc->up = ca;
	  if(cb) cb->up = cc ;
	  cc->d1 = cb;
	  cc->qop = SUBEXP ;
	}
      else if(q == FOLLOW )  /* FOLLOW tag */
	{ if (!cc->d1)
	    { messout("op > with nothing to the right of it") ;
	      return FALSE ;
	    }
          cr = cc->d1 ;
	  cb = cr->d1 ;
          
	  if(cr->qop == TEXT)
	    if (*stackText(s,cr->mark) == '?')
	      { /* Autocomplete to a Class */
		int t = 0;
		char *name;
		for(t=0;t<256;t++)
		  if(  (name = pickClass2Word(t))
		     && pickMatch(name,
				  1 + stackText(s,cr->mark)))
		    { cr->qop = CLASS ;
		      cr->n.class = t ;
		      messout("Autocompleting >%s to >?%s",
			      stackText(s,cr->mark),
			      name) ;
		      break ;
		    }
	      }
	    else
	      { /* Autocomplete to a tag */
		KEY t = 0;
		while(lexNext(0,&t))
		  if(pickMatch(name(t),
			       stackText(s,cr->mark)))
		    { cr->qop = TAG ;
		      cr->n.tag = t ;
		      messout("Autocompleting >%s to >%s",
			      stackText(s,cr->mark),
			      name(t)) ;
		      break ;
		    }
	      }

	  if(cr->qop != CLASS && cr->qop != TAG)
	    { messout( "Missing or unrecognised tag after the leading >") ;
	      return FALSE ;
	    }

	  cc->d1 = cb ;	cc->d2 = cr ;
	  cr->up = cc ;
	  if(cb)
	    cb->up = cc ;
	  cr->d1 = 0 ;
          if(cr->d2)
	    { messout( "Sorry, parsing error after a >");
	      return FALSE ;
	    }
	  cc->qop = FOLLOW ;
	}
      else /* binary operator */
	{ if(!cc->up ||
	     ( cc->up->qop == SUBEXP &&
	       cc->up->d2 == cc ) )
	    { messout("op %d with nothing to the left", q) ;
	      return FALSE ;
	    }
	  if(!cc->d1)
	    { messout("op %d with nothing to the right", q) ;
	      return FALSE ;
	    }
          cl = cc->up ; cr = cc->d1 ;
	  ca = cl->up ; cb = cr->d1 ;
          
	  new = CDTalloc() ;
	  new->qop = cc->qop ;
          new->d1 = cl ; new->d2 = cr ;
	  cl->up = cr->up = new ;
          new->up = cc ;
	  cc->d2 = new ; cc->d1 = 0 ;
	  cl->d1 = 0 ; cr->d1 = 0 ;

          if(ca) 
	    if(ca->d1 == cl) 
	      ca->d1 = cc ;
	    else
	      ca->d2 = cc ;
	  cc->up = ca;
	  if(cb) cb->up = cc ;
	  cc->d1 = cb;
	  cc->qop = SUBEXP ;

	  if(*top == cl)
	    *top = cc ;
	}
    }
  return TRUE ;
}

/***********************/

       /* construct a chain of all operators at a given level */
static CDT cdtConstructLevel(char *text, Stack s)
{
  CDT cdt = 0, top = 0, next ;
  char dummy ;
  char *item ;
  QOP qop ;
  KEY tag ;
  float x ; 
  int i ;

  if(!text || !*text)
    return 0 ;
  cdtItem(0,&item) ;
  while (qop = cdtItem(text, &item))
    {
      next = CDTalloc() ;
      if(!top)
	top = next ;
      if(cdt)
	cdt->d1 = next ;
      next->up = cdt ;
      cdt = next ;
      cdt->qop = qop ;
      switch(qop)
	{
	case TAG:
	  if(!strcmp(item,"EQ"))
	    { cdt->qop = EQ ;
	      break ;
	    }
	  else if(!strcmp(item,"NEQ"))
	    { cdt->qop = NEQ ;
	      break ;
	    }
	  else if(!strcmp(item,"AND"))
	    { cdt->qop = AND ;
	      break ;
	    }
	  else if(!strcmp(item,"OR"))
	    { cdt->qop = OR ;
	      break ;
	    }
	  else if(!strcmp(item,"NOT"))
	    { cdt->qop = NOT ;
	      break ;
	    }
	  else if(!strcmp(item,"XOR"))
	    { cdt->qop = XOR ;
	      break ;
	    }
	  else if (!strcmp(item,"NEXT"))
	    { cdt->n.tag = _bsRight ;
	      break ;
	    }
	  else if (!strcmp(item,"HERE"))
	    { cdt->n.tag = _bsHere ;
	      break ;
	    }
	  else if((*item == '?') && (i = pickWord2Class(item + 1)))
	    { cdt->n.class = i ;
	      cdt->qop = CLASS ;
	      break ;
	    }
          else if(lexword2key(item, &tag, _VSystem))
	    { cdt->n.tag = tag ;
	      /* fall through to match tag with a _Text */
	    }
	  else if(sscanf (item,"%f%c",&x,&dummy) == 1)
                  /* dummy trick checks no unmatched characters */
	    { cdt->qop = NUMBER;
	      cdt->n.x = x ;
	      /* fall through to match 1967 with a _Text 1967 */
	    }
	  else
	    { cdt->qop = TEXT;
	      /* fall through */
	    }

	case TEXT: case SUBEXP :
	  cdt->mark = stackMark(s) ;
	  pushText(s,item) ;
	  break ;
	default:
	  break;
	}
    }
  return top ;
}

/***********************/
 /* Expands subexpressions presented in parenthesis to the scanner */

static BOOL cdtExpand(CDT cdt, Stack s) ;

static CDT cdtConstructExpand(char *text, Stack s) 
{
 CDT cdt =  cdtConstructLevel(text,s) ;
 cdtExpand(cdt,s) ;
 return cdt ;
}

static BOOL cdtExpand(CDT cc, Stack s)
{
 while(cc)
   {
     if (cc->qop == SUBEXP)
	 { cc->d2 = cdtConstructExpand(stackText(s,cc->mark), s) ;
	   cc->d2->up = cc ;
	 }
     cc = cc->d1 ;
   }
 return TRUE ;
}

/**********/
       /* Destruction */
void  condDestroy(COND cond) 
{
  if(cond)
    { stackDestroy (cond->stack) ;
      cdtDestroy(cond->cdt) ;
      CONDfree(cond) ;
    }
}

/**********/
       /* Initialisation part */
static BOOL condConstruct(char *text, COND* cdp)
{
  COND cond = 0 ;
  
  while(*text == ' ')
    text++ ;
  if(!text || !*text)
    return FALSE ;

  cond = CONDalloc() ;

  cond->stack = stackCreate(3*strlen(text)) ; /* wild guess */
  pushText(cond->stack,text) ;
  cond->cdt =   cdtConstructLevel(text, cond->stack) ;
  if ( cdtExpand(cond->cdt,cond->stack) 
      &&  cdtReorder(&(cond->cdt),cond->stack) )
    { *cdp = cond ;
      return TRUE ;
    }
  else
    { condDestroy(cond) ;
      *cdp = 0 ;
      return FALSE ;
    }
}

/**********/
       /* Number comparisons */
static BOOL condCheck (double x, QOP qop, double y)
{
  switch(qop)
    {
    case NEQ:
      return x != y  ? TRUE : FALSE ;
    case LE:
      return x <= y  ? TRUE : FALSE ;
    case LT:
      return x < y ? TRUE : FALSE ;
    case GE:
      return x >= y ? TRUE : FALSE ;
    case GT:
      return x > y ? TRUE : FALSE ;
    default:
      messcrash("Unknown operator %d in condCheck.",qop) ;
    }
  return FALSE ; /* for compiler happiness */
}

/************************************************************************/

static BOOL condMatch(Stack s, KEY key, OBJ* objp, CDT cdt) 
{
  KEY k ; OBJ subObj, surObj ;
  char *cp ;  float f ; int i ;
  switch (cdt->qop)
    {
    case FOLLOW:
      messout("Internal error : condMatch should not parse >Tag'") ;
      return FALSE ;

    case OR:
      if(condMatch(s,key,objp, cdt->d1))
	return TRUE ;
      else
	return condMatch(s,key,objp, cdt->d2) ;
      
    case AND:
      if(!condMatch(s,key,objp, cdt->d1))
	return FALSE ;
      else
	return condMatch(s,key,objp, cdt->d2) ;
      
    case XOR:
      if(condMatch(s,key,objp, cdt->d1))
	return !condMatch(s,key,objp, cdt->d2) ;
      else
	return condMatch(s,key,objp, cdt->d2) ;
      
    case NOT:
      return !condMatch(s,key,objp, cdt->d2) ;
      
    case SUBEXP:
      return condMatch(s,key,objp, cdt->d2) ;
      
    case EQ:
      if(!*objp && !(*objp = bsCreate(key)))
	return FALSE ;
      if ( bsGetKeyTags(*objp, cdt->d1->n.tag,&k))
	while (TRUE)
	  {
	    if (k <= _LastC)   /* Comment */
	      { if (bsGetData(*objp, _bsHere, k, &cp)
		    && pickMatch(cp,
			      stackText(s,cdt->d2->mark)))
		  return TRUE ;
	      }
	    else if (k <= _LastN) /* Number */
	      { if (cdt->d2->qop == NUMBER)
		  if (k == _Int)
		    { if (bsGetData(*objp, _bsHere, k, &i)
			  && ( ((float) i) == cdt->d2->n.x ))
			return TRUE ;
		    }
		  else if(k == _Float)
		    { if (bsGetData(*objp, _bsHere, k, &f)
			  && ( f == cdt->d2->n.x ))
			return TRUE ;
		    }
	      }
	    else             /* ordinary Key */
	      { cp = 0 ;
		while (nextName(k, &cp))
		  if (pickMatch(cp,
				stackText(s,cdt->d2->mark)))
		  return TRUE ;
	      }
	    if (!bsGetKey(*objp,_bsDown, &k))
	      break ;
	  }
      return FALSE ;
      
    case NEQ: case LT: case LE: case GT: case GE: /* Meaningful for numbers only */
      if (cdt->d1 && cdt->d2 &&  cdt->d2->qop == NUMBER
          && (*objp || (*objp = bsCreate(key)))
	  && bsGetKeyTags(*objp, cdt->d1->n.tag,&k))
	    while (TRUE)
	      {
		if (k <= _LastC)   /* Comment */
		  { if (bsGetData(*objp, _bsHere, k, &cp)
			&& (f = (float) atof(cp))
			&& condCheck(f,cdt->qop,cdt->d2->n.x)
			)
		      return TRUE ;
		  }
		else if (k <= _LastN) /* Number */
		  { if (k == _Int)
			{ if (    bsGetData(*objp, _bsHere, k, &i)
			      && condCheck((float) i,cdt->qop,cdt->d2->n.x)
			      )
			    return TRUE ;
			}
		      else if(k == _Float)
			{ if (bsGetData(*objp, _bsHere, k, &f)
			      && condCheck(f,cdt->qop,cdt->d2->n.x)
			      )
			    return TRUE ;
			}
		  }
                /* ordinary Key  cannot eval to a number */
		if (!bsGetKey(*objp,_bsDown, &k))
		  break ;
	      }
      return FALSE ;
      
    case COMPOSITE_TAG:
      if(!*objp && !(*objp = bsCreate(key)))
	return FALSE ;
      if(!condMatch(s,key,objp, cdt->d1))
	return FALSE ;
      if (!bsGetObj(*objp,_bsHere,&subObj))
	return FALSE ;
      surObj = *objp ;
      *objp = subObj ;
      i = condMatch(s,key,objp, cdt->d2) ;
      bsDestroy(subObj) ;
      *objp = surObj ;
      return i ;

    case TAG:
      if(!*objp && !(*objp = bsCreate(key)))
	return FALSE ;
      return  
	bsFindTag(*objp,cdt->n.tag) ;

    case NUMBER:   /* isolated number treat as a TEXT */
    case TEXT:   /* isolated text, applies to the sample itself */
      cp = 0 ;
      if (nextName(key, &cp))
	if (pickMatch(cp, stackText(s,cdt->mark)))
	  return TRUE ;
      return FALSE ;

    case CLASS:
      return class(key) == cdt->n.class ;
    default:
      break;
    }
  messcrash("Unknown operator in condMatch") ;
  return FALSE ; /* for compiler happiness */
}

/***********************************************************/

static void cdtDump(COND cond, CDT cdt, int level)
{
  int i = level ;
  if(!cdt)
    { printf("\n cdtDumping NULL \n") ;
      return ;
    }
  while(i--) 
    printf("\t") ;
  printf("qop = %d", cdt->qop) ;
  if(cdt->qop == TAG && cdt->n.tag)
    printf(" tag = %s ", name(cdt->n.tag) );
  else  if(cdt->qop == CLASS && cdt->n.class)
    printf(" class = %s ", pickClass2Word(cdt->n.tag) );
  if ( cdt->mark)
      printf(" cp = %s ",  stackText(cond->stack, cdt->mark)) ;
  printf("\n") ;
  if(cdt->d2)
    cdtDump(cond,cdt->d2, level + 1);
  if(cdt->d1)
    cdtDump(cond,cdt->d1, level );
}

/**********************************************************/
   /* Works en place */
static KEYSET queryFilter (KEYSET set, CDT cdt, Stack stack) 
{    
  register int i = 0, imax, j = 0 ;
  KEY k ;
  OBJ obj;   /* created if needed by condMatch */
  
  if (!set)
    return keySetCreate() ;
  imax = keySetMax(set) ;
            /* works on a single object at a time */
  for(i=0;i<imax;i++)
    { k = keySet(set,i) ;
      obj = 0 ;
      if (condMatch(stack, k, &obj, cdt) )
	keySet(set,j++) = k ;	/* Keeps set sorted as a subset */
      if(obj)
	bsDestroy(obj) ;
    }
  keySetMax(set) = j ;
  return set ;
}

/***********************************/

static KEYSET queryExpand(KEYSET oldSet, CDT cdt, Stack stack) 
{
  KEYSET s = keySetCreate() ;
  KEY k , k2 ;
  int i , imax = oldSet ? keySetMax(oldSet) : 0  , j = 0 ;
  OBJ obj ;
  KEY tag = cdt->d2->n.tag ;

  for(i=0;i<imax;i++)
    { k = keySet(oldSet,i) ;
      if(obj = bsCreate(k))
	{
	  if( bsGetKey(obj,tag,&k2))
	    { 
	      if(class(k2))
		keySet(s, j++) = k2 ;
	      while( bsGetKey(obj,_bsDown,&k2))
		if(class(k2))
		  keySet(s, j++) = k2 ;
	    }
	  bsDestroy(obj) ;
	}
    }
  keySetSort (s) ;
  keySetCompress(s) ;
  if(cdt->d1)
    queryFilter(s, cdt->d1, stack) ;
  return s ;
}

/***********************************/

static KEYSET queryClass(CDT cdt, Stack stack) 
{
  KEYSET s = keySetCreate() ;
  KEY k = 0 ; /* primes lexanext */
  int t =  cdt->d2->n.class , i = 0 ;

  while(lexNext(t,&k))
   keySet(s,i++) = k ;

  keySetSort(s) ;   /* keysetCompress not needed */
  if(cdt->d1)
    queryFilter(s, cdt->d1, stack) ;

  return s ;
}

/**********************************************************/

static KEYSET queryExecute (KEYSET oldSet, COND condition)
{    
  CDT cdt = condition->cdt , cr =cdt ;
  Stack stack = condition->stack ;

 /* cdtDump(condition, condition->cdt, 0) ; */

  if(cdt->qop == SUBEXP && cdt->d2)
    cr = cdt->d2 ;
  if(cr->qop == FOLLOW)
    if(cr->d2->qop == TAG)
      return queryExpand(oldSet, cr, stack) ;
    else
      return queryClass(cr, stack) ;
  else
    return queryFilter(oldSet, cr, stack) ;
}

/**********************************************************/
  /* Hopefully clear local error message on easy errors */

static BOOL checkParentheses (char *text)
{
  static Stack bra = 0 ;
  char c, old, hold, *cp = text ;

  bra = stackReCreate (bra, 40) ;
  while (c = *cp++)
    switch (c)
      {
      case '{': push (bra, '}', char) ; break ;
      case '[': push (bra, ']', char) ; break ;
      case '(': push (bra, ')', char) ; break ;
      case '}': case ']': case ')':
	if (stackEmpty (bra))
	  { hold = *cp ; *cp = 0 ;
	    messout ("Unbalanced %c in:\n %s", c, text) ;
	    *cp = hold ;
	    return FALSE ;
	  }
	old = pop (bra, char) ;
	if (old != c)
	  { hold = *cp ; *cp = 0 ;
	    messout ("Mismatched %c in:\n %s", c, text) ;
	    *cp = hold ;
	    return FALSE ;
	  }
	break ;
      }

  if (!stackEmpty (bra))
    { messout ("Missing final %c in:\n %s", pop(bra, char), text) ;
      return FALSE ;
    }
  return TRUE ;
}

/**********************************************************/

static KEYSET querySingleCommand (KEYSET oldSet, char *text)
{ COND cond ;   
  KEYSET newSet = 0 ;

  if (!checkParentheses (text))
    return 0 ;
  if (condConstruct (text, &cond))
    newSet = queryExecute (oldSet, cond) ;
  condDestroy (cond) ;
  return newSet ;
}

/**********************************************************/

  /* Can process several command separated by semi columns */
KEYSET query (KEYSET oldSet, char *text)
{ KEYSET newSet = 0, runningSet = keySetCopy(oldSet) ;
  static Stack myText = 0 , runningText = 0 ;
  int mine ;
  char *cp ;
          /* oldSet belongs to calling routine */
  myText = stackReCreate(myText, 80) ;
  pushText(myText, text) ;
  mine = freesettext (stackText(myText, 0),"") ;
  while (freecard(mine))
    { if ((cp = freepos()) && *cp)
	{ runningText = stackReCreate(runningText, 80) ;
	  pushText(runningText, cp) ;
	  newSet = querySingleCommand(runningSet,stackText(runningText,0)) ;
	  if (!newSet)
	    { while (freecard (mine)) ;
	      keySetDestroy (runningSet) ;
	      return oldSet ? oldSet : keySetCreate () ;
	    }
	  if(newSet != runningSet)  
	    keySetDestroy(runningSet) ;
	  runningSet = newSet ;
	}
    }
  return newSet ;
}

/***************************************************************************/
   /* filter text applied to a single key */
   /* avoids having to create s each time */
KEYSET queryKey(KEY key,  char *text)
{
  static KEYSET s = 0 ;
  if(!s)
    s = arrayCreate(1,KEY) ;
  keySet(s,0) = key ;
  return query(s,text) ;
}

/***************************************************************************/
		/* Search all the X classes and all visible names */

KEYSET queryGrep(char *text)
{ KEY k, k1 ;
 int  t , i , n = 0 ;
  OBJ obj ; char *cp ; 
  KEYSET s = keySetCreate(), kS ;
  

  t = 256 ;
  while(t--)
    if(pickXref(t))  /* pick all cross referenced objects */
      { for (k = 0 ; lexNext (t,&k) ;)
	  { cp = 0 ; 
/*	    if (nextName(k, &cp)) looping thru all lexNext anyway */
	    cp = name(k) ;
	      if (pickMatch (cp, text))
		{ if (obj = bsCreate(k))
		    {
		      if (bsGetKey (obj, _Quoted_in, &k1)) 
			do keySet(s,n++) = k1 ;
		      while (bsGetKey (obj, _bsDown, &k1)) ;
		      bsDestroy(obj);
		    }
		}
	  }
      }
    else                   /* pick matching objects */
      if (pickVisible(t))
	{ for (k = 0 ; lexNext (t,&k) ;)
	     { cp = 0 ; 
 /*	       if (nextName(k, &cp))  looping thru all lexNext anyway */
		 cp = name(k) ;
		 if (pickMatch (cp, text))
		   {     /* pick Self and all objects refered by a matching object */
		     if(kS = bsVisibleKeySet(k))
		       { i = keySetMax(kS) ;
			 while(i--)
			   keySet(s, n++) =  keySet(kS,i) ;
			 keySetDestroy(kS) ;
		       }
		   }
	      }
	}
  keySetSort(s) ;
  keySetCompress(s) ;
  return s ;
}

/*********************************************************/
/*
void cdtTest(void)
{ COND cond ;
  char * cp ;
  while(graphPrompt("Test queryScan : ", "", "t") )
   if(cp = freeword() &&
      condConstruct(cp,&cond) )
     {
       cdtDump(cond, cond->cdt, 0) ;
       condDestroy(cond) ;
     }
}
*/
/*********************************************************/
/*********************************************************/


