/*  File: picksubs.c
 *  Author: Jean Thierry-Mieg (mieg@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description:
 * Exported functions:
     pickInit()
       must be called once at start of program.
     pickWord2Class()
       returns a class number
     pickMatch matches a name to a template 
       accepts the wild char * (anything) and ? (single character).
     displayCreate(DISPLAYTYPE)
      graphCreate(), importing title and positions from wspec/displays.wrm
 * HISTORY:
 * Last edited: Apr 22 16:07 1992 (mieg)
 * * Apr  2 13:41 1992 (mieg): Simplified customization by defining
   the new wspec files (sys)options and displays.
 * Created: Thu Apr  2 13:41:10 1992 (mieg)
 *-------------------------------------------------------------------
 */

#include <ctype.h>

#include "acedb.h"
#include "lex.h"
#include "pick.h"
#include "display.h"
#include "graph.h"

FREEOPT pickVocList[256] = { 0,  "Choose from"} ;
PICKLIST pickList[256] ;

/******************************************************************/

static Stack dtStack = 0 ;
static Array clA = 0 ;

static void pickGetDisplayTypes(void)
{
  FILE * fil = filopen("wspec/disptype","wrm","r") ;
  char c, *cp ;
  int state = 0 ;

  if (!fil)
    messcrash("pickGetDisplayTypes cannnot find wspec/disptype.wrm") ;

  dtStack = stackReCreate(dtStack, 80) ;
  while(freeread(fil))
    { 
      while (cp = freeword())
	{ 
	  switch (state)
	    { case 0:
		if (!strcmp(cp, "typedef"))
		  state = 1 ;
		break ;
	      case 1:
		if (!strcmp(cp, "enum"))
		  state = 2 ;
		break ;
	      case 2:
		if (!strcmp(cp, "ZERO"))
		  { pushText(dtStack, cp) ;
		    freewordcut(",", &c) ;
		    state = 3 ;
		  }
		break ;
	      case 3:
		if (*cp == '}')
		  return ;
		if (*cp == ',' || *cp == '=' || isdigit(*cp))
		  break ;
		pushText(dtStack, cp) ;
		state = 3 ;
		break ;
	      }
	}
    }
  messcrash ("pickGetDisplayTypes cannnot interpret wspec/disptype.wrm") ;
}

/********************************************/

char * className(KEY kk)  
{     
 return
   pickList[class(kk)].name ; /* Skip ? */
}

/******************************************************************/

static void pickGetClassNames(char * fileName, char * ending) 
{
  int  k , v ;
  int line = 0 ;
  FILE * fil = filopen(fileName, ending,"r") ;
  char name[32] ;
  register char *cp ;


  if (!fil)
    messcrash("pickInit cannot find Class definition file %s.%s",
	      fileName, ending) ;

  if (!clA)
    clA = arrayCreate(20, int) ;
  v = arrayMax(clA) ;

  while(line ++ ,freeread(fil))
                                                /* read #define */
    if ((cp = freeword()) && !strcmp(cp,"#define"))
      {
	cp =freeword() ;                      /* read class name */
	if(!cp)
	  messcrash("Error parsing line %d of %s.%s", fileName, ending) ;
	if(*cp++ != '_' || *cp++ !='V')
	  messcrash("No _V in line %d : %s of class file",line,cp);
	if(!*cp || !strlen(cp) || strlen(cp)>=32)
	  messcrash("Bad class name %s in line %d of class file ",cp,line);
	strcpy(name,cp);
                    
        k = -999;                           /* read class number */
        if (!freeint(&k)  || k<0 || k>255)
	  messcrash("Impossible class number %d in line %d : %s",
		    k,line,name);
        if (*pickList[k].name)
	  messcrash(" Redefinition of class %d : %s = %s",
		    k, pickList[k].name, name );
	strcpy (pickList[k].name, name );
	array(clA, v++, int) = k ;
      }
} 

/******************************************************************/

static FREEOPT classOptions[] =
{
  7, "class options",
  'h', "Hidden",
  'v', "Visible",
  'a', "Array",
  'b', "Btree",
  'x', "XREF",
  'd', "Display",
  't', "Title",
  } ;

static void pickGetClassProperties(char * fileName, char * ending) 
{
  int d,  k ;
  int line = 0 ;
  FILE * fil = filopen(fileName, ending,"r") ;
  char *cp , *cq ;
  KEY tag , option ;

  if (!fil)
    messcrash("pickInit cannot find Class definition file %s.%s",
	      fileName, ending) ;

  while(line ++ ,freeread(fil))
                                                /* read #define */
    if ((cp = freeword()) && *cp++ == '_' && *cp++ == 'V')
      {
	if(!*cp)
	  messcrash("Isolated _V while parsing line %d of %s.%s", line , fileName, ending) ;

	if (!( k = pickWord2Class(cp)) && strcmp (cp, "System"))
	  messcrash ("In %s.wrm line %d, class name %s does not match classes.wrm",
		    fileName, line, cp) ;
	
	while (TRUE)
	  {
	    freenext() ;
	    if (!freestep('-')) 
	      if (cp = freeword())
		messcrash ("In %s.wrm line %d, no - at start of option %s", 
			   fileName, line, freepos()) ;
	      else
		break ;
	    
	    if (!freekey(&option, classOptions))
	      messcrash ("In options.wrm line %d, unkown option %s", line, freepos()) ;
	    
	    switch (option) 
	      {
	      case 'h':
		pickList[k].visible = 'H' ;  /* default is visible */
		break ;
	      case 'v':
		pickList[k].visible = 'V' ;  /* default is visible */
		break ;
	      case 'a':
		pickList[k].type = 'A' ;
		break ;
	      case 'b':
		pickList[k].type = 'B' ;
		break ;
	      case 'x':
		pickList[k].Xref = TRUE ;
		pickList[k].visible = 'H' ; 
		pickList[k].type = 'B' ;
		break ;
	      case 'd':                       /* Display type (enum) */
		freenext() ;
		cp = freeword() ;
		d = 0 ;  /* I will substract 1 at the end, to be able to select 0  */
		stackCursor(dtStack, 0) ;
		while (d++, cq = stackNextText(dtStack))
		  if (!strcmp(cq, cp))
		    goto goodType ;
		messcrash("Bad display type  in line %d of %s at %s ", 
			  line, fileName, freepos()) ;
	      goodType:
		pickList[k].display = d ;
		break ;
	      case 't':
                               /* Tag for Name expansion  */
		pickList[k].tag = 0 ;
		if (cp = freeword())
		  { 
		    if (!lexword2key(cp, &tag, 0))  /* 0 is _VSystem */
		      messcrash("Unknown ExpandTag  %s in line %d of %s ", 
				cp, line, fileName) ;
		    else
		      pickList[k].tag = tag ;
		  }
		break ;
	      }
	  }
      }
}

/******************************************************************/

static void pickDefaultProperties(void)
{
  int   j, k ;
  static int  v = 0;          /* number of visible classes */
   
      /* clA is used to have the Visible Class in main menu in
       * same order as they appear in classes.wrm.
       */
  for (j = 0 ; j < arrayMax(clA) ; j++)
    { k = array(clA, j, int) ;
      if (*pickList[k].name)
	{
	  if (!pickList[k].type)            /* default is B */
	    pickList[k].type = 'B' ;
	  if (!pickList[k].visible)
	    if (pickList[k].type == 'B')
	      pickList[k].visible = 'V' ;   /* for B default is visible */
	    else
	      pickList[k].visible = 'H' ;   /* for A default is hidden */
	  
	  if ( pickList[k].visible == 'V')
	    {
	      v++ ;
	      pickVocList[v].text = pickList[k].name ;
	      pickVocList[v].key = k;	  
	    }
	  if (!pickList[k].display)
	    if (pickList[k].type == 'B')
	      pickList[k].display = 2 ;   /* for B default is TREE */
	    else
	      pickList[k].display = 1 ;   /* for A default is hidden */
	  pickList[k].display -- ;
	}
    }

  pickVocList[0].key =  v ;
  arrayDestroy(clA) ;
} 

/******************************************************************/

int  pickWord2Class(char *word)     /* returns class of word */
{
 register  int i = 256 ;

 while(--i && strcmp(word,pickList[i].name)) ;
 
/*
 if (i<=0)
   messout("pickWord2Class called on unknown Class %s",word);
*/
 return i ;
}

/*************************************************************************/
/*************************************************************************/

/* match to template with wildcards.   Authorized wildchars are * ? #
     ? represents any single char
     * represents any set of chars
   case sensitive.   Example: *Nc*DE# fits abcaNchjDE23

   returns 0 if not found
           1 + pos of first sigificant match (i.e. not a *) if found
*/

int pickMatch (char *cp,char *tp)
{
  char *c=cp, *t=tp;
  char *ts, *cs, *s = 0 ;
  int star=0;

  while (TRUE)
    switch(*t)
      {
      case '\0':
 /*
        return (!*c ? ( s ? 1 + (s - cp) : 1) : 0) ;
*/
	if(!*c)
	  return  ( s ? 1 + (s - cp) : 1) ;
	if (!star)
	  return 0 ;
        /* else not success yet go back in template */
	t=ts; c=cs+1;
	if(ts == tp) s = 0 ;
	break ;
      case '?' :
	if (!*c)
	  return 0 ;
	if(!s) s = c ;
        t++ ;  c++ ;
        break;
      case '*' :
        ts=t;
        while( *t == '?' || *t == '*')
          t++;
        if (!*t)
          return s ? 1 + (s-cp) : 1 ;
        while (freeupper(*c) != freeupper(*t))
          if(*c)
            c++;
          else
            return 0 ;
        star=1;
        cs=c;
	if(!s) s = c ;
        break;
      default  :
        if (freeupper(*t++) != freeupper(*c++))
          { if(!star)
              return 0 ;
            t=ts; c=cs+1;
	    if(ts == tp) s = 0 ;
          }
	else
	  if(!s) s = c - 1 ;
        break;
      }
}


/******************************************/
/********* Display Customization  *********/
/******************************************/
/******************************************************************/

static Stack gtStack = 0 ;

  /* This list must be constructed by hand and match Graph.h */

static void pickGetGraphTypes(void)
{
  gtStack = stackCreate(128) ;
  pushText(gtStack, "PLAIN") ;
  pushText(gtStack, "TEXT_SCROLL") ;
  pushText(gtStack, "TEXT_FIT") ;
  pushText(gtStack, "MAP_SCROLL") ;
}

/****************************************************************/

static Array dArray = 0 ;
typedef struct acedbDisplayStuct  AcedbDisplay ;
struct acedbDisplayStuct 
  { 
    char title[32], help[32] ;
    float x, y, width, height ;
    int type ;  /* from GraphType */
    } ;


static FREEOPT displayOptions[] =
{
  7, "display options",
  'g', "GraphType",
  't', "Title",
  'w', "Width",
  'h', "Height",
  'x', "Xposition",
  'y', "Yposition",
  'a' , "Help"
  } ;

static void localDisplayInit(void)
{
  AcedbDisplay *dp ;
  int d , gt ;
  int line = 0 ;
  char *cp , *cq ; float x ;
  KEY option ;
  FILE * fil = filopen("wspec/displays", "wrm", "r") ;
  

  if (!fil)
    messcrash("displayInit cannot find display definition file : wspec/displays.wrm") ;

  dArray = arrayCreate (12, AcedbDisplay) ;
  pickGetGraphTypes() ;

	              /* Defaults */
  d = 0 ;
  stackCursor(dtStack, 0) ;
  while (cq = stackNextText(dtStack))
    { dp = arrayp(dArray, d++, AcedbDisplay) ;
      if (! dp->type)
	dp->type = 1 + TEXT_FIT ;
      dp->x = .001 ; dp->y = 0.001 ; dp->width = .3  ; dp->height = .5 ;
      strncpy(dp->title, cq, 31) ;
      strncpy(dp->help, "No_help", 31) ;
    }
  
  while(line ++ ,freeread(fil))
    /* read _D... */
    if ((cp = freeword()) && *cp++ == '_' && *cp++ == 'D')
      {
	if(!*cp)
	  messcrash("Error parsing line %d of wspec/displays.wrm", line ) ;
	
        d = -1  ;
	stackCursor(dtStack, 0) ;
	while (d++, cq = stackNextText(dtStack))
	  if (!strcmp(cq, cp))
	    goto goodType ;
	messcrash("Bad display type %s  in line %d of wspec/displays.wrm at", 
		  cp, line, freepos()) ;
      goodType:
	dp = arrayp(dArray, d, AcedbDisplay) ;
	
	while (TRUE)
	  {
	    freenext() ;
	    if (!freestep('-')) 
	      if (cp = freeword())
		messcrash ("In displays.wrm line %d, no - at start of option %s", line, freepos()) ;
	      else
		break ;
	    
	    freenext() ;
	    if (!freekey(&option, displayOptions))
	      messcrash ("In displays.wrm line %d, unkown option %s", line , freepos()) ;
	    
	    switch (option) 
	      {
	      case 't':
		if (cp = freeword())
		  strncpy(dp->title, cp, 31) ;
		break ;
	      case 'a':
		if (cp = freeword())
		  strncpy(dp->help, cp, 31) ;
		break ;
	      case 'g':                       /* GraphType (enum) */
		freenext() ;
		cp = freeword() ;
		gt = 0 ;  /* I will substract 1 at the end, to be able to select 0  */
		stackCursor(gtStack, 0) ;
		while (gt++, cq = stackNextText(gtStack))
		  if (!strcmp(cq, cp))
		    goto goodGraphType ;
		messcrash("Bad graph type  in line %d of displays at %s ", 
			  line,  freepos()) ;
	      goodGraphType:
		dp->type =  gt ;
		break ;
	      case 'x':
		if (!freefloat(&x))
		  messcrash("Missing float x value in line %d of wspec/displays.wrm at %s",
			    line,  freepos()) ;
		if (x < 0 || x > 1.3)
		  messout ("In wsepc/displays.wrm, line %d, x value %f out of range 0 1.3",
			   line, x ) ;
		else
		  dp->x  = x ;
		break ;
	      case 'w':
		if (!freefloat(&x))
		  messcrash("Missing float width value in line %d of wspec/displays.wrm at %s",
			    line,  freepos()) ;
		if (x < 0 || x > 1.3)
		  messout ("In wsepc/displays.wrm, line %d, width value %f out of range .0 1.3",
			   line, x ) ;
		else
		  dp->width  = x ;
		break ;
	      case 'y':
		if (!freefloat(&x))
		  messcrash("Missing float y value in line %d of wspec/displays.wrm at %s",
			    line,  freepos()) ;
		if (x < 0 || x > 1.0)
		  messout ("In wsepc/displays.wrm, line %d, y value %f out of range 0 1.0",
			   line, x ) ;
		else
		  dp->y  = x ;
		break ;
	      case 'h':
		if (!freefloat(&x))
		  messcrash("Missing float height value in line %d of wspec/displays.wrm at %s",
			    line,  freepos()) ;
		if (x < 0 || x > 1.0)
		  messout ("In wsepc/displays.wrm, line %d, height value %f out of range 0 1.0",
			   line, x ) ;
		else
		  dp->height  = x ;
		break ;
	      }
	  }
      }
}

/********************************************************/
/******************* Public Routine *********************/
/********************************************************/

void pickInit(void)
{ register int k ;

  k = 256 ;
  while(k--)
    pickList[k].type = 0;

  pickGetDisplayTypes() ;
  pickGetClassNames("wspec/classes","wrm") ;
  pickGetClassNames("wspec/sysclass","wrm") ;

  pickGetClassProperties("wspec/options","wrm") ;
  pickGetClassProperties("wspec/sysoptions","wrm") ;

  pickDefaultProperties() ;
}

/******************************************/

Graph displayCreate(DISPLAYTYPE d)
{
 AcedbDisplay *dp ;
 Graph g ;

 if (!arrayExists(dArray))
   localDisplayInit() ;

 if (d < 0 || d > arrayMax(dArray))
   messcrash ("displayCreate received bad type %d", (int) d) ;

 if (d == ZERO)
   return 0 ;

 dp = arrp(dArray, d, AcedbDisplay) ;
 
 g =  graphCreate (dp->type - 1 , dp->title, 
		dp->x, dp->y, dp->width, dp->height) ;

 if (g)
   graphHelp(dp->help) ;
 return g ;
}

/********************************************************/
/********************************************************/
