/*  File: graphxt.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: Xt/Xaw level of X version of graph package
 * Exported functions: many
 * HISTORY:
 * Last edited: Apr  3 00:22 1992 (mieg)
 * * Jan  2 02:29 1992 (rd): exposeCall now restores gActive
 * Created: Thu Jan  2 02:27:24 1992 (rd)
 *-------------------------------------------------------------------
 */

#include "regular.h"
#include "array.h"
#include "graphxt_.h"
#include "key.h"
#include<X11/Xaw/Cardinals.h>
#include<X11/Xaw/Viewport.h>
#include<X11/Xaw/Scrollbar.h>
#include<X11/Xaw/SimpleMenu.h>
#include<X11/Xaw/SmeBSB.h>
#include<X11/Xaw/AsciiText.h>
#include<X11/Xaw/Dialog.h>
#include<X11/Xaw/Command.h>
#define XK_MISCELLANY
#include<X11/keysymdef.h>
#include <signal.h>

#if defined(WCS) && !defined(NO_WCS)
  extern void wcommInit (Display *d, Window w) ;
  extern void wcommFinish (void) ;
  #include "wcomm.h"
#else
  static void wcommInit (Display *d, Window w) { }
  static void wcommFinish (void) { }
  static BOOL WPropMessage (XEvent *ev) { return FALSE ;}
#endif

/* globals within the X graph package */

float  gPixAspect = 1.0 ;
XtAppContext app_con;	/* application context */
Widget root_widget;	/* parent of all */

/*******/

static Arg args[10];		/* convenience variable */

/* typeName used for widget instance names - 
   corresponds to enum GraphType for resource setting
*/
static char* typeName[] = {"plain",
			   "text_scroll",
			   "text_fit",
			   "map_scroll"} ;

/* screenx, screeny are for positioning and sizing created graphs
   SUN and other common screens are 1152x900 (1.3x900 = 1170)
   could/should get these from DISPLAY server in graphDevInit()
*/
static float screenx = 900 ;
static float screeny = 900 ;

static Associator assw2g = 0;	/* = assCreate */

static void fakeResize (Graph_ g) ;
static BOOL activeSet (Widget w) ;

/*****************************************************************/

/* registered functions */

static void exposeCall(Widget w, XEvent* ev, String *params, Cardinal *num_params);
static void mouseCall(Widget w, XEvent* ev, String *params, Cardinal *num_params);
static void pickCall(Widget w, XEvent* ev, String *params, Cardinal *num_params);
static void keyboardCall(Widget w, XEvent* ev, String *params, Cardinal *num_params);
static void resizeCall (Widget w, XEvent* ev, String *params, Cardinal *num_params) ;
static void propertyCall (Widget w, XEvent* ev, String *params, Cardinal *num_params) ;
static void activeCall (Widget w, XEvent* ev, String *params, Cardinal *num_params) 
{ activeSet (w) ; }

static void destroyCallback (Widget w, XtPointer client_data, XtPointer call_data);
static void defaultMenuProc (Widget w, XtPointer client_data, XtPointer call_data);

/*****************************************************************/

/* action translations for the simple widget (= graph itself) */

static String actionTrans = 
  "<Expose>: exposeCall()\n\
   <ConfigureNotify>: resizeCall()\n\
   <Btn1Down>: pickCall()\n\
   <Btn1Motion>: mouseCall(\"1\")\n\
   <Btn1Up>: mouseCall(\"2\")\n\
   <Btn2Down>: mouseCall(\"3\")\n\
   <Btn2Motion>: mouseCall(\"4\")\n\
   <Btn2Up>: mouseCall(\"5\")\n\
   <Btn3Down>: activate() \
               XawPositionSimpleMenu(menu) \
               XtMenuPopup(menu)\n\
   <Key>: keyboardCall()\n\
   <PropertyNotify>: propertyCall()" ;

/* define here all the normal returns on things */

static XtActionsRec regular_actions[] = {
  {"exposeCall", exposeCall},
  {"resizeCall", resizeCall},
  {"mouseCall", mouseCall},
  {"pickCall", pickCall},
  {"keyboardCall", keyboardCall},
  {"activate", activeCall},
  {"promptReturn", promptReturn},
  {"propertyCall", propertyCall}
};

static String fallback_resources[] =
{
  /*
    right,top... refer to geometry management for the form widget they keep
    the distance constant between a widget and the edge of a form
  */
  "*right:  ChainRight",
  "*left:   ChainLeft",
  "*top:    ChainTop",
  "*bottom: ChainBottom",
  "*resizable: TRUE",
/*  "*AllowShellResize*: TRUE",  -- RMD want for prompt only */
/*  "*Text.Translations:\
	Ctrl<Key>F:	forward-character() \n\
	Ctrl<Key>B:     backward-character() \n\
	Ctrl<Key>D:	delete-next-character() \n\
	Ctrl<Key>A:	beginning-of-line() \n\
	Ctrl<Key>E:	end-of-line() \n\
	Ctrl<Key>K:	kill-to-end-of-line() \n\
	Ctrl<Key>W:	kill-selection() \n\
	Ctrl<Key>Y:	unkill() \n\
	<Key>Right:	forward-character() \n\
	<Key>Left:      backward-character() \n\
	<Key>Delete:    delete-previous-character() \n\
	<Key>BackSpace: delete-previous-character() \n\
	<Key>Return:    promptReturn() \n\
	<Key>:		insert-char() \n\
	<Btn1Down>:     select-start() \n\
	<Btn1Motion>:	extend-adjust() \n\
	<Btn1Up>:	extend-end(PRIMARY, CUT_BUFFER0) \n\
	<Btn2Down>:	insert-selection(PRIMARY, CUT_BUFFER0) \n\
	<Btn3Down>:	extend-start() \n\
	<Btn3Motion>:   extend-adjust() \n\
	<Btn3Up>:	extend-end(PRIMARY, CUT_BUFFER0)",
*/
  NULL,
} ;


/**********************/

extern void invokeDebugger (void) ;

static int errorHandler (Display *display, XErrorEvent *err)
{
  char msg_code[80] ;
  char msg_major[80] ;

  XGetErrorText (display, err->error_code, msg_code, 80) ;
  XGetErrorDatabaseText (display, "GraphPackage", 
			 messprintf ("XRequestMajor.%d",err->request_code), 
			 "undefined", msg_major, 80) ;
  fprintf (stderr, "X error: code %d (%s), request (%d, %d) %s\n", 
	   err->error_code, msg_code, 
	   err->request_code, err->minor_code, msg_major) ;
  invokeDebugger () ;
  return 0 ;
}

static void warningHandler (char *msg)
{
  fprintf (stderr, "Xt warning: %s\n", msg) ;
  invokeDebugger () ;
}

static Widget YInit (int *argcp, char** argv)
     /* creates an unrealized top widget to be parent for the rest */
{
  int i = 0 ;

/*   XtSetArg(args[i], XtNmappedWhenManaged, False); i++; */
  root_widget = XtAppInitialize(&app_con, "GraphPackage",
				NULL, ZERO,
				argcp, argv, fallback_resources,
				args, i) ;

/* XSynchronize (XtDisplay(root_widget), True) ; */
  XSetErrorHandler (errorHandler) ;
  XtAppSetWarningHandler (app_con, warningHandler) ;

  XtAppAddActions(app_con, regular_actions, XtNumber(regular_actions) );
/*  XawSimpleMenuAddGlobalActions(app_con) ; */
  return root_widget ;
}

static void intProc (void)  /* default CTRL-C handler */
{ extern int intFlag ;
  if (++intFlag > 1  &&  graphQuery ("Do you want to abort?"))
    messcrash ("User initiated abort") ;
}

void graphDevInit (int *argcptr, char **argv)
{
  assw2g = assCreate();
  (void) YInit (argcptr, argv) ;
  signal (SIGINT,(GraphFunc) intProc) ;
}

void graphDevFinish(void)
{
  wcommFinish() ;
  XtDestroyWidget(root_widget) ;
}

/********** creation ************/

/* comment for when I interface into the graph package:
   I could use the instance name as designated by the
   type--a type2string or something */

static BOOL sneakyTransient = FALSE ;

static Dev YCreate (Graph_ graph, int x, int y, int w, int h)
/*
  YCreate creates a popup shell and puts inside of it a top region and a
  view port with a simple widget inside for me to draw upon.  */
{
  int i;
  Dev dev;
  Widget entry ;

  dev = (Dev) messalloc (DEV_SIZE)  ;

  i=0;
  XtSetArg(args[i], XtNmappedWhenManaged, True); i++;
  XtSetArg(args[i], XtNtitle, graph->name); i++;
  XtSetArg(args[i], XtNiconName, graph->name); i++;
  XtSetArg(args[i], XtNx, x ); i++;
  XtSetArg(args[i], XtNy, y ); i++;

  dev->popup = XtCreatePopupShell(typeName[graph->type],
	               sneakyTransient ? overrideShellWidgetClass
				       : topLevelShellWidgetClass,
				  root_widget, args, i);

  XtAddCallback(dev->popup, 
		XtNdestroyCallback, destroyCallback, "hi folks, I'm dead");

/*
  i=0;
  largeBox = XtCreateManagedWidget("largeBox", formWidgetClass,
				   dev->popup, args, i);
  i=0;				
  XtSetArg(args[i], XtNheight, 10 ); i++;
  XtSetArg(args[i], XtNwidth, w ); i++;
  dev->upper = XtCreateManagedWidget("top", coreWidgetClass,
					       largeBox, args, i);
*/  
  i = 0;
/*  XtSetArg(args[i], XtNfromVert, dev->upper); i++; */
  switch (graph->type)
    {
    case PLAIN: case TEXT_FIT:
      break ;
    case TEXT_SCROLL:
      XtSetArg(args[i], XtNallowVert, TRUE); i++ ;
      break ;
    case MAP_SCROLL:
      XtSetArg(args[i], XtNallowHoriz, TRUE); i++ ;
      break ;
    }
  dev->viewport = XtCreateManagedWidget("viewport", 
					viewportWidgetClass,
					dev->popup, args, i);
  i =0;
  XtSetArg(args[i], XtNheight, h ); i++;
  XtSetArg(args[i], XtNwidth, w ); i++;
  XtSetArg(args[i], XtNtranslations,
	   XtParseTranslationTable(actionTrans) ); i++;
  dev->simple = XtCreateManagedWidget("simple", coreWidgetClass,
				      dev->viewport, args, i);

  dev->menuWidget = XtCreatePopupShell ("menu", simpleMenuWidgetClass,
					dev->simple, NULL, 0) ;
  entry = XtCreateManagedWidget ("Quit", smeBSBObjectClass,
				 dev->menuWidget, NULL, 0) ;
  XtAddCallback (entry, XtNcallback, 
		 defaultMenuProc, (XtPointer) graph ) ;

  return dev ;
}

static void YRealizeWidget(Widget popup)
{
  XtRealizeWidget(popup);
  XtPopup(popup, XtGrabNone);
}


void graphDevCreate (Graph_ graph, float x, float y, float w, float h)
{
  int ix = x*screenx ;
  int iy = y*screeny ;
  int iw = w*screenx ;
  int ih = h*screeny ;

  graph->isClear = TRUE; /* no box drawing */

  graph->dev = YCreate (graph, ix, iy, iw, ih) ;
  graph->dev->isExposed = FALSE ;

  assInsert(assw2g, graph->dev->popup, graph);
  assInsert(assw2g, graph->dev->simple, graph);
  assInsert(assw2g, graph->dev->viewport, graph);

  graph->w = iw;
  graph->h = ih;

  YRealizeWidget (graph->dev->popup) ;
  fakeResize (graph) ;
}  

void graphDevDestroy (void)
{
  Dev dev = gDev ;
  
  if (assRemove (assw2g,dev->popup)) /* i.e. not looping via destroyCallback */
    XtDestroyWidget (dev->popup) ;
  assRemove (assw2g,dev->simple) ;
  assRemove (assw2g,dev->viewport) ;
  assRemove (assw2g,dev->menuWidget) ;

  gActive->dev = gDev = 0;
  messfree(dev);
}

void graphMapBounds (float ux, float uy, float uw, float uh, float aspect)
{
  if (gActive->type != MAP_SCROLL)
    messcrash ("MapBounds called on invalid graph type %d",gActive->type) ;

  gActive->ux = ux ;
  gActive->uy = uy ;
  gActive->uw = uw ;
  gActive->uh = uh ;
  gActive->aspect = aspect ;

  if (gDev)		/* reset width */
    { int i = 0 ;
      gActive->w = gActive->h * gActive->uw / 
		      (gActive->uh * gActive->aspect * gPixAspect) ;
      XtSetArg (args[i], XtNwidth, gActive->w ); i++;
      XtSetValues (gDev->simple, args, i) ;
      graphFacMake () ;
    }
}

void graphTextBounds (int nx, int ny)
{
  if (gActive->type != TEXT_SCROLL)
    messcrash ("textBounds called on invalid graph type %d",gActive->type) ;

  gActive->uw = nx ;
  gActive->uh = ny ;

  if (gDev)		/* reset height */
    { int i = 0 ;
      gActive->h = ny * gActive->yFac ;
      XtSetArg (args[i], XtNheight, gActive->h ); i++;
      XtSetValues (gDev->simple, args, i) ;
    }
}

/*******************************/

static Widget rootPopup = 0 ;
static Widget blockPopup = 0 ;
static int blockReturn = 0 ;

void destroyCallback (Widget w, XtPointer client_data, XtPointer call_data)
{
  if (activeSet(w))
    { assRemove (assw2g,gDev->popup) ;
      gDev->popup = 0 ;
      graphDestroy () ;
    }

  if (w == rootPopup)
    { rootPopup = 0 ;
      blockPopup = 0 ;
    }
  if (w == blockPopup)
    blockPopup = 0 ;
}

static void defaultMenuProc(Widget w, XtPointer graph, XtPointer call_data)
{
  graphDestroy () ;
}

/*****************************/

void graphStart (Graph gId)
{ Graph_ g ;
  XEvent ev ;

  if(!graphActivate(gId))
    messcrash("graphStart cannot activate its own graph") ;

  g = gActive ;
  rootPopup = g->dev->popup ;

  wcommInit (XtDisplay(gDev->simple), XtWindow(gDev->simple)) ;

  while (rootPopup)	/* destroyCallback sets to 0 when killed */
    {
      XtAppNextEvent (app_con, &ev) ;
      XtDispatchEvent (&ev) ;
    }
}

/******* blocking loop *******/

int graphBlock (void)
{
  XEvent ev ;

  if (!gDev || blockPopup)
    return 0 ;

  blockPopup = gDev->popup ;
  blockReturn = 0 ;
  while (blockPopup) /* leave if killed or graphUnBlock() called  */
    {
      XtAppNextEvent (app_con, &ev) ;
      XtDispatchEvent (&ev) ;
    }

  return blockReturn ;
}

void graphUnBlock (int retval)
{
  if (blockPopup != gDev->popup)
    messcrash ("graphUnblock called on the wrong graph") ;

  blockReturn = retval ;
  blockPopup = 0 ;
}

/*****************************************************************/

void graphGoto (float x, float y)  /* tries to center x,y in visible region */
{ 
  Widget sbar ;
  float shown ;

  if (!gDev)
    return ;
  if (sbar = XtNameToWidget (gDev->viewport,"horizontal"))
    {
      XtVaGetValues (sbar, XtNshown, &shown, NULL) ;

      x = (x - gActive->ux)/gActive->uw - 0.5*shown ;
      if (x < 0) x = 0 ;
      if (x > 1 - shown) x = 1 - shown ;

      shown = x ;
      XtCallCallbacks (sbar, XtNjumpProc, &shown) ;
    }
  if (sbar = XtNameToWidget (gDev->viewport,"vertical"))
    {
      XtVaGetValues (sbar, XtNshown, &shown, NULL) ;

      y = (y - gActive->uy)/gActive->uh - 0.5*shown ;
      if (y < 0) y = 0 ;
      if (y > 1 - shown) y = 1 - shown ;

      shown = y ;
      XtCallCallbacks (sbar, XtNjumpProc, &shown) ;
    }
}

void graphEvent (int action, float x, float y)
/* sends an event over the "wire" to active window
   currently can only handle printable ascii events
   and mouse events */
{
  XEvent *event ;
  Widget w ;
  Status s;
  static char buf[2] ;
  long event_mask ;
  int tmp ;

  
  if (!gDev)
    return ;
  w = gDev->simple ;

  if(action < 0 || action > 127) {
    messcrash ("graphEvent() can only handle actions between 0 and \
                127\n");
    return ;
  }
  
  if (action <= RIGHT_UP) /* means a mouse event */ {
    if (action % 3 == 1 ) {
      fprintf (stderr, "Sorry, no drag events yet \n") ;
      return ;
    }
    /* Button up or down */
    event = (XEvent*) messalloc(sizeof (XButtonEvent) ) ;
    event->xbutton.type = ((tmp = action % 3) == 2 ? ButtonRelease :
			   ButtonPress);
    event_mask = ( tmp == 2 ? ButtonReleaseMask :
		  ButtonPressMask ) ;
    event->xbutton.button = ((tmp =  action / 3 )== 0 ? Button1 :
			     (tmp == 1 ? Button2 : Button3 )
			     ) ;
    fprintf(stderr, "Button# = %d , action# = %d \n", tmp+1, action) ;
    event->xbutton.x = uToXabs(x) ;
    event->xbutton.y = uToYabs(y) ;
  }

  /* keyboard events */
  if(action > RIGHT_UP ) {
    
    event = (XEvent*) messalloc(sizeof (XKeyEvent)) ;
    event->xkey.type = KeyPress ;
    
    buf[0] = action ;
    buf[1] = 0 ;
    
    event->xkey.keycode = XKeysymToKeycode(XtDisplay(w),XStringToKeysym(buf)) ;
    event_mask = KeyPressMask ;
    event->xkey.x = uToXabs(x) ;
    event->xkey.y = uToYabs(y) ;
    
  }

  /* information common to all events */

  event->xany.send_event = TRUE ;
  event->xany.display = XtDisplay(w) ;
  event->xany.window = XtWindow(gDev->simple) ;

  s = XSendEvent (XtDisplay(w), XtWindow(w), 
		 FALSE, KeyPressMask,		       
		 event) ;
  /*for debugging*/
  /*fprintf(stderr, "ID %d, status %d, event =%d\n", XtDisplay(w),s, (int ) event);*/
  messfree(event) ;
}

void graphRetitle (char *title)
{
  if (!gDev) return ;
  
  XtSetArg(args[0], XtNtitle, title) ;
  XtSetValues(gDev->popup, args, 1) ;
}

/************/

void graphPop (void)
{
  if (gDev)
    { 
/*
      if (getenv("ACEDB_SCREEN_FIX"))
	XIconifyWindow (XtDisplay(gDev->popup),XtWindow(gDev->popup), 0) ;

      XtVaSetValues (gDev->popup, XtNiconic, FALSE, NULL) ;
*/
      XRaiseWindow (XtDisplay(gDev->popup),XtWindow(gDev->popup)) ;
      XMapWindow(XtDisplay(gDev->popup),XtWindow(gDev->popup)) ;
    }
}

/**********/

static void killMessage (Widget w, XtPointer client_data, XtPointer call_data)
{
  Dev dev = (Dev) client_data ;

  XtDestroyWidget (dev->message) ;
  dev->message = 0 ;
}

static void messageDestroy (Widget w, XtPointer client_data, XtPointer call_data)
{
  Graph_ g = (Graph_) client_data ;
  Graph old = graphActive () ;

  if (graphExists (g->id) && 
      g->func[MESSAGE_DESTROY] && 
      graphActivate (g->id))
    { (*g->func[MESSAGE_DESTROY])() ;
      graphActivate (old) ;
    }
}

void graphMessage (char *text)
{
  Widget dialog, button ;
  Position x,y ;
  int n ;

  if (!gDev || gDev->message)
    return ;

  XtTranslateCoords (gDev->viewport, 
		     (Position) 25, (Position) 20,
		     &x, &y) ;
  n = 0;
  XtSetArg (args[n], XtNx, x) ;	n++ ;
  XtSetArg (args[n], XtNy, y) ;	n++ ;
  gDev->message = XtCreatePopupShell("message", transientShellWidgetClass,
				     gDev->popup, args, n) ;
  dialog = XtCreateManagedWidget ("textmessage", dialogWidgetClass,
				   gDev->message, NULL,0) ;
  XtVaSetValues (dialog, XtNlabel, uBrokenText (text, 40), NULL) ;
  button = XtCreateManagedWidget ("Remove", commandWidgetClass, 
				  dialog, args, 0) ;
  XtAddCallback (button, XtNcallback, killMessage, (XtPointer)gDev) ;
  XtAddCallback (gDev->message, XtNdestroyCallback, 
		 messageDestroy, (XtPointer)gActive) ;
  XtPopup (gDev->message,XtGrabNone) ;
}

void graphUnMessage (void) 
{ 
  if (gDev && gDev->message)
    { XtDestroyWidget (gDev->message) ;
      gDev->message = 0 ;
    }
}

/*****************************************************************/

/* callbacks, associators and all that */

/* The following take care of the callback and action routine
 * interface to the graph package
 * 1. install/interface the graphs callbacks on the widgets actions
 */

/****/

static BOOL activeSet(Widget w)
{
  Graph_ g ;

  if (assFind(assw2g, w, &g))
    { graphActivate (g->id) ;
      return TRUE ;
    }
  else
    return FALSE ;
}

/*****************************************************************/

static Bool exposeCheck (Display* display, XEvent* ev0, char* arg)
{
  Window win = (Window) arg ;
  XAnyEvent* ev = (XAnyEvent*)ev0 ;

  return (ev->type == Expose) && (ev->window == win) ;
}

static void exposeCall (Widget w, XEvent* ev0, String *params, Cardinal *num_params)
{
  XEvent  ev1 ;
  XExposeEvent* exp0 = (XExposeEvent*)ev0 ;
  XExposeEvent* exp1 = (XExposeEvent*)&ev1 ;
  int xmin,xmax,ymin,ymax ;
  Graph gsave = gActive->id ;

  xmin = exp0->x ; xmax = xmin + exp0->width ;
  ymin = exp0->y ; ymax = ymin + exp0->height ;

  while (XCheckIfEvent (exp0->display, 
			&ev1, exposeCheck, 
			(char*)(exp0->window)))
    { if (xmin > exp1->x) 
	xmin = exp1->x ;
      if (xmax < exp1->x + exp1->width) 
	xmax = exp1->x + exp1->width ;
      if (ymin > exp1->y) 
	ymin = exp1->y ;
      if (ymax < exp1->y + exp1->height) 
	ymax = exp1->y + exp1->height ;
    }

  if (activeSet (w))
    { graphClipDraw (xmin,ymin,xmax,ymax) ;
      graphActivate (gsave) ;
    }
}

static void resizeCall (Widget w, XEvent* ev, String *params, Cardinal *num_params)
{
  int width = ev->xconfigure.width ;
  int height = ev->xconfigure.height ;
  int i, isRedraw ;
  Widget sbar ;

  if (!activeSet (w))
    return ;
  
  switch (gActive->type)
    {
    case PLAIN :
      isRedraw = (width <= gActive->w && height <= gActive->h &&
		  (width < gActive->w || height < gActive->h)) ;
      gActive->w = width ;
      gActive->h = height ;
      graphFacMake () ;
      if (isRedraw)
	graphRedraw () ;
      break ;
    case TEXT_FIT :
      gActive->w = width ;
      gActive->h = height ;
      gActive->uw = width / gActive->xFac ;
      gActive->uh = height / gActive->yFac ;
      break ;
    case TEXT_SCROLL :
      gActive->w = width ;
      i=0;				
      XtSetArg (args[i], XtNheight, gActive->h ); i++;
      XtSetValues (w, args, i) ;
      break ;
    case MAP_SCROLL :
      if (height == gActive->h)	/* prevents recursion */
	break ;
      if (sbar = XtNameToWidget (gDev->viewport,"horizontal"))
	{ float xpos,shown ;

	  XtVaGetValues (sbar, 
			 XtNtopOfThumb, &xpos, 
			 XtNshown, &shown,
			 NULL) ;
	  xpos *= height / (float) gActive->h ;
	  if (xpos > 1-shown)
	    xpos = 1-shown ;
	  XtCallCallbacks (sbar, XtNjumpProc, &xpos) ;
	}
      gActive->h = height ;
      gActive->w = gActive->h * gActive->uw / 
		      (gActive->uh * gActive->aspect * gPixAspect) ;
      i=0;				
      XtSetArg (args[i], XtNwidth, gActive->w ); i++;
      XtSetValues (w, args, i) ;
      graphFacMake () ;
      break ;
    }

  if (gActive->func[RESIZE])
    (*gActive->func[RESIZE])() ;
}

static void fakeResize (Graph_ g)
{
  XConfigureEvent ev ;
  ev.height = g->h ;
  ev.width = g->w ;
  resizeCall (g->dev->simple,(XEvent*)(&ev),0,0) ;
}

static void mouseCall (Widget w, XEvent* ev, String *params, Cardinal *num_params)
{
  int myEv ;

  if (!activeSet (w))
    return ;

  if (blockPopup && gDev->popup != blockPopup)
    return ;

  if (!sscanf (*params,"%d",&myEv) || myEv < 1 || myEv >= 9)
    messout ("Bad mouse event %s",*params) ;
  else if (gActive->func[myEv] )
    (*gActive->func[myEv])(XtoUabs(ev->xbutton.x), YtoUabs(ev->xbutton.y) ); 
}

static void pickCall (Widget w, XEvent* ev, String *params, Cardinal *num_params)
{
  if (!activeSet (w))
    return ;

  if (blockPopup && gDev->popup != blockPopup)
    return ;

  gLeftDown (XtoUabs(ev->xbutton.x), YtoUabs(ev->xbutton.y)); 
}

static void keyboardCall (Widget w, XEvent* ev, String *params, Cardinal *num_params)
{
  KeySym keysym ;
  char buffer[2] ;
  int kval ;
  extern void help(void) ;

  if (!activeSet (w))
    return ;

  if (blockPopup && gDev->popup != blockPopup)
    return ;

  if (XLookupString((XKeyEvent*) ev, buffer, 2, &keysym, NULL) )
    kval = buffer[0] ;
  else
    switch (keysym)
      {
      case XK_F1: 
      case XK_F10: 
      case 65386:
	help () ;
	return ;
      case XK_F2:
      case XK_F9:
/*    case 65478: duplicate */
	graphPrint () ;  
	return ;

/* hard coding keys r1 r2 r3 r5 as arrow keys for the sun under openlook */
      case XK_Delete: kval = DELETE_KEY ; break ;
      case XK_BackSpace: kval = BACKSPACE_KEY ; break ;
      case 65491:
      case XK_Up: kval = UP_KEY ; break ;
      case 65494:
      case XK_Down: kval = DOWN_KEY ; break ;
      case 65490:
      case XK_Left: kval = LEFT_KEY ; break ;
      case 65492:
      case XK_Right: kval = RIGHT_KEY ; break ;
      case XK_Home: kval = HOME_KEY ; break ;
      case XK_End: kval = END_KEY ; break ;
      case 65379: kval = INSERT_KEY ; break ;
      default: 
/*
 	if (keysym < 65500)
	  printf ("Got unmapped keysym %d (%d)\n",
		  keysym, XtGetActionKeysym (ev, NULL)) ;
*/
	kval = 0 ; 
        break ;
      }
/*
  printf ("buffer, keysym, kval are: %d %d %d\n", 
	  buffer[0],keysym,kval) ;
*/
  if (kval && gActive->func[KEYBOARD])
    (*gActive->func[KEYBOARD])(kval) ;
}

static void propertyCall (Widget w, XEvent* ev, String *params, Cardinal *num_params)
{
  WPropMessage (ev) ;
}

/********** end of file **********/
