/*  File: graphxlib.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: X Low level stuff for graph package.
		Based on code from Chris Lee.
 * Exported functions:  graphDevActivate, 
 			graphBoxDraw, graphRedraw, graphClipDraw
			graphXorLine, graphXorBox,
			gFontInfo

 * HISTORY:
 * Last edited: Apr 22 20:05 1992 (mieg)
 * * Dec  4 13:55 1991 (mieg): ACEDB_COLOR to force isMono = FALSE
 * * Oct 14 15:56 1991 (rd): added clipping of all vectors, except
	only test the centre (top left) of circles, points, text.
 *-------------------------------------------------------------------
 */

#include "regular.h"
#include "array.h"
#include "graphxt_.h"
#include<X11/Xaw/Scrollbar.h>
#ifdef NEXT
  #include "next.h"
#endif

static void fontInit (Display* display) ;
static void colorInit (Display *display, Screen *screen) ;
static void stippleInit (Window window) ;

/******* statics for drawing - set in graphDevActivate() *********/

static Display	*display ;
static Window   window ;
static GC	gc, gcInvert, gcStipple ;

#define DWG display,window,gc  /* used in all drawing calls */
#define DG  display,gc         /* used in all GC change calls */

static unsigned long gcValuesMask ;
static XGCValues gcValues;

static BOOL isMono ;

/* stuff from Theron Friedman (tzf@cs.arizona.edu) to prevent
   program quitting in Openlook/MacX
*/
#include <X11/Xatom.h>
static Atom	exitHintAtom, myExitAtom ;

/*******************************************************/

static void xlibInit (void)
{
  int screen_num = DefaultScreen(display) ;
  int mydepth, myclass ;

  gc = XCreateGC (display, RootWindow(display,screen_num), 0, 0) ;
  XSetLineAttributes (display, gc, 0, LineSolid, CapRound, JoinRound) ;
  gcInvert = XCreateGC (display, RootWindow(display,screen_num), 0, 0) ;
  XSetFunction (display, gcInvert, GXinvert) ;
  fontInit (display) ;
  colorInit (display, XtScreen (root_widget)) ;

  mydepth = DefaultDepth(display, screen_num) ;
#if defined (__cplusplus) || defined(c_plusplus)
  myclass = DefaultVisual (display, screen_num)->c_class ;
#else
  myclass = DefaultVisual (display, screen_num)->class ;
#endif
  isMono = (getenv ("MONO") != 0 || mydepth == 1 || myclass < 2) ;
  if(getenv("ACEDB_COLOR"))
    isMono = FALSE ;
  if (isMono)
    { stippleInit (RootWindow (display, screen_num)) ;
      gcStipple = XCreateGC (display, RootWindow(display,screen_num), 0, 0) ;
      XSetFillStyle (display, gcStipple, FillOpaqueStippled) ;
    }
	/* tzf quit-safety stuff */
  exitHintAtom = XInternAtom (display, "WM_PROTOCOLS", FALSE) ;
  myExitAtom = XInternAtom (display, "WM_DELETE_WINDOW", FALSE) ;
}

void graphDevActivate (int show) 
{
  display = XtDisplay (gDev->popup) ;
  window = XtWindow (gDev->simple) ;
  if (!gc)
    xlibInit () ;
  XChangeProperty (display, XtWindow(gDev->popup), 
		   exitHintAtom, XA_ATOM, 32, 
		   PropModeReplace, (unsigned char*)&myExitAtom, 1) ;
}

/********* next the fonts - needs work to clean up *********/

/* font policy: use simple fonts so that they should all be there
   load them all in at beginning
   fontInit assumes default font is first in list
   note: this is a display specific object - so fontInit needs the
     display to have been opened.
*/

enum FONTTYPE {F8x13, F5x8, F6x9, F6x10, F6x12, F9x15, NUM_FONTS} ;
static char *fontName[] = 
	      {"8x13","5x8","6x9","6x10","6x12","9x15"} ;
#define Fdefault F8x13
static XFontStruct *fns[NUM_FONTS] ;
static int fontdy ;  /* is simply the ascent used to offset base
			line to make drawtext start at upper left */

static void fontInit (Display* display)
{
  int i ;
  static int isDone = FALSE ;
  extern float textAspect ;

  if (isDone)
    return ;

  if (!(fns[0] = XLoadQueryFont (display, fontName[0])))
    messcrash ("Can't load default font %s",fontName[0]) ;
  textAspect = fns[0]->max_bounds.width / 
    ((float)fns[0]->ascent + fns[0]->descent) ;

  for (i = 1 ; i < NUM_FONTS ; ++i)
    if (!(fns[i] = XLoadQueryFont (display, fontName[i])))
      { fprintf(stderr,"Font %s does not exist - using default",
		 fontName[i]) ;
	fns[i] = fns[0] ;
      }

  isDone = TRUE ;
  return ;
}

static int height2font (int height)
{
  switch (height)
    {
    case 0:			/* default font */
      return F8x13;
    case 1: case 2: case 3: case 4: case 5: case 6: case 7: case 8: 
      return F5x8;
    case 9: case 10:
      return F6x10;
    case 11: case 12:
      return F6x12;
    case 13: case 14:
      return F8x13;
    default:
      return F9x15;
    }
}

BOOL gFontInfo (int height, int* w, int* h)
	/* note that this depends on an integer height, and so
	   is only reliable in the long term for height 0 */
{
  XFontStruct* fnt = fns[height2font (height)] ;

  if (!fnt)
    { fontInit (XtDisplay (root_widget)) ;
      if (!(fnt = fns[height2font (height)]))
	return FALSE ;
    }

  *w = fnt->max_bounds.width;
  *h = fnt->ascent + fnt->descent;
  return TRUE ;
}

static void fontSet (int height)
{
  XFontStruct* fnt = fns[height2font (height)] ;

  if (!fnt)
    fnt = fns[height2font(0)] ;

  XSetFont (DG, fnt->fid) ;
  fontdy = fnt->ascent ;
}

/********** next color control - screen specific ***********/

static unsigned long colorInd[NUM_TRUECOLORS] ;
static char* colorName[] = 
  {"white","black","lightgray","dimgray",
   "red","green","blue",
   "yellow","cyan","magenta",
   "pink","yellowgreen","skyblue",
   "violet","darkgreen","cadetblue"} ;

static void colorInit (Display *display, Screen *screen)
{
  int i ;
  XColor color_used, color_named ;
  Colormap cmap = DefaultColormapOfScreen(screen) ;

  for (i = NUM_TRUECOLORS ; i-- ;)
    if (XAllocNamedColor (display, cmap, colorName[i],
			  &color_used, &color_named))
      colorInd[i] = color_used.pixel ;
    else  /* messout here creates a crash */
      { printf("Can't match color %s",colorName[i]) ;
	colorInd[i] = WhitePixelOfScreen(screen) ;
      }
}

/************ stipples for monochrome systems ***********/

static unsigned short whiteMask[] = {0xffff,0xffff,0xffff,0xffff} ;
static unsigned short lightMask[] = {0x7f7f,0xfbfb,0xdfdf,0xfefe,
				     0xf7f7,0xbfbf,0xfdfd,0xefef} ;
static unsigned short mixMask[]   = {0x7777,0xdddd,0xbbbb,0xeeee} ;
static unsigned short halfMask[]  = {0xaaaa,0x5555,0xaaaa,0x5555} ;
static unsigned short darkMask[]  = {0x8888,0x2222,0x4444,0x1111} ;
static unsigned short blackMask[] = {0x0000,0x0000,0x0000,0x0000} ;
static Pixmap  whiteStipple, lightStipple, mixStipple,
  	       halfStipple,  darkStipple,  blackStipple ;
static Pixmap* stipple[] =
  { &whiteStipple, &blackStipple, &lightStipple, &darkStipple,
    &halfStipple,  &halfStipple,  &halfStipple,
    &mixStipple,   &mixStipple,   &mixStipple,
    &lightStipple, &lightStipple, &lightStipple,
    &darkStipple,  &darkStipple,  &darkStipple
  } ;

static void stippleInit (Window window)
{
  whiteStipple = XCreateBitmapFromData (display, window,
		   (char*) whiteMask, 16, 4) ;
  lightStipple = XCreateBitmapFromData (display, window,
		   (char*) lightMask, 16,8) ;
  mixStipple   = XCreateBitmapFromData (display, window,
		   (char*) mixMask, 16, 4) ;
  halfStipple  = XCreateBitmapFromData (display, window,
		   (char*) halfMask, 16, 4) ;
  darkStipple  = XCreateBitmapFromData (display, window,
		   (char*) darkMask, 16, 4) ;
  blackStipple = XCreateBitmapFromData (display, window,
		   (char*) blackMask, 16, 4) ;
						       
  if (!whiteStipple || !lightStipple || !mixStipple || 
      !halfStipple || !darkStipple || !blackStipple)
    messcrash ("Could not make stipples for monochrome X graphics") ;
}

/******* now draw boxes **********/

static int psize, psize2 ;
static int clipx1, clipx2, clipy1, clipy2 ;

#define xclip(z)  if (z < clipx1) z = clipx1 ; \
		  else if (z > clipx2) z = clipx2
#define yclip(z)  if (z < clipy1) z = clipy1 ; \
		  else if (z > clipy2) z = clipy2
#define xsafe(z)  if (z < -30000) z = -30000 ; \
		  else if (z > 30000) z = 30000
#define ysafe(z)  if (z < -30000) z = -30000 ; \
		  else if (z > 30000) z = 30000

static void drawBox (Box box)
{
  float  t ;
  int    x1,x2,y1,y2,r;
  char   *text ;
  int    action ;
  unsigned long color ;
  BOOL   isDraw = TRUE ;

  if (box->x1 > box->x2 || box->y1 > box->y2)
    isDraw = FALSE ;

  x1 = uToXabs(box->x1) ;
  y1 = uToYabs(box->y1) ;
  x2 = uToXabs(box->x2) ;
  y2 = uToYabs(box->y2) ;

  if (x1 > clipx2 || x2 < clipx1 || y1 > clipy2 || y2 < clipy1)
    isDraw = FALSE ;

  if (!isDraw)
    { int nDeep = 1 ;
      stackCursor (gStk,box->mark) ; /* sets position to mark */
      while (!stackAtEnd (gStk))
	switch (action = stackNext (gStk,int))
	  {
	  case BOX_END :
	    if (!--nDeep)
	      return ;                        /* exit point */
	    break ;
	  case BOX_START :
	    r = stackNext (gStk, int) ;
	    ++nDeep ;
	    break ;
	  case COLOR :
	    r = stackNext (gStk,int) ; 
	    break ;
	  case LINE_WIDTH : case TEXT_HEIGHT : case POINT_SIZE :
	    t = stackNext (gStk,float) ;
	    break ;
	  case LINE : case RECTANGLE : case FILL_RECTANGLE :
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    break ;
	  case CIRCLE : case POINT : case TEXT : case TEXT_PTR :
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    switch (action)
	      {
	      case CIRCLE :
		t = stackNext (gStk,float) ;
		break ;
	      case POINT :
		break ;
	      case TEXT :
		text = stackNextText (gStk) ;
		break ;
	      case TEXT_PTR :
		text = stackNext (gStk,char*) ;
		break ;
	      }
	    break ;
	    default :
	      messout ("Invalid draw action %d received",action) ;
	  }
      return ;
    }

  if (box->bcol != TRANSPARENT)
    { xclip(x1) ; xclip(x2) ; yclip(y1) ; yclip(y2) ;
      if (isMono)
	{ XSetStipple (display, gcStipple, *stipple[box->bcol]) ;
	  XFillRectangle (display, window, gcStipple,
			  x1, y1, (x2-x1), (y2-y1)) ;
	}
      else
	{ XSetForeground (DG, colorInd[box->bcol]) ;
	  XFillRectangle (DWG, x1, y1, (x2-x1), (y2-y1)) ;
	}
    }

  color = colorInd[box->fcol] ;
  XSetForeground (DG, color) ;
  
  stackCursor (gStk,box->mark) ; /* sets position to mark */

  while (!stackAtEnd (gStk))
    switch (action = stackNext (gStk,int))
      {
      case BOX_END :
        return ;                        /* exit point */
      case BOX_START :
        r = stackNext (gStk, int) ;
        drawBox (gBoxGet (r)) ;             /* recursion */
	XSetForeground (DG, color) ;
        break ;
      case COLOR :
        x1 = stackNext (gStk,int) ; 
	color = colorInd[x1] ;
        XSetForeground (DG, color);
        break ;
      case LINE_WIDTH :
        t = stackNext (gStk,float) ;
	gcValues.line_width = uToXrel(t) ;
	gcValuesMask = GCLineWidth; 
        XChangeGC (DG, gcValuesMask, &gcValues);
        break ;
      case TEXT_HEIGHT :
        t = stackNext (gStk,float) ; x1 = uToYrel(t) ;
        fontSet (x1) ;
        break ;
      case POINT_SIZE :
        t = stackNext (gStk,float) ;
        psize = uToXrel(t) ; psize2 = psize/2 ; 
	if (!psize2) psize2 = 1 ;
        break ;
      case LINE : case RECTANGLE : case FILL_RECTANGLE : /* changed */
        t = stackNext (gStk,float) ; x1 = uToXabs(t) ; xsafe(x1) ;
        t = stackNext (gStk,float) ; y1 = uToYabs(t) ; ysafe(y1) ;
        t = stackNext (gStk,float) ; x2 = uToXabs(t) ; xsafe(x2) ;
        t = stackNext (gStk,float) ; y2 = uToYabs(t) ; ysafe(y2) ;
	switch (action)
          {
          case LINE :
	    XDrawLine (DWG, x1, y1, x2, y2);
            break ;
          case RECTANGLE :
	    if (x2 == x1) x2 = x1+1 ;
	    if (y2 == y1) y2 = y1+1 ;
	    if (x2 < x1) { r = x1 ; x1 = x2 ; x2 = r ; }
	    if (y2 < y1) { r = y1 ; y1 = y2 ; y2 = r ; }
	    XDrawRectangle (DWG, x1, y1, (x2-x1), (y2-y1) );
            break ;
          case FILL_RECTANGLE :
      	    if (x2 == x1) x2 = x1+1 ;
	    if (x2 < x1) { r = x1 ; x1 = x2 ; x2 = r ; }
	    if (y2 == y1) y2 = y1+1 ;
	    if (y2 < y1) { r = y1 ; y1 = y2 ; y2 = r ; }
	    XFillRectangle (DWG, x1, y1, (x2-x1), (y2-y1) );
            break ;
          }
        break ;
      case CIRCLE : case POINT : case TEXT : case TEXT_PTR :
        t = stackNext (gStk,float) ; x1 = uToXabs(t) ;
	t = stackNext (gStk,float) ; y1 = uToYabs(t) ;
	isDraw = (x1 < clipx2 + 1000 && x1 > clipx1 - 1000 &&
		  y1 < clipy2 + 1000 && y1 > clipy1 - 1000) ;
        switch (action)
          {
          case CIRCLE :      /* given center x1, y1, radius r*/
            t = stackNext (gStk,float) ; r = uToXrel(t) ;
	    if (!r) r = 1 ;
	    if (isDraw) 
	      XDrawArc (DWG, x1-r, y1-r, 2*r, 2*r, 0, 64*360);
		    /* NB upper left corner and 64ths of a degree */
            break ;
          case POINT :
	    if (isDraw)
	      XFillRectangle (DWG, x1-psize2, y1-psize2, psize, psize);
            break ;
          case TEXT :
            text = stackNextText (gStk) ;
	    /* must adjust for font offset so that x1,y1 is top left */
	    if (isDraw) 
	      XDrawString(DWG, x1, y1+fontdy, text, strlen(text) ); 
            break ;
	  case TEXT_PTR :
	    text = stackNext (gStk,char*) ;
	    if (isDraw)
	      XDrawString(DWG, x1, y1+fontdy, text, strlen(text) ); 
	    break ;
          }
	break ;
      default :
	messout ("Invalid action %d received in drawBox()",action) ;
      }
}

static void setBoxDefaults (Box box)
{
  gcValues.line_width = uToXrel (box->linewidth) ;
  gcValuesMask = GCLineWidth ; 
  XChangeGC (DG, gcValuesMask, &gcValues) ;

  fontSet (uToYrel(box->textheight)) ;
  psize = uToXrel(box->pointsize) ;
  psize2 = psize/2 ;
  if (!psize2)
    psize2 = 1 ;
}

void graphBoxDraw (int k, int fcol, int bcol)
{
  Box box = gBoxGet (k) ;

  if (fcol >= 0)
    box->fcol = fcol ;
  if (bcol >= 0)
    box->bcol = bcol ;

  if (!gDev || !gDev->isExposed || gActive->isClear)
    return ;

  clipx1 = uToXabs(box->x1) ; if (clipx1 < 0) clipx1 = 0 ;
  clipx2 = uToXabs(box->x2) ; if (clipx2 > 30000) clipx2 = 30000 ;
  clipy1 = uToYabs(box->y1) ; if (clipy1 < 0) clipy1 = 0 ;
  clipy2 = uToYabs(box->y2) ; if (clipy2 > 30000) clipy2 = 0 ;
  if (clipx1 > clipx2 || clipy1 > clipy2)
    return ;

  setBoxDefaults (box) ;
  XSetClipMask (DG, None) ;
  if (isMono)
    XSetClipMask (display, gcStipple, None) ;
  drawBox (box) ;
  XFlush (display) ;
}

void graphClipDraw (int x1, int y1, int x2, int y2) /* expose action */
{
  static XRectangle clipRect[1] ;
  Box box = gBoxGet(0) ;

  if (!gActive->stack || !gDev)
    return ;
  gDev->isExposed = TRUE ;

  setBoxDefaults (box) ;

  clipx1 = (x1 > 0) ? x1 : 0 ; 
  clipx2 = (x2 < 30000) ? x2 : 30000 ; 
  clipy1 = (y1 > 0) ? y1 : 0 ; 
  clipy2 = (y2 < 30000) ? y2 : 30000 ;
  if (clipx1 > clipx2 || clipy1 > clipy2)
    return ;
  clipRect->x = clipx1 ; clipRect->y = clipy1 ;
  clipRect->width = clipx2 - clipx1 + 1 ; 
  clipRect->height = clipy2 - clipy1 + 1 ;
  XSetClipRectangles (DG, 0, 0, clipRect, 1, Unsorted) ;
  if (isMono)
    XSetClipRectangles (display, gcStipple, 
			0, 0, clipRect, 1, Unsorted) ;
  drawBox (box) ;
  XFlush (display) ;
}

void graphRedraw (void)	/* now do this by creating expose event */
{
  unsigned short w, h ;	/* true width, height */
  Widget sbar ;
  float  pos ;
  XExposeEvent ev ;

  gActive->isClear = FALSE ;
  if (!gActive->stack || !gDev || !gDev->isExposed)
    return ;

  ev.type = Expose ;
  ev.display = XtDisplay (gDev->simple) ;
  ev.window = XtWindow (gDev->simple) ;
  ev.x = ev.y = 0 ;
  XtVaGetValues (gDev->viewport, XtNwidth, &w, XtNheight, &h, NULL) ;
  ev.width = w ; ev.height = h ;
  if (sbar = XtNameToWidget (gDev->viewport,"horizontal"))
    { XtVaGetValues (sbar, XtNtopOfThumb, &pos, NULL) ;
      ev.x = pos*gActive->w ;
    }
  if (sbar = XtNameToWidget (gDev->viewport,"vertical"))
    { XtVaGetValues (sbar, XtNtopOfThumb, &pos, NULL) ;
      ev.y = pos*gActive->h ;
    }
  XSendEvent (ev.display, ev.window, True, ExposureMask, (XEvent*)&ev) ;
  XFlush (display) ;
}

void graphWhiteOut (void)
{
  static XRectangle rect[1] ;

  if (!gDev)
    return ;

  rect->x = 0 ; rect->width = 2000 ;
  rect->y = 0 ; rect->height = 30000 ;
  XSetClipRectangles (DG, 0, 0, rect, 1, Unsorted) ;
  XSetForeground (DG, colorInd[WHITE]) ;
  XFillRectangle (DWG, 0, 0, 2000, 30000) ;
  XFlush (display) ;
}

void graphXorLine (float x1, float y1, float x2, float y2)
{
  int ix1 = uToXabs(x1) ;
  int iy1 = uToYabs(y1) ;
  int ix2 = uToXabs(x2) ;
  int iy2 = uToYabs(y2) ;
  
  if (gDev)
    XDrawLine (display, window, gcInvert, ix1, iy1, ix2, iy2);
}

void graphXorBox(int k, float x, float y)
{
  Box box = gBoxGet (k) ;
  int x1 = uToXabs(x) ;
  int y1 = uToYabs(y) ;
  int x2 = x1 + uToXrel(box->x2 - box->x1) ;
  int y2 = y1 + uToYrel(box->y2 - box->y1) ;

  if (x2 == x1 || y2 == y1)
    graphXorLine (x, y, x + (x2-x1), y + (y2-y1)) ;
  else if (gDev)
    XDrawRectangle (display, window, gcInvert, 
		    x1, y1, (x2-x1), (y2-y1));
}

/******************/
/******************/
