/*  File: graphsun.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: sunwindows dependent code for graph package
 * Exported functions: (many)
 * HISTORY:
 * Last edited: Apr  3 15:23 1992 (mieg)
 * Created: Thu Oct 10 14:42:23 1991 (rd)
 *-------------------------------------------------------------------
 */

#include "regular.h"
#include "array.h"
#include "graphsun_.h"
#include "key.h"
#include <suntool/scrollbar.h>
#include <suntool/icon.h>
#include <suntool/panel.h>

float  gPixAspect = 1.0 ;

#define CMS_SIZE 16
static unsigned char   red[] = {255,0,192,128,255,0,0,255,0,255,
				  255,200,200,100,0,0} ;
static unsigned char green[] = {255,0,192,128,0,255,0,255,255,0,
				  200,255,200,0,100,0} ;
static unsigned char  blue[] = {255,0,192,128,0,0,255,0,255,255,
				  200,200,255,0,0,100} ;

static int isMono ;
static float screenx = 862 ; /* 1150 / 1.3 */
static float screeny = 900 ;
static char *defFont = "/usr/lib/fonts/fixedwidthfonts/screen.r.11" ;
static Associator assf2g ;
static Associator assc2g ;

/************ forward declarations of static functions ************/

static Notify_value destroyFrame (Frame frame, Destroy_status status) ;
static void resizeProc (Canvas canvas, int width, int height) ;
static void frameEventProc (Window win, Event *ev, caddr_t arg) ;
static void canvasEventProc (Window win, Event *ev, caddr_t arg) ;
static void typeCreate (Graph_ g) ;

/***********************************************************/

static Notify_value graphicsIntProc (int *me, int signal,
                                     Notify_signal_mode when)
        /* CTRL-C handler */
{ 
  if (++intFlag > 1  &&  graphQuery ("Do you want to abort?"))
    messcrash ("User initiated abort") ;
  return (NOTIFY_DONE);
}

void graphDevInit (int *argcptr, char **argv)
{
  static int client ;   /* just to create a unique address */

  notify_set_signal_func (&client, graphicsIntProc, SIGINT, NOTIFY_ASYNC);

  if (!assf2g)
    { assf2g = assCreate () ;
      assc2g = assCreate () ;
    }
  isMono = (getenv ("MONO") != 0) ;
}

void graphStart (Graph gId)
{ Graph_ g ;
  static int isLooping = FALSE ;

  if (isLooping)
    messcrash ("Attempt to call graphStart when already started") ;

  if(!graphActivate(gId))
    messcrash("graphStart cannot activate its own graph") ;

  g = gActive ;

  if (!g || !g->dev)
    messcrash ("called graphStart without a functioning home window") ;
  window_main_loop (g->dev->frame) ;
  isLooping = TRUE ;
}

void graphDevFinish (void)
{
    /* can't destroy assf2g, assc2g because of asynchronous window kill */
}

/*************************************/

void graphDevActivate (int flag)
{ window_set (gDev->frame, FRAME_EMBOLDEN_LABEL, flag, 0) ;
}

/********************************************/
/*
Jean tried this to create an icon but for some reason mpr_static does
not properly compile with gcc and gives parse error before _data
_data is in the macro which is in include/pixrect/memvar.h

25/11/90 RMD also had this poblem with mono bit patterns.  They are put in
a separate source file that is compiled with cc.  The problem is that some
macro substitutions rely on name concatenation, which is not ANSI standard.

static short icon_image[] = {
#include "worm_icon"
};
mpr_static(wormi, 64, 64, 1, icon_image);

graphDevIcon ()
{ Icon icon = icon_create (ICON_IMAGE, &wormi, 0) ;
  window_set (dev->frame, FRAME_ICON, icon, 0) ;
}
*/

/********************************************/

void (*gCreateRoutine)(Graph g) = 0 ;	/* interpolator */

void graphDevCreate (Graph_ graph, float x, float y, float w, float h)
{
  int   error ;
  int ix = x*screenx ;
  int iy = y*screeny ;
  int iw = w*screenx ;
  int ih = h*screeny ;
  Dev dev ;

  dev = graph->dev = (Dev) messalloc (DEV_SIZE) ;

  graph->isClear = TRUE; /* no box drawing */
  dev->frame = window_create (0, FRAME, FRAME_LABEL, graph->name, 
			      WIN_CONSUME_PICK_EVENTS, LOC_WINENTER, 0,
			      WIN_EVENT_PROC, frameEventProc,
			      0);
  if (error = window_set (dev->frame,
                          WIN_X,          ix,     WIN_Y,          iy,
                          WIN_WIDTH,      iw,     WIN_HEIGHT,     ih,
                          WIN_SHOW, TRUE,
                          0))
    messprintf ("Fault %d setting SunView frame",error) ;  /* error = 1! */
  notify_interpose_destroy_func (dev->frame, destroyFrame) ;
  assInsert (assf2g,dev->frame,graph) ;

  if (gCreateRoutine)
    (*gCreateRoutine)(graph->id) ;

  dev->canvas = window_create (dev->frame, CANVAS,
              CANVAS_RETAINED, FALSE,
              CANVAS_RESIZE_PROC, resizeProc,
              WIN_CONSUME_PICK_EVENTS, WIN_MOUSE_BUTTONS, LOC_DRAG,
		    WIN_RESIZE, WIN_REPAINT,
                    LOC_WINENTER, LOC_RGNENTER, WIN_IN_TRANSIT_EVENTS, 0,
              WIN_IGNORE_PICK_EVENTS, LOC_MOVE, LOC_WINEXIT, LOC_RGNEXIT, 0,
              WIN_CONSUME_KBD_EVENTS, WIN_ASCII_EVENTS, 
		       WIN_RIGHT_KEYS, KEY_TOP(1), KEY_TOP(2), 0,
              WIN_EVENT_PROC, canvasEventProc,
	      0) ;
  assInsert (assc2g,dev->canvas,graph) ;
  dev->pixwin = canvas_pixwin (dev->canvas);
  if (!isMono)
    { pw_setcmsname (dev->pixwin, "wormcolours") ;
      pw_putcolormap(dev->pixwin,0,CMS_SIZE,red,green,blue) ;
    }
  graph->w = (int)window_get (dev->canvas,WIN_WIDTH) ;
  graph->h = (int)window_get (dev->canvas,WIN_HEIGHT) ;
  typeCreate (graph) ;
  notify_set_signal_func (dev->canvas, graphicsIntProc, SIGINT, NOTIFY_ASYNC);
  notify_dispatch () ; notify_dispatch () ;
  pw_rop (dev->pixwin,0,0,graph->w,graph->h,(PIX_SRC|PIX_COLOR(WHITE)),0,0,0) ;
}

/********************************/

void graphDevDestroy ()
{
  Dev  dev = gDev ;

  assRemove (assf2g,dev->frame) ;
  assRemove (assc2g,dev->canvas) ;
  gActive->dev = gDev = 0 ;
  window_set (dev->frame, FRAME_NO_CONFIRM, TRUE, 0) ;
  window_destroy (dev->frame) ;   /* forces ASYNCHRONOUS destroyFrame call */
  free (dev) ;
}

/*** next are callback routines registered by DevCreate ***/

static Notify_value destroyFrame (Frame frame, Destroy_status status)
{
  Graph_ g ;

  if (status != DESTROY_CHECKING)
    { if (assFind (assf2g,frame,&g))
	{ free (g->dev) ;		/* prevents graphDevDestroy call */
	  g->dev = 0 ;
	  if (gActive == g)
	    gDev = 0 ;
	  else
	    graphActivate (g->id) ;
	  graphDestroy () ;             /* clean my own structures */
	}
    }
  return notify_next_destroy_func (frame,status) ;
}

/*************************/

#include <sys/time.h>   /* these sys/ files are missing from win_input.h */
#include <sys/types.h>
#include <sunwindow/win_input.h>	/* for arrow key definitions */

static void frameEventProc (Window win, Event *ev, caddr_t arg)
{ Graph_ g ;

  if (assFind (assf2g,win,&g))
    graphActivate (g->id) ;
}

Window gButtonWin = 0 ;	/* needed for waitButtonUp() in sunsubs.c */

static void canvasEventProc (Window win, Event *ev, caddr_t arg)
{
  Graph_ g ;
  int button, kval ;
  Event readev ;
  GraphFunc *funcs = gActive->func ;
  enum MouseEvents {DOWN,DRAG,UP} ;
  extern void help(void) ;
  extern BOOL isBlockedFirst ;

  if (!gDev || win != gDev->canvas)
    { if (!assFind (assc2g,win,&g))
        { assDump (assc2g) ;
	  messcrash ("screwup in graphEventProc: cant find canvas") ;
	}
      graphActivate (g->id) ;
      funcs = gActive->func ;
      if (event_action(ev)==LOC_WINENTER || event_action(ev)==LOC_RGNENTER)
        return ;
    }
/*
  fprintf (stderr,"event id %d, action %d received for %lx\n",
	  event_id(ev), event_action(ev), win) ;
  fprintf (stderr,"  code,flags,shiftmask,locx,locy: %d %x %x %d %d\n",
          ev->ie_code,ev->ie_flags,ev->ie_shiftmask,ev->ie_locx,ev->ie_locy) ;
*/

  switch (event_action(ev))
    {
    case WIN_RESIZE:
      if (funcs[RESIZE])
	(*funcs[RESIZE])() ;
      return ;
    case WIN_REPAINT:
    case SCROLL_REQUEST:
      if (gIsCreating) return ;
      if (isBlockedFirst)
	{ pw_setcmsname (gDev->pixwin, "blockcolours") ;
	  pw_putcolormap (gDev->pixwin, 0, 16, red, green, blue) ;
	  isBlockedFirst = FALSE ;
	}
      graphRedraw () ;
      return ;
    case KEY_TOP(1): case KEY_TOP(10):
      help() ; 
      return ;
    case KEY_TOP(2): case KEY_TOP(9):
      graphPrint () ; 
      return ;
    }

  if (event_is_ascii(ev)  || event_is_key_right(ev))
    { kval = event_action(ev) ;

      if (!funcs[KEYBOARD])
	return ;

      switch (kval)
	{
	case 127: kval = DELETE_KEY ; break ;
	case KEY_RIGHT(2):  kval = UP_KEY ; break ;
	case KEY_RIGHT(5):  kval = DOWN_KEY ; break ;
	case KEY_RIGHT(1):  kval = LEFT_KEY ; break ;
	case KEY_RIGHT(3):  kval = RIGHT_KEY ; break ;
	case KEY_RIGHT(4):  kval = HOME_KEY ; break ;
	case KEY_RIGHT(6):  kval = END_KEY ; break ;
	}
      (*funcs[KEYBOARD])(kval) ;
      return ;
    }

  if (event_is_button(ev) && event_is_down(ev))    /* MOUSE BUTTON */
    { gButtonWin = win ;
      button = event_id(ev) - MS_LEFT ;
      funcs = &funcs[3*button] ;                /* realign funcs */
      if (!button)	/* left button */
	gLeftDown (XtoUabs(event_x(ev)),YtoUabs(event_y(ev))) ;
      else if (funcs[DOWN])
        (*funcs[DOWN])(XtoUabs(event_x(ev)),YtoUabs(event_y(ev))) ;
      if (funcs[UP] || funcs[DRAG])
        while (TRUE)                        /* until button comes up */
          { if (intFlag)
              { intFlag = 0 ; return ;}
            window_read_event (gDev->canvas, &readev) ;
            if (event_is_button(&readev) &&
                (event_id(&readev) - MS_LEFT) == button)
              { if (!event_is_up(&readev))
                  messout ("Up/down mouse button confusion") ;
                if (funcs[UP])
                  (*funcs[UP])(XtoUabs(event_x(&readev)),
                               YtoUabs(event_y(&readev))) ;
                break ;
              }
            else if (event_id(&readev) == LOC_DRAG && funcs[DRAG])
              (*funcs[DRAG])(XtoUabs(event_x(&readev)),
                             YtoUabs(event_y(&readev))) ;
          }
      gButtonWin = 0 ;
      return ;
    }
}

/****************************************************************/
/*** routines handling graph types: resize, create, XxxBounds ***/

static void resizeProc (Canvas canvas, int width, int height)
{
  Graph_ g ; /* first must find relevant graph */
  Scrollbar sbar ;
  float xpos ;

  if (!assFind (assc2g,canvas,&g))
    { assDump (assc2g) ;
      messcrash ("screwup in resizeGraph: cant find canvas %lx",canvas) ;
    }
  graphActivate (g->id) ;

  switch (gActive->type)
    {
    case PLAIN :
      gActive->w = width ;
      gActive->h = height ;
      graphFacMake () ;
      break ;
    case TEXT_FIT :
      gActive->w = width ;
      gActive->h = height ;
      gActive->uw = width / gActive->xFac ;
      gActive->uh = height / gActive->yFac ;
      break ;
    case TEXT_SCROLL :
      break ;
    case MAP_SCROLL :
      sbar = (Scrollbar) window_get (gDev->canvas, 
				     WIN_HORIZONTAL_SCROLLBAR) ;
      xpos = XtoUabs ((int)scrollbar_get (sbar,SCROLL_VIEW_START) +
               0.5 * (int)scrollbar_get (sbar,SCROLL_VIEW_LENGTH)) ;
      gActive->h = height ;
      gActive->w = gActive->h * gActive->uw / 
		      (gActive->uh * gActive->aspect * gPixAspect) ;
      window_set (gDev->canvas,
		  CANVAS_WIDTH, gActive->w,
		  CANVAS_HEIGHT, gActive->h,
		  0) ;
      graphFacMake () ;
      graphGoto (xpos, 0) ;
      break ;
    }
}

/***********************/

static void typeCreate (Graph_ g)
{
  Scrollbar sbar ;
  Canvas canvas = g->dev->canvas ;

  switch (g->type)
    {
    case PLAIN: 
      break ;
    case TEXT_FIT:
      window_set (canvas,
		  CANVAS_FIXED_IMAGE, FALSE,
		  CANVAS_AUTO_EXPAND, TRUE,
		  CANVAS_AUTO_SHRINK, TRUE,
		  0) ;
      break ;
    case TEXT_SCROLL:
      window_set (canvas, 
		  CANVAS_FIXED_IMAGE, TRUE,
		  CANVAS_AUTO_EXPAND, FALSE,
		  CANVAS_AUTO_SHRINK, FALSE,
		  CANVAS_WIDTH, uToXrel(g->uw),
		  CANVAS_HEIGHT, uToYrel(g->uh), 
		  WIN_VERTICAL_SCROLLBAR, sbar = scrollbar_create (0),
		  0) ;
      scrollbar_set (sbar, 
		     SCROLL_NOTIFY_CLIENT, canvas,
/*
                     SCROLL_LINE_HEIGHT, g->yFac,
		     SCROLL_TO_GRID, TRUE, 
*/
		     0) ;
      break ;
    case MAP_SCROLL:
      window_set (canvas,
		  CANVAS_FIXED_IMAGE, FALSE,
		  CANVAS_AUTO_EXPAND, FALSE,
		  CANVAS_AUTO_SHRINK, FALSE,
		  WIN_HORIZONTAL_SCROLLBAR, sbar = scrollbar_create (0), 
		  0) ;
      scrollbar_set (sbar, 
		     SCROLL_NOTIFY_CLIENT, canvas, 
		     0) ;
      break ;
    }
}

/** XxxBounds() routines for scrollable types need device access **/

void graphMapBounds (float ux, float uy, float uw, float uh, float aspect)
{
  if (gActive->type != MAP_SCROLL)
    messcrash ("MapBounds called on invalid graph type %d",gActive->type) ;

  gActive->ux = ux ;
  gActive->uy = uy ;
  gActive->uw = uw ;
  gActive->uh = uh ;
  gActive->aspect = aspect ;

  if (gDev)		/* fake resize event */
    resizeProc (gDev->canvas,
		0, (int) window_get (gDev->canvas, CANVAS_HEIGHT)) ;
}

void graphTextBounds (int nx, int ny)
{
  if (gActive->type != TEXT_SCROLL)
    { messerror ("graphTextBounds called on invalid graph type %d "
		 "This call only make sense for TEXT_SCROLL graphs"
		 " check wspec/displays.wrm", gActive->type) ;
      return ;
    }
  gActive->uw = nx ;
  gActive->uh = ny ;
  gActive->w = nx * gActive->xFac ;
  gActive->h = ny * gActive->yFac ;
  if (gDev)
    window_set (gDev->canvas,
		CANVAS_WIDTH, gActive->w,
		CANVAS_HEIGHT, gActive->h,
		0) ;
}

/*****************************************************************/
/*** other public graphXXXX() routines requiring device access ***/

void graphEvent (int action, float x, float y)
{
static Event ev ;	 /* not really safe - problems if multiple calls */
  Notify_error retval ;

  if (!gDev)
    return ;
  if (action < 0 || action > 127)
    messcrash ("graphEvent() can only handle ASCII actions") ;
  ev.ie_shiftmask = 0 ;
  ev.ie_code = action ;
  switch (retval = notify_post_event (gDev->canvas,&ev,NOTIFY_SAFE))
    {
    case NOTIFY_OK : 
/*    fprintf (stderr,"event %d - action %d - posted for %lx\n",
              action,event_action(&ev),gDev->canvas) ;
*/
      break ;
    case NOTIFY_UNKNOWN_CLIENT : 
      messout ("graphEvent() unknown client") ;
      break ;
    case NOTIFY_NO_CONDITION :
      messout ("graphEvent() no client handler") ;
      break ;
    default :
      messout ("graphEvent() unknown error %d",retval) ;
      break ;
    }
}

/*********************************/

void graphGoto (float x, float y)  /* tries to center x,y in visible region */
{
  long target ;
  Scrollbar sbar ;

  if (!gDev)
    return ;

  if (sbar = (Scrollbar) window_get (gDev->canvas, WIN_VERTICAL_SCROLLBAR))
    { target = uToYabs(y) - (int)window_get(gDev->frame,WIN_HEIGHT) / 2 ;
      if (target < 0)
	target = 0 ;
      scrollbar_scroll_to (sbar,target) ;
    }

  if (sbar = (Scrollbar) window_get (gDev->canvas, WIN_HORIZONTAL_SCROLLBAR))
    { target = uToXabs(x) - (int)window_get(gDev->frame,WIN_WIDTH) / 2 ;
      if (target < 0)
	target = 0 ;
      scrollbar_scroll_to (sbar,target) ;
    }
}

/*****************/

void graphRetitle (char *title)
{
  if (!gDev) return ;
  window_set (gDev->frame, FRAME_LABEL, title, 0) ;
}

/*****************/

static Notify_value destroyMessage (Frame frame, Destroy_status status)
{
  Graph_ g, old = gActive ;

  if (status != DESTROY_CHECKING &&
      assFind (assf2g, frame, &g) &&
      g->func[MESSAGE_DESTROY] && 
      graphActivate (g->id))
    { (*g->func[MESSAGE_DESTROY])() ;
      graphActivate (old->id) ;
    }
  assRemove (assf2g, frame) ;
  return notify_next_destroy_func (frame,status) ;
}

void graphMessage (char *text)
{
  Panel panel ;
  char *cp ;
  int line = 0 ;

  if (gDev->message)
    graphUnMessage () ;

  gDev->message = window_create (gDev->frame, FRAME,
			       FRAME_SHOW_SHADOW, TRUE,
			       0) ;
  notify_interpose_destroy_func (gDev->message, destroyFrame) ;
  assInsert (assf2g, gDev->message, gActive) ;

  panel = window_create (gDev->message, PANEL, 0) ;
  uLinesText (text,40) ;
  while (cp = uNextLine (text))
    panel_create_item (panel, PANEL_MESSAGE,
		       PANEL_LABEL_STRING, cp,
		       PANEL_ITEM_Y, ATTR_ROW(line++),
		       PANEL_ITEM_X, ATTR_COL(0),
		       0) ;

  window_fit (panel) ;
  window_fit (gDev->message) ;
  window_set (gDev->message, WIN_SHOW, TRUE, 0) ;
}

void graphUnMessage (void)
{ if (gDev->message)
    { window_set (gDev->message, FRAME_NO_CONFIRM, TRUE, 0) ;
      window_destroy (gDev->message) ;
      gDev->message = 0 ;
    }
}

void graphPop (void)
{ if (gDev)
    window_set (gDev->frame, WIN_SHOW, TRUE, 0) ;
}

/***************************************************************/
/******************** low level routines ***********************/

#define pr_ops(icol)  (PIX_SRC|PIX_COLOR(icol))

static int      psize,psize2 ;
static Pixwin  *pw ;
static Pixfont *font ;
static int      fontdx,fontdy ;
static int      fontwidth,fontheight ;
static struct pr_brush brush ;  /* linewidth */

static void fontSet (int height)
{
  switch(height)
    {
    case 0:
      font = pf_open (defFont) ; break ;
    case 1: case 2: case 3: case 4: case 5: case 6: case 7:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.r.6") ; break ;
    case 8: case 9: case 10: case 11:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.r.7") ; break ;
    case 12: case 13: case 14:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.r.11") ; break ;
    case 15: case 16:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.r.14") ; break ;
    case 17: case 18: case 19: case 20: case 21: case 22: case 23: case 24:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.r.16") ; break ;
/*
    case -1: case -2: case -3: case -4: case -5: case -6: case -7: case -8:
    case -9: case -10: case -11: case -12:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/cour.b.10") ; break ;
    case -13: case -14:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.b.12") ; break ;
    case -15: case -16:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.b.14") ; break ;
    case -17: case -18: case -19: case -20: case -22: case -23: case -24:
      font = pf_open ("/usr/lib/fonts/fixedwidthfonts/screen.b.16") ; break ;
*/
    default:
      if (height > 0)
	font = pf_open ("/usr/lib/fonts/fixedwidthfonts/cour.r.24") ;
      else
	font = pf_open ("/usr/lib/fonts/fixedwidthfonts/cour.b.24") ;
    }
  if (!font)
    { messout ("Cant find font file for height %d (pixels) - %s",
               height,"using standard sunwindows font") ;
      font = pf_default () ;
    }
  fontdx = - font->pf_char[50].pc_home.x ;      /* to make origin top left */
  fontdy = - font->pf_char[50].pc_home.y ;
  fontwidth = font->pf_defaultsize.x ;

  fontheight = font->pf_defaultsize.y ;
}

BOOL gFontInfo (int height, int *w, int *h)
	/* note that this depends on an integer height, and so
	   is only reliable in the long term for height 0 */
{ 
  fontSet (height) ;	/* can do this because font is reset 
			   explicitly when graphBoxDraw is called */
  *w = fontwidth ;
  *h = fontheight ;
  return TRUE ;
}

/***********************************************/

static BOOL isClip = FALSE ;
static int clipx1, clipx2, clipy1, clipy2 ;

static void drawBox (Box box)
{
  float  t ;
  int    x1,x2,y1,y2,r,rr,r2 ;
  char   *text ;
  int    action ;
  int    pr_op = pr_ops(box->fcol) ;
  extern Pixrect* pr_src[] ;
  BOOL	 isDraw = TRUE ;

  if (box->x1 > box->x2 || box->y1 > box->y2)
    isDraw = FALSE ;

  x1 = uToXabs(box->x1) ;
  y1 = uToYabs(box->y1) ;
  x2 = uToXabs(box->x2) ;
  y2 = uToYabs(box->y2) ;

  if (isClip &&
    (x1 > clipx2 || x2 < clipx1 || y1 > clipy2 || y2 < clipy1))
    isDraw = FALSE ;

  if (!isDraw)
    { int nDeep = 1 ;
      stackCursor (gStk,box->mark) ; /* sets position to mark */
      while (!stackAtEnd (gStk))
	switch (action = stackNext (gStk,int))
	  {
	  case BOX_END :
	    if (!--nDeep)
	      return ;                        /* exit point */
	    break ;
	  case BOX_START :
	    r = stackNext (gStk, int) ;
	    ++nDeep ;
	    break ;
	  case COLOR :
	    x1 = stackNext (gStk,int) ; 
	    break ;
	  case LINE_WIDTH : case TEXT_HEIGHT : case POINT_SIZE :
	    t = stackNext (gStk,float) ;
	    break ;
	  case LINE : case RECTANGLE : case FILL_RECTANGLE :
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    break ;
	  case CIRCLE : case POINT : case TEXT : case TEXT_PTR :
	    t = stackNext (gStk,float) ;
	    t = stackNext (gStk,float) ;
	    switch (action)
	      {
	      case CIRCLE :
		t = stackNext (gStk,float) ;
		break ;
	      case POINT :
		break ;
	      case TEXT :
		text = stackNextText (gStk) ;
		break ;
	      case TEXT_PTR :
		text = stackNext (gStk,char*) ;
		break ;
	      }
	    break ;
	    default :
	      messout ("Invalid draw action %d received",action) ;
	  }
      return ;
    }

  if (box->bcol != TRANSPARENT)
    if (isMono)
      pw_replrop (gDev->pixwin,x1,y1,(x2-x1),(y2-y1),
		  PIX_SRC,pr_src[box->bcol],x1,y1);
    else
      pw_rop (gDev->pixwin,x1,y1,(x2-x1),(y2-y1),
	      pr_ops(box->bcol),0,0,0) ;

  stackCursor (gStk,box->mark) ;

  while (!stackAtEnd (gStk))
    switch (action = stackNext (gStk,int))
      {
      case BOX_END :
        return ;                        /* exit point */
      case BOX_START :
        r = stackNext (gStk, int) ;
        drawBox (gBoxGet (r)) ;             /* recursion */
        break ;
      case COLOR :
        x1 = stackNext (gStk,int) ;
        pr_op = pr_ops(x1) ;
        break ;
      case LINE_WIDTH :
        t = stackNext (gStk,float) ;
        brush.width = uToXrel(t) ;
        break ;
      case TEXT_HEIGHT :
        t = stackNext (gStk,float) ; x1 = uToYrel(t) ;
        fontSet (x1) ;
        break ;
      case POINT_SIZE :
        t = stackNext (gStk,float) ;
        psize = uToXrel(t) ; psize2 = psize/2 ;
        break ;
      case LINE : case RECTANGLE : case FILL_RECTANGLE :
        t = stackNext (gStk,float) ; x1 = uToXabs(t) ;
        t = stackNext (gStk,float) ; y1 = uToYabs(t) ;
        t = stackNext (gStk,float) ; x2 = uToXabs(t) ;
        t = stackNext (gStk,float) ; y2 = uToYabs(t) ;
        switch (action)
          {
          case LINE :
            if (!brush.width)
              pw_vector (pw,x1,y1,x2,y2,pr_op,1) ;
            else
              pw_line (pw,x1,y1,x2,y2,
                       &brush,(struct pr_texture *)0,pr_op) ;
            break ;
          case RECTANGLE :
            pw_batch_on (pw) ;
            pw_vector (pw,x1,y1,x1,y2,pr_op,1);
            pw_vector (pw,x1,y2,x2,y2,pr_op,1);
            pw_vector (pw,x2,y2,x2,y1,pr_op,1);
            pw_vector (pw,x2,y1,x1,y1,pr_op,1);
            pw_batch_off (pw) ;
            break ;
          case FILL_RECTANGLE :
            pw_rop (pw,x1,y1,(x2-x1),(y2-y1),pr_op,0,0,0);
            break ;
          }
        break ;
      case CIRCLE : case POINT : case TEXT : case TEXT_PTR :
        t = stackNext (gStk,float) ; x1 = uToXabs(t) ;
        t = stackNext (gStk,float) ; y1 = uToYabs(t) ;
        switch (action)
          {
          case CIRCLE :      /* bug: seems to only draw in white ! */
            t = stackNext (gStk,float) ; r = uToXrel(t) ;
            rr = r*r ; r2 = (int) (r / sqrt (2.0)) ;
            pw_batch_on (pw) ;
            for (x2=0; x2 <= r2 ; x2++)
              { y2 = (int) sqrt ((double)(rr - x2*x2)) ;
                pw_put (pw, x1+x2, y1+y2, pr_op);
                pw_put (pw, x1-x2, y1+y2, pr_op);
                pw_put (pw, x1-x2, y1-y2, pr_op);
                pw_put (pw, x1+x2, y1-y2, pr_op);
                pw_put (pw, x1+y2, y1+x2, pr_op);
                pw_put (pw, x1-y2, y1+x2, pr_op);
                pw_put (pw, x1-y2, y1-x2, pr_op);
                pw_put (pw, x1+y2, y1-x2, pr_op);
              }
            pw_batch_off (pw) ;
            break ;
          case POINT :
            pw_rop (pw,x1-psize2,y1-psize2,psize,psize,pr_op,0,0,0);
            break ;
          case TEXT :
            text = stackNextText (gStk) ;
            pw_ttext (pw,x1+fontdx,y1+fontdy,pr_op,font,text) ;
            break ;
	  case TEXT_PTR :
	    text = stackNext (gStk,char*) ;
	    pw_ttext (pw,x1+fontdx,y1+fontdy,pr_op,font,text) ;
	    break ;
          }
	break ;
      default :
	messout ("Invalid action %d received in drawBox()",action) ;
      }
}

static void setBoxDefaults (Box box)
{
  brush.width = uToXrel(box->linewidth) ;
  fontSet (uToYrel(box->textheight)) ;
  psize = uToXrel(box->pointsize) ;
  psize2 = psize/2 ;
  pw = gDev->pixwin ;
}

void graphBoxDraw (int k, int fcol, int bcol)
{
  Box box = gBoxGet (k) ;

  if (fcol >= 0)
    box->fcol = fcol ;
  if (bcol >= 0)
    box->bcol = bcol ;

  if (!gDev || gActive->isClear)
    return ;

  setBoxDefaults (box) ;
  isClip = FALSE ;
  drawBox (box) ;
}

void graphClipDraw (int x1, int y1, int x2, int y2)
{
  Box box = gBoxGet(0) ;

  if (!gActive->stack || !gDev)
    return ;

  setBoxDefaults (box) ;

  isClip = TRUE ;
  clipx1 = x1 ; clipx2 = x2 ; clipy1 = y1 ; clipy2 = y2 ;
  drawBox (box) ;
}

void graphRedraw (void)
{
  Scrollbar sbar ;
  Box box = gBoxGet (0) ;

  gActive->isClear = FALSE ;
  if (!gActive->stack || !gDev)
    return ;

  setBoxDefaults (box) ;
  isClip = TRUE ;
  clipx1 = 0 ; clipx2 = 0 ;
  if (sbar = (Scrollbar) window_get (gDev->canvas, WIN_HORIZONTAL_SCROLLBAR))
    clipx1 = (int) scrollbar_get (sbar,SCROLL_VIEW_START) ;
  else
    clipx1 = 0 ;
  if (sbar = (Scrollbar) window_get (gDev->canvas, WIN_VERTICAL_SCROLLBAR))
    clipy1 = (int) scrollbar_get (sbar,SCROLL_VIEW_START) ;
  else
    clipy1 = 0 ;
  clipx2 = clipx1 + (int) window_get (gDev->canvas, WIN_WIDTH) ;
  clipy2 = clipy1 + (int) window_get (gDev->canvas, WIN_HEIGHT) ;
  drawBox (box) ;
}

void graphWhiteOut (void)
{
  int x2, y2 ;

  if (!gDev)
    return ;

  x2 = (int) window_get (gDev->canvas, WIN_WIDTH) ;
  y2 = (int) window_get (gDev->canvas, WIN_HEIGHT) ;
  pw_rop (gDev->pixwin,0,0,x2,y2,pr_ops(WHITE),0,0,0);
}

/******* routines for dragging ********/

#define XOR_op (PIX_SRC ^ PIX_DST)

void graphXorLine (float x1, float y1, float x2, float y2)
{
  int ix1 = uToXabs(x1) ;
  int iy1 = uToYabs(y1) ;
  int ix2 = uToXabs(x2) ;
  int iy2 = uToYabs(y2) ;
  
  if (!gDev || !(pw = gDev->pixwin))
    return ;

  pw_vector (pw,ix1,iy1,ix2,iy2,XOR_op,1) ;
}

void graphXorBox (int k, float x, float y)
{
  Box box = gBoxGet (k) ;
  int x1 = uToXabs(x) ;
  int y1 = uToYabs(y) ;
  int x2 = x1 + uToXrel(box->x2 - box->x1) ;
  int y2 = y1 + uToYrel(box->y2 - box->y1) ;

  if (x2 == x1 || y2 == y1)
    { graphXorLine (x, y, x+box->x2-box->x1, y+box->y2-box->y1) ;
      return ;
    }

  if (!gDev || !(pw = gDev->pixwin))
    return ;

  pw_batch_on (pw) ;
  pw_vector (pw,x1,y1,x1,y2,XOR_op,1);
  pw_vector (pw,x1,y2,x2,y2,XOR_op,1);
  pw_vector (pw,x2,y2,x2,y1,XOR_op,1);
  pw_vector (pw,x2,y1,x1,y1,XOR_op,1);
  pw_batch_off (pw) ;
}

/******** below here some old junk that may one day be useful *******/

/**** START OF COMMENT OUT ***********
   PROBLEMS WITH THIS : must set font first and very costly
void gTextBox (float x, float y, char *text)
{
  struct pr_subregion reg ;
  float t ;

  pf_textbound (reg,strlen(text),font,text) ;
  if (x < gBox->x1) gBox->x1 = x ;
  t = x + xToUrel (reg.size.x + reg.pos.x) ;
  if (t > gBox->x2) gBox->x2 = t ;
  if (y < gBox->y1) gBox->y1 = y ;

  t = y + yToUrel (reg.size.y + reg.pos.y) ;
  if (t > gBox->y2) gBox->y2 = t ;
}

******  END OF COMMENT OUT ***********/
