/*  File: graphsub.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1991
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: graph package - device independent routines acting on
 *	the current active graph.
 * Exported functions: (many)
 * HISTORY:
 * Last edited: Mar 31 19:03 1992 (mieg)
 * * Feb 12 12:52 1992 (mieg): lastXpick, so the mouse will position the cursor.
 * * Feb 11 00:11 1992 (mieg): graphClassEntry -> graphCompletionEntry
 * * Jan 28 1992 (neil): save existing drag routines over a drag operation, 
                         restore them on button up;
                         (changed graphBoxDrag, upBox)
 * * Dec  1 19:21 1991 (mieg): textEntry recognition of arrow keys
 * * Dec  1 19:20 1991 (mieg): graphClassEntry: autocompletes on tab
 * * Oct 10 14:51 1991 (rd): cleaned graphBoxClear - now a toggle
 * * Oct  8 23:00 1991 (mieg): added graphBoxClear
 * * Oct  7 23:00 1991 (rd): added graphBoxShift
 *-------------------------------------------------------------------
 */

#include "regular.h"
#include "graph_.h"     /* defines externals for within graph package */
#include "key.h"
#if defined(THINK_C)
  #define isascii(x) ((x) < 128)
#endif
#include <ctype.h>

/*******************************/

void graphDump ()

 {Graph_ g = gActive ;
  if (g->magic != GRAPH_MAGIC)
   {fprintf (stderr,"Error in graphDUMP: active graph %lx is not a graph\n",
                                                                (long)g) ;
    return ;
   }
  fprintf (stderr,"DUMP of graph %lx  name %s, id %d\n",(long)g,g->name,g->id) ;
  fprintf (stderr,"  user coords: %f %f %f %f\n",g->ux,g->uy,g->uw,g->uh) ;
  fprintf (stderr,"  offsets and factors: %d %d %f %f\n",
                                        g->xWin,g->yWin,g->xFac,g->yFac) ;
  fprintf (stderr,"  device w,h: %d %d\n",g->w,g->h) ;
 }

/*******************************/

void graphFacMake (void)	/* for PLAIN and MAP graphs */
{
  float xfac,yfac ;
  float asp ;

  xfac = gActive->w / gActive->uw ;
  yfac = gActive->h / gActive->uh ;

  if (gActive->aspect)
    { asp = gPixAspect * gActive->aspect ;
      if (xfac*asp > yfac)
	xfac = yfac/asp ;
      yfac = xfac * asp ;
    }
  else
    gActive->aspect = yfac/xfac ;

  gActive->xFac = xfac ;
  gActive->yFac = yfac ;
  gActive->xWin = uToXrel (gActive->ux) ;
  gActive->yWin = uToYrel (gActive->uy) ;
}

void gUpdateBox0 (void)
{
  Box box ;
  
  if (!gActive->nbox)
    return ;

  box = gBoxGet(0) ;
  box->x1 = gActive->ux ; box->x2 = gActive->ux + gActive->uw ;
  box->y1 = gActive->uy ; box->y2 = gActive->uy + gActive->uh ;
}

void graphPlainBounds (float ux, float uy, float uw, float uh, float aspect)
{                                 /* makes ->fac, ->xWin, ->yWin */
  gActive->ux = ux ;
  gActive->uy = uy ;
  gActive->uw = uw ;
  gActive->uh = uh ;
  gActive->aspect = aspect ;
  graphFacMake() ;
  gUpdateBox0 () ;
}

void graphTextInfo(int *nx, int *ny, float *cw, float *ch)
{
  int dx, dy ;
  if (!gFontInfo (uToYrel(gActive->textheight), &dx, &dy)) /* number of pixells of a character */
    messcrash ("Can't get font info for current font") ;
     /* default font dimensions are device dependent */
  *cw = XtoUrel(dx) ; /* char width in user's coord */
  *ch = YtoUrel(dy) ; /* char height in user's coord */
  *nx = gActive->w / dx ;
  *ny = gActive->h / dy ;
}

void graphFitBounds (int *nx, int *ny)
{
  if (gActive->type != TEXT_FIT)
    messcrash 
      ("FitBounds called on invalid graph type %d",gActive->type) ;

  *nx = gActive->w / gActive->xFac ;
  *ny = gActive->h / gActive->yFac ;
}

/* other graphXxxBounds() are in graphdev.c since they actively
   change the size of the scrolled window, which is a device
   dependent operation.
*/

/********************* box control ********************************/

Box gBoxGet (int k)
{ 
  if (k < 0 || k >= gActive->nbox)
    messcrash ("Cannot get box %d - index out of range",k) ;
  return arrp (gActive->boxes, k, struct BoxStruct) ;
}
  

int graphBoxStart (void)
{
  Box box ;
  int k = gActive->nbox++ ;

  box = arrayp (gActive->boxes, k, struct BoxStruct) ;

  box->linewidth = gActive->linewidth ;
  box->pointsize = gActive->pointsize ;
  box->textheight = gActive->textheight ;
  box->fcol = gActive->color ;
  box->bcol = k ? gBox->bcol : BACK_COLOR ;
  box->x1 = 1000000.0 ; box->x2 = -1000000.0 ;
  box->y1 = 1000000.0 ; box->y2 = -1000000.0 ;
  box->flag = 0 ;

  push (gStk, BOX_START, int) ; push (gStk, k, int) ;
  box->mark = stackMark (gStk) ;

  gBox = box ;
  if (k)
    push (gActive->boxstack, gActive->currbox, int) ;
  gActive->currbox = k ;

  return k ;
}

void graphBoxEnd (void)
{ Box box = gBox ;

  if (gActive->currbox == 0)
    messcrash ("Tried to end box 0") ;

  push (gStk, BOX_END, int) ;
  gActive->currbox = pop (gActive->boxstack, int) ;
  gBox = gBoxGet (gActive->currbox) ;
  if (box->x1 < gBox->x1) gBox->x1 = box->x1 ;
  if (box->x2 > gBox->x2) gBox->x2 = box->x2 ;
  if (box->y1 < gBox->y1) gBox->y1 = box->y1 ;
  if (box->y2 > gBox->y2) gBox->y2 = box->y2 ;
}

void graphBoxClear (int k)
{
  Box box = gBoxGet(k) ;

  if(!k)  /* do not clear background box */
    return ;  

  box->x2 = -box->x2 ;		/* will not redraw if x2 < x1 */
  box->x1 = -box->x1 ;

  if (box->x1 > box->x2)	/* Clear the screen */
    graphClipDraw (uToXabs(-box->x1), uToYabs(box->y1),
		   uToXabs(-box->x2), uToYabs(box->y2)) ;
  else
    graphBoxDraw (k, -1, -2) ;	/* expose box */
}

void graphBoxDim (int k, float *x1, float *y1, float *x2, float *y2)
{ Box box = gBoxGet (k) ;
  *x1 = box->x1 ; *x2 = box->x2 ;
  *y1 = box->y1 ; *y2 = box->y2 ;
}

void graphBoxSetPick (int k, BOOL pick)
{
  Box box = gBoxGet (k) ;

  if (pick)
    box->flag &= ~GRAPH_BOX_NOPICK_FLAG ;
  else
    box->flag |= GRAPH_BOX_NOPICK_FLAG ;
}

/********* box picking and dragging **********/

static void (*dragRoutine)(float*,float*,BOOL) ;
static GraphFunc leftDragR, leftUpR, middleDragR, middleUpR;
  /*saved over dragging operations, by graphBoxDrag, restored by upBox*/
static float oldx,oldy,xbase,ybase ;
static int draggedBox ;
static int lastXpick, lastYpick ;

void gLeftDown (float x, float y)
{
  int i = -1 ;
  Box box ;
  VoidRoutine buttonFunc ;

  oldx = x ; oldy = y ;

  for (i = gActive->nbox ; i-- ;)
    { box = gBoxGet (i) ;
      if (box->flag & GRAPH_BOX_NOPICK_FLAG)
	continue ;
      if (x >= box->x1 && x <= box->x2 && 
	  y >= box->y1 && y <= box->y2)
	break ;
    }

  if (i == -1) 	/* outside even the whole drawing area! */
    return ;
  
  if (i && gActive->buttonAss && 
      assFind (gActive->buttonAss, (void*) (i*4), &buttonFunc))
    { graphBoxDraw (i, WHITE, BLACK) ;
      (*buttonFunc)() ;
      if (i < gActive->nbox && (box = gBoxGet(i)) &&
	  box->fcol == WHITE && box->bcol == BLACK &&
	  gActive->buttonAss &&
	  assFind (gActive->buttonAss, (void*) (i*4), &buttonFunc))
	graphBoxDraw (i, BLACK, WHITE) ;
      return ;
    }

  if (gActive->func[PICK])
    { lastXpick = x - box->x1 ;
      lastYpick = y - box->y1 ;
      (*(gActive->func[PICK]))(i,(double)(x - box->x1),
			       (double)(y - box->y1)) ;
      return ;
    }

  if (gActive->func[LEFT_DOWN])
    (*gActive->func[LEFT_DOWN])(x,y) ;
}

static void dragBox (double x, double y)
{
  float oldxbase = xbase, oldybase = ybase ;

  xbase += x-oldx ;
  ybase += y-oldy ;
  (*dragRoutine)(&xbase,&ybase,FALSE) ;
  if (xbase != oldxbase || ybase != oldybase)
    { graphXorBox (draggedBox,oldxbase,oldybase) ;
      graphXorBox (draggedBox,xbase,ybase) ;
    }
  oldx = x ; oldy = y ;
}

static void upBox (double x, double y)
{ 
  graphXorBox (draggedBox,xbase,ybase) ;
  xbase += x-oldx ;
  ybase += y-oldy ;
  (*dragRoutine)(&xbase,&ybase, TRUE) ;

  /*.....restore the saved methods*/
  graphRegister(LEFT_DRAG, leftDragR);
  graphRegister(LEFT_UP, leftUpR);
  graphRegister(MIDDLE_DRAG, middleDragR);
  graphRegister(MIDDLE_UP, middleUpR);

  return;
}

      /* JTM, Drag immediatly to x,y, used with  MIDDLE_DOWN */
void graphBoxDragInit(int k, void (*clientRoutine)(float*,float*,BOOL),
       double x, double y)
{
  Box box = gBoxGet(k) ;
  
  graphBoxDrag(k, clientRoutine) ;
  oldx = (box->x1 + box->x2) / 2 ;
  oldy = (box->y1 + box->y2) / 2 ;
  dragBox(x,y) ;
}

void graphBoxDrag (int k, void (*clientRoutine)(float*,float*,BOOL))
{
  Box box = gBoxGet(k) ;

  dragRoutine = clientRoutine ;
  draggedBox = k ;

  leftDragR=graphRegister(LEFT_DRAG, (VoidRoutine) dragBox);
  leftUpR=graphRegister(LEFT_UP, (VoidRoutine) upBox);
  middleDragR=graphRegister(MIDDLE_DRAG, (VoidRoutine) dragBox);
  middleUpR=graphRegister(MIDDLE_UP, (VoidRoutine) upBox);
    /*saving the existing methods for restoration by upBox*/

  xbase = box->x1 ; ybase = box->y1 ;
  (*dragRoutine) (&xbase,&ybase,FALSE) ;

  graphXorBox (draggedBox,xbase,ybase) ;

  return;
}

/******************** text entry box *********************/

typedef struct EntryBoxStruct
  { int		magic ;
    char*	text ;
    char*	cp ;
    int		len ;
    int		box ;
    int		cursorBox ;
    int         insertMode ;
    GraphFunc	action;
    GraphCompletionFunc  completionFunction ;
    struct EntryBoxStruct *nxt ;
  } *EntryBox ;

#define EBOX_MAGIC 187648

/******************************/

static void graphUnSetCursor(EntryBox ebox)
{ 
  graphBoxDraw (ebox->cursorBox, LIGHTGREEN, TRANSPARENT) ;
}

/******************************/

static void graphSetCursor(EntryBox ebox)
{ 
  float x1,x2,y1,y2 ;
  graphBoxDim (ebox->box, &x1, &y1, &x2, &y2) ;
  graphBoxShift (ebox->cursorBox, x1 + (ebox->cp - ebox->text), y1) ;
  graphBoxDraw (ebox->cursorBox, RED, TRANSPARENT) ;
}

/******************************/

static void entryBoxEntry (int key)
{
  EntryBox ebox ;
  char *cp, *cq , *cr ;
  Graph currGraph = graphActive () ;

  if (!graphAssFind (graphTextEntry,&ebox) ||
        ebox->magic != EBOX_MAGIC )
    messcrash ("entryBoxEntry() can't find entryBox") ;

  cp = ebox->cp ;
  if (strlen(ebox->text) > ebox->len)
    messcrash ("Over flow in entryBoxEntry") ;
  if (cp < ebox->text || cp >= ebox->text + ebox->len)
    cp = ebox->text + strlen(ebox->text) ;
  switch (key)
    {
    case RETURN_KEY:
      ebox->cp = cp ;
      graphSetCursor(ebox) ;
      graphBoxDraw (ebox->box,-1,-1) ;
      if (ebox->action)
	(*(ebox->action))(ebox->text) ;
      graphActivate (currGraph) ;  /* May work or not */
      return ;   /* because action may have destroyed or graphRedrawn */
    case INSERT_KEY:
      ebox->insertMode = 1 - ebox->insertMode ;
      break ;
    case DELETE_KEY:
    case BACKSPACE_KEY:  /* delete one char to my left */
      if (cp == ebox->text)
	break ;
      cq = --cp ;
      while (*cq)
	{ *cq = *(cq + 1) ;
	  cq++ ;
	}
      break ;
    case 4: /* C-d delete at cp */
      cq = cp ;
      while(*cq)
	{ *cq = *(cq + 1) ;
	  cq ++ ;
	}
      break ;
    case 23: /* C-w delete a word leftwards at cp */
      if (cp == ebox->text)
	break ;
      cq = cp ;
      while (*cq == ' ' && cq > ebox->text) cq-- ;
      while (*cq != ' ' && cq > ebox->text) cq-- ;
      cp = cq ;
      while (*cq++ = *cr++) ;	/* copy */
      while (*cq) *cq++ = 0 ;	/* clean up */
      break ;
    case LEFT_KEY:
    case 2:  /* C-b */
      if (cp > ebox->text)
	cp-- ;
      break ;
    case RIGHT_KEY:
    case 6:  /* C-f */
      if (*cp)
	cp++ ;
      if (cp > ebox->text + ebox->len && ebox->len > 0 )
	cp-- ;
      break ;
    case HOME_KEY:
    case 1:			/* C-a */
      cp = ebox->text ; 
      break ;
    case 11:  /* C-k kill from cp to the right of cp */
      cq = ebox->text + ebox->len ;
      while (cq >= cp) 
	*cq-- = 0 ;
      break ;
    case 21:  /* C-u kill whole entry */
      cq = ebox->text + ebox->len ;
      cp = ebox->text ;
      while (cq >= cp) 
	*cq-- = 0 ;
      break ;
    case END_KEY:
    case 5:  /* C-e go to last char = 1 */
      cp = ebox->text + strlen(ebox->text) ;
      if(cp >= ebox->text + ebox->len)
	cp = ebox->text + ebox->len - 
	  (ebox->len > 0 ? 1 : 0) ;
      break ;
    case '\t': /* auto complete */
      if (ebox->completionFunction)
	{ 
	  static EntryBox previousEbox = 0 ;
	  
	  if (ebox != previousEbox)
	    (*ebox->completionFunction)(0,0) ;  /* To reInit the search */
	  (*ebox->completionFunction)(ebox->text, ebox->len) ;
	  previousEbox = ebox ;
	}
      cp = ebox->text + strlen(ebox->text) ;
      break ;
    default:
      if (!isascii(key) || !isprint(key))
	{ fprintf(stderr,"Got unknown char '%c' = %d \n", key, key) ;
	  ebox->cp = cp ; 
	  return ; 
	}
      if (ebox->insertMode)
	{ if (strlen(ebox->text) < ebox->len)
	    { cq = ebox->text + strlen(ebox->text) ;
	      while(--cq >= cp) 
		*(cq + 1) = *cq ;
	      *cp++ = key ;
	    }
	}
      else
	if (cp < ebox->text + ebox->len)
	  *cp++ = key ;
    }

  ebox->cp = cp ;
  graphSetCursor (ebox) ;
  graphBoxDraw (ebox->box,-1,-1) ;
}  

static void entryBoxDestroy (EntryBox ebox)
     /* NB must destroy when clear stack and when destroy graph */
{
  GraphFunc func = 0 ;

  graphBoxDraw (ebox->cursorBox, WHITE, WHITE) ;
  graphBoxDraw (ebox->box, WHITE, WHITE) ;	/* not ideal */

  graphAssRemove (graphTextEntry) ;
  graphAssFind (entryBoxDestroy, &func) ;
  graphRegister (KEYBOARD, func) ;
  graphAssRemove (entryBoxDestroy) ;
  ebox->magic = 0 ;
  free (ebox) ;
}

static int graphTextEntry2 (GraphCompletionFunc f, char* text, int len, float x, float y, void (*fn)())
{
  EntryBox ebox = 0, previous = 0, old ;
  int n = (int)x ;
  char *cp ;

  if(len < 0)
    messcrash("Negative length in graphTextEntry") ;
  if (!text)
    messcrash("Null text in graphTextEntry") ;
  cp = text - 1 ;  /* Clean the buffer */
  while(*++cp) ;
  while(cp < text + len)
    *cp++ = 0 ;

  if (graphAssFind (graphTextEntry,&previous))
    { if (previous->text == text)
	{ GraphFunc oldFunc = graphRegister (KEYBOARD, (GraphFunc) entryBoxEntry) ;
          graphAssociate (entryBoxDestroy, oldFunc) ; /* jtm made graphAss a macro */
	  n = strlen(text) ;
	  if (n > previous->len)
	    messcrash("graphTextEntry2 received text longer than limit %d :\n %s",
		      previous->len, text) ;
	  for( ; n < previous->len ; n++)
	    *(text + n) = 0 ;  /* Clean up usually not done in calling routine */
	  n = lastXpick ;      /* Try to position the cursor */
	  if (n <= 0 || n > strlen(text))
	    n = strlen(text) ;  /* On later call, x is 0 or  position in box */
	  previous->cp = text + n ;
	  
	  graphSetCursor (previous) ;
	  graphBoxDraw (previous->box, BLACK, YELLOW) ;
	  return previous->box ;
	}
      graphUnSetCursor (previous) ;
      graphBoxDraw (previous->box, BLACK, LIGHTGREEN) ;
      graphAssRemove (graphTextEntry) ;
       for (old = previous, ebox = old->nxt ; ebox && ebox->text != text ;)
	{ old = ebox ; 
	  ebox = ebox->nxt ; 
	}
    }

  if (ebox)
    old->nxt = ebox->nxt ;	/* merely unlink */
  else
    { 
      ebox = (EntryBox) messalloc (sizeof (struct EntryBoxStruct)) ;
      ebox->magic = EBOX_MAGIC ;
      ebox->text = text ;
      n = strlen(text) ; /* On first call, x is box position, not position in box */
        /* ebox->cursorOffset = 0 by default (messalloc) */
      ebox->cp = text + n ;
      ebox->len = len ;
      text[len] = 0 ;		/* for subsequent safety */
      ebox->action = fn ;
      ebox->completionFunction = f ;
      ebox->insertMode = 1 ;
      graphAssociate (entryBoxDestroy, 
		      graphRegister (KEYBOARD,(GraphFunc) entryBoxEntry)) ;
      
      ebox->box = graphBoxStart () ;
      graphTextPtr (text,x,y,len) ;
      ebox->cursorBox = graphBoxStart () ;
      graphLine (x+n, y, x+n, y+0.9) ;
      graphBoxEnd () ;
      graphSetCursor(ebox) ;
      graphBoxEnd () ;
    }

  ebox->nxt = previous ;	/* 0 if previous not found */
  n = lastXpick ;      /* Try to position the cursor */
  if (n!=-99)
    { text = ebox->text ;
      if (n <= 0 || n > strlen(text))
	n = strlen(text) ;  /* On later call, x is 0 or  position in box */
      ebox->cp = text + n ;
      graphSetCursor(ebox) ;
    }
  lastXpick = -99 ;
  graphBoxDraw (ebox->box, BLACK, YELLOW) ;
  graphAssociate (graphTextEntry, ebox) ;
 
  return ebox->box ;
}

int graphTextEntry (char* text, int len, float x, float y, void (*fn)())
{ return
    graphTextEntry2 (0, text, len, x, y, fn) ;
}

int graphCompletionEntry (GraphCompletionFunc f, char* text, int len, float x, float y, void (*fn)())
{ return
    graphTextEntry2 (f, text, len, x, y, fn) ;
}

void graphEntryDisable (void)
{
  EntryBox previous ;
  GraphFunc func ;

  if (graphAssFind (graphTextEntry,&previous))
    { 
      graphUnSetCursor(previous) ;
      graphBoxDraw (previous->box, BLACK, LIGHTGREEN) ;
      graphAssFind (entryBoxDestroy, &func) ;
      graphRegister (KEYBOARD, func) ;
    }
}

/********* clear box list, stacks etc - also used during initialisation ***/

void graphClear ()
{
  EntryBox ebox ;

  if (graphAssFind (graphTextEntry,&ebox))
    entryBoxDestroy (ebox) ;

  stackClear (gActive->boxstack) ;
  stackClear (gActive->stack) ;
  if (gActive->buttonAss)
    assClear (gActive->buttonAss) ;

  graphWhiteOut () ;

  gActive->nbox = 0 ;
  graphBoxStart () ;
  graphTextHeight (0.0) ;
  graphColor (BLACK) ;
  gActive->isClear = TRUE ;
}

/********************************* Low Level Graphics ************/

                        /* checks to increase box size */
#define CHECK4\
  if (x0 < x1) { if (x0 < gBox->x1) gBox->x1 = x0 ;\
		   if (x1 > gBox->x2) gBox->x2 = x1 ; }\
  else         { if (x1 < gBox->x1) gBox->x1 = x1 ;\
		   if (x0 > gBox->x2) gBox->x2 = x0 ; }\
  if (y0 < y1) { if (y0 < gBox->y1) gBox->y1 = y0 ;\
		   if (y1 > gBox->y2) gBox->y2 = y1 ; }\
  else         { if (y1 < gBox->y1) gBox->y1 = y1 ;\
		   if (y0 > gBox->y2) gBox->y2 = y0 ; }
#define CHECK3\
  if (x-r < gBox->x1) gBox->x1 = x-r ;\
  if (x+r > gBox->x2) gBox->x2 = x+r ;\
  if (y-r < gBox->y1) gBox->y1 = y-r ;\
  if (y+r > gBox->y2) gBox->y2 = y+r ;

void graphLine (float x0, float y0, float x1, float y1)

{ push (gStk, LINE, int) ;
  push (gStk, x0, float) ; push (gStk, y0, float) ;
  push (gStk, x1, float) ; push (gStk, y1, float) ;
  CHECK4
}

void graphRectangle (float x0, float y0, float x1, float y1)
{ push (gStk, RECTANGLE, int) ;
  push (gStk, x0, float) ; push (gStk, y0, float) ;
  push (gStk, x1, float) ; push (gStk, y1, float) ;
  CHECK4
}

void graphFillRectangle (float  x0, float y0, float x1, float y1)
{ push (gStk, FILL_RECTANGLE, int) ;
  push (gStk, x0, float) ; push (gStk, y0, float) ;
  push (gStk, x1, float) ; push (gStk, y1, float) ;
  CHECK4
}

void graphCircle (float x, float y, float r)
{ push (gStk, CIRCLE, int) ;
  push (gStk, x, float) ; push (gStk, y, float) ; push (gStk, r, float) ;
  CHECK3
}

void graphPoint (float x, float y)
{ float r = 0.5 * gActive->pointsize ;  /* needed for check3 */
  push (gStk, POINT, int) ;
  push (gStk, x, float) ; push (gStk, y, float) ;
  CHECK3
}

float textAspect = 0.583 ; /* 7/12 = SunView default - can be reset elsewhere */
float textXfac = 1.0 ;
float textYfac = 1.0 ;

void graphText (char *text, float x, float y)
{
  float len ;
  float h = textYfac ;
    /* gActive->textheight ? gActive->textheight : 1.0 ; */

  if (!text) 
    return ;

  push (gStk, TEXT, int) ;
  push (gStk, x, float) ;
  push (gStk, y, float) ;
  pushText (gStk, text) ;

  len = strlen(text) * textXfac ;
    /* strlen(text) * h * textAspect * gActive->aspect ; */
  if (x < gBox->x1)         gBox->x1 = x ;
  if (x + len > gBox->x2)   gBox->x2 = x + len ;
  if (y < gBox->y1)         gBox->y1 = y ;
  if (y + h > gBox->y2)     gBox->y2 = y + h ;
}

void graphTextPtr (char *text, float x, float y, int st_len)
{
  float len ;
  float h = textYfac ;
    /* gActive->textheight ? gActive->textheight : 1.0 ; */

  if (!text)
    return ;
  push (gStk, TEXT_PTR, int) ;
  push (gStk, x, float) ;
  push (gStk, y, float) ;
  push (gStk, text, char*) ;

  len = st_len * textXfac ;
    /* st_len * h * textAspect * gActive->aspect ; */
  if (x < gBox->x1)         gBox->x1 = x ;
  if (x + len > gBox->x2)   gBox->x2 = x + len ;
  if (y < gBox->y1)         gBox->y1 = y ;
  if (y + h > gBox->y2)     gBox->y2 = y + h ;
} 

/********************** routines to set static properties ************/

float graphLinewidth (float x)
{ float old = gActive->linewidth ;
  if (x >= 0)
    { push (gStk, LINE_WIDTH, int) ;
      push (gStk, x, float) ;
      gActive->linewidth = x ;
    }
  return old ;
}

float graphPointsize (float x)
{float old = gActive->pointsize ;
  if (x >= 0)
    { push (gStk, POINT_SIZE, int) ;
      push (gStk, x, float) ;
      gActive->pointsize = x ;
    }
  return old ;
}

float graphTextHeight (float x)
{ float old = gActive->textheight ;
  int nx, ny ;
  if (x >= 0)
    { push (gStk, TEXT_HEIGHT, int) ;
      push (gStk, x, float) ;
      gActive->textheight = x ;
      graphTextInfo (&nx, &ny, &textXfac, &textYfac) ;
    }
  return old ;
}

int graphColor (int color)
{ int old = gActive->color ;

  if (color >= 0)
    { push (gStk, COLOR, int) ;
      push (gStk, color, int) ;
      gActive->color = color ;
    }
  return old ;
}

/******** routine to shift a box to a new origin ********/

void graphBoxShift (int kbox, float xbase, float ybase)
{
  Box box = gBoxGet (kbox) ;
  float dx = xbase - box->x1 ;
  float dy = ybase - box->y1 ;
  int nDeep = 1 ;
  int action, r ;
  float t ;
  char* text ;

  stackCursor (gStk,box->mark) ; /* sets position to mark */
  while (!stackAtEnd (gStk))
    switch (action = stackNext (gStk,int))
      {
      case BOX_END :
	if (!--nDeep)
	  goto redraw ;                        /* exit point */
	break ;
      case BOX_START :
	r = stackNext (gStk, int) ;
	++nDeep ;
	break ;
      case COLOR :
	r = stackNext (gStk,int) ; 
	break ;
      case LINE_WIDTH : case TEXT_HEIGHT : case POINT_SIZE :
	t = stackNext (gStk,float) ;
	break ;
      case LINE : case RECTANGLE : case FILL_RECTANGLE :
	stackNext (gStk,float) += dx ;
	stackNext (gStk,float) += dy ;
	stackNext (gStk,float) += dx ;
	stackNext (gStk,float) += dy ;
	break ;
      case CIRCLE : case POINT : case TEXT : case TEXT_PTR :
	stackNext (gStk,float) += dx ;
	stackNext (gStk,float) += dy ;
	switch (action)
	  {
	  case CIRCLE :
	    t = stackNext (gStk,float) ;
	    break ;
	  case POINT :
	    break ;
	  case TEXT :
	    text = stackNextText (gStk) ;
	    break ;
	  case TEXT_PTR :
	    text = stackNext (gStk,char*) ;
	    break ;
	  }
	break ;
	default :
	  messout ("Invalid draw action %d when shifting",action) ;
      }

redraw:
  graphClipDraw (uToXabs(box->x1), uToYabs(box->y1), 
		 uToXabs(box->x2)+1, uToYabs(box->y2)+1) ;
  box->x1 += dx ; box->y1 += dy ; box->x2 += dx ; box->y2 += dy ;
  graphBoxDraw (kbox, -1, -1) ;
}

/******** end of file ********/
