/*  File: graphcon.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: device independent control routines for graph package
 * Exported functions: graphInit, Finish, Create, Destroy, Activate...
 * HISTORY:
 * Last edited: Mar  4 01:58 1992 (rd)
 * Created: Thu Jan  2 02:34:40 1992 (rd)
 *-------------------------------------------------------------------
 */

#include "regular.h"
#include "graph_.h"     /* defines Graph, gXxxx externals, graphDevXxxx() */
#include "array.h"

/******* externals visible from elsewhere *******/

int             intFlag = 0 ;   /* set through notifier in SUNVIEW */
int             isGraphics = FALSE ;
FREEOPT  graphColors[] =        /* for user selection of */
 {16,   "Colours",		/* legal foreground colours */
  WHITE,        "WHITE",
  BLACK,        "BLACK",
  LIGHTGRAY,    "LIGHTGRAY",
  DARKGRAY,     "DARKGRAY",
  RED,          "RED",
  GREEN,        "GREEN",
  BLUE,         "BLUE",
  YELLOW,       "YELLOW",
  CYAN,         "CYAN",
  MAGENTA,      "MAGENTA",
  LIGHTRED,     "LIGHTRED",
  LIGHTGREEN,   "LIGHTGREEN",
  LIGHTBLUE,    "LIGHTBLUE",
  DARKRED,      "DARKRED",
  DARKGREEN,    "DARKGREEN",
  DARKBLUE,     "DARKBLUE"
 } ;

/****** externals only for use within graph package *****/

Graph_   gActive = 0 ;
Dev     gDev = 0 ;
int     gIsCreating = FALSE ;   /* needed for careful activation switch
                                   during creation */
Box     gBox ;
Stack   gStk ;

/****** statics for this file ******/

static int      isInitialised = FALSE ;
static char     *progname ;
static int      ngraph = 0 ;    /* number of graphs made this process */
static Graph_   graphList = 0 ;
static Associator  graphIdAss = 0 ;

/***********************************************************/
/***************** initialisation and finish ****************/

void graphInit (int *argcptr, char **argv)
{
  int isG ;

  if (NEVENTS != DESTROY+1)
    messcrash ("Code error: number of graph register events mismatch") ;

  progname = argv[0] ;

  isG = TRUE ;          /* used to ask if graphics wanted on Suns */

  if (isG && !isGraphics)               /* must initialise */
    { if (!isInitialised)
        {
          isInitialised = TRUE;
	  graphIdAss = assCreate() ;
        }
      isGraphics = TRUE ;
      graphDevInit (argcptr, argv) ;
    }
  isGraphics = isG ;
}

/***********************/

void graphFinish ()
{
  if (isGraphics)
    { while (gActive)
	graphDestroy () ;
      graphDevFinish () ;
    }
  assDestroy(graphIdAss) ;
  graphIdAss = 0 ;
  isGraphics = FALSE ;
}

/***********************/

BOOL graphActivate (Graph gId)
{ Graph_ g ;

  if (gActive && gId == gActive->id)
    return TRUE ;
 
  if (!gId || !assFind(graphIdAss, (void *) gId, &g))
    return FALSE ;

  if (!gIsCreating && gDev)
    graphDevActivate (FALSE) ;
  gActive = g ;
  gDev = gActive->dev ;
  gStk = gActive->stack ;
  gBox = gActive->nbox ? gBoxGet (gActive->currbox) : 0 ;
  if (gDev)
    graphDevActivate (TRUE) ;

  return TRUE ;
}

Graph graphActive (void)
{ return gActive ? gActive->id : 0 ;
}

/********************* create and destroy ************************/

static void typeInitialise (Graph_ g, int type)
	/* This is called before the window is generated, and
	   so can not refer to w,h.
	   After the window is generated the resize procedure
	   will be called once - so any initialisation for that
	   must be placed here.
	   After that xFac,yFac,xWin,yWin must be set, so that
	   drawing can take place.
	*/
{
  int dx,dy ;

  g->type = type ;

  switch (type)
    {
    case PLAIN: case MAP_SCROLL:
      g->ux = g->uy = 0.0 ;
      g->uw = g->uh = g->aspect = 1.0 ;
      break ;
    case TEXT_SCROLL:	/* default 80x25 text window */
      g->uw = 80 ;
      g->uh = 25 ;	/* note deliberate fall through here */
    case TEXT_FIT:
      g->ux = g->uy = 0 ;
      if (!gFontInfo (0, &dx, &dy))
	messcrash ("Can't get font info for default font") ;
      		/* default font dimensions are device dependent */
      g->aspect = dy / (float) dx ;
      g->xFac = dx + 0.00001 ; /* bit to ensure correct rounding */
      g->yFac = dy + 0.00001 ;
      g->xWin = g->yWin = 0 ;
      break ;
    default: 
      messcrash ("Invalid graph type %d requested",type) ;
    }
}

static void positionCheck (float *px, float *py, float *pw, float *ph)
	/* tile if x,y are 0.0 - use old values if w,h are 0.0 */
{
/* coords are device-free: assume screen height = 1.0, width = 1.3 */
  static float oldx = -1.0 ;
  static float oldy = 0.0 ;
  static float oldw = 0.5 ;
  static float oldh = 0.5 ;

  if (*px < 0 || *pw < 0 || *px > 1.3 || *pw > 1.3)
    { messout ("graph bounds x,w = %g,%g must be 0-1.3",*px,*pw) ;
      *px = oldx ;
      *pw = oldw ;
    }
  if (*py < 0 || *ph < 0 || *py > 1 || *ph > 1)
    { messout ("graph bounds y,h = %g,%g must be 0-1",*py,*ph) ;
      *py = oldy ;
      *ph = oldh ;
    }
  if (!*px) *px = (oldx < 0) ? 0 : oldx + oldw ;
  if (!*pw) *pw = oldw ;
    else oldw = *pw ;
  if (!*ph) *ph = oldh ;
   else oldh = *ph ;
  if (!*py) *py = oldy ;
  if (*px + *pw > 1.3)
    { *px = 0 ;
      *py = *py + 0.1 ;
    }
  oldx = *px ;
  if (*py + *ph > 1) *py = 0 ;
  oldy = *py ;
}

Graph graphCreate (int type, char *nam, float x, float y, float w, float h)
{ 
  Graph_ graph;
  int i ;

  if (!isInitialised)
    { messout ("You must graphInit () first") ;
      return (Graph) 0 ;
    }

  graph = (Graph_) messalloc (GRAPH_SIZE) ;

  graph->magic = GRAPH_MAGIC ;
  graph->id = ++ngraph ;
  if (!nam)
    nam = messprintf ("%s%d",progname,ngraph) ;
  graph->name = (char *) messalloc (strlen(nam) + 1);
  strcpy (graph->name,nam) ;

  graph->nxt = graphList ;
  graphList = graph ;         /* add to list of graphs */
  assInsert(graphIdAss, (void *)graph->id, (void *)graph) ;

  graph->boxes = arrayCreate (64, struct BoxStruct) ;
  graph->stack = stackCreate (1024) ;
  graph->boxstack = stackCreate (32) ;
  graph->nbox = 0 ;
  graph->assoc = assCreate() ;
  typeInitialise (graph, type) ;

        /* initialise statics for the graph */
  graph->linewidth = 0.002 ;
  graph->pointsize = 0.005 ;
  graph->textheight = 0.0 ;
  graph->color = FORE_COLOR ;
  for (i = 0 ; i < NEVENTS ; i++)
    graph->func[i] = 0 ;

  if (gDev)
    graphDevActivate (FALSE) ;  /* preparation for devCreate */
  
  gIsCreating = TRUE ;
  if (isGraphics)
    { positionCheck (&x, &y, &w, &h) ;
      graphDevCreate (graph,x,y,w,h) ;
    }
  else
    graph->dev = 0 ;

  graphActivate (graph->id) ;
  gIsCreating = FALSE ;

  graphClear () ;                       /* initialises box, stack structures*/

  return graph->id ;
}

/*******************************/

void graphDestroy ()
{
  Graph_ dying = gActive ;
  Graph_ g ;
static Associator ass = 0 ;            /* used to store set of dying graphs */

  if (!ass)
    ass = assCreate () ;
  if (!assInsert (ass,dying,&g))        /* dying already there */
    return ;

  if (gDev)                             /* device specific functions */
    graphDevDestroy() ;
  devMenuDestroy () ;

  graphClear () ;			/* kills anything attached to boxes */

  if (gActive->func[DESTROY])           /* user registered functions */
    (*(gActive->func[DESTROY]))() ;     /* can not use graph, as gDev gone */

  g = gActive ;

  if (dying == graphList)               /* find in graph list and unlink */
    graphList = dying->nxt ;
  else
    { for (g = graphList ; g && g->nxt != dying ; g = g->nxt) ;
      if (!g)
        messcrash ("Dying graph not in graph list") ;
      g->nxt = dying->nxt ;
    }

  assRemove(graphIdAss, (void *)dying->id) ;

  if (graphList)
    graphActivate (graphList->id) ;         /* must activate something else */
  else
    { gActive = 0 ; gDev = 0 ; gStk = 0 ; gBox = 0 ;
    }

  free (dying->name) ;                  /* deallocate storage */
  stackDestroy (dying->stack) ;
  arrayDestroy (dying->boxes) ;
  stackDestroy (dying->boxstack) ;
  assDestroy (dying->assoc) ;
  if (dying->buttonAss)
    assDestroy (dying->buttonAss) ;
  dying->magic = 0 ;
  free (dying) ;
  assRemove (ass,dying) ;
}

/*********************************/

BOOL graphExists (Graph gId)
{ Graph_ g ;

  return  
    gId && assFind(graphIdAss, (void *)gId, &g) ;
}

/*********************************/

void graphCleanUp (void)  /* kill all windows except current active one */
{ 
  Graph id, currentId = gActive ? gActive->id : 0 ;
  Graph_  g = graphList ;

  if (!graphList)
    return ;

  while (g)
    { 
      id = g->id ;
      g = g->nxt ;
      
      if (id != currentId)
	{ 
	  graphActivate (id) ;
	  graphDestroy() ;
	}
    }
  graphActivate (currentId) ;  /* JTM */
}

/**** routine to register the Help  ****/

char *graphHelp (char *item)
{
  if (item && *item)
    gActive->help = item ;

  return gActive->help ;
}

/**** routine to register functions for events (e.g. mouse) ****/

GraphFunc graphRegister (int i, GraphFunc proc) /* i is a GraphEvent */
{
  GraphFunc old = gActive->func[i] ;
  gActive->func[i] = proc ;
  return old ;
}

/**** button package ****/

int graphButton (char* text, VoidRoutine func, float x, float y)
{
  int k = graphBoxStart () ;
  graphText (text, x+0.4, y+0.2) ;
  graphRectangle (x, y, x + strlen(text) + 0.8, y + 1.2) ;
  graphBoxEnd () ;
  graphBoxDraw (k, BLACK, WHITE) ;
  if (!gActive->buttonAss)
    gActive->buttonAss = assCreate () ; 
  assInsert (gActive->buttonAss, (void*) (k*4), func) ;
  return k ;
}

void graphButtons (MENUOPT *buttons, float x0, float y0, float xmax)
{
  double x = x0 ;
  double y = y0 ;
  int len ;

  while (buttons->text)
    { len = strlen (buttons->text) ;
      if (x + 0.8 + len > xmax && x != x0)
	{ x = x0 ; y += 1.6 ; }
      graphButton (buttons->text, buttons->f, x, y) ;
      x += len + 2 ; 
      ++buttons ;
    }
}

/************** graph associator stuff ***************/

BOOL uGraphAssociate (void* in, void* out)
{
  return assInsert (gActive->assoc,in,out) ;
}

BOOL uGraphAssFind (void *in, void* *out)
{
  return assFind (gActive->assoc,in,out) ;
}

BOOL graphAssRemove (void* in)
{
  return assRemove (gActive->assoc,in) ;
}

/*****************************/
/*****************************/
