 /*  File: parse.c
 *  Author: Richard Durbin (rd@mrc-lmba.cam.ac.uk)
 *  Copyright (C) J Thierry-Mieg and R Durbin, 1992
 *-------------------------------------------------------------------
 * This file is part of the ACEDB genome database package, written by
 * 	Richard Durbin (MRC LMB, UK) rd@mrc-lmba.cam.ac.uk, and
 *	Jean Thierry-Mieg (CRBM du CNRS, France) mieg@frmop11.bitnet
 *
 * Description: ace file parser
 * Exported functions: parseControl(), parseAutomatic()
 * HISTORY:
 * Last edited: Apr 22 16:07 1992 (mieg)
 * * Mar  4 02:48 1992 (rd): added -R option for renaming objects
 * Created: Sun Jan 12 17:52:35 1992 (rd)
 *-------------------------------------------------------------------
 */

#include "acedb.h"
#include "array.h"
#include "a.h"  /* for arraykill */
#include "bs.h"
#include "lex.h"
#include "systags.wrm"
#include "sysclass.wrm"
#include "graph.h"
#include "display.h"
#include "pick.h"
#include "session.h"

static void parseDraw(void), openFile(void), closeFile(void),
  readAll(void), readItem(void), readLine(void), skipItem(void) ;

static MENUOPT parseOpts[] = { 
  openFile, "Open  File",
  readAll, "Read until CTRL/C in home window",
  readItem, "Read one Item",
  readLine, "Read one Line",
  skipItem, "Skip to next Item",
  0,0
  } ;

ParseFunc parseFunc[256] ;  /* == MAXTABLE of lexsubs.c */

/*********** stuff local to the parser routine and initialiser **********/
/************************************************************************/

enum {DONE, OUTSIDE, INSIDE, REF_BLOCKED} ;  /* parser states */

static int itemBox, lineBox ;
static char itemText[64], lineText[64], directory[80], filename[24] ;

static int parseShow = 0 ;
static BOOL haltOnError = TRUE ;

static int state, isError, nerror = 0 ;
static int nob = 0 ;
static FILE *fil = 0 ;
static int level = 0 ;  /* freecard level */
static Stack objStack = 0 ;
/* all these statics are because of callback implementation */

/********************************************/

static void parseArray (KEY key, int class)
{ 
  nob++ ;
  if(class>0 && class<256 &&    
     parseFunc[class])
    { if(parseFunc[class] (level,key))
	return ;
    }
  else
    messout ("No parser available for array class %d",class) ;
  
  if (haltOnError)
    isError = TRUE ;    /* skip to end of this reference */
  ++nerror ;
  while (freecard(level) && freeword()) ;
}

/********************************************/

static void parseLine (int isSkip)		/* deals with one entry */
{
  int   table, ix ;
  float fx ;
  char  *word ;
  KEY   key,tag,type ;
  static BOOL isDelete, isRename ;
  static char *obname ;
  static OBJ obj = 0 , subObj ;
  static KEY  ref ;
  static char *errtext ;

  isError = FALSE ;
  if (isSkip)
    { if (state == DONE)
        goto done ;
      if (state == OUTSIDE)
	return ;
      goto abort ;
    }

  switch (state)
    { 
    case DONE :
      goto done ;
    case OUTSIDE :
      objStack = stackReCreate(objStack, 4) ;
      while (TRUE)
	{ if (!freecard(level))
	    goto done ;
	  if ((word = freeword()) && *word != '!') /* special commenting symbol */
	    break ;
	}

      isDelete = isRename = FALSE ;
      if (!strcmp (word, "-D"))	/* symbol for deletion */
	{ isDelete = TRUE ;
	  if (!(word = freeword()))
	    { errtext = "No class after -D" ;
	      goto abort ;
	    }
	}
      else if (!strcmp (word, "-R"))	/* symbol for renaming */
	{ isRename = TRUE ;
	  if (!(word = freeword()))
	    { errtext = "No class after -R" ;
	      goto abort ;
	    }
	}

      if (pickType (KEYMAKE(table = pickWord2Class(word),0)) != 'A'
	  && pickType (KEYMAKE(table,0)) != 'B')
	{ errtext = messprintf ("Bad class name : %s", word ? word : "NULL") ;
	  goto abort ;
	}
      if (table <  _VKeySet)
	{ errtext = "Trying to read/alter a system class" ;
	  goto abort ;
	}
      freestep(':') ;               /* optional colon after class */
      if (!(obname = freeword()))
	{ errtext = "No object name" ;
	  goto abort ;
	}
      lexaddkey (obname,&ref,table) ;

      if (isRename)
	{ if (!(obname = freeword()))
	    { errtext = "No new name for -R" ;
	      goto abort ;
	    }
	  if (!lexAlias (ref, obname, FALSE, FALSE))
	    { errtext = "Rename failed" ;
	      goto abort ;
	    }
	  return ;
	}

      switch (pickType (ref))
	{ 
	case 'A':
	  if (isDelete)
	    arrayKill (ref) ;
	  else
	    parseArray (ref,table) ;
	  return ;
	case 'B':
	  break ;		/* handled by REF_BLOCKED */
	default:
	  errtext = "Can't handle because unknown class type" ;
	  goto abort ;
	}
      state = REF_BLOCKED ;

    case REF_BLOCKED :			/* note fall through */
      if (!(obj = bsUpdate (ref)))
	{ messout ("Object %s is locked - free before continuing",
		   obname) ;
	  return ;
	}
      push(objStack, obj, OBJ) ;
      ++nob ;
      if (parseShow >= 2 || (parseShow == 1 && !(nob%100)))
	{ strcpy (itemText,messprintf ("%d  %.40s",nob,obname)) ;
	  graphBoxDraw (itemBox,BLACK,WHITE) ;
	}

      if (isDelete)
	{ while (bsGetKeyTags (obj, _bsRight, 0))
	    bsRemove (obj) ;
  /* can't delete the root so must do bit by bit - annoying */
	  bsSave (obj) ;
	  pop(objStack, OBJ) ;
	  state = OUTSIDE ;
	}
      else
	state = INSIDE ;
      return ;

    case INSIDE :
      if (!freecard(level))
	goto done ;
      if (!(word = freeword ()))  /* white line separates objects */
	{ while (!stackEmpty(objStack))
	    { obj = pop(objStack,OBJ) ;
	      bsSave (obj) ;
	    }
	  state = OUTSIDE ;
	  return ;
	}

      if (!strcmp (word, "-D"))	/* symbol for deletion */
	{ isDelete = TRUE ;
	  if (!(word = freeword()))
	    { errtext = "No tag after -D" ;
	      goto abort ;
	    }
	}
      else
	isDelete = FALSE ;

      if (!lexword2key (word,&tag,0)) /* find tag */
	{ errtext = messprintf ("Unknown tag \"%s\"",word) ;
	  goto abort ;
	}
      freestep (':') ;			/* optional colon after tag */

      if (parseShow == 3)
	{ strcpy (lineText,messprintf ("%.15s: %.40s",
				       name(tag),freeword())) ;
          freeback() ;
	  graphBoxDraw (lineBox,BLACK,WHITE) ;
	}

      if (bsAddTag (obj,tag)) /* sub objects are suppose to come on the same line */
	while (word = freeword ())
	  { 
	  scanSubObj:
	    if (!strcmp (word,"-C"))
	      { if (word = freeword())
		  bsAddComment (obj,word,'C') ;
	        continue ;
              }
	    if (!strcmp (word,"-U"))
	      { if (word = freeword())
		  bsAddComment (obj,word,'U') ;
	        continue ;
              }

	    if (!(type = bsType (obj,_bsRight)))
	      { errtext = messprintf ("tag %s : %s - off end of model\n",
		                      name (tag),word) ;
		goto abort ;
	      }

	    if (class(type) && type == KEYMAKE(class(type), 1)) /* a type */
	      {
		if (!bsAddObj(obj, _bsHere, &subObj))
		  { errtext = messprintf ("tag %s : %s - off end of model, addObj failed\n",
					  name (tag),word) ;
		    goto abort ;
		  }
		obj = subObj ;
		push(objStack, obj, OBJ) ;
		goto scanSubObj ;
              }

	    if (table = class(type))
	      { lexaddkey (word,&key,table) ;
		bsAddKey (obj,_bsRight,key) ;
	      }
	    else if (type == _Int)
	      if (sscanf (word,"%d",&ix))
		bsAddData (obj,_bsRight,_Int,&ix) ;
	      else
		{ errtext = messprintf ("%s not an integer after %s\n",
					word, name(tag)) ;
		  goto abort ;
		}
	    else if (type == _Float)
	      if (sscanf (word,"%f",&fx))
		bsAddData (obj,_bsRight,_Float,&fx) ;
	      else
		{ errtext = messprintf ("%s not a float after %s\n",
					word,name(tag)) ;
		  goto abort ;
		}
	    else if (type < _LastC)
	      bsAddData (obj,_bsRight,type,word) ;
	    else          /* a tag */
	      { if (!lexword2key (word,&tag,0)) /* find tag */
		  { errtext = messprintf ("Unknown tag \"%s\"",word) ;
		    goto abort ;
		  }
		if (!bsAddTag(obj, tag))
		  { errtext = messprintf ("Unknown tag \"%s\" in object",word) ;
		    goto abort ;
		  }
	      }
	  }
      else
	{ 
	  errtext = messprintf ("Tag %s does not fit model\n", name(tag)) ;
	  goto abort ;
	}
    
      /* save subObjs, but not obj itself */
      if (stackEmpty(objStack))
	messcrash("stackObj error in parser") ;
      subObj = pop(objStack,OBJ) ;
      while (!stackEmpty(objStack))
	{ obj = pop(objStack,OBJ) ;
	  bsSave(subObj) ;
	  subObj = obj ;
	}
	
      push(objStack, obj, OBJ) ; /* repush obj */  
      if (isDelete)
	bsPrune (obj) ;
			       
      return ;
    }

abort :
  if (obj)
    { fprintf (stderr, "Line %7d  object %.25s : %s\n", 
	       freestreamline(level), name(ref), errtext) ;
      messdump ("  parse error near line %7d in %.25s : %s\n", 
		freestreamline(level), name(ref), errtext) ;
    }
  else
    { fprintf (stderr, "Parse error near line %d: %s\n", 
	       freestreamline(level), errtext) ;
      messdump ("  parse error near line %d: %s\n", 
		freestreamline(level), errtext) ;
    }
  while (freecard(level) && freeword()) ; /* skip to end of this reference */
  while (objStack && !stackEmpty(objStack))
    { obj = pop(objStack,OBJ) ;
      bsDestroy (obj) ;
    }
    
  state = OUTSIDE ;
  if (haltOnError)
    isError = TRUE ;
  ++nerror ;
  return ;

done :
  while (objStack && !stackEmpty(objStack))
    { obj = pop(objStack,OBJ) ;
      bsSave (obj) ;
    }
  
  while (freecard(level)) ; /* skip to end of this stream */
          /* This implicitly closes fil if it was used */
  fil = 0 ;
  *filename = 0 ;
  messout ("%d objects read with %d errors",nob,nerror) ;
/*   messdump ("%d objects read with %d errors",nob,nerror) ; */
  nob = 0 ;
  parseOpts->text = "Open  File" ;
  parseOpts->f = openFile ;
  parseDraw () ;
  state = DONE ;
  return ;
}

/*****************************************************************/

static Graph parseGraph = 0 ;

extern int intFlag ;

static void actionStart (void)	/* common to all actions */
{
  graphActivate (parseGraph) ;
  graphPop () ;
  if (parseShow == 3)	/* clear line item */
    { *lineText = 0 ;
      graphBoxDraw (lineBox,BLACK,WHITE) ;
    }
}

static void readAll (void)
{
  actionStart () ;
  intFlag = 0 ;
  parseShow = 1 ;
  do { parseLine (FALSE) ; 
     } while (!intFlag && !isError && state != REF_BLOCKED && state != DONE) ;
  intFlag = 0 ;
}

static void readItem (void)
{ 
  actionStart () ;
  parseShow = 2 ;
  do { parseLine (FALSE) ;
     } while (state == INSIDE) ;
  parseLine (FALSE) ;		/* get next item's name */
}

static void readLine (void)
{
  actionStart () ;
  parseShow = 3 ;
  parseLine (FALSE) ; 
}

static void skipItem (void)
{
  actionStart () ;
  parseShow = 3 ;
  parseLine (TRUE) ; 
  parseLine (FALSE) ; 
}

static void openFile (void)
{
  actionStart () ;

  if(!isWriteAccess())   /* may be lost doing  a save */
    sessionWriteAccess() ;
  if(!isWriteAccess())  /* may occur is somebody else grabed it */
    { messout("Sorry, you no longer have Write Access");
      graphDestroy() ;
      return ;
    }

  if (!*directory)
    if (getenv ("ACEDB_DATA"))
      strcpy (directory, getenv("ACEDB_DATA")) ;
    else
      strcpy (directory, messprintf ("%s/rawdata",filsetdir(0))) ;

  fil = filqueryopen (directory, filename, "ace", "r") ;
  if (!graphExists (parseGraph)) /* maybe the window is gone */
    { if (fil) 
	fclose(fil) ;
      fil = 0 ;
      return ;
    }
  if (fil)
    { parseOpts->f = closeFile ;
      parseOpts->text = "Close File" ;
      parseDraw () ;
      messdump ("Parsing file %s/%s.ace\n",directory,filename) ;
      state = OUTSIDE ;
    }
  nerror = 0 ;
  level = freesetfile (fil,"") ;
  freespecial ("\n/\\") ;
}

static void closeFile (void)
{
  actionStart () ;
  state = DONE ;
  parseLine (FALSE) ;	/* force file to close */
}

/********************************************/

static void parseDraw (void)
{
  if (!graphActivate (parseGraph))
    return ;
  graphPop () ;
  graphClear () ;

  graphButtons (parseOpts, 1, 3, 55) ;

  if (fil)
    { graphText ("Filename:",1,1) ;
      graphText (directory,11,1) ;
      graphText ("/",11+strlen(directory),1) ;
      graphText (filename,12+strlen(directory),1) ;
      graphText (".ace",12+strlen(directory)+strlen(filename),1) ;
    }

  graphText ("Item:",1,7) ;	
  *itemText = 0 ;
  itemBox = graphBoxStart () ;
  graphTextPtr (itemText,7,7,64) ;
  graphBoxEnd () ;

  graphText ("Line:",1,8.5) ;
  *lineText = 0 ;
  lineBox = graphBoxStart () ;
  graphTextPtr (lineText,7,8.5,64) ;
  graphBoxEnd () ;

  graphRedraw() ;
}

/***************************************/

static void parseDestroy (void)
{
  if (fil)	/* close it correctly */
    { state = DONE ;
      parseLine (FALSE) ;
    }
  parseGraph = 0 ;
}

/********************************************/

void parseControl (void)
{ 
  if(!isWriteAccess())
    { messout("Sorry, you do not have Write Access");
      return ;
    }

  if (parseGraph)
    { graphActivate(parseGraph) ;
      graphPop() ; /* Nicer, isn't it */
      return ;
    }

  if (!*directory)
    if (getenv ("ACEDB_DATA"))
      strcpy (directory, getenv("ACEDB_DATA")) ;
    else
      strcpy (directory, messprintf ("%s/rawdata",filsetdir(0))) ;

  parseGraph = displayCreate(DtAce_Parser) ;
  graphTextBounds (60,10) ;		/* needed to for text box sizing */
  graphRegister (DESTROY, parseDestroy) ;
  
  parseDraw () ;
}

BOOL parseAutomatic (FILE *fileHandle)
{
  Graph original = graphActive() ;

  if (graphActivate (parseGraph))
    { graphDestroy () ;
      parseGraph = 0 ;
      graphActivate (original) ;
    }

  graphText ("Item:",27,1.5) ;	
  *itemText = 0 ;
  itemBox = graphBoxStart () ;
  graphTextPtr (itemText,31,1.5,40) ;
  graphBoxEnd () ;

  nerror = 0 ;

  fil = fileHandle ;
  level = freesetfile (fileHandle,"") ;
  freespecial ("\n/\\") ;
  state = OUTSIDE ;
  readAll () ;
  return (state == DONE && nerror == 0) ;
}

BOOL parseBuffer(char *text)
{
  fil = 0 ;
  state = OUTSIDE ;
  parseShow = 0 ;		/* no status drawing */
  if (!text || !*text)
    return FALSE ;
  level = freesettext(text,"/\n\\") ;
  do { parseLine (FALSE) ; 
     } while (!isError && state != REF_BLOCKED && state != DONE) ;
  if (isError)
    { messout ("Some sort of parse error") ;
      goto abort ;
    }
  if (state == REF_BLOCKED)
    { messout ("Parsing blocked by locked object") ;
      goto abort ;
      return FALSE ;
    }
  return TRUE ;
 abort:
  state = DONE ;
  parseLine (FALSE) ;
  return FALSE ;
}


/***** end of file ****/
