/**********************************************************************/
/*                                                                    */
/*	CRISP - Programmable editor                                   */
/*	===========================                                   */
/*                                                                    */
/*  File:          kbd.c                                              */
/*  Author:        P. D. Fox                                          */
/*  Created:       24 Apr 1991                     		      */
/*                                                                    */
/*  Copyright (c) 1990, 1991 Paul Fox                                 */
/*                All Rights Reserved.                                */
/*                                                                    */
/*                                                                    */
/*--------------------------------------------------------------------*/
/*  Description:  Routines to manipulate key maps and bindings        */
/*                                                                    */
/**********************************************************************/

/*static char sccs_id[] = "%Z% %M% %R%.%L%";*/
# include	"list.h"
# include	"alt.h"
# include	"foxlib/stype.h"

# if !defined(toupper)
int	toupper();
# endif

/**********************************************************************/
/*   Following  structure  used  so  that  we  can map a sequence of  */
/*   input characters to an internal key code.			      */
/**********************************************************************/
typedef struct keyseq_t {
	unsigned short	ks_code;/* Internal keycode.		       */
	char	ks_buf[1];	/* Array of characters null terminated */
				/* containing key sequence. If first byte is */
				/* null then this is a self_insert.    */
	} keyseq_t;

/**********************************************************************/
/*   Following  structure  is  used  to maintain a splay tree of the  */
/*   keyboard  assignments  for  the  current keymap. We use a splay  */
/*   tree  because  it  is FAST and self-ordering here so we avoid a  */
/*   lot of CPU time trying to match characters.		      */
/**********************************************************************/
typedef struct keytree_t {
	int	kt_id;		/* Keyboard ID.				*/
	int	kt_ref;		/* Number of references to this keyboard.*/
	stype_t	*kt_macros;	/* Macro assignments per key-stroke	*/
	} keytree_t;

keytree_t	*cur_kp;	/* Pointer to current keyboard table.	*/
extern struct k_tbl k_tbl[];	/* Default bindings on startup.		*/
ref_t		*push_ref;	/* Pointer to keyboard push back buffer.*/
static int kbd_id;		/* Used to give unique values to keyboards. */
SPTREE	*kseq_tree;		/* Tree of key assignments.		*/
char	user_call[64];		/* Last macro called from keyboard by user*/
static char key_char_buf[16];
				/* Char version of key_buf[]		*/
int	next_multi_seq = 0;	/* Used to generate unique multi-key Ids */
char	**multi_key_seq;	/* Pointers to strings defining multi-keys.*/
char	character;		/* Current character typed.		*/
Head_p	hd_kbd;			/* Currently push keyboards.		*/
Head_p	hd_kstk;		/* Popped keyboards but user still wants */
				/* to access them.			*/

/**********************************************************************/
/*   Following  two  variables  are used by push_back[12] functions.  */
/*   push_back1()  uses  these  when  its  storing  a  button  press  */
/*   keystroke event.						      */
/**********************************************************************/
static	int	store_key_x, store_key_y;
/**********************************************************************/
/*   Following  two  are  used  to  keep  track of (x,y) co-ordinate  */
/*   associated with a keystroke event when the keystroke is read.    */
/**********************************************************************/
static	int	fetch_key_x, fetch_key_y;
/**********************************************************************/
/*   Define  some  tables to make life easier when converting to and  */
/*   from internal codes.					      */
/**********************************************************************/
# define	BACKSPACE	8
# define	ENTER	'\r'
char	*keypad_names[] = {
	"Ins",
	"End",
	"Down",
	"PgDn",
	"Left",
	"5",
	"Right",
	"Home",
	"Up",
	"PgUp",
	"Del",
	"Plus",
	"Minus",
	"Star",
	"Divide",
	"Equals",
	"Enter",
	"Pause",
	"PrtSc",
	"Scroll",
	"NumLock"
	};

struct map {
	int	len;
	char	*name;
	short	modifier;
	short	value;
	};
/**********************************************************************/
/*   Table to help us convert keystrings to internal keycodes.	      */
/**********************************************************************/
struct	map	keystring_tbl[] = {
	{3, "ESC",		0, ESC},
	{1, "{",		0, '{'},
	{1, "}",		0, '}'},
	{5, "SPACE",		0, ' '},
	{5, "ENTER",		0, ENTER},
	{3, "TAB",		0, 0x09},
	{5, "ARROW",		0, 0},
	{4, "CTRL",		MOD_CTRL, 0},
	{5, "SHIFT",		MOD_SHIFT, 0},
	{9, "BACKSPACE",	0, BACKSPACE},
	{4, "BACK",		MOD_SHIFT, 0},
	{7, "PRIVATE",		RANGE_PRIVATE, 0},
	{3, "ALT",		MOD_META, 0},
	{4, "META",		MOD_META, 0},
	{6, "KEYPAD",		RANGE_KEYPAD, 0},
	{4, "GREY",		RANGE_KEYPAD, 0},
	{2, "UP",		RANGE_KEYPAD, KEY_UP},
	{4, "DOWN",		RANGE_KEYPAD, KEY_DOWN},
	{4, "LEFT",		RANGE_KEYPAD, KEY_LEFT},
	{5, "RIGHT",		RANGE_KEYPAD, KEY_RIGHT},
	{4, "HOME",		RANGE_KEYPAD, KEY_HOME},
	{3, "END",		RANGE_KEYPAD, KEY_END},
	{4, "PGUP",		RANGE_KEYPAD, KEY_PAGEUP},
	{4, "PGDN",		RANGE_KEYPAD, KEY_PAGEDOWN},
	{4, "STAR",		RANGE_KEYPAD, KEYPAD_STAR},
	{5, "MINUS",		RANGE_KEYPAD, KEYPAD_MINUS},
	{4, "PLUS",		RANGE_KEYPAD, KEYPAD_PLUS},
	{3, "DEL",		RANGE_KEYPAD, KEY_DEL},
	{3, "INS",		RANGE_KEYPAD, KEY_INS},
	{5, "PRTSC",		RANGE_KEYPAD, KEYPAD_PRTSC},
	{6, "SCROLL",		RANGE_KEYPAD, KEYPAD_SCROLL},
	{5, "MOUSE",		RANGE_MISC, MOUSE_KEY},
	0
	};

/**********************************************************************/
/*   Prototypes.						      */
/**********************************************************************/
char	*find_macro_name PROTO((int));
void	key_init PROTO((void));
char	*key_to_char PROTO((char *, int));
keytree_t *find_keyboard PROTO((int, int));
void	kbd_type PROTO((void));
void	key_add_macro PROTO((int, LISTV *));
void	key_free_macro PROTO((LISTV *));
int key_string_to_code PROTO((char *));
static int	keyname_to_code PROTO((char *));
void	exec_key PROTO((int));
void	inq_command PROTO((void));
void	inq_keyboard PROTO((void));
keytree_t	*new_keyboard PROTO((void));
void	free_keyboard PROTO((keytree_t *));
void	find_assignment PROTO((keytree_t *));
char	*get_listv_str PROTO((LISTV *));
/**********************************************************************/
/*   Function called at startup to initialise the first key map.      */
/**********************************************************************/
void
key_init()
{	register int i;
	LISTV	lbuf;

	hd_kbd = ll_init();
	hd_kstk = ll_init();
	
	/***********************************************/
	/*   Create splay tree for key-sequences.      */
	/***********************************************/
	kseq_tree = spinit();
	cur_kp = new_keyboard();
	/***********************************************/
	/*   Do a keyboard_typeables().		       */
	/***********************************************/
	kbd_type();
	/***********************************************/
	/*   Set  up  the  key definitions from table  */
	/*   in config.c			       */
	/***********************************************/
	lbuf.l_flags = F_LIT;
	for (i = 0; k_tbl[i].name; i++) {
		lbuf.l_str = k_tbl[i].name;
		key_add_macro((int) k_tbl[i].key, &lbuf);
		}
	ll_push(hd_kbd, (char *) cur_kp);
	
	/***********************************************/
	/*   Allocate  some  memory  for  the  inital  */
	/*   push  back  buffer.  We  use a reference  */
	/*   because  we  can  easily  expand this as  */
	/*   necessary.				       */
	/***********************************************/
	push_ref = r_init(F_STR, "123456789", 10);
	push_ref->r_used = 0;
}
/**********************************************************************/
/*   Function  to  define  a  key-sequence.  Called  by  the code in  */
/*   kbdmap.c.  Returns  the  actual  key  code  assigned  if we are  */
/*   given a multi-key code, or -1 if all entries used up.	      */
/**********************************************************************/
int
key_define_key_seq(key, str)
int	key;
char	*str;
{	SPBLK	*sp;
	int	len = strlen(str);
	keyseq_t	*ks;
	int	key_code = key;

	/***********************************************/
	/*   If  we  already  have  a  definition for  */
	/*   this   key   sequence,  remove  it.  For  */
	/*   multi-key  sequences  we  can re-use the  */
	/*   entry.				       */
	/***********************************************/
	if ((sp = splookup(str, kseq_tree)) != NULL) {
		ks = (keyseq_t *) sp->data;
		if (key < 0)
			key_code = ks->ks_code;
		spdeq(sp, kseq_tree);
		spfreeblk(sp);
		}
	else {
		if (key < 0) {
			/***********************************************/
			/*   Make  sure  we  don't run out of room in  */
			/*   the multikey range.		       */
			/***********************************************/
			if (next_multi_seq >= MULTIKEY_SIZE)
				return -1;
			key_code = RANGE_MULTIKEY + next_multi_seq++;
			/***********************************************/
			/*   Keep  a  pointer  to the defining string  */
			/*   handy  so  we can implement int_to_key()  */
			/*   properly for these sequences.	       */
			/***********************************************/
			if (multi_key_seq == NULL)
				multi_key_seq = (char **) chk_alloc(sizeof (char *));
			else
				multi_key_seq = (char **) 
					chk_realloc((char *) multi_key_seq,
						next_multi_seq * sizeof (char *));
			}
		}
	/***********************************************/
	/*   Allocate new node for this entry.	       */
	/***********************************************/
	sp = spblk(sizeof(keyseq_t) + len);
	ks = (keyseq_t *) sp->data;
	ks->ks_code = key_code;
	memcpy(ks->ks_buf, str, len + 1);
	sp->key = ks->ks_buf;
	spenq(sp, kseq_tree);
	if (key < 0)
		multi_key_seq[key_code - RANGE_MULTIKEY] = ks->ks_buf;
	return key_code;
}
/**********************************************************************/
/*   Allocate memory and initialise a new keyboard.		      */
/**********************************************************************/
keytree_t *
new_keyboard()
{	keytree_t *kp;

	kp = (keytree_t *) chk_alloc(sizeof *kp);
	if (kp == NULL)
		return NULL;
	kp->kt_id = ++kbd_id;
	kp->kt_ref = 1;
	kp->kt_macros = st_alloc();

	return kp;
}	
/**********************************************************************/
/*   Function to add an assignment of a macro to a key binding	      */
/**********************************************************************/
void
key_add_macro(key, lp)
int	key;
LISTV	*lp;
{	LISTV	*lp1;
	sentry_t	*sep;
	
	if (cur_kp == NULL)
		return;

	lp1 = (LISTV *) chk_alloc(sizeof (LISTV));
	*lp1 = *lp;
	/***********************************************/
	/*   Look  for  an  old definition, and if so  */
	/*   delete it.				       */
	/***********************************************/
	if ((sep = st_lookup(cur_kp->kt_macros, (unsigned long) key)) != NULL) {
		key_free_macro((LISTV *) sep->se_ptr);
		st_replace(cur_kp->kt_macros, sep, (char *) lp1);
		return;
		}
	st_insert(cur_kp->kt_macros, (unsigned long) key, (char *) lp1);
}
/**********************************************************************/
/*   Function  to  delete  a  key  entry  and  free  the  memory  if  */
/*   necessary.							      */
/**********************************************************************/
void
key_free_macro(lp)
LISTV	*lp;
{
	if (lp == NULL)
		return;
	switch (lp->l_flags) {
	  case F_RSTR:
	  	r_dec(lp->l_ref);
		break;
	  case F_STR:
	  	chk_free(lp->l_str);
		break;
	  default:
	  	break;
	  }
	chk_free((char *) lp);
}
/**********************************************************************/
/*   Primitive  to  add  a  define  a  macro  to  be called when the  */
/*   specified key is pressed.					      */
/**********************************************************************/
void
assign_to_key()
{	int	key_code;
	char	*cp;
	char	buf[BUFSIZ];
	LISTV	lbuf;
	
	acc_assign_int(-1L);
	/***********************************************/
	/*   Get  the  key-name.  If not specified in  */
	/*   macro then prompt for it.		       */
	/***********************************************/
	if ((cp = get_arg1("Enter key: ", buf, sizeof buf - 1)) == NULL)
		return;
	/***********************************************/
	/*   Convert  ascii  string  to  internal key  */
	/*   name.				       */
	/***********************************************/
	key_code = key_string_to_code(cp);
	/***********************************************/
	/*   If  we  have a multi-key then we need to  */
	/*   enter  this  key  into  the key-sequence  */
	/*   table in the private range.	       */
	/***********************************************/
	if (key_code < 0) {
		key_code = key_define_key_seq(-1, key_char_buf);
		}
	acc_assign_int((long) key_code);
	/***********************************************/
	/*   Now  get  the macro name to be assigned.  */
	/*   If  it  comes  from  the command line we  */
	/*   need  to  make  sure  we allocate memory  */
	/*   for it.				       */
	/***********************************************/
	if (argv[2].l_flags == F_NULL) {
		if (ereply("Enter macro name to assign: ", 
			buf, sizeof buf - 1) != TRUE)
			return;
		lbuf.l_flags = F_STR;
		lbuf.l_str = strdup(buf);
		}
	else {
		lbuf.l_flags = argv[2].l_flags;
		switch (argv[2].l_flags) {
		  case F_LIT:
		  	lbuf.l_str = argv[2].l_str;
			break;
		  case F_STR:
		  	lbuf.l_str = strdup(argv[2].l_str);
			break;
		  case F_RSTR:
		  	lbuf.l_ref = r_inc(argv[2].l_ref);
			break;
		  default:
		  	panic("assign_to_key: what?");
		  }
		}
	/***********************************************/
	/*   Now  put  the  definition in the current  */
	/*   key map.				       */
	/***********************************************/
	key_add_macro(key_code, &lbuf);
}
/**********************************************************************/
/*   keyboard_typeable   primitive.   Fill  in  the  ASCII  typeable  */
/*   characters.						      */
/**********************************************************************/
void
kbd_type()
{	register int	i;
	LISTV	lbuf;
	
	/***********************************************/
	/*   NULL  entry  means  self_insert  without  */
	/*   having to allocate memory for this.       */
	/***********************************************/
	lbuf.l_flags = F_NULL;
	for (i = 0; i < 256; i++) {
		key_add_macro(i, &lbuf);
		}
}

/**********************************************************************/
/*   The  following  function  converts a key string in ascii format  */
/*   to  the  internal  key  code value. If a multi-key is specified  */
/*   more than one key-code wide), return -1.			      */
/**********************************************************************/
int
key_string_to_code(cp)
register char *cp;
{	int	i;
	SPBLK	*sp;
	keyseq_t	*ks;
	unsigned short *key_buf;
	int	key_width;

	key_buf = cvt_string_to_code(cp, &key_width);
	if (key_width == 1)
		return key_buf[0];
	/***********************************************/
	/*   Convert  the  short-array  of keystrokes  */
	/*   to a char array.			       */
	/***********************************************/
	for (i = 0; i < key_width; i++) {
		/***********************************************/
		/*   Dont let user specify 16-bit chars.       */
		/***********************************************/
		if (key_char_buf[i] & ~0xff)
			continue;
		key_char_buf[i] = (char) key_buf[i];
		}
	key_char_buf[i] = NULL;
	/***********************************************/
	/*   See if we have code in the table.	       */
	/***********************************************/
	if ((sp = splookup(key_char_buf, kseq_tree)) != NULL) {
		ks = (keyseq_t *) sp->data;
		return ks->ks_code;
		}
	return -1;
}
/**********************************************************************/
/*   Convert  a  sequence  of  keystrokes  in external format to the  */
/*   internal values.						      */
/**********************************************************************/
unsigned short *
cvt_string_to_code(cp, lenp)
register char	*cp;
int	*lenp;
{
	static ref_t	*rp;
	unsigned short us;
	
	if (rp == NULL)
		rp = r_init(F_STR, "", 1);
	rp->r_used = 0;
	while (*cp) {
		/***********************************************/
		/*   <h> is  lower  case  'h'  and  not upper  */
		/*   case version so handle differently here.  */
		/***********************************************/
		if (*cp == '<' && cp[1] && cp[2] == '>') {
			us = (unsigned short) cp[1];
			cp += 3;
			}
		else if (*cp == '<') {
			us = (unsigned short) keyname_to_code(cp);
			while (*cp)
				if (*cp++ == '>')
					break;
			}
		else if (*cp == '#') {
			us = (unsigned short) atoi(++cp);
			while (isdigit(*cp))
				cp++;
			}
		else {
			if (*cp == '%')
				cp++;
			if (*cp == '\\')
				us = (unsigned short) *++cp;
			else if (*cp == '^')
				us = (unsigned short) *++cp & 0x1f;
			else
				us = (unsigned short) *cp;
			cp++;
			}
		r_append(rp, (char *) &us, sizeof us, 64);
		}
	*lenp = rp->r_used / sizeof (unsigned short);
	return (unsigned short *) rp->r_ptr;
}
/**********************************************************************/
/*   This  function  tries  to  convert  a  key  description  to its  */
/*   internal  key  code.  This  function  only handles the key-type  */
/*   descriptions.   The   rest  of  the  key  code  parsing  is  in  */
/*   key_string_to_code() which calls this.			      */
/**********************************************************************/
static int
keyname_to_code(string)
char	*string;
{	char	str[80];
	register char	*cp;
	int	flags = 0;
	int	key = 0;
	register struct map *mp;

	/***********************************************/
	/*   Copy string ignoring the initial '<'.     */
	/***********************************************/
	strcpy(str, string+1);
	
	/***********************************************/
	/*   Make   it  all  upper  case  for  easier  */
	/*   testing down below.		       */
	/***********************************************/
	for (cp = str; *cp; cp++)
		if (islower(*cp))
			*cp = toupper(*cp);

	for (cp = str; *cp && *cp != '>'; ) {
		/***********************************************/
		/*   Ignore whitespace and punctuation.	       */
		/***********************************************/
		if (*cp == ' ' || *cp == '-' || *cp == '\t') {
			cp++;
			continue;
			}
		/***********************************************/
		/*   Check for Function key.		       */
		/***********************************************/
		if (*cp == 'F' && isdigit(cp[1])) {
			key = atoi(++cp) - 1;
			flags |= RANGE_FN;
			while (isdigit(*cp))
				cp++;
			continue;
			}
		/***********************************************/
		/*   Check for button keys.		       */
		/***********************************************/
		if (strncmp(cp, "BUTTON", 6) == 0) {
			cp += 6;
			key = atoi(cp++) - 1;
			if (strncmp(cp, "-UP", 3) == 0)
				key += BUTTON1_UP - BUTTON1_DOWN;
			else if (strncmp(cp, "-MOTION", 7) == 0)
				key += BUTTON1_MOTION - BUTTON1_DOWN;
			flags |= RANGE_BUTTON;
			break;
			}
		/***********************************************/
		/*   Look  up  in  the  table for the special  */
		/*   keywords.				       */
		/***********************************************/
		for (mp = keystring_tbl; mp->name; mp++)
			if (strncmp(mp->name, cp, mp->len) == 0) {
				cp += mp->len;
				flags |= mp->modifier;
				key |= mp->value;
				break;
				}
		if (mp->name == 0) {
			key |= *cp;
			break;
			}
		}
		
	/***********************************************/
	/*   We  now  have  the modifiers and the key  */
	/*   itself.  We  now  can  just  OR  in  the  */
	/*   flags  and  key  value,  except  for the  */
	/*   special cases.			       */
	/***********************************************/
	if (flags == MOD_SHIFT && key == '\t')
		return BACK_TAB;
	if ((flags & RANGE_MASK) == RANGE_KEYPAD &&
	    key >= '0' && key <= '9')
		key -= '0';
	if (flags == MOD_CTRL && key >= 0x40 && key <= 0x7f)
		return key & 0x1f;
	return flags | key;
}
/**********************************************************************/
/*   push_back() primitive.					      */
/**********************************************************************/
void
push_back()
{
	push_back1(push_ref, (int) argv[1].l_int);
}
/**********************************************************************/
/*   Function  to  push  a  character  back  on  the  keyboard input  */
/*   buffer.  Called  by the primitive and when user hits an invalid  */
/*   key on the command line.					      */
/**********************************************************************/
void
push_back1(pp, ch)
ref_t	*pp;
int	ch;
{
	/***********************************************/
	/*   Make  sure  theres room for at least the  */
	/*   key plus the (x,y) event.		       */
	/***********************************************/
	if (pp->r_used + 6 >= pp->r_size) {
		pp->r_size += 16;
		pp->r_ptr = chk_realloc(pp->r_ptr, pp->r_size);
		}
	pp->r_ptr[pp->r_used++] = (char) (ch >> 8);
	pp->r_ptr[pp->r_used++] = (char) (ch & 0xff);
	/***********************************************/
	/*   If  we've  got  a  button  event then we  */
	/*   also  have  an  (x,y)  range  associated  */
	/*   with it, so go for that.		       */
	/***********************************************/
	if ((ch & RANGE_MASK) == RANGE_BUTTON) {
		pp->r_ptr[pp->r_used++] = (char) (store_key_x >> 8);
		pp->r_ptr[pp->r_used++] = (char) (store_key_x & 0xff);
		pp->r_ptr[pp->r_used++] = (char) (store_key_y >> 8);
		pp->r_ptr[pp->r_used++] = (char) (store_key_y & 0xff);
		}
}
/**********************************************************************/
/*   Function  to  push  back  a mouse button down/up event together  */
/*   with the x,y) co-ordinate into the keystroke input buffer.	      */
/**********************************************************************/
void
push_back2(pp, ch, x, y)
ref_t	*pp;
int	ch;
int	x, y;
{
	/***********************************************/
	/*   Store  (x,y)  co-ordinate  in  a  global  */
	/*   because  it  makes  the  code smaller by  */
	/*   not   having   to   have  each  call  to  */
	/*   push_back1() have two extra arguments.    */
	/***********************************************/
	store_key_x = x;
	store_key_y = y;
	push_back1(pp, ch);
}
/**********************************************************************/
/*   Function  to  get  the next character from the pushback buffer.  */
/*   Return zero if no more available.				      */
/**********************************************************************/
int
get_push(pref, get_it)
ref_t	*pref;
int	get_it;
{	int	ch;
	int	n = 2;

	if (!get_it)
		return pref->r_used;

	if (pref->r_used == 0)
		return 0;
	ch = (pref->r_ptr[0] << 8) | (pref->r_ptr[1] & 0xff);
	if ((ch & RANGE_MASK) == RANGE_BUTTON) {
		fetch_key_x = (pref->r_ptr[2] << 8) | (pref->r_ptr[3] & 0xff);
		fetch_key_y = (pref->r_ptr[4] << 8) | (pref->r_ptr[5] & 0xff);
		n = 6;
		}
	/***********************************************/
	/*   Shuffle remaining characters down.	       */
	/***********************************************/
	pref->r_used -= n;
	memcpy(pref->r_ptr, pref->r_ptr + n, pref->r_used);
	return ch;
}
/**********************************************************************/
/*   key_to_int primitive -- convert key name to internal key code.   */
/**********************************************************************/
void
key_to_int()
{	SPBLK	*sp;
	keyseq_t	*ks;
	char	*cp = get_str(1);

	if (argv[2].l_int == 0) {
		acc_assign_int((long) key_string_to_code(cp));
		return;
		}
	if ((sp = splookup(cp, kseq_tree)) == NULL)
		acc_assign_int((long) -1L);
	else {
		ks = (keyseq_t *) sp->data;
		acc_assign_int((long) ks->ks_code);
		}
}
/**********************************************************************/
/*   int_to_key  primitive  --  convert  internal  key  code  to the  */
/*   external name.						      */
/**********************************************************************/
void
int_to_key()
{
	acc_assign_str(code_to_keyname((int) argv[1].l_int), -1);
}
/**********************************************************************/
/*   Function  to  convert a numeric key identifier into a printable  */
/*   string in the standard format.				      */
/**********************************************************************/
char *
code_to_keyname(key)
int	key;
{
	static char buf[48];
	register char *cp, *bp;
	int	i;
	SYMBOL	*sp;
	
	/***********************************************/
	/*   If   we   have   the   kbd_labels   list  */
	/*   defined,  see  if  we  can  get the name  */
	/*   from  the  list in case we have a non-PC  */
	/*   based keyboard.			       */
	/***********************************************/
	if (!IS_ASCII(key) && (sp = sym_lookup("kbd_labels")) != NULL &&
		sp->s_type == F_LIST) {
		LIST	*lp = (LIST *) sp->s_obj->r_ptr;
		int	i;
		/***********************************************/
		/*   List  must  be  a sequence of int/string  */
		/*   pairs.				       */
		/***********************************************/
		while (*lp != F_HALT) {
			if (*lp != F_INT)
				goto do_normal;
			i = (int) LGET32(lp);
			lp += sizeof_atoms[F_INT];
			if (i == key)
				break;
			lp = next_atom(lp);
			}
		if (*lp != F_HALT) {
			switch (*lp) {
			  case F_STR:
			  case F_LIT:
			  	strncpy(buf, (char *) LGET32(lp), sizeof buf);
				return buf;
			  case F_RSTR:
			  	strncpy(buf, ((ref_t *) LGET32(lp))->r_ptr, sizeof buf);
				return buf;
			  }
			}
		}
	/***********************************************/
	/*   Handle normal ASCII case.		       */
	/***********************************************/
do_normal:
	if (IS_ASCII(key)) {
		key_to_char(buf, key);
		return buf;
		}
	buf[0] = '<';
	buf[1] = NULL;
	if (key & MOD_META)
		strcat(buf, "Alt-");
	if (key & MOD_CTRL)
		strcat(buf, "Ctrl-");
	if (key & MOD_SHIFT)
		strcat(buf, "Shift-");
	
	/***********************************************/
	/*   Handle function keys.		       */
	/***********************************************/
	bp = buf + strlen(buf);
	switch (key & RANGE_MASK) {
	  case RANGE_MISC:
	  	switch (key) {
		  case MOUSE_KEY:
		  	strcpy(bp, "Mouse");
			break;
		  case BACK_TAB:
		  	strcpy(bp, "Back-tab");
			break;
		  default:
		  	goto DEFAULT;
		  }
		break;
	  case RANGE_ASCII:
	  	*bp++ = (char) (key & KEY_MASK);
		*bp = NULL;
		break;
	  case RANGE_PRIVATE:
	  	sprintf(bp, "Private-%d", key & KEY_MASK);
		break;
	  case RANGE_FN:
	  	sprintf(bp, "F%d", (key & KEY_MASK) + 1);
		break;
	  case RANGE_BUTTON:
	  	key &= ~(MOD_META | MOD_CTRL | MOD_SHIFT);
	  	if (key >= BUTTON1_MOTION)
		  	sprintf(bp, "Button%d-motion", key - BUTTON1_MOTION + 1);
	  	else if (key >= BUTTON1_UP)
		  	sprintf(bp, "Button%d-up", key - BUTTON1_UP + 1);
		else
		  	sprintf(bp, "Button%d-down", key - BUTTON1_DOWN + 1);
		break;
	  case RANGE_MULTIKEY:
	  case RANGE_MULTIKEY + 0x100:
	  case RANGE_MULTIKEY + 0x200:
	  case RANGE_MULTIKEY + 0x300:
	  	/***********************************************/
	  	/*   Make  sure  we've  got  a definition for  */
	  	/*   this character seq.		       */
	  	/***********************************************/
	  	if (multi_key_seq == NULL || key >= RANGE_MULTIKEY + next_multi_seq)
			strcpy(bp, "undefined");
		else {
			cp = multi_key_seq[key - RANGE_MULTIKEY];
			/***********************************************/
			/*   Modifiers are not valid for these ones.   */
			/***********************************************/
			bp = buf;
			*bp = NULL;
			while (*cp) {
				key_to_char(bp, *cp & 0xff);
				bp += strlen(bp);
				cp++;
				}
			return buf;
			}
	  	break;
	  case RANGE_KEYPAD:
	  	i = key & KEY_MASK;
		if (i < sizeof keypad_names / sizeof keypad_names[0]) {
			cp = keypad_names[i];
			if (isdigit(*cp) || 
			    (key & KEY_MASK) >= (KEYPAD_PLUS & KEY_MASK)) {
			  	strcpy(bp, "Keypad-");
				bp += 7;
				}
			while (*cp && *cp != '-')
				*bp++ = *cp++;
			*bp = NULL;
			break;
			}
	  	strcpy(bp, "Keypad-");
		bp += 7;
		/* Fallthru.. */
	  default:
	  DEFAULT:
		sprintf(bp, "#%d", key);
		break;
	  }
	strcat(buf, ">");
	return buf;
}
/**********************************************************************/
/*   Function  to  convert a single 8-bit character into the canonic  */
/*   notation.							      */
/**********************************************************************/
char *
key_to_char(buf, key)
char	*buf;
int	key;
{
	switch (key) {
	  case ENTER:
	  	strcpy(buf, "<Enter>");
		break;
	  case ESC:
	  	strcpy(buf, "<Esc>");
	  	break;
	  case '{':
	  	strcpy(buf, "<{>");
		break;
	  case '}':
	  	strcpy(buf, "<}>");
		break;
	  case ' ':
	  	strcpy(buf, "<Space>");
		break;
	  case BACKSPACE:
	  	strcpy(buf, "<Backspace>");
		break;
	  case '<':
	  case '\\':
	  	buf[0] = '\\';
		buf[1] = key;
		buf[2] = NULL;
		break;
	  default:
		if (key < ' ') {
			sprintf(buf, "<Ctrl-%c>", key + '@');
			}
		else if (key < 0x7f) {
			buf[0] = (char) key;
			buf[1] = NULL;
			}
		else
			sprintf(buf, "#%d", key);
		break;
	  }
	buf += strlen(buf);
	return buf;
}
/**********************************************************************/
/*   Function  called  to  execute  the  macro  associated  with  an  */
/*   internal key code.						      */
/**********************************************************************/
void
exec_key(c)
int	c;
{	LISTV	*lp;
	sentry_t	*sep = NULL;
	char	*cp = "";

	/***********************************************/
	/*   Terminate  the  undo  chain.  This means  */
	/*   that  when  user hits undo, it will stop  */
	/*   at  the  point  that  the  last  key was  */
	/*   pressed.				       */
	/***********************************************/
	u_chain();
	
	/***********************************************/
	/*   Get  the  macro assigned to this key. We  */
	/*   need  to  check  the  local  keyboard if  */
	/*   there  is  one.  Otherwise  we  use  the  */
	/*   current keyboard table.		       */
	/***********************************************/
	if (curbp && curbp->b_keyboard) {
		sep = st_lookup(curbp->b_keyboard->kt_macros, c);
		}
	/***********************************************/
	/*   If  we  haven't  found  it  yet, then go  */
	/*   for the current keyboard.		       */
	/***********************************************/
	if (sep == NULL)
		sep = st_lookup(cur_kp->kt_macros, c);

	if (sep) {
		lp = (LISTV *) sep->se_ptr;
		switch (lp->l_flags) {
		  case F_LIT:
		  case F_STR:
		  	cp = lp->l_str;
			break;
		  case F_RSTR:
		  	cp = lp->l_ref->r_ptr;
			break;
		  case F_NULL:
		  	cp = "self_insert";
		  	break;
		  default:
		  	panic("exec_key: what?");
		  }
		}
	else
		cp = "";
	/***********************************************/
	/*   Save  the  internal  key code in case we  */
	/*   call self_insert.			       */
	/***********************************************/
	character = (char) c;

	/***********************************************/
	/*   If  we're  recording keystrokes then log  */
	/*   the macros being called as well.	       */
	/***********************************************/
	remember_macro(cp);

	/***********************************************/
	/*   Remember  name  of last global macro for  */
	/*   the inq_command() macro.		       */
	/***********************************************/
	if (cp[0] != '_')
		strcpy(user_call, cp);

	trace_log("\nKEY_EXEC 0x%04x => %s\n", c, cp);
	str_exec(cp);
}
/**********************************************************************/
/*   inq_command()  primitive  --  returns name of last macro called  */
/*   as a result of user hitting a key.				      */
/**********************************************************************/
void
inq_command()
{
	acc_assign_str(user_call, -1);
}
/**********************************************************************/
/*   Return current keyboard identifier.			      */
/**********************************************************************/
void
inq_keyboard()
{
	acc_assign_int((long) (cur_kp ? cur_kp->kt_id : -1));
}
/**********************************************************************/
/*   Return identifier associated with current local keyboard.	      */
/**********************************************************************/
void
inq_local_keyboard()
{
	acc_assign_int((long) (curbp->b_keyboard ? 
			curbp->b_keyboard->kt_id : 0));
}
/**********************************************************************/
/*   Function  called  whilst  we  are assembling a keystroke to see  */
/*   if  we  have  the  full key sequence yet. Return key code if we  */
/*   have  an  unambiguous keystroke. If we have an ambiguity, check  */
/*   for  a  multikey  press,  and  set multi_key if so. This way we  */
/*   can force the caller to wait until the ambiguity is resolved.    */
/**********************************************************************/
int
check_key(sbuf, multi_key, no_ambig)
unsigned short *sbuf;
int	*multi_key;
int	no_ambig;
{	char	buf[32];
	SPBLK	*sp;
	SPBLK	*sp_first;
	int	ambig;
	int	key;
	char	*cp;
	
	/***********************************************/
	/*   Need  to  convert  the  string of 16-bit  */
	/*   bytes  to  a  string  of  8-bit bytes so  */
	/*   that the splay tree stuff will work.      */
	/***********************************************/
	for (cp = buf; *sbuf; )
		*cp++ = (char) *sbuf++;
	*cp = NULL;
	*multi_key = FALSE;
	sp = sp_partial_lookup(buf, kseq_tree, &ambig, &sp_first);
	if (sp == NULL) {
		if (sp_first) {
			key = ((keyseq_t *) sp_first->data)->ks_code;
			if (IS_MULTIKEY(key))
				*multi_key = TRUE;
			}
		return ambig ? -1 : 0;
		}
	/***********************************************/
	/*   If  caller  not  worried about ambiguity  */
	/*   then  forget  that  because he must have  */
	/*   timed out.				       */
	/***********************************************/
	if (no_ambig)
		ambig = 0;
	return ambig ? -1 : ((keyseq_t *) sp->data)->ks_code;
}
/**********************************************************************/
/*   keyboard_push primitive. Save current keyboard on a stack.	      */
/**********************************************************************/
void
kbd_push()
{	register int	i = argv[1].l_int;
	register keytree_t *kp;

	/***********************************************/
	/*   If  non-zero  argument  specified,  then  */
	/*   we   want   to   push  an  existant  old  */
	/*   keyboard on the stack.		       */
	/***********************************************/
	if (i) {
		kp = find_keyboard(i, TRUE);
		if (kp == NULL) {
			/***********************************************/
			/*   Couldn't find the keyboard at all.	       */
			/***********************************************/
			errorf("keyboard_push: %d not found.", i);
			return;
			}
		}
	else
		kp = new_keyboard();
		
	ll_push(hd_kbd, (char *) kp);
	cur_kp = kp;
}
/**********************************************************************/
/*   Routine  to  look  for  a  keyboard.  Its  either on the pushed  */
/*   stack  or  the  popped  stack.  If inc_ref is TRUE, then we are  */
/*   creating  a  new reference to it. Otherwise we're just going to  */
/*   look at it.						      */
/**********************************************************************/
keytree_t *
find_keyboard(id, inc_ref)
int	id;
int	inc_ref;
{	register List_p	lp;
	register keytree_t *kp;
	
	/***********************************************/
	/*   See  if  we can find the keyboard on the  */
	/*   main stack.			       */
	/***********************************************/
	for (lp = ll_first(hd_kbd); lp; lp = ll_next(lp)) {
		kp = (keytree_t *) ll_elem(lp);
		if (kp->kt_id == id) {
			if (inc_ref)
				kp->kt_ref++;
			return kp;
			}
		}
	/***********************************************/
	/*   If  we  didn't find it on the main stack  */
	/*   try the popped stack.		       */
	/***********************************************/
	for (lp = ll_first(hd_kstk); lp; lp = ll_next(lp)) {
		kp = (keytree_t *) ll_elem(lp);
		if (kp->kt_id == id) {
			if (inc_ref)
				ll_delete(lp);
			return kp;
			}
		}
	return NULL;
}
/**********************************************************************/
/*   keyboard_pop   primitive   -   pop  keyboard  possible  keeping  */
/*   current one for later reuse.				      */
/**********************************************************************/
void
kbd_pop()
{	register int	i = argv[1].l_int;
	List_p lp = ll_first(hd_kbd);
	register keytree_t *kp;
	
	/***********************************************/
	/*   If no keyboard active just give up.       */
	/***********************************************/
	if (lp == NULL)
		return;
	/***********************************************/
	/*   Remove   current   keyboard   from   the  */
	/*   keyboard stack.			       */
	/***********************************************/
	kp = (keytree_t *) ll_elem(lp);
	(void) ll_pop(hd_kbd);

	/***********************************************/
	/*   See   if  user  wants  to  discard  this  */
	/*   keymap.				       */
	/***********************************************/
	if (i == 0) {
		/***********************************************/
		/*   Free  up  the  memory  for  the keyboard  */
		/*   table if its the last reference.	       */
		/***********************************************/
		free_keyboard(kp);
		}
	else {
		/***********************************************/
		/*   User  wants  to  keep the keyboard table  */
		/*   so save it here.			       */
		/***********************************************/
		ll_append(hd_kstk, (char *) kp);
		}
	lp = ll_first(hd_kbd);
	cur_kp = lp ? (keytree_t *) ll_elem(lp) : NULL;
}
/**********************************************************************/
/*   Macro primitive to flush any keyboard input.		      */
/**********************************************************************/
void
kbd_flush()
{	int	flag16;

	while (get_typeahead(&flag16) >= 0)
		;
}
/**********************************************************************/
/*   use_local_keyboard  primitive  --  cause current keyboard to be  */
/*   associated with the current buffer.			      */
/**********************************************************************/
void
use_local_keyboard()
{
	register keytree_t *kp;
	register List_p	lp;

	acc_assign_int(0L);
	if (argv[1].l_int == 0) {
		detach_local_keyboard(curbp);
		return;
		}
	for (lp = ll_first(hd_kstk); lp; lp = ll_next(lp)) {
		kp = (keytree_t *) ll_elem(lp);
		if (kp->kt_id == (int) argv[1].l_int) {
			kp->kt_ref++;
			if (curbp->b_keyboard)
				detach_local_keyboard(curbp);
			curbp->b_keyboard = kp;
			return;
			}
		}
	acc_assign_int(-1L);
}
/**********************************************************************/
/*   Function to remove a reference to a local keyboard.	      */
/**********************************************************************/
void
detach_local_keyboard(bp)
BUFFER	*bp;
{
	if (bp && bp->b_keyboard) {
		free_keyboard(bp->b_keyboard);
		bp->b_keyboard = NULL;
		}
}
/**********************************************************************/
/*   Function  to  free  a  keyboard  if the reference count goes to  */
/*   zero.							      */
/**********************************************************************/
void
free_keyboard(kp)
keytree_t	*kp;
{	register int	j;
	sentry_t	*sep;

	if (--kp->kt_ref <= 0) {
		/***********************************************/
		/*   Free  up  memory  allocated to the macro  */
		/*   strings.				       */
		/***********************************************/
		sep = kp->kt_macros->st_block;
		for (j = kp->kt_macros->st_used; j-- > 0; sep++)
			key_free_macro((LISTV *) sep->se_ptr);
		st_free(kp->kt_macros);
		chk_free((char *) kp);
		}
}
/**********************************************************************/
/*   inq_assignment  primitive  --  returns names of macros assigned  */
/*   to keys or keys assigned to macros.			      */
/**********************************************************************/
void
inq_assignment()
{
	int	from_key = argv[2].l_int == 0;
	char	*cp;
	
	if (from_key) {
		cp = find_macro_name((int) (argv[1].l_flags == F_INT 
			? argv[1].l_int
			: key_string_to_code(get_str(1))));
		acc_assign_str(cp, -1);
		return;
		}
	acc_assign_str("", 0);
	if (curbp->b_keyboard)
		find_assignment(curbp->b_keyboard);
	find_assignment(cur_kp);

	if (acc_get_sval()[0] == NULL)
		acc_assign_str("nothing", 7);
}
/**********************************************************************/
/*   Function  to  find  the  name of a macro bound to a key. Called  */
/*   by  inq_assignment()  and  when we get a mouse paste action (we  */
/*   need to check if <Ins> is bound to the paste macro.	      */
/**********************************************************************/
char *
find_macro_name(key)
int	key;
{
	LISTV	*lvp;
	sentry_t *sep;
	char	*cp;

	/***********************************************/
	/*   If  its  a valid keycode check the local  */
	/*   keyboard    and   thence   the   current  */
	/*   keyboard for the mapping.		       */
	/***********************************************/
	sep = NULL;
	if (curbp->b_keyboard)
		sep = st_lookup(curbp->b_keyboard->kt_macros, 
			(unsigned long) key);
	if (sep == NULL)
		sep = st_lookup(cur_kp->kt_macros, (unsigned long) key);
	if (sep == NULL)
		return "nothing";
	lvp = (LISTV *) sep->se_ptr;
	cp = get_listv_str(lvp);
	if (cp == NULL) {
		character = (char) key;
		return "self_insert";
		}
	return cp;
}	
/**********************************************************************/
/*   Function  called  by inq_assignment() to look for keys assigned  */
/*   by a particular macro.					      */
/**********************************************************************/
void
find_assignment(kp)
keytree_t	*kp;
{	char	buf[BUFSIZ];
	char	*cp;
	char	*sp;
	char	*str1 = get_str(1);
	register int j;
	register sentry_t *sep;

	strcpy(buf, acc_get_sval());
	sp = buf + strlen(buf);

	sep = kp->kt_macros->st_block;
	for (j = kp->kt_macros->st_used; j-- > 0; sep++) {
		cp = get_listv_str((LISTV *) sep->se_ptr);
		if (strcmp(cp, str1) == 0) {
			if (sp != buf) {
				strcpy(sp, "<-also>");
				sp += 7;
				}
			strcpy(sp, code_to_keyname(sep->se_key));
			sp += strlen(sp);
			}
		}
	acc_assign_str(buf, -1);
}

/**********************************************************************/
/*   Function  to  return  a  pointer to the string described by the  */
/*   argument.							      */
/**********************************************************************/
char *
get_listv_str(lvp)
LISTV	*lvp;
{

	switch (lvp->l_flags) {
	  case F_LIT:
	  case F_STR:
	  	return lvp->l_str;
	  case F_RSTR:
	  	return lvp->l_ref->r_ptr;
	  case F_NULL:
	  	return "self_insert";
	  default:
	  	panic("get_listv_str: what?");
	  }
	return NULL;
}
/*******************************************************************/
/*   key_list primitive -- return list of key bindings.		   */
/*******************************************************************/
void
key_list()
{	LIST	*new_list;
	int	new_len;
	register LIST *lp;
	LISTV	*lvp;
	register sentry_t *sep, *sep1, *sepend, *sepend1;
	int	do_self_inserts = argv[2].l_flags == F_INT && argv[2].l_int != 0;
	char	*cp;
	keytree_t	*kp, *kp1;
	int	id;
	
	kp1 = NULL;
	/***********************************************/
	/*   If  we  get  passed a buffer ID then use  */
	/*   the local keyboard on that buffer.	       */
	/***********************************************/
	if (argv[3].l_flags != F_NULL) {
		BUFFER *bp = numberb((int) argv[3].l_int);
		kp1 = bp ? bp->b_keyboard : NULL;
		}
	/***********************************************/
	/*   If  keybd  id  not  specified  then  use  */
	/*   local  keyboard  if  available otherwise  */
	/*   go for the current keyboard.	       */
	/***********************************************/
	if (argv[1].l_flags == F_NULL) {
		kp = cur_kp;
		if (kp1 == NULL)
			kp1 = curbp ? curbp->b_keyboard : NULL;
		}
	else {
		/***********************************************/
		/*   See  if  we can find the keyboard on the  */
		/*   main stack.			       */
		/***********************************************/
		id = (int) argv[1].l_int;
		kp = find_keyboard(id, FALSE);
		if (kp == NULL) {
			acc_assign_null();
			return;
			}
		}
		
	if (kp == NULL && kp1 == NULL) {
		acc_assign_null();
		return;
		}
		
	/***********************************************/
	/*   Work  out  how  much  space  we need for  */
	/*   the key table.			       */
	/***********************************************/
	new_len = 0;
	if (kp && kp1) {
		sep = kp->kt_macros->st_block;
		sep1 = kp1->kt_macros->st_block;
		sepend = &sep[kp->kt_macros->st_used];
		sepend1 = &sep1[kp1->kt_macros->st_used];
		}
	else if (kp) {
		sep1 = sep = kp->kt_macros->st_block;
		sepend = sepend1 = &sep[kp->kt_macros->st_used];
		}
	else {
		sep = sep1 = kp1->kt_macros->st_block;
		sepend = sepend1 = &sep[kp1->kt_macros->st_used];
		}
	while (sep < sepend || sep1 < sepend1) {
		if (do_self_inserts) {
			new_len += sizeof_atoms[F_LIT];
			}
		else {
			if (sep >= sepend)
				lvp = (LISTV *) sep1->se_ptr;
			else if (sep1 >= sepend1)
				lvp = (LISTV *) sep->se_ptr;
			else if (sep->se_key >= sep1->se_key)
				lvp = (LISTV *) sep1->se_ptr;
			else
				lvp = (LISTV *) sep->se_ptr;
			if (lvp->l_flags != F_NULL)
				new_len += sizeof_atoms[F_LIT];
			}
		if (sep >= sepend)
			sep1++;
		else if (sep1 >= sepend1)
			sep++;
		else if (sep->se_key == sep1->se_key) {
			sep++;
			sep1++;
			}
		else if (sep->se_key > sep1->se_key)
			sep1++;
		else
			sep++;
		}
		
	/***********************************************/
	/*   Allocate memory for list.		       */
	/***********************************************/
	new_len = 2 * new_len + sizeof_atoms[F_HALT];
	new_list = (LIST *) chk_alloc(new_len);
	if (new_list == NULL) {
		acc_assign_null();
		return;
		}
		
	lp = new_list;
	if (kp && kp1) {
		sep = kp->kt_macros->st_block;
		sep1 = kp1->kt_macros->st_block;
		}
	else if (kp) {
		sep1 = sep = kp->kt_macros->st_block;
		}
	else {
		sep = sep1 = kp1->kt_macros->st_block;
		}
	while (sep < sepend || sep1 < sepend1) {
		int	key;
		if (sep >= sepend) {
			lvp = (LISTV *) sep1->se_ptr;
			key = sep1->se_key;
			}
		else if (sep1 >= sepend1) {
			lvp = (LISTV *) sep->se_ptr;
			key = sep->se_key;
			}
		else if (sep->se_key >= sep1->se_key) {
			lvp = (LISTV *) sep1->se_ptr;
			key = sep1->se_key;
			}
		else {
			lvp = (LISTV *) sep->se_ptr;
			key = sep->se_key;
			}
		if (lvp->l_flags == F_NULL) {
			if (!do_self_inserts)
				goto incr_ptrs;
			}
		cp = code_to_keyname(key);
		*lp = F_STR;
		LPUT32(lp, (long) strdup(cp));
		lp += sizeof_atoms[F_STR];
		*lp = lvp->l_flags;
		switch (*lp) {
		  case F_LIT:
		  	LPUT32(lp, (long) lvp->l_str);
			break;
		  case F_RSTR:
		  	LPUT32(lp, (long) r_inc(lvp->l_ref));
			break;
		  case F_STR:
		  	*lp = F_RSTR;
		  	LPUT32(lp, (long) r_init(F_STR, lvp->l_str, strlen(lvp->l_str)));
			break;
		  case F_NULL:
		  	*lp = F_LIT;
			LPUT32(lp, (long) "self_insert");
		  	break;
		  default:
		  	panic("key_list: what?");
			break;
		  }
		lp += sizeof_atoms[*lp];
incr_ptrs:
		if (sep >= sepend)
			sep1++;
		else if (sep1 >= sepend1)
			sep++;
		else if (sep->se_key == sep1->se_key) {
			sep++;
			sep1++;
			}
		else if (sep->se_key > sep1->se_key)
			sep1++;
		else
			sep++;
		}
	*lp = F_HALT;
	acc_donate_list(new_list, new_len);
}
/**********************************************************************/
/*   Macro to implement the copy_keyboard primitive.		      */
/**********************************************************************/
void
copy_keyboard()
{	keytree_t	*kp;
	LIST		*lp = argv[2].l_list;
	char		*name;
	sentry_t	*sep, *sepend;
	LISTV		*lvp;
	LISTV		lbuf;
	char		*cp;
	
	kp = find_keyboard((int) argv[1].l_int, FALSE);
	if (kp == NULL || kp == cur_kp) {
		acc_assign_int((long) 0);
		return;
		}
	acc_assign_int((long) cur_kp->kt_id);
	
	sepend = &kp->kt_macros->st_block[kp->kt_macros->st_used];
	
	for (; lp && *lp != F_HALT; lp = next_atom(lp)) {
		name = get_str_ptr(lp);
		if (name == NULL)
			continue;
		/***********************************************/
		/*   Perform  a  linear  search  to  see what  */
		/*   key its mapped to.			       */
		/***********************************************/
		for (sep = kp->kt_macros->st_block; sep < sepend; sep++) {
			lvp = (LISTV *) sep->se_ptr;
			cp = getv_str(lvp);
			if (cp == NULL)
				continue;
			if (*cp == *name && strcmp(cp, name) == 0) {
				lbuf.l_flags = lvp->l_flags;
				switch (lvp->l_flags) {
				  case F_LIT:
				  	lbuf.l_str = lvp->l_str;
					break;
				  case F_STR:
				  	lbuf.l_str = strdup(lvp->l_str);
					break;
				  case F_RSTR:
				  	lbuf.l_ref = r_inc(lvp->l_ref);
					break;
				  default:
				  	break;
				  }
				key_add_macro(sep->se_key, &lbuf);
				}
			}
		}
}
/**********************************************************************/
/*   Primitive to return the last mouse x,y co-ordinates.	      */
/**********************************************************************/
void
get_mouse_pos()
{
	argv_assign(1, (long) fetch_key_x);
	argv_assign(2, (long) fetch_key_y);
}
