/* GNU Mailutils -- a suite of utilities for electronic mail
   Copyright (C) 1999, 2000, 2002, 2003,
   2004 Free Software Foundation, Inc.

   GNU Mailutils is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   GNU Mailutils is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with GNU Mailutils; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA  */

 /**
  *
  * Created as an example for using mailutils API
  * Sean 'Shaleh' Perry <shaleh@debian.org>, 1999
  * Alain Magloire alainm@gnu.org
  *
  **/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <mailutils/address.h>
#include <mailutils/argp.h>
#include <mailutils/debug.h>
#include <mailutils/errno.h>
#include <mailutils/header.h>
#include <mailutils/list.h>
#include <mailutils/mailbox.h>
#include <mailutils/message.h>
#include <mailutils/registrar.h>
#include <mailutils/stream.h>
#include <mailutils/nls.h>
#include <mailutils/tls.h>
#include <mailutils/mutil.h>
#include <mailutils/error.h>
#include <mailutils/mime.h>

const char *program_version = "from (" PACKAGE_STRING ")";
static char doc[] = N_("GNU from -- display from and subject");

static struct argp_option options[] = {
  {"debug",  'd', NULL,   0, N_("Enable debugging output"), 0},
  {0, 0, 0, 0}
};

static int debug;

static error_t
parse_opt (int key, char *arg, struct argp_state *state)
{
  switch (key)
    {
    case 'd':
      debug++;
      break;

    default: 
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static struct argp argp = {
  options,
  parse_opt,
  N_("[URL]"),
  doc,
};

static const char *capa[] = {
  "common",
  "license",
  "mailbox",
#ifdef WITH_TLS
  "tls",
#endif
  NULL
};

static void
from_rfc2047_decode (char *buf, size_t buflen)
{
  char *charset = NULL;
  char *tmp;
  int rc;

  /* Try to deduce the charset from LC_ALL or LANG variables */

  tmp = getenv ("LC_ALL");
  if (!tmp)
    tmp = getenv ("LANG");

  if (tmp)
    {
      char *sp;
      char *lang;
      char *terr;

      lang = strtok_r (tmp, "_", &sp);
      terr = strtok_r (NULL, ".", &sp);
      charset = strtok_r (NULL, "@", &sp);

      if (!charset)
	charset = mu_charset_lookup (lang, terr);
    }

  if (!charset)
    return;

  rc = rfc2047_decode (charset, buf, &tmp);
  if (rc)
    {
      if (debug)
	mu_error (_("Can't decode line `%s': %s"),
		  buf, mu_strerror (rc));
    }
  else
    {
      strncpy (buf, tmp, buflen - 1);
      free (tmp);
    }
}

int
main (int argc, char **argv)
{
  mailbox_t mbox;
  size_t i;
  size_t count = 0;
  char *mailbox_name = NULL;
  /* Arbitrary limits.  A better approach would be to allocate
     as we go along but it is not worth the trouble.  */
  char buf[128];
  char personal[128];
  int status;

  /* Native Language Support */
  mu_init_nls ();

  {
    int opt;
    mu_argp_init (program_version, NULL);
#ifdef WITH_TLS
    mu_tls_init_client_argp ();
#endif
    mu_argp_parse (&argp, &argc, &argv, 0, capa, &opt, NULL);
    mailbox_name = argv[opt];
  }

  /* Register the desire formats.  */
  mu_register_all_mbox_formats ();

  status = mailbox_create_default (&mbox, mailbox_name);
  if (status != 0)
    {
      mu_error (_("Couldn't create mailbox <%s>: %s."),
		mailbox_name ? mailbox_name : _("default"),
		mu_strerror (status));
      exit (1);
    }

  /* Debuging Trace.  */
  if (debug)
  {
    mu_debug_t debug;
    mailbox_get_debug (mbox, &debug);
    mu_debug_set_level (debug, MU_DEBUG_TRACE|MU_DEBUG_PROT);
  }

  status = mailbox_open (mbox, MU_STREAM_READ);
  if (status != 0)
    {
      mu_error (_("Couldn't open mailbox <%s>: %s."),
		mailbox_name, mu_strerror (status));
      exit (1);
    }

  mailbox_messages_count (mbox, &count);
  for (i = 1; i <= count; ++i)
    {
      message_t msg;
      header_t hdr;
      size_t len = 0;

      if ((status = mailbox_get_message (mbox, i, &msg)) != 0
	  || (status = message_get_header (msg, &hdr)) != 0)
	{
	  mu_error (_("msg %d : %s"), i, mu_strerror (status));
	  exit (2);
	}

      header_get_value (hdr, MU_HEADER_FROM, buf, sizeof (buf), &len);
      if (len != 0)
	{
	  address_t address = NULL;
	  len = 0;

	  from_rfc2047_decode (buf, sizeof (buf));
	  address_create (&address, buf);
	  address_get_personal (address, 1, personal,
				sizeof (personal), &len);
	  printf ("%s\t", (len != 0) ? personal : buf);
	  address_destroy (&address);
	}
      else
	{
	  status = header_get_value (hdr, MU_HEADER_TO, buf,
				     sizeof (buf), &len);
	  if (status == 0)
	    {
	      from_rfc2047_decode (buf, sizeof (buf));
	      printf ("%s\t", buf);
	    }
	}

      status = header_get_value_unfold (hdr, MU_HEADER_SUBJECT,
					buf, sizeof (buf), NULL);
      if (status == 0)
	{
	  from_rfc2047_decode (buf, sizeof (buf));
	  printf ("%s\n", buf);
	}
    }

  status = mailbox_close (mbox);
  if (status != 0)
    {
      mu_error (_("Couldn't close <%s>: %s."),
		mailbox_name, mu_strerror (status));
    }

  mailbox_destroy (&mbox);
  return 0;
}
