;=========================================================================
;
;   CLOCK.SYS                            03-13-95
;
;   CLOCK$ Device driver.  For use on AT-type machines only.
;   Like the driver in PC DOS v2.00 and v2.10, except that:
; 
;   1. The `midnight' bug is fixed.
;   2. The real-time clock date is read.
;   3. The CONFIG.SYS command line is read to get the correct 
;      local time zone.
;
;   It is assumed that the CMOS Clock is set to UCT (GMT).
;
;   Copy CLOCK.SYS to PATH and enter something like
;              DEVICE = PATH\CLOCK.SYS /TZ=+5
;   in your CONFIG.SYS file.  
;
;   The value of TZ must be an integer, -12 <= TZ <= 12
;
;   Examples: 
;      Eastern Standard Time (U.S.)  (EST)  TZ = +5
;      Eastern Daylight Saving Time  (EDT)  TZ = +4
;      Pacific Standard Time (U.S.)  (PST)  TZ = +8
;      Pacific Daylight Saving Time  (PDT)  TZ = +7
;
;      Eastern Standard Time (U.S.) is 5 hours BEHIND UCT.
;      Its OFFSET, relative to UCT, is therefore -5 H.
;      However, TZ is defined to be the number of hours that
;      must be added to local time to obtain UCT. That is,
;      TZ = -OFFSET.
;
;   Don't forget that the DOS time function sets the CMOS clock,
;   as well as DOS time. So, if you use it, you should set the
;   time to UCT. Then, if you want DOS to show the correct local
;   time, you should reboot.
;
;   Bugs: 
;
;      Daylight Saving Time is not accounted for. You will still
;      have to edit your CONFIG.SYS file at the appropriate times.
;
;      Fractional Time Zones are not implemented. That is, CLOCK.SYS
;      cannot handle Newfoundland.
;
;      CLOCK.SYS has been tested only under MS-DOS ver. 5.0a. 
;      The DOS version is not checked.
;
;   Assemble with:   
;      masm clock.asm
;      link clock.obj,,,;
;      exe2bin clock.exe clock.sys
;
;   Original program ATCLKFIX.ASM by Mike Geyer. See ATCLKFIX.DOC.
;
;   Time Zone feature added by Dr. Eugene Zaustinsky  
;                              Department of Mathematics
;                              State University of New York
;                              Stony Brook, NY 11794
;
;   Please send comments and bug reports to:  ezaust@math.sunysb.edu
;
;=========================================================================

;******** Prototype for Request Header
REQ_HDR      STRUC
RH_LEN       DB      ?         ; Offset 0
RH_UNIT_CODE DB      ?         ; Offset 1
RH_CMD_CODE  DB      ?         ; Offset 2
RH_STATUS    DW      ?         ; Offset 3
RH_RESERVED  DB      8 DUP(?)  ; Offset 5
RH_MEDIA     DB      ?         ; Offset 13
RH_ADDR_OFF  DW      ?         ; Offset 14
RH_ADDR_SEG  DW      ?         ; Offset 16
RH_BYTPSEC   DW      ?         ; Offset 18 Offset  Config.sys cmd line
RH_CD_SEG    DW      ?         ; Offset 20 Segment Config.sys cmd line
             DB      ?         ; Offset 22
REQ_HDR      ENDS

;******** Prototype for Clock Data
CLOCK_DATA   STRUC
CD_DAYS      DW      ?
CD_HR_MIN    DW      ?
CD_SECONDS   DW      ?
CLOCK_DATA   ENDS

;******** Equates
ZERO         EQU     0
SIXTY        EQU     60
HUNDRED      EQU     100
CON59        EQU     59659               ; 59659 / (5*64K) ticks/100th-sec
CR           EQU     0DH
LF           EQU     0AH

;==========================================================================
;
;            Executable Code
;
;==========================================================================

DEVSEG       SEGMENT PARA PUBLIC 'CODE'

             ASSUME  CS:DEVSEG, DS:DEVSEG
             ORG     ZERO

DEVPROC      PROC    FAR

;==========================================================================
;            Data Storage Area
;==========================================================================

;******** Header for DOS Device Drivers
NEXT_DEVIP   DW      -1                  ; fake pointer to ..
NEXT_DEVCS   DW      -1                  ; .. next device driver
ATTRIBUTE    DW      8008H               ; Char device and current clock
STRAT_PTR    DW      OFFSET STRATEGY
INTRP_PTR    DW      OFFSET COMMANDS
DEV_NAME     DB      'CLOCK$  '
;
;******** Pointer to DOS Request Header
REQ_HDR_PTR  LABEL   DWORD
 RHP_OFF     DW      ZERO
 RHP_SEG     DW      ZERO
;
;******** Numerical data
DAYCOUNT     DW      ZERO                ; 0 corresponds to 01 Jan 80
NUMTICKSHI   DW      ZERO
;
;******** Jump Table for Interrupt Routine Functions
FN_TABLE     DW      INITIALIZE          ;  0: Initialization Procedure
             DW      ERROR_EXIT          ;  1: Media Check Procedure
             DW      ERROR_EXIT          ;  2: Build BIOS Parameter Block
             DW      ERROR_EXIT          ;  3: IOCTL Input
             DW      READ_DEV            ;  4: Read from device
             DW      READ_EXIT           ;  5: Non-destructive no-wait read
             DW      GOOD_EXIT           ;  6: Return status of input device
             DW      GOOD_EXIT           ;  7: Flush input buffer
             DW      WRITE_DEV           ;  8: Write to device
             DW      WRITE_DEV           ;  9: Write and verify
             DW      GOOD_EXIT           ; 10: Return status of output device
             DW      GOOD_EXIT           ; 11: Flush output buffer

;==========================================================================
;            Device Strategy Routine
;            External Entry Point
;==========================================================================

             ASSUME  CS:DEVSEG, DS:NOTHING, ES:NOTHING

STRATEGY     PROC    FAR
             MOV     CS:RHP_OFF,BX
             MOV     CS:RHP_SEG,ES
             RETF
STRATEGY     ENDP


;==========================================================================
;            Interrupt Handler Routine
;            External Entry Point
;==========================================================================

             ASSUME  CS:DEVSEG, DS:NOTHING, ES:NOTHING

COMMANDS     PROC    FAR

;=======================================
;======== Main Routine
;=======================================
;
;******** Common setup for all functions
;-------- Clear direction
             CLD
;-------- Push registers
             PUSH    AX
             PUSH    BX
             PUSH    CX
             PUSH    DX
             PUSH    SI
             PUSH    DI
             PUSH    BP
             PUSH    DS
             PUSH    ES
;-------- Make DS:BX point at Request Header
             LDS     BX,REQ_HDR_PTR
;
;******** Processing based on Request Header command code
;-------- Place the command code in AL
             MOV     AL,[BX].RH_CMD_CODE
;-------- Branch and exit if command code is greater than 11
             CMP     AL,0BH
             JA      ERROR_EXIT
;-------- Load CX from Request Header `bytes per sector' field
             MOV     CX,[BX].RH_BYTPSEC
;-------- Load ES:DI from Request Header `address' field
             LES     DI,DWORD PTR [BX].RH_ADDR_OFF
;-------- Point DS at this segment
             PUSH    CS
             POP     DS
             ASSUME  DS:DEVSEG
;-------- Jump to routine called for in command code
             MOV     SI,OFFSET FN_TABLE
             MOV     AH,ZERO
             SHL     AX,1
             ADD     SI,AX
             JMP     WORD PTR [SI]
;
;******** Common exit from Interrupt Handler routine
;-------- Assign result status and branch to STOR_STATUS
READ_EXIT:   MOV     AX,0300H            ; Code for `done and busy'
             JMP     SHORT STOR_STATUS
;-------- Assign bad result status and branch to STOR_STATUS
ERROR_EXIT:  MOV     AX,8103H            ; Code for `unknown command'
             JMP     SHORT STOR_STATUS
;-------- Assign result status
GOOD_EXIT:   MOV     AX,100H             ; Code for `done'
;-------- Store result status in Request Header
STOR_STATUS: LDS     BX,REQ_HDR_PTR
             MOV     [BX].RH_STATUS,AX
;-------- Restore registers
             POP     ES
             POP     DS
             POP     BP
             POP     DI
             POP     SI
             POP     DX
             POP     CX
             POP     BX
             POP     AX
;-------- Return
             RETF

;=======================================
;======== `Write to Device' Processing routine
;=======================================
;
;         Input: 6 byte block provided by DOS and pointed to by
;                RH_ADDR_SEG:RH_ADDR_OFF (which has already been
;                placed in ES:DI).  The 6 byte block has the format
;                of CLOCK_DATA.
;         Processing: Saves input days info in location DAYCOUNT.
;                     Converts input hours/minutes/seconds info to the
;                     number of 55 msec ticks since midnight, as follows:
;                        a) Convert hr/min/sec to hundredths-of-seconds
;                        b) Multiply by 59659 / (5*64K) = 0.182
;         Output: Writes the number of 55 msec ticks to BIOS Data Area
;                 (at 0040:006C) using BIOS Interrupt 1AH.
;
WRITE_DEV    LABEL   NEAR
;
;******** Move data from DOS to this program
;-------- Place days since 1/1/80 in DAYCOUNT
             MOV     AX,ES:[DI].CD_DAYS
             MOV     DAYCOUNT,AX
;-------- Place hours in CH, minutes in CL
             MOV     CX,ES:[DI].CD_HR_MIN
;-------- Place hundredth of seconds in DL, seconds in DH
             MOV     DX,ES:[DI].CD_SECONDS
;
;******** Convert hr/min/sec/100ths to hundredths-of-seconds, and put in DX,CX
;-------- Convert from hr:min to total minutes, and place in AX
             MOV     AL,SIXTY
             MUL     CH
             MOV     CH,ZERO
             ADD     AX,CX
;-------- Convert total minutes to hundredth of seconds, and place in DX,CX
             MOV     CX,SIXTY*HUNDRED    ; Hundredths-of-seconds per minute
             MOV     BX,DX               ; Stores CD_SECONDS in BX
             MUL     CX
             MOV     CX,AX
;-------- Convert seconds to hundredths-of-seconds and add to CX
             MOV     AL,HUNDRED          ; Hundredths-of-seconds per second
             MUL     BH
             ADD     CX,AX
             ADC     DX,ZERO             ; Take care of the carry
;-------- Add in the hundredths-of-seconds
             MOV     BH,ZERO
             ADD     CX,BX
             ADC     DX,ZERO
;
;******** Multiply DX,CX by 59,659 and discard lowest word of three-word result
;-------- Place hundredths-of-seconds in CX,AX and free DX
             XCHG    DX,AX
             XCHG    CX,AX
;-------- Multiply hundredths-of-seconds low-word in AX by 59,659
             MOV     BX,CON59
             MUL     BX
;-------- Place results in CX,DX, and hundredths-of-second high-word in AX
             XCHG    DX,CX
             XCHG    DX,AX
;-------- Multiply hundredths-of-seconds high-word by 59,659, over-writing DX
             MUL     BX
;-------- Add in most significant half of low-word results
             ADD     AX,CX
             ADC     DX,ZERO
;-------- Place result in AX,DX -- new high- and low-words
             XCHG    DX,AX
;
;******** Divide AX,DX by 5, and place result in CX,DX
;-------- Divide AX by 5 (byte operation)
             MOV     BX,5
             DIV     BL
;-------- Save quotient in CX (high-word of tick count)
             MOV     CL,AL
             MOV     CH,ZERO
;-------- Place remainder in DX, low-word in AX
             MOV     AL,AH
             CBW
             XCHG    DX,AX
;-------- Divide DX,AX by 5 (word operation)
             DIV     BX
;-------- Place quotient in DX (low-word of tick count), over-writing remainder
             MOV     DX,AX
;
;******** Save result and exit
;-------- Save the high-word part of tick count
             MOV     NUMTICKSHI,CX
;-------- Invoke BIOS routine to set the tick count (now in CX,DX)
             MOV     AH,01H
             INT     1AH
;-------- Jump to common exit
             JMP     GOOD_EXIT

;=======================================
;======== Read Processing routine
;=======================================
;
;         Input: Two-word block (provided by BIOS Interrupt 1AH) containing
;                the number of 55 msec ticks currently in the BIOS Data
;                Area at 0040:006C
;         Processing: Updates internal variable DAYCOUNT if midnight passed.
;                     Converts number of 55 msec clock ticks to hours/minutes/
;                     seconds/100ths info as follows:
;                        a) Multiply by (5*64K) / 59659 = 1 / 0.182
;                        b) Convert hundredths-of-seconds to hr/min/sec/100ths
;         Output: Information is placed in 6 byte block pointed to by
;                 RH_ADDR_SEG:RH_ADDR_OFF -- block has the format of
;                 CLOCK_DATA.
;
READ_DEV     LABEL   NEAR
;
;******** Get tick count into CX,DX, and increment DAYCOUNT if midnight passed
;-------- Invoke BIOS routine to read the tick count into CX,DX
             XOR     AX,AX
             INT     1AH
;-------- If NUMTICKSHI has decreased since last saved, then increment DAYCOUNT
             CMP     CX,NUMTICKSHI
             JNB     READ_1
             INC     DAYCOUNT
;-------- Save the high-word part of tick count
READ_1:      MOV     NUMTICKSHI,CX
;
;******** Multiply CX,DX by 5 and put result in DX,AX
;-------- Save tick count in AX,BX
             MOV     AX,CX
             MOV     BX,DX
;-------- Multiply CX,DX by 2 using shifts
             SHL     DX,1
             RCL     CX,1
;-------- Multiply CX,DX by 2 using shifts
             SHL     DX,1
             RCL     CX,1
;-------- Add AX,BX to CX,DX and put result in AX,DX
             ADD     DX,BX
             ADC     AX,CX
;-------- Exchange AX and DX -- result in DX,AX
             XCHG    DX,AX
;
;******** Convert to hundredths-of-seconds by multiplying by 64K / 59659
;-------- Divide DX,AX by 59,659 (word operation)
             MOV     CX,CON59
             DIV     CX
;-------- Place quotient in BX
             MOV     BX,AX
;-------- Append word to remainder and divide by 59,659 again
             XOR     AX,AX
             DIV     CX
;
;******** Find number of 100th-sec and total seconds since midnight
;-------- Place hundredths-of-seconds in DX,AX
             MOV     DX,BX
;-------- Divide DX,AX by 200 so quotient (total-seconds/2) fits in one word
             MOV     CX,200
             DIV     CX
;-------- Branch around if remainder is less than 100
             CMP     DL,HUNDRED
             JB      READ_2              ; Carry Flag now 1
;-------- Subtract 100 from remainder (sets CF to 0)
             SUB     DL,HUNDRED
;-------- Complement carry flag (set it to 1 if remainder was 100 or greater)
READ_2:      CMC
;-------- Place 100th-sec byte into BL
             MOV     BL,DL
;
;******** Convert total seconds since midnight to hours/minutes/seconds
;-------- Place total seconds since midnight in DX,AX
             RCL     AX,1                ; Rotate thru carry
             MOV     DL,ZERO
             RCL     DX,1                ; Rotate thru carry
;-------- Divide total seconds by 60
             MOV     CX,SIXTY
             DIV     CX
;-------- Place remainder (seconds byte) in BH
             MOV     BH,DL
;-------- Divide quotient (total minutes since midnight) by 60
             DIV     CL
;-------- Put hours byte in AH, minutes byte in AL
             XCHG    AH,AL
;
;******** Put result in 6-byte block pointed to by ES:DI, and exit
;-------- Push hour/minute info on stack
             PUSH    AX
;-------- Get day info from internal variable and put in block
             MOV     AX,DAYCOUNT
             STOSW                       ; Store ax to es:[di]
;-------- Pop hour/minute info and put in block
             POP     AX
             STOSW                       ; Store ax to es:[di]
;-------- Get seconds info and put in block
             MOV     AX,BX
             STOSW                       ; Store ax to es:[di]
;-------- Jump to common exit
             JMP     GOOD_EXIT

;=======================================
;======== Initialization Routine
;=======================================
;
DAYS_MONTH   DW      0,31,59,90,120,151,181,212,243,273,304,334
;
tz_neg_flag  db      0     ; if tz < 0 then tz_neg_flag = 1
B10          db      10         
W10          dw      10
;
zone_table   dw      0000h ; time offset = 0 hours
             dw      0000h ;             = 0 ticks     
 
             dw      0001h ; time offset = 1 hour
             dw      0007h ;             = 65543 ticks

             dw      0002h ; time offset = 2 hours
             dw      000Fh ;             = 131087 ticks

             dw      0003h ; time offset = 3 hours
             dw      0016h ;             = 196630 ticks

             dw      0004h ; time offset = 4 hours
             dw      001Dh ;             = 262173 ticks

             dw      0005h ; time offset = 5 hours
             dw      0025h ;             = 327717 ticks

             dw      0006h ; time offset = 6 hours
             dw      002Ch ;             = 393260 ticks

             dw      0007h ; time offset = 7 hours
             dw      0033h ;             = 458803 ticks

             dw      0008h ; time offset = 8 hours
             dw      003Bh ;             = 524347 ticks

             dw      0009h ; time offset = 9 hours
             dw      0042h ;             = 589890 ticks

             dw      000Ah ; time offset = 10 hours
             dw      0049h ;             = 655433 ticks

             dw      000Bh ; time offset = 11 hours
             dw      0051h ;             = 720977 ticks

             dw      000Ch ; time offset = 12 hours
             dw      0058h ;             = 786520 ticks
;
installmsg1  db      13, 10, 'CLOCK.SYS installed TZ = '
TZ_Str       db      10 dup( 0 )
installmsg2  db      13, 10, 0
;
badmsg       db      13, 10, 'Defective config.sys command line' 
             db      13, 10, 0
;
BUFFER       db      128 dup(?)
;
INITIALIZE   LABEL   NEAR
;
;******** Point DOS at end of resident code
;-------- Make ES:BX point at Request Header
             LES     BX,REQ_HDR_PTR
;-------- Place end of device address in Request Header `address' field
             MOV     ES:[BX].RH_ADDR_OFF,OFFSET DAYS_MONTH
             MOV     ES:[BX].RH_ADDR_SEG,CS
;
;******** Read real-time clock date info and convert from BCD to binary
;-------- Do read (CH=year high digits, CL=year low digits, DH=month, DL=day)
             MOV     AH,04H
             INT     1AH
;-------- Convert month
             MOV     AL,DH
             CALL    BCD_TO_BIN
             MOV     DH,AL
             DEC     DH
;-------- Convert day
             MOV     AL,DL
             CALL    BCD_TO_BIN
             MOV     DL,AL
             DEC     DL
;-------- Convert last two digits of year
             MOV     AL,CL
             CALL    BCD_TO_BIN
             MOV     CL,AL
;-------- Convert first two digits of year and place in AX
             MOV     AL,CH
             CALL    BCD_TO_BIN
             MOV     AH,HUNDRED
             MUL     AH
;-------- Add two halves of year together and place in CX
             XOR     CH,CH
             ADD     CX,AX
;
;******** Convert from year/month/day form to number-of-days since 1/1/80
;-------- Put number of days in this (assumed non-leap) year in BX
             XOR     BX,BX
             MOV     BL,DL
             XOR     AX,AX
             MOV     AL,DH
             SHL     AX,1
             MOV     SI,OFFSET DAYS_MONTH
             ADD     SI,AX
             ADD     BX,WORD PTR [SI]
;-------- Account for current leap year
             TEST    CX,0003H            ; Test for leap year
             JNZ     INIT_1              ; Loop around if not
             CMP     DX,0201H            ; Test against March 1
             JB      INIT_1              ; Loop around if less
             INC     BX                  ; Increment otherwise
;-------- Account for previous leap years
INIT_1:      MOV     AX,CX
             SUB     AX,1977
             SHR     AX,1                ; Divide ..
             SHR     AX,1                ; .. by 4
             ADD     BX,AX
;-------- Account for 365 days per previous year
             MOV     AX,365
             SUB     CX,1980
             MUL     CX
             ADD     AX,BX
;
;******** Save number-of-days and branch to exit
;
;-------- Save number-of-days in internal storage

             MOV     DAYCOUNT, AX

;-------- Get TZ from CONFIG.SYS Command Line Tail and
;-------- convert it to Clock Ticks before Greenwich

             call    get_params          ; parse config.sys command line  
             mov     di, OFFSET TZ_Str
             call    bin2dec             ; save tz as a decimal string
             call    tbg                 ; get ticks before Greenwich
             cmp     tz_neg_flag, 1
             je      skip_negate         ; if tz > 0 then negate
             neg     dx                  ; negate dx:ax
             neg     ax
             sbb     dx, 0000H
skip_negate:
             push    ax
             push    dx

;-------- Modify Tick and Day Counts by Time Zone

             mov     ah, 00H
             int     1AH             ; get tick count
             mov     ax, dx          ; mov cx:dx to dx:ax
             mov     dx, cx
             pop     cx
             pop     bx
             add     ax, bx          ; new_ticks = old_ticks + tbg
             adc     dx, cx   
             test    dx, dx          ; is dx:ax negative?
             jge     sign_done
             add     ax, 00B0H       ; add ticks per day (tpd)
             adc     dx, 0018H       
             dec     DAYCOUNT        ; and decrement DAYCOUNT
             jmp     done
sign_done:
             cmp     dx, 0018H       ; is dx > tpd_hi
             jg      too_big
             jne     done
             cmp     ax, 00B0H       ; is ax > tpd_lo
             jbe     done
too_big:     
             sub     ax, 00B0H       ; subtract tpd
             sbb     dx, 0018H 
             inc     DAYCOUNT        ; and increment DAYCOUNT
done:
             mov     cx, dx          ; mov dx:ax to cx:dx
             mov     dx, ax          
             mov     ah, 01H
             int     1AH             ; set new tick count

;-------- Display messages

             MOV     SI, OFFSET installmsg1
             call    print_line
             MOV     SI, OFFSET installmsg2
             call    print_line

;-------- Jump to common exit
             JMP     GOOD_EXIT

;======== Subroutine BCD_TO_BIN
;
;         Input: Integer between 0 and 99, stored in AL, coded in BCD
;                format (viz, a decimal integer in each nibble)
;         Output: Same integer, stored in AL, coded in binary format
;
BCD_TO_BIN   PROC    NEAR
;-------- BL holds low nibble, AL holds high nibble
             MOV     BL,AL
             AND     BL,0FH
             AND     AL,0F0H
;-------- Multiply high nibble by 10
             ROR     AL,1                ; Rotate
             MOV     AH,AL
             ROR     AL,1                ; Rotate
             ROR     AL,1                ; Rotate
             ADD     AL,AH               ; Low nibble*8 + low nibble*2
;-------- Add low nibble to result
             ADD     AL,BL
;-------- Return to caller
             RETN
BCD_TO_BIN   ENDP

;*****************************************
;
; Parse CONFIG.SYS Command Line Tail
;
; on entry: nothing
;
; on return: ax contains abs( tz )
;            if tz < 0 then tz_neg_flag = 1 
;            if tz >= 0 then tz_neg_flag = 0 
;
;*****************************************

get_params   proc    near
             push    ds
             push    cs
             pop     es
             mov     di, OFFSET BUFFER
             lds     bx, REQ_HDR_PTR
             lds     si, DWORD PTR [BX].RH_BYTPSEC
             mov     cx, 32
rep          movsb              ; move config.sys command line
                                ; tail to BUFFER
             pop     ds         ; restore DS

             mov     cx, 128    ; start parse
             cld
             mov     si, OFFSET BUFFER
next:        lodsb              ; find / or -
             cmp     al, '/'
             jz      good
             cmp     al, '-'
             jz      good
             cmp     al, 10
             jz      bad
             cmp     al, 13
             jz      bad
             loop    next

good:        call    get_char   ; find "tz ="
             jz      bad
             or      al, 20h
             cmp     al, 't'
             jnz     bad
             call    get_char
             jz      bad
             or      al, 20h
             cmp     al, 'z'
             jnz     bad
             call    get_char
             jz      bad
             cmp     al, '='
             jnz     bad
             call    get_char
             jz      bad
             cmp     al, '-'    ; is it - ?
             jz      negative
             cmp     al, '+'    ; is it + ?
             jz      positive
             jmp     number
negative:    mov     tz_neg_flag, 1 ; set if tz < 0
positive:    call    get_char
             jz      bad
number:      call    test_num   ; if we got here, al should
             jnz     bad        ; contain a number
             xor     cx, cx
             sub     al, '0'
             cbw
             xchg    ax, cx     ; put first digit in cx
             call    get_char
             jz      finish_up  ; if at end, finish up
             call    test_num
             jnz     bad
             sub     al, '0'
             xor     bx, bx
             mov     bl, al     ; put second digit in bl
             xchg    ax, cx     ; put first digit back in ax
             mul     W10
             add     ax, bx
             xchg    ax, cx
finish_up:
             xchg    ax, cx
             cmp     ax, 12
             ja      bad        ; jump if tz > 12
             jmp     done_parse
bad:         mov     ax, 0000h  
             mov     si, OFFSET badmsg
             call    print_line
done_parse:
             ret

get_char     proc
             cmp     al, 10
             je      get_char_exit
             cmp     al, 13
             je      get_char_exit
again:       lodsb
             cmp     al, 20h
             je      again       ; skip spaces
             cmp     al, 10
             je      get_char_exit
             cmp     al, 13
get_char_exit:
             ret
get_char     endp

test_num     proc
             cmp     al, '0'
             jb      test_num_exit
             cmp     al, '9'
             ja      test_num_exit
             cmp     al, al          ; set Z if numeric
test_num_exit:
             ret
test_num     endp

get_params   endp


;*****************************************
;
; Calculate Ticks Before Greenwich
;
; on entry: unsigned short int abs( tz ) in ax
;
; on exit:  unsigned long int tbg in dx:ax  hi-word = dx, lo-word = ax
;
;*****************************************

tbg          proc    near           
             mov     bx, OFFSET zone_table ; point to time offset table 
             mov     ah, 0
             shl     ax, 1
             shl     ax, 1                 ; make it an index
             add     bx, ax                ; bx points to tz_ticks_hi
             mov     dx, [bx]
             inc     bx
             inc     bx
             mov     ax, [bx]
             ret     
tbg          endp


;*****************************************
;
;  Convert small binary word in ax 
;  to two decimal digit string at DS:DI
;
;*****************************************

bin2dec      proc
             push    ax
             push    di
             cmp     tz_neg_flag, 0
             je      pr_plus
             mov     [di], 45         ; '-' 
             inc     di
             jmp     pos
pr_plus:
             mov     [di], 43         ; '+' 
             inc     di
pos:
             div     B10
             cmp     al, 0
             jz      skip_zero
             add     al, '0'
             mov     [di], al
             inc     di
skip_zero:
             xchg    al, ah
             add     al, '0'
             mov     [di], al
             pop     di
             pop     ax
             ret
bin2dec      endp


;*****************************************
;
; Print ASCIIZ string at DS:SI
;
;*****************************************

print_line   proc    near
             push    ax
print_line2:
             lodsb                        ;get next byte to print
             cmp     al, 0                ;terminating char 0?
             jz      print_line9          ;yes, exit
             call    print                ;print this char
             jmp     print_line2          ;continue till 0
print_line9:
             pop ax
             ret
print_line   endp


;********************************************************
;
; Output AL to screen
;
;********************************************************

print        proc    near
             push    bx
             push    ax
             mov     bx, 14 
             mov     ah, 0Eh
             int     010h
             pop     ax
             pop     bx
             ret
print        endp


COMMANDS     ENDP

DEVPROC      ENDP

DEVSEG       ENDS

             END
