/************************************************************************\
*                                                                        *
*  GS: A Generalized, stochastic petri net Simulator                     *
*      V0.01      March 1989      Andreas Nowatzyk (agn@unh.cs.cmu.edu)  *
*      Carnegie-Mellon Univerity, School of Computer Science             *
*      Schenley Park, Pittsburgh, PA 15213                               *
*                                                                        *
\************************************************************************/

/*
 * Parameters and other definitions
 *
 */

#define INIT_H_SIZE	4			/* default histogram size	    */

#define TTY_IMM	    0				/* transition types		    */
#define TTY_EXP	    1
#define	TTY_DET	    2
#define TTY_NRM	    3
#define TTY_BIM	    4

#define EPSILON	    1e-12			/* used in zero-tests for fp-numbers */

#ifdef mainpgm					/* main program ?		    */
#define cdext					/* define data struct.		    */
#define dinit(x) = x				/* var initialize		    */
#else
#define cdext extern				/* declare data struct.		    */
#define dinit(x)				/* no var init			    */
#endif
#define _ ,

/******************** Data structure definitions ********************/

struct transition {
    char		*name;			/* transition name */
    unsigned		type:2;			/* transition type */
    unsigned short	ena_cnt;		/* enable count = #of conditions NOT met to fire :
						   0: the transition can fire */
    unsigned short	ini_ena;		/* initial enable count */
    double		delay;			/* inverse of rate (saves some div's) */
    long		prob;			/* rescaled probablility (for immediate transition only) */
    unsigned long	fire_cnt;		/* # of times the transition fired */
    unsigned long	tot_fire_cnt;		/* total # of times the transition fired */
    
    struct transition	*prev, *next;		/* forw/backw pointer for event queue */
    double		t_fire;			/* time scheduled to fire */

    double		sum, sum_sq;		/* for rate computation */
    
    void		(*f_func)();		/* pointer to fire function */
};

struct hist {					/* histogram element */
    double		prob;			/* accumulated time within one generation */
    double		sum;			/* sum (prob) */
    double		sum_sq;			/* sum (prob^2) */
    double		sum_cr;			/* sum (prob * t(gen)) */
};

struct place {
    char		*name;			/* name of place */
    unsigned short	ini_mrk, cur_mrk;	/* initial & current #of marks */
    unsigned short	max_mrk;		/* max number of marks (for histogram size) */
    double		t_last;			/* time of last change */
    struct hist		*H;			/* histogram structure */
};

struct parameter {				/* generic parameter */
    char		*name;			/* name (to be accessed via command line) */
    unsigned		type:1;			/* 0: int (marking), 1: double (rate) */
    unsigned short	n_used;			/* #of incarnations */
    unsigned short	ind;			/* index to pointer array */
    double		value;			/* current value */
};

struct result {
    char		*name;			/* name of result */
    unsigned		type:2;			/* 0: dimensionless (E,P) 1: timed (D) 2: rate (F) */
    double		t_last;			/* time of last change */
    double		cur;			/* current value */
    double		acc;			/* accumulator */

    double		sum;			/* sum (prob) */
    double		sum_sq;			/* sum (prob^2) */
    double		sum_cr;			/* sum (prob * t(gen)) */

    void		(*upd_fnctn)();		/* update function */
};

/******************** Storage definitions ********************/

cdext double		clock dinit(0.0);	/* simulation time */
cdext double		total_clock dinit(0.0);	/* accumulated simulation time */
cdext double		total_sq_clock dinit(0.0);	/* accumulated simulation time^2 */
cdext struct transition	*top dinit(0);		/* top of fire queue */
cdext struct transition	*i_top dinit(0);	/* top of fire queue (immediate transitions) */
cdext long		i_prob_acc dinit(0);	/* total probability of immediate transitions */

						/* bimodal distribution control: */
/*
 *  Transition delays alternate between the nominal delay (= same as for an deterministic transition)
 *  and an alternate delay. The ratio is globally controlled as is the alternate delay.
 *  The ratio may vary between 0 (= det. delay only) and 100 (= alt. delay only).
 *  The alternate delay is experessed relative to the det. delay: 0.5 takes half the time,
 *  2.0 takes twice the nominal delay.
 */
cdext int		bim_ratio dinit(50);	/* %of times the alternate delay is used */
cdext double		bim_alt dinit(0.5);	/* alternate delay (the primary is 1.0) */


extern struct parameter	PARAMETER[];		/* paramerter */
extern unsigned long	n_parameters;
extern double		*d_param[];
extern unsigned short	*i_param[];

extern struct transition TRANSITION[];		/* transitions */
extern unsigned long	n_transitions;

extern struct place 	PLACE[];		/* places */
extern unsigned long	n_places;

extern struct result	RESULT[];		/* results */
extern unsigned long	n_results;

extern struct transition *TBUF[];		/* temporary list of all enabled transitions */

#define DEQUEUE_X_TR(x)	{if (x.prev) x.prev->next = x.next; else top = x.next;\
			 if (x.next) x.next->prev = x.prev;}

#define DEQUEUE_I_TR(x)	{if (x.prev) x.prev->next = x.next; else i_top = x.next;\
			 if (x.next) x.next->prev = x.prev;\
			 i_prob_acc -= x.prob;}

#define ENQUEUE_I_TR(x)	{x.prev = 0;\
			 if (x.next = i_top) i_top->prev = &(x);\
			 i_top = &(x);\
			 i_prob_acc += x.prob;}

cdext FILE		*dstf;			/* output and results go here */
cdext FILE		*yyin;			/* for future use (standart get_opt feature) */

double			acc_dwt();		/* accumulate dwell time */

/******************** Option processing (the usual chore) ********************/

cdext long		n_fire dinit(10000);	/* # of transition fireings	    */
cdext long		n_rgen dinit(5);	/* # of regenerations		    */
cdext long		random_seed dinit(123);	/* seed for random number generation */
cdext char		*comment dinit(0);	/* optional comment		    */
cdext char		*out_file dinit(0);	/* optional output file		    */

		    /* basics */
#define FIRE 0x0001				/* #of fireing per generation	    */
#define RGEN 0x0002				/* #of regeneration cycles	    */

		    /* control */
#define VERB 0x0010				/* verbose option		    */
#define BM_R 0x0020				/* bimodal ratio (0..100)	    */
#define BM_A 0x0040                             /* bimodal alternate delay	    */
		    /* misc */
#define COM  0x0100				/* comment supplied		    */
#define OUTF 0x0200				/* output file supplied		    */
#define SEED 0x0400				/* seed for random number gen	    */

		    /* app-specific */
#define PARM 0x1000				/* load (user interpret)	    */

cdext unsigned long OPTV dinit (0);		/* option vector		    */

struct options {				/* list of program options	    */
    char c;					/* mnemonic character		    */
    unsigned bc;			        /* bit code			    */
    char ty;					/* type: 0=option 1=var 2=string 3=parameter */
    char **s;					/* string destination		    */
    long *x;					/* variable destination		    */
    unsigned cf;				/* conflicting option		    */
    double *d;					/* floating point option	    */
} optv[]
#ifdef mainpgm
    = {
	{'b', BM_R, 1, 0, &bim_ratio, 0, 0},
	{'a', BM_A, 6, 0, 0, 0, &bim_alt},
	{'v', VERB, 0, 0, 0, 0, 0},
	{'f', FIRE, 1, 0, &n_fire, 0, 0},
	{'r', RGEN, 1, 0, &n_rgen, 0, 0},
	{'s', SEED, 1, 0, &random_seed, 0, 0},
	{'c', COM, 2, &comment, 0, 0, 0},
	{'o', OUTF, 2, &out_file, 0, 0, 0},
	{'p', PARM, 5, 0, 0, 0, 0}, 
 	{0, 0, 0, 0, 0, 0, 0}}
#endif
						;



#undef cdext
#undef dinit
#undef _
