#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
#reducible.g
#
############################################################################
##
#F  InfoMeatAxe (...)  . . . . . . . . . . . for debugging assistance
##
##
if not IsBound(InfoMeatAxe)  then InfoMeatAxe := Ignore;  fi;
###############################################################################
##
#F  OrthogVec( subbasis ) . . . . . single vector othogonal to a submodule,
## N.B. subbasis is assumed to consist of normed vectors, 
## submodule is assumed proper.
##
OrthogVec := function( subbasis )
    local v, i, j, k, x, dim, len, F;
    Sort(subbasis);
    subbasis := Reversed(subbasis);
# Now subbasis is in order so that the vector whose leading coefficient
# comes furthest to the left comes first.
    len := Length(subbasis);
    dim := Length(subbasis[1]);
    F := Field(Flat(subbasis));
    i:= 1;
    v:=[];
    for i in [1..dim] do
      v[i] := F.zero;
    od;
    i:=1;
    while i <= len and subbasis[i][i]=F.one do
      i:= i+1;
    od;
    v[i] := F.one;
    for j in Reversed([1..i-1]) do
      x := F.zero;
      for k in [j+1..i] do
        x := x + v[k]*subbasis[j][k];
      od;
      v[j] := -x;
    od;

    return v;
end;

#############################################################################
##
#F  SpinBasis( v,matrices,[ngens] ) . . . . basis for the module generated by a 
##  vector under the action of some matrices
##
## SpinBasis( v,matrices,[ngens]) computes a basis for the submodule defined by 
## the action of the
## matrix group generated by the list matrices on the vector v.
## It is returned as a list of normed vectors.
## If the optional third argument is present, then only the first ngens
## matrices in the list are used.
SpinBasis := function ( arg  )
    local   v, matrices, ngens,
            ans, dim, subdim, leadpos, w, i, j, k, l, m, f, zero;

    if Number(arg)<2 or Number(arg)>3 then
      Error("Usage:  SpinBasis( v,matrices,[ngens] )");
    fi;
    v := arg[1];
    matrices := arg[2];
    if Number(arg)=3 then
      ngens := arg[3];
      if ngens <= 0 or ngens > Length(matrices) then
         ngens := Length(matrices);
      fi;
    else
      ngens := Length(matrices);
    fi;
    f:=Field(Flat(matrices)); zero:=f.zero;
    ans := [];
    dim := Length(v);
    leadpos := [];
    
    j:=1;
    while j<=dim and v[j]=zero do j:=j+1; od;
    if j>dim then
       return ans;
    fi;
    subdim:=1;
    leadpos[1]:=j;
    w := (v[j]^-1)*v;
    Add( ans, w );

    i:=1;
    while i<=subdim do
        for l in [1..ngens] do
           m := matrices[l];
           # apply generator m to submodule generator i
           w := ans[i]*m;
           # try to express w in terms of existing submodule generators
           j:=1;
           for  j in [1..subdim] do
              k := w[leadpos[j]];
              if k<>zero then
                  w := w - k*ans[j];
              fi;
           od;

           j:=1;
           while j<=dim and w[j]=zero do j:=j+1; od;
           if j<=dim then
           #we have found a new generator of the submodule
              subdim:=subdim+1;
              leadpos[subdim]:=j;
              w := (w[j]^-1)*w;
              Add( ans, w );
              if subdim = dim then
                 return ans;
              fi;
           fi;
        od;
        i:=i+1;
    od;
           
    
    return ans;
end;

#############################################################################
##
#F  SubGMod( module,sub ) . . . . . construct submodule
##
## module is a module record, and sub is a list of generators of a submodule.
## IT IS ASSUMED THAT THE GENERATORS OF SUB ARE NORMED.
## (i.e. each has leading coefficient 1 in a unique place).
## SubGMod( module,sub ) computes the submodule of module for which
## sub is the basis.
## If  sub is empty or generates the whole space, then the empty sequence
## is returned.
## If sub does not generate a submodule then false is returned.
SubGMod := function ( module,sub )
    local   ans, dim, subdim, leadpos,  w, i, j, k, m,
            g, newg, smodule,matrices, smatrices,
            im, newim, F, zero, one;

    ans := [];
    subdim := Length(sub);
    if subdim=0 then
        return ans;
    fi;
    dim := DimFlag(module);
    if subdim=dim then
        return ans;
    fi;
    matrices := MatricesFlag(module);
    F:=FieldFlag(module); zero:=F.zero; one:= F.one;
    sub := ShallowCopy(sub);

## As in SpinBasis, leadpos[i] gives the position of the first nonzero entry
## (which will always be 1) of sub[i].

    leadpos:=[];
    for i in [1..subdim] do
       j:=1;
       while j<=dim and sub[i][j]=zero do j:=j+1; od;
       leadpos[i]:=j; 
       for k in [1..i-1] do
         if leadpos[k]=j then
           Error("Subbasis isn't normed.\n");
         fi;
       od;
    od;


## Now work out action of generators on submodule
   smatrices:=[];
   for g in matrices do
      newg := [];
      for i in [1..subdim] do
          im := sub[i] * g;
          newim := [];
          for j in [1..subdim] do
              k := im[leadpos[j]];
              newim[j] := k;
              if k<> zero then
                  im := im - k*sub[j];
              fi;
          od;
# Check that the vector is now zero - if not, then sub was not the basis of a
# submodule at all.
          if im <> im*zero then return false; fi;
          Add(newg,newim);   
      od;
      Add(smatrices,newg);
   od;
   smodule := GModule(smatrices,F);
   return smodule;
end;


#############################################################################
##
#F  SubGModAction( sub,g ) . . . . . construct matrix for g relative
##                          to the basis sub (spanning a space preserved by g).
##
## IT IS ASSUMED THAT THE SUB IS NORMED.
## This code is taken from SubGMod,
## and allows us to add new matrices into a submodule
## without recomputing everything else.
##
SubGModAction := function(sub,g)
    local dim,subdim,zero,leadpos,i,j,k,im,newim,newg;

    subdim := Length(sub);
    dim := Length(sub[1]);
    zero := Field(sub[1]).zero;
    
    leadpos:=[];
    for i in [1..subdim] do
       j:=1;
       while j<=dim and sub[i][j]=zero do j:=j+1; od;
       leadpos[i]:=j; 
       for k in [1..i-1] do
         if leadpos[k]=j then
           Error("Subbasis isn't normed.\n");
         fi;
       od;
    od;

## Now work out action of g on submodule
    newg := [];
    for i in [1..subdim] do
       im := sub[i] * g;
       newim := [];
       for j in [1..subdim] do
          k := im[leadpos[j]];
          newim[j] := k;
          if k<> zero then
            im := im - k*sub[j];
          fi;
       od;
       Add(newg,newim);   
    od;
    return newg;
end;

#############################################################################
##
#F  QuotGMod( module,sub ) . . . . . generators of submodule
##
## module is a module record, and sub is a list of generators of a submodule.
## IT IS ASSUMED THAT THE GENERATORS OF SUB ARE NORMED.
## (i.e. each has leading coefficient 1 in a unique place).
## If  sub is empty or generates the whole space, then the empty sequence
## is returned. Otherwise, qmodule is returned, where qmodule
## is the quotient module.
QuotGMod := function ( module,sub )
    local   ans, dim, subdim, leadpos, cfleadpos,  w, i, j, k, m,
            g, newg, qmodule,matrices, qmatrices,
            im, newim, F, zero, one;

    ans := [];
    subdim := Length(sub);
    if subdim=0 then
        return ans;
    fi;
    dim := DimFlag(module);
    if subdim=dim then
        return ans;
    fi;
    matrices := MatricesFlag(module);
    F:=FieldFlag(module); zero:=F.zero; one:= F.one;
    sub := ShallowCopy(sub);

## As in SpinBasis, leadpos[i] gives the position of the first nonzero entry
## (which will always be 1) of sub[i].

    leadpos:=[];
    cfleadpos:=[];
    for i in [1..dim] do cfleadpos[i]:=0; od;
    for i in [1..subdim] do
       j:=1;
       while j<=dim and sub[i][j]=zero do j:=j+1; od;
       leadpos[i]:=j; cfleadpos[j]:=1;
       for k in [1..i-1] do
         if leadpos[k]=j then
           Error("Subbasis isn't normed.\n");
         fi;
       od;
    od;

## Now add a further dim-subdim vectors to the list sub, to complete a basis.

    k:=subdim;
    for i in [1..dim] do
        if cfleadpos[i]=0 then
            k:=k+1;
            w:=[];
            for m in [1..dim] do  w[m]:=zero; od;
            w[i]:=one;
            leadpos[k]:=i;
            Add(sub,w);
        fi;
    od;
## Now work out action of generators on quotient module
   qmatrices:=[];
   for g in matrices do
      newg := [];
      for i in [subdim+1..dim] do
          im := sub[i] * g;
          newim := [];
          for j in [1..dim] do
              k := im[leadpos[j]];
              if j>subdim then
                 newim[j-subdim] := k;
              fi;
              if k<> zero then
                  im := im - k*sub[j];
              fi;
          od;
          Add(newg,newim);   
      od;
      Add(qmatrices,newg);
   od;
   qmodule := GModule(qmatrices,F);
   return qmodule;

end;
  
#############################################################################
##
#F  SubQuotGMod( module,sub ) . . .  generators of sub- and quotient-module
##                               . . .  and original module wrt new basis
## module is a module record, and sub is a list of generators of a submodule.
## IT IS ASSUMED THAT THE GENERATORS OF SUB ARE NORMED.
## (i.e. each has leading coefficient 1 in a unique place).
## SubQuotGMod( module,sub ) computes the submodule and quotient module.
## and the original module with its matrices written wrt to the basis used
## to compute smodule and qmodule. 
## If  sub is empty or generates the whole space, then the empty sequence
## is returned. Otherwise, [smodule,qmodule, nmodule] is returned, where smodule
## is the submodule and qmodule the quotient module.
## The matrices of nmodule have the form  A  0  where  A  and  B  are the
##                                        C  B
## corresponding matrices of smodule and qmodule resepctively.
## If sub does is not the basis of a submodule then false is returned.
SubQuotGMod := function ( module,sub )
    local   ans, dim, subdim, leadpos, cfleadpos,  w, i, j, k, m, ct,
            g, newg, newgn,  smodule,qmodule,nmodule,
            matrices, smatrices, qmatrices, nmatrices,
            im, newim, newimn, F, zero, one;

    ans := [];
    subdim := Length(sub);
    if subdim=0 then
        return ans;
    fi;
    dim := DimFlag(module);
    if subdim=dim then
        return ans;
    fi;
    matrices := MatricesFlag(module);
    F:=FieldFlag(module); zero:=F.zero; one:= F.one;
    sub := ShallowCopy(sub);

## As in SpinBasis, leadpos[i] gives the position of the first nonzero entry
## (which will always be 1) of sub[i].

    leadpos:=[];
    cfleadpos:=[];
    for i in [1..dim] do cfleadpos[i]:=0; od;
    for i in [1..subdim] do
       j:=1;
       while j<=dim and sub[i][j]=zero do j:=j+1; od;
       leadpos[i]:=j; cfleadpos[j]:=1;
       for k in [1..i-1] do
         if leadpos[k]=j then
           Error("Subbasis isn't normed.\n");
         fi;
       od;
    od;

## Now add a further dim-subdim vectors to the list sub, to complete a basis.

    k:=subdim;
    for i in [1..dim] do
        if cfleadpos[i]=0 then
            k:=k+1;
            w:=[];
            for m in [1..dim] do  w[m]:=zero; od;
            w[i]:=one;
            leadpos[k]:=i;
            Add(sub,w);
        fi;
    od;

## Now work out action of generators on submodule
   smatrices:=[];
   nmatrices:=[];
   for g in matrices do
      newg := []; newgn := [];
      for i in [1..subdim] do
          im := sub[i] * g;
          newim := []; newimn := [];
          for j in [1..subdim] do
              k := im[leadpos[j]];
              newim[j] := k; newimn[j] := k;
              if k<> zero then
                  im := im - k*sub[j];
              fi;
          od;
# Check that the vector is now zero - if not, then sub was not the basis of a
# submodule at all.
          if im <> im*zero then return false; fi;
          for j in [subdim+1..dim] do newimn[j]:=zero; od;
          Add(newg,newim);  Add(newgn,newimn);
      od;
      Add(smatrices,newg);
      Add(nmatrices,newgn);
   od;
   smodule := GModule(smatrices,F);

## Now work out action of generators on quotient module
   qmatrices:=[];
   ct := 0;
   for g in matrices do
      ct := ct+1;
      newg := []; newgn := nmatrices[ct];
      for i in [subdim+1..dim] do
          im := sub[i] * g;
          newim := []; newimn := [];
          for j in [1..dim] do
              k := im[leadpos[j]];
              if j>subdim then
                 newim[j-subdim] := k;
              fi;
              newimn[j] := k;
              if k<> zero then
                  im := im - k*sub[j];
              fi;
          od;
          Add(newg,newim);   
          Add(newgn,newimn);
      od;
      Add(qmatrices,newg);
   od;
   qmodule := GModule(qmatrices,F);
   nmodule := GModule(nmatrices,F);

   ans := [smodule,qmodule,nmodule];
   return ans;

end;
  
#############################################################################
##
#F  IsIrredGMod( module ) . . try to reduce a module over a finite field
##
## module is a module record
## IsIrredGMod( ) attempts to decide whether module is irreducible.
## When it succeeds it returns true or false.
## We choose at random elements of the group algebra of the group.
## If el is such an element, we define M,p,fac,N,e and v as follows:-
## M is the matrix corresponding to el, p is its characteristic polynomial,
## fac an irreducible factor of p, N the nullspace of the matrix fac(M),
## ndim the dimension of N, and v a vector in N.
## If we can find the above such that ndim = deg(fac) then we can test
## conclusively for irreducibility. Then, in the case where irreducibility is
## proved, we store the information as fields for the module, since it may be
## useful later (e.g. to test for absolute irreducibility, equivalence with
## another module).
## These  fields are accessed by the functions
## AlgElFlag() (el), AlgElMatFlag (M), AlgElCharPolFlag (p),
## AlgElCharPolFacFlag (fac), AlgElNullspaceDimFlag (ndim), and
## AlgElNullspaceVecFlag (v).
## If we cannot find such a set with ndim=deg(fac) we may nonetheless prove
## reducibility  by finding a submodule. However we can never prove
## irreducibility without such a set (and hence the algorithm could run
## forever, but hopefully this will never happen!)
## Where reducibility is proved, we set the field .subbasis
## (a basis for the submodule, normed in the sense that the first non-zero
## component of each basis vector is 1, and is in a different position from
## the first non-zero component of every other basis vector).
## The test for irreducibility is based on the meataxe method  (but in the
## meataxe, ndim is always very small, usually 1. The modification here is put
## in to enable the method to work over modules with large centralizing fields).
## We simply spin v. If we do
## not get  the whole space, we have a submodule, on the other hand,
## if we do get the whole space, we calculate the nullspace NT of the transpose
## of fac(M), spin that under the group generated by the transposes of the
## generating matrices, and thus either find the transpose of a submodule
## or conclusively prove irreducibility.

IsIrredGMod := function ( module )
    local matrices, tmatrices, ngens, ans,  M, mat, g1,g2,maxdeg,
          newgenlist,coefflist,orig_ngens,
          N, NT, v, subbasis, sq, fac, sfac, pol, orig_pol, q, dim,ndim, i, k, 
          l, trying, dim, deg, facno, bestfacno, F,count, R, rt0,rt1;

    rt0:=Runtime();
    InfoMeatAxe("Calling MeatAxe. All times will be in milliseconds.\n");
    if IsGModule(module)=false then 
      return Error("Argument of IsIrredGMod is not a module.");
    elif ReducibleFlag(module) <> "unknown" then
      return not ReducibleFlag(module);
    fi;
    matrices := MatricesFlag(module);
    dim := DimFlag(module);
    ngens := Length(matrices);
    orig_ngens:=ngens;
    F := FieldFlag(module);
    R := PolynomialRing(F);

# Now compute random elements M of the group algebra, calculate their
# characteristic polynomials, factorize, and apply the irreducible factors
# to M to get matrices with nontrivial nullspaces.
# tmatrices will be a list of the transposed generators if required.

   tmatrices := [];
   trying := true; 
#trying will become false when we have an answer
   maxdeg:=1;
   newgenlist := [];
# Do a small amount of preprocessing to increase the generator set.
   for i in [1..1] do
      g1:=Random([1..ngens]);
      g2:=g1;
      while g2=g1 and Length(matrices)>1 do
         g2:=Random([1..ngens]);
      od;
      ngens:=ngens+1;
      matrices[ngens]:=matrices[g1]*matrices[g2];
      Add(newgenlist,[g1,g2]);
   od;
   rt1:=Runtime();
   InfoMeatAxe("Done preprocessing. Time = ",rt1-rt0,".\n");
   count := 0;

#Main loop starts - choose a random element of group algebra on each pass
   while trying  do
      count := count+1;
      if count>50 then
    Error("Have generated 50 random elements and failed to prove or disprove irreducibility.\n");
      fi;
      maxdeg:=maxdeg*2;
# On this pass, we only consider irreducible factors up to degree maxdeg.
# Using higher degree factors is very time consuming, so we prefer to try
# another element.
# To choose random element, first add on a new generator as a product of
# two randomly chosen unequal existing generators
# Record the product in newgenlist.
      InfoMeatAxe("Choosing random element number ",count,".\n");
      g1:=Random([1..ngens]);
      g2:=g1;
      while g2=g1 do
         g2:=Random([1..ngens]);
      od;
      ngens:=ngens+1;
      matrices[ngens]:=matrices[g1]*matrices[g2];
      Add(newgenlist,[g1,g2]);
# Now take a random linear sum of the existing generators as new generator.
# Record the sum in coefflist
      coefflist:=[];
      M := NullMat(dim,dim,F);
      for g1 in [1..ngens] do
        g2:=Random(F);
        M := M + g2*matrices[g1];
        Add(coefflist,g2);
      od;
      rt1:=Runtime();
      InfoMeatAxe("Evaluated random element in algebra. Time = ",rt1-rt0,".\n");
      pol := CharacteristicPolynomial(M);
      pol := EmbeddedPolynomial(R,pol);
#That is necessary in case pol is defined over a smaller field that F.
      orig_pol := Copy(pol);
      rt1:=Runtime();
      InfoMeatAxe("Evaluated characteristic polynomial. Time = ",rt1-rt0,".\n");
#Now we extract the irreducible factors of pol starting with those of low degree
      deg := 0;
      fac := [];
#The next loop is through the degrees of irreducible factors
      while Degree(pol)>0 and deg<maxdeg and trying do
         repeat
            deg := deg+1;
            if deg > Int(Degree(pol)/2) then
                fac := [pol];
            else
                fac := FactorsPolDeg(R,pol,deg);
                rt1:=Runtime();
                InfoMeatAxe(
Length(fac)," factors of degree ",deg,", Time = ",rt1-rt0,".\n");
            fi;
         until fac <> [] or deg=maxdeg;

         if fac <> [] then
           if Degree(fac[1])=dim then 
# In this case the char poly is irreducible, so the module is irreducible.
             ans := true;
             trying := false; 
             bestfacno := 1;
             v := matrices[1][1] * F.zero;
             v[1] := F.one;
             ndim := dim;
           fi; 
# Otherwise, first see if there is a non-repeating factor.
# If so it will be decisive, so delete the rest of the list
           l := Length(fac);
           facno := 1;
           while facno <= l  and trying do
             if facno = l  or  fac[facno] <> fac[facno+1] then
                fac := [fac[facno]]; l:=1;
             else
                 while facno < l and fac[facno] = fac[facno+1] do
                    facno:=facno+1;
                 od;
             fi;
             facno := facno+1;
           od;
# Now we can delete repetitions from the list fac
           sfac := Set(fac);
   
           if Degree(fac[1]) <> dim then
# Now go through the factors and attempt to find a submodule
             facno := 1; l:=Length(sfac);
             while facno<=l and trying do
                mat := Value(sfac[facno],M);
                rt1:=Runtime();
                InfoMeatAxe(
                  "Evaluated matrix on factor. Time = ",rt1-rt0,".\n");
                N := NullspaceMat(mat);
                v := N[1];
                ndim := Length(N);
                rt1:=Runtime();
                InfoMeatAxe(
        "Evaluated nullspace. Dimension = ",ndim,". Time = ",rt1-rt0,".\n");
                subbasis := SpinBasis(v,matrices,orig_ngens);
                rt1:=Runtime();
                InfoMeatAxe(
   "Spun up vector. Dimension = ",Length(subbasis),". Time = ",rt1-rt0,".\n");
                if Length(subbasis) < dim then
# Proper submodule found !
                   trying := false;
                   ans := false;
                   SetSubbasisFlag(module,subbasis);
                elif ndim = deg then
                   trying := false;
# In this case, if we transpose and find no proper submodule, then the
# module is definitely irreducible. 
                   mat := TransposedMat(mat);
                   if Length(tmatrices)=0 then
                      for i in [1..orig_ngens] do
                        Add(tmatrices,TransposedMat(matrices[i]));
                      od;
                   fi;
                  rt1:=Runtime();
                  InfoMeatAxe(
                    "Transposed matrices. Time = ",rt1-rt0,".\n");
                   NT := NullspaceMat(mat);
                   rt1:=Runtime();
                   InfoMeatAxe(
     "Evaluated nullspace. Dimension = ",Length(NT),". Time = ",rt1-rt0,".\n");
                   subbasis := SpinBasis(NT[1],tmatrices,orig_ngens);
                   rt1:=Runtime();
                   InfoMeatAxe(
   "Spun up vector. Dimension = ",Length(subbasis),". Time = ",rt1-rt0,".\n");
                   if Length(subbasis) < dim then
# subbasis is a basis for a submodule of the transposed module, and 
# the othogonal complement of this is a submodule of the original 
# module. So we find a vector v in that, and then spin it. Of course we won't
# necessarily get the full orthogonal complement that way, but we'll
# certainly get a proper submodule.
                     v := OrthogVec(subbasis);
                     SetSubbasisFlag(module,SpinBasis(v,matrices,orig_ngens));
                     ans := false;
                   else
                     ans := true;
                     bestfacno := facno;
                   fi;
                fi;
                facno := facno+1;
             od; # going through irreducible factors of fixed degree.
# If trying is false at this stage, then we don't have an answer yet,
# so we have to go onto factors of the next degree.
# Now divide p by the factors used if necessary
             if trying and deg<maxdeg then
               for q in fac do
                   pol := Quotient(R,pol,q);
               od;
             fi; 
           fi;           #Degree(fac[1]) <> dim
         fi;             #fac <> []
      od; #loop through degres of irreducible factors

# if we have not found a submodule and trying is false, then the module
# must be irreducible.
     if trying=false and ans=true then
       SetAlgElFlag(module,[newgenlist,coefflist]);
       SetAlgElMatFlag(module,M);
       SetAlgElCharPolFlag(module,orig_pol);
       SetAlgElCharPolFacFlag(module,sfac[bestfacno]);
       SetAlgElNullspaceVecFlag(module,v);
       SetAlgElNullspaceDimFlag(module,ndim);
     fi;

   od;  #main loop

   if ans=true then SetReducibleFlag(module,false);
   else SetReducibleFlag(module,true);
   fi;
   for i in [orig_ngens+1..ngens] do
     Unbind(matrices[i]);
   od;
   rt1:=Runtime();
   InfoMeatAxe("Total time = ",rt1-rt0," microseconds.\n");
   return ans;

end;

#############################################################################
##
#F RandomIrredSubGMod( module ) . . .find a basis for a random irreducible
## submodule of module, and return that basis and the submodule, with all
## the irreducibility flags set.
## Returns false if module is irreducible.
RandomIrredSubGMod := function ( module )
    local  ranSub, subbasis, submodule, subbasis2,submodule2, copymodule,
    F,dim,el, M, p, fac, N, ndim, v, i, matrices, ngens, orig_ngens, genpair;

    if IsGModule(module)=false then 
      return Error("Argument of IsIrredGMod is not a module.");
    elif ReducibleFlag(module)=false then
      return false;
    fi;
    if ReducibleFlag(module)=true then
      copymodule := Copy(module);
      UndoReducibleFlag(copymodule);
# Do this to avoid changing the flags in the original module
# We need to undo the reducible falgs before calling IsIrredGMod so that it
# actually runs and doesn't merely select the submodule already listed as a
# field of module.
    else
      copymodule := module;
    fi;

    if IsIrredGMod(copymodule) then
       return false;
# in this case copymodule is actually equal to module, because copymodule is
# only actually a copy of module when module is known to be reducible
# So when IsIrredGMod sets all the flags it sets as it proves irreducibility,
# they'll be set in the original module.
    fi;

    subbasis := SubbasisFlag(copymodule);
    submodule := SubGMod(copymodule,subbasis);
    ranSub := RandomIrredSubGMod(submodule);
    if ranSub = false then
# submodule has been proved irreducible in a call to this function,
# so the flags have been set.
       return [ subbasis,submodule] ;
    else 
# ranSub[1] is given in terms of the basis for the submodule, but we want it
# to be given in terms of the basis of the original module.
# So we multiply it by subbasis.
# Then we need our basis to be normed. The function Base ensures that it is. 
      F := FieldFlag(module);
      subbasis2 := Base(RowSpace(ranSub[1]*subbasis,F)); 
# But now since we've normed the basis subbasis2,
# the matrices of the submodule ranSub[2] are given with respect to 
# the wrong basis.
# So we have to recompute the submodule.
      submodule2 := SubGMod(module,subbasis2);
# Unfortunately, although it's clear that this submodule is irreducible, we'll## have to reset the flags that IsIrredGMod sets. Some will be the same
# as in ranSub[2], but some are effected by the base change. 
# or at least part of it, since the flags gets screwed up by the base change.
# We need to set the following flags:-
# ReducibleFlag
# AlgElFlag() (el), AlgElMatFlag (M), AlgElCharPolFlag (p),
# AlgElCharPolFacFlag (fac), AlgElNullspaceDimFlag (ndim), and
# AlgElNullspaceVecFlag (v).
# Most of these can simply be copied.
      SetReducibleFlag(submodule2,false);
      el := AlgElFlag(ranSub[2]); 
      SetAlgElFlag(submodule2,el);
      p:= AlgElCharPolFlag(ranSub[2]); 
      SetAlgElCharPolFlag(submodule2,p);
      fac:=AlgElCharPolFacFlag(ranSub[2]); 
      SetAlgElCharPolFacFlag(submodule2,fac);
      ndim:=AlgElNullspaceDimFlag(ranSub[2]);
      SetAlgElNullspaceDimFlag(submodule2,ndim);
# Only two of the flags actually have to be recomputed.
# This code is essentially from IsomGMod
# To compute the matrix that represents this element we first
# have to add extra generators  
      dim := DimFlag(submodule2);
      matrices:=MatricesFlag(submodule2); 
      ngens := Length(matrices);
      orig_ngens := ngens;
      for genpair in el[1] do
        ngens:=ngens+1;
        matrices[ngens] := matrices[genpair[1]]*matrices[genpair[2]];
      od;
      M := NullMat(dim,dim,F);
      for i in [1..ngens] do M := M + el[2][i]*matrices[i]; od;
# Having done that, we no longer want the extra generators ,
# so we throw them away again.
      for i in [orig_ngens+1..ngens] do Unbind(matrices[i]); od;
      SetAlgElMatFlag(submodule2,M);
      N := NullspaceMat(Value(fac,M));
      SetAlgElNullspaceVecFlag(submodule2,N[1]);
      return [subbasis2,submodule2];
    fi;

end;


#############################################################################
##
#F  GoodElMod( module ) . .  find good group algebra element in an
##                      . .  irreducible module
##
## module is a module that is already known to be irreducible.
## GoodElMod finds a group algebra element with nullspace of 
## minimal possible dimension. This dimension is 1 if the module is absolutely
## irreducible, and the degree of the relevant field extension otherwise.
## This is needed for testing for equivalence of modules.
GoodElMod := function ( module )
    local matrices, ngens, el, M, mat,  N, newgenlist,coefflist,orig_ngens,
    g1,g2, fac, sfac, p, oldp,  q, deg, i, k, l,
    trying, dim,mindim, deg,  F,R,count, rt0, rt1;

    rt0:=Runtime();
    if IsGModule(module)=false then 
       return Error("Argument is not a module.");
    elif ReducibleFlag(module)<>false then
       return Error("Module is not known to be irreducible");
    fi;
    if AbsReducibleFlag(module)="unknown" then
        IsAbsIrredGMod(module);
    fi;
    if  AbsReducibleFlag(module)=false then
      mindim := 1;
    else mindim := FieldExtDegFlag(module);
    fi;
    if AlgElNullspaceDimFlag(module)=mindim then return; fi;
# This is the condition that we want. If it holds already, then there is
# nothing else to do.
    dim := DimFlag(module);
    matrices := MatricesFlag(module);
    ngens := Length(matrices);
    orig_ngens:=ngens;
    F := FieldFlag(module);
    R := PolynomialRing(F);

# Now compute random elements el of the group algebra, calculate their
# characteristic polynomials, factorize, and apply the irreducible factors
# to el to get matrices with nontrivial nullspaces.

   trying := true; 
   count:=0;
   newgenlist := [];
   while trying do
      count := count+1;
      if count>50 then
    Error("Have generated 50 random elements and failed to find a good one.\n");
      fi;
      InfoMeatAxe("Choosing random element number ",count,".\n");
# To choose random element, first add on a new generator as a product of
# two randomly chosen unequal existing generators
# Record the product in newgenlist.
      g1:=Random([1..ngens]);
      g2:=g1;
      while g2=g1 and ngens>1 do
         g2:=Random([1..ngens]);
      od;
      ngens:=ngens+1;
      matrices[ngens]:=matrices[g1]*matrices[g2];
      Add(newgenlist,[g1,g2]);
# Now take a random linear sum of the existing generators as new generator.
# Record the sum in coefflist
      coefflist:=[];
      M := NullMat(dim,dim,F);
      for g1 in [1..ngens] do
        g2:=Random(F);
        M := M + g2*matrices[g1];
        Add(coefflist,g2);
      od;
      rt1:=Runtime();
      InfoMeatAxe("Evaluated random element in algebra. Time = ",rt1-rt0,".\n");

      p := CharacteristicPolynomial(M);
      p := EmbeddedPolynomial(R,p);
      rt1:=Runtime();
      InfoMeatAxe("Evaluated characteristic polynomial. Time = ",rt1-rt0,".\n");
#That is necessary in case p is defined over a smaller field that F.
      oldp := Copy(p);
#Now we extract the irreducible factors of p starting with those of low degree
      deg := 0;
      fac := [];
      while deg <=mindim and trying do
         repeat
            deg := deg+1;
            if deg > mindim then
                fac := [p];
            else
                fac := FactorsPolDeg(R,p,deg);
                rt1:=Runtime();
                InfoMeatAxe(
Length(fac)," factors of degree ",deg,", Time = ",rt1-rt0,".\n");
                sfac := Set(fac);
            fi;
         until fac <> [];
         l := Length(fac);
         if trying and deg<=mindim then
            i := 1;
            while i <= l  and trying do
               mat := Value(fac[i],M);
               rt1:=Runtime();
               InfoMeatAxe(
                  "Evaluated matrix on factor. Time = ",rt1-rt0,".\n");
               N:=NullspaceMat(mat);
               rt1:=Runtime();
               InfoMeatAxe(
      "Evaluated nullspace. Dimension = ",Length(N),". Time = ",rt1-rt0,".\n");
               if Length(N) = mindim then
                 trying:=false;
                 SetAlgElFlag(module,[newgenlist,coefflist]);
                 SetAlgElMatFlag(module,M);
                 SetAlgElCharPolFlag(module,oldp);
                 SetAlgElCharPolFacFlag(module,fac[i]);
                 SetAlgElNullspaceVecFlag(module,N[1]);
                 SetAlgElNullspaceDimFlag(module,Length(N));
               fi;
               i := i+1;
            od;
         fi;
   
         if trying then
             for q in fac do
                 p := Quotient(R,p,q);
             od;
         fi; 
      od;
   od;
   for i in [orig_ngens+1..ngens] do
     Unbind(matrices[i]);
   od;
   rt1:=Runtime();
   InfoMeatAxe("Total time = ",rt1-rt0," microseconds.\n");
   
end;

#############################################################################
##
#F  EnlargeIrreducibleModule(module,mat) . .add a generator to a module that
##                                       . .is already known to be irreducible.
##
##  This function adds an extra matrix <mat> to the module <module> which
##  has already been shown to be irreducible, and has all of its flags set.
##  Of course, the module remains irreducible. The only flag that needs changing
##  is the AlgElFlag. Its second component is a coefficient list, and a 0
##  has to be inserted here in the place where the new generator sits.
##  WARNING: If the module has been shown to be absolutely irreducible, then
##  this may no longer be true. Similarly, flags relating to decompositions
##  into tensor products, blocks of imprimitivity, etc. will need updating.
##  This is the users responsibility. USE THIS FUNCTION WITH CIRCUMSPECTION.
EnlargeIrreducibleModule := function( module,mat )
   local mats, ngens, F, el, coeffs, lc, i;
   mats:=MatricesFlag(module);
   Add(mats,mat);
   ngens:=Length(mats);
   F:= FieldFlag(module);
   el := AlgElFlag(module);
   coeffs := el[2];
   lc := Length(coeffs);
   for i in Reversed([ngens,lc]) do
     coeffs[i+1] := coeffs[i];
   od;
   coeffs[ngens] := F.zero;
end;

