(*  :Title:	Region of Convergence  *)

(*  :Authors:	Brian Evans, James McClellan  *)

(*  :Summary:	To provide region of convergence routines for transforms  *)

(*  :Context:	SignalProcessing`Support`ROC`  *)

(*  :PackageVersion:  2.4	*)

(*
    :Copyright:	Copyright 1989-1991 by Brian L. Evans
		Georgia Tech Research Corporation

	Permission to use, copy, modify, and distribute this software
	and its documentation for any purpose and without fee is
	hereby granted, provided that the above copyright notice
	appear in all copies and that both that copyright notice and
	this permission notice appear in supporting documentation,
	and that the name of the Georgia Tech Research Corporation,
	Georgia Tech, or Georgia Institute of Technology not be used
	in advertising or publicity pertaining to distribution of the
	software without specific, written prior permission.  Georgia
	Tech makes no representations about the suitability of this
	software for any purpose.  It is provided "as is" without
	express or implied warranty.
 *)

(*  :History:	*)

(*  :Keywords:	region of convergence, open set	*)

(*  :Source:	*)

(*  :Warning:	*)

(*  :Mathematica Version:  1.2 or 2.0 *)

(*  :Limitation:  *)

(*  :Discussion:  *)

(*  :Functions:	  FindRMinus
		  FindRPlus
		  GetRMinus
		  GetRPlus
		  InfCheck
		  IntersectsROC
		  WithinROC
 *)


If [ TrueQ[ $VersionNumber >= 2.0 ],
     Off[ General::spell ];
     Off[ General::spell1 ] ]


(*  B E G I N     P A C K A G E  *)

BeginPackage[ "SignalProcessing`Support`ROC`",
	      "SignalProcessing`Support`SupCode`" ]


(*  U S A G E     I N F O R M A T I O N  *)

FindRMinus::usage =
	"FindRMinus[rminus1, rminus2] and FindRMinus[rminus1, rminus2, \
	lowerlimit] returns the greater of the two coordinates rminus1 \
	and rminus2.  This function and FindRPlus[] are used to find \
	the intersection of two regions of convergence."

FindRPlus::usage =
	"FindRPlus[rplus1, rplus2] and FindRPlus[rplus1, rplus2, lowerlimit] \
	returns the lesser of the two coordinates rminus1 and rminus2. \
	This function and FindRMinus[] are used to find the intersection of \
	two regions of convergence."

GetRMinus::usage =
	"GetRMinus returns the value of the Rminus component of the \
	region of convergence contained in the passed argument. \
	So, GetRMinus[ZTransData[x,rm,rp,z]] returns the value of rm; \
	GetRMinus[Rminus[x]] returns x; and GetRMinus[x] returns x."

GetRPlus::usage =
	"GetRPlus returns the value of the Rplus component of the \
	region of convergence contained in the passed argument. \
	GetRPlus[ZTransData[x,rm,rp,z]] returns the value of rp; \
	GetRPlus[Rplus[x]] returns x; and GetRPlus[x] returns x."

InfCheck::usage =
	"InfCheck[expr] returns Infinity if expr is any Infinity form; \
	otherwise, expr is returned."

IntersectsROC::usage =
	"Intersects[rm1,rp1,rm2,rp2] returns True if the interval (rm1,rp1) \
	overlaps with the interval (rm2,rp2) in any way.  Therefore, \
	IntersectsROC will return False when the two intervals only overlap \
	at endpoints."

LTransData::usage =
	"LTransData is a data tag for a valid Laplace transform object."

LVariables::usage =
	"LVariables is a data tag for variables in a Laplace transform object."

Rminus::usage =
	"Rminus[ <r-minus> ] is the head for the R- value in a transform."

Rplus::usage =
	"Rplus[ <r-plus> ] is the head for the R+ value in a transform."

Transform::usage =
	"Transform[transform, rminus, rplus] represents a transform. \
	Transform is also a tag for exporting errors messages like \
	Transform::twosided.  This object is used exclusively by the \
	various transform rule bases."

WithinROC::usage =
	"WithinROC[rm1,rp1,rm2,rp2] returns True if the interval [rm1,rp1] \
	is a subset of the interval [rm2,rp2]."

ZTransData::usage =
	"ZTransData is a data tag for a valid z-transform object."

ZVariables::usage =
	"ZVariables is a data tag for the variables in a z-transform object. \
        When applied to an object x containing z-transform information, \
	ZVariables[x] returns the z-variables contained in the z-transform. \
	For example, if x is a z-transform object of the form \
	ZTransData[ zexpr, Rminus[r-], Rplus[r+], ZVariables[z] ], then \
	ZVariables would return z."

(*  E N D     U S A G E     I N F O R M A T I O N  *)


Begin["`Private`"]


(*  R E G I O N     O F     C O N V E R G E N C E  *)

(*  FindRMinus  *)
FindRMinus[rminus1_, rminus2_, lowlimit_:0] :=
	Which [ SameQ[rminus1, Infinity], Infinity,
		SameQ[rminus2, Infinity], Infinity,
		SameQ[rminus1, lowlimit], rminus2,
		SameQ[rminus2, lowlimit], rminus1,
		True, Max[rminus1, rminus2] ]

(*  FindRPlus  *)
FindRPlus[rminus1_, rminus2_, lowlimit_:0] :=
	Which [ SameQ[rminus1, Infinity], rminus2,
		SameQ[rminus2, Infinity], rminus1,
		SameQ[rminus1, lowlimit], lowlimit,
		SameQ[rminus2, lowlimit], lowlimit,
		True, Min[rminus1, rminus2] ]

(*  GetRMinus  *)
GetRMinus[Transform[x_, rm_, rp_]] := rm
GetRMinus[ZTransData[x_, rm_, rp_, z_]] := GetRMinus[rm]
GetRMinus[LTransData[x_, rm_, rp_, z_]] := GetRMinus[rm]
GetRMinus[Rminus[x__]] := x

(*  GetRPlus  *)
GetRPlus[Transform[x_, rm_, rp_]] := InfCheck[rp]
GetRPlus[ZTransData[x_, rm_, rp_, z_]] := GetRPlus[rp]
GetRPlus[LTransData[x_, rm_, rp_, z_]] := GetRPlus[rp]
GetRPlus[Rplus[x_]] := InfCheck[x]

(*  InfCheck  *)
SetAttributes[InfCheck, Listable]
InfCheck[x_] := If [ InfinityQ[x], Infinity, x ]

(*  IntersectsROC  *)
IntersectsROC[rm1_, rp1_, rm2_, rp2_] :=
	(InRange[rm2,rm1,rp2,0,Infinity] || InRange[rm2,rp1,rp2,0,Infinity]) &&
	(! SameQ[rp1, rm2] && ! SameQ[rp2, rm1] )

(*  WithinROC  *)
WithinROC[rm1_, rp1_, rm2_, rp2_] := N[ ( rm1 >= rm2 ) && ( rp1 <= rp2 ) ]

(*  ZVariables  *)
ZTransData/: ZVariables[ZTransData[f_, rm_, rp_, z_]] := ZVariables[z]
ZVariables[ZVariables[z_]] := z


(*  E N D     P A C K A G E  *)

End[]
EndPackage[]

If [ TrueQ[ $VersionNumber >= 2.0 ],
     On[ General::spell ];
     On[ General::spell1 ] ]


(*  H E L P     I N F O R M A T I O N  *)

Block [	{newfuns},
	newfuns =  { FindRMinus, FindRPlus,	GetRMinus, GetRPlus,
		     InfCheck,   IntersectsROC, WithinROC };
	Combine[ SPfunctions, newfuns ];
	Apply[ Protect, newfuns ] ]

Protect[Rminus, Rplus]


(*  E N D I N G     M E S S A G E  *)

Print["Region of convergence routines for the transform rule bases are loaded."]
Null
