/* adapted by David Cortes Provencio */
/* this is a subset with changes of: */
/*===================================*/
/*                                   */
/* Standard MIDI File Routines       */
/*                                   */
/* Copyright (c) 1990                */
/* By Music Quest, Inc.              */
/*                                   *==================================*/
/*                                                                      */
/* For more information on standard MIDI files, consult the             */
/* Standard MIDI File 1.0 Specification which is available from the:    */
/*                                                                      */
/* International MIDI Association.                                      */
/* 5316 W. 57th Street                                                  */
/* Los Angeles, CA 90056                                                */
/* (213) 649-6434                                                       */
/*                                                                      */
/* Updates                                                              */
/* PR238 09/16/90 - large model use of DS versus SS                     */
/*                                                                      */
/*======================================================================*/

#include <io.h>
#include <dos.h>
#include <stdio.h>
#include <malloc.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <memory.h>
#include "stdmblok.h"
#include <string.h>

/*===================================*/
/* Return data length for given      */
/* MIDI status.                      */
/*===================================*/
static int m_cmddl[8] = {2,2,2,2,1,1,2,0}; /* midi command data lengths */
int midi_datalng(unsigned char rs)
{
  return(m_cmddl[(rs >> 4) & 7]);
}

/*===================================*/
/* Validity check SMF header         */
/*===================================*/
valid_smf(unsigned char *smfhdr)
{
  static char msf_header[] = "MThd";
  char *vp;

  vp=msf_header;
  while (*vp)
    if (*vp++ != *smfhdr++)
      return(0);
  return(1);
}

/*===================================*/
/* smfblok constructor               */
/*===================================*/
smfblok::smfblok()
{
  smf_data=0L;
  smf_open=0;
}

/*===================================*/
/* Open a standard MIDI file         */
/*===================================*/
int smfblok::open_std_mfile(unsigned char *fn)
{
  unsigned char hchunk[14];
  unsigned char *tdata;
  unsigned char buffer[1024];
  unsigned lng;
  int rc;

  if (smf_open)                         /* already open smfblok */
    return(-1);

  if ((smf_fh=open((char *)fn,O_RDONLY|O_BINARY)) >= 0) /* if file opened */
    {
      /* read header chunk */
      read(smf_fh,hchunk,sizeof(hchunk));
      memmove(hc_type,hchunk,sizeof(hc_type));
      hc_chunklng=smfmakelong(&hchunk[4]);
      hc_format=smfmakeint(&hchunk[8]);
      hc_ntrks=smfmakeint(&hchunk[10]);
      hc_division=smfmakeint(&hchunk[12]);

      if (valid_smf(hchunk))            /* if file passes validity checks */
        {
          /* allocate space for file data */
          tdata=smf_data=(unsigned char *)malloc(filelength(smf_fh));

          /* read file data */
          do
            {
              lng=read(smf_fh,buffer,sizeof(buffer));
              if (lng)
                /*movedata(read_SS(),FP_OFF(buffer),FP_SEG(tdata),FP_OFF(tdata),lng);*/
                memmove(tdata,(char *)buffer,lng);
              tdata+=lng;
            }
          while (lng > 0);
          smf_open=rc=1;                /* file now open */
        }
      else                              /* if file not recognized */
        {
          rc=0;
        }
    }
  else                                  /* if file open failed */
    {
      rc=0;
    }
  return(rc);
}

/*===================================*/
/* smfblok destructor                */
/*===================================*/
smfblok::~smfblok()
{
  close_std_mfile();                    /* make sure file was closed */
}

/*===================================*/
/* Close a standard MIDI file        */
/*===================================*/
int smfblok::close_std_mfile()
{
  if (smf_open)
    {
      close(smf_fh);
      smf_fh=-1;
      if (smf_data != 0L)               /* if track was being read */
	free((unsigned char  *)smf_data);
      smf_open=0;
    }
  return(1);
}

/*===================================*/
/* tiob constructor                  */
/*===================================*/
tiob::tiob()
{
  t_rwaddr=0L;
  t_ceaddr=0L;
  t_trkaddr=0L;
  t_rwflag=t_eof=t_open=0;
  t_first_event=1;
  t_cmdl=2;
  t_cmd=0x90;
}

/*===================================*/
/* tiob destructor                   */
/*===================================*/
tiob::~tiob()
{
  close_track();                        /* make sure file was closed */
}

/*===================================*/
/* Logically rewind the track        */
/*===================================*/
void tiob::rewind_track()
{
  if (t_rwflag == 0)
    {
      t_rwaddr=t_trkaddr+8;
      t_rwflag=t_eof=0;
      t_first_event=1;
      t_cmdl=2;
      t_cmd=0x90;
    }
}


/*===================================*/
/* Open a track for reading          */
/*===================================*/
int tiob::open_track(smfblok *smfb, unsigned n)
{
  unsigned char ldata[4];

  if (t_open)                           /* tiob already open */
    return(-1);

  t_smfb=smfb;
  t_trkaddr=smfb->smflocate_track(n);
  memmove((char *)ldata,t_trkaddr+4,sizeof(ldata));
  t_length=smfmakelong(ldata);
  rewind_track();
  t_open=1;

  return(1);
}

/*===================================*/
/* Close a track                     */
/*===================================*/
int tiob::close_track()
{
  long next_track;
  unsigned char l32[4];

  if (t_open)
    if (t_rwflag)                       /* if track was being written */
      {
	next_track=tell(t_smfb->get_smf_fh());
	lseek(t_smfb->get_smf_fh(),t_wlength_fptr,0); /* position to track length field */
	write_long(l32,t_length);
	write(t_smfb->get_smf_fh(),l32,sizeof(l32));
	lseek(t_smfb->get_smf_fh(),next_track,0); /* position to start of new track */
      }
  return(1);
}

/*===================================*/
/* Read the next byte from a track   */
/*===================================*/
unsigned char tiob::read_byte()
{
  return(*t_rwaddr++);
}

/*===================================*/
/* Read the next track event         */
/*===================================*/
int tiob::read_next_event(unsigned char *etype, long *etime, unsigned char *estatus,
			  unsigned char  *(*edata), unsigned *elng)
{
  unsigned char db;
  unsigned dl;
  unsigned mdx;
  unsigned char *mdata;

  t_ceaddr=(unsigned char  *)t_rwaddr; /* checkpt current event */
  *etime=smfread_vlq();                 /* get delta timing value */

  db=read_byte();                       /* get first data byte */
  switch (db)                           /* cases based on cmd PR85 */
    {
      case 0xF0:                        /* sysex event */
      case 0xF7:
	*etype=SYSEXEVENT;
	*estatus=db;
	*elng=(unsigned)smfread_vlq();  /* meta event length */
	*edata=t_rwaddr;                /* ptr to event data within track */
	t_rwaddr+=*elng;                /* advance r/w pointer */
	break;
      case 0xFF:                        /* meta event */
	*etype=METAEVENT;
	*estatus=read_byte();           /* meta event type */
	*elng=(unsigned)smfread_vlq();  /* meta event length */
	*edata=t_rwaddr;                /* ptr to event data within track */
        t_rwaddr+=*elng;                /* advance r/w pointer */
        if (*estatus == METAEOT)        /* end of track event */
          t_eof=1;
        break;
      default:
	*etype=MIDIEVENT;
        if (db & 0x80)                  /* new status */
          {
            *estatus=t_cmd=db;          /* new track running status */
            dl=t_cmdl=*elng=midi_datalng(t_cmd); /* transmit data length */
            for (mdx=0, mdata=t_mdata; mdx < dl; mdx++) /* fetch data */
	      *mdata++=read_byte();
            *edata=(unsigned char  *)t_mdata; /* ptr to data */
          }
        else
          {
            *estatus=t_cmd;             /* return current status */
	    *elng=dl=t_cmdl;            /* repeated command data len */
            mdata=t_mdata;              /* return data in work area */
            *mdata++=db;
            for (mdx=1; mdx < dl; mdx++) /* fetch data */
              *mdata++=read_byte();
            *edata=(unsigned char  *)t_mdata; /* ptr to data */
	  }
        break;
    }

  return(t_eof);                        /* return eof status */
}

/*===================================*/
/* Make a long out of a string       */
/*===================================*/
long smfmakelong(unsigned char *s)
{
  long v;

  v=((long)s[0] << 24) + ((long)s[1] << 16) + ((long)s[2] << 8) + s[3];
  return(v);
}

/*===================================*/
/* Make an integer out of a string   */
/*===================================*/
unsigned smfmakeint(unsigned char *s)
{
  return((s[0] << 8) + s[1]);
}

/*===================================*/
/* Locate track 'n' within file      */
/*===================================*/
unsigned char  *smfblok::smflocate_track(unsigned n)
{
  unsigned char  *hptr;
  unsigned char  *fptr;
  unsigned char ldata[4];
  long tlng;

  hptr=smf_data;
  while (n--)
    {
      fptr=(unsigned char *)hptr; 
      memmove((char *)ldata,fptr+4,sizeof(ldata));
      tlng=smfmakelong(ldata)+8L;
      hptr+=tlng;
    }
  return((unsigned char *)hptr);
}

/*===================================*/
/* Read a variable length quantity   */
/*===================================*/
long tiob::smfread_vlq()
{
  long v;
  unsigned char c;

  if ((v=read_byte()) & 0x80)
    {
      v&=0x7F;
      do
	{
	  v=(v << 7) + ((c=read_byte()) & 0x7F);
	}
      while (c & 0x80);
    }
  return(v);
}

/*===================================*/
/* Write variable length quantity    */
/*===================================*/
unsigned char *write_vlq(unsigned char *s /* output string */, unsigned long v /* value */)
{
  unsigned char sv[10];
  unsigned char *svp;

  svp=sv;
  *svp++=v & 0x7F;
  while ((v >>= 7) > 0)
    *svp++=(v & 0x7F) | 0x80;

  svp--;
  for ( ; ;)
    {
      *s++=*svp;
      if (*svp & 0x80)
	svp--;
      else
	break;
    }
  return(s);                            /* return ptr to next available byte */
}

/*===================================*/
/* Write 32 bit integer              */
/*===================================*/
unsigned char *write_long(unsigned char *s /* output string */, unsigned long v /* value */)
{
  int i;

  s+=3;
  for (i=0; i < 4; i++, v >>= 8)
    *s--=v & 0xFF;
  return(s+5);                          /* return ptr to next available byte */
}
