/***********************************************************************\
 *                                PC2.c                                *
 *              Copyright (C) by Stangl Roman, 1993, 1994              *
 * This Code may be freely distributed, provided the Copyright isn't   *
 * removed, under the conditions indicated in the documentation.       *
 *                                                                     *
 * PC/2 - Program Commander/2 is a configurable program starter for    *
 * OS/2 2.x PM. If the user clicks button 1 on the DESKTOP, a user     *
 * modifyable popup menu is displayed. The user then selects a program *
 * to be started, or configuration of PC/2 or dismisses it.            *
 * You can define Hotkeys to switch to a program, or start it if it is *
 * not already running.                                                *
 * PC/2 is an alternative method of starting programs compared to      *
 * icons and uses no space on DESKTOP, and no folder must be opended   *
 * to start a program. For frequently used programs, this reduces the  *
 * time to start an application.                                       *
 * PC/2 also implements an optional virtual Desktop and sliding focus. *
 *                                                                     *
\***********************************************************************/

static char RCSID[]="@(#) $Header: PC2.c/PC2.h Version 1.70 06,1994 $ (LBL)";

#define         _FILE_  "PC/2 - PC2.c V1.70"

#include        "PC2.h"                 /* User include files */
#include        "Error.h"

HEV             hevPc2;                 /* Handle of PC/2 semaphore */
BOOL            InstallHelp;            /* True if we're installing */
UCHAR           *pucFilenameProfile;    /* The buffer holding the filename of the profile */
UCHAR           *pucFilenameINI;        /* Path and filename of PC2.INI */
UCHAR           *pucFilenameHLP;        /* The buffer holding the filename of the HLP file */
HMODULE         hDllPc2;                /* Handle of PC/2 DLL */
TID             tidThread;              /* Working thread ID */
/*                                                                                      *\
 * Reserve data referenced generally through all modules. This isn't the best way, hope *
 * that I have time to port to C++...                                                   *
\*                                                                                      */
HAB             hab;                    /* Handle of PM anchor block */
HMQ             hmq;                    /* Handle of message queue */
HWND            hwndFrame;              /* Frame handle of window */
HWND            hwndClient;             /* Client handle of window */
HWND            hwndPopupMenu;          /* Handel of popup menu window */
                                        /* Input options: mouse button 1 depressed,
                                           Input devices keyboard or mouse button 1,
                                           popup menu allways visible on DESKTOP,
                                           position so that ID_CONFIGDIALOG is under the
                                           pointer */
HWND            hwndHelp;               /* Help window handle */
SESSIONDATA     SessionData;            /* Used by Menu Installation dialog and by
                                           Program Installation dialog to store menu or
                                           program data, to be filled from the user or
                                           to be presented to the user. */
MENUDATA        *pPopupMenu=NULL;       /* Used by all procedures as the starting point
                                           of a linked list of menu entries. */
MENUDATA        *pMenuData;             /* This pointer points to the current level of
                                           Submenus and Menuitems within the configuration
                                           dialog procedure */
                                        /* Create linked list by starting with this ID */
USHORT          MenuDataId=ID_POPUPMENU;
USHORT          DialogResult;           /* Each dialog procedure returns the result in
                                           this variable to enable the calling routine to
                                           check, if there is valid data or not. */
FILE            *Pc2Profile;            /* Open the profile, where the user entered menu
                                           data is stored, with this handle */
SWP             swpScreen;              /* The screen dimensions */
BOOL            bConfigDialog=FALSE;    /* TRUE only if the Configuration dialog exists */
WINDOWS         Windows;                /* All windows on PM are traced in this structure */

/*--------------------------------------------------------------------------------------*\
 * The main procedure.                                                                  *
 * Req:                                                                                 *
 *      argc, argv, envp                                                                *
 * Returns:                                                                             *
 *      int ........... Exitcode (0, or errorlevel)                                     *
\*--------------------------------------------------------------------------------------*/
int main(int argc, char *argv[], char *envp[])
{
QMSG    qmsg;                           /* Message queue */
ULONG   counter;                        /* Temporary counter */
                                        /* Frame creation control flag */
ULONG   flCreate=FCF_ICON | FCF_SIZEBORDER | FCF_TASKLIST | FCF_MENU | FCF_TITLEBAR | FCF_HIDEBUTTON | FCF_ACCELTABLE;

                                        /* Default the Popup-Menu is displayed after a
                                           WM_BUTTON1DBLCLK on the Desktop, but users may
                                           prefer WM_BUTTON1CLICK to popup the menu */
HookParameters.ulClickFlag=WM_BUTTON1DBLCLK;
/*                                                                                      *\
 * Get the full path and filename of the running copy of PC/2 and change the extension  *
 * .EXE into .cfg to open the configuration file under this name. If the user supplies  *
 * [-,/Profile filename.ext] then use this filename as the Profile. Also change .EXE    *
 * into .HLP, .INI and PC/2 directory as the current directory to access .DLL.          *
\*                                                                                      */
                                        /* Long enough to hold user Profile name */
pucFilenameProfile=malloc(strlen(argv[0])+64);
pucFilenameINI=malloc(strlen(argv[0])+1);
pucFilenameHLP=malloc(strlen(argv[0])+1);
strcpy(pucFilenameProfile, argv[0]);
strcpy(pucFilenameINI, argv[0]);
strcpy(pucFilenameHLP, argv[0]);
strcpy(strchr(pucFilenameProfile, '.'), ".cfg");
strcpy(strchr(pucFilenameINI, '.'), ".ini");
strcpy(strchr(pucFilenameHLP, '.'), ".hlp");
InstallHelp=FALSE;                      /* Assume no installation */
for(counter=1; counter<argc; counter++)
    {
    strupr(argv[counter]);              /* Convert to uppercase */
                                        /* Test for /PROFILE or -PROFILE to get a
                                           profile name */
    if((strstr(argv[counter], "/PROFILE")!=NULL) ||
        (strstr(argv[counter], "-PROFILE")!=NULL))
        strcpy((pucFilenameProfile+strlen(pucFilenameProfile)-7), argv[counter+1]);
                                        /* Test for /INSTALL or -INSTALL to start the help
                                           panels during initialization */
    if((strstr(argv[counter], "/INSTALL")!=NULL) ||
        (strstr(argv[counter], "-INSTALL")!=NULL))
        InstallHelp=TRUE;
                                        /* Test for /DOUBLECLICK or -DOUBLECLICK to display
                                           the Popup-Menu after a double-click instead of
                                           a single click */
    if((strstr(argv[counter], "/DOUBLECLICK")!=NULL) ||
        (strstr(argv[counter], "-DOUBLECLICK")!=NULL))
        HookParameters.ulClickFlag=WM_BUTTON1DBLCLK;
                                        /* Test for /SINGLECLICK or -SINGLECLICK to display
                                           the Popup-Menu after a double-click instead of
                                           a single click */
    if((strstr(argv[counter], "/SINGLECLICK")!=NULL) ||
        (strstr(argv[counter], "-SINGLECLICK")!=NULL))
        HookParameters.ulClickFlag=WM_BUTTON1CLICK;
    }
INIAccess(pucFilenameINI, TRUE);        /* Get data from PC2.INI into HookParameters */
do
{
                                        /* Initialize anchor block and message queue */
    if(WinStartUp(&hab, &hmq)==FALSE) break;
    HookParameters.habWindow=hab;       /* Save anchor block handle */
    if(!WinRegisterClass(               /* Register window class */
        hab,                            /* Handle of anchor block */
        (PSZ)PC2_CLASSNAME,             /* Window class name */
        (PFNWP)PC2_MainWindowProc,      /* Address of window procedure */
                                        /* Class style */
        CS_SIZEREDRAW | CS_SAVEBITS | CS_MOVENOTIFY,
        0))                             /* Extra window words */
        {
        GEN_ERR(hab, (HWND)NULL, (HWND)NULL);
        break;
        }
/*                                                                                      *\
 * Check if we are allready loaded before by querying a semaphore that is defined the   *
 * first time PC/2 runs.                                                                *
\*                                                                                      */
    if(DosCreateEventSem(               /* Create a semaphore */
        PC2_SEM,                        /* Name */
        &hevPc2,                        /* Handle */
        (ULONG)0,                       /* Named semaphores are allways shared */
        (BOOL32)FALSE))                 /* Initially set */
        {                               /* If an error occurs, either we can't create
                                           the semaphore or it allready exists. We assume
                                           that it exists, meaning PC/2 allready loaded.
                                           Parameters NULL force an immediate exit(1). */
        USR_ERR("PC/2 already loaded - exiting...", (HWND)NULL, (HWND)NULL);
        break;                          /* Not needed but for safety... */
        }
/*                                                                                      *\
 * Now get device resolution and save it and other initilization data to the structure  *
 * used to communicate with PC2HOOK.DLL.                                                *
\*                                                                                      */
                                        /* Query and save the device resolution, f.e.
                                           1024 * 768 */
    swpScreen.cx=WinQuerySysValue(HWND_DESKTOP, SV_CXSCREEN);
    swpScreen.cy=WinQuerySysValue(HWND_DESKTOP, SV_CYSCREEN);
                                        /* Now initialized the virtual Desktop data */
    HookParameters.DesktopSize.x=swpScreen.cx;
    HookParameters.DesktopSize.y=swpScreen.cy;
    HookParameters.LLHotBorder.x=HookParameters.DesktopSize.x*0.15;
    HookParameters.LLHotBorder.y=HookParameters.DesktopSize.y*0.15;
    HookParameters.URHotBorder.x=HookParameters.DesktopSize.x*0.85;
    HookParameters.URHotBorder.y=HookParameters.DesktopSize.y*0.85;
    HookParameters.VirtualDesktopPos.x=0;
    HookParameters.VirtualDesktopPos.y=0;
    HookParameters.VirtualDesktopMin.x=-swpScreen.cx;
    HookParameters.VirtualDesktopMin.y=-swpScreen.cy;
    HookParameters.VirtualDesktopMax.x=swpScreen.cx;
    HookParameters.VirtualDesktopMax.y=swpScreen.cy;
    HookParameters.SlidingXFactor=(HookParameters.ulScrollPercentage*
        HookParameters.DesktopSize.x)/100;
    HookParameters.SlidingYFactor=(HookParameters.ulScrollPercentage*
        HookParameters.DesktopSize.y)/100;
    HookParameters.hwndDesktop=0;       /* Initialize to a value not used by PM (hopefully) */
    HookParameters.hwndWPS=0;
/*                                                                                      *\
 * Pc2Hook.dll is already loaded from the directory PC/2 was started from or via the    *
 * LIBPATH. We just need to query the handle.                                           *
\*                                                                                      */
    if(DosQueryModuleHandle(            /* Get the module handle of a DLL that was previously loaded */
        "PC2Hook",                      /* ASCIIZ name of the DLL */
        &hDllPc2)!=NO_ERROR)            /* DLL handle returned */
        {
        USR_ERR("Can't determine module handle of PC2HOOK.DLL - exiting...", HWND_DESKTOP, HWND_DESKTOP);
        break;
        }
    PC2DLL_SetParameters();             /* Initialize data in PC2Hook DLL */
/*                                                                                      *\
 * Start frame window, which creates window(s) and WM_CREATE message.                   *
\*                                                                                      */
    if((hwndFrame=WinCreateStdWindow(   /* Create a standard window */
        HWND_DESKTOP,                   /* DESKTOP is parent */
        0,                              /* Standard window styles */
        &flCreate,                      /* Frame control flags */
        (PSZ)PC2_CLASSNAME,             /* Client window class name */
        "",                             /* No window text */
        0,                              /* No special class style */
        (HMODULE)0,                     /* Ressource is in .EXE file */
        ID_PC2MAINWINDOW,               /* Frame window identifier */
        &hwndClient)                    /* Client window handle */
        )==NULLHANDLE)
        {
        GEN_ERR(hab, (HWND)NULL, (HWND)NULL);
        break;
        }
    HookParameters.hwndPC2=hwndClient;  /* Save window handle to send message to display
                                           the Popup-Menu */
/*                                                                                      *\
 * Now initilize Help, if it can't be initialized the we get no help but that's no      *
 * reason to terminate.                                                                 *
\*                                                                                      */
    if(WinStartHelp(hab, pucFilenameHLP, &hwndHelp)==FALSE)
        USR_ERR("Can't find PC2.HLP, please check HLP file and HELP - ignoring help requests...",
            hwndFrame, hwndClient);
/*                                                                                      *\
 * Set text to titlebar and size & position. Ensure that position is limited.           *
\*                                                                                      */
    if((HookParameters.swpPC2.x<HookParameters.VirtualDesktopMin.x) ||
        (HookParameters.swpPC2.x>HookParameters.VirtualDesktopMax.x))
        HookParameters.swpPC2.x=8;
    if((HookParameters.swpPC2.y<HookParameters.VirtualDesktopMin.y) ||
        (HookParameters.swpPC2.y>HookParameters.VirtualDesktopMax.y))
        HookParameters.swpPC2.y=8;
    if(HookParameters.swpPC2.cx>HookParameters.DesktopSize.x)
        HookParameters.swpPC2.cx=256;
    if(HookParameters.swpPC2.cy>HookParameters.DesktopSize.y)
        HookParameters.swpPC2.cy=192;
    WinSetWindowText(hwndFrame, "PC/2");
    if(!WinSetWindowPos(                /* Set window postion */
        hwndFrame,                      /* Window handle */
        HWND_BOTTOM,
        HookParameters.swpPC2.x,
        HookParameters.swpPC2.y,
        HookParameters.swpPC2.cx,
        HookParameters.swpPC2.cy,
        SWP_HIDE | SWP_MOVE | SWP_SIZE | SWP_DEACTIVATE))
        GEN_ERR(hab, (HWND)NULL, (HWND)NULL);
                                        /* Display overview window if requested */
    if(HookParameters.ulStatusFlag & OVERVIEW)
        WinSetWindowPos(hwndFrame, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER | SWP_SHOW | SWP_DEACTIVATE);
/*                                                                                      *\
 * Now setup the Popup-Menu by loading the data from the profile and install the hook   *
 * into the system input queue.                                                         *
\*                                                                                      */
    hwndPopupMenu=WinLoadMenu(          /* Load popup menu */
        hwndClient,                     /* Owner window handle */
        (HMODULE)0,                     /* Ressource in .EXE file */
        ID_PC2POPUPMENU);               /* Menu identifier in ressource file */
                                        /* Load the data from the profile */
    WinPostMsg(hwndClient, WM_SETPOPUPMENU, NULL, NULL);
                                        /* Now install the hook */
    WinPostMsg(hwndClient, WM_LOADHOOK, NULL, NULL);
/*                                                                                      *\
 * Here we loop dispatching the messages...                                             *
\*                                                                                      */
    while(WinGetMsg(hab, &qmsg, 0, 0, 0))
        WinDispatchMsg(hab, &qmsg);     /* Dispatch messages to window procedure */
    WinDestroyWindow(hwndFrame);        /* Close window */
} while (FALSE);

if(WinCloseDown(&hwndHelp, &hab, &hmq)==FALSE)
    {
                                        /* Wait for working thread to close */
    DosWaitThread(&tidThread, DCWW_WAIT);
    return(1);
    }
else
    {
                                        /* Wait for working thread to close */
    DosWaitThread(&tidThread, DCWW_WAIT);
    return(0);
    }
}

/*--------------------------------------------------------------------------------------*\
 * This procedure is the PC/2 window procedure.                                         *
\*--------------------------------------------------------------------------------------*/
MRESULT EXPENTRY PC2_MainWindowProc(HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2)
{
static USHORT   usMenuCommand;          /* Last message from smarticons */
static HPOINTER hPointerMove;           /* Pointer when moving windows on overview window */
static HPOINTER hPointerAction;         /* Pointer when performing the other actions on windows */
static HWND     hwndMenu;               /* PC/2's menu window handle */

switch(msg)
{
case WM_CREATE:                         /* Create window by WinCreateStdWindow() */
    {
                                        /* First call default window procedure */
    WinDefWindowProc(hwnd, msg, mp1, mp2);
                                        /* Load pointers from resource */
    hPointerMove=WinLoadPointer(HWND_DESKTOP, NULLHANDLE, ID_MOVEPOINTER);
    hPointerAction=WinLoadPointer(HWND_DESKTOP, NULLHANDLE, ID_ACTIONPOINTER);
    usMenuCommand=ID_ICONMOVE;          /* Setup smarticon's control */
    Windows.ulDesktop=(ULONG)-1;        /* Set to -1 because currently we don't know Desktop's name */
    Windows.ulWindowList=(ULONG)-1;     /* Set to -1 because currently we don't know Window List's name */
    if(InstallHelp==TRUE)               /* For installation display help panels */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_HELP), NULL);
                                        /* Start working thread */
    tidThread=_beginthread(PC2_Thread, NULL, 65536, NULL);
    if(tidThread==(TID)-1)
        {
        USR_ERR("Can't start working thread - exiting...",
            hwndFrame, hwndClient);
                                        /* Post exit message */
        WinPostMsg(hwnd, WM_QUIT, NULL, NULL);
        }
    }
    break;

case WM_PAINT:                          /* Draw overview of virtual Desktop */
    {
    HPS         hpsClient;
                                        /* Get a cached presentation space */
    hpsClient=WinBeginPaint(hwnd, NULLHANDLE, NULL);
                                        /* Request repaint from working thread */
    WinPostMsg(hwndThread, WM_REPAINT, NULL, NULL);
    WinEndPaint(hpsClient);
    }
    break;

/*                                                                                      *\
 * Syntax: WM_SETPOPUPMENU, NULL, NULL                                                  *
\*                                                                                      */
case WM_SETPOPUPMENU:
/*                                                                                      *\
 * Open the profile for reading the linked list containing the popup menu data. If the  *
 * profile can't be opened, the file is assumed to be empty so the popup menu is empty. *
\*                                                                                      */
    if((Pc2Profile=fopen(pucFilenameProfile, "r"))==NULL)
        {
        pPopupMenu=AllocateMenuData();  /* Allocate an empty MENUDATA structure used as
                                           the first element of linked list */
        USR_ERR("Cannot open confguration file - assuming empty file", hwndFrame, hwndClient);
        }
    else
        {
        static UCHAR    Buffer[256];

        bConfigDialog=FALSE;            /* Assume the Configuration Dialog isn't part
                                           of the profile */
        pPopupMenu=AllocateMenuData();  /* Allocate an empty MENUDATA structure used as
                                           the first element of linked list */
        fgets(Buffer, sizeof(Buffer), Pc2Profile);
        if(strcmp(Buffer, "PROFILE START\n")==0)
            LoadMenu(pPopupMenu);       /* Load the rest by calling a recursive procedure */
        fclose(Pc2Profile);
        }
    pMenuData=pPopupMenu;               /* Initialize *MENUDATA for Configuration dialog
                                           procedure to a known value */
    break;

/*                                                                                      *\
 * Syntax: WM_LOADHOOK, NULL, NULL                                                      *
\*                                                                                      */
case WM_LOADHOOK:
/*                                                                                      *\
 * Install the hook into the system input queue pointing to the PC2DLL_Hook() procedure *
 * in the DLL PC2HOOK.DLL. If we can't do this we exit after an error message box.      *
\*                                                                                      */
                                        /* Query and set the window handle of the Desktop */
    WinSendMsg(hwnd, WM_SETDESKTOPHANDLE, NULL, NULL);
    if(WinSetHook(                      /* Set a hook */
        hab,                            /* Handle of anchor block */
        NULLHANDLE,                     /* Hook into system message queue */
        HK_INPUT,                       /* Hook of system input queue */
        (PFN)PC2DLL_Hook,               /* Pointer to hook procedure */
        hDllPc2)==FALSE)
        {
        USR_ERR("Hooking the system input queue failed - exiting...", hwndFrame, hwndClient);
        WinPostMsg(hwnd, WM_QUIT, NULL, NULL);
        }
    break;

/*                                                                                      *\
 * Syntax: WM_DESKTOPMOVE, LONG SlidingXFactor, LONG SlidingYFactor                     *
\*                                                                                      */
case WM_DESKTOPMOVE:
                                        /* Post message to working thread */
    WinPostMsg(hwndThread, WM_DESKTOPMOVE, mp1, mp2);
    break;

case WM_SIZE:
                                        /* Request repaint from working thread */
    WinPostMsg(hwndThread, WM_REPAINT, NULL, NULL);
case WM_MOVE:
                                        /* Get the frame area size */
    WinQueryWindowPos(hwndFrame, &HookParameters.swpPC2);
    break;

/*                                                                                      *\
 * Syntax: WM_DESKTOPMOVE, NULL, NULL                                                   *
\*                                                                                      */
case WM_MOUSEMOVE:                      /* Set mouse pointer accordingly to last smarticon pressed */
    if(usMenuCommand==ID_ICONMOVE) WinSetPointer(HWND_DESKTOP, hPointerMove);
    else WinSetPointer(HWND_DESKTOP, hPointerAction);
    break;
 
/*                                                                                      *\
 * Syntax: WM_HOTKEY, (USHORT usFlags, USHORT usCh), ULONG ulKeyDataIndex               *
\*                                                                                      */
case WM_HOTKEY:
                                        /* Just pass this message to the working thread */
    WinPostMsg(hwndThread, WM_HOTKEY, MPFROMLONG(mp1), MPFROMLONG(mp2));
    break;

case WM_BUTTON1DOWN:
    return((MRESULT)TRUE);              /* Avoid default window procedure which sets the
                                           focus to PC/2 */
    break;

/*                                                                                      *\
 * Syntax: WM_BUTTON2DOWN, (LONG lClickX), (LONG lClickY)                               *
\*                                                                                      */
case WM_BUTTON2DOWN:
/*                                                                                      *\
 * This message detected and passed from the PC/2 window procedure is used to track or  *
 * perform certain actions on windows on the Virtual Desktops.                          *
\*                                                                                      */
    {
    LONG        lClickX, lClickY;       /* Pointer position during click */
    ULONG       ulWindowIndex;          /* Index in Windows.wdWindow[] */
    BOOL        bFound=FALSE;           /* TRUE if one window was found */
    TRACKINFO   tiWindow;               /* Trackinfo structure for one window */

                                        /* Get position of mouse button 1 down */
    lClickX=(ULONG)(SHORT1FROMMP(mp1));
    lClickY=(ULONG)(SHORT2FROMMP(mp1));
    lClickX=((float)(lClickX-HookParameters.ptlOrigin.x))/HookParameters.fScaleX;
    lClickY=((float)(lClickY-HookParameters.ptlOrigin.y))/HookParameters.fScaleY;
                                        /* Loop for all windows, from topmost to bottommost, on
                                           overview window */
    for(ulWindowIndex=0;
        Windows.ulWindowLast!=(ULONG)-1 && ulWindowIndex<=Windows.ulWindowLast;
        ulWindowIndex++)
        {
                                        /* Ignore invisible windows */
        if(!(Windows.wdWindow[ulWindowIndex].ulStatus & VISIBLE)) continue;
        if((Windows.wdWindow[ulWindowIndex].swpWindow.x<=lClickX) &&
            (Windows.wdWindow[ulWindowIndex].swpWindow.x+Windows.wdWindow[ulWindowIndex].swpWindow.cx>=lClickX) &&
            (Windows.wdWindow[ulWindowIndex].swpWindow.y<=lClickY) &&
            (Windows.wdWindow[ulWindowIndex].swpWindow.y+Windows.wdWindow[ulWindowIndex].swpWindow.cy>=lClickY))
            {                           /* We have found one */
            bFound=TRUE;
            break;
            }
        }
    if(bFound)
        {
        SWP     swpWindow;              /* Copy of Window's position, because working thread may update it */

        memcpy(&swpWindow, &Windows.wdWindow[ulWindowIndex].swpWindow, sizeof(SWP));
        switch(usMenuCommand)
        {
        case ID_ICONMOVE:
                                        /* Calculate and draw starting tracking rectangle */
            tiWindow.rclTrack.xLeft=HookParameters.ptlOrigin.x+
                (float)swpWindow.x*HookParameters.fScaleX;
            tiWindow.rclTrack.yBottom=HookParameters.ptlOrigin.y+
                (float)swpWindow.y*HookParameters.fScaleY;
            tiWindow.rclTrack.xRight=tiWindow.rclTrack.xLeft+
                (float)swpWindow.cx*HookParameters.fScaleX;
            tiWindow.rclTrack.yTop=tiWindow.rclTrack.yBottom+
                (float)swpWindow.cy*HookParameters.fScaleY;
            tiWindow.cxBorder=1;        /* Border width */
            tiWindow.cyBorder=1;
            tiWindow.cxGrid=1;          /* Grid width */
            tiWindow.cyGrid=1;
            tiWindow.cxKeyboard=1;      /* Movement in pixel for keyboard (keyboard currently can't
                                           start tracking) */
            tiWindow.cyKeyboard=1;
                                        /* Minimum tracking size */
            tiWindow.ptlMinTrackSize.x=2;
            tiWindow.ptlMinTrackSize.y=2;
                                        /* Maximum tracking size */
            tiWindow.ptlMaxTrackSize.x=HookParameters.swpPC2Client.cx;
            tiWindow.ptlMaxTrackSize.y=HookParameters.swpPC2Client.cy;
                                        /* Boundary rectangle */
            tiWindow.rclBoundary.xLeft=0;
            tiWindow.rclBoundary.yBottom=0;
            tiWindow.rclBoundary.xRight=HookParameters.swpPC2Client.cx;
            tiWindow.rclBoundary.yTop=HookParameters.swpPC2Client.cy;
                                        /* Tracking options */
            tiWindow.fs=TF_SETPOINTERPOS | TF_MOVE | TF_ALLINBOUNDARY;
            if(WinTrackRect(hwnd, NULLHANDLE, &tiWindow))
                {                       /* If tracking was successful reposition tracked window */
                swpWindow.x=((float)(tiWindow.rclTrack.xLeft-HookParameters.ptlOrigin.x))/HookParameters.fScaleX;
                swpWindow.y=((float)(tiWindow.rclTrack.yBottom-HookParameters.ptlOrigin.y))/HookParameters.fScaleY;
                WinSetWindowPos(swpWindow.hwnd, HWND_TOP,
                    swpWindow.x, swpWindow.y, 0, 0, SWP_MOVE | SWP_NOADJUST);
                }
            break;
 
        case ID_ICONZORDERTOP:          /* Set selected window to top of Desktop */
            WinSetWindowPos(swpWindow.hwnd, HWND_TOP, 0, 0, 0, 0, SWP_ZORDER);
            break;

        case ID_ICONZORDERBOTTOM:       /* Set selected window to bottom of Desktop */
            WinSetWindowPos(swpWindow.hwnd, HWND_BOTTOM, 0, 0, 0, 0, SWP_ZORDER);
            break;

        case ID_ICONCLOSE:              /* Close selected window by simulating the selection of
                                           the "close" menuitem of the applications's system menu */
            WinPostMsg(swpWindow.hwnd, WM_SYSCOMMAND, MPFROMLONG(SC_CLOSE), MPFROM2SHORT(CMDSRC_MENU, FALSE));
            break;

        case ID_ICONMAXIMIZE:           /* Maximize selected window */
            WinSetWindowPos(swpWindow.hwnd, HWND_TOP, 0, 0, 0, 0, SWP_MAXIMIZE);
            break;

        case ID_ICONMINIMIZE:           /* Maximize selected window */
            WinSetWindowPos(swpWindow.hwnd, HWND_TOP, 0, 0, 0, 0, SWP_MINIMIZE);
            break;

        case ID_ICONRESTORE:            /* Maximize selected window */
            WinSetWindowPos(swpWindow.hwnd, HWND_TOP, 0, 0, 0, 0, SWP_RESTORE);
            break;
        }
        }
    }
    break;

case WM_BUTTON1DBLCLK:
                                        /* Just pass this message to the working thread */
    WinPostMsg(hwndThread, WM_BUTTON1DBLCLK, MPFROMLONG(mp1), MPFROMLONG(mp2));
    break;

/*                                                                                      *\
 * Syntax: WM_WINDOWLIST, (USHORT x, USHORT y), NULL                                    *
\*                                                                                      */
case WM_WINDOWLIST:
                                        /* Pass this message to the working thread */
    WinPostMsg(hwndThread, WM_WINDOWLIST, MPFROMLONG(mp1), MPFROMLONG(mp2));
    break;

/*                                                                                      *\
 * Syntax: WM_POPUPMENU, (SHORT x, SHORT y), HWND hwndPopup                             *
\*                                                                                      */
case WM_POPUPMENU:
/*                                                                                      *\
 * The hook found that button 1 was clicked on the Desktop and sent us this message. It *
 * is either a WM_BUTTON1CLICK or WM_BUTTON1DBLCLK. First we obtain the focus, to be    *
 * able to start our programs in the foreground.                                        *
\*                                                                                      */
    {
    POINTL      ptlPopupPosition;
    HWND        hwndPopup;
    USHORT      fsOptions=PU_NONE | PU_KEYBOARD | PU_MOUSEBUTTON1 |
                          PU_HCONSTRAIN | PU_VCONSTRAIN;

    if(bConfigDialog==FALSE)            /* Display Configuration dialog if the Popup-Menu
                                           doesn't contain one */
                                        /* Post message to load Configure PC/2 dialog box.
                                           We post it because send is synchronous and will
                                           block the message queue which however must
                                           not be blocked in order to handle the
                                           Configuration dialog window procedure */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_CONFIGDIALOG), MPFROMSHORT(CMDSRC_OTHER));
    else
        {                               /* If there is one menuentry in the Popup-Menu
                                           position on the first */
        if(pPopupMenu->Item!=ENTRYEMPTY) fsOptions|=PU_POSITIONONITEM;
                                        /* Get the position and window, where the user
                                           clicked to get the Popup-Menu */
        ptlPopupPosition.x=(ULONG)SHORT1FROMMP(mp1);
        ptlPopupPosition.y=(ULONG)SHORT2FROMMP(mp1);
        hwndPopup=HWNDFROMMP(mp2);
                                        /* Map these window coordinated to display
                                           coordinates, so that the Menu will be
                                           displayed relative to the lower, left edge
                                           of the display */
        WinMapWindowPoints(hwndPopup, HWND_DESKTOP, &ptlPopupPosition,
            sizeof(ptlPopupPosition)/sizeof(POINTL));
                                        /* Set focus to our window */
        WinSetFocus(HWND_DESKTOP, hwnd);
        if(!WinPopupMenu(               /* Pop up the popup menu */
            HWND_DESKTOP,               /* Parent window handle */
            hwnd,                       /* Owner window handle that receives all the
                                           notification messages generated by the pop-up
                                           menu */
            hwndPopupMenu,              /* Popup menu window handle */
            ptlPopupPosition.x,         /* x-coordinate of mouse pointer for popup menu */
            ptlPopupPosition.y,         /* y-coordinate of mouse pointer for popup menu */
            pPopupMenu->id,             /* Input item identity, if PU_POSITIONONITEM or
                                           PU_SELECTITEM is set */
            fsOptions)                  /* Input options */
        ) GEN_ERR(hab, (HWND)NULL, (HWND)NULL);
        }
    break;
    }

/*                                                                                      *\
 * Syntax: WM_MOVEREQUEST, (USHORT usMouseXPos, USHORT usMouseYPos), (ULONG ulMoveFlag) *
\*                                                                                      */
case WM_MOVEREQUEST:
    WinPostMsg(hwndThread, WM_MOVEREQUEST, MPFROMLONG(mp1), MPFROMLONG(mp2));
    break;

case WM_CLOSE:
    if(WinMessageBox(                   /* Ask the user if he really wants to exit */
        HWND_DESKTOP, HWND_DESKTOP,
        "Are you sure you want to close PC/2?",
        "PC/2 - Program Commander/2",
        ID_PC2MAINWINDOW,
        MB_OKCANCEL | MB_ICONQUESTION | MB_DEFBUTTON1)!=MBID_OK)
        return((MRESULT)TRUE);          /* Only exit if OK is pressed */
                                        /* Send WM_QUIT to thread window, because this message
                                           terminates the message loop */
    WinPostMsg(hwndThread, WM_QUIT, NULL, NULL);
                                        /* Write changes to PC2.INI */
    INIAccess(pucFilenameINI, FALSE);
    if(WinReleaseHook(                  /* Release hook */
        hab,                            /* Handle  of anchor block */
        NULLHANDLE,                     /* Release from system hook chain */
        HK_INPUT,                       /* Hook of system input queue */
        (PFN)PC2DLL_Hook,               /* Pointer to hook procedure */
        hDllPc2)==FALSE)
        {
        USR_ERR("Unhooking the system input queue failed, System ShutDown suggested - exiting...",
            hwndFrame, hwndClient);
        WinPostMsg(hwnd, WM_QUIT, NULL, NULL);
        }
    DosFreeModule(hDllPc2);             /* Free DLL reference */
                                        /* Post exit message */
    WinPostMsg(hwnd, WM_QUIT, NULL, NULL);
    break;

case HM_ERROR:
   {
   GEN_ERR(hab, hwndFrame, hwndClient);
   break;
   }

case WM_COMMAND:
    {
    USHORT      command;

    command=SHORT1FROMMP(mp1);          /* Extract the command value */
/*                                                                                      *\
 * Filter the IDs of the user defined items of the Popup-Menu. If one is found, call    *
 * SearchItem() to search for the corresponding MENUDATA structure, copy it to a        *
 * SESSIONDATA structure and start the session.                                         *
\*                                                                                      */
    if((command>=USERITEMFIRST) && (command<=USERITEMLAST))
        {
        ULONG           id=(ULONG)command;
        MENUDATA        *pMD=NULL;

                                        /* Search in the linked list for this entry */
        if((pMD=SearchItem(pPopupMenu, &id))!=NULL)
            if(pMD->Item==ENTRYMENUITEM)
                {  
                                        /* Load SessionData with MENUDATA structure */
                LoadMenuData2SessionData(pMD, &SessionData);
                                        /* If user defined size and position defined
                                           set SWP_MOVEWINDOW flag, which invokes a
                                           reposition of the window, the first time the
                                           started application's window is found. The flag
                                           will be reset afterwards */
                if(pMD->PgmControl & SSF_CONTROL_SETPOS)
                    {                   /* If have to move the window create it invisible
                                           first to avoid drawing of window before movement.
                                           If the window should be not invisible set flag
                                           SWP_MOVEWINDOWVISIBLE to show window after movement.
                                           The flag will be reset afterwards. */
                    if(pMD->PgmControl & SSF_CONTROL_INVISIBLE)
                        pMD->SwpFlag|=SWP_MOVEWINDOW;
                    else
                        pMD->SwpFlag|=(SWP_MOVEWINDOW | SWP_MOVEWINDOWVISIBLE);
                                        /* Create window invisible before movement */
                    SessionData.PgmControl|=SSF_CONTROL_INVISIBLE;
                    }
                                        /* Start the session */
                StartSession(&SessionData);
                }
        break;                          /* We don't need further testing for this command */
        }
    hwndMenu=WinWindowFromID(hwndFrame, FID_MENU);
    switch(command)
    {
                                        /* Test for messages sent from smarticons */
    case ID_ICONEXIT:
        if(usMenuCommand!=ID_ICONMOVE)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
                                        /* Post exit to PC/2 */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_EXIT), MPFROMSHORT(CMDSRC_MENU));
        break;

    case ID_ICONMOVE:
        if(usMenuCommand!=ID_ICONMOVE)
            {                           /* Remove frame attribute from previous smarticon to move smarticon */
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
        break;

    case ID_ICONZORDERTOP:
        if(usMenuCommand!=ID_ICONZORDERTOP)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONZORDERTOP, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONZORDERTOP;
        break;

    case ID_ICONZORDERBOTTOM:
        if(usMenuCommand!=ID_ICONZORDERBOTTOM)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONZORDERBOTTOM, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONZORDERBOTTOM;
        break;

    case ID_ICONCLOSE:
        if(usMenuCommand!=ID_ICONCLOSE)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONCLOSE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONCLOSE;
        break;

    case ID_ICONMAXIMIZE:
        if(usMenuCommand!=ID_ICONMAXIMIZE)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMAXIMIZE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMAXIMIZE;
        break;

    case ID_ICONMINIMIZE:
        if(usMenuCommand!=ID_ICONMINIMIZE)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMINIMIZE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMINIMIZE;
        break;

    case ID_ICONRESTORE:
        if(usMenuCommand!=ID_ICONRESTORE)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONRESTORE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONRESTORE;
        break;

    case ID_ICONSHUTDOWN:
        if(usMenuCommand!=ID_ICONMOVE)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
                                        /* Post ShutDowm OS/2 to PC/2 */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_SHUTDOWN), MPFROMSHORT(CMDSRC_MENU));
        break;

    case ID_ICONHELP:
        if(usMenuCommand!=ID_ICONMOVE)
            {
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(usMenuCommand, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, 0));
            WinSendMsg(hwndMenu, MM_SETITEMATTR,
                MPFROM2SHORT(ID_ICONMOVE, TRUE), MPFROM2SHORT(MIA_FRAMED|MIA_HILITED, MIA_FRAMED|MIA_HILITED));
            }
        usMenuCommand=ID_ICONMOVE;
                                        /* Post help to PC/2 */
        WinPostMsg(hwnd, WM_COMMAND, MPFROMSHORT(ID_HELP), MPFROMSHORT(CMDSRC_MENU));
        break;

    case ID_HELP:                       /* Display general help panel */
        if(hwndHelp!=NULLHANDLE) WinSendMsg(
            hwndHelp,                   /* Help window */
            HM_DISPLAY_HELP,            /* Display a help panel */
            MPFROMSHORT(ID_HELP),       /* Panel ID in ressource file */
            HM_RESOURCEID);             /* MP1 points to the help window identity */
        break;

    case ID_CONFIGDIALOG:               /* Popup menuitem Configure Menu selected */
        if(!WinDlgBox(                  /* Start Configure PC/2 dialog box */
            HWND_DESKTOP,               /* DESKTOP is parent */
            HWND_DESKTOP,               /* DESKTOP is owner */
            CD_DialogProcedure,         /* Dialog procedure of Program Installation
                                           dialog */
            0,                          /* Ressource is .EXE file */
            CDID_CONFIGDIALOG,          /* ID of Configure PC/2 dialog */
            0))                         /* No initialization data */
        GEN_ERR(hab, hwndFrame, hwndClient);
        break;

    case ID_DESKTOPDIALOG:              /* Popup menuitem Configure Desktop selected */
        if(!WinDlgBox(HWND_DESKTOP, HWND_DESKTOP, DD_DialogProcedure,
            0, DDID_DESKTOPDIALOG, 0))
        GEN_ERR(hab, hwndFrame, hwndClient);
        break;

    case ID_SHUTDOWN:                   /* ShutDown OS/2 menuitem selected */
        if(WinMessageBox(               /* Ask the user if he really wants to shut down OS/2 */
        HWND_DESKTOP, HWND_DESKTOP,
        "Are you really sure you want to ShutDown OS/2?",
        "PC/2 - Program Commander/2",
        ID_PC2MAINWINDOW,
        MB_OKCANCEL | MB_ICONQUESTION | MB_DEFBUTTON1)!=MBID_OK)
        return((MRESULT)TRUE);          /* Only shut down if OK is pressed */
        if(!WinDlgBox(                  /* Start ShutDown OS/2 dialog box */
            HWND_DESKTOP, HWND_DESKTOP, SD_DialogProcedure, 0,
            SDID_SHUTDOWNDIALOG, 0))
        GEN_ERR(hab, hwndFrame, hwndClient);
        break;

    case ID_EXIT:                       /* User selected F3 to shutdown PC/2 */
        WinPostMsg(hwnd, WM_CLOSE, 0, 0);
        break;

    case ID_ABOUTDIALOG:                /* User selected About PC/2 dialog */
        if(!WinDlgBox(                  /* Start About PC/2 dialog box */
            HWND_DESKTOP,               /* DESKTOP is parent */
            HWND_DESKTOP,               /* DESKTOP is owner */
            AD_DialogProcedure,         /* Dialog procedure of Program Installation
                                           dialog */
            0,                          /* Ressource is .EXE file */
            ADID_ABOUTDIALOG,           /* ID of Program Installation dialog */
            0))                         /* No initialization data */
            GEN_ERR(hab, hwndFrame, hwndClient);
        break;
    }
    break;
    }

default:                                /* Default window procedure must be called */
    return((MRESULT)WinDefWindowProc(hwnd, msg, mp1, mp2));
}
return((MRESULT)FALSE);                 /* We have handled the message */
}

/*--------------------------------------------------------------------------------------*\
 * This dialog procedure handles the PC/2 - Configuration (Setup) dialog.               *
 * Req: none                                                                            *
\*--------------------------------------------------------------------------------------*/
MRESULT  EXPENTRY CD_DialogProcedure(HWND hwndDlg, ULONG msg, MPARAM mp1, MPARAM mp2)
{
switch(msg)
{
case WM_INITDLG:
    {
    SWP         swp;

    WinQueryWindowPos(                  /* Query position of dialog window */
        hwndDlg,                        /* Handle of dialog window */
        &swp);                          /* Fill with position */
    WinSetWindowPos(                    /* Set dialog window position */
        hwndDlg,                        /* Handle of dialog window */
        HWND_TOP,                       /* Position on top and center of DESKTOP */
        (swpScreen.cx-swp.cx)/2,
        (swpScreen.cy-swp.cy)/2,
        0,
        0,
        SWP_MOVE);
                                        /* Initialize the listbox */
    WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pMenuData), NULL);
    break;
    }

/*                                                                                      *\
 * Syntax: WM_LOADPOPUPMENU, *MENUDATA, NULL                                            *
\*                                                                                      */
case WM_LOADPOPUPMENU:                  /* Load the current level of the Popup-Menu in
                                           the listbox after removing the old items */
    {
    MENUDATA    *pMD;

    pMD=PVOIDFROMMP(mp1);               /* Get the pointer to the first MENUDATA of the
                                           current level */
    WinSendDlgItemMsg(                  /* Send message to listbox */
        hwndDlg,                        /* Handle of dialog window */
        CDLB_MENUPROGRAM,               /* Submenu & Program listbox */
        LM_DELETEALL,                   /* Delete all list box items */
        (MPARAM)NULL,
        (MPARAM)NULL);
    if(pMD==NULL) break;                /* If linked list is empty break out */
    do
    {
        if(pMD->Item==ENTRYSUBMENU)     /* It is a Submenu */
            {
            UCHAR       Buffer[MAXNAMEL+4];
                                        /* Add >> for a Submenu */
            sprintf(Buffer, "%s >>", pMD->PgmTitle);
            WinSendDlgItemMsg(
                hwndDlg,
                CDLB_MENUPROGRAM,
                LM_INSERTITEM,          /* Insert Submenu Title at the end */
                MPFROMSHORT(LIT_END),
                MPFROMP(Buffer));
            }
        if(pMD->Item==ENTRYMENUITEM)    /* It's a Menuitem */
            WinSendDlgItemMsg(
                hwndDlg,
                CDLB_MENUPROGRAM,
                LM_INSERTITEM,          /* Insert Menuitem Title at the end */
                MPFROMSHORT(LIT_END),
                MPFROMP(pMD->PgmTitle));
        if(pMD->Item==ENTRYCONTROL)     /* It's a Control */
            {
            WinSendDlgItemMsg(
                hwndDlg,
                CDLB_MENUPROGRAM,
                LM_INSERTITEM,          /* Insert Control Title at the end */
                MPFROMSHORT(LIT_END),
                MPFROMP(pMD->PgmTitle));
            }
                                        /* It may also be an empty entry, but then we
                                           ignore it, because it must be filled with
                                           Menuitem or Submenu data first */
        if(pMD->Next!=NULL)             /* Get through linked list without diving into
                                           Submenus */
                pMD=pMD->Next;
        else break;                     /* We're at the end of the linked list */
    }while(TRUE);
    break;
    }

/*                                                                                      *\
 * Syntax: WM_SAVEPOPUPMENU, NULL, NULL                                                 *
\*                                                                                      */
case WM_SAVEPOPUPMENU:                  /* Save the Popup-Menu to the configuraion file */
    if((Pc2Profile=fopen(pucFilenameProfile, "w"))==NULL)
        USR_ERR("Cannot open confguration file - changes won't be saved",
                hwndFrame, hwndClient);
    else
        {
        fprintf(Pc2Profile, "PROFILE START\n");
        SaveMenu(pPopupMenu);           /* Save the menu linked list */
        fprintf(Pc2Profile, "PROFILE END\n");
        fclose(Pc2Profile);
        }
    break;

case WM_HELP:                           /* Help pressed */
    WinSendMsg(
        hwndHelp,                       /* Help window */
        HM_DISPLAY_HELP,                /* Display a help panel */
        MPFROMSHORT(ID_CONFIGDIALOG),   /* Panel ID in ressource file */
        HM_RESOURCEID);                 /* MP1 points to the help window identity */
    break;

case WM_COMMAND:                        /* Button pressed */
    switch(SHORT1FROMMP(mp1))
    {
/*                                                                                      *\
 * Chain up the linked list until we find the node, where this part-list comes from or  *
 * the beginning of the complete list. The pointer pMenuData is adjusted.               *
\*                                                                                      */
    case CDID_LEVELUP:                  /* Get up one level in the linked list */
        {
        MENUDATA        *pMD;           /* Pointer to a MENUDATA structure to find the
                                           Submenu where this part-list starts */

        pMD=pMenuData;                  /* Point to the first element of the linked list
                                           at the current level */
        if(pMD->Back==NULL)             /* If we're at the beginning of the complete linked
                                           list ignore button */
            return((MRESULT)FALSE);
        else pMD=pMD->Back;             /* Submenu which started current level */
                                        /* Now chain back through the linked list and find
                                           the element, where the pointer to a Submenu
                                           equals the back pointer of the first element
                                           in this Submenu. Then we've found the node */
        while(TRUE)
            {
            if(pMD->Back==NULL)         /* If we're now at the beginning break */
                break;
            if((pMD->Back)->Submenu==pMD)
                break;
            else pMD=pMD->Back;
            }
        pMenuData=pMD;                  /* Load as the top element of the current item */
                                        /* Now redraw items in listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pMenuData), NULL);
        return((MRESULT)FALSE);         /* We handled this button */
        }

/*                                                                                      *\
 * Test the user selection for being a Submenu. If one found chain into this submenu    *
 * and adjust the pointer pMenuData.                                                    *
\*                                                                                      */
    case CDID_LEVELDOWN:                /* Get down one level in the linked list */
        {
        MENUDATA        *pMD;           /* Pointer to a MENUDATA structure to find the
                                           Submenu to chain into */
        SHORT           sCount;

        pMD=pMenuData;                  /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
                                        /* If no item selected, ignore this button */
        if(sCount==LIT_NONE)
            return((MRESULT)FALSE);
        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
        if(pMD->Item!=ENTRYSUBMENU)     /* It's not a Submenu that's selected, ignore */
            return((MRESULT)FALSE);
        pMenuData=pMD->Submenu;         /* Otherwise chain into this part-list */
                                        /* Now redraw items in listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pMenuData), NULL);
        return((MRESULT)FALSE);         /* We handled this button */
        }

/*                                                                                      *\
 * The user selected to add a (Sub)Menu. Thus dismiss the PC/2 Configuration dialog and *
 * load the (Sub)Menu Installation dialog. The new (Sub)Menu is entered in a            *
 * STARTSESSION structure named StartSession. Save the changes and reload the PC/2      *
 * Configuration dialog again.                                                          *
\*                                                                                      */
    case CDID_ADDMENU:                  /* Add a Menu to PC/2 Configuration selected */
/*                                                                                      *\
 * The user selected to add a Program. Thus dismiss the PC/2 Configuration dialog and   *
 * load the Menucontrol Addition dialog. The new control style is selected via          *
 * autoradiobuttons and is fille STARTSESSION structure named StartSession. Save the    *
 * changes and reload the PC/2 Configuration dialog again.                              *
\*                                                                                      */
    case CDID_ADDPROGRAM:               /* Add a Program to PC/2 Configuration selected */
/*                                                                                      *\
 * The user selected to add a Control Style. Thus dismiss the PC/2 Configuration dialog *
 * and load the Program Installation dialog. The new session data is entered in a       *
 * STARTSESSION structure named StartSession. Save the changes and reload the PC/2      *
 * Configuration dialog again.                                                          *
\*                                                                                      */
    case CDID_ADDCONTROL:               /* Add a Control Entry to PC/2 Configuration selected */
        {
        UCHAR           *pU;            /* Temporary character pointer */
        MENUDATA        *pMD;           /* Pointer to a MENUDATA structure to insert a
                                           new MENUDATA stucture after */
        MENUDATA        *pMDNew;        /* Temporary pointer for the new item to be inserted
                                           after pMD */
        SHORT           sCount;

        pMD=pMenuData;                  /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
                                        /* If no item selected, and there exists one,
                                           add the new Menuitem after the last available
                                           Menuitem by querying the number from the listbox.
                                           Subtract 0 because we use 0-based instead 1-based. */
        if((sCount==LIT_NONE) && (pMenuData->Item!=ENTRYEMPTY))
            sCount=(SHORT)WinSendDlgItemMsg(hwndDlg, CDLB_MENUPROGRAM, LM_QUERYITEMCOUNT,
                MPFROM2SHORT(NULL, NULL), (MPARAM)NULL)-1;

        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
                                        /* Allocate a new item */
        pMDNew=AllocateMenuData();
        if(SHORT1FROMMP(mp1)!=CDID_ADDCONTROL)
            {                           /* Don't modify MenuData structure if a control entry
                                           is added */
            strcpy(pU=malloc(strlen("Insert here please")+1), "Insert here please");
            free(pMDNew->PgmTitle);
            pMDNew->PgmTitle=pU;
            pMDNew->id=MenuDataId++;    /* Increment ID only for non control entries, because
                                           control entries get their own unique ID */
            }
        LoadMenuData2SessionData(pMDNew, &SessionData);
        WinDismissDlg(hwndDlg, TRUE);   /* Clear up Configuration dialog */
        if(SHORT1FROMMP(mp1)==CDID_ADDMENU)
            {
            if(!WinDlgBox(              /* Start Addmenu PC/2 dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                HWND_DESKTOP,           /* DESKTOP is owner */
                MI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                MIID_MENUDIALOG,        /* ID of Addmenu PC/2 dialog */
                0))                     /* No initialization data */
                GEN_ERR(hab, hwndFrame, hwndClient);
            }
        if(SHORT1FROMMP(mp1)==CDID_ADDPROGRAM)
            {
            if(!WinDlgBox(              /* Start Program Installation dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                HWND_DESKTOP,           /* DESKTOP is owner */
                PI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                PIID_PROGRAMDIALOG,     /* ID of Addprogram PC/2 dialog */
                0))                     /* No initialization data */
                GEN_ERR(hab, hwndFrame, hwndClient);
            }
        if(SHORT1FROMMP(mp1)==CDID_ADDCONTROL)
            {
            if(!WinDlgBox(              /* Start Menucontrol Addition dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                HWND_DESKTOP,           /* DESKTOP is owner */
                MD_DialogProcedure,     /* Dialog procedure of Menucontrol Addition
                                           dialog */
                0,                      /* Ressource is .EXE file */
                MDID_CONTROLDIALOG,     /* ID of Addmenucontrol PC/2 dialog */
                0))                     /* No initialization data */
                GEN_ERR(hab, hwndFrame, hwndClient);
            }
        if(DialogResult==DID_OK)        /* If manipulation is done successfully, then load
                                           the SESSIONDATA structure back to the MENUDATA
                                           structure and save the changes */
            {
            LoadSessionData2MenuData(pMDNew, &SessionData);
            if(pMD->Item!=ENTRYEMPTY)   /* Add new entry, if the current entry isn't empty */
                {
                if(SHORT1FROMMP(mp1)==CDID_ADDMENU)
                    {                   /* It it is a Submenu, we also must add an empty
                                           first item for it */
                    MENUDATA    *pMDTemp;

                    pMDTemp=AllocateMenuData();
                    pMDNew->Submenu=pMDTemp;
                    pMDTemp->Back=pMDNew;
                    pMDNew->Item=ENTRYSUBMENU;
                    }
                if(SHORT1FROMMP(mp1)==CDID_ADDPROGRAM) pMDNew->Item=ENTRYMENUITEM;
                if(SHORT1FROMMP(mp1)==CDID_ADDCONTROL)
                    {                   /* For controls also add the ID of the control. These
                                           controls have an predefined ID because the action
                                           they should start is allways the same */
                    pMDNew->Item=ENTRYCONTROL;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGMENU))
                        {               /* If the Configuration Dialog is found set flag
                                           to true */
                        bConfigDialog=TRUE;
                        pMDNew->id=ID_CONFIGDIALOG;
                        }
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGDESKTOP)) pMDNew->id=ID_DESKTOPDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_ABOUT)) pMDNew->id=ID_ABOUTDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SHUTDOWN)) pMDNew->id=ID_SHUTDOWN;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_HELP)) pMDNew->id=ID_HELP;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_EXIT)) pMDNew->id=ID_EXIT;
                                        /* We assume that more than 1 separator may occur
                                           so to be able add, modify or delete on menuitems
                                           define a unique one */
                    if(!strcmp(pMDNew->PgmTitle, CTRL_BREAKSEPARATOR)) pMDNew->id=MenuDataId++;;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SEPARATOR)) pMDNew->id=MenuDataId++;;
                    }
                if(pMD->Next!=NULL) (pMD->Next)->Back=pMDNew;
                pMDNew->Next=pMD->Next;
                pMDNew->Back=pMD;
                pMD->Next=pMDNew;
                                        /* Insert item after the existing item */
                SetPopupMenu(MM_INSERTITEMMENUITEM, MPFROMP(pMDNew), MPFROMLONG(pMD->id));
                }
            else                        /* If it is an empty entry fill it with user data */
                {
                UCHAR   *pU;            /* Temporary character pointer */

                pMD->id=pMDNew->id;
                if(SHORT1FROMMP(mp1)==CDID_ADDMENU)
                    {                   /* It it is a Submenu, we also must add an empty
                                           first item for it */
                    MENUDATA    *pMDTemp;

                    pMDTemp=AllocateMenuData();
                    pMD->Submenu=pMDTemp;
                    pMDTemp->Back=pMD;
                    pMD->Item=ENTRYSUBMENU;
                    }
                if(SHORT1FROMMP(mp1)==CDID_ADDPROGRAM) pMD->Item=ENTRYMENUITEM;
                if(SHORT1FROMMP(mp1)==CDID_ADDCONTROL)
                    {                   /* For controls also add the ID of the control. These
                                           controls have an predefined ID because the action
                                           they should start is allways the same */
                    pMD->Item=ENTRYCONTROL;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGMENU))
                        {               /* If the Configuration Dialog is found set flag
                                           to true */
                        bConfigDialog=TRUE;
                        pMD->id=ID_CONFIGDIALOG;
                        }
                    if(!strcmp(pMDNew->PgmTitle, CTRL_CONFIGDESKTOP)) pMD->id=ID_DESKTOPDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_ABOUT)) pMD->id=ID_ABOUTDIALOG;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SHUTDOWN)) pMD->id=ID_SHUTDOWN;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_HELP)) pMD->id=ID_HELP;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_EXIT)) pMD->id=ID_EXIT;
                                        /* We assume that more than 1 separator may occur
                                           so to be able add, modify or delete on menuitems
                                           define a unique one */
                    if(!strcmp(pMDNew->PgmTitle, CTRL_BREAKSEPARATOR)) pMD->id=MenuDataId++;;
                    if(!strcmp(pMDNew->PgmTitle, CTRL_SEPARATOR)) pMD->id=MenuDataId++;;
                    }
                strcpy(pU=malloc(strlen(pMDNew->PgmTitle)+1), pMDNew->PgmTitle);
                free(pMD->PgmTitle);
                pMD->PgmTitle=pU;
                strcpy(pU=malloc(strlen(pMDNew->WindowTitle)+1), pMDNew->WindowTitle);
                free(pMD->WindowTitle);
                pMD->WindowTitle=pU;
                strcpy(pU=malloc(strlen(pMDNew->PgmName)+1), pMDNew->PgmName);
                free(pMD->PgmName);
                pMD->PgmName=pU;
                strcpy(pU=malloc(strlen(pMDNew->PgmDirectory)+1), pMDNew->PgmDirectory);
                free(pMD->PgmDirectory);
                pMD->PgmDirectory=pU;
                strcpy(pU=malloc(strlen(pMDNew->PgmInputs)+1), pMDNew->PgmInputs);
                free(pMD->PgmInputs);
                pMD->PgmInputs=pU;
                pMD->SessionType=pMDNew->SessionType;
                pMD->PgmControl=pMDNew->PgmControl;
                pMD->FgBg=pMDNew->FgBg;
                pMD->InitXPos=pMDNew->InitXPos;
                pMD->InitYPos=pMDNew->InitYPos;
                pMD->InitXSize=pMDNew->InitXSize;
                pMD->InitYSize=pMDNew->InitYSize;
                (pMD->KeyData).usFlags=(pMDNew->KeyData).usFlags;
                (pMD->KeyData).usCh=(pMDNew->KeyData).usCh;
                pMD->PriorityClass=pMDNew->PriorityClass;
                pMD->SwpFlag=pMDNew->SwpFlag;
                pMD->PriorityDelta=pMDNew->PriorityDelta;
                if(pMD->Back!=NULL)     /* This is the first item of a Submenu, then
                                           insert it there */
                    SetPopupMenu(MM_INSERTITEMSUBMENU, MPFROMP(pMD), MPFROMLONG((pMD->Back)->id));
                else                    /* This is the complete first item of the linked
                                           list, so insert at the end */
                    SetPopupMenu(MM_INSERTITEMMENUITEM, MPFROMP(pMD), MPFROMLONG(MIT_END));
                free(pMDNew->PgmTitle); /* Free temporary used structure */
                free(pMDNew->WindowTitle);
                free(pMDNew->PgmName);
                free(pMDNew->PgmDirectory);
                free(pMDNew->PgmInputs);
                free(pMDNew);
                }
            }
        else
            {
            free(pMDNew->PgmTitle);     /* Free temporary MENUDATA structure */
            free(pMDNew->WindowTitle);
            free(pMDNew->PgmName);
            free(pMDNew->PgmDirectory);
            free(pMDNew->PgmInputs);
            free(pMDNew);
            }
                                        /* Initialize the listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pMenuData), NULL);
        if(!WinDlgBox(                  /* Now reload the Configuration dialog */
            HWND_DESKTOP,
            HWND_DESKTOP,
            CD_DialogProcedure,
            0,
            CDID_CONFIGDIALOG,
            0)) GEN_ERR(hab, hwndFrame, hwndClient);
        break; 
        }

/*                                                                                      *\
 * The user selected to change an item. Thus dismiss the PC/2 Configuration dialog and  *
 * load the Menu or Program Installation dialog. The new session data is entered in a   *
 * STARTSESSION structure named StartSession.                                           *
 * Then reload the PC/2 Configuration dialog again.                                     *
\*                                                                                      */
    case CDID_CHANGEENTRY:              /* Change a Menu or Program configuration selected */
        {
        MENUDATA        *pMD;
        SHORT           sCount;

        pMD=pMenuData;                  /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
        if(sCount==LIT_NONE)            /* If no item selected ignore this button */
            return((MRESULT)FALSE);
        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
                                        /* Controls can't be changed, they must be
                                           removed and newly inserted */
        if(pMD->Item==ENTRYCONTROL) return((MRESULT)FALSE);
                                        /* Now load the MENUDATA to SESSIONDATA structure
                                           where the manipulations will take effect */
        LoadMenuData2SessionData(pMD, &SessionData);
        WinDismissDlg(hwndDlg, TRUE);   /* Clear up Configuration dialog */
        if(pMD->Item==ENTRYMENUITEM)
                                        /* It's a Menuitem  so call the Program
                                           Installation dialog box */
            if(!WinDlgBox(              /* Start Program Installation dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                HWND_DESKTOP,           /* DESKTOP is owner */
                PI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                PIID_PROGRAMDIALOG,     /* ID of Program Installation PC/2 dialog */
                0))                     /* No initialization data */
                GEN_ERR(hab, hwndFrame, hwndClient);
        if(pMD->Item==ENTRYSUBMENU)
                                        /* It's a Submenu so call the Menu Installation
                                           dialog box */
            if(!WinDlgBox(              /* Start Addmenu PC/2 dialog box */
                HWND_DESKTOP,           /* DESKTOP is parent */
                HWND_DESKTOP,           /* DESKTOP is owner */
                MI_DialogProcedure,     /* Dialog procedure of Program Installation
                                           dialog */
                0,                      /* Ressource is .EXE file */
                MIID_MENUDIALOG,        /* ID of Addmenu PC/2 dialog */
                0))                     /* No initialization data */
                GEN_ERR(hab, hwndFrame, hwndClient);
        if(DialogResult==DID_OK)        /* If manipulation is done successfully, then load
                                           the SESSIONDATA structure back to the MENUDATA
                                           structure and save the changes */
            {
            LoadSessionData2MenuData(pMD, &SessionData);
                                        /* Now change the menuitem text to the new one */
            SetPopupMenu(MM_SETITEMTEXT, MPFROMP(pMD), MPFROMLONG(pMD->id));
                                        /* Initialize the listbox */
            WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pMenuData), NULL);
            }
        if(!WinDlgBox(                  /* Now reload the Configuration dialog */
            HWND_DESKTOP,
            HWND_DESKTOP,
            CD_DialogProcedure,
            0,
            CDID_CONFIGDIALOG,
            0)) GEN_ERR(hab, hwndFrame, hwndClient);
        break;
        }

/*                                                                                      *\
 * The user selected to remove an item. If thist item is the only one in the linked     *
 * list or the first item of a submenu set it to empty otherwise free it's ressources   *
 * and remove the entry.                                                                *
\*                                                                                      */
    case CDID_REMOVEENTRY:              /* Remove a item of the PC/2 Configuration selected */
        {
        UCHAR           *pU;
        MENUDATA        *pMD;
        SHORT           sCount;

        pMD=pMenuData;                  /* Point to the first element of the linked list
                                           at the current level */
                                        /* Send message to listbox */
        sCount=(SHORT)WinSendDlgItemMsg(
            hwndDlg,                    /* Handle of dialog window */
            CDLB_MENUPROGRAM,           /* Submenu & Program listbox */
            LM_QUERYSELECTION,          /* Query first selected list box item */
            MPFROMSHORT(LIT_FIRST),
            (MPARAM)NULL);
        if(sCount==LIT_NONE)            /* If no item selected ignore this button */
            return((MRESULT)FALSE);
        for( ;sCount>0; sCount--)       /* Walk through the linked list to the selected
                                           item */
            pMD=pMD->Next;
        while(TRUE)
            {
            if((pMD->Back==NULL) && (pMD->Next!=NULL))
                {                       /* Remove the first item of the complete linked list */
                if(pMD->Item==ENTRYSUBMENU)
                if((pMD->Submenu)->Item==ENTRYEMPTY)
                    {                   /* If it is an empty Submenu remove it completely */
                                        /* Remove the Submenu and the empty first item
                                           from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMLONG(pMD->id));
                    free((pMD->Submenu)->PgmTitle);
                    free((pMD->Submenu)->WindowTitle);
                    free((pMD->Submenu)->PgmName);
                    free((pMD->Submenu)->PgmDirectory);
                    free((pMD->Submenu)->PgmInputs);
                    free(pMD->Submenu);
                    (pMD->Next)->Back=NULL;
                                        /* Now next element is the first one */
                    pPopupMenu=pMD->Next;
                    pMenuData=pMD->Next;
                    free(pMD->PgmTitle);
                    free(pMD->WindowTitle);
                    free(pMD->PgmName);
                    free(pMD->PgmDirectory);
                    free(pMD->PgmInputs);
                    free(pMD);
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                if((pMD->Item==ENTRYMENUITEM) || (pMD->Item==ENTRYCONTROL))
                    {                   /* If it is an empty Menuitem or Control remove it completly */
                                        /* If the Configuration Dialog is found set flag
                                           to false */
                    if(!strcmp(pMD->PgmTitle, CTRL_CONFIGDESKTOP)) bConfigDialog=FALSE;
                    (pMD->Next)->Back=NULL;
                                        /* Now next element is the first one */
                    pPopupMenu=pMD->Next;
                    pMenuData=pMD->Next;
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMLONG(pMD->id));
                    free(pMD->PgmTitle);
                    free(pMD->WindowTitle);
                    free(pMD->PgmName);
                    free(pMD->PgmDirectory);
                    free(pMD->PgmInputs);
                    free(pMD);
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                }
            if((pMD->Back==NULL) && (pMD->Next==NULL))
                {                       /* If it is the one and only item of the linked list
                                           set it to empty */
                if(pMD->Item==ENTRYSUBMENU)
                if((pMD->Submenu)->Item==ENTRYEMPTY)
                    {                   /* If it is an empty Submenu remove the empty
                                           item completely */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMLONG(pMD->id));
                    free((pMD->Submenu)->PgmTitle);
                    free((pMD->Submenu)->WindowTitle);
                    free((pMD->Submenu)->PgmName);
                    free((pMD->Submenu)->PgmDirectory);
                    free((pMD->Submenu)->PgmInputs);
                    free(pMD->Submenu);
                    free(pMD->PgmTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmTitle=pU;
                    free(pMD->WindowTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->WindowTitle=pU;
                    free(pMD->PgmName);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmName=pU;
                    free(pMD->PgmDirectory);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDirectory=pU;
                    free(pMD->PgmInputs);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmInputs=pU;
                    pMD->Item=ENTRYEMPTY;
                    pMD->Back=NULL;
                    pMD->Submenu=NULL;
                    pMD->Next=NULL;
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                if((pMD->Item==ENTRYMENUITEM) || (pMD->Item==ENTRYCONTROL))
                    {                   /* If it is a Menuitem or Control set it to empty */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMLONG(pMD->id));
                    free(pMD->PgmTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmTitle=pU;
                    free(pMD->WindowTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->WindowTitle=pU;
                    free(pMD->PgmName);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmName=pU;
                    free(pMD->PgmDirectory);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDirectory=pU;
                    free(pMD->PgmInputs);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmInputs=pU;
                    pMD->Item=ENTRYEMPTY;
                    pMD->Back=NULL;
                    pMD->Submenu=NULL;
                    pMD->Next=NULL;
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                }
            if(pMD->Back!=NULL)
                {                       /* It is any item of more than one item and not
                                           the first one */
                if(((pMD->Back)->Submenu==pMD) && (pMD->Submenu==NULL) && (pMD->Next==NULL))
                {                       /* If it is the first item of a Submenu not followed
                                           by any item, set it to empty */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMLONG(pMD->id));
                    free(pMD->PgmTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmTitle=pU;
                    free(pMD->WindowTitle);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->WindowTitle=pU;
                    free(pMD->PgmName);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmName=pU;
                    free(pMD->PgmDirectory);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->PgmDirectory=pU;
                    free(pMD->PgmInputs);
                    strcpy(pU=malloc(strlen("")+1), "");
                    pMD->Item=ENTRYEMPTY;
                    break;              /* Ensure we only test once, because each routine
                                           changes the pointers */
                    }
                if(pMD->Item==ENTRYSUBMENU)
                if((pMD->Submenu)->Item==ENTRYEMPTY)
                    {                   /* If it is an empty Submenu so also remove the
                                           first item in the Submenu */
                                        /* Remove the Submenu and the empty first item
                                           from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMLONG(pMD->id));
                    free((pMD->Submenu)->PgmTitle);
                    free((pMD->Submenu)->WindowTitle);
                    free((pMD->Submenu)->PgmName);
                    free((pMD->Submenu)->PgmDirectory);
                    free((pMD->Submenu)->PgmInputs);
                    free(pMD->Submenu);
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next==NULL))
                        {               /* If the previous item is a Submenu, this item is
                                           the first item of it. If none item follows, set
                                           this item to empty */
                        free(pMD->PgmTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmTitle=pU;
                        free(pMD->WindowTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->WindowTitle=pU;
                        free(pMD->PgmName);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmName=pU;
                        free(pMD->PgmDirectory);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmDirectory=pU;
                        free(pMD->PgmInputs);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->Item=ENTRYEMPTY;
                        pMD->Submenu=NULL;
                        pMD->Next=NULL;
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next!=NULL))
                        {               /* If the previous item is a Submenu, this item ist
                                           the first item of it. If one item follows adjust
                                           the pointer to the current level of items */
                        pMenuData=pMD->Next;
                        (pMD->Back)->Submenu=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if((pMD->Back)->Submenu!=pMD)
                        {               /* If this item isn't the first item of a Submenu */
                        (pMD->Back)->Next=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    }
                if((pMD->Item==ENTRYMENUITEM) || pMD->Item==ENTRYCONTROL)
                    {                   /* If it is a Menuitem or Control just remove it completly */
                                        /* Remove the item from the Popup-Menu */
                    SetPopupMenu(MM_DELETEITEM, MPFROMP(pMD), MPFROMLONG(pMD->id));
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next==NULL))
                        {               /* If the previous item is a Submenu, this item is
                                           the first item of it. If none item follows, set
                                           this item to empty */
                        free(pMD->PgmTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmTitle=pU;
                        free(pMD->WindowTitle);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->WindowTitle=pU;
                        free(pMD->PgmName);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmName=pU;
                        free(pMD->PgmDirectory);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->PgmDirectory=pU;
                        free(pMD->PgmInputs);
                        strcpy(pU=malloc(strlen("")+1), "");
                        pMD->Item=ENTRYEMPTY;
                        pMD->Submenu=NULL;
                        pMD->Next=NULL;
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if(((pMD->Back)->Submenu==pMD) && (pMD->Next!=NULL))
                        {               /* If the previous item is a Submenu, this item ist
                                           the first item of it. If one item follows adjust
                                           the pointer to the current level of items */
                        pMenuData=pMD->Next;
                        (pMD->Back)->Submenu=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    if((pMD->Back)->Submenu!=pMD)
                        {               /* If this item isn't the first item of a Submenu */
                        (pMD->Back)->Next=pMD->Next;
                        if(pMD->Next!=NULL) (pMD->Next)->Back=pMD->Back;
                        free(pMD->PgmTitle);
                        free(pMD->WindowTitle);
                        free(pMD->PgmName);
                        free(pMD->PgmDirectory);
                        free(pMD->PgmInputs);
                        free(pMD);
                        break;          /* Ensure we only test once, because each routine
                                           changes the pointers */
                        }
                    }
                }
            break;                      /* If we come here, we're trying to remove an not
                                           empty Submenu, but we also must exit the
                                           endless loop */
            }
                                        /* Initialize the listbox */
        WinSendMsg(hwndDlg, WM_LOADPOPUPMENU, MPFROMP(pMenuData), NULL);
        return((MRESULT)FALSE);         /* We have done everything */
        }

/*                                                                                      *\
 * The user selected to resort the current level of the menuentries. Load the dialog    *
 * and let the user resort the linked list of menues pointed to by pMenuData and to     *
 * resort the menuentries of the Popup-Menu.                                            *
\*                                                                                      */
    case CDID_RESORT:                   /* Load the resort dialog */
        {
        WinDismissDlg(hwndDlg, TRUE);   /* Clear up Configuration dialog */
        if(!WinDlgBox(                  /* Start Resort dialog box */
            HWND_DESKTOP,               /* DESKTOP is parent */
            HWND_DESKTOP,               /* DESKTOP is owner */
            RD_DialogProcedure,         /* Dialog procedure of Program Installation
                                           dialog */
            0,                          /* Ressource is .EXE file */
            RDID_RESORTDIALOG,          /* ID of Program Installation PC/2 dialog */
            0))                         /* No initialization data */
            GEN_ERR(hab, hwndFrame, hwndClient);
        if(!WinDlgBox(                  /* Now reload the Configuration dialog */
            HWND_DESKTOP,
            HWND_DESKTOP,
            CD_DialogProcedure,
            0,
            CDID_CONFIGDIALOG,
            0)) GEN_ERR(hab, hwndFrame, hwndClient);
        }
        break;

    case DID_OK:                        /* Enter key pressed */
                                        /* Save the changes */
        WinSendMsg(hwndDlg, WM_SAVEPOPUPMENU, NULL, NULL);
        DialogResult=DID_OK;            /* Dialog terminated with DID_OK */
        break;

    case DID_CANCEL:                    /* Escape or Cancel pressed */
        DialogResult=DID_CANCEL;        /* Dialog terminated with DID_CANCEL */
        break;

    default:
        return(WinDefDlgProc(hwndDlg, msg, mp1, mp2));
    }
    WinDismissDlg(hwndDlg, TRUE);       /* Clear up dialog */
    break;

default:                                /* Default window procedure must be called */
    return(WinDefDlgProc(hwndDlg, msg, mp1, mp2));
}
return((MRESULT)FALSE);                 /* We have handled the message */
}

