/*
 FILE: 		SourceBuilder.java
 
 Provides a GUI for building a sources.xml file and corresponding XSpecs for sources. 
 For more information see:  www.unityjdbc.com
 */

package unity.extractor;

import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.GridBagConstraints;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.ArrayList;

import javax.swing.BoxLayout;
import javax.swing.filechooser.FileFilter;


public class SourceBuilder extends JFrame {
	private static final long serialVersionUID = 1L;

	private static int FrameWidth = 700, FrameHeight = 600;
	private String startMessage = "Select a location for the sources.xml file to begin \n  OR accept the default location, enter your source information, then click the add button.";
	private String addFailedMessage = "ERROR encountered.  Add data source failed.";
	private String addSuccessMessage = "SUCCESS.  Data source successfully added.";	

	private JTextField txtSources, txtName, txtURL, txtDriver, txtXSpec;

	private JList lstSourceType;

	private JTextArea txtMessages;

	private JButton btnSourceBrowse, btnXSpecBrowse, btnExit, btnAdd;

	private static int fieldSize = 60;
	private String startDirectory = null;
	private Source[] sources;

	public SourceBuilder() {
		super();
		startDirectory = System.getProperty("user.dir");
		initSources();
		initialize();
	}

	private void initialize() 
	{			
		FlowLayout flowLayout4 = new FlowLayout();
		flowLayout4.setAlignment(java.awt.FlowLayout.LEFT);
		FlowLayout flowLayout3 = new FlowLayout();
		flowLayout3.setAlignment(java.awt.FlowLayout.LEFT);
		FlowLayout flowLayout2 = new FlowLayout();
		flowLayout2.setAlignment(java.awt.FlowLayout.LEFT);
		FlowLayout flowLayout1 = new FlowLayout();
		flowLayout1.setAlignment(java.awt.FlowLayout.LEFT);
		GridLayout gridLayout = new GridLayout(5, 1);
		gridLayout.setHgap(5);
		gridLayout.setVgap(5);
		Font labelFont = new java.awt.Font("DialogInput", java.awt.Font.BOLD, 12);
		Font listFont = new java.awt.Font("SansSerif", java.awt.Font.PLAIN, 12);
		Font textFont = new java.awt.Font("SansSerif", java.awt.Font.PLAIN, 11);
		
		GridBagConstraints gridBagConstraints21 = new GridBagConstraints(0, 0, 1, 1, 0.0, 0.0, GridBagConstraints.SOUTH, GridBagConstraints.HORIZONTAL, new Insets(0, 0, 2, 0), 185, 79);
		gridBagConstraints21.ipady = 0;
		gridBagConstraints21.gridwidth = 1;
		gridBagConstraints21.gridheight = 1;
		gridBagConstraints21.anchor = java.awt.GridBagConstraints.CENTER;		
		GridBagConstraints gridBagConstraints2 = new GridBagConstraints();
		gridBagConstraints2.insets = new java.awt.Insets(2,0,1,0);
		gridBagConstraints2.gridy = 2;
		gridBagConstraints2.ipadx = 650;
		gridBagConstraints2.ipady = 300;
		gridBagConstraints2.fill = java.awt.GridBagConstraints.BOTH;
		gridBagConstraints2.gridheight = 1;
		gridBagConstraints2.anchor = java.awt.GridBagConstraints.CENTER;
		gridBagConstraints2.gridx = 0;
		GridBagConstraints gridBagConstraints1 = new GridBagConstraints();
		gridBagConstraints1.insets = new java.awt.Insets(2,0,2,0);
		gridBagConstraints1.gridy = 1;
		gridBagConstraints1.ipadx = 10;
		gridBagConstraints1.ipady = 0;
		gridBagConstraints1.anchor = java.awt.GridBagConstraints.WEST;
		gridBagConstraints1.fill = java.awt.GridBagConstraints.HORIZONTAL;
		gridBagConstraints1.gridx = 0;
		GridBagConstraints gridBagConstraints = new GridBagConstraints();
		gridBagConstraints.insets = new java.awt.Insets(0,0,2,0);
		gridBagConstraints.gridy = 0;
		gridBagConstraints.ipadx = 185;
		gridBagConstraints.ipady = 79;
		gridBagConstraints.gridwidth = 1;
		gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
		gridBagConstraints.gridx = 0;
		FlowLayout flowLayout = new FlowLayout();
		flowLayout.setAlignment(java.awt.FlowLayout.LEFT);
		this.setSize(new java.awt.Dimension(FrameWidth, FrameHeight));
		this.setResizable(false);
		this.setTitle("Source Builder Utility");
		this.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		// Main panel for frame
		JPanel mainPanel = new JPanel();
		mainPanel.setLayout(new GridBagLayout());
		
		// Source file panel
		JPanel sourcePanel = new JPanel();	
		sourcePanel.setLayout(flowLayout);
		sourcePanel.setBorder(javax.swing.BorderFactory.createEtchedBorder(javax.swing.border.EtchedBorder.RAISED));
		
		txtSources = new JTextField(40);
		txtSources.setText(startDirectory+"\\sources.xml");
		sourcePanel.add(new JLabel("Sources file:"), null);
		btnSourceBrowse = new JButton("Browse");
		btnSourceBrowse.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent e) {
				File f = getFile(txtSources.getText());
				if (f != null)				
					txtSources.setText(f.getPath());				
			}
		});
		
		sourcePanel.add(txtSources);
		sourcePanel.add(btnSourceBrowse);
		btnExit = new JButton("Exit");
		btnExit.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent e) {
				System.exit(0); 
			}
		});
		sourcePanel.add(btnExit);

		// Add data source panel
		JPanel addPanel = new JPanel();
		addPanel.setBorder(javax.swing.BorderFactory.createEtchedBorder(javax.swing.border.EtchedBorder.RAISED)); 
		JPanel leftAddPanel = new JPanel();
		addPanel.setLayout(new BoxLayout(addPanel, BoxLayout.X_AXIS));
		
		leftAddPanel.setLayout(new BoxLayout(leftAddPanel, BoxLayout.PAGE_AXIS));
		lstSourceType = new JList(sources);
		lstSourceType.setFont(listFont);
		lstSourceType.setBorder(javax.swing.BorderFactory.createEtchedBorder(javax.swing.border.EtchedBorder.RAISED));
		lstSourceType.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JLabel lbl = new JLabel("Source Type");
		lbl.setAlignmentX(Component.LEFT_ALIGNMENT);
		lstSourceType.setAlignmentX(Component.LEFT_ALIGNMENT);
		lstSourceType.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
			public void valueChanged(javax.swing.event.ListSelectionEvent e) 
			{	// Change the fields to the defaults for that source type
				int idx = lstSourceType.getSelectedIndex();				
				txtURL.setText(sources[idx].defaultURL);
				txtDriver.setText(sources[idx].driverName);
				txtURL.repaint();
				txtDriver.repaint();
			}
			});
		leftAddPanel.add(lbl);
		leftAddPanel.add(lstSourceType);
		addPanel.add(leftAddPanel);
		
		JPanel rightAddPanel = new JPanel();
		JPanel tmp = new JPanel();	
		lbl = new JLabel("Name:  ");
		lbl.setFont(labelFont);
		tmp.setLayout(flowLayout1);
		tmp.add(lbl, null);
		rightAddPanel.setLayout(gridLayout);
		txtName = new JTextField(fieldSize);	
		txtName.setFont(textFont);
		tmp.add(txtName);		
		rightAddPanel.add(tmp);
		tmp = new JPanel();
		lbl = new JLabel("URL:   ");
		lbl.setFont(labelFont);
		tmp.setLayout(flowLayout2);
		tmp.add(lbl, null);
		txtURL = new JTextField(fieldSize);
		txtURL.setFont(textFont);
		tmp.add(txtURL);
		rightAddPanel.add(tmp);
		tmp = new JPanel();
		lbl = new JLabel("Driver:");
		lbl.setFont(labelFont);
		tmp.setLayout(flowLayout3);
		tmp.add(lbl, null);
		txtDriver = new JTextField(fieldSize);
		txtDriver.setFont(textFont);
		tmp.add(txtDriver);		
		rightAddPanel.add(tmp);
		tmp = new JPanel();
		lbl = new JLabel("XSpec: ");
		lbl.setFont(labelFont);
		tmp.setLayout(flowLayout4);
		tmp.add(lbl, null);
		txtXSpec = new JTextField(fieldSize-2);
		txtXSpec.setFont(textFont);
		txtXSpec.setText(startDirectory+"\\output.xml");
		tmp.add(txtXSpec);
		btnXSpecBrowse = new JButton("...");
		btnXSpecBrowse.setFont(textFont);
		btnXSpecBrowse.setPreferredSize(new Dimension(18,18));
		btnXSpecBrowse.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent e) {
				File f = getFile(txtXSpec.getText());
				if (f != null)				
					txtXSpec.setText(f.getPath());				
			}
		});
		tmp.add(btnXSpecBrowse);
		rightAddPanel.add(tmp);
		btnAdd = new JButton("Add Source");
		btnAdd.addActionListener(new java.awt.event.ActionListener() {
			public void actionPerformed(java.awt.event.ActionEvent e) {
				doAdd();				
			}
		});
		rightAddPanel.add(btnAdd);
		
		addPanel.add(rightAddPanel);

		// Messages panel
		JPanel messagePanel = new JPanel();
		messagePanel.setBorder(javax.swing.BorderFactory.createEmptyBorder(0,0,0,0));
		txtMessages = new JTextArea(20, 60);
		txtMessages.setText(startMessage);
		JScrollPane scrollPane = new JScrollPane(txtMessages);								
		scrollPane.setComponentOrientation(java.awt.ComponentOrientation.LEFT_TO_RIGHT);
		messagePanel.add(scrollPane, BorderLayout.CENTER);

		this.setContentPane(mainPanel);
		mainPanel.add(sourcePanel, gridBagConstraints21);
		mainPanel.add(addPanel, gridBagConstraints1);
		mainPanel.add(messagePanel, gridBagConstraints2);
			addPanel.setFont(new java.awt.Font("Dialog", java.awt.Font.ITALIC, 12));
			rightAddPanel.setFont(new java.awt.Font("Dialog", java.awt.Font.ITALIC, 12));
	}

	public static void main(String[] argv) {
		SourceBuilder sb = new SourceBuilder();
		sb.setVisible(true);
	}

	private class Source {
		public String driverName; // Name of JDBC driver for this source

		public String sourceName; // Name of source type (MySQL, etc.)

		public String defaultURL; // Syntax of default URL for source

		public Source(String dn, String sn, String URL) {
			driverName = dn;
			sourceName = sn;
			defaultURL = URL;
		}

		public String toString() {
			return sourceName;
		}
	}

	private void initSources() { // Informix, Sybase?, Microsoft Access?
		sources = new Source[7];
		sources[0] = new Source("oracle.jdbc.driver.OracleDriver", "Oracle", "jdbc:oracle:thin:<user>/<password>@<server>:1521/<service>");
		sources[1] = new Source("com.microsoft.jdbc.sqlserver.SQLServerDriver", "Microsoft SQL Server", "jdbc:microsoft:sqlserver://<server>;DatabaseName=<database>;User=<uid>;Password=<pw>");
		sources[2] = new Source("com.ibm.db2.jcc.DB2Driver", "IBM DB2", "jdbc:db2//<server>:<port>/<database>:user=<userId>;password=<password>;");
		sources[3] = new Source("com.mysql.jdbc.Driver", "MySQL", "jdbc:mysql://<server>/<database>?user=<userId>&password=<password>");
		sources[4] = new Source("org.postgresql.Driver", "Postgres", "jdbc:postgresql://<server>/<database>?user=<userId>&password=<password>");
		sources[5] = new Source("", "Generic - JDBC", "");
		sources[6] = new Source("sun.jdbc.odbc.JdbcOdbcDriver", "Generic - ODBC", "jdbc:odbc:<database>;UID=<userId>;PWD=<password>");
	}
	
	private File lastDirectory = null;
	
	private class XMLFileFilter extends FileFilter
	{
		public boolean accept(File f) {
			if(f != null) 
			{
			    if(f.isDirectory()) 
			    	return true;
			    
			    String extension = getExtension(f);
			    if(extension != null && extension.toLowerCase().equals("xml")) 
			    	return true;			    
			}
			return false;		    
		}

		public String getDescription() {
			return "XML Files";
		}
		
		public String getExtension(File f) 
		{
			if(f != null) {
			    String filename = f.getName();
			    int i = filename.lastIndexOf('.');
			    if(i>0 && i<filename.length()-1) 
			    	return filename.substring(i+1).toLowerCase();			    
			}
			return null;
		}
	}
	
	private File getFile(String orgFile)
	{
		JFileChooser chooser;
		if (orgFile != null && !orgFile.equals(""))
		{	// Set chooser to point to current file selected
			File tmp = new File(orgFile);
			chooser = new JFileChooser(tmp);
			chooser.setSelectedFile(tmp);
		}		
		else
		{
			chooser = new JFileChooser();
			if (lastDirectory != null)
				chooser.setCurrentDirectory(lastDirectory);						// Set directory to last one viewed
			else
				chooser.setCurrentDirectory(new File(startDirectory));			// Otherwise set to directory where code was run
		}
		
		chooser.setDialogTitle("Select Save File");		
		chooser.setFileFilter(new XMLFileFilter());
		
		if (chooser.showSaveDialog(new JFrame()) == JFileChooser.APPROVE_OPTION)
		{
			File f = chooser.getSelectedFile();
			if (f.isDirectory())
				return null;
			else
			{	lastDirectory = chooser.getCurrentDirectory();
				return f;
			}			
		}		
		return null;  
	}
	
	private class ExtractThread extends Thread
	{
		private SourceBuilder sb = null;
		
		public ExtractThread(SourceBuilder s)
		{	sb = s; }
		
		public void run()
		{	sb.threadDoAdd();					
		}
	}
	
	private void doAdd()
	{
		ExtractThread et = new ExtractThread(this);
		et.start();				
	}
	
	private void threadDoAdd() 
	{	// Adds a data source to sources.xml file and creates a new XSpec for it
		this.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

		txtMessages.setText("");
		txtMessages.append("Preparing to add data source...\n");

		String sourceFileName = txtSources.getText();
		String xspecFileName = txtXSpec.getText();
							
		// Make a connection and perform statistics gathering
		Connection con=null;
		String driverName = txtDriver.getText();
		String url = txtURL.getText();
		
		try
		{	// Make database connection
			txtMessages.append("Registering driver: "+driverName+"\n");
		    Class.forName(driverName);
		    txtMessages.append("Driver successfully registered.\n");
		    txtMessages.append("Connecting to database...\n");
		    con = DriverManager.getConnection(url);
		    txtMessages.append("Connection established to database.\n");
        	
        	// Extract database schema information into local data structure
            AnnotatedExtractor extractor = new AnnotatedExtractor();
            extractor.setOutputArea(txtMessages);
            txtMessages.append("Extracting schema information.");
            extractor.extract(con);
            
            // Allow user to provide different name for this database than is referenced by the system.
            String dbName = txtName.getText();
            if (dbName != null && !dbName.equals(""))
            	extractor.setDatabaseName(dbName);
            	
            // Write out schema information into an XML file
            txtMessages.append("Writing information into XML file.\n");
            File xSpec = new File(xspecFileName);
            extractor.exportXML(xSpec);
            
            txtMessages.append("Updating sources file.\n");
            File sourceFile = new File(sourceFileName);
            ArrayList lines = new ArrayList(200);
            FileWriter writer = null;
    		if (!sourceFile.exists())
    		{	txtMessages.append("Creating new sources file.\n");
    			writer = new FileWriter(sourceFile);
    			writer.write("<SOURCES>\n");
    		}
    		else
    		{	// Read entire file and then write it out again except for last line
    			FileReader reader = new FileReader(sourceFileName);
    			BufferedReader breader = new BufferedReader(reader);
    			String st = breader.readLine();
    			while (st != null)
    			{	lines.add(st);
    				st = breader.readLine();
    			}
    			breader.close();
    			reader.close();
    			
    			// Write all lines back out except for last one
    			writer = new FileWriter(sourceFile);
    			for (int i=0; i < lines.size()-1; i++)
    				writer.write((String) lines.get(i)+"\n"); 
    		}
    		
    		// Write information and close file
    		writer.write("\t<DATABASE>\n");
    		writer.write("\t\t<URL>"+url+"</URL>\n");
	    	writer.write("\t\t<DRIVER>"+driverName+"</DRIVER>\n");
	    	writer.write("\t\t<XSPEC>"+xspecFileName+"</XSPEC>\n");
	    	writer.write("\t</DATABASE>\n");
	    	writer.write("</SOURCES>\n");
            writer.close();
            
            txtMessages.append(addSuccessMessage);            
		}
		catch(ClassNotFoundException ce)
		{	txtMessages.append("ClassNotFoundException for Driver: "+ce.getMessage()+"\n");
			txtMessages.append("Make sure driver is in your classpath.\n");
			txtMessages.append(addFailedMessage);
		}
		catch (SQLException sqlEx)
        { 	txtMessages.append("SQLException during extract: "+ sqlEx.getMessage()+"\n"); 
        	txtMessages.append(addFailedMessage);
        }
		catch (IOException ex)
        { 	txtMessages.append("IOException during extract: "+ ex.getMessage()+"\n"); 
        	txtMessages.append(addFailedMessage);
        }
		finally{
			this.setCursor(null);
		}
	}
}
