/**
  * The JEP class is the main interface with which the user should
  * interact. It contains all neccessary methods to parse and evaluate 
  * expressions.
  * @author Nathan Funk
  */

package nFunk.JEP;

import nFunk.JEP.*;
import java.io.*;
import nFunk.JEP.postfixMathCommand.*;


public class JEP {

	/** symTab: Symbol Table */
	private SymbolTable symTab;
	
	/** funTab: Function Table*/
	private FunctionTable funTab;
	
	/** topNode: Node at the top of the parse Tree*/
	private Node topNode;

	/** traverse: Traverse Option*/
	private boolean traverse;

	/** parser: The parser object*/
	private Parser parser;
	
	/** hasError: Boolean value; Is set to false
	when expression has been parsed succesfully*/
	private boolean hasError;

	/** Constructor; initializes all internal variables*/
	public JEP()
	{
		topNode = null;
		traverse = false;
		hasError = true;
		initSymTab();
		initFunTab();
		parser = new Parser(new StringReader(""));
	}

	/**
	  * Initializes the symbol table
	  */
	public void initSymTab() {
		//Init SymbolTable
		symTab = new SymbolTable();
	}
	
	/**
	  * Initializes the function table
	  */
	public void initFunTab() {
		//Init FunctionTable
		funTab = new FunctionTable();
	}

	/**
	  * Adds the standard functions to the parser.
	  */
	public void addStandardFunctions()
	{
		//add functions to Function Table
		funTab.put("sin", new Sine());
		funTab.put("cos", new Cosine());
		funTab.put("tan", new Tangent());
		funTab.put("asin", new ArcSine());
		funTab.put("acos", new ArcCosine());
		funTab.put("atan", new ArcTangent());
		funTab.put("log", new Logarithm());
		funTab.put("angle", new Angle());
		funTab.put("abs", new Abs());
		funTab.put("mod", new Modulus());
	}

	/**
	  * Adds the constants pi and e to the parser.
	  */
	public void addStandardConstants()
	{
		//add constants to Symbol Table
		symTab.put("pi", new Double(Math.PI));
		symTab.put("e", new Double(Math.E));
	}

	/**
	  * Parses the expression
	  * @param expression_in The input expression string
	  */
	public void parseExpression(String expression_in)
	{
		Reader reader = new StringReader(expression_in);
		
		try
		{
			topNode = parser.parseStream(reader, symTab, funTab, traverse);
			hasError = false;
		} catch (Throwable e)
		{
			topNode = null;
			hasError = true;
		}
	}

	/**
	  * Reports whether there is an error in the expression
	  * @return Returns true if the expression has an error
	  */
	public boolean hasError()
	{
		return hasError;
	}

	/**
	  * Reports information on the error in the expression
	  * @return Returns a string containing information on the error
	  */
	public String getErrorInfo()
	{
		if (hasError)
			return "Has Error";
		else
			return "";	
	}


	/**
	  * Evaluates and returns the value of the expression.
	  * @return The calculated value of the expression
	  */
	public double getValue()
	{
		if (!hasError)
			return topNode.getValue();
		else
			return 0;
	}


	/**
	  * Adds a new function to the parser. This must be done before parsing
	  * an expression so the parser is aware that the new function may be
	  * contained in the expression.
	  */
	public void addFunction(String functionName, Object function)
	{
		funTab.put(functionName, function);
	}

	/**
	  * Adds a new variable to the parser, or updates the value of an 
	  * existing variable. This must be done before parsing
	  * an expression so the parser is aware that the new variable may be
	  * contained in the expression.
	  * @param name Name of the variable to be added
	  * @param value Initial value or new value for the variable
	  */
	public void addVariable(String name, double value)
	{
		symTab.put(name, new Double(value));
	}

	/**
	  * setTraverse is useful for debugging purposes. When setTraverse(true)
	  * is called, the parse-tree will be traversed with command-line output.
	  * The default setting is traverse = false.
	  */
	public void setTraverse(boolean traverse_in)
	{
		traverse = traverse_in;
	}

/*	public void setVarValue()
	{

	}*/
}

