/* ex:set ts=4 sw=4:
 *
 * travel.t:	a smarter map structure
 *
 * This module provides a slightly smarter map structure for TADS games.
 * It adds the ability for methods to determine the exits which are
 * available from a specific room without triggering dubious messages,
 * traps, etc.
 *
 * It is *not* compatible with the original TADS room structures in that it
 * requires extra parameters on every travel method.  Where a room used to
 * have a method "north", it now needs a method "north(actor)".  If this
 * method is called with Me, it is a normal player travel; if a different
 * actor is used (or nil) then its a non-player-character or a map-walking
 * subroutine - no output or nasty effects should be generated.
 *
 * It is also necessary to modify the noexit() method to take *two* args so
 * that a room which had "noexit" now needs "noexit(actor,horizontal)".
 * The first is the actor performing the travelling - same conditions apply
 * as for normal methods.  The second is a parameter which defines whether
 * the travel was horizontal (n,s,e,w,in,out, etc) or vertical (u,d,etc).
 * This can be used for smarter error messages (so "You bump your head on
 * the wall" doesnt come out when you type "DOWN" in a place where down is
 * not a valid direction.
 * 
 * The module also provides two new travel verbs. "climb up" and "climb down"
 * are two new ways to travel and "exits" will display a message describing
 * the current possible exits.
 *
 * It also provices a new method which allows the complete destruction of a
 * room.  I needed this for the port of Dungeon since it is possible to
 * explode a bomb in a room and all other rooms would no longer let you in
 * there.
 *
 * Adds:
 *	room.climbUp(actor)
 *	room.climbDown(actor)
 *	room.kaboom(msg)
 *		This method will reduce the specified room to rubble.  All exits
 *		which previouly ended there will return a message instead.
 *	room.rubbled
 *		This flag indicates that a room has been 'kaboom'ed
 *
 * Replaces:
 *	room.(all travel types)
 *
 * This module is Copyright (c) 1993 Jeff Laing.  Permission to use any or all
 * of this code in other TADS games is granted provided credit is given in
 * your "CREDITS" command for this code and that you leave this copyright
 * message in the source whenever distributed.  Please make all changes in
 * a backward compatible manner (where possible) and make them available
 * for free (like I did ;-)
 */
travelVersion : versionTag
	id="$Id: travel.t_v 1.10 1994/05/09 04:15:03 jeffl Exp jeffl $\n"
	author='Jeff Laing'
	func='enhanced travelling'
;


/*
 * Room's are extended in a number of ways.
 *
 * The room exit handlers all pass the actor attempting the motion and
 * whether the motion is horizontal or vertical.  This allows dynamic
 * mapping of the dungeon and more meaningful messages (i.e. 'up' doesn't
 * generate "There is a wall there."
 */
modify class room 

    /*
     *   Direction handlers.  The directions are all set up to
     *   the default, which is no travel allowed.  To make travel
     *   possible in a direction, just assign a room to the direction
     *   property.
     */
    replace north(a)	= (self.noexit(a,0))
    replace south(a)	= (self.noexit(a,0))
    replace east(a)		= (self.noexit(a,0))
    replace west(a) 	= (self.noexit(a,0))
    replace ne(a)  		= (self.noexit(a,0))
    replace nw(a)		= (self.noexit(a,0))
    replace se(a)		= (self.noexit(a,0))
    replace sw(a)		= (self.noexit(a,0))
    replace in(a)		= (self.noexit(a,0))
    replace out(a)		= (self.noexit(a,0))

    replace up(a)   	= (self.noexit(a,1))
    replace down(a)		= (self.noexit(a,1))

	climbUp(a)			= (self.noexit(a,1))
	climbDown(a)		= (self.noexit(a,1))

    /*
     *   noexit displays a message when the player attempts to travel
     *   in a direction in which travel is not possible.
     */
    replace noexit(actor, horizontal) = {
		if (actor = Me) {	// only send message if player is asking
	    	"%You% can't go that way. ";
		}
    	return( nil );
    }

	/*
	 * by default, we have not been destroyed yet
	 */
	rubbled=nil
	kaboom(msg)={
		self.rubbled := msg;
	}

	/*
	 * return a list of the valid directions from here
	 */
	roomexits={
		local i, dirs, prop, txt, here, typ, dest, list;
		
		dirs := [	&north, 'north',
					&south,	'south',
					&east,	'east',
					&west,	'west',
					&ne,	'northeast',
					&nw,	'northwest',
					&se,	'southeast',
					&sw,	'southwest',
					&in,	'in',
					&out,	'out',
					&up,	'up',
					&down,	'down'
				];

		list := [];
		i:= 1;
		while (i < 25) {
			prop := dirs[i++];
			txt := dirs[i++];
			if (roomexit(self,prop)) {
				list += txt;
			}
		}

		return(list);
	}

;

/*
 * darkroom: correct parameter list for noexit().  
 */
modify class darkroom

	// If its not the player asking, don't complain.
    replace noexit(actor, flag) = {
        if ( self.islit or actor <> Me ) pass noexit;
        darkTravel();
		return( nil );
    }
;

/*
 * vehicle: correct parameter list for noexit()
 */
modify class vehicle

	// don't issue messages if the caller is not the player
    noexit(actor, horizontal) =
    {
        if (actor=Me) {
			"%You're% not going anywhere until %you% get%s% out of ";
			self.thedesc; ". ";
		}
        return( nil );
    }

	out(actor) = (self.location);
;

/*
 * travelVerb: all travel verbs have the same action - why not inherit?
 */
modify class travelVerb
    action( actor ) = {
		local r;
		r := self.travelDir(actor);
		if (r<>nil) {
			if (r.rubbled<>nil) {
				say(r.rubbled);
			} else {
				actor.travelTo(r);
			}
		}
	}
;

/*
 * each travel verb needs extra parameters passed to the room method
 */
replace eVerb: travelVerb
	sdesc="go east"
    verb = 'e' 'east' 'go east'
    travelDir( actor ) = ( actor.location.east(actor) )
;
replace sVerb: travelVerb
	sdesc="go south"
    verb = 's' 'south' 'go south'
    travelDir( actor ) = ( actor.location.south(actor) )
;
replace nVerb: travelVerb
	sdesc="go north"
    verb = 'n' 'north' 'go north'
    travelDir( actor ) = ( actor.location.north(actor) )
;
replace wVerb: travelVerb
	sdesc="go west"
    verb = 'w' 'west' 'go west'
    travelDir( actor ) = ( actor.location.west(actor) )
;
replace neVerb: travelVerb
	sdesc="go northeast"
    verb = 'ne' 'northeast' 'go ne' 'go northeast'
    travelDir( actor ) = ( actor.location.ne(actor) )
;
replace nwVerb: travelVerb
	sdesc="go northwest"
    verb = 'nw' 'northwest' 'go nw' 'go northwest'
    travelDir( actor ) = ( actor.location.nw(actor) )
;
replace seVerb: travelVerb
	sdesc="go southeast"
    verb = 'se' 'southeast' 'go se' 'go southeast'
    travelDir( actor ) = ( actor.location.se(actor) )
;
replace swVerb: travelVerb
	sdesc="go southwest"
    verb = 'sw' 'southwest' 'go sw' 'go southwest'
    travelDir( actor ) = ( actor.location.sw(actor) )
;
replace inVerb: travelVerb
    sdesc = "enter"
    verb = 'in' 'go in' 'enter'
    doAction = 'Enter'
    travelDir( actor ) = ( actor.location.in(actor) )
;
replace outVerb: travelVerb
	sdesc="leave"
    verb = 'out' 'go out' 'exit' 'leave'
    travelDir( actor ) = ( actor.location.out(actor) )
;
replace dVerb: travelVerb
	sdesc="go down"
    verb = 'd' 'down' 'go down'
    travelDir( actor ) = ( actor.location.down(actor) )
;
replace uVerb: travelVerb
	sdesc="go up"
    verb = 'u' 'up' 'go up'
    travelDir( actor ) = ( actor.location.up(actor) )
;
climbUpVerb: travelVerb
	sdesc="climb up"
    verb = 'climb up'
    travelDir( actor ) = ( actor.location.climbUp(actor) )
;
climbDownVerb: travelVerb
	sdesc="climb down"
    verb = 'climb down'
    travelDir( actor ) = ( actor.location.climbDown(actor) )
;

/*
 * this function will return the location that is in a particular direction
 * from a specific room.  The function can look past closed doors but only
 * if they are unlocked.
 */
roomexit: function( loc, prop )
{
	local dest;

	switch( proptype(loc, prop) ) {

	case 2:							// Object
		dest := loc.(prop);			// get destination object
		break;

	case 6:							// Code
		dest := loc.(prop)(nil);	// Ask with a NIL actor
		break;

	default:						// all others
		dest := nil;				// are not real exits
	}

	// we got back something.  if its not nil, check for a door
	// if it is a door, we can get through if its open or unlocked.
	if (dest <> nil and dest.isdoor) {
		if (dest.isopen or not dest.islocked) {
			dest := dest.doordest;	// its a door, look on other side
		} else {
			dest := nil;			// closed & locked.
		}
	}

	return( dest );
}

/*
 * find all rooms which can be reached from the specified room and return them
 * as a list.  nil lists mean you can't get here.
 */
exitsfrom : function( here )
{
	local val,dirs,i,len,r,x, temp;

	// catch bad parameters and return 'no way out of here'
	if (here=nil) return [];
	if (not isclass(here,room)) return [];

	// standard direction properties
	dirs := [	&north, &south, &east, &west, &ne, &nw, &se, &sw,
				&in,	&out,	&up,   &down, &climbUp, &climbDown ];

	val := [];
	i := 1;
	len := length(dirs);
	while (i<=len) {
		x := dirs[i];
		r := roomexit( here, x );
		if (r <> nil and not isclass(r,nestedroom)) {
			temp := [] + r - val;
			val := val + temp;
		}
		i := i + 1;
	}
	return val;
}

/*
 * exits: This verb is useful for players who don't want to have to read
 * the game text.  It will give a listing of all the exits which seem to
 * lead somewhere (excluding locked doors)
 */
exitsVerb : sysverb
	verb='exits'
	action(actor) = {
		self.dumpexits(actor.location);
		abort;
	}

	dumpexits(loc) = {
		local i, list;

		list := loc.roomexits;
		"\n[";
		if (datatype(list)=7) {
			switch (length(list)) {
			case 0:	"I don't know where you can go from here"; break;
			case 1: "You can only go <<list[1]>>"; break;
			default:
				"It would appear that you can go ";
				for (i:=1; i<length(list);i++) {
					if (i>1) ", ";
					say(list[i]);
				}
				" or <<list[length(list)]>>"; break;
			}
		} else {
			say(list);
		}
		"]\n";
	}
;

