#include <adv3.h>
#include <en_us.h>



/* --- shared items - the ground and so on --- */

/* the ordinary ground replaces the default ground */
ordinaryGround: defaultGround
    vocabWords = 'ordinary '
    disambigName = 'ordinary ground'
;


modify OutdoorRoom
    roomParts = static inherited - defaultGround + ordinaryGround
;


lawnGrass: defaultGround
    'big neat empty mown mowed soft cool damp fresh sweet grass/lawn'
    'lawn'
    "The lawn is cool and damp and looks like it has just been mowed. "
    feelDesc = "The grass is soft and cool and damp."
    smellDesc = "The grass smells fresh and sweet. It probably <i>has</i> just
        been mowed. "
;


fence: MultiFaceted
    locationList = [cubbyArea, lotsaBushes, behindShed, underTree, vegeGarden]
    instanceObject: Fixture
    {
        'tall green smooth fence'
        'fence'
        "The fence is tall and smooth and green and made out of some kind of 
        special stuff that they advertise on TV. It's supposed to last a 
        really long time in the sun or something like that. "
        
        cannotClimbMsg = 'The fence is too high and slippery. You can\'t 
            get a grip on it. '
    }
;



house: MultiFaceted
    locationList = [lawnWest, lawnEast, vegeGarden, underTree, frontGarden]
    instanceObject: Building
    {
        'emma\'s family\'s really new cute little sloping ceilings ceiling 
        windows (in) (the) roof house back real
        door/side/ceilings/ceiling/windows/roof/attic/house'
        'house'
        "Emma and her family only just moved here, and the house is new.
        Emma's room is in the attic. It's really cool --- it has sloping 
        ceilings and cute little windows peeking out through the roof. She 
        showed you when you got here. "
        disambigName = 'real house'
        
        connector = enterHouse
        
        /* 
         *   if the player tries to GO THROUGH BACK DOOR, remap to entering 
         *   the house
         */
        dobjFor(GoThrough)
        {
            remap()
            {
                if (gAction.getOrigText.toLower.find('door'))
                {
                    return [EnterAction, self];
                }
                else
                {
                    return inherited();
                }
            }
        }
    }
    
    /* TODO: response to LOOK IN */
;



shed: MultiFaceted
    locationList = [lawnWest, lawnEast, lotsaBushes, behindShed, cubbyArea]
    instanceObject: Building
    {
        'pretty big emma\'s dad\'s shed/workshop'
        'shed'
        "This is Emma's dad's workshop or something like that. It's a pretty big 
        shed, but not very interesting. You know how some
        sheds have funny signs and cool stuff hanging on the walls and things
        like that? Well, this one doesn't --- at least not on the outside. "
        
        connector = enterShed
    }
    
    /* TODO: response to LOOK IN */
;



/* --- lawnWest --- */



/* --- lawnEast --- */

flowerBedsBack: Fixture
    'few weird spiky nearly empty flower 
    bed/flowerbed/plant/dirt/flower-bed*beds*plants*flowerbeds*flowers*flower-beds'
    'flower beds'
    @lawnEast
    "There are a few flowers in the flower beds, but mostly
    it's just weird spiky plants that you've never seen before. But there
    aren't even that many of those --- a lot of the flower beds are nearly 
    empty."
    
    smellDesc = "The flowers don't smell like much. Maybe they're not even
        flowers. "
;



/* --- cubbyArea --- */

cubbyHouse: ComplexContainer, Fixture, HidingSpot, PlayThing
    'really cool big cubby cubby/(house)/cubbyhouse/cubby-house/side'
    'cubby house'
    @cubbyArea
    "The cubby house is big, painted in vivid shades of red and yellow.
    A ladder leads up into the cubby on one side, and on another side 
    a slide runs down into the sandpit. "
    
    subContainer: ComplexComponent, Booth
    {
        performEntry(posture)
        {
            local origin = gActor.location;
            inherited(posture);
            
            if (origin != gActor.location)
            {
                "You scramble up the ladder into the cubby house. ";
            }
        }
    }
    
    pcSearchMsg = 'Standing on the ladder, you peek into the cubby house. 
        It\'s completely empty. No toys, no cupboards, no anything. And 
        definitely no Emma. '
    npcSearchMsg = '{The actor/she} scrambles up the cubby house ladder to look 
        inside the cubby. '
    
    /* This is for both the cubby house itself and the slide. */
    playWithTiana = "<q>Do you want to play in the cubby house with me?</q>
        you ask.<.p>
        {The tiana/she} hesitates. <q>That sounds like fun,</q> she admits,
        <q>but what about Emma's game? Won't she be mad if we just go and
        play without her?</q><.p>
        <q>Who cares? She didn't ask us what we wanted to play, she just 
        decided for us. Anyway, this will be heaps more fun.</q><.p>
        <<tiana.isProperName ? 'Tiana' : 'The girl'>> grins suddenly.
        <q>Okay,</q> she says. <q>Come on then.</q><.p>
        You and Tiana <<tiana.learnNameInPassing>> slide down the slide 
        a few times before Emma 
        <<yvonne.isIn(bushNearCubby) ? 'and Yvonne emerge' : 'emerges'>>
        from the bush near the cubby. <q>What are you doing?</q>
        <<yvonne.isIn(bushNearCubby) ? 'Emma' : 'she'>> protests. <q>What
        about the game?</q><.p>
        <q>Oh no!</q> you shout, <q>the enemy's army is attacking! Quick, 
        Tiana, pull up the drawbridge! Get the bows and arrows!</q><.p>
        Emma scowls at you. <q>You're a mean spoilsport! Come on, Yvonne,
        let's go do our own thing on our own, they're just being 
        boring.</q><.p>
        She and Yvonne sulk in the house for the rest of the day while
        you and Tiana defend the cubby house from the invading army. By the
        time you have to go home, you've been invited to Tiana's house
        for a visit next weekend. "
    
    /* Instead of climbing the cubby, climb the ladder. */
    dobjFor(Climb) remapTo(Climb, cubbyLadder)
;



+ cubbyLadder: Component
    'wooden wood (cubby) (house) ladder'
    'ladder'
    "The sturdy wooden ladder goes up into the cubby house. "
    
    dobjFor(Climb)
    {
        verify()
        {
            logicalRank(150, 'is a ladder');
        }
        
        check()
        {
            if (gActor.isIn(cubbyHouse))
                failCheck('You\'re already in the cubby house. ');
        }
        
        action()
        {
            replaceAction(Enter, cubbyHouse);
        }
    }
    
    dobjFor(ClimbDown)
    {
        verify() { }
        
        check()
        {
            if (!gActor.isIn(cubbyHouse))
                failCheck('You\'re already as far down as you can go. ');
        }
        
        action()
        {
            replaceAction(GetOut);
        }
    }
;



+ cubbySlide: Component, PlayThing
    'bright red (and) yellow slide'
    'slide'
    "The slide is bright red and yellow. It comes out of an opening in the side
    of the cubby house and goes down into the sandpit. It isn't very long, but
    it's still a slide. "
    
    actorInPrep = 'on'
    
    dobjFor(Climb)
    {
        verify() {}
        
        check()
        {
            if (gActor.isIn(cubbyHouse))
            {
                failCheck('{You/he} {is} already in the cubby house. ');
            }
        }
        
        action()
        {
            nestedAction(Enter, cubbyHouse);
            "You scramble up the slide into the cubby house. ";
        }
    }
    
    dobjFor(ClimbDown)
    {
        verify()
        {
            logicalRank(80, 'not a ladder');
        }
        
        check()
        {
            if (!gActor.isIn(cubbyHouse))
            {
                failCheck('You\'re already as far down as you can go. ');
            }
        }
        
        action()
        {
            "You scramble awkwardly down the slide, trying not to slip and
            fall. ";
            replaceAction(GetOut);
        }
    }
    
    dobjFor(SlideDown)
    {
        verify()
        {
            logicalRank(150, 'is a slide');
        }
        
        check()
        {
            if (!gActor.isIn(cubbyHouse))
            {
                failCheck('You\'re already as far down as you can go. ');
            }
        }
        
        action()
        {
            "You sit down at the top of the slide and push yourself off.
            At the bottom, you land with a soft bump in the sandpit. ";
            gPlayerChar.moveIntoForTravel(sandpit);
            gPlayerChar.makePosture(sitting);
        }
    }
    
    /* 
     *   Playing on the slide is the same as climbing it or sliding down it, 
     *   depending on where the PC is.
     */
    dobjFor(PlayOn)
    {
        remap()
        {
            if (gPlayerChar.isIn(cubbyHouse))
                return [SlideDownAction, DirectObject];
            else
                return [ClimbAction, DirectObject];
        }
    }
    
    /* 
     *   Entering the slide is the same as sliding down it, if you're in the 
     *   cubby.
     */
    dobjFor(Enter) asDobjFor(SlideDown)
    dobjFor(SitOn) maybeRemapTo(gActor.isIn(cubbyHouse), SlideDown, self)
    
    /* Not much point in sitting on the slide otherwise. */
    cannotSitOnMsg = 'There\'s not much point in sitting at the bottom of the
        slide. '
    
    playWithTiana = cubbyHouse.playWithTiana
;



sandpit: Immovable, Booth, PlayThing
    'big smooth-surfaced smooth surfaced
     sand pit/sand/sandpit/sand-pit/log/logs'
    'sandpit'
    @cubbyArea
    "Big, smooth-surfaced logs keep the sand in the sandpit. There are 
    no toys or anything in it, just sand. "
    
    feelDesc = "The sand is slightly damp on top but dry underneath. "
    
    /* Digging in the sandpit */
    dobjFor(Dig)
    {
        verify()
        {
            logicalRank(150, 'is a sandpit');
        }
        
        check() { }
        
        action()
        {
            "After the rain last night, the sand is slightly damp on top 
            but dry underneath. You gather up some of the dampest sand
            and build it into a rough sandcastle, then try to tunnel
            through it. But the sand isn't wet enough to hold together,
            and it collapses. ";
        }
    }
    
    
    /* Playing in the sandpit is the same as digging in it. */
    dobjFor(PlayOn) asDobjFor(Dig)
    
    
    /* Messing about with the sand */
    cannotTakeMsg = 'You pick up a handful of sand and let it trickle
        through your fingers. '
    cannotTakeLocationMsg = cannotTakeMsg
    cannotMoveMsg = 'You push a pile of sand up into a sort of bank, and
        then knock it down again. '
    
    playWithTiana = "<q>Do you want to play in the sandpit with me?</q> you
        ask {the tiana/her}. <q>We can build a really big sandcastle 
        together.</q><.p>
        She hesitates. <q>That sounds like fun, but what about Emma's 
        game?</q><.p>
        <q>Who cares about that? Emma didn't even ask us what we wanted to
        play. Come on, this will be heaps more fun.</q><.p>
        {The tiana/she} laughs suddenly. <q>Okay then,</q> she says,
        <q>let's do it then.</q><.p>
        As you and <<tiana.isProperName ? 'Tiana' : 'the girl'>> settle down
        in the sandpit, Emma <<yvonne.isIn(bushNearCubby) ? 'and Yvonne 
            scramble' : 'scrambles'>> out of <<gPlayerChar.knowsWhereEmmaIs ?
          'the' : 'a nearby'>> bush. <q>Why aren't you 
        playing the game? You\'re a mean spoilsport!</q> 
        <<yvonne.isIn(bushNearCubby) ? 'Emma' : 'she'>> protests.<.p>
        But you and Tiana <<tiana.learnNameInPassing>> ignore her and build a big 
        sandcastle, with a moat (but no water) and towers on all the corners. It's 
        a bit hard with no tools and Emma and Yvonne make fun of it, but that's
        okay, because next weekend you're going to visit Tiana at her place 
        instead.<.p> "
;


bushNearCubby: Fixture, Booth, HidingSpot
    vocabWords = 'bush (near) (the) (cubby) bush'
    /* 
     *   TODO: understand 'hiding spot' as the bush if the player knows Emma 
     *   is hiding there
     */
    location = cubbyArea
    desc = "You don't know what sort of bush it is --- you don't really know 
        anything about bushes, actually. But it's big and thick and it looks 
        like there's room to hide in it. "
    
    disambigName = 'bush near the cubby'
    
    name()
    {
        if (gPlayerChar.isIn(cubbyArea))
        {
            return 'bush';
        }
        else
        {
            return disambigName;
        }
    }
    
    /* 
     *   shouldn't be able to see or hear people hiding in the bush unless
     *   you're right next to it
     */
    transSensingIn(sense)
    {
        if (gActor && !gActor.isIn(location))
            return opaque;
        
        return inherited(sense);
    }
    
    /* sitting in the bush is more probable if the PC knows Emma is there */
    dobjFor(SitOn)
    {
        verify()
        {
            inherited();
            
            if (gActor.knowsWhereEmmaIs)
                logicalRank(120, 'hiding with Emma');
        }
    }
    
    /* 
     *   searching the bush allows the player to discover Emma (and Yvonne, 
     *   if Yvonne is there
     */
    dobjFor(Search)
    {
        /* it's always okay to search this hiding spot */
        check() { }
        
        action()
        {
            if (gActor.isPlayerChar)
            {
                /* 
                 *   If the player is the last to find the hiding spot, she 
                 *   loses. Otherwise, she discovers who's hiding here and the 
                 *   game continues.
                 */
                if (tiana.isIn(self) && yvonne.isIn(self))
                {
                    "You push aside a few branches to find Emma, Yvonne and 
                    {the tiana/she} all huddled in the bush.<.p>";
                    searching.loseGame();
                }
                else
                {
                    if (!searchedBy(gActor) && !gActor.isCheating)
                    {
                        if (gActor.isPlayingHnS)
                        {
                            "You lift a few branches and peek into the bush. 
                            Hey, ";
                        }
                        else
                        {
                            "You look behind the bush --- nothing. So you lift a
                            few branches  and peek underneath. Aha! ";
                        }
                        searchedByList += gActor;
                    }

                    gMessageParams(contents);
                    "{The/list contents} {are} sitting there. ";
                    foreach (local girl in contents)
                    {
                        girl.isHidden = nil;
                    }
                }
            }
            else
            {
                /* 
                 *   When Yvonne or Tiana finds Emma, mark the hiding spot 
                 *   as searched by her. On the next turn, she'll hide here.
                 */
                searchedByList += gActor;
                
                /* Report that she searched the bush. */
                "{The actor/she} gets down on her hands and knees to look under 
                the bush behind the cubby house. ";
            }
        }
    }
    
    /* 
     *   What happens when the player enters the bush. This always ends the 
     *   story. There are various endings depending on who is already hiding 
     *   here, but it should be impossible for the player to win without 
     *   cheating.
     */
    actorTravelingWithin(origin, dest)
    {
        if (gActor == gPlayerChar)
        {
            if (dest == self)
            {
                /* 
                 *   If the PC is the last to arrive, remap to searching the 
                 *   bush.
                 */
                if (yvonne.isIn(self) && tiana.isIn(self))
                {
                    replaceAction(Search, self);
                }
                
                /* 
                 *   Quickly stick the PC in the appropriate posture, to make sure 
                 *   the status line says the right thing.
                 */
                gPlayerChar.makePosture(hiding);
                
                /* 
                 *   Betrayal endings - when you trick Tiana and/or Yvonne into 
                 *   playing Hide and Seek and then hide with Emma anyway.
                 */
                if (gPlayerChar.isPlayingHnS)
                {
                    if (!yvonne.isPlayingHnS)
                    {
                        "You watch carefully for a moment when no one is looking,
                        and then sneak underneath the bush. Emma grins at
                        you<<yvonne.isIn(bushNearCubby) ? ' and Yvonne shifts over 
                            to make a bit more room.' : '. Pretty soon Yvonne
                                finds you and hides with you as well.'>><.p>
                        Finally {the tiana/she} <<tiana.roleInHnS ==
                          roleCounting ? 'finishes counting' : 'wonders what
                              happened to you'>> and comes
                        looking for you. <q>What took you so long?</q> Emma
                        demands. <q>You're last, you lose, losers smell like
                        dirty shoes. I'm bored of this game now. Come on,
                        let's go back inside, I want some cordial.</q>
                        She scrambles out of the bush and scampers off 
                        towards the house.<.p>
                        {The tiana/she} stares at you. <q>But --- what about
                        playing Hide and Seek? You said...</q><.p>
                        You grin. <q>Tricked you! You totally fell for 
                        it!</q><.p>
                        {The's tiana/her} lips quiver. She hesitates a moment,
                        then spins around and runs away.<.p>
                        Yvonne is giving you a funny look, but she doesn't 
                        say anything. ";

                        
                        endGame('You have betrayed {the tiana/her}');
                    }
                    /* 
                     *   else endings where you win by treachery (or try to, but 
                     *   Emma knows what you're up to)
                     */
                    else
                    {
                        "You wriggle under the bush next to Emma. She grins at
                        you and shifts over to make more room.<.p>
                        It's ages before {the tiana/she} comes to find you.
                        Yvonne is close behind her, helping her search.<.p> 
                        <q>What took you two so long?</q> Emma demands. <q>You 
                        two are the slowest slowpokes ever.</q><.p>
                        Yvonne stares at you. <q>What about playing
                        Hide and Seek?</q> she says. <q>You asked me and 
                        Tiana<<tiana.isProperName ? '' : '</q> (of course, 
                            that\'s the red-haired girl\'s name) <q>'>> to
                        play with you and then you just started playing with
                        Emma again?</q><.p>
                        <q>You started playing Hide and Seek instead?</q>
                        Emma protests. <q>But we were playing my game!</q><.p>
                        You grin at Yvonne and Tiana. <q>Tricked you! It was
                        a good trick, wasn't it? You both totally fell for 
                        it!</q><.p>
                        The next thing you know, Yvonne has knocked you to the 
                        ground. Emma screams and the grownups come running 
                        and pull Yvonne away, but not before she's given you
                        a new loose tooth. Yvonne's mum apologises and 
                        Yvonne refuses to apologise. Mum takes you home soon
                        afterwards. ";
                        
                        endGame('Your plan has backfired');
                    }
                }
                
                /* If the PC is the first to arrive... */
                if (!yvonne.isIn(self) && !tiana.isIn(self))
                {
                    if (gPlayerChar.isCheating && !tiana.isCheating)
                    {
                        /* 
                         *   If the PC is the first to arrive and *is* 
                         *   cheating (but Tiana isn't), she gets the 
                         *   victorious-by-underhanded-means ending.
                         */
                        "You duck under the branches and wriggle in next to Emma. She
                        grins at you and shuffles over to make room. Pretty soon
                        Yvonne joins you, and finally {the Tiana/she} appears too.<.p>
                        Emma scrambles out of the bush, pushing past her. <q>You 
                        were ages, what took you so long?</q> she demands. 
                        <q>Doesn't matter. You're last, you lose, losers smell like 
                        dirty shoes.\ I'm bored of this game now, let's go inside. 
                        I'm thirsty.</q><.p>
                        {The Tiana/she} screws up her face like she's going to cry. 
                        Emma turns to you. <q>You won, so you can decide what we do 
                        next. D'you want to play Barbies? Come on, I'll show you all my 
                        Barbies and you can choose which to play with first.</q>";
                        endGame('You have cheated');
                    }
                    else if (gPlayerChar.isCheating && tiana.isCheating)
                    {
                        /* 
                         *   If the PC is first to arrive and both she and Tiana 
                         *   are cheating, Yvonne loses. In this case, all the 
                         *   characters will arrive more or less at once...
                         */
                        "You duck under the branches and wriggle in beside Emma,
                        who shifts to make room for you. {The tiana/she} crawls in a
                        moment later and gives you a secret grin when Emma isn't
                        looking.<.p>
                        Not much later, Yvonne finds the hiding spot too. Emma 
                        pushes past her and scrambles out of the bush. <q>You 
                        were ages, what took you so long?</q> she demands. 
                        <q>Doesn't matter. You're last, you lose, losers smell like 
                        dirty shoes.\ I'm bored of this game now, let's go inside. 
                        I'm thirsty.</q><.p>
                        Yvonne scowls and stomps off, but as she turns you see 
                        something shining in the corner of her eye. Emma stares 
                        after her. <q>What's the matter with her? She's always so 
                        grumpy when she loses. Oh, never mind her, let's go inside. 
                        There's some cordial in the fridge.</q> ";
                        endGame('Yvonne has lost');
                    }
                    else
                    {
                        /* 
                         *   If the PC is the first to arrive and is not cheating, 
                         *   it's a bug. This shouldn't be possible.
                         */
                        "[It shouldn't be possible for the PC to win without 
                        cheating.] ";
                        endGame('This is a BUG');
                    }
                }
                
                if (yvonne.isIn(self) && !tiana.isIn(self))
                {
                    /* 
                     *   If the PC just follows Yvonne across the lawn, Yvonne 
                     *   accuses her of cheating.
                     */
                    if ((!yvonne.isCheating || yvonne.isCheatingOfHerOwnAccord)
                        && 
                        gPlayerChar.pathToHidingSpot == Person.pathAcrossLawn)
                    {
                        "You duck under the branches and wriggle in next to Emma.
                        She grins at you, but Yvonne scowls.<.p>
                        <q>You're cheating,</q> she whispers fiercely. <q>Emma,
                        she cheated, you saw her. She just followed me straight
                        across the lawn instead of playing properly.</q><.p>
                        <q>So?</q> you retort. <q>You didn't do much looking 
                        either, did you? How come you knew to come straight
                        here?</q><.p>
                        Yvonne flushes. <<yvonne.isCheating && 
                          yvonne.isCheatingOfHerOwnAccord? '(Maybe you guessed 
                          right.)' : ''>> <q>You're a liar! I didn't-- I'm 
                        not a cheater, you are!</q><.p>
                        Before you can reply, Emma jumps to her feet. <q>You're
                        both mean spoilsports! You've ruined the game, and I hate you
                        both!</q> Tears sparkle in her eyes a moment, before she
                        turns and runs off towards the house.
                        <<tiana.isIn(cubbyArea) ? '' : '<.p>{The tiana/she}, 
                            coming around the corner, stares after her in 
                            confusion, but you and Yvonne don\'t explain.'>> ";
                        endGame('You have all lost');
                    }
                    /* Otherwise, Tiana loses. */
                    else
                    {
                        "You duck under the branches and wriggle in next to Emma and
                        Yvonne. Emma grins at you <<gPlayerChar.isScaredOfYvonne ?
                          'and shifts to make a bit more room but Yvonne ignores you' :
                          'and Yvonne smiles and moves to make a bit more space'>>.
                        Finally {the's Tiana/her} face peeks through the branches. Emma
                        pushes past her and scrambles out of the bush. <q>You 
                        were ages, what took you so long?</q> she demands. 
                        <q>Doesn't matter. You're last, you lose, losers smell like 
                        dirty shoes.\ I'm bored of this game now, let's go inside. 
                        I'm thirsty.</q><.p>
                        {The Tiana/she} screws up her face like she's going to cry, 
                        but Emma and Yvonne scamper off to the house, ignoring 
                        her. ";
                        endGame('{The Tiana/she} has lost');
                    }
                }
                
                /* 
                 *   If the player arrives after Tiana but before Yvonne, the 
                 *   player must have helped Tiana by cheating. (If not, it's a 
                 *   bug.) In this case, Yvonne loses. She's a bad loser.
                 */
                if (!yvonne.isIn(self) && tiana.isIn(self))
                {
                    if (tiana.isCheating)
                    {
                        /* this is as it should be */
                        "You duck under the branches and wriggle in beside Emma and 
                        {the Tiana/her}. Emma shifts to make room for you and 
                        {the Tiana/she} grins at you when
                        Emma isn't watching. Pretty soon Yvonne finds the hiding spot
                        too. Emma pushes past her and scrambles out of the bush. <q>You 
                        were ages, what took you so long?</q> she demands. 
                        <q>Doesn't matter. You're last, you lose, losers smell like 
                        dirty shoes.\ I'm bored of this game now, let's go inside. 
                        I'm thirsty.</q><.p>
                        Yvonne scowls and stomps off, but as she turns you see something
                        shining in the corner of her eye. Emma stares after her. <q>What's
                        the matter with her? She's always so grumpy when she loses.
                        Never mind her, let's go inside. There's some lemon cordial in
                        the fridge.</q> ";
                        endGame('Yvonne has lost');
                    }
                    else
                    {
                        /* this is a bug */
                        "[Tiana shouldn't be able to get to the bush before Yvonne
                        unless Tiana is cheating (because the PC cheated and told 
                        her where Emma is hiding).] ";
                        endGame('This is a BUG');
                    }
                }
            }
        }
        else /* if the actor isn't the PC */
        {
            inherited(origin, dest);
        }
    }
    
    
    /* Report an NPC hiding in the bush, if necessary. */
    roomOkayPostureChangeMsg(posture, obj)
    {
        gMessageParams(obj);
        
        if (gActor == tiana)
        {
            return('Pushing aside a branch, {the tiana/she} crawls under the
                bush near the cubby. ');
        }
        else if (gActor == yvonne)
        {
            if (!yvonne.isCheatingOfHerOwnAccord &&
                !tiana.isIn(bushNearCubby))
            {
                return('Yvonne glances about to make sure that {the tiana/she}
                    isn\'t watching and ducks under the bush near the cubby. ');
            }
            else
            {
                return('Thinking no one is watching her, Yvonne ducks under the 
                    bush near the cubby. ');
            }
        }

        return inherited(posture, obj);
    }
    
    
    /* 
     *   This is the most likely place to hide if the PC knows Emma is 
     *   hiding there and the characters aren't playing Hide and Seek.
     */
    dobjFor(HideIn)
    {
        verify()
        {
            if ((gActor.canSee(emma) || gActor.isCheating) &&
                !gActor.isPlayingHnS)
            {
                logicalRank(200, 'hiding with Emma');
            }
            else
            {
                inherited();
            }
        }
    }
    
    
    /* 
     *   So the player doesn't betray Tiana by accident, if she tries to hide in
     *   the bush without knowing Emma is there, just search the bush instead so
     *   she has a choice to hide somewhere else instead.
     */
    beforeTravel(traveler, connector)
    {
        if (traveler == gPlayerChar && traveler.isPlayingHnS && connector == nil 
            && gDobj == self && !gPlayerChar.knowsWhereEmmaIs)
        {
            nestedAction(Search, self);
            
            "<.p>If Emma is already hiding here, you should probably hide
            somewhere else instead. ";
            
            exit;
        }
    }
;



/* --- behindShed --- */

longGrass: Fixture
    'long grass'
    'long grass'
    @behindShed
    "The grass behind the shed is long and tangly. Some of it comes up past 
    your knees. It must not have been cut for a long time. "
    
    cannotTakeMsg = 'Weeding is a generous idea, but there\'s way too much grass 
        here for you to get anywhere with it. '
    
    notAHidingSpotMsg = 'The grass is long, but not <i>that</i> long. You\'d
        definitely be able to see Emma if she were hiding in it. '
;



junkPile: Heavy, HidingSpot
    'old junk broken wheel rusty odds (and) 
    bits/wood/ends/barrow/wheelbarrow/wheel-barrow/pile'
    'pile of junk'
    @behindShed
    "A broken wheelbarrow, bits of wood, rusty odds and ends... that sort of 
    thing. Just junk. It's leaning up against the back wall of the shed, but 
    there's a big enough space between it and the wall for someone to be 
    hiding there. "
    
    actorInPrep = 'behind'
    
    pcSearchMsg = 'You shift a few bits of junk and look behind the wheelbarrow.
        Emma isn\'t hiding here. '
    npcSearchMsg = '{The actor/she} rummages through a pile of junk. '
;



/* --- lotsaBushes --- */

numerousBushes: Fixture, HidingSpot
    'bush bushes big rough wild (next) (near) (to) (the) (shed)
    clump/patch/bush/bushes/stuff/triangle/heaps/bunch/jungle/shrub/shrubs'
    'bushes'
    @lotsaBushes
    "<<gActor.isIn(location) ? 'This' : 'That'>> place is almost like a jungle. 
    Searching all those bushes <<searchedBy(gPlayerChar) ? 'took' : 'is going 
        to take'>> ages. "
    
    disambigName = 'bushes next to the shed'
    
    isPlural = true
    
    /* 
     *   The bushes have special search handling because it takes several 
     *   tries to search them all.
     *
     *   This is really just a ruse to delay Tiana so the player can talk to 
     *   her, or show her the baby bird, or whatever. So searching the 
     *   bushes definitely needs to take long enough that the player go 
     *   southwest to the tree, climb it, examine the bird... Maybe even 
     *   enough time for the player to take the bird, climb down and go show 
     *   it to Tiana? The PC can talk to Tiana from up the tree (fingers 
     *   crossed), but the player may not realise this, since rooms in IF 
     *   tend to be more closed off than they are in "It".
     *
     *   TODO: Clean up and abstract this fugly bunch of code.
     */
    dobjFor(Search)
    {
        action()
        {
            if (gActor.isPlayerChar())
                pcSearchMsgList.doScript();
            else if (gActor == tiana)
                tianaSearchMsgList.doScript();
            else
                yvonneSearchMsgList.doScript();
        }
    }
    
    /* 
     *   The list of PC search messages. The PC manages to search the bushes 
     *   a bit faster than Tiana.
     */
    pcSearchMsgList: EventList
    {
        [
            'You push aside some branches and look behind a few of the bushes.
            Nothing, but there are still heaps of bushes you haven\'t searched. ',
            'You get down on your hands and knees and peek under some larger
            shrubs. Still no sign of Emma, but you still have a few bushes 
            left to search. ',
            new function()
            {
                "You poke about in the <q><<lotsaBushes.name()>></q> some more.
                Finally it seems you've looked everywhere. Emma isn't here. "; 
                lexicalParent.markAsSearched(); 
            }
        ]
    }
    
    /* 
     *   The list of Tiana's search messages. She takes a LONG time to search
     *   the bushes - perhaps six turns?
     */
    tianaSearchMsgList: EventList
    {
        [
            '{The tiana/she} looks behind one of the bushes by the shed.',
            '{The tiana/she} pushes aside a branch to search in another
            bush.',
            '{The tiana/she} gets down on her hands and knees and looks
            under a bush.',
            '{The tiana/she} wriggles quickly out of the bush and scrambles
            to her feet.',
            '{The tiana/she} wanders around a few bushes, brushing the dirt
            off her knees. ',
            new function() { "{The tiana/she} messes about in the bushes
                a bit more. "; lexicalParent.markAsSearched(); }
        ]
    }
    
    /* 
     *   The list of Yvonne's search messages. Yvonne searches much faster 
     *   than either the PC or Tiana. Maybe she already knows the best hiding
     *   places around here - she is Emma's best friend, after all.
     */
    yvonneSearchMsgList: EventList
    {
        [
            'Yvonne looks behind one of the bushes by the shed.',
            'Yvonne pushes aside a couple of branches and looks into another
            bush.',
            new function() { "Yvonne searches under another bush."; 
            lexicalParent.markAsSearched(); }
        ]
    }
    
    /* Note that the current actor has searched this hiding spot. */
    markAsSearched()
    {
        searchedByList += gActor;
    }
;


lineOfRocks: CustomImmovable
    'squiggly line/rock/rocks'
    'line of rocks'
    @lotsaBushes
    "The rocks are half dug into the ground. Better be careful you don't trip
    over them. They're different shapes, but all about the same colour and 
    size --- small enough that you could maybe pick them up, big enough 
    to <i>really</i> hurt if you dropped one on your foot. "
    
    cannotTakeMsg = 'You\'d better not mess up Emma\'s family\'s garden. 
        Anyway, the rocks look pretty heavy --- you might hurt yourself if 
        you tried playing around with them. '
    
    uselessToAttackMsg = 'You\'d only hurt yourself. '
;



/* --- underTree --- */

bigTree: Fixture, HidingSpot
    'pretty old big tree thick lowest tree/leaves/branches/branch/trunk/base'
    'big tree'
    @underTree
    "The tree is big and it has lots of leaves, but the lowest branches are 
    well within reach. Someone could hide in it. "
    
    /* 
     *   If the PC is not at the top of the tree, the player most likely 
     *   means to do something to the base of the tree.
     */
    vocabLikelihood = (!gActor.isIn(upTree) ? inherited() : -50)
    
    /* remap CLIMB TREE to UP */
    dobjFor(Climb)
    {
        verify()
        {
            if (gActor.canTouch(self))
                logicalRank(150, 'is a tree');
        }
        
        /* 
         *   Only remap the action if the actor is close enough to touch the 
         *   tree.
         */
        remap()
        {
            if (gActor.canTouch(self))
                return [UpAction];
            else
                return nil;
        }
    }
    
    /* can't climb down the tree when already down */
    cannotClimbMsg = 'You\'re already at the bottom of the tree. '
    
    /* Sliding down the tree is the same as climbing down it. */
    dobjFor(SlideDown) asDobjFor(ClimbDown)
    
    /* Playing in the tree is the same as climbing it. */
    dobjFor(PlayOn) asDobjFor(Climb)
    
    /* remap HIDE IN TREE to UP */
    dobjFor(HideIn)
    {
        remap = (remapDobjClimb)
    }
    
    pcSearchMsg = 'You walk around the tree, peering up into the branches.
        The leaves are pretty thick, but you\'re sure you\'d see anyone who was
        hiding there. Guess Emma isn\'t here. '
    npcSearchMsg = '{The actor/she} stares up into the big tree. '
    
    /* 
     *   Can't sit, stand or lie on the tree. In retrospect, making 
     *   HidingSpot a sub-class of Booth was probably a bad idea.
     */
    dobjFor(SitOn)
    {
        verify() { inherited Thing(); }
    }
    
    dobjFor(StandOn)
    {
        verify() { inherited Thing(); }
    }
    
    dobjFor(LieOn)
    {
        verify() { inherited Thing(); }
    }
    
    dobjFor(Hug)
    {
        verify() { }
        
        action()
        {
            "Your arms don't even reach halfway around the trunk. ";
        }
    }
;



lumpyGround: defaultGround
    'lumpy (tree) ground/roots/lump/lumps/bump/bumps'
    'lumpy ground'
    @underTree
    "The ground here is all lumpy from the tree roots, and you have to be 
    careful not to trip over. "
    
    feelDesc = "The ground is rough, full of lumps and bumps. "
;



swing: Chair, CustomImmovable, PlayThing
    'bright red bendy plastic chains/chain/seat/stuff/swing'
    'swing'
    @underTree
    "The swing isn't just a bit of wood hung on ropes, it's like something 
    from a proper playground. The seat is bright red bendy plastic stuff,
    and the chains are covered in plastic stuff too so you can't pinch your
    fingers.
    <.p>The only thing that's wrong is that the branch it's attached to 
    is too low. You could just about reach it if you stood on the swing. "
    
    /* Climbing - remap to standing on the swing */
    dobjFor(Climb) asDobjFor(StandOn)
    
    /* 
     *   Climbing down the swing is the same as climbing down the tree (if 
     *   up the tree) or just getting off the swing (if on the swing).
     */
    dobjFor(ClimbDown)
    {
        verify() 
        {
            if (gActor.isIn(self))
                logicalRank(150, 'getting off the swing');
        }
        
        /* 
         *   Don't require the PC to be able to touch the swing in order to 
         *   climb down it when up the tree.
         */
        preCond()
        {
            if (gActor.isIn(upTree))
                return inherited - touchObj;
            else
                return inherited;
        }
        
        check()
        {
            if (!gActor.isIn(upTree) && !gActor.isIn(self))
                failCheck('You\'re already on the ground. ');
        }
        
        action()
        {
            /* If up the tree, remap to climbing down the tree. */
            if (gActor.isIn(upTree))
                replaceAction(ClimbDown, topOfTree);
            
            /* If in the swing, remap to getting off the swing. */
            if (gActor.isIn(self))
                replaceAction(GetOffOf, self);
        }
    }
    
    /* Custom message for standing/sitting on the swing */
    roomOkayPostureChangeMsg(posture, obj)
    {
        gMessageParams(obj);
        if (posture == standing)
            return ('You clamber up onto the swing, clinging to the 
                chains. From here, you can just reach the lowest 
                branch. ');
        else if (posture == sitting)
            return ('You settle into the swing. Digging your toe into the 
                ground, you push it back and forth a little. The chains
                creak faintly under your weight. ');
        else
            return inherited(posture, obj);
    }
    
    /* Swinging */
    dobjFor(SwingOn)
    {
        verify() { }
        
        action()
        {
            if (gActor.posture == sitting)
                "You kick off and launch the swing into the air. The wind tangles
                your hair across your face as you fly higher and higher. The
                short chains don't let you swing too high, though, so after 
                a minute or so you dig your heels into the ground and stop the
                swing. ";
            else
                /* standing */
                "Standing on the swing, you lean back, then forwards, pushing 
                the swing into motion. You can't get it to swing very high,
                though; the chains are too short. ";
        }
    }
    
    /* Playing on the swing is the same as swinging on it. */
    dobjFor(PlayOn) asDobjFor(SwingOn)
    
    /* The response to asking Tiana to play on the swing with you. */
    playWithTiana = "<q>Hey, <<tiana.isProperName ? 'Tiana, ' : ''>> do you 
        want to play on the swing with me? You can go first and I'll push 
        you, and then you can push me?</q><.p>
        {The tiana/she} hesitates. <q>That sounds like fun,</q> she admits,
        <q>but what about Emma's game?</q><.p>
        <q>Who cares about Emma's game? She never asked us what we wanted to
        play.</q><.p>
        {The tiana/she} grins suddenly. <q>Okay,</q> she says. <q>Let's 
        go!</q><.p>
        You push Tiana <<tiana.learnNameInPassing>> on the swing first, and
        then she gets off and pushes you on the swing. Eventually Emma and
        Yvonne come to see what you're doing, but you ignore them and they
        go away again. By the time you have to go home, you've been invited to
        Tiana's house next weekend. "
    
    /* Responses to pushing and pulling the swing. */
    cannotTakeMsg = 'Your touch swings it back and forth, but eventually it 
        comes to a stop again. '
;


/* --- upTree --- */

topOfTree: Fixture
    'thick big pretty old tree 
    top/treetop/tree-top/branch/branches/tree/leaves/leaf/fork'
    'big tree'
    @upTree
    "The tree would make a good hiding place, actually. The leaves are so thick,
    you're pretty sure no one could see you unless they looked really hard. "
    
    /* 
     *   If the player is up the tree, she most likely means to do something 
     *   to the top of the tree.
     */
    vocabLikelihood = (gActor.isIn(upTree) ? inherited() : -50)
    
    /* Reject attempts to climb the tree while already up the tree. */
    dobjFor(Climb)
    {
        verify() { }
        check()
        {
            if (gActor.canTouch(self))
                failCheck('You\'ve climbed as high as you can go. ');
        }
    }
    
    /* Allow climbing down the tree */
    dobjFor(ClimbDown)
    {
        verify()
        {
            if (gActor.isIn(upTree))
                logicalRank(150, 'up the tree');
        }
        
        remap = [DownAction]
    }
    
    /* Sliding down the tree is the same as climbing down it. */
    dobjFor(SlideDown) asDobjFor(ClimbDown)
    
    /* Getting off/out of the tree is the same as climbing down it. */
    dobjFor(GetOutOf) maybeRemapTo(gActor.isIn(upTree), ClimbDown, self)
    dobjFor(GetOffOf) maybeRemapTo(gActor.isIn(upTree), ClimbDown, self)
    
    /* Allow the PC to hide in the tree-top if playing Hide and Seek. */
    dobjFor(HideIn)
    {
        verify() { }
        
        check()
        {
            delegated HidingSpot();
            
            if (!gActor.isPlayingHnS)
                failCheck('You\'re already hidden in the tree. ');
        }
        
        action()
        {
            delegated HidingSpot();
        }
    }
    
    /* 
     *   Just ignore attempts to enter the top of the tree - the PC is 
     *   already there.
     */
    performEntry(posture) { }
    
    /* Be polite about attempts to search the top of the tree. */
    notAHidingSpotMsg = 'Now you\'ve climbed the tree, it\'s clear 
        there\'s no one hiding here. '
    
    /* 
     *   Treat searching the top of the tree the same way as searching the 
     *   base of the tree when the PC is on the ground.
     */
    dobjFor(Search) maybeRemapTo(!gActor.isIn(upTree), Search, bigTree)
    
    dobjFor(Take)
    {
        verify() { }
        
        check()
        {
            failCheck('You\'re already holding onto the tree as tight as you
                can. ');
        }
    }
    
    dobjFor(Hug)
    {
        verify() { }
        remap = [TakeAction, self]
    }
;


+ BeforeAction
    actor = gPlayerChar
    action = JumpAction
    condition = (gPlayerChar.isIn(upTree))
    
    stopAction = true
    
    react()
    {
        "You'd probably break your leg, if not your neck. ";
    }
;



nest: CustomImmovable, Container
    'bird\'s nest'
    'nest'
    @upTree
    "The nest is almost perfectly round, neatly woven from twigs and grass.
    It's hard to believe a bird could make something so complicated; you 
    could never manage it.<.p>
    A baby bird squats inside the nest, staring up at you. "
    
    /* 
     *   When the baby bird is in the nest, its specialDesc mentions the 
     *   nest, so don't bother mentioning the nest on its own.
     */
    useSpecialDesc = nil
    isListedInContents = nil
    
    /* Can't see it from a distance */
    sightSize = small
    
    /* 
     *   Custom message for searching the nest, on the grounds that the 
     *   player is hardly going to expect to find Emma in it.
     */
    notAHidingSpotMsg = 'The baby bird peers out of the nest at you. '
    
    /* Leave the poor bird alone. */
    cannotTakeMsg = 'If you move the nest you might frighten the baby bird. '
;


babyBird: Actor, CustomImmovable
    'baby bird'
    'baby bird'
    @nest
    "The baby is pinkish grey and wrinkled and almost featherless, like a 
    shrunken old man. Its tiny black eyes sparkle as it turns its head to 
    look at you. "
    
    specialDesc = "A nest snuggles in the fork of a branch, and a baby bird 
        peers over the edge at you. "
    
    /* Can't see it from a distance */
    sightSize = small
    
    /* Descriptions for the various senses */
    soundDesc = "The baby bird gives a high, piercing cheep, opening its
        beak wide and looking meaningfully at you. "
    feelDesc = "You stroke the bird gently with one finger. Its skin
        feels thin and fragile, with slight prickles where its tiny
        feathers are growing, and its heartbeat flutters against your
        fingers. The bird flinches away from your hand and stares
        suspiciously at you. "
    smellDesc = "The baby bird smells sharp and earthy, like rain in
        summer. "
    tasteDesc = "You lick the baby bird's head experimentally. It tastes 
        slightly salty. "
    
    /* Leave the poor bird alone. */
    cannotTakeMsg = 'If you move the baby bird, its parents might not be able to 
        find it again. '
    
    dobjFor(Kiss)
    {
        /* Custom message for kissing the bird. */
        action()
        {
            "You kiss the baby bird softly on top of its head. Its tiny feathers
            prickle against your skin. ";
        }
    }
    
    dobjFor(Hug) asDobjFor(Take)
    
    dobjFor(PlayWith)
    {
        remap = nil
        
        check()
        {
            failCheck(&cannotTakeMsg);
        }
    }
    
    /* Don't hurt the poor bird! */
    uselessToAttackMsg = 'That would be extremely cruel and totally pointless. '
    dobjFor(Break) asDobjFor(Attack)
;


+ DefaultSayAskTellTopic
    "The baby bird stares at you. "
;

+ HelloTopic
    "The baby bird stares at you. "
;



neighbourhoodNoises: SimpleNoise
    'neighbourhood neighborhood dog bark barking car slowing somebody laugh 
    laughter laughing people going (about) everyday every-day business
    gentle noise/noises/sound/sounds/breeze'
    'neighbourhood noises'
    @upTree
    desc = "The breeze carries with it the faint sounds of people going about 
        their everyday business. "
    
    isPlural = true
    
    notWithIntangibleMsg = 'All those noises are way off in the distance. '
;


/* --- frontGarden --- */

cars: CustomImmovable, HidingSpot, CollectiveGroup
    '*car*cars'
    'cars'
    @frontGarden
    "Emma's family's four-wheel-drive and Yvonne's mum's car are parked close to
    the house, with your mum's car and {the's tiana/her} mum's van behind them. 
    Someone could hide between them, maybe. Or under them. Or even in one of 
    them, if it wasn\'t locked. "
    isPlural = true
    
    /* 
     *   It sounds a bit odd to say that the PC hides "in" the cars 
     *   (especially when they're all locked), so change the preposition.
     */
    actorInPrep = 'among'
    actorIntoPrep = 'in between'
    objInPrep = 'among'
    actorOutOfPrep = 'out from among'
    
    pcSearchMsg = 'You look between and underneath all the cars. And just in 
        case, you peer in through the windows --- although they\'re probably 
        all locked anyway. Nope, Emma isn\'t hiding in any of them. '
    /* 
     *   No npcSearchMsg necessary, since Yvonne never comes this way and 
     *   Tiana's search is handled differently.
     */
    
    /* 
     *   When Tiana takes the front-of-house route, she gets to the hiding 
     *   spot a bit too soon. To slow her down, I'll make her search each of 
     *   the cars individually. Poor girl, she can't catch a break.
     */
    dobjFor(Search)
    {
        action()
        {
            if (gActor == tiana)
            {
                tianaSearchMsgList.doScript();
            }
            else
            {
                inherited();
            }
        }
    }
    
    tianaSearchMsgList: EventList
    {
        [
            '{The tiana/she} wanders around the cars, searching between and
            behind them all. ',
            '{The tiana/she} gets down on her hands and knees to look under 
            the cars. ',
            new function {
                "{The tiana/she} stands on her tiptoes and peeks in the window 
                of Emma's parents' four-wheel-drive. ";
                lexicalParent.searchedByList += gActor;
            }
        ]
    }
    
    /* 
     *   All collective actions on the cars should use the cars group object,
     *   not the individual objects.
     */
    isCollectiveAction(action, whichObj)
    {
        return true;
    }
    
    /* Handle attempts to open/unlock the group of cars. */
    cannotLockMsg = 'You don\'t have the keys. '
    cannotUnlockMsg = cannotLockMsg
    cannotOpenMsg = 'The cars are all locked and you don\'t have the keys. '
    cannotCloseMsg = 'The cars are all already closed. '
    
    /* Handle attempts to move the cars. */
    cannotTakeMsg = 'You can\'t drive, and you couldn\'t possibly move them
        by hand. '
    
    /* Custom message for hiding in the cars. */
    roomOkayPostureChangeMsg(posture, obj)
    {
        return 'Lying flat on your tummy, {you/he} wriggle{s} under one of the 
            cars. ';
    }
    
    /* Custom message for getting out of the cars. */
    okayNotStandingOnMsg = 'Okay, {you\'re} no longer hiding {on dobj}. '
;


emmaCar: Car
    'family\'s big shiny green brand new 
    4 four wheel (drive)/four-wheel-drive/4-wheel-drive/4x4/4wd/car*cars'
    'Emma\'s family\'s four-wheel-drive'
    @frontGarden
    "Emma's family's car is a big shiny green four-wheel-drive. It looks brand
    new. "
    
    owner = emma
;


yvonneCar: Car
    '(mum\'s) ordinary white car*cars'
    'Yvonne\'s mum\'s car'
    @frontGarden
    "Yvonne's mum drives an ordinary white car, a lot like your mum's, only
     a bit newer. "
    
    owner = yvonne
;


yourCar: Car
    'mum\'s plain boring white scratched scratch/scratches/car*cars'
    'your mum\'s car'
    @frontGarden
    "Your mum's car is a plain boring white one. You've had it for a while now
    and it's got some scratches where Mum accidentally scraped the letter box
    while backing out of the driveway. Also it kind of needs washing. "
    
    owner = me
;


tianaCar: Car
    vocabWords = '(mum\'s) old big yellow smiley smily smile back 
        face/van/car/dirt/window*cars'
    name = tiana.theName + '\'s mum\'s van'
    location = frontGarden
    
    desc = "{The's tiana/her} family has a big old yellow van. You don't know
        why --- maybe they have lots of kids or maybe they just like plenty of
        space. Someone has drawn a smiley face in the dirt on the back window. "
    
    owner = tiana
;



frontGardenShrubs: Fixture, HidingSpot
    'thick shiny dark leaf/leaves/shrub/shrubs/bush/bushes'
    'shrubs'
    @frontGarden
    "The shrubs are all in flowerbeds, behind the flowers. They all seem to be the
    same kind --- something with shiny dark leaves. They look thick enough to hide
    in. "
    
    isPlural = true
    
    pcSearchMsg = 'Looking in the shrubs means stepping in the flowerbeds. You\'re
        very careful not to tread on any flowers, and when you\'ve finished you 
        smooth away your footprints so Emma\'s parents won\'t notice anything. It\'s
        a waste of time, though --- Emma isn\'t there. '
    npcSearchMsg = '{The actor/she} picks her way through the flowerbeds to look in
        the shrubs. '
;



flowerBedsFront: CustomImmovable
    'plain neat tidy flower 
    bed/flowerbed/flower-bed/plant/beds/plants/flowerbeds/flowers/flower-beds'
    'flower beds'
    @frontGarden
    "Like the flower beds behind the house, the front garden doesn't have many 
    flowers in it at the moment, just plain plants. But it's very neat and tidy,
    no weeds or anything. "
    
    isPlural = true
    
    smellDesc = "The flowers smell faintly sweet. "
    
    cannotTakeMsg = 'Emma\'s parents might not like you picking their plants. '
    
    dobjFor(Search) remapTo(Search, frontGardenShrubs)
;



birdBath: CustomImmovable
    'clean fresh stone (bird) bath/birdbath/bird-bath/water'
    'bird bath'
    @frontGarden
    "The bird bath sits in the exact centre of the lawn. The water looks clean
    and fresh, but there's no sign of any birds having been here. "
    
    cannotTakeMsg = 'The bird bath is much too heavy for you. '
    
    dobjFor(Drink)
    {
        verify() { }
        
        preCond = [touchObj]
        
        action()
        {
            "You cup some of the water in your hands and sip it. It tastes all
            right<<tiana.canSee(gPlayerChar) ? ', but {the tiana/she} looks
                horrified' : ''>>. ";
        }
    }
    
    dobjFor(Taste) asDobjFor(Drink)
    
    cannotSitOnMsg = 'You couldn\'t sit on the bird bath without getting wet. '
;



driveway: defaultGround
    'red brick parking drive way/driveway/drive/area'
    'driveway'
    @frontGarden
    "The red brick driveway runs up from the road and widens into a parking
    area in front of the house. "
;


road: PathPassage, NoTravelMessage
    'empty road'
    'road'
    @frontGarden
    "Emma's family live right out on the edge of town. Hardly any cars pass
    by. "
    
    travelDesc = "Mum always tells you not to play on the road, in case you
        get kidnapped or run over or something. "
;



/* --- vegeGarden --- */


vegePlants: CustomImmovable
    'various tidy furry little tall tomato vegetable vege vegie
    stem/plants/rows/kinds/bushes/herbs/plant/row/kind/herb/vegetables/vegetable/vege/veges/vegie/vegies/tomatos/tomatoes'
    'vegetable plants'
    @vegeGarden
    "The tall plants with a sort of furry stem look like tomato bushes, and the 
    little plants on the border of the garden could be herbs, maybe? "
    isPlural = true
    
    cannotTakeMsg = 'Emma\'s parents probably wouldn\'t be happy if you damaged
        their plants. '
    cannotEatMsg = 'You\'re not going to eat vegetables if you don\'t have to. '
;
