/* Execute.c - code bytecode interpreter */
/*
	Copyright (c) 1993, by David Michael Betz
	All rights reserved
*/

#include "Drool.h"
#include "Objects.h"
#include "Streams.h"
#include "Execute.h"

/* local variables */
int executing;

/* prototypes */
static int getwoperand(void);
void CallFunction(int n);
void CallNext(void);
void Return(void);
static void TypeError(void);
static void CallNextError(void);

/* Execute - execute code */
void Execute(ObjectPtr fcn)
{
    register ObjectPtr p2,p3,p1,*p;
    unsigned int pcoff;
    long n;

    /* make sure this is a method */
    if (!MethodP(fcn))
	TypeError();
    
    /* setup to execute the code */
    ResetStack();
    code = fcn;
    pc = cbase = StringDataAddress(MethodCode(code));
	
    /* the instruction dispatch loop */
    for (executing = true; executing; ) {
	/*info("%04x: %02x",(int)(pc-cbase),*pc);*/
	switch (*pc++) {
	case OP_CALL:
		check(4);
		n = *pc++;
		p1 = sp[n];
		if (ObjectP(p1)) {
		    p2 = sp[n - 1];
		    if (n < 1) error("too few arguments");
		    if ((p3 = GetProperty(p1,p2)) != nil)
			p2 = nilObject;
		    else {
			ObjectPtr c;
			for (c = ObjectSearchList(p1); c != nilObject; c = Cdr(c))
			    if ((p3 = GetSharedProperty(Car(c),p2)) != nil)
				break;
			if (c == nilObject)
			    error("No method for this message");
			p2 = c;
		    }
		    sp[n] = PropertyValue(p3);
		    cpush(p1);
		    cpush(p2);
		    n += 2;
		}
		CallFunction(n);
		break;
	case OP_RETURN:
		Return();
		break;
	case OP_CALLNEXT:
		CallNext();
	 	break;
	case OP_TFRAME:
		check(2);
		push(BoxNumber(*pc++));
		push(BoxNumber(stackTop - fp));
		fp = sp;
		break;
	case OP_TPOP:
		p2 = *sp;
		sp = fp;
		fp = stackTop - UnboxNumber(pop());
		p3 = pop();
		drop(UnboxNumber(p3));
		*sp = p2;
		break;
	case OP_ARG:
		n = *pc++;
		for (p = fp; --n >= 0; )
		    p = stackTop - UnboxNumber(p[0]);
		n = *pc++;
		if (n >= UnboxNumber(p[1]))
		    error("too few arguments");
		*sp = p[n + 2];
		break;
	case OP_ASET:
		n = *pc++;
		for (p = fp; --n >= 0; )
		    p = stackTop - UnboxNumber(p[0]);
		n = *pc++;
		if (n >= UnboxNumber(p[1]))
		    error("too few arguments");
		p[n + 2] = *sp;
		break;
	case OP_BRT:
		pcoff = getwoperand();
		if (*sp != falseObject)
		    pc = cbase + pcoff;
		break;
	case OP_BRF:
		pcoff = getwoperand();
		if (*sp == falseObject)
		    pc = cbase + pcoff;
		break;
	case OP_BR:
		pc = cbase + getwoperand();
		break;
	case OP_T:
		*sp = trueObject;
		break;
	case OP_NIL:
		*sp = nilObject;
		break;
	case OP_PUSH:
		cpush(nilObject);
		break;
	case OP_NOT:
		*sp = *sp == falseObject ? trueObject : falseObject;
		break;
	case OP_ADD:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = BoxNumber(UnboxNumber(*sp) + n);
		break;
	case OP_SUB:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = BoxNumber(UnboxNumber(*sp) - n);
		break;
	case OP_MUL:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = BoxNumber(UnboxNumber(*sp) * n);
		break;
	case OP_DIV:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = BoxNumber(n == 0 ? 0 : UnboxNumber(*sp) / n);
		break;
	case OP_REM:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = BoxNumber(n == 0 ? 0 : UnboxNumber(*sp) % n);
		break;
	case OP_LT:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = UnboxNumber(*sp) < n ? trueObject : falseObject;
		break;
	case OP_LEQ:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = UnboxNumber(*sp) <= n ? trueObject : falseObject;
		break;
	case OP_EQ:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = UnboxNumber(*sp) == n ? trueObject : falseObject;
		break;
	case OP_NEQ:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = UnboxNumber(*sp) != n ? trueObject : falseObject;
		break;
	case OP_GEQ:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = UnboxNumber(*sp) >= n ? trueObject : falseObject;
		break;
	case OP_GT:
		p2 = pop();
		if (!NumberP(p2)) TypeError();
		if (!NumberP(*sp)) TypeError();
		n = UnboxNumber(p2);
		*sp = UnboxNumber(*sp) > n ? trueObject : falseObject;
		break;
	case OP_LIT:
		*sp = VectorElement(MethodLiterals(code),getwoperand());
		break;
	case OP_VAR:
		*sp = SymbolValue(VectorElement(MethodLiterals(code),getwoperand()));
		break;
	case OP_SET:
		SetSymbolValue(VectorElement(MethodLiterals(code),getwoperand()),*sp);
		break;
	case OP_GETP:
		p2 = pop();
		p1 = *sp;
		if (!ObjectP(p1)) TypeError();
		*sp = getp(p1,p2);
		break;
	case OP_SETP:
		p3 = pop();
		p2 = pop();
		p1 = *sp;
		if (!ObjectP(p1)) TypeError();
		*sp = setp(p1,p2,p3);
		break;
	case OP_CONS:
		p2 = pop();
		p1 = *sp;
		*sp = Cons(p1,p2);
		break;
	case OP_CAR:
		p1 = *sp;
		if (!ConsP(p1)) TypeError();
		*sp = Car(p1);
		break;
	case OP_SETCAR:
		p2 = pop();
		p1 = *sp;
		if (!ConsP(p1)) TypeError();
		*sp = SetCar(p1,p2);
		break;
	case OP_CDR:
		p1 = *sp;
		if (!ConsP(p1)) TypeError();
		*sp = Cdr(p1);
		break;
	case OP_SETCDR:
		p2 = pop();
		p1 = *sp;
		if (!ConsP(p1)) TypeError();
		*sp = SetCdr(p1,p2);
		break;
	case OP_VSIZE:
		p1 = *sp;
		if (StringP(p1))
		    *sp = BoxNumber(StringSize(p1));
		else if (VectorP(p1))
		    *sp = BoxNumber(VectorSize(p1));
		else
		    TypeError();
		break;
	case OP_VREF:
		p2 = pop();
		p1 = *sp;
		if (!NumberP(p2)) TypeError();
		if (StringP(p1)) {
		    if ((n = UnboxNumber(p2)) < 0 || n >= StringSize(p1))
			error("index out of bounds");
		    *sp = BoxNumber(StringElement(p1,n));
		}
		else if (VectorP(p1)) {
		    if ((n = UnboxNumber(p2)) < 0 || n >= VectorSize(p1))
			error("index out of bounds");
		    *sp = VectorElement(p1,n);
		}
		else
		    TypeError();
		break;
	case OP_VSET:
		p3 = pop();
		p2 = pop();
		p1 = *sp;
		if (!NumberP(p2)) TypeError();
		if (StringP(p1)) {
		    if (!NumberP(p3)) TypeError();
		    if ((n = UnboxNumber(p2)) < 0 || n >= StringSize(p1))
			error("index out of bounds");
		    *sp = BoxNumber(SetStringElement(p1,n,UnboxNumber(p3)));
		}
		else if (VectorP(p1)) {
		    if ((n = UnboxNumber(p2)) < 0 || n >= VectorSize(p1))
			error("index out of bounds");
		    *sp = SetVectorElement(p1,n,p3);
		}
		else
		    TypeError();
		break;
	case OP_NUMBERP:
		*sp = NumberP(*sp) ? trueObject : falseObject;
		break;
	case OP_STRINGP:
		*sp = StringP(*sp) ? trueObject : falseObject;
		break;
	case OP_SYMBOLP:
		*sp = SymbolP(*sp) ? trueObject : falseObject;
		break;
	case OP_CONSP:
		*sp = ConsP(*sp) ? trueObject : falseObject;
		break;
	case OP_VECTORP:
		*sp = VectorP(*sp) ? trueObject : falseObject;
		break;
	case OP_OBJECTP:
		*sp = ObjectP(*sp) ? trueObject : falseObject;
		break;
	case OP_METHODP:
		*sp = MethodP(*sp) ? trueObject : falseObject;
		break;
	case OP_CMETHODP:
		*sp = CMethodP(*sp) ? trueObject : falseObject;
		break;
	case OP_PACKAGEP:
		*sp = PackageP(*sp) ? trueObject : falseObject;
		break;
	case OP_NULLP:
		*sp = *sp == nilObject ? trueObject : falseObject;
		break;
	default:
		error("bad opcode: %02x",pc[-1]);
		break;
	    }
	}
}

/* getwoperand - get data word */
static int getwoperand(void)
{
    int data;
    data = *pc++;
    data = (*pc++ << 8) | data;
    return (data);
}

void CallFunction(int n)
{
    push(BoxNumber(n));
    push(BoxNumber(stackTop - fp));
    fp = sp;
    push(BoxNumber(pc - cbase));
    push(code);
    code = fp[n + 2];
    if (MethodP(code))
	cbase = pc = StringDataAddress(MethodCode(code));
    else if (CMethodP(code)) {
	cpush(nilObject);
	CMethodHandler(code)();
	Return();
    }
    else
	TypeError();
}

void CallNext(void)
{
    ObjectPtr obj,tag,prop,c,*ap;
    long n,ac;
    n = UnboxNumber(fp[1]);
    if (n < 3)
	CallNextError();
    obj = fp[3];
    tag = fp[n + 1];
    if (!ObjectP(obj))
	CallNextError();
    if ((c = fp[2]) == nilObject)
	c = ObjectSearchList(obj);
    else
	c = Cdr(c);
    for (; c != nilObject; c = Cdr(c))
	if ((prop = GetSharedProperty(Car(c),tag)) != nil)
	    break;
    if (c == nilObject)
	*sp = nilObject;
    else {
	cpush(PropertyValue(prop));
	cpush(tag);
	for (ap = &fp[n], ac = n - 3; --ac >= 0; --ap)
	    cpush(*ap);
	cpush(obj);
	cpush(c);
	CallFunction(n);
    }
}

void Return(void)
{
    ObjectPtr p2,p3;
    if (fp == stackTop)
	executing = false;
    else {
	p2 = *sp;
	code = fp[-2];
	cbase = StringDataAddress(MethodCode(code));
	pc = cbase + UnboxNumber(fp[-1]);
	sp = fp;
	fp = stackTop - UnboxNumber(pop());
	p3 = pop();
	drop(UnboxNumber(p3));
	*sp = p2;
    }
}

static void TypeError(void)
{
    error("wrong type");
}

static void CallNextError(void)
{
    error("call-next-method must be called from within a method");
}

/* getp - get the value of an object property */
ObjectPtr getp(ObjectPtr obj,ObjectPtr tag)
{
    ObjectPtr p;
    if ((p = GetProperty(obj,tag)) != nil)
	return PropertyValue(p);
    else if ((p = GetInheritedProperty(obj,tag)) != nil)
	return PropertyValue(p);
    return nilObject;
}

/* setp - set the value of an object property */
ObjectPtr setp(ObjectPtr obj,ObjectPtr tag,ObjectPtr val)
{
    ObjectPtr p;
    if ((p = GetProperty(obj,tag)) != nil)
	SetPropertyValue(p,val);
    else if ((p = GetInheritedProperty(obj,tag)) != nil)
	SetPropertyValue(p,val);
    else
	error("no matching property");
    return val;
}

/* PrintValue - print a value */
void PrintValue(ObjectPtr val)
{
    if (StringP(val)) {
	unsigned char *p = StringDataAddress(val);
	long size = StringSize(val);
	while (--size >= 0)
	    TranscriptPutC(*p++);
    }
    else {
	Str255 buf; int i;
	ObjectPrint(val,buf);
	for (i = 1; i <= buf[0]; ++i)
	    TranscriptPutC(buf[i]);
    }
}
