/*
// Abstract:
//	INSTRUCT---Unassemble Instructions
//
//	The Unassemble Instructions module unassembles the Saturn
//	instruction set.
//
// Author:
//	Derek S. Nickel
//
// Creation date;
//	13 October 1990
//
// History:
// V01-002	Derek S. Nickel		18-FEB-1991
//	Disassemble remaining 'unknown' instructions.
//
// V01-001	Derek S. Nickel		13-OCT-1991
//	Modified version of Pete M. Wilson's TAUA.C
//
*/

#include <string.h>
#include <ctype.h>

#include "modes.h"
#include "pager.h"
#include "memory.h"
#include "index.h"
#include "instruct.h"

typedef char *(*PcpFc)(char);
typedef char *(*PcpFv)(void);
typedef PcpFc RDA[16];
typedef PcpFv RDAv[16];

static char instrBuf[33], *ibp;
static char disBuf[240];
static char goyes_buf[240];
PcpFv *SecondNibble[16];
static int StopAns, Follow;
static char unknownInstr[] = "??????";

static char *q03and8B[4] = {
	"AB", "BC", "CA", "DC"
};

static char *q47andCF[4] = {
	"BA", "CB", "AC", "CD"
};

static char *field[] = {
	"P", "WP", "XS", "X", "S", "M", "B", "W",
	NULL, NULL, NULL, NULL, NULL, NULL, NULL, "A"
};

static char *load_reg_with_data(char *iname);
static void testWithNibble(char *typ);
static char *ua808(void);
static char *ua8081(void);
static char *ua818(void);
static char *ua819(void);
static char *ua81A(void);
static char *ua81B(void);

/***********************************************************************
	
***********************************************************************/

static void follow(bin5_t new_addr)
{
	if (modes.follow) {
		SetWorkPtr(new_addr);
		Follow = 1;
	} else {
		StopAns = 1;
	}
}

/***********************************************************************
	
***********************************************************************/

static char *uaSN(char firstNib)
{
	char secondNib;

	*ibp++ = (secondNib = GetNibble());

	return (SecondNibble[hexval(firstNib)][hexval(secondNib)])();
}

/***********************************************************************
	
***********************************************************************/

static char *ua0(char dummy)
{
	static char *imap[] = {
		"RTNSXM",
		"RTN   ",
		"RTNSC ",
		"RTNCC ",
		"SETHEX",
		"SETDEC",
		"RSTK=C",
		"C=RSTK",
		"CLRST ",
		"C=ST  ",
		"ST=C  ",
		"CSTEX ",
		"P=P+1 ",
		"P=P-1 ",
		NULL,
		"RTI   "
	};

	char ch = (*ibp++ = GetNibble());
	int secondNibble = hexval(ch);

	if (secondNibble == 0xE) {
		char ch;
		int f, q;

		*ibp++ = (ch = GetNibble());
		f = hexval(ch);
		*ibp++ = (ch = GetNibble());
		q = hexval(ch);

		if (f > 0x7 && f < 0xF)
			return unknownInstr;

		switch (q/4) {
		  case 0:
			sprintf(disBuf, "%c=%c&%c  %s",
				q03and8B[q][0],
				q03and8B[q][0],
				q03and8B[q][1],
				field[f]);
			break;

		  case 1:
			sprintf(disBuf, "%c=%c&%c  %s",
				q47andCF[q-4][0],
				q47andCF[q-4][0],
				q47andCF[q-4][1],
				field[f]);
			break;

		  case 2:
			sprintf(disBuf, "%c=%c!%c  %s",
				q03and8B[q-8][0],
				q03and8B[q-8][0],
				q03and8B[q-8][1],
				field[f]);
			break;

		  case 3:
			sprintf(disBuf, "%c=%c!%c  %s",
				q47andCF[q-12][0],
				q47andCF[q-12][0],
				q47andCF[q-12][1],
				field[f]);
			break;
		}

		return disBuf;
	}
	else {
		StopAns = (secondNibble < 4);
		return imap[secondNibble];
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua10(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNib = hexval(ch);

	if (0 <= thirdNib && thirdNib <= 4)
		sprintf(disBuf, "R%d=A  ", thirdNib);
	else if (8 <= thirdNib && thirdNib <= 12)
		sprintf(disBuf, "R%d=C  ", thirdNib-8);
	else
		return unknownInstr;

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua11(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNib = hexval(ch);

	if (0 <= thirdNib && thirdNib <= 4)
		sprintf(disBuf, "A=R%d  ", thirdNib);
	else if (8 <= thirdNib && thirdNib <= 12)
		sprintf(disBuf, "C=R%d  ", thirdNib-8);
	else
		return unknownInstr;

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua12(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNib = hexval(ch);

	if (0 <= thirdNib && thirdNib <= 4)
		sprintf(disBuf, "AR%dEX ", thirdNib);
	else if (8 <= thirdNib && thirdNib <= 12)
		sprintf(disBuf, "CR%dEX ", thirdNib-8);
	else
		return unknownInstr;

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua13(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNib = hexval(ch);
	static char *dmap[] =  { "D0", "D1" };
	static char *rmap[] = { "A", "C" };
	static char *imap[] = {
		"%s=%s  ",
		"%s%sEX ",
		"%s=%s  ",
		"%s%sEX ",
		"%s=%sS ",
		"%s%sXS ",
		"%s=%sS ",
		"%s%sXS "
	};

	if (!(thirdNib & 0x2)) {
		sprintf(disBuf, imap[thirdNib/2], dmap[thirdNib & 0x1],
			rmap[(thirdNib & 0x4) /4]);
	}
	else {
		sprintf(disBuf, imap[thirdNib/2], rmap[(thirdNib & 0x4) /4],
			dmap[thirdNib & 0x1]);
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua14(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);
	char regch = "AC"[(thirdNibble & 0x04) / 4],
		fieldch = "AB"[thirdNibble / 8],
		datch = (char) ('0'+(thirdNibble & 0x01));

	if (thirdNibble & 0x02)
		sprintf(disBuf, "%c=DAT%c %c", regch, datch, fieldch);
	else
		sprintf(disBuf, "DAT%c=%c %c", datch, regch, fieldch);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua15(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNib = hexval(ch), f;

	*ibp++ = (ch = GetNibble());
	f = hexval(ch);

	if (thirdNib < 8) {
		static char *map[] = {
			"DAT%d=A %s",
			"A=DAT%d %s",
			"DAT%d=C %s",
			"C=DAT%d %s"
		};

		if (f < 8)
			sprintf(disBuf, map[thirdNib/2], thirdNib & 0x1,
				field[f]);
		else
			return unknownInstr;
	} else {
		static char *map[] = {
			"DAT%d=A %d",
			"A=DAT%d %d",
			"DAT%d=C %d",
			"C=DAT%d %d"
		};

	    sprintf(disBuf, map[(thirdNib-8)/2], thirdNib & 0x1, f+1);
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *adjDat(char reg, char op)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);

	sprintf(disBuf, "D%c=D%c%c %d", reg, reg, op, thirdNibble+1);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua16(void)
{
	return adjDat('0', '+');
}

/***********************************************************************
	
***********************************************************************/

static char *ua17(void)
{
	return adjDat('1', '+');
}

/***********************************************************************
	
***********************************************************************/

static char *ua18(void)
{
	return adjDat('0', '-');
}

/***********************************************************************
	
***********************************************************************/

static void ua19_F(char secondNib, int whichD)
{
	char dBuf[6];
	int n = 0;

	switch (secondNib) {
	    case '9':
	    case 'D':
		n = 2;
		break;

	    case 'A':
	    case 'E':
		n = 4;
		break;

	    case 'B':
	    case 'F':
		n = 5;
		break;
	}

	GetNNibbles(dBuf, n);
	strcpy(ibp, dBuf);
	ibp += n;
	strrev(dBuf);

	sprintf(disBuf, "D%d=HEX %s", whichD, dBuf);

	if (n == 5) {
		char *acom = get_address_comment(str2adr(dBuf, 0));
		if (*acom) sprintf(disBuf, "%-20s  ; %s", disBuf, acom);
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua19_A_B(void)
{
	ua19_F(instrBuf[1], 0);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua1C(void)
{
	return adjDat('1', '-');
}

/***********************************************************************
	
***********************************************************************/

static char *ua1D_E_F(void)
{
	ua19_F(instrBuf[1], 1);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua2(char dummy)
{
	char ch = (*ibp++ = GetNibble());
	int secondNib = hexval(ch);

	sprintf(disBuf, "P=     %d", secondNib);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua3(char dummy)
{
	return load_reg_with_data("LCHEX ");
}

/***********************************************************************
	
***********************************************************************/

static char *load_reg_with_data(char *iname)
{
	char ch = (*ibp++ = GetNibble()), *wp, ascii[9], buf[17];
	int ct = hexval(ch)+1, j1;

	wp = ibp;
	for (j1 = 0; j1 < ct; ++j1)
		*ibp++ = GetNibble();
	*ibp = '\0';
	strcpy(buf, wp);
	strrev(buf);

	if (ct & 0x1) {
		sprintf(disBuf, "%s %s", iname, buf);
		if (ct == 5) {
			char *acom = get_address_comment(str2adr(buf, 0));

			if (*acom)
				sprintf(disBuf, "%-20s  ; %s", disBuf, acom);
		}
	} else {
		/* Even number of nibbles */
		for (j1 = 0; j1 < ct/2; ++j1) {
			ascii[j1] = (hexval(wp[2*j1+1]) << 4) + hexval(wp[2*j1]);
			if (!isprint(ascii[j1]) || ascii[j1] < ' ')
				ascii[j1] = '.';
		}
		ascii[ct/2] = '\0';

		sprintf(disBuf, "%s %s", iname, buf);
		sprintf(disBuf, "%-20s  ! \"%s\"", disBuf, ascii);
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static int getOffset(int n, char *dBuf, int adjust)
{
	int ofs;

	GetNNibbles(dBuf, n);
	strcpy(ibp, dBuf);
	ibp += n;
	strrev(dBuf);

	ofs = (int) str2adr(dBuf, 0);

	if (ofs > (0x8 << 4*(n-1))-1)
		ofs -= (0x8 << (4*(n-1)+1));

	ofs -= adjust;

	return ofs;
}

/***********************************************************************
	
***********************************************************************/

static void goTarget(char *typ)
{
	char dBuf[3];
	int ofs;
	char *ccom;
	int bet = 0;

	ccom = get_code_comment(WorkPtr-1);
	if (strcmpi(ccom,"BET") == 0) bet = 1;

	ofs = getOffset(2, dBuf, 0);

	if (ofs == 0) {
		sprintf(disBuf, "RTN%s ", typ);
		if (bet) StopAns = 1;

	} else if (ofs == 2 && *instrBuf == '4') {
		sprintf(disBuf, "NOP3  ");

	} else {
		bin5_t target = WorkPtr+ofs-2;
		char *acom = get_address_comment(target);

		sprintf(disBuf, "GO%s   %05lX", typ, target);

		if (*acom)
			sprintf(disBuf, "%-20s  ; %s", disBuf, acom);

		if (bet) follow(target);
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua4(char dummy)
{
	goTarget("C ");

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua5(char dummy)
{
	goTarget("NC");

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static bin5_t offsetTarget(char *instr, int n, int adjust)
{
	char dBuf[5], *acom;
	int ofs = getOffset(n, dBuf, adjust);
	bin5_t target = WorkPtr+ofs;

	sprintf(disBuf, "%s %05lX", instr, target);

	acom = get_address_comment(target);
	if (*acom)
		sprintf(disBuf, "%-20s  ; %s", disBuf, acom);

	return target;
}

/***********************************************************************
	
***********************************************************************/

static char *ua6(char dummy)
{
	/*
	// 6300 is NOP4
	// 64000 is NOP5
	*/

	follow(offsetTarget("GOTO  ", 3, 3));

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua7(char dummy)
{
	(void) offsetTarget("GOSUB ", 3, 0);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static void offsetPTarget(char *instr)
{
	char dBuf[5];
	int ofs = getOffset(2, dBuf, 0);

	sprintf(disBuf, "%s", instr);

	if (ofs == 0)
		sprintf(goyes_buf, "\t\t  RTNYES");
	else {
		bin5_t target = WorkPtr+ofs-2;
		char *acom;

		sprintf(goyes_buf, "\t\t  GOYES  %05lX", target);

		acom = get_address_comment(target);
		if (*acom)
			sprintf(goyes_buf, "%s\t\t; %s", goyes_buf, acom);

	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua9(char dummy)
{
	static char instrBuf[8];
	char ch;
	int f, q;

	*ibp++ = (ch = GetNibble());
	f = hexval(ch);
	*ibp++ = (ch = GetNibble());
	q = hexval(ch);

	if (f <= 7)
		switch (q/4) {
		  case 0:
			sprintf(instrBuf, "?%c=%c   %s",
				q03and8B[q][0], q03and8B[q][1], field[f]);
			break;
		  case 1:
			sprintf(instrBuf, "?%c#%c   %s",
				q47andCF[q-4][0], q47andCF[q-4][1], field[f]);
			break;
		  case 2:
			sprintf(instrBuf, "?%c=0   %s", 'A'+q-8, field[f]);
			break;
		  case 3:
			sprintf(instrBuf, "?%c#0   %s", 'A'+q-12, field[f]);
			break;
		}
	else
		switch (q/4) {
		  case 0:
			sprintf(instrBuf, "?%c>%c  %s", q03and8B[q][0], q03and8B[q][1],
				field[f-8]);
			break;
		  case 1:
			sprintf(instrBuf, "?%c>%c   %s", q47andCF[q-4][0], q47andCF[q-4][1],
				field[f-8]);
			break;
		  case 2:
			sprintf(instrBuf, "?%c>=%c  %s", q03and8B[q-8][0], q03and8B[q-8][1],
				field[f-8]);
			break;
		  case 3:
			sprintf(instrBuf, "?%c>=%c  %s", q47andCF[q-12][0],
				q47andCF[q-12][1], field[f-8]);
			break;
		}

	offsetPTarget(instrBuf);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *uaA(char dummy)
{
	char ch;
	int f, q;

	*ibp++ = (ch = GetNibble());
	f = hexval(ch);
	*ibp++ = (ch = GetNibble());
	q = hexval(ch);

	if (f <= 7)
		switch (q/4) {
		  case 0:
			sprintf(disBuf, "%c=%c+%c  %s", q03and8B[q][0], q03and8B[q][0],
				q03and8B[q][1], field[f]);
			break;
		  case 1:
			sprintf(disBuf, "%c=%c+%c  %s", 'A'+q-4, 'A'+q-4,
				'A'+q-4, field[f]);
			break;
		  case 2:
			sprintf(disBuf, "%c=%c+%c  %s", q03and8B[q-8][1],
				q03and8B[q-8][1],
				q03and8B[q-8][0], field[f]);
			break;
		  case 3:
			sprintf(disBuf, "%c=%c-1  %s", 'A'+q-12, 'A'+q-12, field[f]);
			break;
		}
	else
		switch (q/4) {
		  case 0:
			sprintf(disBuf, "%c=0    %s", 'A'+q, field[f-8]);
			break;
		  case 1:
			sprintf(disBuf, "%c=%c    %s", q47andCF[q-4][1], q47andCF[q-4][0],
				field[f-8]);
			break;
		  case 2:
			sprintf(disBuf, "%c=%c    %s", q03and8B[q-8][1], q03and8B[q-8][0],
				field[f-8]);
			break;
		  case 3:
			sprintf(disBuf, "%c%cEX   %s", q47andCF[q-12][0], q47andCF[q-12][1],
				field[f-8]);
			break;
		}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *uaB(char dummy)
{
	char ch;
	int f, q;

	*ibp++ = (ch = GetNibble());
	f = hexval(ch);
	*ibp++ = (ch = GetNibble());
	q = hexval(ch);

	if (f <= 7)
		switch (q/4) {
		  case 0:
			sprintf(disBuf, "%c=%c-%c  %s", q03and8B[q][0], q03and8B[q][0],
				q03and8B[q][1], field[f]);
			break;
		  case 1:
			sprintf(disBuf, "%c=%c+1  %s", 'A'+q-4, 'A'+q-4,
				field[f]);
			break;
		  case 2:
			sprintf(disBuf, "%c=%c-%c  %s", q03and8B[q-8][1], q03and8B[q-8][1],
				q03and8B[q-8][0], field[f]);
			break;
		  case 3:
			sprintf(disBuf, "%c=%c-%c  %s", q47andCF[q-12][1],
				q47andCF[q-12][0], q47andCF[q-12][1], field[f]);
			break;
		}
	else
		switch (q/4) {
		  case 0:
			sprintf(disBuf, "%cSL    %s", 'A'+q, field[f-8]);
			break;
		  case 1:
			sprintf(disBuf, "%cSR    %s", 'A'+q-4, field[f-8]);
			break;
		  case 2:
			sprintf(disBuf, "%c=-%c   %s", 'A'+q-8, 'A'+q-8,
				field[f-8]);
			break;
		  case 3:
			sprintf(disBuf, "%c=-%c-1 %s", 'A'+q-12, 'A'+q-12,
				field[f-8]);
			break;
		}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *uaC(char dummy)
{
	char ch;
	int q;

	*ibp++ = (ch = GetNibble());
	q = hexval(ch);

	switch (q/4) {
	  case 0:
		sprintf(disBuf, "%c=%c+%c  A", q03and8B[q][0], q03and8B[q][0],
			q03and8B[q][1]);
		break;
	  case 1:
		sprintf(disBuf, "%c=%c+%c  A", 'A'+q-4, 'A'+q-4, 'A'+q-4);
		break;
	  case 2:
		sprintf(disBuf, "%c=%c+%c  A", q03and8B[q-8][1], q03and8B[q-8][1],
			q03and8B[q-8][0]);
		break;
	  case 3:
		sprintf(disBuf, "%c=%c-1  A", 'A'+q-12, 'A'+q-12);
		break;
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *uaD(char dummy)
{
	char ch;
	int q;

	*ibp++ = (ch = GetNibble());
	q = hexval(ch);

	switch (q/4) {
	  case 0:
		sprintf(disBuf, "%c=0    A", 'A'+q);
		break;
	  case 1:
		sprintf(disBuf, "%c=%c    A", q47andCF[q-4][1], q47andCF[q-4][0]);
		break;
	  case 2:
		sprintf(disBuf, "%c=%c    A", q03and8B[q-8][1], q03and8B[q-8][0]);
		break;
	  case 3:
		sprintf(disBuf, "%c%cEX   A", q47andCF[q-12][0], q47andCF[q-12][1]);
		break;
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *uaE(char dummy)
{
	char ch;
	int q;

	*ibp++ = (ch = GetNibble());
	q = hexval(ch);

	switch (q/4) {
	  case 0:
		sprintf(disBuf, "%c=%c-%c  A", q03and8B[q][0], q03and8B[q][0],
			q03and8B[q][1]);
		break;
	  case 1:
		sprintf(disBuf, "%c=%c+1  A", 'A'+q-4, 'A'+q-4, 'A'+q-4);
		break;
	  case 2:
		sprintf(disBuf, "%c=%c-%c  A", q03and8B[q-8][1], q03and8B[q-8][1],
			q03and8B[q-8][0]);
		break;
	  case 3:
		sprintf(disBuf, "%c=%c-%c  A", q47andCF[q-12][1], q47andCF[q-12][0],
			q47andCF[q-12][1]);
		break;
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *uaF(char dummy)
{
	char ch;
	int q;

	*ibp++ = (ch = GetNibble());
	q = hexval(ch);

	switch (q/4) {
	  case 0:
		sprintf(disBuf, "%cSL    A", 'A'+q);
		break;
	  case 1:
		sprintf(disBuf, "%cSR    A", 'A'+q-4);
		break;
	  case 2:
		sprintf(disBuf, "%c=-%c   A", 'A'+q-8, 'A'+q-8);
		break;
	  case 3:
		sprintf(disBuf, "%c=-%c-1 A", 'A'+q-12, 'A'+q-12);
		break;
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua80(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);

	static char *imap0_B[] = {
		"OUT=CS",
		"OUT=C ",
		"A=IN  ",
		"C=IN  ",
		"UNCNFG",
		"CONFIG",
		"C=ID  ",
		"SHUTDN",
		"??????",
		"C+P+1 ",
		"RESET ",
		"BUSCC ",
		"C=P   ",
		"P=C   ",
		"SREQ? ",
		"CPEX "
	};

	switch (thirdNibble) {
	    case 0x8:
		return ua808();
		break;

	    case 0xC:
	    case 0xD:
	    case 0xF:
		ch = (*ibp++ = GetNibble());
		sprintf(disBuf, "%s %d", imap0_B[thirdNibble], hexval(ch));
		return disBuf;

	    default:
		return imap0_B[thirdNibble];
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua808(void)
{
	char ch = (*ibp++ = GetNibble());
	int fourth_nibble = hexval(ch);

	static char *imap[] = {
		"INTON ",
		"??????",
		"LAHEX ",
		"BUSCB ",
		"ABIT=0",
		"ABIT=1",
		"?ABIT=0",
		"?ABIT=1",
		"CBIT=0",
		"CBIT=1",
		"?CBIT=0",
		"?CBIT=1",
		"PC=(A)",
		"BUSCD ",
		"PC=(C)",
		"INTOFF"
	};

	switch (fourth_nibble) {
	  case 0x1:
		return ua8081();
		break;

	  case 0x2:
		return load_reg_with_data(imap[fourth_nibble]);
		break;

	  case 0x4:
	  case 0x5:
	  case 0x8:
	  case 0x9:
		ch = (*ibp++ = GetNibble());
		sprintf(disBuf, "%s %d", imap[fourth_nibble], hexval(ch));
		return disBuf;
		break;

	  case 0x6:
	  case 0x7:
	  case 0xA:
	  case 0xB:
		testWithNibble(imap[fourth_nibble]);
		return disBuf;
		break;

 	  case 0xC:
	  case 0xE:
		StopAns = 1;
		return imap[fourth_nibble];
		break;

	  default:
		return imap[fourth_nibble];
		break;
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua8081(void)
{
	char ch = (*ibp++ = GetNibble());
	int fifth_nibble = hexval(ch);

	static char *imap[] = {
		"RSI   ",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????"
	};

	switch (fifth_nibble) {
	  case 0x0:
		return imap[fifth_nibble];
		break;

	  default:
		return unknownInstr;
		break;
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua81(void)
{
	char ch = (*ibp++ = GetNibble());
	int r = hexval(ch);
	static char *imap[] = {
		"ASLC  ",
		"BSLC  ",
		"CSLC  ",
		"DSLC  ",
		"ASRC  ",
		"BSRC  ",
		"CSRC  ",
		"DSRC  ",
		"??????",
		"??????",
		"??????",
		"??????",
		"ASRB  ",
		"BSRB  ",
		"CSRB  ",
		"DSRB  "
	};

	switch (r) {
	  case 0x8:
		return ua818();
		break;

	  case 0x9:
		return ua819();
		break;

	  case 0xA:
		return ua81A();
		break;

	  case 0xB:
		return ua81B();
		break;

	  default:
		return imap[r];
		break;
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua818(void)
{
	char ch;
	int fs,r,d;
	static char *imap[] = {
		"A=A+CON",
		"B=B+CON",
		"C=C+CON",
		"D=D+CON",
		"??????",
		"??????",
		"??????",
		"??????",
		"A=A-CON",
		"B=B-CON",
		"C=C-CON",
		"D=D-CON",
		"??????",
		"??????",
		"??????",
		"??????"
	};

 	ch = (*ibp++ = GetNibble());
	fs = hexval(ch);

 	ch = (*ibp++ = GetNibble());
	r = hexval(ch);

 	ch = (*ibp++ = GetNibble());
	d = hexval(ch) + 1;

	sprintf(disBuf, "%s %s,%d", imap[r], field[fs], d);
	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua819(void)
{
	char ch;
	int r,fs;
	static char *imap[] = {
		"ASRB  ",
		"BSRB  ",
		"CSRB  ",
		"DSRB  ",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????"
	};

 	ch = (*ibp++ = GetNibble());
	fs = hexval(ch);

 	ch = (*ibp++ = GetNibble());
	r = hexval(ch);

	sprintf(disBuf, "%s %s", imap[r], field[fs]);
	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua81A(void)
{
	static char *imap[] = {
		"R%c=%c   %s",
		"%c=R%c   %s",
		"%cR%cEX  %s",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????"
	};

	char ch;
	int fs;
	int r;
	int reg;
	char reg1;
	char reg2;

	ch = (*ibp++ = GetNibble());
	fs = hexval(ch);

	ch = (*ibp++ = GetNibble());
	r = hexval(ch);

	ch = (*ibp++ = GetNibble());
	reg = hexval(ch);

	if (reg < 8) {
		reg1 = 'A';
		reg2 = '0' + reg;
	} else {
		reg1 = 'C';
		reg2 = '0' + reg - 8;
	}

	switch (r) {
	  case 0x0:
		sprintf(disBuf, imap[r], reg2, reg1,field[fs]);
		return disBuf;
		break;

	  case 0x1:
	  case 0x2:
		sprintf(disBuf, imap[r], reg1, reg2,field[fs]);
		return disBuf;
		break;

	    default:
		return unknownInstr;
		break;
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua81B(void)
{
	char ch = (*ibp++ = GetNibble());
	int r = hexval(ch);
	static char *imap[] = {
		"??????",
		"??????",
		"PC=A  ",
		"PC=C  ",
		"A=PC  ",
		"C=PC  ",
		"APCEX ",
		"CPCEX ",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????",
		"??????"
	};

	return imap[r];
}

/***********************************************************************
	
***********************************************************************/

static char *ua82(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);

	switch (thirdNibble) {
	  case 0x1:
		return "XM=0  ";

	  case 0x2:
		return "SB=0  ";

	  case 0x4:
		return "SR=0  ";

	  case 0x8:
		return "MP=0  ";
		break;

	  case 0xF:
		return "CLRHST";

	  default:
		sprintf(disBuf, "HST&=  #%X", ~thirdNibble & 0xF);
		return disBuf;
	}
}

/***********************************************************************
	
***********************************************************************/

static char *ua83(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);

	switch (thirdNibble) {
	  case 0x1:
		offsetPTarget("?XM=0");
		break;
	  case 0x2:
		offsetPTarget("?SB=0");
		break;
	  case 0x4:
		offsetPTarget("?XR=0");
		break;
	  case 0x8:
		offsetPTarget("?MP=0");
		break;
	  default:
		return unknownInstr;
	}

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua84(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);

	sprintf(disBuf, "ST=0   %d", thirdNibble);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua85(void)
{
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);

	sprintf(disBuf, "ST=1   %d", thirdNibble);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static void testWithNibble(char *typ)
{
	static char instrBuf[10];
	char ch = (*ibp++ = GetNibble());
	int thirdNibble = hexval(ch);

	sprintf(instrBuf, "%s %d", typ, thirdNibble);
	offsetPTarget(instrBuf);
}

/***********************************************************************
	
***********************************************************************/

static char *ua86(void)
{
	testWithNibble("?ST=0 ");

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua87(void)
{
	testWithNibble("?ST=1 ");

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua88(void)
{
	testWithNibble("?P#   ");

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua89(void)
{
	testWithNibble("?P=   ");

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua8A(void)
{
	static char instrBuf[7];
	char ch = (*ibp++ = GetNibble());
	int q = hexval(ch);

	switch (q/4) {
	  case 0:
		sprintf(instrBuf, "?%c=%c   A", q03and8B[q][0], q03and8B[q][1]);
		break;
	  case 1:
		sprintf(instrBuf, "?%c#%c   A", q47andCF[q-4][0], q47andCF[q-4][1]);
		break;
	  case 2:
		sprintf(instrBuf, "?%c=0   A", 'A'+q-8);
		break;
	  case 3:
		sprintf(instrBuf, "?%c#0   A", 'A'+q-12);
		break;
	}
	offsetPTarget(instrBuf);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua8B(void)
{
	static char instrBuf[7];
	char ch = (*ibp++ = GetNibble());
	int q = hexval(ch);

	switch (q/4) {
	  case 0:
		sprintf(instrBuf, "?%c>%c   A", q03and8B[q][0], q03and8B[q][1]);
		break;
	  case 1:
		sprintf(instrBuf, "?%c>%c   A", q47andCF[q-4][0], q47andCF[q-4][1]);
		break;
	  case 2:
		sprintf(instrBuf, "?%c>=%c  A", q03and8B[q-8][0], q03and8B[q-8][1]);
		break;
	  case 3:
		sprintf(instrBuf, "?%c>=%c  A", q47andCF[q-12][0], q47andCF[q-12][1]);
		break;
	}
	offsetPTarget(instrBuf);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua8C(void)
{
	follow(offsetTarget("GOLONG", 4, 4));

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static bin5_t absTarget(char *instr)
{
	char dBuf[6], *acom;
	bin5_t target;

	GetNNibbles(dBuf, 5);
	strcpy(ibp, dBuf);
	ibp += 5;
	strrev(dBuf);

	target = str2adr(dBuf, 0);
	sprintf(disBuf, "%s %s", instr, dBuf);

	acom = get_address_comment(target);

	if (*acom)
		sprintf(disBuf, "%-20s  ; %s", disBuf, acom);

	return target;
}

/***********************************************************************
	
***********************************************************************/

static char *ua8D(void)
{
	follow(absTarget("GOVLNG"));

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua8E(void)
{
	(void) offsetTarget("GOSUBL", 4, 0);

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static char *ua8F(void)
{
	(void) absTarget("GOSBVL");

	return disBuf;
}

/***********************************************************************
	
***********************************************************************/

static RDA FirstNibble = {
	ua0, uaSN, ua2, ua3, ua4, ua5, ua6, ua7,
	uaSN, ua9, uaA, uaB, uaC, uaD, uaE, uaF
};

static RDAv ua1SecondNibble = {
	ua10, ua11, ua12, ua13, ua14, ua15, ua16, ua17,
	ua18, ua19_A_B, ua19_A_B, ua19_A_B, ua1C, ua1D_E_F, ua1D_E_F, ua1D_E_F
};

static RDAv ua8SecondNibble = {
	ua80, ua81, ua82, ua83, ua84, ua85, ua86, ua87,
	ua88, ua89, ua8A, ua8B, ua8C, ua8D, ua8E, ua8F
};

PcpFv *SecondNibble[16] = {
	NULL, ua1SecondNibble, NULL, NULL, NULL, NULL, NULL, NULL,
	ua8SecondNibble, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};

void unassem1instr(int *stop)
{
	/*
	// Format a line of machine code.
	*/

	char nib;
	char *ans, *acom, *ccom;
	bin5_t origWP = WorkPtr;

	ibp = instrBuf;

	nib = GetNibble();
	if (isxdigit(nib)) {
		*ibp++ = nib;

		StopAns = 0;
		Follow = 0;
		goyes_buf[0] = '\0';
		ans = (FirstNibble[hexval(nib)])(nib);
		*ibp = '\0';

		acom = get_address_comment(origWP);
		if (*acom) {
			pager(0);
			printf("%05lX: ; *** %s ***\n", origWP, acom);
		}

		pager(0);
		ccom = get_code_comment(origWP);
		if (*ccom)
			printf("%05lX: %-10s %-20s  ; %s\n",
				origWP, instrBuf, ans, ccom);
		else
			printf("%05lX: %-10s %-20s\n",
				origWP, instrBuf, ans);

		if (*goyes_buf) {
			pager(0);
			puts(goyes_buf);
		}

		if (Follow) {
			pager(0);
			putchar('\n');
		}

		*stop = StopAns;
	} else {
		pager(0);
		puts("Error: invalid data at current address");
		*stop = 1;
	}
}
