(* Copyright (C) 1990, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* Last modified on Wed Jul 15 09:31:29 PDT 1992 by kalsow     *)
(*      modified on Thu Mar  7 03:13:05 1991 by muller         *)

UNSAFE MODULE RTType;

IMPORT RT0, RT0u, RTMisc, Cstdlib, Cstring, SmallIO, Word, RTHeapRep;

(*-------------------------------------------------------- initialization ---*)

VAR init_done := FALSE;

PROCEDURE Init () =
  BEGIN
    <* ASSERT NOT init_done *>
    init_done := TRUE;
    RegisterTypes ();
    CheckBrands ();
    IdentifyOpaques ();
    FindChildren ();
    AssignTypecodes ();
    CheckRevelations ();
    FixSizes ();
    CallSetupProcs ();
    (* ShowTypes (TRUE); *)
    RTHeapRep.CheckTypes ();
  END Init;

PROCEDURE RegisterTypes () =
  (* "register" each typecell with a distinct temporary typecode *)
  VAR t: RT0.TypeDefinition;  tp: RT0.TypeDefinitionPtr;  cnt := 0;
  BEGIN
    FOR i := 0 TO RT0u.nModules-1 DO
      tp := RT0u.modules[i].type_cells;
      IF (tp # NIL) THEN
        LOOP
          t := tp^;
          IF (t = NIL) THEN EXIT END;
          IF (t.selfLink^ = NIL) THEN
            t.selfLink^ := t;
            t.typecode := cnt;
            RT0u.types[cnt] := t;
            INC (cnt);
          END;
          INC (tp, ADRSIZE (tp^));
        END;
      END;
    END;
    <* ASSERT cnt <= RT0u.nTypes *>
    RT0u.nTypes := cnt;
  END RegisterTypes;

PROCEDURE CheckBrands () =
  (* ensure that all brands are distinct *)
  VAR
    t, a, b: RT0.TypeDefinition;
    tp: RT0.TypeDefinitionPtr;
    hash: INTEGER;
    buckets: ARRAY [0..292] OF RT0.TypeDefinition;
  BEGIN
    (* Hash each type with a non-nil brand into the table
       using the type's sibling pointer to resolve collisions. *)
    FOR i := 0 TO RT0u.nModules-1 DO
      tp := RT0u.modules[i].type_cells;
      IF (tp # NIL) THEN
        LOOP
          t := tp^;
          IF (t = NIL) THEN EXIT END;
          IF (t.brand # NIL) THEN
            hash := HashString (t.brand) MOD LAST (buckets);
            t.sibling := buckets[hash];
            buckets[hash] := t;
          END;
          INC (tp, ADRSIZE (tp^));
        END;
      END;
    END;

    (* Run the naive O(n^2) check on each hash bucket. *)
    FOR i := 0 TO LAST (buckets) DO
      a := buckets[i];
      WHILE (a # NIL) DO
        b := a.sibling;
        WHILE (b # NIL) DO
          IF Cstring.strcmp (a.brand, b.brand) = 0 THEN
            PutText ("Two types have the same brand: ");
            PutType (a);
            PutText (" and ");
            PutType (b);
            RTMisc.FatalError (NIL, 0, "bad types");
          END;
          b := b.sibling;
        END;
        a := a.sibling;
      END;
    END;

    (* Reset the sibling pointers. *)
    FOR i := 0 TO RT0u.nModules-1 DO
      tp := RT0u.modules[i].type_cells;
      IF (tp # NIL) THEN
        LOOP
          t := tp^;
          IF (t = NIL) THEN EXIT END;
          t.sibling := NIL;
          INC (tp, ADRSIZE (tp^));
        END;
      END;
    END;
  END CheckBrands;

PROCEDURE HashString (cp: UNTRACED REF CHAR): INTEGER =
  VAR hash := 0;
  BEGIN
    WHILE (cp^ # '\000') DO
      hash := Word.Plus (Word.LeftShift (hash, 1), ORD (cp^));
      INC (cp, BYTESIZE (cp^));
    END;
    RETURN hash;
  END HashString;

PROCEDURE IdentifyOpaques () =
  (* ensure that all opaque types were revealed *)
  VAR pairs: RT0.RevelationPairs;
  BEGIN
    FOR i := 0 TO RT0u.nModules - 1 DO
      pairs := RT0u.modules[i].revelations.exported.full;
      IF (pairs # NIL) THEN
        WHILE (pairs.lhs # NIL) DO
          IF (pairs.lhs^ # NIL) THEN
            BadPair ("multiple revelations for opaque type: ", pairs^);
          ELSIF (pairs.rhs^ = NIL) THEN
            BadPair ("cannot identify two opaque types: ", pairs^);
          ELSE (* identify the opaque and concrete *)
            pairs.lhs^ := pairs.rhs^;
          END;
          INC (pairs, ADRSIZE (pairs^));
        END;
      END;
    END;
  END IdentifyOpaques;

PROCEDURE FindChildren () =
  VAR child, parent: RT0.TypeDefinition;
  BEGIN
    FOR i := 0 TO RT0u.nTypes -1 DO
      child := RT0u.types[i];
      IF (child.parentLink # NIL) THEN
        parent := child.parentLink^;
        child.parent := parent;
        child.sibling := parent.children;
        parent.children := child;
      END;
    END;
  END FindChildren;

PROCEDURE AssignTypecodes () =
  VAR
   t, u, null, text, root, uroot: RT0.TypeDefinition;
    next_typecode: INTEGER;
  BEGIN
    (* first, use the compiler & the temporary typecodes to find the
       types with reserved typecodes *)
    null  := RT0u.types [TYPECODE (NULL)];
    text  := RT0u.types [TYPECODE (TEXT)];
    root  := RT0u.types [TYPECODE (ROOT)];
    uroot := RT0u.types [TYPECODE (UNTRACED ROOT)];

    (* reset the temporary typecodes *)
    FOR i := 0 TO RT0u.nTypes-1 DO
      RT0u.types[i].typecode := LAST (Typecode);
    END;

    (* assign the fixed typecodes *)
    null.typecode := RT0.NilTypecode;   null.lastSubTypeTC := RT0.NilTypecode;
    text.typecode := RT0.TextTypecode;  text.lastSubTypeTC := RT0.TextTypecode;
    next_typecode := MAX (RT0.NilTypecode, RT0.TextTypecode) + 1;

    (* assign the OBJECT typecodes *)
    AssignObjectTypecode (root, next_typecode);
    AssignObjectTypecode (uroot, next_typecode);

    (* assign the remaining REF typecodes *)
    FOR i := 0 TO RT0u.nTypes-1 DO
      t := RT0u.types[i];
      IF (t.typecode = LAST (Typecode)) THEN
        t.typecode := next_typecode;
        t.lastSubTypeTC := next_typecode;
        INC (next_typecode);
      END;
    END;

    <* ASSERT next_typecode = RT0u.nTypes *>

    (* shuffle the typecells into their correct slots *)
    FOR i := 0 TO RT0u.nTypes-1 DO
      t := RT0u.types[i];
      WHILE (t.typecode # i) DO
        u := RT0u.types [t.typecode];
        RT0u.types [t.typecode] := t;
        t := u;
      END;
      RT0u.types[i] := t;
    END;
  END AssignTypecodes;

PROCEDURE AssignObjectTypecode (t: RT0.TypeDefinition;  VAR next: INTEGER) =
  VAR u: RT0.TypeDefinition;
  BEGIN
    <* ASSERT t.typecode = LAST (Typecode) *>
    t.typecode := next;  INC (next);
    u := t.children;
    WHILE (u # NIL) DO
      AssignObjectTypecode (u, next);
      u := u.sibling;
    END;
    t.lastSubTypeTC := next-1;
  END AssignObjectTypecode;

PROCEDURE CheckRevelations () =
  (* ensure that all partial revelations are consistent *)
  VAR r: RT0.Revelations;
  BEGIN
    FOR i := 0 TO RT0u.nModules - 1 DO
      r := RT0u.modules[i].revelations;
      CheckEqual   (r.imported.full);
      CheckSubtype (r.exported.partial);
      CheckSubtype (r.imported.partial);
    END;
  END CheckRevelations;

PROCEDURE CheckEqual (pairs: RT0.RevelationPairs) =
  BEGIN
    IF (pairs = NIL) THEN RETURN END;
    WHILE (pairs.lhs # NIL) DO
      IF (pairs.lhs^ = NIL) OR (pairs.rhs^ = NIL) THEN
        BadPair ("undefined type!  ", pairs^);
      ELSIF (pairs.lhs^ # pairs.rhs^) THEN
        BadPair ("identified types not equal: ", pairs^);
      END;
      INC (pairs, ADRSIZE (pairs^));
    END;
  END CheckEqual;

PROCEDURE CheckSubtype (pairs: RT0.RevelationPairs) =
  BEGIN
    IF (pairs = NIL) THEN RETURN END;
    WHILE (pairs.lhs # NIL) DO
      IF (pairs.lhs^ = NIL) OR (pairs.rhs^ = NIL) THEN
        BadPair ("undefined type!  ", pairs^);
      ELSIF NOT IsSubtype (pairs.lhs^.typecode, pairs.rhs^.typecode) THEN
        BadPair ("inconsistent revelation: ", pairs^);
      END;
      INC (pairs, ADRSIZE (pairs^));
    END;
  END CheckSubtype;

PROCEDURE FixSizes () =
  (* fix the data(method) sizes and offsets *)
  VAR t: RT0.TypeDefinition;
  BEGIN
    (* make sure that all the REF types are some multiple of header words *)
    FOR i := 0 TO RT0u.nTypes-1 DO
      t := RT0u.types[i];
      IF (t.typecode # RT0.NilTypecode)
        AND (t.parent = NIL)
        AND (t.children = NIL) THEN
        t.dataSize := RTMisc.Upper (t.dataSize, BYTESIZE (RTHeapRep.Header));
      END;
    END;

    (* fix the objects *)
    FixObjectSizes (RT0u.types [TYPECODE (ROOT)]);
    FixObjectSizes (RT0u.types [TYPECODE (UNTRACED ROOT)]);
  END FixSizes;

PROCEDURE FixObjectSizes (t: RT0.TypeDefinition) =
  VAR u: RT0.TypeDefinition;
  BEGIN
    (* fix my sizes *)
    u := t.parent;
    IF (u # NIL) THEN
      t.dataOffset := RTMisc.Upper (u.dataSize, t.dataAlignment);
      INC (t.dataSize, t.dataOffset);
      t.dataAlignment := MAX (t.dataAlignment, u.dataAlignment);
      t.methodOffset := u.methodSize;
      INC (t.methodSize, t.methodOffset);
    END;
    t.dataSize := RTMisc.Upper (t.dataSize, BYTESIZE (RTHeapRep.Header));

    (* allocate my default method list *)
    t.defaultMethods := Cstdlib.malloc (t.methodSize);
    IF (t.defaultMethods = NIL) THEN
      RTMisc.FatalError (NIL, 0, "unable to allocate method suite");
    END;

    (* fix my children *)
    u := t.children;
    WHILE (u # NIL) DO
      FixObjectSizes (u);
      u := u.sibling;
    END;
  END FixObjectSizes;

PROCEDURE CallSetupProcs () =
  VAR t: RT0.TypeDefinition;
  BEGIN
    (* set up the REF types *)
    FOR i := 0 TO RT0u.nTypes-1 DO
      t := RT0u.types[i];
      IF (t.parent = NIL) AND (t.children = NIL) AND (t.setupProc # NIL) THEN
        t.setupProc (t);
      END;
    END;

    (* set up the objects *)
    SetupObject (RT0u.types [TYPECODE (ROOT)]);
    SetupObject (RT0u.types [TYPECODE (UNTRACED ROOT)]);
  END CallSetupProcs;

PROCEDURE SetupObject (t: RT0.TypeDefinition) =
  VAR u: RT0.TypeDefinition;
  BEGIN
    (* initialize my method suite from my parent *)
    u := t.parent;
    IF (u # NIL) THEN
      RTMisc.Copy (u.defaultMethods, t.defaultMethods, u.methodSize);
    END;
    LOOPHOLE (t.defaultMethods, UNTRACED REF INTEGER)^ := t.typecode;

    (* call my setup proc *)
    IF (t.setupProc # NIL) THEN t.setupProc (t) END;

    (* set up my children *)
    u := t.children;
    WHILE (u # NIL) DO
      SetupObject (u);
      u := u.sibling;
    END;
  END SetupObject;

(*------------------------------------------------ user callable routines ---*)

PROCEDURE MaxTypeCode (): Typecode =
  BEGIN
    RETURN RT0u.nTypes - 1;
  END MaxTypeCode;

PROCEDURE IsSubtype (tc1, tc2: Typecode): BOOLEAN =
  VAR t: RT0.TypeDefinition;
  BEGIN
    IF (tc2 >= RT0u.nTypes) THEN BadType (tc2) END;
    IF (tc1 = 0)            THEN RETURN TRUE   END;
    IF (tc1 >= RT0u.nTypes) THEN BadType (tc1) END;
    t := RT0u.types[tc2];
    RETURN (t.typecode <= tc1 AND tc1 <= t.lastSubTypeTC);
  END IsSubtype;

(*--------------------------------------------------------------- tracing ---*)

PROCEDURE Visit (arg, r: REFANY;  p: VisitProc;
                 mask: RefTypeSet := AllRefTypes) RAISES ANY =
  VAR x: VisitProc := p; (* force the check for top-level p's *)
  VAR proc: RT0.TypeMapProc;
  BEGIN
    IF r = NIL THEN RTMisc.FatalError (NIL,0,"RTType.Visit: NIL argument") END;
    proc := RT0u.types [TYPECODE (r)].mapProc;
    IF (proc # NIL) THEN proc (x, arg, LOOPHOLE (r, ADDRESS), mask) END;
  END Visit;

(*---------------------------------------------------------------- errors ---*)

PROCEDURE BadPair (msg: TEXT;  READONLY p: RT0.RevelationPair) =
  BEGIN
    PutText (msg);
    PutText ("_t");
    PutHex  (p.lhs_id);
    PutType (p.lhs^);
    PutText (" and ");
    PutText ("_t");
    PutHex  (p.rhs_id);
    PutType (p.rhs^);
    RTMisc.FatalError (NIL, 0, "bad revelations");
  END BadPair;

PROCEDURE BadType (tc: Typecode) =
  BEGIN
    RTMisc.FatalErrorI ("improper typecode: ", tc);
  END BadType;

(*---------------------------------------------------- internal debugging ---*)

PROCEDURE ShowTypes (full := TRUE) =
  VAR t: RT0.TypeDefinition;  isObject: BOOLEAN;
  BEGIN
    PutILine ("Here are the types: nTypes = ", RT0u.nTypes);
    FOR i := 0 TO RT0u.nTypes-1 DO
      t := RT0u.types[i];
      isObject := FALSE;
      PutType (t); PutText ("\n");
      PutText  ("  typecode= "); PutInt (t.typecode);
        PutILine (" .. ", t.lastSubTypeTC);
      IF (t.parent # NIL) THEN
        PutText ("  parent= "); PutInt (t.parent.typecode);
        isObject := TRUE;
      END;
      IF (t.children # NIL) THEN
        PutText ("  firstChild= "); PutInt (t.children.typecode);
        isObject := TRUE;
      END;
      IF (t.sibling # NIL) THEN
        PutText ("  siblings= "); PutInt (t.sibling.typecode);
        isObject := TRUE;
      END;
      IF (isObject) THEN PutText ("\n") END;
      IF full THEN
        PutText ("  data   ");
        PutText ("  S= "); PutInt (t.dataSize);
        PutText ("  A= "); PutInt (t.dataAlignment);
        PutText ("  O= "); PutInt (t.dataOffset);
        PutText ("\n");
        IF (isObject) THEN
          PutText ("  method ");
          PutText ("  S= ");  PutInt (t.methodSize);
          PutText ("  O= ");  PutInt (t.methodOffset);
          PutText ("\n");
        END;
        IF (t.nDimensions # 0) OR (t.elementSize # 0) THEN
          PutText (" array   ");
          PutText ("  D= ");  PutInt (t.nDimensions);
          PutText ("  S= ");  PutInt (t.elementSize);
          PutText ("\n");
        END;
      END;
    END;
    SmallIO.Flush (SmallIO.stderr);
    EVAL ShowTypes; (* to prevent an "unused symbol" warning *)
  END ShowTypes;

PROCEDURE PutType (t: RT0.TypeDefinition) =
  BEGIN
    PutText ("[");
    PutAddr (t);
    PutText ("  ");

    IF (t.name # NIL)
      THEN PutString (t.name);
      ELSE PutText ("<anonymous>");
    END;
    PutText ("  ");

    IF (t.brand # NIL) THEN
      PutText   ("\"");
      PutString (t.brand);
      PutText   ("\"");
    ELSE
      PutText ("<no brand>");
    END;

    PutText ("]");
  END PutType;

PROCEDURE PutString (cp: UNTRACED REF CHAR) =
  BEGIN
    WHILE (cp^ # '\000') DO
      SmallIO.PutChar (SmallIO.stderr, cp^); 
      INC (cp, BYTESIZE (cp^));
    END;
  END PutString;

(******************************
PROCEDURE PutALine (msg: TEXT;  a: ADDRESS) =
  BEGIN
    PutText (msg);
    PutAddr (a);
    PutText ("\n");
  END PutALine;
*************************)

PROCEDURE PutILine (msg: TEXT;  i: INTEGER) =
  BEGIN
    PutText (msg);
    PutInt  (i);
    PutText ("\n");
  END PutILine;

PROCEDURE PutText (t: TEXT) =
  BEGIN
    SmallIO.PutText (SmallIO.stderr, t);
  END PutText;

PROCEDURE PutInt (i: INTEGER) =
  BEGIN
    SmallIO.PutInt (SmallIO.stderr, i);
  END PutInt;

PROCEDURE PutAddr (a: ADDRESS) =
  BEGIN
    PutHex (LOOPHOLE (a, INTEGER));
  END PutAddr;

PROCEDURE PutHex (i: INTEGER) =
  BEGIN
    SmallIO.PutHexa (SmallIO.stderr, i);
  END PutHex;

BEGIN
END RTType.

