/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/VideoEdit/RCS/VideoEdit.c,v 0.40 92/05/13 00:20:55 drapeau Exp $ */
/* $Log:	VideoEdit.c,v $
 * Revision 0.40  92/05/13  00:20:55  drapeau
 * Minor change, called new function "SetPlayerHandler()" instead
 * of "SetPlayer()".
 * 
 * Revision 0.39  92/05/12  19:04:10  drapeau
 * Minor cosmetic change to the code to conform to coding style specifications.
 * 
 * Revision 0.38  92/01/07  14:07:24  drapeau
 * Cosmetic changes only, to make code more readable.
 * 
 * Revision 0.37  92/01/03  15:44:36  drapeau
 * Changed occurrances of "NULL" in calls to Browse() to use "0" instead.
 * This is due to the ANSI definition of NULL as "(void*)0".
 * 
 * Revision 0.36  91/11/14  10:11:47  collab
 * No code changes; only changing ownership of this file.
 * 
 * Revision 0.34  91/09/29  14:31:23  lim
 * Removed mySerialPort.
 * 
 * Revision 0.33  91/09/27  16:53:10  lim
 * Added "None" as default item on options Player stg.
 * 
 * Revision 0.32  91/09/26  11:47:55  lim
 * Fixed bug in SetDuration for the case when no device has been 
 * initialized.
 * 
 * Revision 0.31  91/09/24  21:34:20  lim
 * Changed 'vEdit' to 'VideoEdit' everywhere.
 * 
 * Revision 0.30  91/09/04  14:01:55  lim
 * Play sets paused to 0.
 * 
 * Revision 0.29  91/08/30  15:43:43  lim
 * Check for null sender in Quit().
 * 
 * Revision 0.28  91/08/28  13:09:55  lim
 * Corrected duration.
 * 
 * Revision 0.27  91/08/24  17:54:54  lim
 * Implemented PrintDiagnostics. 
 * Using command line flag '-d', the user can choose to launch the
 * app with diagnostic messages. Otherwise, no diag messages will
 * be printed.
 * 
 * Revision 0.26  91/08/23  17:14:52  lim
 * 1. Duration can be calculated regardless of whether
 * myVideo has been initialized or not.
 * 2. PlayerPlay becomes PlayerForwardPlay due to change
 * in 'PlayerStatus.h'
 * 3. Forward and Reverse Steps both also implement still.
 * 4. ClearMarker has been removed as a video object function.
 * 5. Document format has been changed.
 * 
 * Revision 0.25  91/08/21  10:47:28  lim
 * Added new variable, mySerialPort, to denote serial port setting on
 * options panel. Initialized to "NOTHING".
 * 
 * Revision 0.24  91/08/17  20:53:05  lim
 * 1. Pause After Search implemented.
 * 2. OpenPanel is now Browse.
 * 
 * Revision 0.23  91/08/16  13:08:53  lim
 * 1. No more editFilename. Filename now put in title bar. 
 * 2. Call Browse() with OPENPANEL_CHECK for loading command line file.
 * 3. Moved info button and function to options panel.
 * 
 * Revision 0.22  91/08/11  18:48:29  lim
 * Replaced call to XOpenDisplay to xv_get(obj, XV_DISPLAY)
 * 
 * Revision 0.21  91/08/09  17:30:29  lim
 * Included OpenPanel
 * 
 * Revision 0.20  91/08/08  17:18:44  lim
 * 1. CalcSpeed() - one parameter added, 'playMode'. If playMode is 1, it is used to calculate
 *    device speed for segment play. If playMode is 0, it is used to calculate device speed for
 *    normal play.
 * 
 * Revision 0.19  91/08/07  13:18:24  lim
 * 1. ErrorCheck() has been removed. The responsibility for notification of
 * error has been shifted to the device drivers, which will call DisplayError()
 * in 'videoObjects.c'.
 * 2. Added instance pointer, "theObject" to all public function calls.
 * 3. The application now sets up a device driver when started, if "vEdit.defaultPlayer"
 * is set in ~/.Xdefaults.
 * 
 * Revision 0.18  91/08/02  12:53:28  lim
 * 1. Pause() now is implemented with Still() and Play() and the
 * variable 'paused'.
 * 2. DevInit has been moved to the file "videoObjects.c"
 * 3. allDevices has been changed from an array of device configurations
 * to an array of names of devices supported.
 * 
 * Revision 0.17  91/07/29  22:28:58  lim
 * Status codes updated to conform with specs.
 * Added driver for Sony1550.
 * 
 * Revision 0.16  91/07/27  22:28:11  lim
 * 1. Changed speed from double to int.
 * 2. Added Pioneer6000Driver.
 * 
 * Revision 0.15  91/07/24  11:01:19  lim
 * Ran through xsaber - removed redundant variables.
 * 
 * Revision 0.14  91/07/23  15:54:31  lim
 * 1. CalcSpeed now always takes in the number of frames/sec and
 * returns closest number of frames per second the device can play.
 * 2. Duration rounding has been fixed.
 * 3. Added Panasonic Optical Player object.
 * 4. Added optionsPlayerStg to select the appropriate driver. It
 * starts up with no drivers installed.
 * 
 * Revision 0.13  91/07/19  16:15:36  lim
 * Set default value for speed in edit window to be 30.
 * 
 * Revision 0.12  91/07/19  14:00:13  lim
 * Made changes to accommodate addition of speed field
 * in edit.
 * 1. SetDuration is changed so that duration depends on 
 * speed to be played (depends on device).
 * 2. Search is modified to fit the function prototype for
 * PlayFromTo.
 * Eject is now a separate button than Stop.
 * PlayMode is moved to vEditEdit.c. It is no longer called
 * by functions outside vEditEdit.c
 * 
 * Revision 0.11  91/07/17  22:49:46  lim
 * Now we use Pioneer4200Pause in place of Pioneer4200Stop, because
 * the latter calls puts the player into park position, which slows 
 * down the TimeLine considerably the next time 'Play' is hit. Thus
 * the player will always be in 'ready' state.
 * 
 * Revision 0.10  91/07/12  16:18:24  lim
 * Initial revision implementing VideoObject.
 *  */

#include "VideoEdit.h"


static char mainrcsid[] = "$Header: /Source/Media/collab/VideoEdit/RCS/VideoEdit.c,v 0.40 92/05/13 00:20:55 drapeau Exp $";

/* Edit list variables */
int lines;							    /* number of lines in the edit list */
int editnum;							    /* the current selection in the edit list */
int change;							    /* indicates if unsaved changes exist in the edit list */
int clearframe;							    /* determines if the frame/chapter textfield is to be 
								       cleared */
int search;							    /* flag to indicate if a notify procedure is to be executed
								       This is necessary because a textfield notify procedure 
								       is executed twice for each key depressed (possibly a 
								       XView bug) */
int startframe[MAX_LISTSIZE]; 
int endframe[MAX_LISTSIZE];					    /* arrays to store the start/end frames of each entry in the 
								       edit list */
char		label[MAX_LISTSIZE][MAX_LABELSIZE+1];		    /* array to store the label of each entry in the edit list */
enum AudioType	audioStg[MAX_LISTSIZE];				    /* array to store audio settings for each entry */
int		speedStg[MAX_LISTSIZE];				    /* array to store speed settings for each entry */
Xv_font*	font;						    /* Font used in the panel list */

/* Command line variables */
int receiverPort;						    /* Command line receiver port */
char* startFilename;				                    /* Name of file in command line */
char hostname[MAX_HOSTNAME];					    /* Name of host */

/* Player status variables */
VideoObject* myVideo;						    /* Current video object */
char deviceName[MaxNameLength];					    /* Name of device currently used */
enum Direction skipDir = Reverse;				    /* flag to indicate which direction to skip */
int paused = 0;							    /* Indicates if player is paused. */
int hitQuit = 0;						    /* Set when Quit button is hit. */

enum Boolean diagMode = No;					    /* Flag to print out diagnostic messages */

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute	INSTANCE;


/* ERROR CHECKING ROUTINES */


/* check if a selection is valid */
int
CheckSelection(start, end)
     int start;
     int end;
{
  char buf[50];

  if (start > end) 
  {
    DisplayError("Invalid current selection:",
		 "Start frame is greater than end frame.");
    return -1;
  }
  if ((start < 1) || (end < 1)) 
  {
    DisplayError("Frame number must be at least 1", " ");
    return -1;
  }
  if ((start > MAX_FRAME) || (end > MAX_FRAME)) 
  {
    sprintf (buf, "Maximum frame number allowed is %d\n", MAX_FRAME);
    DisplayError(buf, " ");
    return -1;
  }
  return 0;
}

/*
 * This function parses the command line and retrieves all the known options and their arguments.
 * Currently, the two options are hostname and portnumber.
 * After parsing the options, the variable optind will point to the start of those non-option arguments.  
 * In this case, it will be the filename to be loaded.  At present, only one filename will be handled.  So if
 * there are multiple filenames typed, the last one will be loaded.
 */
void CheckOptions(argc, argv)
     int 	argc;
     char 	**argv;
{
  int optionChar;  
  int option_index = 0;
  static struct option long_options[] =
  {
    {"hostname", 1, 0, 'h'},
    {"portnumber", 1, 0, 'p'},
    {"diagnostics", 0, 0, 'd'},
    {0, 0, 0, 0}
  };

  while (1)							    /* Start parsing all known options */
  {
    optionChar = getopt_long_only (argc, argv, "h:p:d:",
			      long_options, &option_index);
    if (optionChar == EOF)					    /* Done with all known options */
    {
      break;
    }
    switch (optionChar)
    {
     case 'h':
      if (optarg) 
      {
	strcpy (hostname, optarg);
      }
      break;
     case 'p':
      if (optarg) 
      {
	receiverPort = atoi(optarg);
      }
      break;
     case 'd':
      diagMode = Yes;						    /* We want to print diagnostics */
      break;
     default:
      break;
    }
  }
  if (optind < argc)						    /* Check if a filename has been specified */
  {
    startFilename = (char *) malloc (256);
    strcpy (startFilename, argv[optind]);
  }
}
   

/* GENERAL PROCEDURES USED BY THE FUNCTIONS */

/*  Procedure to display the approximate duration of the current selection */
void
SetDuration()
{
  int start;
  int end;
  int duration;
  int speedInDevice;
  int speedInFrames;
  char buf[80];
  
  start = xv_get(VideoEdit_editPopup->editStartTxt, 
		 PANEL_VALUE);
  end = xv_get(VideoEdit_editPopup->editEndTxt, 
	       PANEL_VALUE);
  if (start >= end) 
  {
    duration = 0;
  }
  else 
  {
    speedInFrames = (int) xv_get(VideoEdit_editPopup->editSpeedTxt,
				    PANEL_VALUE);
    if (myVideo)
    {
      speedInDevice = DevCalcSpeed(myVideo, speedInFrames, 1);
      
      if ((speedInDevice) && (end - start))
	duration = (((end - start)%speedInDevice)/speedInDevice > 0.5 ? 
		    ((end - start)/speedInDevice +1) : (end - start)/speedInDevice);
      else
	duration = 0;
    }
    else
    {
      if ((speedInFrames) && (end - start))
	duration = ((end - start)/speedInFrames +1);
      else
	duration = 0;
    }      
  }
  sprintf (buf, "%d", duration);
  xv_set(VideoEdit_editPopup->editDurTxt, 
	 PANEL_VALUE, buf, 
	 NULL);
  return;
  
}


/*      REMOTE CONTROLLER FUNCTIONS	*/

/* accept number input using the mouse.  The current value on the textfield is
   shifted to the left as input is made (if 5 digits are exceeded) */
void
Num(n)
     int n;
{
  int value;
  
  if (clearframe) xv_set(VideoEdit_window1->frameTxt, 
			 PANEL_VALUE, 0,
			 NULL);
  clearframe = 0;
  value = xv_get(VideoEdit_window1->frameTxt, PANEL_VALUE);
  value = value*10 + n;
  if (value > 99999) value = value % 100000;			    /* remove most significant digit */
  xv_set (VideoEdit_window1->frameTxt, 
	  PANEL_VALUE, value, 
	  NULL);
}

/*
  Button notify proc to open the options popup window (optionsButton)
 */
void
Options(item, event)
     Panel_item      item;
     Event           *event;
{
  xv_set(VideoEdit_optionsPopup->optionsPopup, 
	 FRAME_CMD_PUSHPIN_IN, TRUE, 
	 NULL);
  xv_set(VideoEdit_optionsPopup->optionsPopup, 
	 XV_SHOW, TRUE, 
	 NULL);
}

/*
  Button notify proc to open the edit popup window (editButton)
 */
void
Edit(item, event)
     Panel_item      item;
     Event           *event;
{
  xv_set(VideoEdit_editPopup->editPopup, 
	 FRAME_CMD_PUSHPIN_IN, TRUE, 
	 NULL);
  xv_set(VideoEdit_editPopup->editPopup, 
	 XV_SHOW, TRUE, 
	 NULL);
}


/*
 * Notify callback function for `stopButton'.
 * Stop function
 */
void
Stop(item, event)
     Panel_item	item;
     Event      *event;
{
  DevStop(myVideo);
}


/*
 * Notify callback function for `ejectButton'.
 */
void
Eject(item, event)
     Panel_item	item;
     Event		*event;
{
  DevEject(myVideo);
}


/*
 * Notify callback function for `searchButton'.
 * Search frame/chapter
 */
void
Search(item, event)
     Panel_item	item;
     Event		*event;
{
  int searchNum;
  int searchType;
  
  searchNum = xv_get(VideoEdit_window1->frameTxt, PANEL_VALUE);
  if (CheckSelection (searchNum, MAX_FRAME) == -1) 
  {
    clearframe = 1;
    return;
  }
  								    
  searchType =							    /* check if addressing mode is frame or chapter */
    xv_get(VideoEdit_optionsPopup->optionsSearchStg, PANEL_VALUE);

  if (searchType)
    DevSetAddMode(myVideo, PlayerChapterMode);
  else
    DevSetAddMode(myVideo, PlayerFrameMode);
  
  DevPlayFromTo(myVideo, searchNum, searchNum, 30);			    
  clearframe = 1;
  
}

/*
 * Notify callback function for `frameTxt'.
 * Textfield for entering the frame/chapter number for searching and also for
 * displaying the current frame/chapter number if so desired. 
 */
Panel_setting
SearchKbd(item, event)
     Panel_item      item;
     Event           *event;
{
  int n;
  int value;
  
  /* If return key is pressed, go to search mode */

  if (event_action(event) == '\r') 
  {
    if (search == 0)
    {
      Search(item, event);
      search = 1;
    }
    else search = 0;
    return PANEL_NONE;
  }
  
  
  if ((clearframe) && (search == 0) && !event_is_button(event))
    xv_set(VideoEdit_window1->frameTxt, 
	   PANEL_VALUE, 0, 
	   NULL);
  
  /* This code segment allows the current value to be shifted left as new input is entered */

  if ((event_action(event) >= '0') || (event_action(event) <= '9')) 
  {
    clearframe = 0;
    if (event_is_button(event)) return PANEL_NONE;
    if (search == 0) 
    {
      n = event_action(event) - '0';
      value = xv_get(VideoEdit_window1->frameTxt, PANEL_VALUE);
      value = value*10 + n;
      if (value > 99999) value = value % 100000;		    /* remove most significant digit */
      xv_set (VideoEdit_window1->frameTxt, 
	      PANEL_VALUE, value, 
	      NULL);
      search = 1;
    }
    else search = 0;
  }
  return PANEL_NONE;
}


/* The following 10 procedures are the notify procedures for the numeric
   key-pad */

void
One(item, event)
     Panel_item	item;
     Event		*event;
{
  
  Num(1);
}

void
Two(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(2);
}

void
Three(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(3);
}

void
Four(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(4);
}

void
Five(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(5);
}

void
Six(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(6);
}

void
Seven(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(7);
}

void
Eight(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(8);
}

void
Nine(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(9);
}


void
Zero(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(0);
}


/*
 * Notify callback function for `currFrameButton'.
 * Gives the current frame/chapter number.
 */
void
CurrentFrame(item, event)
     Panel_item	item;
     Event		*event;
{
  int current;
  int value;
  
  /* check to see if frame or chapter is desired */
  value = xv_get(VideoEdit_optionsPopup->optionsSearchStg, 
		 PANEL_VALUE);
  if (value == 0) 
    current = DevQueryFrame(myVideo);
  else
    current = DevQueryChapter(myVideo);

  if (current >= 0)
    xv_set (VideoEdit_window1->frameTxt, 
	    PANEL_VALUE, current, 
	    NULL);

  clearframe = 1;
}

/*
 * Notify callback function for `scanRevButton'.
 * Scan backwards about 1000 frames
 */
void
ScanReverse(item, event)
     Panel_item      item;
     Event           *event;
{
  DevReverse(myVideo);
}

/*
 * Notify callback function for `scanFwdButton'.
 * Scan forward about 1000 frames
 */
void
ScanForward(item, event)
     Panel_item      item;
     Event           *event;
{
  DevFastForward(myVideo);
}


/*
 * Notify callback function for `playButton'.
 * Play
 */
void
Play(item, event)
     Panel_item	item;
     Event		*event;
{
  DevPlay(myVideo);
  paused = 0;
}

/*
 * Notify callback function for `revSkipButton'.
 * Skip back to the beginning of the current chapter
 */
void
ReverseSkip(item, event)
     Panel_item	item;
     Event		*event;
{
  
  int currentChapter;
  int value;
  int status;
  
  status = DevQueryStatus(myVideo);
  
 /* find current chapter number */

  currentChapter = DevQueryChapter(myVideo);
  if (currentChapter == -1) return;

  DevSetAddMode(myVideo, PlayerChapterMode);
  if (skipDir == Reverse)
  {
    DevPlayFromTo(myVideo, currentChapter, currentChapter,  30);	    /* Implements a search */
  }
  else 
  {
    skipDir = Reverse;
    DevPlayFromTo(myVideo, currentChapter+1, currentChapter+1,  30);
  }
  
  if (status == PlayerForwardPlay) DevPlay(myVideo);
  
  /* check if necessary to change back mode to frame */

  value = xv_get(VideoEdit_optionsPopup->optionsSearchStg, 
		 PANEL_VALUE);
  if (value == 0)
    DevSetAddMode(myVideo, PlayerFrameMode);
  
}

/*
 * Notify callback function for `fwdSkipButton'.
 * Skip to the next chapter
 */
void
ForwardSkip(item, event)
     Panel_item	item;
     Event		*event;
{
  skipDir = Forward;
  ReverseSkip(item, event);
}


/*
 * Notify callback function for `pauseButton'.
 * Puts the player into still mode on first call. 
 * On second call, the player goes back into playback mode.
 */
void
Pause(item, event)
     Panel_item	item;
     Event		*event;
{
  if (paused)
  {
    DevPlay(myVideo);
    paused = 0;
  }
  else
  {
    DevStill(myVideo);    
    paused = 1;
  }
 
}


/*
 * Notify callback function for `revStepButton'.
 * Also implements still.
 */
void
ReverseStep(item, event)
     Panel_item	item;
     Event		*event;
{
  DevStill(myVideo);
  DevStep(myVideo, Reverse);
}


/*
 * Notify callback function for `fwdStepButton'.
 * Also implements still.
 */
void
ForwardStep(item, event)
     Panel_item	item;
     Event		*event;
{
  DevStill(myVideo);
  DevStep(myVideo, Forward);
}


/*
 * Notify callback function for `speedSlider'.
 * To control multi-speed play
 */
void
Speed(item, value, event)
     Panel_item      item;
     int             value;
     Event           *event;
{
  char s[5];
  static int oldspeed;
  enum Direction direction;
  int framesPerSecond;
  
  oldspeed = 0;
  direction = Forward;

  /* put in still mode if mouse button is up */

  if (event_is_up(event)) 
  {
    oldspeed = 0;
    xv_set(item, PANEL_VALUE, 0, NULL);
    xv_set(VideoEdit_window1->speedTxt, PANEL_VALUE, " ", NULL);
    DevStill(myVideo);
    return;
  }
  
  /* check if forward or reverse play */
  
  if (value < 0) 
  {
    s[0] = '-';
    value = -value;
    direction = Reverse;
  }
  else 
    s[0] = ' ';
  
  framesPerSecond = DevCalcSpeed(myVideo, value, 0);
  sprintf (&s[1], "%d", framesPerSecond);
  
  if ((framesPerSecond != oldspeed) && (framesPerSecond))
  {
    oldspeed = value;
    xv_set(VideoEdit_window1->speedTxt, 
	   PANEL_VALUE, s, 
	   NULL);
    DevPlayAtSpeedDir(myVideo, framesPerSecond, direction);
  }
}



/*
 * Notify callback function for `quitButton'.
 * Quit from the whole program
 */
void
Quit(item, event)
     Panel_item	item;
     Event		*event;
{
  int result;
  
  hitQuit = 1;
  
  /* check if unsaved changes exist in edit list */
  if ((change) && (lines > 0)) 
  {
    result = notice_prompt(VideoEdit_window1->window1, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   "Unsaved changes exist in the edit list.",
			   "Go ahead and quit? ",
			   NULL,
			   NOTICE_BUTTON_NO,	"Yes",		    /* the labels are inverted to set the default to NO */
			   NOTICE_BUTTON_YES,	"No",
			   NULL);
    if (result == NOTICE_YES)
    {
      hitQuit = 0;
      return;
    }
    
  }

  if (sender)
    SenderDisconnectFromPortMgr(sender,&(receiver->receivePort));
  xv_destroy_safe(VideoEdit_window1->window1);
}


/* FUNCTIONS FOR INFO POPUP WINDOW */
/*
 * Notify callback function for `infoOKButton'.
 * Close the info popup window
 */
void
InfoDone(item, event)
     Panel_item      item;
     Event           *event;
{
  xv_set(VideoEdit_infoPopup->infoPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(VideoEdit_infoPopup->infoPopup, XV_SHOW, FALSE, NULL);
}


#ifdef MAIN

void
main(argc, argv)
     int	argc;
     char	**argv;
{
  Rect  tempRect;
  int i;
  int value;
  Display* display;
  char* defaultPlayer;
   
  /* Initialize XView. */ 

  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);
  INSTANCE = xv_unique_key();
  
  /* Command line parsing */

  strcpy(hostname, "localhost");
  startFilename = NULL;
  receiverPort = AnyPort;
  CheckOptions(argc, argv);
  
  /* Initialize user interface components. */
  
  VideoEdit_window1 = VideoEdit_window1_objects_initialize(NULL, NULL);
  VideoEdit_optionsPopup = VideoEdit_optionsPopup_objects_initialize(NULL, VideoEdit_window1->window1);
  VideoEdit_editPopup = VideoEdit_editPopup_objects_initialize(NULL, VideoEdit_window1->window1);
  VideoEdit_previewPopup = VideoEdit_previewPopup_objects_initialize(NULL, VideoEdit_window1->window1);
  VideoEdit_infoPopup = VideoEdit_infoPopup_objects_initialize(NULL, VideoEdit_window1->window1);

  /* Initialize position of main window */

  frame_get_rect(VideoEdit_window1->window1,&tempRect);	    /* Get the size of the top-level window */
  tempRect.r_top = 250;
  tempRect.r_left = 200;
  frame_set_rect(VideoEdit_window1->window1,&tempRect);	    /* Set the position of the top-level window */
    
  
  /* Initialize position of other windows */
  
  xv_set(VideoEdit_optionsPopup->optionsPopup, 
	 XV_X, 200,
	 XV_Y, 50,
	 NULL);
  xv_set(VideoEdit_editPopup->editPopup, 
	 XV_X, 450,
	 XV_Y, 250,
	 FRAME_DONE_PROC, EditDone,
	 NULL);
  xv_set(VideoEdit_previewPopup->previewPopup, 
	 XV_X, 850,
	 XV_Y, 525,
	 NULL);
  xv_set(VideoEdit_infoPopup->infoPopup, 
	 XV_X, 200,
	 XV_Y, 250,
	 NULL);

  /* Initialize other window attributes */

  if (startFilename)						    /* Load file at start-up if included in command line */
    Browse(startFilename, BrowseCheckOpen, 0,
	   "#Video Edit Document#", "VideoEdit");				
		 
  xv_set(VideoEdit_window1->window1, 
	 FRAME_LEFT_FOOTER, "Display : On",
	 FRAME_RIGHT_FOOTER, "Audio : Stereo",
	 NULL);
  xv_set(VideoEdit_optionsPopup->optionsAudioStg,		    /* Set audio to stereo */
	 PANEL_VALUE, 3, 
	 NULL);
  xv_set(VideoEdit_optionsPopup->optionsDisplayStg,		    /* Set address display to on */
	 PANEL_VALUE, 1, 
	 NULL);
  xv_set(VideoEdit_window1->speedSlider,			    
	 PANEL_NOTIFY_LEVEL, PANEL_ALL, 
	 NULL); 
  xv_set(VideoEdit_previewPopup->previewPlayFirstTxt,		    /* Set number of seconds to be previewed */
	 PANEL_VALUE, 4,					    /* to 4 */
	 NULL);
  xv_set(VideoEdit_previewPopup->previewPlayLastTxt, 
	 PANEL_VALUE, 4, 
	 NULL);
  xv_set(VideoEdit_editPopup->editModButton,			    /* set modify, delete buttons to inactive */
	 PANEL_INACTIVE, TRUE, 
	 NULL);
  xv_set(VideoEdit_editPopup->editDelButton, 
	 PANEL_INACTIVE, TRUE, 
	 NULL);
  xv_set(VideoEdit_editPopup->editSpeedTxt,			    /* set speed text to 30 */
	 PANEL_VALUE, 30, 
	 NULL);

  font = (Xv_font *)xv_find(VideoEdit_editPopup->editPopup,	    /* Set font for the panel list */
			    FONT,
			    FONT_FAMILY, FONT_FAMILY_LUCIDA_FIXEDWIDTH,
			    FONT_STYLE,  FONT_STYLE_NORMAL,
			    FONT_SIZE, 12,
			    NULL);
  
  InitNetwork(hostname);					    /* Set up protocol sender/receiver */

  display =  (Display*) xv_get(VideoEdit_window1->window1, XV_DISPLAY);
  defaultPlayer = XGetDefault(display, "VideoEdit", "defaultPlayer");

  xv_set(VideoEdit_optionsPopup->optionsPlayerStg, 
	 PANEL_NCHOICES, numDevices, NULL);

  value = 0;
  for (i=1; i < numDevices+1; i++)
  {
    xv_set(VideoEdit_optionsPopup->optionsPlayerStg,
	   PANEL_CHOICE_STRING, i, allDevices[i-1], 
	   NULL);
    if (defaultPlayer)
      if (strcmp(defaultPlayer, allDevices[i-1]) == 0)
	value = i+1;
  }

  if (value)
    xv_set(VideoEdit_optionsPopup->optionsPlayerStg,		    /* Set player setting to default player */
	   PANEL_VALUE, value-1,
	   NULL);
   
  if (defaultPlayer)
    SetPlayerHandler(NULL, NULL);			    /* Set up default player */
  else
    Options(NULL, NULL);					    /* Show options popup */
  
  CreateBrowse(OpenHandler, SaveHandler, 
		  VideoEdit_window1->window1);			    /* Set up Open Panel */

  xv_main_loop(VideoEdit_window1->window1);			    /* Turn control over to XView */
  
  if (!hitQuit)							    /* If used 'quit' on frame menu, call Quit */
    Quit(NULL, NULL);						    /* to simulate hitting Quit button. */
  
  exit(0);
}

#endif
