/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
/*
/*    file:      torusd.c
/*
/*    Purpose:   Interactive user interface command interpreter.
*/

#include "include.h"

int go_display_flag = 0; /* for displaying each iteration */

void old_menu()
{
  char response[140];
  body_id b_id;
  REAL total_volume;
  double val;  /* for scanf */
  int gocount = 0;        /* number of iterations to do at once */ 
  int i,n;
  int znum;
  char jchar;

restart:  /* for resetting jumpbuf after starting new file */
  if ( setjmp(jumpbuf) )   /* return here after recoverable errors */
    { if ( commandfd ) { fclose(commandfd); commandfd = NULL;} /* revert */
    }

  while ( 1 )  /* main event loop of program */
    {
	 free_discards(); /* from previous cycle */
         breakflag = 0;
         outstring("Enter command: ");
         memset(response,0,sizeof(response));
         getstring(response);        
	 if ( isalpha(response[0]) && isalpha(response[2]) )
	   { newcommand(response);
	     continue;
	   }
         switch ( response[0] )
         {
	  /* history list */
	  case '!': old_history(response);
		    break;

          /* Reporting */
          case 'C':  run_checks();
                     outstring("Checks completed.\n");
                     break;

          case 'c': /* report count of elements and status */
                     memory_report();
                     break;

          case 'e' : extrapolate(); break;

          case 'i':  /* i for information */
                     sprintf(msg,"Total energy: %17.15f\n",web.total_energy);
                     outstring(msg);
                     if ( fabs(web.spring_energy) > 1e-10 )
                       {
                         sprintf(msg,"Spring energy: %17.15f\n",web.spring_energy);
                         outstring(msg);
                       }
                     sprintf(msg,"Total area:   %17.15f\n",web.total_area);
                     outstring(msg);
                     memory_report();
                     if ( web.area_norm_flag ) 
                       { outstring("Area normalization ON.");
                         if ( web.norm_check_flag )
                           { sprintf(msg,"  Max normal change %4.2f\n",
                                web.norm_check_max);
                             outstring(msg);
                           }
                         else outstring("  Normal change checking OFF.\n");
                       }

                     if ( web.modeltype ==  LINEAR )
                       outstring("Representation: LINEAR\n");
                     else
                       outstring("Representation: QUADRATIC\n");

                     if ( web.motion_flag )
                       sprintf(msg,"Motion scale factor fixed at %f\n",
                           web.scale);
                     else
                       sprintf(msg,"Motion scale factor optimizing at %f; upper bound %f\n",
                           web.scale,web.maxscale);
                     outstring(msg);

                     if ( web.diffusion_flag )
                       { sprintf(msg,"Diffusion is ON; diffusion constant: %f\n,",  
                            web.diffusion_const);
                         outstring(msg);
                       }
                     else outstring("Diffusion is OFF.\n");

                     if ( web.gravflag )
                       sprintf(msg,"Gravity is ON; gravitational constant %f.\n",
                          web.grav_const);
                     else sprintf(msg,"Gravity is OFF.\n");
                     outstring(msg);

                     if ( web.jiggle_flag )
                       sprintf(msg,"Jiggling is ON; temperature is %f.\n",
                          web.temperature);
                     else
                       sprintf(msg,"Jiggling is OFF; temperature is %f.\n",
                          web.temperature);
                     outstring(msg);

                     sprintf(msg,"Convexity spring constant is %f.\n",
                         web.spring_constant); 
                     outstring(msg);

                     if ( web.pressure_flag )
                       { sprintf(msg,"Ambient pressure: %f\n",web.pressure);
                         outstring(msg);
                       }

                     total_volume = 0.0;
                     FOR_ALL_BODIES(b_id)
                       { if ( equal_id(b_id,web.outside_body) ) continue;
                         total_volume += get_body_volume(b_id);
                       }
                     if ( total_volume > 1e-20 )
                       { sprintf(msg,"Area^3/volume^2 figure of merit: %17.15f\n\n",
                             web.meritfactor*pow(web.total_area,3.0)/
                                            total_volume/total_volume);
                         outstring(msg);
                       }

                     break;

          case 'v' : /* show volumes and quantities */ 
                     if ( (web.bodycount == 0) && (web.quantity_count == 0) )
                       { outstring("No bodies or quantities.\n");
                         break;
                       }
                     if ( web.bodycount )
                      {
                       outstring("Body       target volume         actual volume        pressure\n");
                       FOR_ALL_BODIES(b_id)
                         { if ( equal_id(b_id,web.outside_body) ) continue;
                           sprintf(msg,"%3d     %17.15g     %17.15g   %17.15g\n",
                             1+ordinal(b_id),get_body_fixvol(b_id),
                             get_body_volume(b_id),get_body_pressure(b_id));
                           outstring(msg);
                          }
                      }
                     if ( web.quantity_count )
                       { outstring("Quantity   target value          actual value\n");
                         for ( i = 0 ; i < web.quantity_count ; i++ )
                           { struct quantity *quan = get_quant(i);
                             if ( !(quan->attr & IN_USE) ) continue;
                             if ( quan->attr & QFIXED )
                               sprintf(msg,"%3d     %17.15g    %17.15g\n",
                                i,quan->target,quan->value);
                             else
                               sprintf(msg,"%3d         ---------         %17.15g\n",
                                i,quan->value);
                             outstring(msg);
                           }
                       }
                     break;

          case 'z' : /* curvature test */
		     if ( web.simplex_flag )
		       outstring("Not implemented for simplex representation.\n");
                     curtest();
                     break;

          /* Controlling model characteristics */

          case 'A' : /* set adjustable parameters */
                     /* list */
                     outstring("Adjustable parameters: \n");
                     for ( n = 0 ; n < web.paramcount ; n++ )
                       { sprintf(msg,"%2d. %31.31s  %f\n",
                            n+1,web.params[n].name,web.params[n].value);
                         outstring(msg);
                       }
                     if ( logfd ) fprintf(logfd,"A\n");
                     /* ask for new values */
                     for(;;)
                       { int retval;

                         outstring("Number and new value: ");
                         getstring(response);
                         retval = sscanf(response,"%d %lf",&n,&val);
                         if ( retval == 2 )
                           { if ( (n > 0) && (n <= web.paramcount) )
                              { web.params[n-1].value = val;
                                if ( logfd ) 
                                  fprintf(logfd,"%d %17.15f\n",n,val);
                              }
                             else
                               outstring("Bad parameter number.");
                           }
                         else if ( retval == 1 )
                           outstring("Need number AND value.\n");
                         else break;
                       }
                     if ( logfd ) fprintf(logfd,"\n");
                     convert_all_expr();
                     recalc_verts();
                     calc_energy();
                     if ( OOGL_flag ) UpdateOOGL();
                     break;


          case 'a' : /* toggle area normalization of force */
                     web.area_norm_flag = !web.area_norm_flag;
                     if ( logfd ) fprintf(logfd,"a\n");
                     if ( web.area_norm_flag ) 
                       { outstring("Area normalization ON.\n");
                         outstring("Enter maximum fraction facet normal change: ");
                         getstring(response);
                         if ( logfd ) fprintf(logfd,"%s\n",response);
                         if ( sscanf(response,"%lf",&val) == 1 )
                          if ( val > 0.0000001 )
                           { web.norm_check_flag = 1;
                             web.norm_check_max = val;
                           }
                          else web.norm_check_flag = 0;
                         calc_energy();  /* to make sure vertex areas set */
                       }
                     else outstring("Area normalization OFF.\n");
                     break;

          case 'b' :
                     if ( logfd ) fprintf(logfd,"b\n");
                     { body_id b_id;
                       double pp;

                       FOR_ALL_BODIES(b_id)
                         { 
                           if ( get_battr(b_id) & PRESSURE )
                             {
                               pp = get_body_pressure(b_id);
                               sprintf(msg,"Body %d. Current pressure %f.  Enter new: ",
                                  ordinal(b_id)+1,pp);
                               outstring(msg);
                               getstring(response);
                               if ( logfd ) fprintf(logfd,"%s\n",response);
                               if ( sscanf(response,"%lf",&pp) == 1 )
                                 set_body_pressure(b_id,pp);
                              }
                            else /* edit volumes */
                              { 
                               pp = get_body_fixvol(b_id);
                               sprintf(msg,"Body %d. Current target volume %f.  Enter new: ",
                                  ordinal(b_id)+1,pp);
                               outstring(msg);
                               getstring(response);
                               if ( logfd ) fprintf(logfd,"%s\n",response);
                               if ( sscanf(response,"%lf",&pp) == 1 )
                                 { set_body_fixvol(b_id,pp);
				   set_attr(b_id,FIXEDVOL);
				 }
                              }
                         }
                      }
                     break;
 
          case 'f' : /* Set diffusion */
                     if ( logfd ) fprintf(logfd,"f\n");
                     sprintf(msg,"Diffusion constant is %f.  Enter new: ",
                         web.diffusion_const); 
                     outstring(msg);
                     getstring(response);
                     if ( logfd ) fprintf(logfd,"%s\n",response);
                     if ( sscanf(response,"%lf",&val) == 1 )
                       web.diffusion_const = val;
                     if ( web.diffusion_const < 1e-20 )
                        web.diffusion_flag = 0;
                     else web.diffusion_flag = 1;
                     break;

          case 'G' : /* control gravity */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"G\n");
                     if ( web.gravflag )
                       sprintf(msg,"Gravity is now ON with gravitational constant %f.\n",
                          web.grav_const);
                     else sprintf(msg,"Gravity is now OFF.\n");
                     outstring(msg);
                     outstring("Enter new constant (0 for OFF): ");
                     getstring(response);
                     if ( logfd ) fprintf(logfd,"%s\n",response);
                     if ( sscanf(response,"%lf",&val)  == 1)
                       web.grav_const = val;
                     if ( fabs(web.grav_const) > 1e-12 ) web.gravflag = 1;
                     else web.gravflag = 0;
                     convert_all_expr();
                     calc_energy();
                     break;

          case 'J' : /* toggle jiggling on every move */
                     if ( logfd ) fprintf(logfd,"J\n");
                     web.jiggle_flag = !web.jiggle_flag;
                     if ( web.jiggle_flag ) 
                       { outstring("Now jiggling on every move.\n");
                         sprintf(msg,
                           "Enter temperature for jiggling (default %f): ",
                             web.temperature);
                         outstring(msg);
                         getstring(response);
                         if ( logfd ) fprintf(logfd,"%s\n",response);
                         if ( sscanf(response,"%lf",&val) == 1 )
                           web.temperature = val;
                       }
                     else outstring("Jiggling on every move disabled.\n");
                     break;

          case 'k' : /* Magnitude of force opposing boundary short-circuiting */
                     if ( logfd ) fprintf(logfd,"k\n");
                     sprintf(msg,"Spring constant is %f.  Enter new: ",
                         web.spring_constant); 
                     outstring(msg);
                     getstring(response);
                     if ( logfd ) fprintf(logfd,"%s\n",response);
                     if ( sscanf(response,"%lf",&val) == 1 )
                       web.spring_constant = val;
                     if ( fabs(web.spring_constant) < 1e-12 )
                       web.convex_flag = 0;
                     else
                       web.convex_flag = 1;
                     calc_energy();
                     break;

          case 'M' : /* change model type */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"M\n");
                     change_model();
                     break;

          case 'm' : /* Setting motion scale factor */
                     if ( logfd ) fprintf(logfd,"m\n");
                     web.motion_flag = !web.motion_flag;
                     if ( web.motion_flag )
                      {
                        sprintf(msg,"Enter scale factor (%f): ",web.scale);
                        outstring(msg);
                        getstring(response);
                        if ( logfd ) fprintf(logfd,"%s\n",response);
                        if ( sscanf(response,"%lf",&val) == 1 )
                          web.scale = val;
                      }
                     else
                      { energy_init = 0;
                        sprintf(msg,"Scale optimizing. Enter scale limit (%f): ",web.maxscale);
                        outstring(msg);
                        getstring(response);
                        if ( logfd ) fprintf(logfd,"%s\n",response);
                        if ( sscanf(response,"%lf",&val) == 1 )
                          web.maxscale = val;
                      }
                     break;

          case 'p' : /* Ambient pressure for compressible volumes */ 
                     if ( logfd ) fprintf(logfd,"p\n");
                     if ( web.bodycount == 0 )
                       { outstring("Can't do pressure without bodies.\n");
                         break;
                       }
                     sprintf(msg,"Pressure now %f\n",web.pressure);
                     outstring(msg);
                     outstring("Enter pressure (0 for rigid volumes): ");
                     getstring(response);        
                     if ( logfd ) fprintf(logfd,"%s\n",response);
                     if ( sscanf(response,"%lf",&val) ==  1)
                       web.pressure = val;
                     if ( web.pressure > 0.00000001 )
                       {
                         if ( !web.full_flag && !valid_id(web.outside_body) )
                           add_outside();
                         web.projection_flag = 0;
                         web.pressure_flag = 1;
                       }
                     else
                       {
                         web.projection_flag = 1;
                         web.pressure_flag = 0;
                       }
                     calc_pressure();
                     calc_energy();
                     outstring(msg);
                     break;

           case 'Q': /* quantities */

                     if ( web.quantity_count )
                       { outstring("Quantity   target value          actual value\n");
                         for ( i = 0 ; i < web.quantity_count ; i++ )
                           { struct quantity *quan = get_quant(i);
                             if ( !(quan->attr & IN_USE) ) continue;
                             if ( quan->attr & QFIXED )
                               sprintf(msg,"%3d     %17.15g    %17.15g\n",
                                i,quan->target,quan->value);
                             else
                               sprintf(msg,"%3d         ---------         %17.15g\n",
                                i,quan->value);
                             outstring(msg);
                           }
                       }
                     for(;;)
                      { char *turn;
                        struct quantity *quan;
                        outstring("Change (number ON|OFF [value]): ");
                        getstring(response);
                        if ( response[0] == '\0' ) break; 
                        i = atoi(strtok(response," \t,:;"));
                        if ( (i < 0)||(i >= web.quantity_count)||
                              !((quan=get_quant(i))->attr & IN_USE) )
                          { outstring("Invalid quantity number.\n");
                            continue;
                          }
                        turn = strtok(NULL," \t,;:");
                        strupr(turn);
                        if ( strcmp(turn,"ON") == 0 )
                          { quan->attr |= QFIXED;
                            turn = strtok(NULL," \t;,:");
                            if ( turn == NULL )
                              { outstring("Missing value.\n");
                                continue;
                              }
                            quan->target = atof(turn);
                          }
                        else if ( strcmp(turn,"OFF") == 0 )
                          quan->attr &= ~QFIXED;
                        else 
                          { outstring("Missing ON or OFF.\n");
                            continue;
                          }
                       }
                     break;
           case 'W': /* homothety toggle */
                     if ( logfd ) fprintf(logfd,"W\n");
                     web.homothety = !web.homothety;
                     sprintf(msg,"Homothety adjustment is %s.\n",
                       web.homothety ? "ON" : "OFF");
                     outstring(msg);
                     break;

          /* Surface modification */

          case 'g' : gocount = atoi(response+1);
                     if ( logfd ) fprintf(logfd,"%s\n",response);
                     if ( gocount == 0 ) gocount = 1;
                     for ( ; gocount ; gocount-- ) 
                       { if ( breakflag ) break;
                          iterate();
                         if ( go_display_flag )
                           display();
                       }
                     break;

         case 'j' : /* jiggling */
                     if ( logfd ) fprintf(logfd,"%s\n",response);
                     jchar = response[1];  /* save for long jiggle test */
                     if ( jchar == 'j' )
                       { long_jiggle();
                       }
                     else
                       { jiggle();
                       }
                     energy_init = 0;
                     recalc_verts();
                     calc_content();
                     calc_pressure();
                     calc_energy();
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

          case 'l' : /* long edge subdivide */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"l\n");
                     for (;;)
                       {
                         outstring("Enter maximum edge length desired(h for histogram): ");
                         getstring(response);
                         if ( logfd ) fprintf(logfd,"%s\n",response);
                         if ( tolower(response[0]) == 'h' )
                           edge_histogram();
                         else break;
                       }
                     web.max_len = 0.0;
                     if ( sscanf(response,"%lf",&val) == 1 )
                       { web.max_len = val;
                         if ( web.max_len <= 0.0 ) break;
                         sprintf(msg,"New facets: %d\n",articulate(web.max_len));
                         outstring(msg);
                         if ( OOGL_flag ) UpdateOOGL();
                       }
                     break;

          case 'L' : /* lower bound estimate */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"L\n");
                     lowbound();
                     break;

          case 'N' : /* Normalize target volumes to current volumes */
                     if ( logfd ) fprintf(logfd,"N\n");
                     FOR_ALL_BODIES(b_id)
                       set_body_fixvol(b_id,get_body_volume(b_id));
                     outstring("Target volumes adjusted.\n");
                     break;

          case 'n' : /* notching ridges and valleys */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"n\n");
                     for (;;)
                       {
                         outstring("Enter maximum angle(radians) between normals(h for histogram): "); 
                         getstring(response);
                         if ( logfd ) fprintf(logfd,"%s\n",response);
                         if ( tolower(response[0]) == 'h' )
                           ridge_histogram();
                         else break;
                       }
                     if ( sscanf(response,"%lf",&val) == 1)
                       { web.max_angle = val;
                         if ( web.max_angle <= 0.0 ) break;
                         sprintf(msg,"Number of edges notched: %d\n",
                                       ridge_notcher(web.max_angle));
                         outstring(msg);
                         energy_init = 0;
                         if ( OOGL_flag ) UpdateOOGL();
                       }
                     break;

          case 'O' : /* pop nonminimal edges */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"O\n");
                     if ( web.dimension == STRING )
                       sprintf(msg,"Number of vertices popped: %d\n",
                                                             verpop_str());
                     else
                       sprintf(msg,"Number of vertices popped: %d\n",
                                                               edgepop_film());
                     outstring(msg);
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

          case 'o' : /* pop nonminimal vertices */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"o\n");
                     if ( web.dimension == STRING )
                       sprintf(msg,"Number of vertices popped: %d\n",
                                                             verpop_str());
                     else
                       sprintf(msg,"Number of vertices popped: %d\n",
                                                               popfilm());
                     outstring(msg);
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

          case 'r' : refine(); energy_init = 0;
                     if ( logfd ) fprintf(logfd,"r\n");
                     memory_report();
                     web.min_area /= 4;
                     web.max_len /= 2;
                     web.min_length /= 2;
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

          case 't' : /* tiny edge subdivide */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"t\n");
                     for (;;)
                       {
                         outstring("Enter minimum edge length desired(h for histogram): ");
                         getstring(response);
                         if ( logfd ) fprintf(logfd,"%s\n",response);
                         if ( tolower(response[0]) == 'h' )
                           edge_histogram();
                         else break;
                       }
                     web.min_length = 0.0;
                     if ( sscanf(response,"%lf",&val) == 1 )
                       web.min_length = val;
                     else break;
                     if ( web.min_length <= 0.0 ) break;
                     sprintf(msg,"Deleted edges: %d\n",edgeweed(web.min_length));
                     outstring(msg);
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

	  case 'U' : /* conjugate gradient */
		     if ( conj_grad_flag )
		       { if ( cg_hvector ) free((char *)cg_hvector);
			 cg_hvector = NULL;
			 cg_oldsum = 0.0;
			 conj_grad_flag = 0;
			 outstring("Conjugate gradient now OFF.\n");
	               }
		     else
		       { conj_grad_flag = 1;
			 outstring("Conjugate gradient now ON.\n");
		       }
                     break;

          case 'u' : /* equiangulate */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"u\n");
                     sprintf(msg,"Edges switched in equiangulation: %d\n",
                       equiangulate() );
                     outstring(msg);
                     energy_init = 0;
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

          case 'V' : /* move vertex to average of neighbors */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"V\n");
                     vertex_average();
                     if ( OOGL_flag ) UpdateOOGL();
                     outstring("Vertex averaging done.\n");
                     break;

          case 'w' : /* ameliorate triangulation */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"w\n");
                     for (;;)
                       {
                         outstring("Enter minimum area desired(h for histogram): ");
                         getstring(response);
                         if ( logfd ) fprintf(logfd,"%s\n",response);
                         if ( tolower(response[0]) == 'h' )
                           area_histogram();
                         else break;
                       }
                     web.min_area = 0.0;
                     if ( sscanf(response,"%lf",&val) == 1)
                       web.min_area = val;
                     else break; 
                     if ( web.min_area <= 0.0 ) break;
                     sprintf(msg,"Skinny triangles weeded: %d\n",
                         areaweed(web.min_area));
                     outstring(msg);
                     energy_init = 0;
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

          case 'y' : /* torus duplication */
		     if ( web.simplex_flag )
		       { outstring(
			  "Not implemented for simplex representation.\n");
			 break;
		       }
                     if ( logfd ) fprintf(logfd,"y\n");
                     if ( ! web.torus_flag )
                       { outstring("Torus model not in effect.\n");
                         break;
                       }
                     outstring("Duplicate which period(1,2,3)? ");
                     getstring(response);
                     if ( logfd ) fprintf(logfd,"%s\n",response);
                     i = atoi(response);
                     if ( i < 1 || i > web.space_dimension )
                       outstring("Improper period number.\n");
                     else tordup(i-1);
                     if ( OOGL_flag ) UpdateOOGL();
                     break;                  
                     
          case 'Z' : /* zooming in on vertex */
            if ( logfd ) fprintf(logfd,"Z\n");
            znum = ordinal(web.zoom_v) + 1;
            sprintf(msg,"Enter zoom vertex number (%d): ",znum);
            outstring(msg);
            getstring(response);
            if ( logfd ) fprintf(logfd,"%s\n",response);
            sscanf(response,"%d",&znum);
            /* check vertex still exists */
              { vertex_id v_id;
                int found = 0;
                FOR_ALL_VERTICES(v_id)
                 { if ( znum == (ordinal(v_id)+1) )
                     { web.zoom_v = v_id; found = 1; break; }
                 }
                if ( !found )
                 { error("Vertex not found.\n",WARNING);
                   break;
                 }
              }
            sprintf(msg,"Enter cut-off radius (%f): ",web.zoom_radius);
            outstring(msg);
            getstring(response);
            if ( logfd ) fprintf(logfd,"%s\n",response);
            if ( sscanf(response,"%lf",&val) == 1)
             { web.zoom_radius = val;
               zoom_vertex(web.zoom_v,web.zoom_radius);
	       resize();
             }
            else break;
            if ( OOGL_flag ) UpdateOOGL();
            break;

          case '1' : /* global adjust */
                     if ( logfd ) fprintf(logfd,"1\n");
                     global_adjust();
             calc_content();
             calc_pressure();
             calc_energy();  /* energy after motion */
             if ( OOGL_flag ) UpdateOOGL();
                     break;

          /* Graphical output */

          case 'D' : go_display_flag = !go_display_flag; 
                     if ( go_display_flag )
                       outstring("Automatic display ON.\n");
                     else
                       outstring("Automatic display OFF.\n");
                     break;
          case 'd' : if ( response[1] == 'f' ) dump_force();
                     else dump(); 
                     break;
          case 'P' : /* create graphics data file */
                     display_file();
                     break;
          case 's' : /* Show image */ 
		     command("show  everything where 1");
                     /* torshow();*/
		     break;


          /* Process control */

          case 'F' : /* log commands */
                     if ( logfd ) 
                       { fclose(logfd);
                         logfd = NULL;
                         outstring("Command logging OFF.\n");
                         break;
                       }
                     outstring("Name of log file: ");
                     getstring(response);
                     logfd = fopen(response,"w");
                     if ( logfd == NULL )
                       { perror(response); break; }
                     break;        

          case 'S' : /* save binary form */
                     outstring("Binary save not currently functional.\nUse 'D' Ascii dump.\n"); 
                     break;
                    /*  save(); break; */

          case 'R' :  /* restore(NULL);  */
                     outstring("R (restore) command discontinued.\n");
                     if ( OOGL_flag ) UpdateOOGL();
                     break;

          case 'x' : 
          case 'q' : /* Exit */
                     if ( logfd ) 
                       { fclose(logfd);
                         logfd = NULL;
                         outstring("Command logging OFF.\n");
                       }
                     startup(NULL);
                     goto restart;  /* to top of this function */

          case '?' :
          case 'H' :
          case 'h' : /* help screen */
                     main_help();
                     break;

          default  : if ( response[0] != '\0' )
                      { sprintf(msg,"Illegal command: %c. Type h for help.\n",
                          response[0]);
                        outstring(msg);
                      }
                     break;
         }
    }
}

/********************************************************************
*
*  Function: extrapolate()
*
*  Purpose:  prints extrapolation of results to infinite resolution 
*
*/

void extrapolate()
{
  int m;
  REAL d1,d2,ext;

  for ( m = 0 ; m <= level ; m++ )
    { sprintf(msg,"refinement: %3d  area: %20.15f  ",m,extrap_val[m]);
      outstring(msg);
      if ( m > 1 ) /* can extrapolate */
        { d1 = extrap_val[m-1] - extrap_val[m-2];
          d2 = extrap_val[m] - extrap_val[m-1];
          ext = extrap_val[m] - d2*d2/(d2 - d1);
          sprintf(msg,"extrapolation: %20.15f\n",ext);
          outstring(msg);
        }
      else outstring("\n");
    }
}





/*****************************************************************
*
*  Function: save()
*
*  Purpose: Save torusd internal state.  Will ask for filename
*           to save in.  Stamps file with version number
*           to prevent bad restores.
*
*/

#ifdef BINARYSAVE
static char savename[100];

void save()
{
  do    
   {
     outstring("Enter name of file to save state in: ");
     getstring(savename);
     savefd = fopen(savename,"wb");
     if ( savefd == NULL )
       { 
         sprintf(msg,"Cannot open file %s.\n",savename);
         error(msg,RECOVERABLE);
       }
   }
  while ( savefd == NULL );
  do_save();
}


void restore(name)
char *name;
{
  do    
   {
     if ( name == NULL ) 
       { outstring("Enter name of file to restore state from: ");
         getstring(savename);
         name = savename;
       }
     savefd = fopen(name,"rb");
     if ( savefd == NULL )
       { 
         sprintf(msg,"Cannot open file %s.\n",savename);
         error(msg,RECOVERABLE);
       }
     name = NULL;
   }
  while ( savefd == NULL );
  do_restore();
}

#endif
/*****************************************************************
*
*  function: recalc_verts()
*
*  purpose:  recalculate vertex coordinates after boundaries or
*            constraints changed.
*
*/

void recalc_verts()
{
  vertex_id v_id;

  FOR_ALL_VERTICES(v_id)
   {
     if ( get_vattr(v_id) & BOUNDARY )
       { struct boundary *boundary = get_boundary(v_id);
         REAL *param = get_param(v_id);
         REAL *x = get_coord(v_id);
         int j;
         
         for ( j = 0 ; j < web.space_dimension ; j++ )
          x[j] = eval(boundary->coordf[j],param);
       }
     if ( get_vattr(v_id) & CONSTRAINT )
          project_v_constr(v_id);
   }
}

