#include <stdio.h>
#include "cmap.h"

int lastbox = -1;  /* Current box & selected control point, or -1 if none */
int lastpt = -1;

void
reset_boxes()
{
    int		box;

    initLut();

    for (box = 0; box < NBOX; box++) {
	register struct cval *c = &cbox[box].cval[0];

	cbox[box].count = 2;
	c->x = 0;  c->y = 0; c->flags = LOCK_X;  c->pump = 0;
	c++;
	c->x = XSCALE-1;  c->y = YSCALE-1;  c->flags = LOCK_X;  c->pump = 0;
	boxClear(box, 0, XSCALE-1);
	redo(box, 0, 2-1, RECALC|REDOLUT);
    }
}

/*
 * Handle an attempt to pick a control point
 */
void
box_point(box, x, y, flags)
    int box;
    int x, y;	/* Internal coords */
    int flags;	/* unused for now */
{
    int i;
    register struct cval *c;

    i = find_point(box, x, y, 0);
    lastbox = box;		/* remember selection, for box_drag etc. */
    lastpt = i;

    boxAnnounce(box, i);
}

void
box_spot(box, x, y, flags)
	int box;
	int x, y;	/* Internal coordinates of picked spot */
	int flags;	/* 0=>normal, 1=>don't deselect point */
{
	int v;

	if(x < 0 || x >= XSCALE || y < 0 || y > YSCALE)
		return;	/* Forget it if out of bounds */
	
	if((v = find_point(box, x, y, 0)) < 0) {
	    if(flags == 0) {
		v = eval_box(x, box);
		boxMessage(box, "at x%d y%d f(x)%d",
			ItoUX(x, box), ItoUY(y, box), ItoUY(v, box));
		pick_cbar(x, -1, 0);
		lastpt = -1;
	    }
	} else {
		lastpt = v;
		lastbox = box;
		boxAnnounce(box, v);
	}
}

void
box_drag(box, ix, iy, flags)
	int box;	
	int ix, iy;
	int flags;
{
	if(lastbox != box)
		lastpt = -1;
	if(lastpt < 0 || lastpt >= cbox[box].count)
		return;

	move_point(box, lastpt, ix, iy, flags);
}

/*
 * Find a control point given its graphics position.
 * Returns point number,
 * or -1 if no such point appears within 1/32 of the display.
 */
int
find_point(box, x, y, flags)
    int box;
    int x, y;	/* In internal coords */
    int flags;	/* unused for now */
{
    int i, besti, n;
    int bestd;
    register struct cval *c;
    register int dx, dy;

    c = &cbox[box].cval[0];
    bestd = YSCALE/32;	/* Worst acceptable */
    besti = -1;
    n = cbox[box].count;
    for(i = 0; i < n; i++, c++) {
	dx = x - c->x;  if(dx < 0) dx = -dx;
	dy = y - c->y;  if(dy < 0) dy = -dy;
	if(bestd > dx*YSCALE/XSCALE + dy) {
	    bestd = dx*(YSCALE/XSCALE) + dy;
	    besti = i;
	}
    }
    return(besti);
}

void
box_lock_x(box, str)
	int box;
	char *str;
{
	register struct cval *c;

	if(lastpt < 0 || lastbox != box) {
		boxMessage(box, "Select a point first");
		lastpt = -1;
	} else if(lastpt == 0 || lastpt == cbox[box].count-1) {
		boxMessage(box, "Can't unlock end points");
	} else {
		c = &cbox[box].cval[lastpt];
		c->flags ^= LOCK_X;
		boxAnnounce(box, lastpt);
	}
}

void
box_pump(box, incr)
	int box;
	int incr;
{
	if(lastbox != box || lastpt < 0 || lastpt >= cbox[box].count) {
		boxMessage(box, "Select a point first");
		return;
	}
	redo(box, lastpt, lastpt, UNDRAW);
	cbox[box].cval[lastpt].pump += incr;
	redo(box, lastpt, lastpt, RECALC|REDOLUT);
}

void
box_delete(box, str)
	int box;
	char *str;
{
	if(lastpt < 0 || lastbox != box) {
		boxMessage(box, "Select a point first");
	} else {
		del_point(box, lastpt);
		lastpt = -1;
	}
}

void
box_setadd(box, str)
	int box;
	char *str;
{
	int x, y;
	int moveit;

	/* Magic!  Read the contents of the dialog box (which the user
	 * may have edited).  3 cases:
	 * "x %d y %d ..." -- from box_spot, this is the last place the
	 *			user pointed at; add a control point.
	 * "at x %d y %d ..." -- from box_point, this is the last
	 *			control point.  Presumably the user has
	 *			edited the X and/or Y coords.
	 *			Set the control point to be there.
	 * "%d %d" --
	 *		If we had a point selected, move it there.
	 *		If not, add a new one at that position.
	 */
	if(lastbox != box)
		lastpt = -1;
	if(sscanf(str, "at x %d y %d", &x, &y) >= 2) {
		moveit = 0;
	} else if(sscanf(str, "x %d y %d", &x, &y) >= 2) {
		moveit = 1;
	} else if(sscanf(str, "%d %d", &x, &y) >= 2) {
		moveit = (lastpt >= 0);
	} else {
		return;
	}
	x = UtoIX(x, box);
	y = UtoIY(y, box);
	if(moveit == 0)
		add_point(box, x, y, 0);
	else if(lastpt >= 0) {
		register struct cval *c = &cbox[box].cval[lastpt];

		c->flags &= ~(LOCK_X|LOCK_Y);		/* unlock for move */
		move_point(box, lastpt, x, y, 1);
		c->flags |= LOCK_X;
		boxAnnounce(box, lastpt);
	}
}

void
pick_cbar(ix, iy, flags)
	int ix, iy;
	int flags;
{
	struct tuple t;
	char buf[64];
	char *wow;

	map_to_tuple(ix, &t);
	sprintf(buf, "#%d = r%d g%d b%d a%d",
		ItoUX(ix, 3),
		ItoUY(t.v[0], 3),
		ItoUY(t.v[1], 3),
		ItoUY(t.v[2], 3),
		ItoUY(t.v[3], 3));
	switch(colorscheme) {
	case HSV:  wow = " (h%d s%d v%d)"; break;
	case YIQ:  wow = " (y%d i%d q%d)"; break;
	default:   wow = "";
	}
	sprintf(buf + strlen(buf), wow,
	    ItoUY(eval_box(ix, 0), 0),
	    ItoUY(eval_box(ix, 1), 1),
	    ItoUY(eval_box(ix, 2), 2));
	panelMessage("%s", buf);

	if(flags == 1) {
		/* we're called this way when the rightmouse is pressed */
		/* We create a locked-X point on each box */
	    register int box;

	    for(box = 0; box < 4; box++) {
		add_point(box, ix, eval_box(ix, box), LOCK_X);
	    }
	}
}
