/*

Copyright 1987 by Rensselaer Polytechnic Institute

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of RPI not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.
RPI makes no representations about the suitability of
this software for any purpose.  It is provided "as is"
without express or implied warranty.

*/
/*****************************************************************************
**
**			query.c
**
**  function to query for a single variable
**
**
*****************************************************************************/
#include "sgmp.h"			/* the general include file */
#include "sgmplookup.h"			/* application include file */

jmp_buf env;				/* environment for timeouts */

extern char *recv_sgmp_mesg();

query(gwaddr,session_id,var,timeout,maxretries)
long gwaddr;				/* gateway to query */
char *session_id;			/* session id to use in query */
char *var;				/* variable to be queried for */
int timeout;				/* timeout */
int maxretries;				/* maximum number of times to retry */
{ struct itimerval interval;		/* timeout interval */
  struct itimerval disable;		/* to disable timer */
  int req_id;				/* request id for query */
  strng *queryvar;			/* variable name to be queried for */
  int i;				/* general index/counter */
  get_req_msg_type *msg;		/* request message */
  get_rsp_msg_type *reply;		/* response to request message */
  int rc;				/* return code */
  int rreq_id;				/* request id of reply */
  long from;				/* source of reply */
  int type;				/* reply is this type of message */
  strng *name;				/* name of variable after conversion */
  int onintr();				/* interrupt handler for recv timeout */
  int retries;				/* number of times to retry */
/* check that we have a gateway, and a session name */
  if(gwaddr == 0)			/* no gateway address */
   { fprintf(stderr,"no gateway to query\n");
     return(-1);
   }
  if(session_id == NULL)		/* no session id */
   { fprintf(stderr,"no session name specified for query\n");
     return(-1);
   }
/* set up timing stuff */
  interval.it_interval.tv_sec = 0;	/* don't want to reload timer */
  interval.it_interval.tv_usec = 0;
  interval.it_value.tv_sec = timeout;	/* set timeout */
  interval.it_value.tv_usec = 0;
  disable.it_interval.tv_sec = 0;	/* to disable timer */
  disable.it_interval.tv_usec = 0;
  disable.it_value.tv_sec = 0;
  disable.it_value.tv_usec = 0;
/* just in case variable had a dot notation suffix */
  for(i = 0; i < strlen(var); i++)
   if(var[i] == ' ')
    var[i] = '_';
/* set up variable name to be queried */
  if((queryvar = (strng *)malloc(sizeof(strng))) == NULL)
   { fprintf(stderr,"memory allocation failure in setting up variable\n");
     return(-1);
   }
  queryvar->len = strlen(var);
  if((queryvar->str = (char *)malloc(queryvar->len + 2)) == NULL)
   { fprintf(stderr,"memory allocation failure in setting up variable\n");
     return(-1);
   }
  sprintf(queryvar->str,"%s",var);
/* convert variable name from symbolic to numeric */
  if((queryvar = sgmp_var_cvt(queryvar,SYMTONUM)) == NULL)
   { fprintf(stderr,"couldn't convert symbolic variable to numeric\n");
     return(-1);
   }
/* create a request message */
  req_id = getpid() ^ (int)time(0); 	/* create a request id */
  if((msg=(get_req_msg_type *)create_sgmp_req(req_id,1,&queryvar))==NULL)
   { fprintf(stderr,"cannot create a request message\n");
     return(-1);
   }
/* set things up for timeouts, preserve environment etc. */
  retries = 0;				/* we haven't retried yet */
  if(setjmp(env))
   if(retries < maxretries)		/* if we should retry again */
    { fprintf(stderr,"receive timed out. Retrying....\n");
      retries++;
    }
   else					/* we exceeded number of retries */
    { fprintf(stderr,"giving up\n");
      term_sgmp_comm(req_id);
      return(-1);			/* continue with querying */
    }
/* OK, we have a request message, now send it off */
  signal(SIGALRM,SIG_IGN);		/* ignore timeouts here */
  if((rc = send_sgmp_mesg(GET_REQ_MESG,req_id,gwaddr,SERVICE,msg,
    			  session_id,strlen(session_id),timeout)) < 0)
  switch(rc) {				/* find out what error occurred */
   case SEND_TIMEOUT:			/* timed out */
    fprintf(stderr,"timeout occurred in send\n");
    return(-1);
   case GEN_ERR:			/* generic error occurred in send */
    fprintf(stderr,"generic error occurred in send\n");
    return(-1);
  }
/* prepare to receive */
  rreq_id = req_id;			/* only want to hear this */
  signal(SIGALRM,onintr);		/* want to hear timeouts now */
  setitimer(ITIMER_REAL,&interval,NULL); /* start timer */
  reply = (get_rsp_msg_type *)recv_sgmp_mesg(&from,NULL,&rreq_id,&type,
				             &session_id);
  setitimer(ITIMER_REAL,&disable,NULL);	/* disable timer */
  term_sgmp_comm(req_id);		/* close this session */
  if(reply == NULL)			/* error in reply */
   { fprintf(stderr,"reply returned NULL\n");
     return(-1);
   }     
  if(type != GET_RSP_MESG)		/* if we didn't get a response mesg */
   { fprintf(stderr,"response was not a response message\n");
     return(-1);
   }
  if(reply->error_status != 0)		/* if we got an error */
   { fprintf(stderr,"response error, err index= %d\n",reply->error_index);
     return(-1);
   }
/* just being paranoid here */
  if(rreq_id != req_id)			/* if got back wrong message */
   { fprintf(stderr,"got back unsolicited response\n");
     return(-1);
   }     
/* everything all right. Print out response */
/* we know that variables represented symbolically start with '_' */
  if((reply->var_op_list->var_name->str)[0] == '_')  /* symbolic? */
   if((name=sgmp_var_cvt(reply->var_op_list->var_name,CVTIPADD)) == NULL)
    { fprintf(stderr,"couldn't convert symbolic variable name\n");
      return(-1);
    }
   else					/* we could convert name */
    printf("%s\t",name->str);
  else				/* name is in symbolic form */
   if((name=sgmp_var_cvt(reply->var_op_list->var_name,NUMTOSYM)) == NULL)
    { fprintf(stderr,"couldn't convert numeric variable name in reply\n");
      fprintf(stderr,"reply was: ");
      for(i=0; i< reply->var_op_list->var_name->len; i++)
       fprintf(stderr,"%x ",(((reply->var_op_list->var_name->str)[i]) & 0x000000ff));
      fprintf(stderr,"\n");
      return(-1);
    }
   else				/* we could convert name */
    printf("%s\t",name->str);
  switch(reply->var_op_list->var_value.type) { /* decide what value is */
   case INTEGER:			/* value is an integer */
    printf("0x%x",reply->var_op_list->var_value.value.intgr);
    printf("\t\t%d",reply->var_op_list->var_value.value.intgr);
    break;
   case PRIM_STR:			/* a primitive string */
/* kludge because internet addresses can be returned as strings */
    if(isalnum((reply->var_op_list->var_value.value.str->str)[0]))
     printf("%s",reply->var_op_list->var_value.value.str->str);
    else
     { long address;
       ((char *)(&address))[0] = reply->var_op_list->var_value.value.str->str[0];
       ((char *)(&address))[1] = reply->var_op_list->var_value.value.str->str[1];
       ((char *)(&address))[2] = reply->var_op_list->var_value.value.str->str[2];
       ((char *)(&address))[3] = reply->var_op_list->var_value.value.str->str[3];
       printf("%s",stdtodot(ntohl(address)));
     }
    break;
   default:			/* unknown value type */
    printf("Value returned was of unknown type\n");
    break;
  }
  printf("\n");
 return(0);
}

/* onintr - process recv timeouts */
onintr()
{
  longjmp(env);
}
