/*

Copyright 1987 by Rensselaer Polytechnic Institute

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of RPI not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.
RPI makes no representations about the suitability of
this software for any purpose.  It is provided "as is"
without express or implied warranty.

*/
/******************************************************************************
**
**			mquery.c
**
** function to query a gateway for all instances of a variable class
**
**
******************************************************************************/
#include "sgmp.h"		/* the general include file */
#include "sgmplookup.h"		/* application specific include file */

jmp_buf env;			/* environment for timeouts */

mquery(gwaddr,session_id,var_class,timeout,maxretries)
long gwaddr;			/* address of gateway to query */
char *session_id;		/* session id to use in query */
char *var_class;		/* the prefix common to all variables */
int timeout;			/* timeout */
int maxretries;			/* number of times to retry */
{ int onrintr();		/* the interrupt handler for recv timeouts */
  int rc;			/* return code for sending */
  struct itimerval interval;	/* the timeout interval */
  struct itimerval disable;	/* to disable timers */
  int retries;			/* number of times we've retried */
  get_req_msg_type *req_msg;	/* the request message */
  get_rsp_msg_type *reply;	/* the response message */
  int req_id;			/* request id */
  int rreq_id;			/* request id of message received */
  long from;			/* dummy - to store an IP address */
  int msg_type;			/* type of message returned */
  int done = 0;			/* are we done querying yet? */
  int i;			/* just an index */
  strng uncvted;		/* strng for var_class before conversion */
  strng *cvtedvar;		/* ptr to var_class after conversion */
  strng *name;			/* name of variable after conversion */
  if(gwaddr == 0)		/* no gateway address specified? */
   { fprintf(stderr,"no gateway to query\n");
     return(-1);
   }
  if(session_id == NULL)	/* no session to use? */
   { fprintf(stderr,"no session name specified\n");
     return(-1);
   }
  if(var_class == NULL)		/* anything to query? */
   { fprintf(stderr,"no variable class specified\n");
     return(-1);
   }
/* set up timer stuff once and for all */
  interval.it_interval.tv_sec = 0; /* don't want to reload timer */
  interval.it_interval.tv_usec = 0; 
  interval.it_value.tv_sec = timeout; /* wait this long before timing out */
  interval.it_value.tv_usec = 0;
  disable.it_interval.tv_sec = 0; /* don't want to reload timer */
  disable.it_interval.tv_usec = 0; 
  disable.it_value.tv_sec = 0;	/* wait this long before timing out */
  disable.it_value.tv_usec = 0;
/* set up variable class for conversion */
  uncvted.len = strlen(var_class);
  uncvted.str = var_class;
/* convert the variable class */
  if((cvtedvar = sgmp_var_cvt(&uncvted,SYMTONUM)) == NULL)
   { fprintf(stderr,"couldn't convert %s\n",uncvted.str);
     return(-2);
   }
/* set up first request */
   req_id = getpid() ^ (int)time(0); /* the request id */
/* create a request */
   if((req_msg=(get_req_msg_type *)create_sgmp_req(req_id,1,&cvtedvar)) == NULL)
    { fprintf(stderr,"cannot create request message\n");
      exit(1);
    }
  do {				/* iterate to get information */
   retries = 0;			/* we haven't timed out yet */
/* preserve the environment */
   if(setjmp(env))
    if(retries < maxretries)	/* if we should retry again */
     { fprintf(stderr,"receive timed out. Retrying.....\n");
       retries++;		/* retry one more time */
     }
    else			/* we exceeded the number of retries */
     { fprintf(stderr,"giving up\n");
       term_sgmp_comm(req_id);
       return(-1);
     }
/* send off request */
   signal(SIGALRM,SIG_IGN);	/* ignore timeouts for now */
   if((rc = send_sgmp_mesg(GET_REQ_MESG,req_id,gwaddr,SERVICE,req_msg,
			      session_id,strlen(session_id),timeout)) < 0)
    switch(rc) {
     case SEND_TIMEOUT:		/* timed out */
      fprintf(stderr,"send timeout\n");
      exit(1);
     case GEN_ERR:		/* some unknown error */
      fprintf(stderr,"generic error in send\n");
      exit(1);
     default:			/* strange error */
      fprintf(stderr,"unknown error return in send\n");
      exit(1);
    }
/* sent off request, now wait for a response */
   rreq_id = req_id;
   signal(SIGALRM,onrintr);	/* catch interrupts */
   setitimer(ITIMER_REAL,&interval,NULL); /* start timer */
   reply = (get_rsp_msg_type *)recv_sgmp_mesg(&from,SERVICE,&rreq_id,&msg_type,&session_id);
   setitimer(ITIMER_REAL,&disable,NULL); /* disable the receive timer */
   if(reply == NULL)
    { fprintf(stderr,"could not get a coherent reply\n");
      exit(1);
    }
   if(msg_type != GET_RSP_MESG)	/* if this wasn't a response message */
    { fprintf(stderr,"incorrect response from gateway\n");
      return(-1);
    }
   if(reply->error_status != 0)	/* check if we got back an error */
    { fprintf(stderr,"sgmpconfig: received error indication\n");
      fprintf(stderr,"error was in variable %d\n",reply->error_index);
      return(-1);
    }
   if(rreq_id != req_id)	/* if we got the wrong message */
    { fprintf(stderr,"mismatch in req_id of reply\n");
      return(-1);
    }
/* everything OK, now print out what we got back */
/* we know that variables represented symbolically start with '_' */
  if((reply->var_op_list->var_name->str)[0] == '_')  /* symbolic? */
   { if((name=sgmp_var_cvt(reply->var_op_list->var_name,CVTIPADD)) == NULL)
      { fprintf(stderr,"couldn't convert symbolic variable name\n");
        return(-1);
      }
   } 
  else				/* name is in symbolic form */
   { if((name=sgmp_var_cvt(reply->var_op_list->var_name,NUMTOSYM)) == NULL)
      { fprintf(stderr,"couldn't convert numeric variable name in reply\n");
        fprintf(stderr,"reply was: ");
        for(i=0; i< reply->var_op_list->var_name->len; i++)
         fprintf(stderr,"%x ",(((reply->var_op_list->var_name->str)[i]) & 0x000000ff));
        fprintf(stderr,"\n");
        return(-1);
      }
   }
  if(strncmp(uncvted.str,name->str,uncvted.len) != 0)
   done = 1;			/* we're done */
  else				/* not done. prepare next message */
   { printf("%s\t",name->str);	/* print out the name */
     switch(reply->var_op_list->var_value.type) { /* decide what value is */
      case INTEGER:			/* value is an integer */
       printf("0x%x",reply->var_op_list->var_value.value.intgr);
       printf("\t\t%d",reply->var_op_list->var_value.value.intgr);
       break;
      case PRIM_STR:			/* a primitive string */
/* kludge because internet addresses can be returned as strings */
       if(isalnum((reply->var_op_list->var_value.value.str->str)[0]))
        printf("%s",reply->var_op_list->var_value.value.str->str);
       else
        { long address;
          ((char *)(&address))[0] = reply->var_op_list->var_value.value.str->str[0];
          ((char *)(&address))[1] = reply->var_op_list->var_value.value.str->str[1];
          ((char *)(&address))[2] = reply->var_op_list->var_value.value.str->str[2];
          ((char *)(&address))[3] = reply->var_op_list->var_value.value.str->str[3];
          printf("%s",stdtodot(ntohl(address)));
       }
      break;
     default: 
      printf("unrecognizable type for value returned\n");
      break;
     }
     putchar('\n');
/*  OK, done with output. If we aren't done, prepare next message */
     free(req_msg);		/* get rid of request message */
     req_msg = reply;		/* the new request message */
   } 
  
  } while (!done);
  term_sgmp_comm(req_id);
  return(0);
}

onrintr()
{ 
  longjmp(env);
}
