/*

Copyright 1987 by Rensselaer Polytechnic Institute

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of RPI not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.
RPI makes no representations about the suitability of
this software for any purpose.  It is provided "as is"
without express or implied warranty.

*/
/*****************************************************************************
**
**			varcvt.c
**
** This is the src file for the translator that converts SGMP
** variables from symbolic to numeric form and vice-versa. This converter
** will convert variables of form:
**	<symbolic prefix><dot notation IP address> to 
**	<numeric prefix><32 bit IP address>;
**	<symbolic prefix><32 bit IP address> to 
**	<numeric prefix><32 bit IP address>;
** as well as
**	<numeric prefix><32 bit IP address> to
**	<symbolic prefix><dot notation IP address>
**
**
************************************************************************/
#include "sgmp.h"
#include "varcvt.h"

/* put this here to prevent it from interfering with other includes */
#define CONVMASK	((u_long)(~0x00) >> (sizeof(long)-1)*8)

typedef struct cll {			/* a tree cell */
		char 	numeric;	/* numeric representation */
		char *	symbolic;	/* symbolic representation */
		struct cll *child;	/* eldest child */
		struct cll *sib;	/* sibling */
		   } cvtcell;

cvtcell *numtosym,*symtonum;	/* the two conversion tables */

sgmp_init_cvt(initfile)
char *initfile;			/* initialization file */
{ char line[BUFSIZ];		/* to read in lines of file */
  cvtcell *current1;		/* current position in tree */
  cvtcell *current2;		/* current position in tree */
  int i;			/* an index into the line read */
  FILE *init;			/* pointer to init file */
  char sympfx[64];		/* current portion of symbolic prefix */
  char numpfx[3];		/* current portion of numeric prefix */
  int symlen,numlen;		/* lengths of prefix portions */
  cvtcell *p,*q,*r,*s;		/* temporary pointers */
  if((init = fopen(initfile,"r")) == NULL) /* open init file */
   { fprintf(stderr,"cannot open variable initialization file %s\n",initfile);
     return(-1);
   }
  while(fgets(line,BUFSIZ-1,init) != NULL) /* while not end of file */
/* skip white space preceding beginning of symbolic representation */
   { for(i=0; (line[i] == ' ' || line[i] == '\t') && line[i] != '\n'; i++)
      		;
     if(line[i] == '\n')	/* reached end of line? */
      continue;
     if(line[i] != '_')		/* all variables begin with an underscore */
	continue;
/* at the beginning of symbolic representation of variable now */
     current1 = NULL;		/* beginning of tree */
/* while we haven't reached end of symbolic representation */
     while((line[i] != ' ' && line[i] != '\t') && line[i] != '\n')
      { i++;			/* skip past underscore */
/* find next portion of prefix */
        for(symlen=0;
            line[i]!='_' && line[i]!='\n' && line[i]!=' ' && line[i]!='\t' ;
            symlen++,i++)
         sympfx[symlen] = line[i]; /* copy in character of prefix */
/* got the new portion of the symbolic prefix now. Insert into tree */
        if((p = (cvtcell *)malloc(sizeof(cvtcell))) == NULL)
         { fprintf(stderr,"cannot allocate new tree cell\n");
           return(-1);
         }
	sympfx[symlen++] = '\0'; /* null terminate */
        if((p->symbolic = (char *)malloc((symlen)*sizeof(char))) == NULL)
	 { fprintf(stderr,"cannot allocate space for new symbol %s\n",sympfx);
	   return(-1);
         }
	strncpy(p->symbolic,sympfx,symlen); /* copy in prefix portion */
	(p->symbolic)[symlen-1] = '\0';	    /* null terminate */
        p->child = NULL;	/* no children yet */
	p->sib = NULL;		/* no siblings either */
	p->numeric = '\0';	/* no numeric counterpart yet */
/* insert prefix portion into symbolic to numeric tree */
	if(symtonum == NULL)	/* if this is first entry in tree */
           symtonum = p;	/* insert first cell in tree */
	else			/* this is not first cell */
	 { if(current1 == NULL)	/* if we are at the top of the tree */
	    q = symtonum;	/* start at beginning of sibling list */
	   else			/* we are not at top of tree */
	    q = current1->child;	/* start at first child of current */
           if (q == NULL)	/* if there are no children */
            current1->child = p;	/* this is first child */
	   else			/* there is at least one child */
/* search for a match, or for the appropriate place to insert */
	    { r = NULL;		/* no predecessor yet, we are at beginning */
	      while(q != NULL)
/* if we have found insertion point, or a match */
               { if(strcmp(q->symbolic,p->symbolic) == 0)
		  break;
		 r = q;
		 q = q->sib;
	       }
/* if reached end of list without finding a match */
/* Note that there is at least one child, so q and r can't both be NULL */
	      if(q == NULL)
	       r->sib = p;	/* append to list of children */
	      else		/* we found a match or insertion point */
               { free(p);	/* deallocate the new cell we created */
	         p = q;	/* point at cell that matched */
	       }
	    }			/* end of code dealing with at least one child*/
         }			/* end of code dealing with non-first child */
	current1 = p;		/* move down 1 level in tree */
      }				/* end of code dealing with this prefix */
/* found the symbolic prefix, now skip the white space to get numeric stuff */
     for(;(line[i] == ' ' || line[i] == '\t') && line[i] != '\n'; i++)
		;
     if(line[i] == '\n')	/* reached end of line? */
      { fprintf(stderr,"unexpected end of line: %s\n",line);
        return(-1);
      }
/* at the beginning of numeric representation of variable now */
     current1 = NULL;		/* beginning of tree */
     current2 = NULL;		/* beginning of tree */
/* find entire prefix */
     for(;(line[i] != ' ' && line[i] != '\t') && line[i] != '\n'; i++)
      {  numlen = 0;		/* length of number == 0 right now */
         numpfx[numlen++] = line[i++]; /* copy in 1st character of prefix */
         numpfx[numlen] = line[i]; /* copy in 2nd character of prefix */
/* got the new portion of the symbolic prefix now. Insert into tree */
        if((p = (cvtcell *)malloc(sizeof(cvtcell))) == NULL)
         { fprintf(stderr,"cannot allocate new tree cell\n");
           return(-1);
         }
        p->numeric = ((hexcvt(numpfx[0]) << 4) | (hexcvt(numpfx[1]) & 0x0f));
        p->child = NULL;	/* no children yet */
	p->sib = NULL;		/* no siblings either */
	p->symbolic = NULL;	/* no numeric counterpart yet */
/* insert prefix portion into symbolic to numeric tree */
	if(numtosym == NULL)	/* if this is first entry in tree */
         { numtosym = p;	/* insert first cell in tree */
	   if(init_peer(symtonum,numtosym)<0) /* fill in fields of other tree */
		return(-1);
	   p = numtosym;	/* tree has one level now */
	   s = symtonum;	/* currently at first level of tree */
         }
	else			/* this is not first cell */
	 { if(current2 == NULL)	/* if we are at the top of the tree */
	    { q = numtosym;	/* start at beginning of sibling list */
              s = symtonum;	/* on other tree too */
	      if(s==NULL)	/* mistake? */
	       { fprintf(stderr,"unexpected end of tree in line %s\n",line);
		 return(-1);
	       }
            }
	   else			/* we are not at top of tree */
	    { s = current1->child;	/* start at first child */
	      if(s==NULL)	/* mistake? */
	       { fprintf(stderr,"unexpected end of tree in line %s\n",line);
		 return(-1);
	       }
	      q = current2->child;	/* start at first child of current */
	    }
           if (q == NULL)	/* if there are no children */
            { current2->child = p;	/* this is first child */
	      if(init_peer(current1->child,current2->child) < 0)
		return(-1);
	    }
	   else			/* there is at least one child */
/* search for a match, or for the appropriate place to insert */
	    { r = NULL;		/* no predecessor yet, we are at beginning */
	      while(q != NULL)
/* if we have found insertion point, or a match */
               { if(q->numeric == p->numeric)
		  break;
		 r = q;
		 q = q->sib;
		 s = s->sib;
	         if(s==NULL)	/* mistake? */
	          { fprintf(stderr,"unexpected end of tree in line %s\n",line);
		    return(-1);
	          }
	       }
/* if reached end of list without finding a match */
/* Note that there is at least one child, so q and r can't both be NULL */
	      if(q == NULL)
	       { r->sib = p;	/* append to list of children */
		 if(init_peer(s,p) < 0) /* fill in other tree */
		  return(-1);
	       }
	      else		/* we found a match or insertion point */
	       { free(p);
	         p = q;		/* point at matching cell */
	       }
	    }			/* end of code dealing with at least one child*/
         }			/* end of code dealing with non-first child */
	current2 = p;		/* move down 1 level in tree */
	current1 = s;		/* move down 1 level in other tree */
      }				/* end of code dealing with prefix */
   }				/* end of code dealing with this line */
  return(0);
}

/* varcvt - convert a given variable name */
strng *
sgmp_var_cvt(name,cvtflag)
strng *name;			/* name to be converted */
int cvtflag;			/* to indicate what sort of conversion to do */
{ strng *convname;		/* the name after conversion */
  char strbuf[BUFSIZ];		/* buffer for building converted name */
  int namlen;			/* length of this part of name */
  short i;			/* index into name */
  cvtcell *current;		/* to traverse tree */
  u_long intbuf;		/* buffer to convert integers to strings */
  int j;			/* index and counter */
/* allocate space for the strng to be returned */
  if((convname = (strng *)malloc(sizeof(strng))) == NULL)
   { fprintf(stderr,"cannot allocate a new strng\n");
     return((strng *)NULL);
   }
  convname->len = 0;		/* the converted name is of zero length now */
  switch(cvtflag) {		/* decide what sort of conversion to do */
   case SYMTONUM:		/* completely symbolic to numeric */
    current = symtonum;		/* start at top of tree */
/* start at 1 because we want to skip first '_' */
    for(i = 1;i < name->len;)	/* iterate through entire name */
/* find length of that part of name that corresponds to a cell in the tree */
     { for(namlen=0; i < name->len && (name->str)[i] != '_';i++,namlen++)
			;
       if(i > name->len)	/* if we've converted entire name, return */
        { if((convname->str=(char *)malloc((convname->len+1)*sizeof(char)))==NULL)
	   return((strng *)NULL);
          strncpy(convname->str,strbuf,convname->len); /* copy in name */
          (convname->str)[convname->len] = '\0'; /* null terminate */
	  return(convname);
        }
/* find corresponding cell in tree */
       while(current != NULL)	/* go through all cells at this level */
/* if we found a match, or have reached point where a match is impossible */
	if(strncmp((((name->str)+i)-namlen),current->symbolic,namlen) == 0)
	 break;			/* break out of while loop */
	else			/* a match is still possible */
	 current = current->sib; /* go on to next cell */
       if(current != NULL)	/* if we found a match */
	strbuf[(convname->len)++] = current->numeric;
       else			/* didn't find a match */
        { convname->len = -1;
	  convname->str = NULL;
	  return((strng *)NULL);
        }
/* must have found a match if we reached this point. On to next part of name */
       i++;			/* skip the '_' */
       current = current->child; /* go to child */
/* if we have fallen off tree */
       if(current == NULL && i < name->len)	
        { if ((name->str)[i] == '_') /* if there is a rest of name */
	   ++i;
          if(!isdigit((name->str)[i])) /* if rest of name isn't digits */
           { convname->len = -1;
	     convname->str = NULL;
	     return((strng *)NULL);
           }
          else			/* rest  of name is digits */
	   { long digits;	/* temporary storage for digits */
/* convert digits in rest of string from characters to integers */
	     while(i < name->len)
	      { for(j=i; (name->str)[j] != '\0' && (name->str)[j] != '.'; j++)
			;
		(name->str)[j] = '\0'; /* null terminate */
		digits = atol((name->str)+i);
		strbuf[(convname->len)++] = ((char *)(&digits))[sizeof(long)-1];
		i = j+1;	/* go to one after the digits converted */
	      }
	     i = name->len + 1;
	   }
	}
     };
/* converted entire variable, now return it */
    if((convname->str=(char *)malloc((convname->len+1)*sizeof(char)))==NULL)
	return((strng *)NULL);
    strncpy(convname->str,strbuf,convname->len); /* copy in name */
    (convname->str)[convname->len] = '\0'; /* null terminate */
    return(convname);
    break;
   case NUMTOSYM:		/* completely numeric to symbolic */
    current = numtosym;		/* start at top of tree */
    for(i = 0;i < name->len;) /* iterate through entire name */
/* find corresponding cell in tree */
     { while(current != NULL)	/* go through all cells at this level */
/* if we found a match, or have reached point where a match is impossible */
	if((name->str)[i] == current->numeric)
	 break;			/* break out of while loop */
	else			/* a match is still possible */
	 current = current->sib; /* go on to next cell */
       if(current != NULL)	/* if we found a match */
	{ strbuf[(convname->len)++] = '_'; /* add an underscore */
	  sprintf((strbuf+convname->len),"%s",current->symbolic);
	  convname->len += strlen(current->symbolic);
        }
       else			/* didn't find a match */
        { convname->len = -1;
	  convname->str = NULL;
	  return((strng *)NULL);
        }
/* must have found a match if we reached this point. On to next part of name */
       i++;			/* move on to next byte */
       current = current->child; /* go to child */
       if(current == NULL && i < name->len) /* if we have fallen off tree */
        { strbuf[(convname->len)++] = '_'; /* add an underscore */
/* foreach extra digit add digit in string form to convname */
	  for(;i < name->len; i++)
           { sprintf((strbuf+convname->len),"%d",((int)((name->str)[i]) & CONVMASK));
	     convname->len = strlen(strbuf);
	     strbuf[(convname->len)++] = '.';
	   }
	  strbuf[--(convname->len)] = '\0'; /* to get rid of extra '.' */
	  i = name->len + 1;	/* to escape loop */
	}
     };
/* converted entire variable, now return it */
    if((convname->str=(char *)malloc((convname->len+1)*sizeof(char)))==NULL)
     return((strng *)NULL);
    strncpy(convname->str,strbuf,convname->len); /* copy in name */
    (convname->str)[convname->len] = '\0'; /* null terminate */
    return(convname);
    break;
   case CVTIPADD:		/* convert a 32 bit IP add. to dot notation */
    current = symtonum;		/* start at top of tree */
    for(i = 0;i < name->len;)	/* iterate through entire name */
/* find length of that part of name that corresponds to a cell in the tree */
     { strbuf[i++] = '_';	/* add the underscore */
       (convname->len)++;	/* add to length */
       for(namlen=0; i < name->len && (name->str)[i] != '_';i++,namlen++)
	{ strbuf[i] = (name->str)[i];
	  (convname->len)++;
        }
       if(i > name->len)	/* if we've converted entire name, return */
        { if((convname->str=(char *)malloc((convname->len+1)*sizeof(char)))==NULL)
	   return((strng *)NULL);
          strncpy(convname->str,strbuf,convname->len); /* copy in name */
          (convname->str)[convname->len] = '\0'; /* null terminate */
	  return(convname);
        }
/* find corresponding cell in tree */
       while(current != NULL)	/* go through all cells at this level */
/* if we found a match, or have reached point where a match is impossible */
	if(strncmp((((name->str)+i)-namlen),current->symbolic,namlen) == 0)
	 break;			/* break out of while loop */
	else			/* a match is still possible */
	 current = current->sib; /* go on to next cell */
       if(current == NULL)	/* if we didn't find a match */
        { convname->len = -1;
	  convname->str = NULL;
	  return((strng *)NULL);
        }
/* must have found a match if we reached this point. On to next part of name */
       current = current->child; /* go to child */
/* if we have fallen off tree */
       if(current == NULL && i < name->len) /* if we have fallen off tree */
        { strbuf[(convname->len)++] = '_';
	  i++;				/* move past '_' */
/* foreach extra digit add digit in string form to convname */
	  for(;i < name->len; i++)
           { sprintf((strbuf+convname->len),"%d",((int)((name->str)[i]) & CONVMASK));
	     convname->len = strlen(strbuf);
	     strbuf[(convname->len)++] = '.';
	   }
	  strbuf[--(convname->len)] = '\0'; /* to get rid of extra '.' */
	  i = name->len + 1;	/* to escape loop */
	}
     };
/* converted entire variable, now return it */
    if((convname->str=(char *)malloc((convname->len+1)*sizeof(char)))==NULL)
	return((strng *)NULL);
    strncpy(convname->str,strbuf,convname->len); /* copy in name */
    (convname->str)[convname->len] = '\0'; /* null terminate */
    return(convname);
    break;
   default:			/* unknown conversion */
    fprintf(stderr,"unknown conversion\n");
    return((strng *)NULL);
  }
}

/* hexcvt - convert a given character to hex format and return as an integer */
char
hexcvt(c)
char c;
{ u_int result;
  switch (c) {
   case '0':
   case '1':
   case '2':
   case '3':
   case '4':
   case '5':
   case '6':
   case '7':
   case '8':
   case '9': result = (u_int)(c - '0'); break;
   case 'A':
   case 'a': result = 10; break;
   case 'B':
   case 'b': result = 11; break;
   case 'C':
   case 'c': result = 12; break;
   case 'D':
   case 'd': result = 13; break;
   case 'E':
   case 'e': result = 14; break;
   case 'F':
   case 'f': result = 15; break;
   default: fprintf(stderr,"unknown hex digit %c\n",c); result = 16; break;
  }
  return((char)(((char *)&result)[sizeof(int)-1]));
}

/* init_peer - initialize peer cells in the two trees (tables) */
init_peer(symptr,numptr)
cvtcell *symptr;		/* cell in symbolic to numeric table */
cvtcell *numptr;		/* cell in numeric to symbolic table */
{ if(symptr == NULL || numptr == NULL)
    return(-1);
  if(symptr->numeric != '\0')	/* we have a peer already? */
   if(symptr->numeric != numptr->numeric) /* if the peers are not the same */
    { fprintf(stderr,"warning: symbol cell %s had peer.\n",symptr->symbolic);
      return(-1);
    }
   else				/* we have the same peer as last time */
   return(0);
  else				/* we don't have peer */
     symptr->numeric = numptr->numeric;
  if(numptr->symbolic != NULL)
   if(strcmp(numptr->symbolic,symptr->symbolic) != 0)
    { fprintf(stderr,"warning: symbol cell %d had peer.\n",numptr->numeric);
      return(-1);
    }
   else				/* we have same peer as before */
    return(0);
  else				/* we don't have peer */
   { numptr->symbolic=(char *)malloc((strlen(symptr->symbolic)+1)*sizeof(char));
     if(numptr->symbolic == NULL)
      return(-1);
     sprintf(numptr->symbolic,"%s",symptr->symbolic);
   }
  return(0);
}
