/*

Copyright 1987 by Rensselaer Polytechnic Institute

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of RPI not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.
RPI makes no representations about the suitability of
this software for any purpose.  It is provided "as is"
without express or implied warranty.

*/

/*****************************************************************************
**
**			parse_util.c
**
** This file contains all the lowest level functions used to parse
** a packet in the Simple Gateway Monitoring Protocol. This is the
** file where functions that actually perform bit and octet manipulations
** reside.
**
**
*****************************************************************************/
#include "sgmp.h"		/* the general include file */
#include "X409.h"		/* X.409 specific definitions */

PACKET
parse_header(packet,pkt_len,type,len)
PACKET packet;			/* the packet whose header is to be parsed */
int *pkt_len;			/* length of packet */
int *type,*len;			/* the type and the length of the contents */
{ 
  if ((*pkt_len != ISINDEFINITE) && (*pkt_len <= 0)) /* packet end  reached? */
    return((PACKET)NULL);
  switch (*packet) {
/* these cases for getting message type */
    case REQ_MSG: *type = GET_REQ_MESG;	/* Get Request Message */
		  break;
    case RSP_MSG: *type = GET_RSP_MESG;	/* Get Response Message */
		  break;
    case TRP_MSG: *type = TRP_REQ_MESG;	/* Trap Request Message */
		  break;
    case SET_MSG: *type = SET_REQ_MESG;	/* Set Request Message */
		  break;
/* these cases for detecting types types */
    case SEQ_TYP: *type = SEQUENCE;
		  break;
    case INT_TYP: *type = INTEGER;
		  break;
    case PRM_STR: *type = PRIM_STR;
                  break;
    case CON_STR: *type = CONS_STR;
		  break;
    default:	  *type = UNKNOWN_TYPE;
		  return((PACKET)NULL);
  }
  if ((*pkt_len != ISINDEFINITE) && (--(*pkt_len) < 0))
    return((PACKET)NULL);
  ++packet;			/* move on to next field - length */
/* get the length of the packet contents */
  if ((*packet & ISLONG) != ISLONG) /* length is in short form? */
   { *len = (unsigned int)(*packet & LENMASK);
     if ((*pkt_len != ISINDEFINITE) && (--(*pkt_len) < 0))
       return((PACKET)NULL);
     ++packet;			/* move on to next field */
   }
  else				/* length is in not in short form */
   if (*(char *)packet != INDEFINITE)/* if the length is not an indefinite */
     { int lenlen;		/* length of length */
       lenlen = (unsigned int)(*packet & LENMASK);
       if ((*pkt_len != ISINDEFINITE) && (--(*pkt_len) < 0))
         return((PACKET)NULL);
       ++packet;			/* move on to actual length */
       *len = 0;			/* prepare to accumulate length */
       while(lenlen > 0)		/* iterate to accumulate length */
        { *len = (*len << 8) + (unsigned int)(*packet); /* add to length */
	  lenlen--;			/*    decrement length of length */
          if ((*pkt_len != ISINDEFINITE) && (--(*pkt_len) <= 0))
            return((PACKET)NULL);
	  ++packet;			/*    move on to next octet */
        }
     }
   else					/* length is an indefinite */
     { if ((*pkt_len != ISINDEFINITE) && (--(*pkt_len) < 0))
          return((PACKET)NULL);
       ++packet;			/*    move on to next octet */
       *len = ISINDEFINITE;
     }
  return(packet);
}

/* parse_int - parse an integer (a long, actually) */
PACKET
parse_int(packet,value,cont_len)
PACKET packet;
long *value;
int *cont_len;
{ int lenval;			/* length of integer value */
  int type;			/* type - this should be integer */
/* get header information */
  packet = parse_header(packet,cont_len,&type,&lenval);
  if(packet == NULL)		/* if there was an error */
   { *value = GEN_ERR;		/* generic error: don't know what happened */
     return((PACKET)NULL);
   }
  if(type != INTEGER)		/* if this wasn't an integer */
   { *value = PARSE_ERROR;	/* parse error: should be an integer */
     return((PACKET)NULL);
   }
  if(*cont_len != ISINDEFINITE)	/* if length of mesg contents !indefinite */
   { if (lenval > LONG_LEN)	/* check that integer isn't too large */
      { *value = PARSE_ERROR;	/* parse error: integer too long */
        return((PACKET)NULL);
      } 
    *value = 0;		/* prepare to accumulate value of integer */
/* an integer is a primitive structure, so the length has to be definite */
     if(lenval != ISINDEFINITE)
      for(;lenval > 0;--lenval) /* iterate to get integer */
        { *value = (*value * 256) + ((unsigned int)(*packet) & 0x000000ff);
          if (--(*cont_len) < 0) /* if end of packet is reached */
            return((PACKET)NULL);
          ++packet;		/* move on to next octet */
        }
    else			/* length of the integer is indefinite */
     return((PACKET)NULL);	/* integers can't have indefinite lengths */
   }				/* end of code dealing with definite lengths */
  else				/* length of mesg contents is indefinite */
   { if(lenval > LONG_LEN)	/* check that integer isn't too large */
      { *value = PARSE_ERROR;	/* parse error: integer too long */
        return((PACKET)NULL);
      } 
     *value = 0;		/* prepare to accumulate value of integer */
     if(lenval != ISINDEFINITE)	/* if length of integer is not indefinite */
       for(;lenval > 0;--lenval) /* iterate to get integer */
         { *value = (*value << 8) + (unsigned int)(*packet);
           if (--(*cont_len) <= 0) /* if end of packet is reached */
             return((PACKET)NULL);
           ++packet;		/* move on to next octet */
         }
      else			/* length of the integer is indefinite */
	 return((PACKET)NULL);	/* integers can't be of indefinite length */
   }				/* end of indefinite length code */
  return(packet);
}

/* parse_string - parse an octet string */
PACKET
parse_string(current,string,typelen)
PACKET current;			/* packet part of which is to be parsed */
strng **string;			/* this string should be filled in */
/* NOT THE LENGTH OF THE STRING!! */
int *typelen;			/* the length of the parent type */
{ int type;			/* type - should be a string */
  int strlength;		/* length of string */
  char *realloc();		/* define realloc to be of right type */
  int len;			/* length of a string */
/* parse the header */
  current = parse_header(current,typelen,&type,&strlength);
  if(current == NULL)		/* if there was an error */
   return((PACKET)NULL);	/* die... */
/* get space for string structure */
  if((*string = (strng *)malloc(sizeof(strng))) == NULL)
    return((PACKET)NULL);
  switch(type)	{		/* decide what sort of string it is */
   case PRIM_STR:		/* primitive string */
    if(strlength == ISINDEFINITE) /* length of string is indefinite */
/* primitive strings can't be of indefinite length */
	return((PACKET)NULL);	/* return error indication */
    else			/* string is of fixed length */
     { (*string)->str = (char *)malloc((strlength*sizeof(char)+1));
       if((*string)->str == NULL)	/* couldn't allocate space? */
        return((PACKET)NULL);
       bcopy(current, (*string)->str, strlength);
       ((*string)->str)[strlength] = '\0'; /* null terminate */
       current += strlength;
       (*string)->len = strlength;
       if(*typelen != ISINDEFINITE)
	{ *typelen -= strlength; /* decrement length of parent type */
	  if (*typelen < 0)	/* if the parent type isn't this long */
           return((PACKET)NULL);
        }
     }
    break;
   case CONS_STR:		/* constructed string */
    if(strlength == ISINDEFINITE) /* length of string is indefinite */
     { strng *s;			/* temporary */
       (*string)->str = "";		/* an unfortunate kludge */
       (*string)->len = 0;
/* iterate until we reach end of string */
       while((current[0] != EOCID) || (current[1] != EOCLEN))
        { current = parse_string(current,&s,&strlength); 
	  if(current == NULL)	/* some error? */
           return((PACKET)NULL);
          if(((*string)->str=realloc(*string,((*string)->len)+sizeof(s)))==NULL)
           return((PACKET)NULL);
          (*string)->len += sizeof(s);
          strcat((*string)->str,s); /* concatenate new string to old */
          if(*typelen != ISINDEFINITE)
           { *typelen -= sizeof(s); /* decrement typelen */
	     if (*typelen < 0)	/* if typelen wasn't too long */
	      return((PACKET)NULL); 
           }
        }
       current += 2;		/* move past EOC */
       if(*typelen != ISINDEFINITE)
         *typelen  -= 2;	/* decrement typelen accordingly */
     }
    else			/* length of string is fixed */
     { (*string)->str = (char *)malloc(strlength*sizeof(char));
       if((*string)->str == NULL)
	return((PACKET)NULL);
       (*string)->len = 0;
/* check that we have enough of the parent type left */
        if(*typelen != ISINDEFINITE)
         { *typelen -= strlength; /* decrement typelen */
	   if (*typelen < 0)	/* if typelen wasn't too long */
	    return((PACKET)NULL); 
         }
       while(strlength > 0)
        { strng *s;		/* a temporary string */
          if((s = (strng *)malloc(sizeof(strng))) == NULL)
		return((PACKET)NULL);
	  current = parse_string(current,&s,&strlength);
          if(current == NULL)	/* an error? */
	   return((PACKET)NULL);
	  (*string)->len += s->len; /* add to string length */
          strcat((*string)->str,s); /* concatenate string */
	}
     }
    break;
   default:			/* Unknown! */
    return((PACKET)NULL);
  }
  return(current);
}
