/*

Copyright 1987 by Rensselaer Polytechnic Institute

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of RPI not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.
RPI makes no representations about the suitability of
this software for any purpose.  It is provided "as is"
without express or implied warranty.

*/
/******************************************************************************
**
**			build_util.c
**
** This file contains the absolute lowest level functions in the packet
** builder, the functions that do bit and byte manipulations, and actually
** build X.409 structures into the packet. 
** 
** Contents of this file: build_header.c,build_int.c, build_string.c
**
**
*****************************************************************************/
#include "sgmp.h"		/* the general include file */
#include "X409.h"		/* X409 specific definitions */

/* build_header - build an X409 header given 'type' and 'len' */
PACKET
build_header(type,len,head_len,buf,pkt_len)
int type;			/* type of header to be built */
int len;			/* length of contents of type to be built */
int *head_len;			/* length of header itself returned here */
PACKET buf;			/* buffer to build packet in */
int *pkt_len;			/* length of packet so far */
{ 
/* make sure we have at least two bytes - one for id, one for length */
  if (*pkt_len > (MAXLEN - 2))	/* if packet is too long */
   { *pkt_len = PKT_OVERFLOW;
     return((PACKET)NULL);
   }
/* build the type */
  switch(type) {		/* decide what type to build */
   case GET_REQ_MESG:		/* Get Request Message */
	*buf = REQ_MSG;
	break;
   case SET_REQ_MESG:		/* Get Request Message */
	*buf = SET_MSG;
	break;
   case GET_RSP_MESG:		/* Get Response Message */
	*buf = RSP_MSG;
	break;
   case TRP_REQ_MESG:		/* Trap Request Message */
	*buf = TRP_MSG;
	break;
   case SEQUENCE:		/* a list or a sequence */
	*buf = SEQ_TYP;
	break;
   case INTEGER:		/* integer */
	*buf = INT_TYP;
	break;
   case PRIM_STR:		/* primitive string */
	*buf = PRM_STR;
	break;
   case CONS_STR:		/* constructed string */
	*buf = CON_STR;
	break;
   default:			/* unknown type */
	return((PACKET)NULL);
  }
  buf++; (*pkt_len)++;		/* move along buffer; increment pkt length */
  *head_len = 1;		/* length of header is 1 byte now */
/* build the length now */
  if (len < 0)			/* make sure we don't have a spurious length */
   return((PACKET)NULL);
  if (len < 128)		/* if length will fit in short form length */
   if ((++(*pkt_len)) <= MAXLEN) /* if max length of packet not exceeded */
    { *buf = (LENMASK & ((char *)(&len))[sizeof(int)-1]);
      (*head_len)++; buf++;	/* increment what needs incrementing */
    }
   else				/* not enough packet left */
    { *pkt_len = PKT_OVERFLOW;	/* indicate that packet is too long */
      return((PACKET)NULL);
    }
  else				/* length requires long form length */
    if (len < 65536) 		/* will fit in two bytes? */
     if(*pkt_len <= (MAXLEN-3))	/* is there enough packet left */
      { buf[0] = (ISLONG | 0x02); /* length of length */
	buf[1] = ((char *)(&len))[sizeof(int)-2];
	buf[2] = ((char *)(&len))[sizeof(int)-1];
	buf += 3;		/* move buffer along */
	(*head_len) += 3;	/* header length just increased by 3 bytes */
	(*pkt_len) += 3;	/* the length of the packet just increased */
      }
     else			/* not enough packet left */
      { *pkt_len = PKT_OVERFLOW; /* indicate that packet is too long */
        return((PACKET)NULL);
      }
    else 			/* needs more than two bytes */
/*
   if the length of the contents of anything needs more than two bytes
   to encode, the contents must be > 65535 bytes in length, so there
   is no way it'll fit in a packet.
*/
     { *pkt_len = PKT_OVERFLOW;
       return((PACKET)NULL);
     }
  return(buf);
}

/* build_int - build an integer X409 structure */
PACKET
build_int(buf,val,int_len,pkt_len)
PACKET buf;			/* place to build packet */
long val;			/* value of integer to be built */
int *int_len;			/* length of integer returned here */
int *pkt_len;			/* length of packet so far */
{ PACKET int_head;		/* header of integer to be constructed */
  PACKET int_cont;		/* contents of integer to be constructed */
  int head_len,cont_len;	/* length of header, contents */
  int i;			/* for copying things in */
  val = ntohl(val);		/* convert to proper byte order 0123 */
/* check that we have at least three bytes left (for id, length and int) */
  if(*pkt_len > (MAXLEN-3))
   { *pkt_len = PKT_OVERFLOW;
     return((PACKET)NULL);
   }
/* allocate space for header,contents */
  if ((int_head = (PACKET)malloc(HEADLEN*sizeof(char))) == NULL)
   return((PACKET)NULL);
/* since we know an integer cannot require more than 4 bytes, don't allocate
   such a large buffer */
  if ((int_cont = (PACKET)malloc(8*sizeof(char))) == NULL)
   return((PACKET)NULL);
/* construct integer contents */
/* a negative integer always requires 4 bytes to represent */
  if (val >= 0 && val < 256)
   if (*pkt_len <= (MAXLEN-3))	/* if we have enough packet left */
    { *int_cont =  ((char *)(&val))[sizeof(long)-1];
      (*pkt_len)++;		/* the length of the packet just increased */
      cont_len = 1;		/* contents are one byte long */
    }
   else				/* not enough packet left */
    { *pkt_len = PKT_OVERFLOW;	/* indicate error */
      return((PACKET)NULL);
    }
  else				/* integer requires two bytes at least */
   if(val >= 256 && val < 65536)
    if (*pkt_len <= (MAXLEN-4))	/* if we have enough packet left */
     { int_cont[0] =  ((char *)(&val))[sizeof(long)-2];
       int_cont[1] =  ((char *)(&val))[sizeof(long)-1];
       (*pkt_len) += 2;		/* the length of the packet just increased */
       cont_len = 2;		/* contents are one byte long */
     }
    else			/* not enough packet left */
     { *pkt_len = PKT_OVERFLOW;	/* indicate error */
       return((PACKET)NULL);
     }
   else				/* integer requires three bytes at least */
    if (val >= 65536 && val < 16777216)
     if (*pkt_len <= (MAXLEN-5)) /* if we have enough packet left */
      { int_cont[0] =  ((char *)(&val))[sizeof(long)-3];
        int_cont[1] =  ((char *)(&val))[sizeof(long)-2];
        int_cont[2] =  ((char *)(&val))[sizeof(long)-1];
        (*pkt_len) += 3;	/* the length of the packet just increased */
        cont_len = 3;		/* contents are one byte long */
      }
     else			/* not enough packet left */
      { *pkt_len = PKT_OVERFLOW; /* indicate error */
        return((PACKET)NULL);
      }
    else			/* integer requires four bytes */
     if (*pkt_len <= (MAXLEN-6)) /* if we have enough packet left */
      { int_cont[0] =  ((char *)(&val))[sizeof(long)-4];
        int_cont[1] =  ((char *)(&val))[sizeof(long)-3];
        int_cont[2] =  ((char *)(&val))[sizeof(long)-2];
        int_cont[3] =  ((char *)(&val))[sizeof(long)-1];
        (*pkt_len) += 4;	/* the length of the packet just increased */
        cont_len = 4;		/* contents are one byte long */
      }
     else			/* not enough packet left */
      { *pkt_len = PKT_OVERFLOW; /* indicate error */
        return((PACKET)NULL);
      }
/* construct header now */
  if(build_header(INTEGER,cont_len,&head_len,int_head,pkt_len) == NULL)
   return((PACKET)NULL);
/* build entire integer now */
  for(i=0; i < head_len ; i++)
   buf[i] = int_head[i];
  buf += head_len;
  for(i=0; i < cont_len ; i++)
   buf[i] = int_cont[i];
  buf += cont_len;
  *int_len = head_len + cont_len; /* update integer length */
  free(int_cont); free(int_head); /* free what we don't need */
  return(buf);
}

/* build_string - build an X409 string */
PACKET
build_string(buf,string,strlength,pkt_len)
PACKET buf;			/* buffer in which to build string */
strng *string;			/* string to be built */
int *strlength;			/* length (excluding '\0') returned here */
int *pkt_len;			/* length of packet so far */
{ int head_len;			/* length of header */
  int i;			/* for copying things */
  buf = build_header(PRIM_STR,(int)(string->len),&head_len,buf,pkt_len);
  if(buf == NULL)		/* if couldn't create header */
   return((PACKET)NULL);
  if (*pkt_len > (MAXLEN - string->len)) /* if insufficient space left */
   { *pkt_len = PKT_OVERFLOW;
     return((PACKET)NULL);
   }
/* create string */
  for(i=0; i < string->len; i++)
   buf[i] = (string->str)[i];
  *strlength = head_len + string->len;	/* get length of string */
  *pkt_len += string->len;	/* increment packet length */
  buf += string->len;		/* move along buf */
  return(buf);
}
