/*

Copyright 1987 by Rensselaer Polytechnic Institute

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of RPI not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.
RPI makes no representations about the suitability of
this software for any purpose.  It is provided "as is"
without express or implied warranty.

*/
/******************************************************************************
**
**			build_types.c
**
** This file contains small functions that do not justify a file of
** their own, but are not sufficiently low-level that they can be in
** build_util.c either (these functions don't do manipulations at the
** bit or byte level).
**
**
*****************************************************************************/
#include "sgmp.h"			/* the general include file */
#include "X409.h"			/* X.409 specific definitions */

#define build_name(a,b,c,d)		build_string(a,b,c,d)

/* build_value - build a var_value given the structure */
PACKET
build_value(current,val,val_len,pkt_len)
PACKET current;				/* the place to build the value */
var_value_type *val;			/* value to be built */
int *val_len;				/* length of value returned here */
int *pkt_len;				/* length of packet so far */
{ 
  if(val->type == INTEGER)		/* if value is an integer */
   current = build_int(current,(long)(val->value.intgr),val_len,pkt_len);
  else					/* it's a string */
   current = build_string(current,(strng *)(val->value.str),val_len,pkt_len);
  return(current);
}

/* build_var_list - build a X.409 var list given the list as a structure */
/* this function builds the 'var_op_list_type' */
PACKET
build_var_list(current,list,list_len,pkt_len)
PACKET current;				/* place to build list */
var_op_list_type list;			/* list to be built as a structure */
int *list_len;				/* length of list returned here */
int *pkt_len;				/* amount of packet remaining */
{ int header_len;			/* length of X409 header for list */
  int var_len;				/* length of a single variable */
  int i;				/* for copying things in */
  var_op_list_type p;			/* temporary - for traversing list */
  PACKET list_head;			/* temporary - for list header */
  PACKET list_cont;			/* temporary - for list contents */
  PACKET q;				/* temporary - for building list */
/* allocate space for list header,contents */
  if((list_head = (PACKET)malloc(HEADLEN*sizeof(char))) == NULL)
   return((PACKET)NULL);
  if((list_cont = (PACKET)malloc(CONTLEN*sizeof(char))) == NULL)
   return((PACKET)NULL);
/* build list contents */
  q = list_cont;			/* point at beginning of contents */
  *list_len = 0;			/* list is of zero length for now */
  for(p = list;p != NULL;p = p->var_next) /* traverse variable list */
   { q = build_var_type(q,p,&var_len,pkt_len);
     if (q == NULL)		/* if there was an error */
	return((PACKET)NULL);
     *list_len += var_len;		/* add length of this variable */
   }
/* build list header */
  q = build_header(SEQUENCE,*list_len,&header_len,list_head,pkt_len);
  if(q == NULL)			/* if header couldn't be constructed */
   return((PACKET)NULL);
/* build entire list */
  for(i=0; i < header_len; i++)
   current[i] = list_head[i];
  current += header_len;
  for(i=0; i < *list_len; i++)
   current[i] = list_cont[i];
  current += *list_len;			/* move current along */
  *list_len += header_len;		/* update list_len to reflect header */
  free(list_head); free(list_cont);
/* needn't check list_len - if MAXLEN was exceeded, then pkt_len overflowed */
  return(current);
}

/* build_var_type - build the 'var_op_type' from a structure */
PACKET
build_var_type(current,var,var_len,pkt_len)
PACKET current;				/* place to build variable */
var_op_list_type var;			/* pointer to variable */
int *var_len;				/* length of variable returned here */
int *pkt_len;				/* amount of packet remaining */
{ int name_len;				/* length of variable name */
  int val_len;				/* length of value */
  int header_len;			/* length of variable header */
  int i;				/* for copying things in */
  PACKET var_head;			/* temporary - to construct header */
  PACKET var_cont;			/* temporary - to construct contents */
  PACKET p;				/* temporary - to build variable */
/* allocate space for variable header, contents */
  if((var_head = (PACKET)malloc(HEADLEN*sizeof(char))) == NULL)
	return((PACKET)NULL);
  if((var_cont = (PACKET)malloc(CONTLEN*sizeof(char))) == NULL)
	return((PACKET)NULL);
/* build the packet contents */
  p = build_name(var_cont,(strng *)(var->var_name),&name_len,pkt_len);
  if(p == NULL)
	return((PACKET)NULL);
  p = build_value(p,&(var->var_value),&val_len,pkt_len); 
  if (p == NULL)
	return((PACKET)NULL);
/* build the packet header */
  p = build_header(SEQUENCE,name_len+val_len,&header_len,var_head,pkt_len);
  if (p == NULL)
	return((PACKET)NULL);
/* build the packet */
  for(i=0; i < header_len; i++)
   current[i] = var_head[i];
  current += header_len;
  for(i=0; i < (name_len+val_len); i++)
   current[i] = var_cont[i];
  current += (name_len + val_len);
  *var_len = header_len + name_len + val_len;
  free(var_head); free(var_cont);
  return(current);
}

/* build_val_list - build a value list given the structure */
PACKET
build_val_list(current,list,list_len,pkt_len)
PACKET current;				/* place to build list */
val_list_type *list;			/* list to be built */
int *list_len;				/* length of list */
int *pkt_len;				/* length of packet so far */
{ val_list_type *p;			/* to traverse list */
  int val_len;				/* for length of value */
  int header_len;			/* length of header */
  int i;
  PACKET list_head;			/* temporary - to build list header */ 
  PACKET list_cont;			/* temporary - to build list contents */
  PACKET q;				/* temporary - to help build list */
/* allocate space for list header, contents */
  if((list_head = (PACKET)malloc(HEADLEN*sizeof(char))) == NULL)
	return((PACKET)NULL);
  if((list_cont = (PACKET)malloc(CONTLEN*sizeof(char))) == NULL)
	return((PACKET)NULL);
/* build list contents */
  q = list_cont;
  *list_len = 0;
  for (p = list;p != NULL;p = p->var_next) /* traverse list */
   { q = build_value(q,&(p->var_op),&val_len,pkt_len);
     if (q == NULL)
	return((PACKET)NULL);
     *list_len += val_len;
   }
/* build list header */
  q = build_header(SEQUENCE,*list_len,&header_len,list_head,pkt_len);
  if(q == NULL)			/* if header couldn't be constructed */
   return((PACKET)NULL);
/* build entire list */
  for(i=0; i < header_len; i++)
   current[i] = list_head[i];
  current += header_len;
  for(i=0; i < *list_len; i++)
   current[i] = list_cont[i];
  current += *list_len;			/* move current along */
  *list_len += header_len;		/* update list_len to reflect header */
  free(list_head); free(list_cont);
  return(current);
}
