/* $Copyright:	$
 * Copyright (c) 1991,1992,1993 by Thomas Moore and Steve Baker
 * All rights reserved
 *
 * This software is provided as is without any express or implied
 * warranties, including, without limitation, the implied warranties
 * of merchantability and fitness for a particular purpose.
 */

/* Main entry points:
 *
 * parse_wildcards()  -- substitute all wildcards in entire command line
 * match_wildcards()  -- substitute all wildcards in arglist
 * isregex()          -- check if string contains ~ or {[*?
 * dirget()           -- replace one wildcard in arglist
 * patmatch()         -- check if string matches wildcard
 */

#include "shell.h"
#include <sys/types.h>
#include <sys/dir.h>
#include <pwd.h>

extern int err;
char *errstr,*erru;

extern char _nonomatch;	  /* TRUE: don't crash on no match */
extern char _nodots;	  /* TRUE: don't include dot-files in search unless explicit */
extern char *_home;	  /* contains the name of user's home dir */

void *malloc();
char **dirget(), **_dirget();
char *index(),*strcat();

char ***parse_wildcards(), **match_wildcards(), **insertelt(), **delelt();

static struct stat ST;

char ***parse_wildcards(arg)
char ***arg;
{
  int i;

  for(i=0;arg[i];i++)
    arg[i] = (char **)match_wildcards(arg[i]);
  return (char ***)arg;
}

char **match_wildcards(arg)
char **arg;
{
  int i,j,cnt,lvl=0,match=2,k;
  char *p;

  for(cnt = 0;arg[cnt];cnt++);
  for(i=0;arg[i];i++) {
    if (arg[i][0] == '(') lvl++;
    if (arg[i][0] == ')') lvl--;
    if (!err && !lvl && (j=isregex(arg[i]))) {
      k = cnt;
      p = arg[i];
      arg = dirget(p,arg,&cnt,i,NULL,j==1,0);
      if(k<=cnt){ /* the pattern was replaced */
	if(!match)
	  free(errstr);
	match = 1;
	free(p);
      }
      else {
	if(_nonomatch){
	  j = i - 1;
	  arg = insertelt(arg,p,i,&j,&cnt);
	}
	else {
	  if(match == 2){
	    errstr = p;
	    match = 0;
	  }
	  else
	    free(p);
	}
      }
      i += (cnt-k);
    }
  }
  if(err == ERR_NO_USER) {
    fprintf(stderr,"No such user: %s.\n",erru);
    free(erru);
  } else if(!match && !_nonomatch){
    err = ERR_NO_MATCH;
    fprintf(stderr,"No match: %s.\n",errstr);
  }
  if(!match)
    free(errstr);
  return arg;
}

/****************************************************************/

isregex(s)
char *s;
{
  int r= 0;

  if (*s == '"' || *s == '\'' || *s == '$') return FALSE;
  if (*s == '~') r = 1;
  while(*s) {
    switch(*s++) {
      case '*':
      case '[':
      case '?':
      case '{':
	return 2;
      case '\\':
	if(*s)
	  s++;
    }
  }
  return r;
}

/****************************************************************/

/*  handle { both here and in patmatch so that dirs are handled here and
    patmatch still handles {}

    Handling {^..} was considered at length.
    Several options were available:
    a) assume {^a/b} meant a/{^b} (as in AmigaDOS)
    b) use same procedure as patmatch
    c) assume {^a/b} meant {^a}/{^b} (gad...)

    Due to the complexity of implementing a) and c), I decided to let
    the user do the {^} replacements for those cases himself.  Therefore
    '/' is not really handled in {^} (or in [] and [^], for that matter)
*/

char **dirget(pat,argarray,arglen,reppos,_repend,just_tilde,wc)
char *pat, **argarray,just_tilde,wc;
int *arglen, *_repend;
{
  int pl,nl,mpl = 0,al;
  char *moreloc = NULL,*stmp,*tp,*ne,*path = NULL,*n=NULL;
  struct passwd *pass = NULL;
  int repend = reppos;

  if(!_repend) {
    _repend = &repend;
    delelt(argarray,reppos,_repend,arglen);
  }
  if(!just_tilde) {
    for(n=pat;*n && *n != '{';n++)
      if(*n == '\\' && n[1])
	n++;
    if(!*n || n[1] == '^')
      n = NULL;
  }
  if(al = !n && *pat == '~') {
    if(moreloc = index(pat,'/'))
      *moreloc = 0;
    if(isregex(pat+1) != 2) {
      n = (char *)malloc(strlen(pat));
      for(stmp=n,ne=pat+1;*ne;ne++,stmp++) {
	if(*ne == '\\') { /* get rid of backslashes */
	  if(*stmp = ne[1])
	    ne++;
	} else
	  *stmp = *ne;
      }
      *stmp = 0;
      if(*n) {
	if(!(pass=getpwnam(n))) {
	  if(wc) {
	    free(n);
	    return argarray;
	  }
	  erru = n;
	  err = ERR_NO_USER;
	  return argarray;
	}
	free(n);
	n = pass->pw_dir;
      } else {
	free(n);
	n = _home;
      }
      pl = strlen(n);
      if(just_tilde){
	mpl = pl > 1 || *stmp != '/'; /* home != root dir */
	/* if root dir, delete excess '/' */
	nl = moreloc?strlen(moreloc+1)+mpl:0;
	stmp = (char *)malloc(pl + 1 + nl);
	strcpy(stmp,n);
	if(moreloc) {
	  if(mpl)
	    stmp[pl] = '/';
	  strcpy(stmp+pl+mpl,moreloc+1);
	}
	if(!wc || !al)
	  argarray = insertelt(argarray,stmp,reppos,_repend,arglen);
	else {
	  argarray = _dirget(stmp+1,"/",argarray,arglen,reppos,_repend);
	  free(stmp);
	}
	if(moreloc)
	  *moreloc = '/';
	return argarray;
      } else { /* moreloc must != NULL */
	*moreloc = '/';
	pat = moreloc+1;
	path = (char *)ALLOCA(pl+2);
	strcpy(path,n);
	if(pl>1 || *path != '/'){
	  path[pl+1] = 0;
	  path[pl] = '/';
	}
      }
    } else {
      nl = strlen(moreloc+1);
      setpwent();
      for(pass=getpwent();pass;pass=getpwent()) {
	if(!patmatch(pass->pw_name,pat+1))
	   continue;
	n = pass->pw_dir;
	pl = strlen(n);
	if(moreloc && n[pl-1] == '/')
	   pl--;
	n = (char *)malloc(pl+2);
	bcopy(pass->pw_dir,n,pl);
	if(moreloc) {
	  n[pl++] = '/';
	}
	n[pl] = 0;
	if(moreloc && moreloc[1]) {
	  argarray = _dirget(moreloc+1,n,argarray,arglen,reppos,_repend); /* was pat+1 as 1st arg */
	  free(n);
	} else
	  argarray = insertelt(argarray,n,reppos,_repend,arglen);
      }
      endpwent();
      if(moreloc)
	*moreloc = '/';
      return argarray;
    }
  } else {
    if(n) {
      nl = n-pat;
      moreloc = ++n;
      for(pl=0;(*moreloc != '}' || pl) && *moreloc;moreloc++){
	if(*moreloc == '{') pl++;
	if(*moreloc == '}') pl--;
	if(*moreloc == '\\' && moreloc[1]) moreloc++;
      }
      if(!*moreloc++)
	return argarray;
      al = moreloc - n - 1;
      tp = stmp = (char *)ALLOCA(al+1);
      bcopy(n,stmp,al);
      stmp[al] = 0;
      mpl = strlen(moreloc);
      n = (char *)ALLOCA(nl+mpl+al+1);
      bcopy(pat,n,nl);
      while(*stmp){
	for(pl=al=0;(stmp[al] != ',' || pl) && stmp[al];al++){
	  if(stmp[al] == '{') pl++;
	  if(stmp[al] == '}') pl--;
	  if(*stmp == '\\' && stmp[1]) stmp++;
	}
	bcopy(stmp,n+nl,al);
	bcopy(moreloc,n+al+nl,mpl);
	n[al+mpl+nl] = 0;
	if(!(just_tilde = isregex(n)) && !wc) {
	  ne = (char *)malloc(strlen(n)+1);
	  strcpy(ne,n);
	  argarray = insertelt(argarray,ne,reppos,_repend,arglen);
	}
	else
	    argarray = dirget(n,argarray,arglen,reppos,_repend,just_tilde==1,1);
	if(*(stmp += al))
	  stmp++;
      }
      FREEA(tp);
      FREEA(n);
      return argarray;
    }
    if(*pat == '/') {
      path = "/";
      pat++;
    } else path = "";
  }
  argarray = _dirget(pat,path,argarray,arglen,reppos,_repend);
  if(al)
    FREEA(path);
  return argarray;
}

char **_dirget(pat,path,argarray,arglen,repstart,repend)
char *pat,*path,**argarray;
int *arglen,*repend;
{
  DIR *curdir;
  struct direct *curent;
  int pl,nl,mpl = 0;
  char *moreloc,*stmp,*n;

  if(moreloc = index(pat,'/'))
    *moreloc++ = 0;
  if(!(curdir = opendir(path))){
    if(moreloc)
      *--moreloc = '/';
    return argarray;
  }
  if(*pat != '.' || (pat[1]&&pat[1] != '.'||pat[2])){
    readdir(curdir);
    readdir(curdir);
  }
  while(curent = readdir(curdir)){
    n = curent->d_name;
    if((*n != '.' || *pat == '.' || !_nodots) && patmatch(n,pat)) {
      pl = strlen(path);
      nl = strlen(n);
      if(moreloc) {
	if(pl+nl+2 > mpl){
	  if(mpl)
	    FREEA(stmp);
	  mpl = ((pl+nl+6)/5)*5;
	  stmp = (char *)ALLOCA(mpl);
	}
	strcpy(stmp,path);
	strcpy(stmp+pl,n);
	stat(stmp,&ST);
	if((ST.st_mode&S_IFMT) == S_IFDIR){/* || (ST.st_mode&S_IFMT) == S_IFLNK){*/
	  strcpy(stmp+pl+nl,"/");
	  if(!*moreloc)
	    argarray = insertelt(argarray,(char *)strcpy(malloc(pl+nl+2),stmp),repstart,repend,arglen);
	  else
	    argarray = _dirget(moreloc,stmp,argarray,arglen,repstart,repend);
	}
      }
      else{
	stmp = (char *)malloc(pl+nl+1);
	strcpy(stmp,path);
	strcpy(stmp+pl,n);
	argarray = insertelt(argarray,stmp,repstart,repend,arglen);
      }
    }
  }
  if(moreloc){
    *--moreloc = '/';
    if(mpl)
      FREEA(stmp);
  }
  closedir(curdir);
  return argarray;
}

char **insertelt(argarray,p,s,repend,cnt)
char **argarray;
char *p;
int *repend,*cnt;
{
  int m,c,e = *repend;

  if(s>e){
    c = 0;
    m = s;
  }
  else
    while(s<=e){
      m = (s + e) >> 1;
      c = strcmp(p,argarray[m]);
      if(!c){
	free(p);
	return argarray;
      }
      if(c<0)
	e = m - 1;
      else
	s = m + 1;
    }
  if(c>0)
    m++;
  (*cnt)++;
  (*repend)++;
  if(!(*cnt%5))
    argarray = (char **)realloc(argarray,(*cnt+5)*sizeof(*argarray));
  for(s = *cnt;s>m;s--)
    argarray[s] = argarray[s-1];
  argarray[m] = p;
  return argarray;
}


char **delelt(argarray,m,repend,cnt)
char **argarray;
int *repend,*cnt;
{
  int e = *repend;

  if(m>e)
    return NULL;
  (*cnt)--;
  (*repend)--;
  e = *cnt;
  for(;m<=e;m++)
    argarray[m] = argarray[m+1];
  if(*cnt%5== 4)
    argarray = (char **)realloc(argarray,(*cnt+1)*sizeof(*argarray));
  return argarray;
}

/****************************************************************/

patmatch(buf,pat)
char *buf,*pat;
{
  int match = 1,m,n,l,tm;
  char *p,*t,*tp;

  while(*pat && match) {
    switch(*pat) {
      case '[':
	pat++;
	if(*pat != '^') {
	  n = 1;
	  match = 0;
	} else {
	  pat++;
	  n = 0;
	}
	while(*pat != ']'){
	  if(*pat == '\\') pat++;
	  if(!*pat /* || *pat == '/' */ ) return -1;
	  if(pat[1] == '-'){
	    m = *pat;
	    pat += 2;
	    if(*pat == '\\' && *pat)
		pat++;
	    if(*buf >= m && *buf <= *pat)
		match = n;
	    if(!*pat)
		pat--;
	  } else if(*buf == *pat) match = n;
	  pat++;
	}
	buf++;
	break;
      case '{':
	p = ++pat;
	if(*p == '^'){
	  p = ++pat;
	  tm = 0;
	} else tm = 1;
	n = l = 0;
	while((*pat != '}' || l) && *pat){
	  if(*pat == '{') l++;
	  if(*pat == '}') l--;
	  n++;
	  pat++;
	}
	if(!*pat++)
	  return -1;
	t = (char *)ALLOCA(n+1);
	bcopy(p,t,n);
	t[n] = 0;
	tp = t;
	m = strlen(pat);
	n += !m; /* if no chars to match, make sure there's room for a '*' */
	p = (char *)ALLOCA(m+n+1);
	while(*t){
	  for(n=0;(t[n] != ',' || l) && t[n];n++){
	    if(t[n] == '{') l++;
	    if(t[n] == '}') l--;
	  }
	  bcopy(t,p,n);
	  bcopy(pat,p+n,m);
	  p[n+m] = 0;
	  if(patmatch(buf,p)){
	    FREEA(p);
	    FREEA(tp);
	    return tm;
	  }
	  if(*(t += n))
	    t++;
	}
	FREEA(tp);
	if(tm ^= 1){
	  *p = '*';
	  bcopy(pat,p+1,m);
	  p[m+1] = 0;
	  tm = patmatch(buf,p);
	}
	FREEA(p);
	return tm;
      case '*':
	pat++;
	if(!*pat) return 1;
	while(*buf && !(match = patmatch(buf++,pat)));
	return match;
      case '?':
	if(!*buf) return 0;
	buf++;
	break;
      case '\\':
	if(*pat)
	    pat++;
      default:
	match = (*buf++ == *pat);
	break;
    }
    pat++;
    if(match<1) return match;
  }
  if(!*buf) return match;
  return 0;
}

