/*                               -*- Mode: C -*- 
 * 
 * uSystem Version 4.3.2, Copyright (C) Peter A. Buhr and Richard A. Stroobosscher 1990
 * 
 * uStream.c -- Stream I/O routines.
 * 
 * Author           : Rick Stroobosscher
 * Created On       : Fri Feb  9 15:17:47 1990
 * Last Modified By : Peter A. Buhr
 * Last Modified On : Mon Mar 11 10:37:13 1991
 * Update Count     : 263
 */

#define __U_KERNEL__

#include "uUnix.h"
#include "uSystem.h"
#include "uStream.h"

#ifdef __sgi__
#include <bsd/sys/file.h>
#else
#include <sys/file.h>
#endif

#ifdef __mips__
#include "configure/stdarg-mips.h"
#else
#include <stdarg.h>
#endif

/* definition of timeval */
#include <sys/time.h>

/*
 * This type is used to indicate whether and when a stream should be set into
 * non-blocking mode.
 *
 * U_FNDELAY_NEVER:  it never blocks (e.g. file access) so don't bother with FNDELAY
 * U_FNDELAY_DEMAND: it blocks, but we want it to be blocking mode most of the time (stdin)
 *			set it to non-blocking mode on each syscall.
 * U_FNDELAY_ALWAYS: it blocks.  Set it to non-blocking mode when the file is opened.
 *			It is reset to the original mode when the file is closed.
 */
typedef enum { U_FNDELAY_NEVER, U_FNDELAY_DEMAND, U_FNDELAY_ALWAYS } uBlockVal;

typedef struct uStreamD {
    uSemaphore mutex;
    FILE *file;
    uCluster cluster;
    uBlockVal block;
    int flags;
} uStreamD;

uStream uStdin  = NULL;
uStream uStdout = NULL;
uStream uStderr = NULL;

void uInstallStandardStream( void ) {
    uStdin = uFinstall( stdin );
    uStdout = uFinstall( stdout );
    uStderr = uFinstall( stderr );
} /* uInstallStandardStream */

void uDestallStandardStream( void ) {
    uFdestall( uStdin );
    uFdestall( uStdout );
    uFdestall( uStderr );
} /* uInstallStandardStream */

inline void uStreamEnter( uStream stream ) {
    uP( &(stream->mutex) );
    stream->cluster = uMigrate( uThisTask(), stream->cluster );    
} /* uStreamEnter */

inline void uStreamLeave( uStream stream ) {
    stream->cluster = uMigrate( uThisTask(), stream->cluster );
    uV( &(stream->mutex) );
} /* uStreamLeave */

inline static uStream uCreateStream() {
    uStream stream;

    stream = uMalloc( sizeof( uStreamD ) );		/* allocate a stream descriptor */
    stream->mutex = U_SEMAPHORE( 1 );			/* initialize the semaphore */
    stream->cluster = uCreateCluster( 1, 0 );		/* create cluster with one processor and no time slicing */

    return stream;
} /* uCreateStream */

inline static void uDestroyStream( uStream stream ) {
    uDestroyCluster( stream->cluster );
    uFree( stream );
} /* uDestroyStream */

inline static uBlockVal uDoesStreamBlock( uStream stream ) {
    
    /*
     * Determine the type of the file, so that we know whether to block on access or not.
     */
    
    struct stat buf;
    int fd = fileno( stream->file );
    
    if ( fstat( fd, &buf ) < 0 ) {			/* can't stat the file...  set the FNDELAY flag on demand */
	return U_FNDELAY_DEMAND;
    } /* if */
    	
    /*
     * examine the "type" bits
     * 
     * File access may block if file type is:
     * 	S_IFIFO  : named pipe
     * 	S_IFCHR  : character special file (tty/pty)
     * 	S_IFSOCK : socket
     */

    if ( (fd == 0 || fd == 1 || fd == 2) && ( (buf.st_mode & S_IFCHR) == S_IFCHR ) ) {
	/*
	 * Special case the standard streams if a character special device (tty).   
	 * We don't want blocking always turned off for these,
	 * since the tty may be left in that mode on an interrupt.
	 */
	return U_FNDELAY_DEMAND;
    } else {
	return
#if defined(S_IFIFO)
	       (buf.st_mode & S_IFIFO)  == S_IFIFO ||
#endif
	       (buf.st_mode & S_IFCHR)  == S_IFCHR ||
	       (buf.st_mode & S_IFSOCK) == S_IFSOCK
		   ? U_FNDELAY_ALWAYS : U_FNDELAY_NEVER;
    } /* if */
} /* uDoesStreamBlock */

inline static void uSetStreamAccess( uStream stream ) {
    /* if the stream is a blocking stream, set the fcntl bits to allow non blocking access */
    if ( stream->block == U_FNDELAY_ALWAYS ) {
	stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
    } /* if */
} /* uSetStreamAccess */

inline static void uClearStreamAccess( uStream stream ) {
    /* if the stream is a blocking stream, set the fcntl bits to what they originally were */
    if ( stream->block == U_FNDELAY_ALWAYS ) {
	fcntl( fileno( stream->file ), F_SETFL, stream->flags );
    } /* if */
} /* uClearStreamAccess */

static int uSelectStream( uStream stream, fd_set *set ) {

    struct timeval timeout = { 1, 0 };			/* wake up every second because of race condition between signal and select */
    int found;
    uCluster cluster = uThisCluster();			/* remember this for accessing cluster variables */
    
    /* add this file descriptor to the mask of pending file descriptors */
    
    uP( &(cluster->mutex) );
    FD_SET( fileno( stream->file ), set );
    cluster->nfds += 1;
    
    /* if there is still work to be done, give up the cpu */
    
    if ( uReadyTasks() - cluster->nfds > -1 ) {
	uV( &(cluster->mutex) );			/* release cluster lock */
	uYield();					/* give up the cpu */
	
	/* if the mask is still intact, that means that nobody blocked on a select */
	/* so, remove my intent, and try again. */
	uP( &(cluster->mutex) );
	if ( FD_ISSET( fileno( stream->file ), set ) ) {
	    FD_CLR( fileno( stream->file ), set );
	    cluster->nfds -= 1;
	} /* if */
	uV( &(cluster->mutex ) );	
	found = 1;					/* tell the caller to continue polling */
    } else {
	/* prepare to block, waiting for i/o on any of tasks who declared their intent to do i/o */
	
	uThisProcessor()->state = U_IDLE;		/* change this processor's state to idle */
	uThisProcessor()->cluster->ready.idle = U_TRUE;	/* set the idle flag so that the processor gets woken up */
	
	found = select( FD_SETSIZE, &(cluster->rfds), &(cluster->wfds), &(cluster->efds), &timeout );

	FD_ZERO( &(cluster->rfds) );
	FD_ZERO( &(cluster->wfds) );
	FD_ZERO( &(cluster->efds) );
	cluster->nfds = 0;
	
	uV( &(cluster->mutex) );
    } /* if */
    
    return found;
    
} /* uSelectStream */

int uFeof( uStream stream ) {
    return feof( stream->file );
}/* uFeof */

int uFerror( uStream stream ) {
    return ferror( stream->file );
} /* uFerror */

void uClearerr( uStream stream ) {
    clearerr( stream->file );
} /* uClearerr */

int uFileno( uStream stream ) {
    return fileno( stream->file );
} /* uFileno */

inline static void uFopenError( uStream stream, char *path, char* type ) {
    uOpenExMsg msg;
    uException* except;
    
    msg.header.errno = errno;
    msg.path = path;
    msg.perms = type;
    msg.flags = 0;                      /* unused for uFopen  */
    msg.mode = 0;                       /* unused for uFopen  */
    
    uStreamLeave( stream );
    uDestroyStream( stream );
    
    switch( msg.header.errno ) {
      case EPERM:
	msg.header.msg = "uFopen: pathname contains high order bit set\n";
	except = &uBadPathEx;
	break;
      case ENOTDIR:
	msg.header.msg = "uFopen: a portion of the path is not a directory\n";
	except = &uBadPathEx;
	break;
      case ENOENT:
	msg.header.msg = "uFopen: file doesn't exist or component doesn't exist or path too long\n";
	except = &uBadPathEx;
	break;
      case EFAULT:
	msg.header.msg = "uFopen: path parameter points outside process address space\n";
	except = &uBadPathEx;
	break;
      case ELOOP:
	msg.header.msg = "uFopen: symbolic link loop\n";
	except = &uBadPathEx;
	break;
      case EOPNOTSUPP:
	msg.header.msg = "uFopen: Attempt to open socket\n";
	except = &uBadPathEx;
	break;
      case EACCES:
	msg.header.msg = "uFopen: wrong permissions on file or directory\n";
	except = &uNoPermsEx;
	break;
      case EISDIR:
	msg.header.msg = "uFopen: file is a directory, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EROFS:
	msg.header.msg = "uFopen: read only file system, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case ETXTBSY:
	msg.header.msg = "uFopen: shared text being executed, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EBUSY:
	msg.header.msg = "uFopen: special file already opened with exlusive access\n";
	except = &uNoPermsEx;
	break;
      case EMFILE:
	msg.header.msg = "uFopen: this process has too many files open\n";
	except = &uNoFilesEx;
	break;
      case ENFILE:
	msg.header.msg = "uFopen: too many files open in the system\n";
	except = &uNoFilesEx;
	break;
      case EIO:
	msg.header.msg = "uFopen: I/O failed\n";
	except = &uOpenIOFailedEx;
	break;
      case ENOSPC:
	msg.header.msg = "uFopen: no space on filesystem\n";
	except = &uOpenNoSpaceEx;
	break;
      case ENXIO:
	msg.header.msg = "uFopen: device doesn't exist or opening comm device with no delay and no carrier\n";
	except = &uOpenEx;
	break;
      default:
	msg.header.msg = "uFopen: unknown open error\n";
	except = &uOpenEx;
	break;
    } /* switch */
    uRaise( NULL, except, &msg, sizeof(msg) );
} /* uFopenError */

uStream uFopen( char *path, char *type ) {

    uStream stream;

    stream = uCreateStream();
    uStreamEnter( stream );
    
    stream->file = fopen( path, type );			/* open the file */
    if ( stream->file == NULL ) {			/* if error while opening file */
#ifdef __U_UNIXRC__
	uStreamLeave( stream );
	uDestroyStream( stream );
	stream = NULL;
#else
	uFopenError( stream, path, type );
#endif
    } else {
	stream->block = uDoesStreamBlock( stream );
	uSetStreamAccess( stream );
	uStreamLeave( stream );
    } /* if */

    return( stream );					/* return reference to stream */
} /* uFopen */

inline static void uFflushError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFflush: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFflush: no space on filesystem\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFflush: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFflush: writing to pipe w/o reader, or unconnected socket\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFflush: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      case EFBIG:
	msg.msg = "uFflush: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      default:
	msg.msg = "uFflush: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFflushError */

int uFflush( uStream stream ) {
    int code;

    uStreamEnter( stream );
    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    code = fflush( stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    code = fflush( stream->file );
	}
	if ( code != EOF ) break;
	if ( errno != EWOULDBLOCK && errno != EINTR ) {
#ifdef __UNIXRC__
	    break;
#else
	    uFflushError( stream );
#endif
	} /* if */
	code = uSelectStream( stream, &(uThisCluster()->wfds ) );
	if ( code < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __UNIXRC__
		break;
#else
		uFflushError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */
    uStreamLeave( stream );
    return code;
} /* uFflush */

inline static void uFcloseError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFclose: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFclose: no space on filesystem\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFclose: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFclose: writing to pipe w/o reader, or unconnected socket\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFclose: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      case EFBIG:
	msg.msg = "uFclose: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      default:
	msg.msg = "uFclose: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* stream */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFcloseError */

int uFclose( uStream stream ) {
    int code;

    uStreamEnter( stream );
    
    uClearStreamAccess( stream );
    
    code = fclose( stream->file );
    if ( code == EOF ) {
#ifdef __U_UNIXRC__
#else
	uFcloseError( stream );
#endif
    } /* if */
    
    uStreamLeave( stream );
    
    if( code != EOF ) {
	uDestroyStream( stream );
    } /* if */

    return code;
} /* uFclose */

inline static void uFgetcError( uStream stream ) {
	    
    uReadWriteExMsg msg;
    uException* except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFgetc: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uFgetc: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uFgetc: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFgetc: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFgetc: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFgetcError */

int uFgetc( uStream stream ) {

    int byte;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif

    uStreamEnter( stream );
    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    byte = fgetc( stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    byte = fgetc( stream->file );
	}
	if ( byte != EOF ) break;
	if ( feof( stream->file ) ) {
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* if */
	if ( errno != EWOULDBLOCK && errno != EINTR ) {
#ifdef __U_UNIXRC__
	    break;
#else
	    uFgetcError( stream );
#endif
	} /* if */
	byte = uSelectStream( stream, &(uThisCluster()->rfds) );
	if ( byte < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uFgetcError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */
    
#ifndef __U_UNIXRC__
    if( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uFgetc: End of File\n";

	uStreamLeave( stream );
    
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );
    } /* if */
#endif
    uStreamLeave( stream );
    return byte;
} /* uFgetc */

inline static void uFgetsError( uStream stream ) {
    
    uReadWriteExMsg msg;
    uException* except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFgets: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uFgets: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uFgets: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFgets: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFgets: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFgetsError */

char *uFgets( char *str, int len, uStream stream ) {
    int byte;
    char *code;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif

    uStreamEnter( stream );
    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    code = fgets( str, len, stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    code = fgets( str, len, stream->file );
	}
	if ( code != NULL ) break;
	if ( feof( stream->file ) ) {
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* if */
	if ( errno != EWOULDBLOCK && errno != EINTR ) {
#ifdef __U_UNIXRC__
	    break;
#else
	    uFgetsError( stream );
#endif
	} /* if */
	byte = uSelectStream( stream, &(uThisCluster()->rfds) );
	if ( byte < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uFgetsError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */

#ifndef __U_UNIXRC__
    if( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uFgets: End Of File\n";

	uStreamLeave( stream );
    
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    uStreamLeave( stream );
    return code;
} /* uFgets */

int uGetc( uStream stream ) {
    return( uFgetc( stream ) );
} /* uGetc */

int uGetchar( void ) {
    return( uFgetc( uStdin ) );
} /* uGetchar */

char *uGets( char *str, int len ) {
    return( uFgets( str, len, uStdin ) );
} /* uGets */

inline static void uUngetcError( uStream stream ) {
    uReadWriteExMsg msg;
    
    uStreamLeave( stream );
    
    msg.errno = 0;
    msg.msg = "uUngetc: Cannot push character back\n";
    
    uRaise( stream, &uReadWriteEx, &msg, sizeof(msg) );
} /* uUngetcError */

int uUngetc( int c, uStream stream ) {
    int byte;

    uStreamEnter( stream );

    byte = ungetc( c, stream->file );
    if ( ferror( stream->file ) ) {
#ifdef __U_UNIXRC__
#else
	uUngetcError( stream );
#endif
    } /* if */

    uStreamLeave( stream );
    
    return byte;
} /* uUngetc */

inline static void uFputcError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFputc: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFputc: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFputc: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFputc: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFputc: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uFputc: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFputc: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFputcError */

int uFputc( char byte, uStream stream ) {
    int code;
    
    uStreamEnter( stream );
    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    code = fputc( byte, stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    code = fputc( byte, stream->file );
	}
	if ( code != EOF ) break;
	if ( errno != EWOULDBLOCK && errno != EINTR ) {
#ifdef __UNIXRC__
	    break;
#else
	    uFputcError( stream );
#endif
	} /* if */
	code = uSelectStream( stream, &(uThisCluster()->wfds ) );
	if ( code < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __UNIXRC__
		break;
#else
		uFputcError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */
    uStreamLeave( stream );
    return code;
} /* uFputc */

inline static void uFputsError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFputs: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFputs: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFputs: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFputs: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFputs: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uFputs: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFputs: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFputsError */

void uFputs( char *str, uStream stream ) {
    int code;
    
    uStreamEnter( stream );
    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    fputs( str, stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    fputs( str, stream->file );
	}
	if ( ! ferror( stream->file ) ) break;
	if ( errno != EWOULDBLOCK && errno != EINTR ) {
#ifdef __UNIXRC__
	    break;
#else
	    uFputsError( stream );
#endif
	} /* if */
	code = uSelectStream( stream, &(uThisCluster()->wfds) );
	if ( code < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __UNIXRC__
		break;
#else
		uFputsError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */
uStreamLeave( stream );
} /* uFputs */

int uPutc( char byte, uStream stream ) {
    return( uFputc( byte, stream ) );
} /* uPutc */

int uPutchar( char byte ) {
    return( uFputc( byte, uStdout ) );
} /* uPutchar */

void uPuts( char *str ) {
    uFputs( str, uStdout );
    uFputc( '\n', uStdout );				/* to simulate puts() behaviour */
} /* uPuts */

inline static void uDoScanError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uDoscan: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uDoscan: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uDoscan: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uDoscan: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uDoscan: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    
    uStreamLeave( stream );
    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uDoScanError */

static int uDoscan( uStream stream, char *fmt, void *args ) {

    int count;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif
    
    uStreamEnter( stream );

    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    count = _doscan( stream->file, fmt, args );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    count = _doscan( stream->file, fmt, args );
	}
	if ( count != -1 ) break;

	if ( feof( stream->file ) ) {
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* if */
	
	if ( errno != EINTR && errno != EWOULDBLOCK ) {
#ifdef __U_UNIXRC__
	    break;
#else
	    uDoScanError( stream );
#endif
	} /* if */

	count = uSelectStream( stream, &(uThisCluster()->rfds) );
	if ( count < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uDoScanError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */
    
#ifndef __U_UNIXRC__
    if( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uDoscan: End of File\n";

	uStreamLeave( stream );
	    
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );
    } /* if */
#endif

uStreamLeave( stream );
    
    return( count );
} /* uDoScan */

int uFscanf( uStream stream, char *fmt, ... ) {

    int count;
    va_list args;

    va_start( args, fmt );
    count = uDoscan( stream, fmt, args );
    va_end( args );

    return( count );
} /* uFscanf */

int uScanf( char *fmt, ... ) {

    int count;
    va_list args;

    va_start( args, fmt );
    count = uDoscan( uStdin, fmt, args );
    va_end( args );

    return( count );
} /* uScanf */

inline static void uDoprntError( uStream stream ) {

    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uDoprnt: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uDoprnt: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uDoprnt: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uDoprnt: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uDoprnt: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uDoprnt: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uDoprnt: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uDoprntError */

static int uDoprnt( char *fmt, void *args, uStream stream ) {

    int code;

    uStreamEnter( stream );

    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    code = _doprnt( fmt, args, stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    code = _doprnt( fmt, args, stream->file );
	}
	if ( code != -1 ) break;
	if ( errno != EINTR && errno != EWOULDBLOCK ) {
	    if ( ferror( stream->file ) ) {
		/* We get spurious errors when printing on glass ttys that result in the loss of */
		/* a carriage return character. We don't know why, but it is checked for, and the */
		/* carriage return is inserted */
		if ( stream->file->_flag & _IOLBF ) {
		    if( stream->block == U_FNDELAY_DEMAND ) {
			stream->flags = fcntl( fileno( stream->file ), F_GETFL );
			fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
			fputc( '\n', stream->file );
			fcntl( fileno( stream->file ), F_SETFL, stream->flags );
		    } else {
			fputc( '\n', stream->file );
		    }
		    clearerr( stream->file );
		    break;
		} /* if */
	    } /* if */
#ifdef __U_UNIXRC__
	    break;
#else
	    uDoprntError( stream );
#endif
	} /* if */
	code = uSelectStream( stream, &(uThisCluster()->rfds ) );
	if ( code < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uDoprntError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */
    
#if 0
    /* VAX BSD 4.3 generates spurious errors from _doprnt so we gave up checking. */
    /* It doesn't seem to cause any problems that we can detect. */
    if ( code < 0 ) {
	uError( "uDoprnt( %s, %x, %x ) : STREAM ERROR.\n", fmt, args, stream );
    } /* if */
#endif
    
    uStreamLeave( stream );

    return code;
} /* uDoprnt */
    
int uFprintf( uStream stream, char *fmt, ... ) {
    
    int code;
    va_list args;

    va_start( args, fmt );
    code = uDoprnt( fmt, args, stream );
    va_end( args );

    return code;
} /* uFprintf */

int uPrintf( char *fmt, ... ) {

    int code;
    va_list args;

    va_start( args, fmt );
    code = uDoprnt( fmt, args, uStdout );
    va_end( args );

    return code;
} /* uPrintf */

inline static void uFreadError( uStream stream ) {
    uReadWriteExMsg msg;
    uException* except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFread: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case EFAULT:
	msg.msg = "uFread: buffer points ouside address space\n";
	except = &uReadWriteEx;
	break;
      case EBADF:
	msg.msg = "uFread: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case ENOMEM:
	msg.msg = "uFread: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFread: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */	    
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFreadError */

int uFread( char *buf, int size, int count, uStream stream ) {

    int code;
    int total = 0;
#ifndef __U_UNIXRC__
    int endoffile = U_FALSE;
#endif

    uStreamEnter( stream );

    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    code = fread( buf, size, count, stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    code = fread( buf, size, count, stream->file );
	}
	total += code;
	buf += code * size;
	count -= code;
	if ( code != 0 ) break;
	if ( feof( stream->file ) ) {
#ifndef __U_UNIXRC__
	    endoffile = U_TRUE;
#endif
	    break;
	} /* if */
	if ( errno != EWOULDBLOCK && errno != EINTR ) {
#ifdef __U_UNIXRC__
	    break;
#else
	    uFreadError( stream );
#endif
	} /* if */

	code = uSelectStream( stream, &(uThisCluster()->rfds ) );
	if ( code < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uFreadError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */

#ifndef __U_UNIXRC__
    if( endoffile ) {
	uEofExMsg msg;

	msg.msg = "uFread: End Of File\n";

	uStreamLeave( stream );
	uRaise( stream, &uEofEx, &msg, sizeof( msg ) );
    } /* if */
#endif

    uStreamLeave( stream );
    return total;
} /* uFread */

inline static void uFwriteError( uStream stream ) {
    uReadWriteExMsg msg;
    uException *except;
    
    msg.errno = errno;
    
    uStreamLeave( stream );
    
    switch( msg.errno ) {
      case EIO:
	msg.msg = "uFwrite: I/O failed\n";
	except = &uIOFailedEx;
	break;
      case ENOSPC:
	msg.msg = "uFwrite: no space on file system\n";
	except = &uNoSpaceEx;
	break;
      case EFBIG:
	msg.msg = "uFwrite: write exceeds process or system file size limit\n";
	except = &uNoSpaceEx;
	break;
      case EBADF:
	msg.msg = "uFwrite: bad file descriptor\n";
	except = &uBadFileEx;
	break;
      case EPIPE:
	msg.msg = "uFwrite: writing to pipe w/o reader or unconnected socket.\n";
	except = &uBadFileEx;
      case ENOMEM:
	msg.msg = "uFwrite: cannot increase memory size\n";
	except = &uReadWriteEx;
	break;
      default:
	msg.msg = "uFwrite: unknown error\n";
	except = &uReadWriteEx;
	break;
    } /* switch */
    uRaise( stream, except, &msg, sizeof(msg) );
} /* uFwriteError */

int uFwrite( char *buf, int size, int count, uStream stream ) {

    int code;

    uStreamEnter( stream );

    for ( ;; ) {
	if( stream->block == U_FNDELAY_DEMAND ) {
	    stream->flags = fcntl( fileno( stream->file ), F_GETFL );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags | FNDELAY );
	    code = fwrite( buf, size, count, stream->file );
	    fcntl( fileno( stream->file ), F_SETFL, stream->flags );
	} else {
	    code = fwrite( buf, size, count, stream->file );
	}
	if ( code != 0 ) break;
	if ( errno != EWOULDBLOCK && errno != EINTR ) {
#ifdef __U_UNIXRC__
	    break;
#else
	    uFwriteError( stream );
#endif
	} /* if */
	code = uSelectStream( stream, &(uThisCluster()->wfds) );
	if ( code < 0 ) {
	    if ( errno != EINTR ) {
#ifdef __U_UNIXRC__
		break;
#else
		uFwriteError( stream );
#endif
	    } /* if */
	} /* if */
    } /* for */

    uStreamLeave( stream );

    return code;
} /* uFwrite */

inline static void uFseekError( uStream stream ) {
    uReadWriteExMsg msg;

    msg.msg = "uFseek: seek failed\n";
    msg.errno = errno;

    uRaise( stream, &uReadWriteEx, &msg, sizeof(msg) );
} /* uFseekError */

int uFseek( uStream stream, long numbytes, int origin ) {

    int code;

    uStreamEnter( stream );

    code = fseek( stream->file, numbytes, origin );
    if ( code == -1 ) {
#ifdef __UNIXRC__
#else
	uFseekError( stream );
	uStreamLeave( stream );
#endif
    } /* if */
    uStreamLeave( stream );
    return code;
} /* uFseek */

void uRewind( uStream stream ) {

    uStreamEnter( stream );
    rewind( stream->file );
    uStreamLeave( stream );
    
} /* uRewind */

inline static void uFreopenError( char *path, char *perms, uStream stream ) {

    uOpenExMsg msg;
    uException* except;
    
    msg.header.errno = errno;
    msg.path = path;
    msg.perms = perms;
    msg.flags = 0;                      /* unused for uFopen  */
    msg.mode = 0;                       /* unused for uFopen  */
    
    uStreamLeave( stream );
    
    switch( msg.header.errno ) {
      case EPERM:
	msg.header.msg = "uFreopen: pathname contains high order bit set\n";
	except = &uBadPathEx;
	break;
      case ENOTDIR:
	msg.header.msg = "uFreopen: a portion of the path is not a directory\n";
	except = &uBadPathEx;
	break;
      case ENOENT:
	msg.header.msg = "uFreopen: file doesn't exist or component doesn't exist or path too long\n";
	except = &uBadPathEx;
	break;
      case EFAULT:
	msg.header.msg = "uFreopen: path parameter points outside process address space\n";
	except = &uBadPathEx;
	break;
      case ELOOP:
	msg.header.msg = "uFreopen: symbolic link loop\n";
	except = &uBadPathEx;
	break;
      case EOPNOTSUPP:
	msg.header.msg = "uFreopen: Attempt to open socket\n";
	except = &uBadPathEx;
	break;
      case EACCES:
	msg.header.msg = "uFreopen: wrong permissions on file or directory\n";
	except = &uNoPermsEx;
	break;
      case EISDIR:
	msg.header.msg = "uFreopen: file is a directory, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EROFS:
	msg.header.msg = "uFreopen: read only file system, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case ETXTBSY:
	msg.header.msg = "uFreopen: shared text being executed, cannot open for write\n";
	except = &uNoPermsEx;
	break;
      case EBUSY:
	msg.header.msg = "uFreopen: special file already opened with exlusive access\n";
	except = &uNoPermsEx;
	break;
      case EMFILE:
	msg.header.msg = "uFreopen: this process has too many files open\n";
	except = &uNoFilesEx;
	break;
      case ENFILE:
	msg.header.msg = "uFreopen: too many files open in the system\n";
	except = &uNoFilesEx;
	break;
      case EIO:
	msg.header.msg = "uFreopen: I/O failed\n";
	except = &uOpenIOFailedEx;
	break;
      case ENOSPC:
	msg.header.msg = "uFreopen: no space on filesystem\n";
	except = &uOpenNoSpaceEx;
	break;
      case ENXIO:
	msg.header.msg = "uFreopen: device doesn't exist or opening comm device with no delay and no carrier\n";
	except = &uOpenEx;
	break;
      default:
	msg.header.msg = "uFreopen: unknown open error\n";
	except = &uOpenEx;
	break;
    } /* switch */
    uRaise( NULL, except, &msg, sizeof(msg) );
} /* uFreopenError */

uStream uFreopen( char *path, char *perms, uStream stream ) {

    FILE *fp;

    uStreamEnter( stream );

    uClearStreamAccess( stream );
    
    fp = freopen( path, perms, stream->file );

    if ( fp == NULL ) {					/* if error in opening file */
#ifdef __U_UNIXRC__
	uStreamLeave( stream );
	stream = NULL;
#else
	uFreopenError( path, perms, stream );
#endif
    } else {						/* success */
	stream->file = fp;
	stream->block = uDoesStreamBlock( stream );
	uSetStreamAccess( stream );
	uStreamLeave( stream );
    } /* if */

    return( stream );					/* return reference to stream, or NULL */
} /* uFreopen */
    
uStream uFinstall( FILE *file ) {

    uStream stream;

    stream = uMalloc( sizeof( uStreamD ) );		/* allocate a stream descriptor */
    stream->mutex = U_SEMAPHORE( 1 );			/* initialize the semaphore */
    stream->cluster = uThisCluster();			/* assign stream to the System cluster */
    stream->file = file;
    stream->block = uDoesStreamBlock( stream );
    uSetStreamAccess( stream );

    return stream;
} /* uFinstall */
 
void uFdestall( uStream stream ) {

    uFree( stream );

} /* uFdestall */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
