/*                               -*- Mode: C -*- 
 *
 * uSystem Version 4.3.2, Copyright (C) Hamish Macdonald 1990
 *
 * uExcept.c -- uSystem exception handling routines.
 *
 * Author          : Hamish Macdonald
 * Created On      : Mon Jun 11 15:15:38 1990
 * Last Modified By: Peter A. Buhr
 * Last Modified On: Fri Mar  1 15:51:13 1991
 * Update Count    : 100
 */

#define  __U_KERNEL__

#include "uUnix.h"
#include "uSystem.h"
#include "uKernel.i"
#include "uMachine.i"

uException uAny = U_EXCEPTION( U_NULL );	                        /* The "catch-all" exception */

int uExceptSave( register uEBlock block ) {
    
    /*
     * This function saves the context of a region in which exception
     * handlers are active.  Called by the uExcept macro.
     */
    
    register uCoroutine cor = uWorkCoroutine;
    
    block->prev = cor->exceptstack;
    cor->exceptstack = block;
    block->exception = NULL;
    block->data = NULL;
    block->len = 0;
    block->return_addr = NULL;
    block->reraise = U_FALSE;
    
    uSaveContext( block->context );
    return 0;
} /* uExceptSave */

void uExceptEnd( register uEBlock block ) {
    
    /*
     * This function does the cleanup required at the end of a
     * region in which exception handlers are active.  It is 
     * called if no exceptions were raised, or when a handler 
     * has completed execution.
     */
    
    if( block->data ) {
	uFree( block->data );
	block->data = NULL;
    } /* if */
    
    uWorkCoroutine->exceptstack = block->prev;
} /* uExceptEnd */

int uExceptCheck( uEBlock block, void* obj, uException *excp, void *dataptr, int *lenptr ) {
    
    /*
     * This function is called by the uWhen macro to check if the user supplied
     * exception pointer in the uWhen macro matches the exception raised.  If so,
     * the data pointer is copied into the user's pointer, the length is put into the
     * user's length variable and a 1 is returned.
     * Otherwise, 0 is returned.
     *
     * The "obj" parameter must either match the "object" pointer in the exception block or
     * be NULL.
     *
     * An exception matches if the exception pointed to by the "block" parameter
     * (the exception actually raised) either is equal to the "excp" parameter, or
     * the "excp" parameter is equal to one of the addresses in the parent chain
     * of the exception pointed to by the "block" parameter.   If the passed
     * exception address is the address of uAny, then it is a match.
     */
    
    if( obj && obj != block->object )
	return 0;
    
    if( excp != &uAny ) {
	register uException *e = block->exception;
	
	while( e && e != excp) {
	    e = e->parent;
	} /* while */
	
	if( !e || e != excp ) {
	    return 0;
	} /* if */
    } /* if */
    
    if( !dataptr && block->data ) {               /* Error if there is data but no data pointer supplied */
	uError( "uWhen(0x%x,0x%x,0x%x,0x%x) : NULL DATA POINTER PROVIDED (0x%x,%d)\nException raised from 0x%x\n",
	       obj, excp, dataptr, lenptr, block->data, block->len, block->return_addr );
    } /* if */

    if( !lenptr && block->data ) {		  /* Error if there is data but no length pointer supplied */
	uError( "uWhen(0x%x,0x%x,0x%x,0x%x) : NULL LENGTH POINTER PROVIDED (0x%x,%d)\nException raised from 0x%x\n",
	       obj, excp, dataptr, lenptr, block->data, block->len, block->return_addr );
    } /* if */

    if( block->data ) {				  /* if there is data, copy the data and length pointers */
	(*(void**)dataptr) = block->data;
	*lenptr = block->len;
    } else {					  /* no data, but the user may have supplied pointers */
	if( dataptr ) {				  /* if there is a data pointer, set the data ptr to NULL */
	    (*(void**)dataptr) = NULL;
	} /* if */
	if( lenptr ) {				  /* if there is a length pointer, set the length to 0 */
	    *lenptr = 0;
	} /* if */
    } /* if */
    
    return 1;
} /* uExceptCheck */

void uRaise( register void* obj,
	    register uException* exception,
	    register void* data,
	    register int len ) {
    
    /*
     * This function causes an exception to be raised in the current
     * coroutine.  The address of the exception to be raised is given
     * by the first argument, the data to be passed to the handler in the
     * second argument, and the length of the data in the third argument.
     */
    
    register uCoroutine cor = uWorkCoroutine;
    register uEBlock block = cor->exceptstack;
    register uEBlock iblock = block;                                  /* Initial exception block */
    
    /*
     * Search for an exception block which
     * has a NULL exception field.
     * If the exception field is non-NULL,
     * it means that the exception block is
     * already handling an exception.
     */
    while( block && block->exception ) {
	/*
	 * If the data pointer is non-null and this is NOT a reraise,
	 * free it, since we are bypassing an active exception block (i.e. we
	 * are executing in a handler for that block) and the exception
	 * data must be freed.
	 */
	if( block->data && !block->reraise ) {
	    uFree( block->data );
	    block->data = NULL;
	}
	block = block->prev;
    } /* while */
    cor->exceptstack = block;
    
    if( block ) {
	/*
	 * Set up the exception address, object address, 
	 * data pointer and length in the
	 * exception block.
	 */
	block->exception = exception;
	block->object = obj;
	
	/*
	 * If iblock (the first exception block on the stack) is not the same as block
	 * and the reraise field of iblock is non-zero, then this is a reraise done by the
	 * uExceptEnd macro.  The data pointer is copied from the old block to the new block,
	 * and no new memory is allocated.  The return address (where uRaise was called from) is
	 * also propagated from the exception block.
	 */
	if( block != iblock && iblock && iblock->reraise ) {
	    block->data = data;
	    block->len = len;
	    block->return_addr = iblock->return_addr;
	} else {
	    /*
	     * Must allocate data area for exception data (but only if passed data is non-null).
	     */
	    if( data && len > 0 ) {
		block->data = uLowMalloc( len );
		if( block->data ) {
		    uCopy( data, block->data, len );
		} else {
		    /* need to raise exception ? */
		    uError( "uRaise(0x%x,0x%x,0x%x,%d): Memory allocation failed.  Called from 0x%x\n", obj,
			   exception, data, len, uReadReturnAddress() );
		} /* if */
	    } else {
		block->data = NULL;
	    }
	    block->len = len;
	    block->return_addr = uReadReturnAddress();
	}
	
	uRestoreContext( block->context );
    }
    
    /*
     * Check if it is one of the uSystem defined exceptions.  If it is, then the message will be displayed and execution aborted.
     */
    if( exception == &uSystemEx ||
       exception == &uSystemEx ||
       exception == &uCreationEx ||
       exception == &uActiveTasksEx ) {
	uSystemExMsg msg = (uSystemExMsg)data;

	uError( msg );
    } else if ( exception == &uDataCommEx ) {
	uDataCommExMsg *msg = (uDataCommExMsg*)data;

	uFputs( msg->msg, uStderr );
	uError( "uDataCommEx( 0x%x, %d, 0x%x, %d ) : ERROR IN DATA COMMUNICATION.\n",
	       msg->sbuf, msg->slen, msg->rbuf, msg->rlen );
    } else if ( exception == &uDataCopyEx ) {
	uDataCopyExMsg *msg = (uDataCopyExMsg*)data;

	uFputs( msg->msg, uStderr );
	uError( "uDataCopyEx( 0x%x, %d, 0x%x, %d ) : ERROR IN DATA COPYING FROM DATA SENDER TO DATA RECEIVER.\n",
	       msg->sbuf, msg->slen, msg->rbuf, msg->rlen );
    } else if ( exception == &uSendMsgTooLongEx ) {
	uSendMsgTooLongExMsg *msg = (uSendMsgTooLongExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uSendMsgTooLongEx(%s, 0x%x, %d, 0x%x, %d) : MESSAGE IS TOO LONG TO BE SENT FROM SENDING TASK: %s.\n",
	       uGetName(msg->receiver), msg->head.sbuf, msg->head.slen, msg->head.rbuf, msg->head.rlen,
	       uGetName(msg->sender) );
    } else if ( exception == &uAbsorbAreaTooShortEx ) {
	uAbsorbAreaTooShortExMsg *msg = (uAbsorbAreaTooShortExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uAbsorbAreaTooShortEx(%s, 0x%x, %d, 0x%x, %d) : ABSORB MESSAGE BUFFER IS TOO SHORT TO RECEIVE MESSAGE FROM ABSORBING TASK: %s.\n",
	       uGetName(msg->dier), msg->head.sbuf, msg->head.slen, msg->head.rbuf, msg->head.rlen,
	       uGetName(msg->absorber) );
    } else if ( exception == &uForwardMsgTooLongEx ) {
	uForwardMsgTooLongExMsg *msg = (uForwardMsgTooLongExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uForwardMsgTooLongEx(%s, 0x%x, %d, 0x%x, %d) : MESSAGE TOO LONG TO BE FORWARDED FROM TASK: %s BY FORWARDING TASK: %s.\n",
	       uGetName(msg->receiver), msg->head.sbuf, msg->head.slen, msg->head.rbuf, msg->head.rlen,
	       uGetName(msg->sender), uGetName(msg->forwarder) );
    } else if ( exception == &uReplyAreaTooShortEx ) {
	uReplyAreaTooShortExMsg *msg = (uReplyAreaTooShortExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uReplyAreaTooShortEx(%s, 0x%x, %d) : REPLY MESSAGE BUFFER IS TOO SHORT TO TAKE MESSAGE FROM REPLYING TASK: %s.\n",
	       uGetName(msg->sender), msg->head.rbuf, msg->head.rlen, uGetName(msg->replier) );
    } else if ( exception == &uSuspendMsgTooLongEx ) {
	uSuspendMsgTooLongExMsg *msg = (uSuspendMsgTooLongExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uSuspendMsgTooLongEx(0x%x, %d, 0x%x, %d) : MESSAGE IS TOO LONG TO BE SENT FROM SUSPENDING COROUTINE: %s.\n",
	       msg->head.sbuf, msg->head.slen, msg->head.rbuf, msg->head.rlen, uGetName(msg->restarter) );
    } else if ( exception == &uResumeMsgTooLongEx ) {
	uResumeMsgTooLongExMsg *msg = (uResumeMsgTooLongExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uResumeMsgTooLongEx(%s, 0x%x, %d, 0x%x, %d) : MESSAGE IS TOO LONG TO BE SENT FROM RESUMING COROUTINE: %s.\n",
	       uGetName(msg->resumed), msg->head.sbuf, msg->head.slen, msg->head.rbuf, msg->head.rlen, uGetName(msg->restarter) );
    } else if ( exception == &uSyncFailEx ) {
	uSyncFailExMsg *msg = (uSyncFailExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uSyncFailEx : SYNCHRONIZATION FAILED BETWEEN TASK %s and TASK %s.\n",
	       uGetName(msg->sender), uGetName(msg->receiver) );
    } else if ( exception == &uNotReplyBlockedEx ) {
	uNotReplyBlockedExMsg *msg = (uNotReplyBlockedExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uNotReplyBlockedEx(%s, 0x%x, %d) : REPLYING TASK: %s HAS NOT RECEIVED A MESSAGE FROM SENDER %s TASK OR IT HAS ALREADY REPLIED TO THAT TASK.\n",
	       uGetName(msg->receiver), msg->head.sbuf, msg->head.slen, uGetName(msg->receiver), uGetName(msg->sender) );
    } else if ( exception == &uInvalidForwardEx ) {
	uInvalidForwardExMsg *msg = (uInvalidForwardExMsg*)data;

	uFputs( msg->head.head.msg, uStderr );
	uError( "uInvalidForwardEx(%s, 0x%x, %d, %s) : FORWARDING TASK: %s DID NOT RECEIVE A MESSAGE FROM SPECIFIED RECEIVER TASK OR IT HAS ALREADY FORWARDED THE MESSAGE.\n",
	       uGetName(msg->head.receiver), msg->head.head.sbuf, msg->head.head.slen, uGetName(msg->head.sender), uGetName(msg->forwarder) );
    } else if ( exception == &uCreateClusterEx ) {
	uCreateClusterExMsg *msg = (uCreateClusterExMsg*)data;

	uFputs( msg->msg, uStderr );
	uError( "uCreateClusterEx(%ld, %ld, %ld, %ld, %ld) : CANNOT CREATE CLUSTER.\n",
	       msg->cv.Processors, msg->cv.TimeSlice, msg->cv.Spin, msg->cv.StackSize, msg->cv.ArgLen );
    } else if ( exception == &uEmitEx ) {
	uEmitExMsg *msg = (uEmitExMsg*)data;

	uFputs( msg->msg, uStderr );
	uError( "uEmitEx(0x%x, %ld, 0x%x, %ld) : CANNOT EMIT TASK.\n", msg->cluster, msg->space, msg->begin,
	       msg->arglen );
    } else if ( exception == &uCocallEx ) {
	uCocallExMsg *msg = (uCocallExMsg*)data;

	uFputs( msg->msg, uStderr );
	uError( "uCocallEx(0x%x, %d, %ld, 0x%x, %ld) : CANNOT COCALL COROUTINE.\n", msg->rbuf, msg->rlen,
	       msg->space, msg->begin, msg->arglen );
    } else if ( exception == &uCreateProcessorEx ) {
	uCreateProcessorExMsg *msg = (uCreateProcessorExMsg*)data;

	uFputs( msg->msg, uStderr );
	uError( "uCreateProcessorEx() : COULDN'T CREATE PROCESSORS.  CLUSTER CURRENTLY HAS %d PROCESSORS.\n", msg->num_proc );
    } else if ( exception == &uBadCoroutineEx ) {
	uBadCoroutineExMsg *msg = (uBadCoroutineExMsg*)data;

	uFputs( msg->head.msg, uStderr );
	uError( "uBadCoroutineEx(%s, 0x%x, %d, 0x%x, %d) : ATTEMPT TO RESUME A COROUTINE THAT WAS NOT COCALLED BY TASK: %s.\n    OWNER TASK OF COROUTINE TO BE RESUMED: %s.\n",
	       uGetName(msg->restarter), msg->head.rbuf, msg->head.rlen, msg->head.sbuf, msg->head.slen,
	       uGetName(msg->thistask), uGetName(msg->restarter->owner));
    } else if ( exception == &uOutOfMemoryEx ) {
	uOutOfMemoryExMsg* msgp = (uOutOfMemoryExMsg*)data;
	
	uFputs( msgp->msg, uStderr );
	uError( "uOutOfMemoryEx( %d )\n", msgp->size );
    } else if ( exception == &uNoExtendEx ) {
	uNoExtendExMsg* msgp = (uNoExtendExMsg*)data;
	
	uFputs( msgp->head.msg, uStderr );
	uError( "uNoExtendEx( %d, 0x%x)\n", msgp->head.size, msgp->addr );
    } else if( exception == &uIOEx ) {
	uIOExMsg* msgp = (uIOExMsg*)data;
	
	uError( msgp->msg );
    } else if ( exception == &uEofEx ) {
	uEofExMsg* msgp = (uEofExMsg*)data;
	
	uError( msgp->msg );
    } else if ( exception == &uIOErrorEx ||
	       exception ==    &uSocketErrorEx ||
	       exception ==      &uNotSockEx ||
	       exception ==      &uNoBufsEx ||
	       exception ==    &uReadWriteEx ||
	       exception ==      &uIOFailedEx ||
	       exception ==      &uNoSpaceEx ||
	       exception ==      &uBadFileEx ) {
	uIOErrorExMsg* msgp = (uIOErrorExMsg*)data;
	
	uFprintf( uStderr, "IO error with Unix errno %d\n", msgp->errno );
	uError( msgp->msg );
    } else if ( exception == &uCreateSockEx ) {
	uCreateSockExMsg* msgp = (uCreateSockExMsg*)data;
	
	uFputs( msgp->header.msg, uStderr );
	uError( "uCreateSockEx( %d, %d, %d ) with Unix errno %d\n", msgp->af, msgp->type, msgp->protocol,
	       msgp->header.errno );
    } else if ( exception == &uBadSockAddressEx ) {
	uBadSockAddressExMsg* msgp = (uBadSockAddressExMsg*)data;
	
	uFprintf( uStderr, "uBadSockAddressEx( 0x%x, %d )\n", msgp->name, msgp->namelen );
	uError( msgp->header.msg );
    } if ( exception == &uConnFailedEx ) {
	uConnFailedExMsg* msgp = (uConnFailedExMsg*)data;
	
	uFprintf( uStderr, "uConnFailedEx( 0x%x, %d )\n", msgp->name, msgp->namelen );
	uError( msgp->header.msg );
    } else if ( exception == &uOpenEx ||
	       exception ==    &uOpenIOFailedEx ||
	       exception ==    &uOpenNoSpaceEx ||
	       exception ==    &uBadPathEx ||
	       exception ==    &uNoPermsEx ||
	       exception ==    &uNoFilesEx ||
	       exception ==    &uBadParmEx ) {
	uOpenExMsg* msgp = (uOpenExMsg*)data;
	
	uFprintf( uStderr, "%sIO error with Unix errno %d\n", msgp->header.msg, msgp->header.errno );
	uError( "uOpenEx( %s, %s, %d, %d )\n", msgp->path, msgp->perms ? msgp->perms : "(nil)", msgp->flags, msgp->mode );
    } /* if */

    /*
     * no exception handler for this exception
     *
     * The return address used is the one from the initial exception block
     * if this was a reraise by the uExceptEnd macro.
     */
    uError( "uRaise(0x%x,0x%x,0x%x,%d): NO EXCEPTION HANDLER.  Called from 0x%x\n", obj, exception, data, len,
	   (iblock && iblock->reraise) ? iblock->return_addr : uReadReturnAddress() );
} /* uRaise */

void uCause( uTask task, uException* exception, void *data, int len ) {
    /*
     * If the given task is the current task, just do a uRaise.
     */
    if( task == uWorkTask ) {
	uRaise( task, exception, data, len );
    } else if( 0 ) {
	/*
	 * Ensure that the given task exists.
	 */
    } else {
	/*
	 * Stop the task from running somehow.
	 */
	/*
	 * Patch the task descriptor with values taken
	 * from its exception stack entries, if any
	 */
	/*
	 * Allow the task to continue executing.
	 */
    } /* if */
} /* uCause */

/* Local Variables: */
/* compile-command: "dmake" */
/* End: */
