/* RCS      -- $Header: /u2/dvadura/src/generic/dmake/src/RCS/infer.c,v 1.1 90/10/06 12:03:54 dvadura Exp $
-- SYNOPSIS -- infer how to make a target.
-- 
-- DESCRIPTION
--	This file contains the code to infer a recipe, and possibly some new
--	prerequisites for a target which dmake does not know how to make, or
--	has no explicit recipe.
--
--	The inference fails if no path through the inference graph can be
--	found by which we can make the target.
-- 
-- AUTHOR
--      Dennis Vadura, dvadura@watdragon.uwaterloo.ca
--      CS DEPT, University of Waterloo, Waterloo, Ont., Canada
--
-- COPYRIGHT
--      Copyright (c) 1990 by Dennis Vadura.  All rights reserved.
-- 
--      This program is free software; you can redistribute it and/or
--      modify it under the terms of the GNU General Public License
--      (version 1), as published by the Free Software Foundation, and
--      found in the file 'LICENSE' included with this distribution.
-- 
--      This program is distributed in the hope that it will be useful,
--      but WITHOUT ANY WARRANTY; without even the implied warrant of
--      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--      GNU General Public License for more details.
-- 
--      You should have received a copy of the GNU General Public License
--      along with this program;  if not, write to the Free Software
--      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
--
-- LOG
--     $Log:	infer.c,v $
 * Revision 1.1  90/10/06  12:03:54  dvadura
 * dmake Release, Version 3.6
 * 
*/

#include "extern.h"
#include "alloc.h"
#include "db.h"

/* attributes that get transfered from the % start cell to the inferred
 * cells. */

#define A_TRANSFER  (A_EPILOG | A_PRECIOUS | A_SILENT | A_SHELL |\
		     A_SEQ | A_LIBRARY | A_IGNORE | A_PROLOG | A_SWAP)

/* Define local static functions */
static DFALINKPTR _dfa_subset  ANSI((DFALINKPTR, DFASETPTR));
static void 	  _free_dfas   ANSI((DFALINKPTR));
static int	  _count_dots  ANSI((char *));
static char *	  _build_name  ANSI((char *, char *, char *));

static int _prep = -1;	/* Integer value of Prep variable */
static int _dmax;

CELLPTR
Infer_recipe( cp, how, dfa_stack, setdirroot )/*
================================================
	Infer a set of rules for making a target.  We know we have a HOW
	cell attached, but it's prerequisite list may be NIL as is its
	recipe!  A NIL recipe is a prerequisite for calling this routine. */
CELLPTR    cp;
HOWPTR     how;
DFASETPTR  dfa_stack;
CELLPTR    setdirroot;
{
   DFALINKPTR	pdfa;
   DFALINKPTR	dfas;
   CELLPTR	infcell;
   CELLPTR	meta;
   DFASET	top_dfa_stack;
   EDGEPTR      pedge;

   DB_ENTER( "Infer_recipe" );
   DB_PRINT( "inf", (">>> Infer for target [%s]", cp->CE_NAME) );
   DB_PRINT( "mem", ("%s:-> mem %ld", cp->CE_NAME, (long)coreleft()));

   /* If no further transitive closure on this cell then don't perform
    * any more inferences */
   if( cp->ce_attr & A_NOINFER ) DB_RETURN(NIL(CELL));
   if( _prep == -1 ) _prep = atoi(Prep);  /* _dfa_subset needs _prep */

   if( dfa_stack == NIL(DFASET) )
      _dmax = _prep + _count_dots(cp->CE_NAME);

   /* If none of the inference nodes match then forget about the inference.
    * The user did not tell us how to make such a target.  We also stop the
    * Inference if the new set of DFA's is a proper subset of a previous
    * subset and it's PREP counts exceed the value of Prep.
    */
   dfas = _dfa_subset( Match_dfa(cp->CE_NAME), dfa_stack );

#ifdef DBUG
   { char      *tmp;
     DFASETPTR ds;

      tmp = _strdup("");
      for( pdfa = dfas; pdfa != NIL(DFALINK); pdfa = pdfa->dl_next )
         tmp = _strapp( tmp, pdfa->dl_meta->CE_NAME );

      DB_PRINT( "inf", ("Working DFA subset [%s]", tmp) );
      FREE( tmp );

      tmp = _strdup( "{" );
      for( ds = dfa_stack; ds != NIL(DFASET); ds = ds->df_next ) {
	 tmp = _strapp( tmp, "[" );
	 for( pdfa = ds->df_set; pdfa != NIL(DFALINK); pdfa = pdfa->dl_next )
	    tmp = _strapp( tmp, pdfa->dl_meta->CE_NAME );
	 tmp = _strapp( tmp, "]" );
      }
      tmp = _strapp( tmp, "}" );

      DB_PRINT( "inf", ("DFA stack:  %s", tmp) );
      FREE(tmp);
   }
#endif

   if( dfas == NIL(DFALINK) ) {
      DB_PRINT( "mem", ("%s:<- mem %ld",cp->CE_NAME, (long)coreleft()));
      DB_PRINT( "inf", ("<<< Exit, no dfas, cp = %04x", NIL(CELL)) );
      DB_RETURN( NIL(CELL) );
   }


   /* We have a set of %-meta's that have not previously been considered, or
    * whose counts do not violate the Prep count.  So we should consider
    * them, and put them on the top of the stack.
    */
   top_dfa_stack.df_set  = dfas;
   top_dfa_stack.df_next = dfa_stack;


   /* Run through the %-meta cells, build the prerequisite cells.  If we are
    * performing transitive closure, then call Infer_recipe with the new
    * prerequisite.
    */
   for( pdfa = dfas, infcell = NIL(CELL), pedge = NIL(EDGE);
        pdfa != NIL(DFALINK);
        pdfa = pdfa->dl_next ) {
      int     push   = 0;
      int     _trans = 0;
      EDGEPTR edge;
      EDGEPTR edge_noprq;	/* No prerequisite for this edge */

      DB_PRINT( "inf", ("Using dfa:  [%s]", pdfa->dl_meta->CE_NAME) );
      meta = pdfa->dl_meta;


      /* Change to the required directory prior to checking the prerequisites
       * Only if the meta has a directory and we haven't changed dir's
       * for the CELL already.
       *
       * Ignore the %-meta if we had to change to a dir, and the dir
       * did not exist.
       */
      if( !(cp->ce_attr & A_SETDIR) && meta->ce_dir != NIL(char) )
	 if( (setdirroot == NIL(CELL) || setdirroot->ce_dir != meta->ce_dir) &&
	     (push = Push_dir(meta, TRUE)) )
	    setdirroot = cp;
	 else {
	    DB_PRINT( "inf", ("Failed PUSH of [%s]", meta->ce_dir) );
	    continue;
	 }


      /* Now run through the list of prerequisite edge's for the %-meta.
       * Build each prerequisite in turn, and then see if it needs to be
       * inferred.  We treat the edge that has NO prerequisites last, as
       * it gives a recipe for making a %-meta from no prerequisites, and
       * we should really check all NON-NULL prerequisites first.
       */
      edge = meta->CE_EDGES;
      edge_noprq = NIL(EDGE);

      do {
         pedge = edge;

	 if( edge->ed_prq == NIL(CELL) )
	    edge_noprq = edge;
	 else {
	    HASHPTR  thp;	/* temporary hash table pointer		*/
	    HASH     iprqh;	/* hash cell for new prerequisite	*/
	    CELL     iprq;	/* inferred prerequisite to look for	*/
	    CELLPTR  sdir;	/* local setdir root for inference	*/
	    STRINGPTR sp   = NIL(STRING); /* pointer to rcp of inferred target*/
	    int      ipush = 0;		  /* flag for push on inferred prereq */
	    char     *name = edge->ed_prq->CE_NAME;
	    int	      _dmax_fix;

	    DB_PRINT( "inf", ("Trying edge from [%s] to [%s] for [%s]",
	    	      meta->CE_NAME, name, cp->CE_NAME) );

	    _trans = Transitive &&
		     !((Glob_attr | edge->ed_prq->ce_attr) & A_NOINFER);

	    /* Set the temp CELL used for building prerequisite candidates to
	     * all zero so that we don't have to keep initializing all the
	     * fields. */
	    {
	       register char *s = (char *) &iprq;
	       register int   n = sizeof(CELL);
	       while( n ) { *s++ = '\0'; n--; }
	    }


	    /* Build the prerequisite name from the %-meta prerequisite given
	     * for the %-meta rule. */
	    iprqh.ht_name = _build_name( cp->CE_NAME, name, pdfa->dl_per );
	    if( strcmp(cp->CE_NAME, iprqh.ht_name) == 0 )
	       goto try_next_edge;
	    if((_dmax_fix = (_count_dots(name)-_count_dots(meta->CE_NAME))) < 0)
	       _dmax_fix = 0;
	    if( _count_dots(iprqh.ht_name) > _dmax + _dmax_fix )
	       goto try_next_edge;

	    DB_PRINT( "inf", ("Checking prerequisite [%s]", iprqh.ht_name) );

	    if( Verbose )
	       printf( "%s:  Trying prerequisite [%s] for [%s]\n", Pname,
		       iprqh.ht_name, cp->CE_NAME );


	    /* See if the prerequisite CELL has been previously defined.  If
	     * it has, then make a copy of it into iprq, and use it to try
	     * the inference.  We make the copy so that we don't modify the
	     * stat of the inferred cell if the inference fails.
	     */
	    thp = Get_name( iprqh.ht_name, Defs, FALSE, setdirroot );
	    if(thp != NIL(HASH)) {
	       iprq = *thp->CP_OWNR;
	       if( iprq.CE_HOW != NIL(HOW) ) sp = iprq.CE_HOW->hw_recipe;
	    }
	    else
	       iprq.ce_name  = &iprqh;


	    /* If the prerequisite has the .SETDIR attr set, then we
	     * should try to CD to the directory, if it is really different
	     * from the one we are currently in.  If the CD fails we ignore
	     * the edge and go try another edge.
	     */
	    sdir = setdirroot;
	    if( iprq.ce_dir != NIL(char) )
	       if( sdir==NIL(CELL) || sdir->ce_dir != iprq.ce_dir )
	          if( ipush = Push_dir(&iprq, TRUE) )
		     sdir = &iprq;
		  else
		     goto try_next_edge;


	    /* Stat the inferred prerequisite.
	     */
	    if( !(iprq.ce_flag & F_STAT) ) Stat_target( &iprq, FALSE );


	    /* If the STAT succeeded or if the prerequisite has a recipe for
	     * making it, then infer it.  We may later try to perform a
	     * second inference on this prerequisite when we actually go to
	     * make it.
	     */
	    if( (iprq.ce_time != (time_t)0L) || (sp != NIL(STRING)) ) {
	       infcell = Def_cell( iprqh.ht_name, setdirroot );
	       infcell->ce_flag |= F_REMOVE;
	    }
	    else
	       /* The STAT did not succeed, so call Infer_recipe recursively
	        * to see if we know how to make the prerequisite.  If it
		* returns not NULL, then we have an inferred prerequisite
		* and we should Define it and attach it to the CELL pointed at
		* by cp.  This recursive inference is performed only if
		* Transitive closure is enabled.
		*/
	       if( _trans ) {
		  int _save = _dmax;
		  if( !_dmax ) _dmax += _dmax_fix;
		  infcell = Infer_recipe( &iprq, iprq.CE_HOW, &top_dfa_stack,
					  sdir);
		  _dmax = _save;

		  if( infcell != NIL(CELL) ) {
		     /* We found we can make the prerequisite, so make it into
		      * a real node.  This means, mark it for possible
		      * removal, and when you make it into a node make sure
		      * you don't clobber the Def_cell name.
		      */
		     infcell  = Def_cell( iprqh.ht_name, setdirroot );
		     thp      = infcell->ce_name;
		     *infcell = iprq;
		     infcell->ce_name = thp;
		     infcell->ce_flag |= F_REMOVE;
	          }
	       }

	    /* If we pushed a directory for the inferred prerequisite then
	     * pop it.
	     */
	    if( ipush ) Pop_dir(FALSE);

try_next_edge:
	    FREE( iprqh.ht_name );
	 }

	 edge = edge->ed_next;
      }
      while( infcell == NIL(CELL) && edge != meta->CE_EDGES );


      /* If none of the previous edges were any good, and there was an
       * edge with no prerequsite, then use it.
       */
      if( infcell == NIL(CELL) ) 
         if( edge_noprq != NIL(EDGE) ) 
	    pedge = edge_noprq;
	 else
	    pedge = NIL(EDGE);
      else {
	 /* We have a match, so make the edge's pointer for the corresponding
	  * %-meta target point at the matched prerequisite so that the next
	  * time we perform an inference we will check this edge first.
	  */
         meta->CE_EDGES = pedge;
	 if( !_trans ) infcell->ce_attr |= A_NOINFER;
      }


      /* If we pushed a dir when we treated this %-meta, then
       * pop it.
       */
      if( push ) Pop_dir(FALSE);


      /* Need this so that pdfa does not get advanced, and remains pointing
       * to the dfa, that just resulted in a successful match.
       */
      if( pedge != NIL(EDGE) ) break;
   }


   /* If pedge is not NIL then we have found an edge and possibly inferred
    * a prerequisite.  In any case we should definitely attach the recipe to
    * the HOW node of the cell pointed at by cp.  If the CELL has no HOW node
    * the we allocate one.
    */
   if( pedge != NIL(EDGE) ) {
      LINKPTR lp;
      HOWPTR  nhow, ihow;

      DB_PRINT("inf", ("taking edge [%s] to [%s]", pedge->ed_tg->CE_NAME,
	       (pedge->ed_prq == NIL(CELL)) ? "(none)":pedge->ed_prq->CE_NAME));

      if( Verbose )
	 printf("%s:  Inferred recipe using edge from [%s] to [%s]\n",
		Pname, pedge->ed_tg->CE_NAME,
		(pedge->ed_prq == NIL(CELL)) ? "(none)":pedge->ed_prq->CE_NAME);

      if( how == NIL(HOW) ) {
         /* Get a new HOW node, this should happen only for inferred
	  * cells, as such they have no prior HOW node */
         TALLOC( how, 1, HOW );
	 cp->CE_HOW = how;
      }
      

      /* Attach the recipe to the HOW node.  Note that if the %-meta recipe
       * is a :: recipe then we will attach all of the HOW cells belonging to
       * the %-meta :: rule that we matched to the CELL, and we will place
       * them so that they preceed subsequent HOW cells in the list.  Under
       * these circumstances the CELL is marked as a MULTI cell.
       */
      nhow = how->hw_next;
      for( ihow=meta->CE_EDGES->ed_how; ihow != NIL(HOW); ihow=ihow->hw_next ) {
	 how->hw_per     = pdfa->dl_per;
	 how->hw_flag   |= (ihow->hw_flag & (F_SINGLE | F_GROUP)) | F_INFER;
	 how->hw_attr   |= (ihow->hw_attr & A_TRANSFER);
	 how->hw_recipe  = ihow->hw_recipe;

	 /* Add global prerequisites to the first HOW cell
	  */
	 for( lp=ihow->hw_indprq; lp != NIL(LINK); lp=lp->cl_next ) {
	    char    *name = lp->cl_prq->CE_NAME;
	    CELLPTR tcp;

	    name = _build_name( cp->CE_NAME, name, pdfa->dl_per );
	    tcp  = Def_cell( name, setdirroot );
	    tcp->ce_flag |= F_REMOVE;
	    Add_prerequisite( how, tcp, FALSE );

	    if( Verbose )
	       printf( "%s:  Inferred indirect prerequisite [%s]\n",
	               Pname, name );
	    FREE(name);
	 }

	 /* If infcell is not NIL then we have inferred a prerequisite, so
	  * add it to the first HOW cell as well.
	  */
	 if( infcell != NIL(CELL) ) {
	    (Add_prerequisite( how, infcell, FALSE))->cl_flag |= F_TARGET;

	    if( Verbose )
	       printf( "%s:  Inferred prerequisite [%s]\n",
	               Pname, infcell->CE_NAME );
	 }

	 /* If the recipe is a :: recipe then Insert a new HOW node after
	  * the inferred recipe HOW node and prior to any previous
	  * :: nodes.
	  */
	 if( ihow->hw_next != NIL(HOW) ) {
	    cp->ce_flag |= F_MULTI;
	    TALLOC( how->hw_next, 1, HOW );
	    how = how->hw_next;
	 }
      }
      how->hw_next = nhow;
      pdfa->dl_per = NIL(char);		/* We used it, so don't FREE it */

      /* Make sure to set the FLAGS, and ATTRIBUTES of the CELL so that it
       * gets made correctly.
       */
      cp->ce_flag |= F_RULES | F_TARGET | F_INFER;

      if( !(cp->ce_attr & A_SETDIR) ) {
         cp->ce_attr |= (meta->ce_attr & A_SETDIR);
	 cp->ce_dir   = meta->ce_dir;
      }
   }
   else
      cp = NIL(CELL);

   _free_dfas( dfas );

   DB_PRINT( "mem", ("%s:-< mem %ld", (cp!=NIL(CELL)) ? cp->CE_NAME : "(none)",
	     (long)coreleft()));
   DB_PRINT( "inf", ("<<< Exit, cp = %04x", cp) );
   DB_RETURN( cp );
}


static char *
_build_name( tg, meta, per )
char *tg;
char *meta;
char *per;
{
   char    *name;

   name = Apply_edit( meta, "%", per, FALSE, FALSE );
   if( strchr(name, '$') ) {
      HASHPTR m_at;
      char *tmp;

      m_at = Def_macro( "@", tg, M_MULTI );
      tmp = Expand( name );

      if( m_at->ht_value != NIL(char) ) {
	 FREE( m_at->ht_value );
	 m_at->ht_value = NIL(char);
      }

      if( name != meta ) FREE( name );
      name = tmp;
   }
   else if( name == meta )
      name = _strdup( name );

   return(name);
}


static DFALINKPTR
_dfa_subset( pdfa, stack )/*
============================
   This is the valid DFA subset computation.  Whenever a CELL has a Match_dfa
   subset computed this algorithm is run to see if any of the previously
   computed sets on the DFA stack are proper subsets of the new set.  If they
   are, then any elements of the matching subset whose Prep counts exceed
   the allowed maximum given by Prep are removed from the computed DFA set,
   and hence from consideration, thereby cutting off the cycle in the
   inference graph. */
DFALINKPTR	   pdfa;
register DFASETPTR stack;
{
   register DFALINKPTR element;
   DFALINKPTR          nelement;

   DB_ENTER( "_dfa_subset" );

   for(; pdfa != NIL(DFALINK) && stack != NIL(DFASET); stack = stack->df_next) {
      int subset = TRUE;

      for( element=stack->df_set; subset && element != NIL(DFALINK);
           element=element->dl_next ) {
         register DFALINKPTR subel;

	 for( subel = pdfa;
	      subel != NIL(DFALINK) && (subel->dl_meta != element->dl_meta);
	      subel = subel->dl_next );

	 if( subset = (subel != NIL(DFALINK)) ) element->dl_member = subel;
      }

      if( subset )
	 for( element=stack->df_set; element != NIL(DFALINK);
	      element=element->dl_next ) {
	    DFALINKPTR mem = element->dl_member;
	    int        npr = element->dl_prep + 1;

	    if( npr > _prep )
	       mem->dl_delete++;
	    else
	       mem->dl_prep = npr;
	 }
   }

   for( element = pdfa; element != NIL(DFALINK); element = nelement ) {
      nelement = element->dl_next;

      if( element->dl_delete ) {
	 /* A member of the subset has a PREP count equal to PREP, so
	  * it should not be considered further in the inference, hence
	  * we remove it from the doubly linked set list */
	 if( element == pdfa )
	    pdfa = element->dl_next;
	 else
	    element->dl_prev->dl_next = element->dl_next;

	 if( element->dl_next != NIL(DFALINK) )
	    element->dl_next->dl_prev = element->dl_prev;

	 DB_PRINT("inf", ("deleting dfa [%s]", element->dl_meta->CE_NAME));
	 FREE( element->dl_per );
	 FREE( element );
      }
   }

   DB_RETURN( pdfa );
}



static void
_free_dfas( chain )/*
=====================
   Free the list of DFA's constructed by Match_dfa, and linked together by
   LINK cells.  FREE the % value as well, as long as it isn't NIL. */
DFALINKPTR chain;
{
   register DFALINKPTR tl;

   DB_ENTER( "_free_dfas" );

   for( tl=chain; tl != NIL(DFALINK); chain = tl ) {
      tl = tl->dl_next;

      DB_PRINT( "inf", ("Freeing DFA [%s], %% = [%s]", chain->dl_meta->CE_NAME,
                chain->dl_per) );

      if( chain->dl_per != NIL(char) ) FREE( chain->dl_per );
      FREE( chain );
   }

   DB_VOID_RETURN;
}


static int
_count_dots( name )/*
=====================*/
char *name;
{
   register char *p;
   register int  i = 0;

   for( p = name; *p; p++ ) if(*p == '.') i++;

   return( i );
}
