(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: AssertStmt.m3                                         *)
(* Last modified on Mon Mar  2 11:08:30 PST 1992 by kalsow     *)
(*      modified on Sat Mar 16 02:01:05 1991 by muller         *)

MODULE AssertStmt;

IMPORT Expr, Token, Scanner, Stmt, StmtRep, Error, Temp, Type, Bool;
IMPORT Emit, PackedType, Host, EnumExpr, Fault;

TYPE
  P = Stmt.T OBJECT
        e: Expr.T;
      OVERRIDES
        check    := Check;
	compile  := Compile;
        outcomes := GetOutcome;
      END;

PROCEDURE Parse (READONLY fail: Token.Set): Stmt.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    StmtRep.Init (p);
    Scanner.Match (Token.T.tASSERT, fail, Token.ExprStart);
    p.e := Expr.Parse (fail);
    IF (Scanner.cur.token # Token.T.tENDPRAGMA) THEN
      Scanner.Fail ("missing \'*>\'", fail + Token.Set {Token.T.tENDPRAGMA});
    END;
    Scanner.cur.token := Token.T.tSEMI;  (* for the statement parser *)
    RETURN p;
  END Parse;

PROCEDURE Check (p: P;  VAR cs: Stmt.CheckState) =
  BEGIN
    Expr.TypeCheck (p.e, cs);
    IF NOT Type.IsEqual (PackedType.Strip (Expr.TypeOf(p.e)), Bool.T, NIL) THEN
      Error.Msg ("you can only ASSERT boolean expressions");
    END;
  END Check;

PROCEDURE Compile (p: P): Stmt.Outcomes =
  VAR t: Temp.T;  v: Expr.T;  i: INTEGER;  u: Type.T; 
  BEGIN
    IF Host.doAsserts THEN
      i := -1;
      v := Expr.ConstValue (p.e);
      IF (v = NIL) THEN
        t := Expr.Compile (p.e);
        Emit.OpT ("if (!@) ", t);
        Fault.Assert ();
        Temp.Free (t); 
      ELSIF EnumExpr.Split (v, i, u) AND (i = 0) THEN
        (* ASSERT (FALSE) *)
        Fault.Assert ();
        RETURN Stmt.Outcomes {};
      ELSE <* ASSERT i = 1 *>
        (* ASSERT (TRUE) *)
      END;
    END;
    RETURN Stmt.Outcomes {Stmt.Outcome.FallThrough};
  END Compile;

PROCEDURE GetOutcome (<*UNUSED*> p: P): Stmt.Outcomes =
  BEGIN
    RETURN Stmt.Outcomes {Stmt.Outcome.FallThrough};
  END GetOutcome;

BEGIN
END AssertStmt.
