/*
 * in.fingerd.c --
 *	Return information from the standard Finger port.
 *
 * Copyright (C) 1988,1990 Free Software Foundation, Inc.
 * Copyright (C) 1991 International Computer Science Institute, Berkeley, USA.
 *
 * This file is part of GNU Finger.
 * 
 * GNU Finger is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 1, or (at your
 * option) any later version.
 *
 * GNU Finger is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GNU Finger; see the file COPYING.  If not, write to the
 * Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#if !defined(lint) && !defined(SABER)
static char *rcsid = "$Id: in.fingerd.c,v 2.78 1995/01/26 07:21:10 stolcke Exp $ ICSI (Berkeley)";
#endif

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <netdb.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/file.h>
#ifndef X_OK
#include <unistd.h>
#endif
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <pwd.h>
#include <signal.h>
#include <sys/wait.h>
#include <sys/param.h>
#if defined(sun) && defined(HAVE_VFORK)
#include <vfork.h>
#endif

#include "../config.h"

#include "fingerpaths.h"
#include "general.h"
#include "client.h"
#include "call.h"
#include "bitmap.h"
#include "error.h"
#include "os.h"
#include "packet.h"
#include "util.h"
#include "getservhost.h"
#include "flock.h"

#include "../patchlevel.h"
#include "version.h"

#define DEFAULT_TARGET	"default"
#define TEMPFILE	"/tmp/fingerXXXXXX"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

/* the finger flag extends the public interface in call.h is for internal
   use.  It is used by the in.fingerd on a clients machine to tell the
   in.fingerd on the serer that it is on the other side.
 */
#define INFINGERD	0x80
#define INFINGERD_FLAG	"/FiNgEr"

/* **************************************************************** */
/*								    */
/*		    INET Daemon Finger Replacement		    */
/*								    */
/* **************************************************************** */

#define IP 0			/* Internet Protocol */

/* A single line of arguments as read from the input stream. */
char arguments[256];

/* Offset of the next available character in arguments. */
int next_char = 0;

char *serverhost;		/* name of host running server daemon */
char *lserverhost;		/* name of host doing local polling */
int im_the_server = 0;		/* I'm the server host daemon! */
int im_a_lserver = 0;		/* I'm a local polling host! */
int local_connect = 0;		/* Is this a local connection? */
int self_connect = 0;		/* Connection from localhost ? */

char peer_name[MAXHOSTNAMELEN + 1];	/* address of peer host */
char user_name[64];			/* user name on peer host */

char *widtharg = NULL;			/* the widths arguments */

struct passwd *current_pw; /* current passwd structure while scanning
				   over password file */

/* Get args from stdin, and print info to stdout. */
/*ARGSUSED*/
main (argc, argv)
     int argc;
     char **argv;
{
  char *next_argument();
  char *defwidths;

  default_error_handling (argv[0], LOG_SYS);

  /* Change uid to nobody to make sure no private info is inadvertedly
     given out */
  if (getuid() == 0)
    {
#ifdef NOBODY_USER
      struct passwd *nobody = getpwnam(NOBODY_USER);

      if (!nobody)
        handle_error(FATAL, "user %s unknown", NOBODY_USER);

      if (setgid(nobody->pw_gid) < 0)
        file_error(FATAL, "setgid");
      if (setuid(nobody->pw_uid) < 0)
        file_error(FATAL, "setuid");
#endif /* NOBODY_USER */
    }

  (void) open_host_config ();

  local_connect = local_socket(fileno(stdin), peer_name, sizeof(peer_name),
					      user_name, sizeof(user_name));
  self_connect = (stricmp(peer_name, "localhost") == 0) ||
                 (strnicmp(peer_name, "localhost.", 10) == 0);

  /* Gobble line of arguments from user.  If the connection is already
     closed, then just exit. */
  if (!fgets (arguments, sizeof (arguments), stdin))
    exit (0);

  /* Check for buffer overflow */
  if (strlen(arguments) >= 200)
    {
      handle_error(WARNING, "attempted buffer overflow from %s@%s",
                   user_name, peer_name);
      printf("No such luck, buddy.\r\n");
      exit(2);
    }

  if (deny_connection (user_name, peer_name))
    {
      handle_error(WARNING, "connection from %s@%s denied",
                   user_name, peer_name);
      printf("Access denied.\r\n");
      exit(1);
    }
  
#ifdef USE_SYSLOG
  /* we can only do this with syslogging since the message would otherwise
     appear in the finger output ... */
  if (match_host_config ("logconnect", peer_name, NULL))
    {
      char *cp = arguments + strlen(arguments);
    
      /* strip final \r\n */
      while (cp > arguments && (*(cp-1) == '\r' || *(cp-1) == '\n'))
	cp--;
      *cp = '\0';
    
      debug(1, "connection from %s@%s: %s", user_name, peer_name, arguments);
    }
#endif /* USE_SYSLOG */

  /* Are we the finger daemon running on the server host? */
  serverhost = getservhost (GETSERV_GLOBAL);
  if (!serverhost)
    exit (1);

  /* If this is the server host, then handle the request directly.
     Otherwise, open a connection to the server and pass the input
     and output back and forth. */
  im_the_server = same_hostname (get_full_hostname (), serverhost);

  /* Also need name of local polling host later.  */
  lserverhost = getservhost (GETSERV_LOCAL);
  if (!lserverhost)
    exit (1);

  im_a_lserver = same_hostname (get_full_hostname (), lserverhost);

  /* set default print widths */
  (void) open_host_config ();
  if (defwidths = get_config_entry ("printwidths", NULL))
    {
      set_print_widths (defwidths);
      free (defwidths);
    }

  finger_reply (next_argument(), stdout);

  exit(0);
}

/* send a finger request to another server */
static void
client_finger (serverhost, arguments, verbose, stream)
     char *serverhost;
     char *arguments;
     int verbose;
     FILE *stream;
{
  struct hostent *host;
  FILE *server;
  int connection, fd;

  host = gethostbyname (serverhost);

  if (!host)
    if (verbose)		/* this hostname was requested by user */
      {
        fprintf(stream, "%s: unknown host\r\n", serverhost);
	exit(1);
      }
    else
      {
        handle_error(FATAL, "%s: server host lookup failed", serverhost);
      }

  if ( verbose )
    fprintf(stream, "[%s]\r\n", host->h_name);

  connection = tcp_to_service ("finger", (struct in_addr *)host->h_addr);

  if (connection < 0)
    file_error (FATAL, host->h_name);

  fd = dup (connection);

  if ((server = fdopen (fd, "r+")) == (FILE *)NULL)
    file_error (FATAL, host->h_name);

  /* Pass the arguments through, outputting the responses. */
  {
    int i = strlen (arguments);

    if (i && (arguments[i - 1] == '\n'))
      arguments[i - 1] = '\0';
  }

  fprintf (server, "%s\r\n", arguments);
  fflush (server);
  fseek (server, 0l, 1);

  /* We have to pass everything byte by byte because we don't know
     wether this is face information, etc. */
  {
    byte single_byte;

    while (read (fd, &single_byte, 1) == 1)
      fwrite (&single_byte, sizeof(single_byte), 1, stream);
  }

  fflush (stream);
  fclose (server);
}

/* wrap a finger request so it can be forwarded */
static void
forward_finger (otherhost, user, options, verbose, stream)
    char *otherhost, *user;
    int options, verbose;
    FILE *stream;
{
    char *newuser;

    /* modify user name according to options */
    if (options & FACE)
      {
	newuser = (char *)xmalloc (strlen (user) + sizeof(FACE_COLON));
        sprintf (newuser, "%s%s%s", FACE_COLON, user, FACE_COLON);
      }
    else
      {
        newuser = (char *)xmalloc(strlen(user) + 1 +
				    sizeof(INFINGERD_FLAG) +
				    sizeof(WIDE_FLAG) +
				    sizeof(WIDTH_FLAG) +
				    (widtharg ? strlen(widtharg) : 0));
        sprintf (newuser, "%s%s%s%s%s%s%s%s",
               (options&INFINGERD ? INFINGERD_FLAG : ""),
               (options&INFINGERD ? " " : ""),
               (options&INFO ? WIDE_FLAG : ""),
               (options&INFO ? " " : ""),
               (options&WIDTH ? WIDTH_FLAG : ""),
               (options&WIDTH ? widtharg : ""),
               (options&WIDTH ? " " : ""),
               user);
      }

    client_finger(otherhost, newuser, verbose, stream);

    free(newuser);
}



/* **************************************************************** */
/*                                                                  */
/*                 Getting and Parsing Argumements                  */
/*                                                                  */
/* **************************************************************** */

#define ADVANCE 1
char *next_arg_internal ();

/* Return the next argument available.   Return a new string which is
   the argument, or a pointer to NULL if no arguments remain. */
char *
next_argument ()
{
  return (next_arg_internal (ADVANCE));
}

/* Return the next argument available in arguments, but leave us
   still pointing at that argument. */
char *
peek_argument ()
{
  return (next_arg_internal (!ADVANCE));
}

/* Return the next argument available.   Return a new string which is
   the argument, or a pointer to NULL if no arguments remain.  If ADVANCE
   is non-zero, then advance the argument pointer to the following argument
   so that subsequent calls can get subsequent arguments. */
char *
next_arg_internal (advance)
     int advance;
{
  register int i;
  int start, end, len;
  char *arg;

  if (next_char >= strlen (arguments))
    return ((char *)NULL);

  i = next_char;

  /* Isolate argument. */
  while (arguments[i] && isspace (arguments[i]))
    i++;

  start = i;

  while (arguments[i] && !isspace (arguments[i]))
    i++;

  end = i;

  if (start == end)
    return ((char *)NULL);

  arg = (char *)xmalloc (len = end - start + 1);
  strncpy (arg, &arguments[start], len);
  arg[len - 1] = '\0';

  if (advance)
    next_char = end;

  return (arg);
}


/* **************************************************************** */
/*								    */
/*			Long Finger Information		            */
/*								    */
/* **************************************************************** */
/* Produce long finger information for USER.  Print the textual output
   to STREAM. Returns number of matching users with local accounts. */
int
finger_long (user, stream)
     char *user;
     FILE *stream;
{
  FINGER_PACKET **packets = get_finger_data (NULL, 1);
  struct passwd *entry = (current_pw && strcmp(user, current_pw->pw_name)==0) ?
				current_pw : getpwnam (user);
  int entries_printed = 0;

  if (!entry)
    {
      setpwent ();

      while (current_pw = getpwent ()) 
	if (strindex (pw_real_name (current_pw), user) != (char *)NULL)
	  {
	    if (entries_printed == 0)
	      fprintf (stream,
		       "Users who have `%s' in their real names:\r\n", user);

	    if (!maybe_user_script (current_pw, stream, packets))
	      display_finger_info (current_pw, stream, packets);
	    entries_printed++;
	  }
      endpwent ();

      /* Print alias expansion, if any. */
      if (mail_list (user, stream, peer_name))
	entries_printed++;

      if (entries_printed == 0 &&
	  (maybe_special_target (DEFAULT_TARGET, stream, 'l', user) ||
	   maybe_special_target (DEFAULT_TARGET, stream, 'x', user)))
	entries_printed++;

      if (entries_printed == 0)
	fprintf (stream, "Nobody matching %s known on %s.\r\n",
			 user, get_full_hostname ());
    }
  else
    {
      if (!maybe_user_script (entry, stream, packets))
        display_finger_info (entry, stream, packets);
      entries_printed++;
    }
  free_array (packets);
  fflush (stream);
  return entries_printed;
}


/* **************************************************************** */
/*								    */
/*			How the main server replies.		    */
/*								    */
/* **************************************************************** */

/* A minute is 60 seconds. */
#define MINUTES(x) (x * 60)

/* How old the hostdata and clientstatus files can get before we suspect
   something is wrong. */
#ifndef MIN_UPDATE_INTERVAL
#define MIN_UPDATE_INTERVAL (2 * DEFAULT_POLL_INTERVAL)
#endif

/* How long the console of a machine has to be idle before we think it
   is free. */
#ifndef FREE_IDLE_TIME
#define FREE_IDLE_TIME MINUTES(15)
#endif

/* How long the console of a machine has to be idle in order to look like
   it might be free in the future.  This is used when answering the .clients
   message. */
#ifndef IDLE_TIME
#define IDLE_TIME MINUTES(3)
#endif

/* How many machines should be free for any given site?  Although a
   percentage of the available machines would be reasonable, I think
   that there should be a lower limit. */
#ifndef ENOUGH_FREE_MACHINES
#define ENOUGH_FREE_MACHINES 0
#endif

/* Generate output to STREAM for USER.  USER can be one of the following:

	username	Return the collected finger packets for USERNAME.

	/W		Return the long (database) information about the
			following USERNAME.

	.free		Return a list of free machines. (Those
			idle for a "long" time.)

	.all		Return the information about every machine that
			we know about, in a format that is useful.

	.clients	Returns a list of the clients that this server
			serves.

	:user:		If the username is surrounded with colons, then
			this is a request for the user's face.  If we
			can find the data, then return that.

        .faces		Return a list of the available faces.

	.local		Finger only at this machine.

	.debug		print debugging info (not advertised)

	.version	similar to .debug (GNU compatible)

	user@host	finger to be forwarded

	/WIDTHS		Change the width of the output according to the 
			following string. This is a comma seperated list
			of integers, giving the width of the fields for
			longname, login-host, remote-host. The widths may
			be preceeded by '+' or '-', then the width will
			be incremented, or decremented.

   The initial period can be replaced by a `-' for compatibility with
   UNIX option syntax.
*/
finger_reply (user, stream)
     char *user;
     FILE *stream;
{
  int options = 0;

  /* connection from peer in.fingerd ? */
  if (user && strcmp (user, INFINGERD_FLAG) == 0)
    {
      options |= INFINGERD;

      /* Skip the magic cookie  */
      user = next_argument ();
    }

  /* Is this a `wide' finger? */
  if (user && stricmp (user, WIDE_FLAG) == 0)
    {
      options |= INFO;

      /* Skip the /W and print out the finger info. */
      user = next_argument ();
    }

  /* Have other widths */
  if (user && strnicmp(user, WIDTH_FLAG, sizeof(WIDTH_FLAG) - 1) == 0)
    {
      /* only allow users to reset the format if they're local */
      if (local_connect)
	{
          widtharg = user + sizeof(WIDTH_FLAG) - 1;
          options |= WIDTH;
          set_print_widths(widtharg);
	}
      user = next_argument ();
    }
  else if (widtharg)
    options |= WIDTH;	/* use widths from an earlier invocation */

  /* Figure out what to do as default action if no target is specified */
  if (!user || !*user)
    {
      char *defuser;

      (void)open_host_config();
      if ((defuser = get_config_entry(self_connect ? "defaultself" :
	                              (local_connect ? "defaultlocal"
                                                     : "defaultremote"), NULL))
          && *defuser)
	user = defuser;
      else
	user = ".all";
    }

  /* Is this a face finger ? */
  if (strncmp(user, FACE_COLON, sizeof(FACE_COLON)-1) == 0)
    {
      char *t = user + 1;

      while (*t && strncmp(t, FACE_COLON, sizeof(FACE_COLON)-1) != 0) t++;
      if (*t)
	{
	  options |= FACE;
	  user++;
	  *t = '\0';
        }
    }

  /* A finger to be forwarded ? */
  {
    char *otherhost = strrchr(user, '@');

    if (otherhost)
      {
        if ((local_connect &&
 	     match_host_config ("allowindirect", "localnet", NULL)) ||
	    match_host_config ("allowindirect", peer_name, NULL))
	  {
	    /* isolate host name to call */
	    *otherhost++ = '\0';
	    if ( !*otherhost )
		otherhost = LOCALHOST;

	    forward_finger(otherhost, user, options, !(options&FACE), stream);
	  }
	else
	  {
	    fprintf (stream, "Sorry, indirect fingering is not allowed.\r\n");
	    fflush (stream);
	  }
	return;
      }
  }

  /* If the requesting machine is expecting a face, then send it along
     if we have it. */
  if (options & FACE)
    {
      /* face retrieval is handled by global server */
      if (!im_the_server)
	{
	  forward_finger(serverhost, user, (options|INFINGERD), 0, stream);
	  return;
	}

      send_face (user, stream);
      return;
    }

  /*
   * special requests
   */
  /* option compatibility: inital `-' is same as `.' */
  if (*user == '-')
	*user = '.';

  /* Check for special, externally implemented targets. */
  if (maybe_special_target (user, stream, (options&INFO) ? 'l' : 's', NULL) ||
      maybe_special_target (user, stream, 'x', NULL))
    return;

  if (stricmp (user, ".debug") == 0 ||
      stricmp (user, ".version") == 0)
    {
      fprintf(stream, "ICSI Finger version %d.%d.%d #%d\r\n\
Built by %s on %s\r\n",
		MAJORVERSION, MINORVERSION, PATCHLEVEL,
		BUILDNO, BUILDER, BUILDDATE);

      if (stricmp (user, ".debug") == 0)
	{
	  fprintf(stream, "%s: peer = %s@%s (%slocal, %sself)\r\n",
			    (im_the_server ? "global server" :
			     (im_a_lserver ? "local server" : "client")),
			    user_name, peer_name,
	                    local_connect ? "": "not ",
	                    self_connect ? "" : "not ");

	  if (im_a_lserver || im_the_server)
	    {
	      show_last_modified ("Client status", HOSTSTAT, stream);
	      show_last_modified ("Host data", HOSTDATA, stream);
	      show_last_modified ("User data", USERDATA, stream);
	    }
	}
      fflush(stream);

      if (!im_a_lserver)
        forward_finger(lserverhost, user, (options|INFINGERD), 1, stream);
      else if (!im_the_server)
        forward_finger(serverhost, user, (options|INFINGERD), 1, stream);

      return;
    }

  /* Is this a local finger?  If so, just call the local client finger
     daemon to get packets, and print them out. */
  if (stricmp (user, ".local") == 0)
    {
      finger_local (user, options, stream);
      return;
    }

  /* ".users" returns a listing of everyone in the userdata file. */
  if (stricmp (user, ".users") == 0)
    {
      finger_users (user, options, stream);
      return;
    }

  /* ".clients" returns a listing of everyone in the hoststatus file. */
  if (stricmp (user, ".clients") == 0)
    {
      finger_clients (user, options, stream);
      return;
    }

  /* .faces returns a list of all of the available faces. */
  if (stricmp (user, ".faces") == 0)
    {
      finger_faces (user, options, stream);
      return;
    }

  /* .free lists unused machines. */
  if (stricmp (user, ".free") == 0)
    {
      finger_free (user, options, stream);
      return;
    }

  /* At this point, we have a regular person (or wildcard) to finger.
     `Long' fingers are handled locally, everything else is handled on
     the server.
     Long finger info returned if
     - user requested it explicitly (-l, -info)
     - the "longoutput" hostconfig attribute tells us so
     The "longoutput" attribute is not consulted if this is just a
     forwarded request from a peer finger daemon.
   */
  if (stricmp (user, ".all") != 0 &&
      ((options & INFO) ||
       !(options & INFINGERD) &&
       match_host_config ("longoutput", peer_name, NULL)))
    {
      if (finger_long (user, stream) > 0)
        return;

      /* No long info available for user, probably because it's not a local
	 account. Fall back on short finger info from server. */
      options &= ~INFO;
    }

  finger_default (user, options, stream);
}

/* finger local login sessions */
finger_local (user, options, stream)
    char *user;
    int options;
    FILE *stream;
{
  int packets_output = 0;
  int i;

  FINGER_PACKET **lpackets = get_finger_data (NULL, 1);

  if (! lpackets)
    handle_error (FATAL, "couldn't get local finger data");

  sort_packets (lpackets);

  for (i = 0; lpackets[i]; i++)
    {
      /* skip dummy packets */
      if (lpackets[i]->name[0] == '\0')
	continue;

      ascii_packet (lpackets[i], stream, packets_output == 0);
      packets_output++;
    }

  if ( packets_output == 0 )
    fprintf(stream, "No one logged on.\r\n");

  free_array (lpackets);
}

/* finger known users */
finger_users (user, options, stream)
    char *user;
    int options;
    FILE *stream;
{
  static FINGER_PACKET **upackets = NULL;

  /* user listings are handled by global servers only */
  if (!im_the_server)
    {
      forward_finger(serverhost, user, (options|INFINGERD), 0, stream);
      return;
    }

  if (! upackets)
    {
      int f;

      warn_if_not_recent (USERDATA);

      if ((f = open (USERDATA, O_RDONLY)) < 0 ||
	  flock (f, LOCK_SH) < 0 ||
	  ! (upackets = read_packets (f, 0)))
	file_error (FATAL, USERDATA);

      sort_packets (upackets);
    }

  if (*upackets)
    {
      int i;

      for (i = 0; upackets[i]; i++)
	if (*(upackets[i]->real_name))
	  fprintf (stream, "%s (%s) seen at %s on %s",
		   upackets[i]->real_name, upackets[i]->name,
		   strip_hostname (upackets[i]->host),
		   ctime (&upackets[i]->idle_time));
	else
	  fprintf (stream, "%s seen at %s on %s",
		   upackets[i]->name,
		   strip_hostname (upackets[i]->host),
		   ctime (&upackets[i]->idle_time));
    }
  else
    {
      handle_error (WARNING, "%s: file is empty", USERDATA);
      fprintf (stream, "No users.\r\n");
    }

  fflush (stream);
}

/* finger known client hosts */
finger_clients (user, options, stream)
    char *user;
    int options;
    FILE *stream;
{
  CLIENT **clients, **get_clients();
  int i, clients_count;
  int plural;
  time_t now = time((time_t *)0);

  /* client listings are handled by local servers only */
  if (!im_a_lserver)
    {
      forward_finger(lserverhost, user, (options|INFINGERD), 0, stream);
      return;
    }

  clients = get_clients ();
  clients_count = array_len (clients);
  plural = (clients_count != 1);

  fprintf (stream, "There %s %d client%s polled by %s.",
	   plural? "are" : "is", clients_count, plural? "s" : "",
	   lserverhost);
  if (!same_hostname(lserverhost, serverhost))
      fprintf (stream, " Finger server is %s.", serverhost);
  fprintf (stream, "\r\n");

  fprintf (stream, "%-24s", "Hostname");
  fprintf (stream, "%-16s", "Address");
  fprintf (stream, "%-8s", "Status");
  fprintf (stream, "Users");
  fprintf (stream, " Idle/Down Time\r\n");
  fflush (stream);

  for (i = 0; clients[i]; i++)
    {
      char *status;
      char *idle_time = NULL;
      struct in_addr address;

      if (clients[i]->status & CLIENT_UP)
	{
	  if (clients[i]->status & CLIENT_GATEWAY)
	      status = "gateway";
	  else if (clients[i]->users == 0)
	      status = "free";
	  else
	    {
	      if (clients[i]->idle_time < IDLE_TIME)
		status = "up";
	      else
		{
		  status = "idle";
		  idle_time = idle_time_string (clients[i]->idle_time, 0);
		}
	    }
	}
      else if (clients[i]->idle_time != 0)
	{
	  status = "down";
	  idle_time = idle_time_string (now - clients[i]->idle_time, 0);
	}
      else
	{
	  status = "";
	  idle_time = "";
	}
    

      fprintf (stream, "%-24.24s", strip_hostname (clients[i]->hostname));
      address = clients[i]->address;
      fprintf (stream, "%-16s", (clients[i]->status & CLIENT_LOCAL) ?
		       "(local)" : inet_ntoa (address));
      fprintf (stream, "%-8s", status);
      if ((clients[i]->status & CLIENT_UP) &&
	  clients[i]->users > 0)
	  fprintf (stream, "%5d", clients[i]->users);
      else
	  fprintf (stream, "%5s", " ");

      if (idle_time)
	{
	  fprintf (stream, " %s", idle_time);
	  free (idle_time);
	}
      fprintf (stream, "\r\n");

    }

  free_array (clients);
  fflush (stream);
}

/* finger face listing */
finger_faces (user, options, stream)
     char *user;
     int  options;
     FILE *stream;
{
  int faces_listed;
  int plural;

  /* faces listings are handled by global servers only */
  if (!im_the_server)
    {
      forward_finger(serverhost, user, (options|INFINGERD), 0, stream);
      return;
    }

  faces_listed = site_list_faces (stream) ;
  plural = (faces_listed != 1);

  fprintf (stream, "%d face%s listed.\r\n",
		   faces_listed, plural ? "s" : "");
}

/* If the user wants to know about free machines then look the information
   up in the database kept on the availability of hosts.  Print statistics
   that seem strange, like if the file of host information hasn't been
   written in a while. */
finger_free (user, options, stream)
     char *user;
     int  options;
     FILE *stream;
{
  int not_responding, free_hosts_index;
  int i, packets_output = 0;
  char **free_hosts;
  CLIENT **clients, **get_clients();

  /* free host listings are handled by local servers only */
  if (!im_a_lserver)
    {
      forward_finger(lserverhost, user, (options|INFINGERD), 0, stream);
      return;
    }

  clients = get_clients ();

  free_hosts = (char **)xmalloc (array_len (clients) * sizeof (char *));
  free_hosts_index = 0;
  not_responding = 0;

  for (i = 0; clients[i]; i++)
    {
      if (! (clients[i]->status & CLIENT_UP))
	{
	  not_responding++;
	}
      else
	{
	  /* A host is free if it is up with no users, or if it has
	     been idle for at least as long as FREE_IDLE_TIME. */
	  if (clients[i]->users == 0 ||
	      clients[i]->idle_time >= FREE_IDLE_TIME)
	    {
	      char *description =
		    savestring (strip_hostname (clients[i]->hostname));

	      /* If the reason this machine is free is because it
		 has been idle for a long time, then show how long
		 it has been idle in the description. */
	      if (clients[i]->users)
		{
		  char *itime = idle_time_string (clients[i]->idle_time, 0);
		  char *t;

		  t = (char *)xmalloc (1 + strlen (" (idle )") +
				       strlen (itime) +
				       strlen (description));

		  sprintf (t, "%s (idle %s)", description, itime);
		  free (itime);
		  free (description);
		  description = t;
		}
	      free_hosts[free_hosts_index++] = description;
	      packets_output++;
	    }
	}
    }
  free_hosts[free_hosts_index] = (char *)NULL;
  free_array (clients);

  {
    /* If there aren't enough free machines, something is wrong, even
       if it only means that this site doesn't have enough machines to
       go around. */

    int percent_not_replied = 0, col = 0;
    int plural = packets_output != 1;

    /* Print a summary report. */
    fprintf (stream, "%d host%s free", packets_output, plural ? "s" : "");

    if (not_responding)
      fprintf (stream, ", %d not responding.\r\n", not_responding);
    else
      fprintf (stream, ".\r\n");

    /* Print the list of free hosts. */

    if (packets_output)
      {
	fprintf (stream, "Free hosts are ");
	col = strlen ("Free hosts are ");
      }

    for (i = 0; free_hosts[i]; i++)
      {
	if ((col + 2 + strlen (free_hosts[i])) > 79)
	  {
	    fprintf (stream, "\r\n");
	    col = 0;
	  }

	fprintf (stream, "%s%s", free_hosts[i],
		 free_hosts[i + 1] ? ", " : ".\r\n");

	if (i + 2 == packets_output)
	  {
	    if (col + strlen ("and ") > 79)
	      {
		fprintf (stream, "\r\n");
		col = 0;
	      }
	    fprintf (stream, "and ");
	    col += strlen ("and ");
	  }

	col += 2 + strlen (free_hosts[i]);
      }

    free_array (free_hosts);

    if (not_responding)
      {
	double x, x1, x2;
	x1 = free_hosts_index; x2 = not_responding;
	x = x1 / x2;
	x = 100 / x;
	percent_not_replied = x;
      }

    fprintf (stream, "\r\n");

#if ENOUGH_FREE_MACHINES > 0
    /* What's wrong?  Not enough machines? */
    if (percent_not_replied < 10)
      {
	if (packets_output < ENOUGH_FREE_MACHINES)
	  fprintf (stream, "\
There %s %d machine%s free, even though %d%% of the hosts are responding.\r\n\
Maybe you should ask whoever is in charge to purchase more machines,\r\n\
since there obviously aren't enough to go around.\r\n\
", plural? "are" : "is", packets_output,
		     plural? "s" : "", 100 - percent_not_replied);
	}
      else fprintf (stream, "\
There %s %d known free machine%s, but %d%% of all the hosts did not\r\n\
respond.  It is likely that for some reason those hosts are not\r\n\
running the appropriate daemon.  You might want to complain to the\r\n\
system administrator about this, since this program can only report\r\n\
the status of machines that it can talk to.\r\n\
", plural? "are" : "is", packets_output,
		    plural? "s" : "", percent_not_replied);
#endif /* ENOUGH_FREE_MACHINES > 0 */

  }
}

finger_default (user, options, stream)
     char *user;
     int  options;
     FILE *stream;
{
  /* This function may be called several times
     by the same finger reply. Make sure we keep
     the finger data around. */
  static FINGER_PACKET **hpackets = NULL, **upackets = NULL;
  int i, packets_output = 0;
  int match_all;

  if (!im_the_server)
    {
      forward_finger(serverhost, user, (options|INFINGERD), 0, stream);
      return;
    }

  match_all = (stricmp(user, ".all") == 0);

  /*
   * get host packets (in case we don't have them already)
   */
  if (! hpackets)
    {
      int f;

      warn_if_not_recent (HOSTDATA);

      if ((f = open (HOSTDATA, O_RDONLY)) < 0 ||
	  flock (f, LOCK_SH) < 0 ||
	  ! (hpackets = read_packets (f, 0)))
	{
	  file_error (FATAL, HOSTDATA);
	}

      sort_packets (hpackets);

      /* For long listings, avoid duplication of users */
      if (options & INFO)
	  uniq_packets(hpackets);
    }

  for (i = 0; hpackets[i]; i++)
    {
      if (match_all || (stricmp (user, hpackets[i]->name) == 0))
	{
          /* skip dummy packets */
	  if (hpackets[i]->name[0] == '\0')
	    continue;

	  if (options & INFO)
	    finger_long (hpackets[i]->name, stream);
	  else
	    ascii_packet (hpackets[i], stream, packets_output == 0);
	  packets_output++;
	}
    }

  /* If no packets were output, and the caller wanted to match everybody,
     sorry, there is no one logged in anywhere that we know of.  But if the
     caller wanted a specific user, check the old user file for old logins.
     If it isn't found there, then try a match on the real_name field. */

  if (packets_output == 0)
    {
      if (match_all)
	{
	  fprintf (stream, "No one logged in on client machines.\r\n");
	  goto exit_function;
	}

      /*
       * get user packets (in case we don't have them already)
       */
      if (! upackets)
        {
	  int f;

	  if ((f = open (USERDATA, O_RDONLY)) < 0 ||
	      flock (f, LOCK_SH) < 0 ||
	      ! (upackets = read_packets (f, 0)))
	    {
	      file_error (FATAL, USERDATA);
	    }

	  sort_packets (upackets);

	  /* For long listings, avoid duplication of users */
	  if (options & INFO)
	      uniq_packets(upackets);
        }

      if (!match_all)
	{
	  for (i = 0; upackets[i]; i++)
	    {
	      if (stricmp (upackets[i]->name, user) == 0)
		{
		  if (options & INFO)
		    finger_long (upackets[i]->name, stream);
		  else
		    show_unlogged_packet (upackets[i], stream);
		  packets_output++;
		  break;
		}
	    }
	}

      /* If we haven't found any packets, then try real names.
	 Start with people who are currently logged in.
	 Then try the people who are not. */
      if (packets_output == 0)
	{
	  for (i = 0; hpackets[i]; i++)
	    if (hpackets[i]->name[0] /* no dummy packets */ &&
	        strindex (hpackets[i]->real_name, user))
	      {
		if (options & INFO)
		  finger_long (hpackets[i]->name, stream);
		else
		  ascii_packet (hpackets[i], stream, packets_output == 0);
		packets_output++;
	      }

	  if (packets_output == 0)
	    {
	      for (i = 0; upackets[i]; i++)
		{
		  if (strindex (upackets[i]->real_name, user))
		    {
		      if (options & INFO)
		        finger_long (upackets[i]->name, stream);
		      else
		        show_unlogged_packet (upackets[i], stream);
		      packets_output++;
		    }
		}
	    }
	}
    }
  if (packets_output == 0 &&
      !(maybe_special_target (DEFAULT_TARGET, stream, 's', user) ||
	maybe_special_target (DEFAULT_TARGET, stream, 'x', user)))
    {
      fprintf(stream, "Nobody matching %s ever logged in.\r\n", user);
      fprintf(stream, "Try `finger .help'.\r\n");
    }

exit_function:
  fflush(stream);
}

/* What to show in the case that a user is not logged in on any clients.
   Print information about the user owning PACKET to STREAM. */
show_unlogged_packet (packet, stream)
     FINGER_PACKET *packet;
     FILE *stream;
{
  char *the_time = (char *)ctime (&packet->idle_time);
  struct passwd *pw = (current_pw &&
		       strcmp(packet->name, current_pw->pw_name)==0) ?
				current_pw : getpwnam (packet->name);
  char *message = (pw ? "is not presently logged in" :
			"no longer has an account here");

  the_time[strlen(the_time) - 1] = '\0';	/* delete newline */

  if (*(packet->real_name))
    fprintf (stream, "%s (%s) %s.\r\n",
	     packet->real_name, packet->name, message);
  else
    fprintf (stream, "%s %s.\r\n", packet->name, message);

  fprintf (stream, "Last seen at %s on %s",
           strip_hostname (packet->host), the_time);
  if (*packet->ttyloc)
    fprintf (stream, " from %s", *packet->ttyloc == '>' ?
				packet->ttyloc + 1 : packet->ttyloc);
  fprintf (stream, "\r\n");
}

/* Return a list of clients that this server serves. */
CLIENT **
get_clients ()
{
  CLIENT **clients = (CLIENT **)NULL;
  CLIENT client;
  int clients_index, clients_size;
  int fd;

  warn_if_not_recent (HOSTSTAT);

  if ((fd = open (HOSTSTAT, O_RDONLY, 0666)) < 0 ||
      flock (fd, LOCK_SH) < 0)
    {
      file_error (FATAL, HOSTSTAT);
    }

  clients_index = 0;
  clients_size = 0;

  while ((read (fd, &client, sizeof (client))) == sizeof (CLIENT))
    {
      if (clients_index + 1 >= clients_size)
	{
	  if (!clients)
	    clients = (CLIENT **)
	      xmalloc ((clients_size = 20) * sizeof (CLIENT *));
	  else
	    clients = (CLIENT **)
	      xrealloc (clients, (clients_size += 20) * sizeof (CLIENT *));
	}

      clients[clients_index] = (CLIENT *)xmalloc (sizeof (CLIENT));
      bcopy (&client, clients[clients_index], sizeof (CLIENT));
      clients[++clients_index] = (CLIENT *)NULL;
    }
  close (fd);

  if (!clients)
    {
      handle_error (FATAL, "%s: no client info found", HOSTSTAT);
    }

  return (clients);
}

/* Simply complain if FILENAME hasn't been written in a while.  This can be
   an indication that the finger server has wedged or otherwise stopped
   running.  Return non-zero if the file hasn't been written recently. */
int
warn_if_not_recent (filename)
     char *filename;
{
  int result = 0;
  struct stat finfo;

  if (stat (filename, &finfo) != -1)
    {
      time_t last_change = time ((time_t *)0) - finfo.st_mtime;

      if (last_change > MIN_UPDATE_INTERVAL)
	{
	  char *itime = idle_time_string (last_change, 0);

	  handle_error (ALERT, "%s: file has not changed in %s",
		   filename, itime);

	  free (itime);
	  result = 1;
	}
    }
  else
    {
    /* don't report error here, will do so later anyway
      file_error (WARNING, filename);
     */
      result = 1;
    }

  return (result);
}

/* Report modification time of file */
int
show_last_modified (what, filename, stream)
     char *what;
     char *filename;
     FILE *stream;
{
  struct stat finfo;

  if (stat (filename, &finfo) < 0)
    {
      file_error (WARNING, filename);
      return (-1);
    }
  else
    {
      time_t last_change = time ((time_t *)0) - finfo.st_mtime;
      char *itime = idle_time_string (last_change, 0);

      fprintf (stream, "%s last updated %d seconds ago.\r\n", what, last_change);

      free (itime);
      return (0);
    }
}

/* Run a target SCRIPT for TARGET, redirecting the output to STREAM.
   Input is redirected to stream INPUT.  Current directory set to CD.
   Return 1 if execution was successful.
   The script is run with stdin set to
   /dev/null and the following arguments: 
	1st = domain or IP address of peer host
	2nd = "local" or "remote" depending on source of request
	3rd = remote user name
	4th = ARG
*/
int
run_target_script(script, target, cd, input, stream, arg)
  char *script;
  char *target;
  char *cd;
  FILE *input, *stream;
  char *arg;
{
  int pid;
#ifdef SYSV
  int status;
#else
#ifndef WEXITSTATUS
# define WEXITSTATUS(w) ((w).w_retcode)
#endif
  union wait status;
#endif

  if (access (script, X_OK) >= 0)
    {
      signal (SIGPIPE, SIG_IGN);

      /* Create child */
#ifdef HAVE_VFORK
      pid = vfork ();
#else /* !HAVE_VFORK */
      pid = fork ();
#endif /* HAVE_VFORK */
      if (pid < 0)
	{
	  file_error (WARNING, "vfork");

	  return (0);
	}
      else if (pid == 0)
	{
	  /* Make input be stdin */
	  if (fileno (input) != 0)
	    {
	      dup2 (fileno (input), 0);
	    }

	  /* Make child stdout and stderr be the TCP stream */
	  if (fileno (stream) != 1)
	    {
	      dup2 (fileno (stream), 1);
	    }

	  if (fileno (stream) != 2)
	    {
	      dup2 (fileno (stream), 2);
	    }

	  /* Set default directory */
	  if (chdir (cd) < 0)
	    {
	      file_error (WARNING, cd);
	      _exit (1);
	    }

	  execlp (script, target, peer_name,
	          local_connect ? "local" : "remote", user_name, arg, NULL);

	  file_error (WARNING, target);

	  _exit (1);
	}

      if (wait (&status) < 0)
	{
	  file_error (WARNING, "wait");

	  /* return 1 so target is aborted */
	  return (1);
	}

      return (WIFEXITED(status) && WEXITSTATUS(status) == 0);
    }

  return 0;
}

/* Run special-target script for TARGET, if any, redirecting the
   output to STREAM. The request is of FINGER_TYPE, which is either 'l'
   (long), 's' (short), or 'x' (either). Returns a non-zero value if
   TARGET is a special target. */
int
maybe_special_target (target, stream, finger_type, arg)
  char *target, finger_type;
  FILE *stream;
  char *arg;
{
  char *target_script;
  FILE *input;
  int result;

  if (!target || !stream)
    return (0);
  
  target_script = xmalloc (strlen (TARGETDIR) + 1 + 2 + strlen (target) + 1);

  sprintf (target_script, "%s/%c-%s", TARGETDIR, finger_type, target);
  
  if (!(input = fopen ("/dev/null", "r")))
    {
      file_error (WARNING, "/dev/null");
      free (target_script);
      return (0);
    }

  result = run_target_script (target_script, target, TARGETDIR,
                              input, stream, arg);
  free (target_script);

  return (result);
}

/* Try to run the fingerrc script for a user identified by the passwd ENTRY. */
int
maybe_user_script (entry, stream, packets)
  struct passwd *entry;
  FILE *stream;
  FINGER_PACKET **packets;
{
  char *ashell, *getusershell();
  struct stat sbuf;
  char *user_script;
  char temp_file[sizeof(TEMPFILE)];
  FILE *long_output;
  int result;
  extern char *mktemp();

#ifndef FINGERRC
  return (0);
#else /* FINGERRC */

  /* Check that the user is allowed to run programs on this machine,
     i.e., has a valid login shell. */
  setusershell ();
  ashell = NULL;
  while (ashell = getusershell ())
    {
      if (strcmp (entry->pw_shell, ashell) == 0)
        break;
    }
  endusershell ();

  if (!ashell)
    return (0);

  user_script = xmalloc (strlen (entry->pw_dir) + 1 + strlen (FINGERRC) + 1);

  sprintf (user_script, "%s/%s", entry->pw_dir, FINGERRC);
  
  if (stat (user_script, &sbuf) < 0 ||
      !(sbuf.st_mode & S_IFREG))
    {
      free (user_script);
      return (0);
    }

  /* Check some security restrictions on the finger script */
  if (sbuf.st_uid != entry->pw_uid ||	/* not owned by user */
      (sbuf.st_mode & 00022) ||		/* group or world writable */
      (sbuf.st_mode & 06000))		/* suid/sgid */
    {
      warning ("%s: script is insecure\n", user_script);
      free (user_script);
      return (0);
    }

  /* Collect regular long finger output in file */
  strcpy (temp_file, TEMPFILE);
  
  if (!mktemp (temp_file) ||
      !(long_output = fopen (temp_file, "w+")))
    {
      file_error (WARNING, temp_file);
      free (user_script);
      return (0);
    }

  unlink (temp_file);
  display_finger_info (entry, long_output, packets);
  rewind (long_output);
  
  /* run the script */
  result = run_target_script (user_script, entry->pw_name, entry->pw_dir,
                              long_output, stream, NULL);
  return result;
#endif /* FINGERRC */
}

/* check a connection peer against deny list */
int
deny_connection (user, host)
  char *user, *host;
{
  char *entry;
  int result = 0;

  /* check host first */
  result = match_host_config ("denyhost", host, NULL);

  if (result || *user == '\0')
    return (result);

  while (!result && (entry = get_config_entry ("denyuser", NULL)))
    {
      if (stricmp (user, entry) == 0)
	result = 1;
      free (entry);
    }

  return (result);
}

